/************************************************************************
 *			 				V e k t o r . c p p							*
 ************************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "math.h"

#include "Vektor.h"

/************************************************************************
 *  Vektor.cpp			   C V e k t o r   CONSTRUCTOR			    	*
 ************************************************************************/
CVektor::CVektor (double X, double Y, double Z)
{										
x	= X;
y	= Y;
z	= Z;
}

/************************************************************************
 *  Vektor.cpp			   C V e k t o r  COPY CONSTRUCTOR			    *
 ************************************************************************/
CVektor::CVektor (const CVektor& other)
{					
x		= other.x;
y		= other.y;
z		= other.z;
}

/************************************************************************
 *  Vektor.cpp				~ C V e k t o r   DESTRUCTOR				*
 ************************************************************************/
CVektor::~CVektor()
{

}


/************************************************************************
 *  Vektor.cpp				o p e r a t o r  + 							*
 ************************************************************************/
CVektor CVektor::operator +(const CVektor& V)
{
return CVektor (x + V.x, 
				y + V.y, 
				z + V.z);
}

/************************************************************************
 *  Vektor.cpp				o p e r a t o r  - 							*
 ************************************************************************/
CVektor CVektor::operator -(const CVektor& V)
{
return CVektor (x - V.x, 
				y - V.y, 
				z - V.z);
}

/************************************************************************
 *  Vektor.cpp				o p e r a t o r  - 							*
 ************************************************************************/
CVektor CVektor::operator -()
{
return CVektor (-x, 
				-y, 
				-z);
}


/************************************************************************
 *  Vektor.cpp				o p e r a t o r  * 							*
 ************************************************************************/
CVektor CVektor::operator *(const double fFaktor)
{
return CVektor (fFaktor * x, 
				fFaktor * y, 
				fFaktor * z);
}

/************************************************************************
 *  Vektor.cpp				o p e r a t o r  *= 						*
 ************************************************************************/
const CVektor& CVektor::operator *=(const double fFaktor)
{
x = x * fFaktor; 
y = y * fFaktor;  
z = z * fFaktor;  
return *this;
}

/************************************************************************
 *  Vektor.cpp				o p e r a t o r  = 							*
 ************************************************************************/
CVektor& CVektor::operator =(const CVektor& V)
{
x = V.x, 
y = V.y, 
z = V.z;

return *this;
}
 
/************************************************************************
 *  Vektor.cpp						S e t  								*
 ************************************************************************/
void CVektor::Set (double X, double Y, double Z)
{
x = X;
y = Y;
z = Z;
}

/************************************************************************
 *								A b s 									*
 *	return  : Betrag des Vektors										*
 ************************************************************************/
double CVektor::Abs ()
{
return sqrt(x * x  +  y * y  +  z * z);
}


/************************************************************************
 *							 N o r m a l i z e							*
 *	Ausgabe : norms the given vektor (lenght = 1)						*
 ************************************************************************/
void CVektor::Normalize (CVektor& V)
{
double Betr;

Betr = V.Abs();
x  = V.x/Betr;
y  = V.y/Betr;
z  = V.z/Betr;
}

/************************************************************************
 *							N o r m a l i z e							*
 *	Norms the own vektor (length = 1)									*
 ************************************************************************/
void CVektor::Normalize ()
{
double Betr;

Betr = this->Abs();
x  = x/Betr;
y  = y/Betr;
z  = z/Betr;
}

/************************************************************************
 *						S k a l a r P r o d u k t						*
 *	Eingabe:  Vektor2	[CVektor&]										*
 *	return :  Skalarprodunkt mit Vektor 2								*
 ************************************************************************/
double CVektor::SkalarProdukt (CVektor& V2)
{
return x * V2.x  +  y * V2.y  +  z * V2.z;
}


/************************************************************************
 *	Vektor.cpp				D e t e r m i n a n t e	2					*
 *	Eingabe: Vektor2	[CVektor&]										*
 *	return:  Determinante mit Vektor 2									* 
 ************************************************************************/
double CVektor::Det2 (CVektor& V2)
{
return x * V2.y  -  y * V2.x;
}


/************************************************************************
 *	Vektor.cpp				D e t e r m i n a n t e	3					*
 *	Eingabe: Vektor2 und Vektor3 [CVektor&]								*
 *	return:  Determinante mit Vektor 2 und 3							* 
 ************************************************************************/
double CVektor::Det3 (CVektor& V2, CVektor& V3)
{
double w1, w2, w3;

w1 = x * (V2.y * V3.z  -  V2.z * V3.y);
w2 = y * (V2.x * V3.z  -  V2.z * V3.x);
w3 = z * (V2.x * V3.y  -  V2.y * V3.x); 

return w1 - w2 + w3;
}


/************************************************************************
 *  Vektor.cpp						L G S 2								*
 *  this vorinitialisiert mit Ergebnis-Vektor							*
 *	Eingabe: Vektor X, Vektor Y, this = Ergebnisvektor					*
 *	Ausgabe: Vektor aus den Komponenten X und Y in "this"				*
 ************************************************************************/
BOOL CVektor::LGS2 (CVektor& VX, CVektor& VY)
{ 
BOOL bOK;
double Det0;
CVektor E;

CVektor VE(x, y, z);

Det0  = VX.Det2 (VY);
bOK = (Det0 != 0);

if (bOK)
	{
	E.Set (	VE.Det2 (VY) / Det0,
			VX.Det2 (VE) / Det0);
	x = E.x;
	y = E.y;
	z = E.z;
	}

return bOK;
}

/************************************************************************
 *  Vektor.cpp						L G S 3								*
 *  this vorinitialisiert mit Ergebnis-Vektor							*
 *	Eingabe: Vektor X, Vektor Y, Vektor Z, this = Ergebnisvektor		*
 *	Ausgabe: Vektor aus den Komponenten X, Y und Z in "this"			*
 ************************************************************************/
BOOL CVektor::LGS3 (CVektor& VX, CVektor& VY, CVektor& VZ)
{
BOOL bOK;
double Det0;
CVektor E;

CVektor VE(x, y, z);

Det0  = VX.Det3 (VY,VZ);
bOK = (Det0 != 0);

if (bOK)
	{
	E.Set (	VE.Det3 (VY,VZ) / Det0,
			VX.Det3 (VE,VZ) / Det0,
			VX.Det3 (VY,VE) / Det0);
	x = E.x;
	y = E.y;
	z = E.z;
	}

return bOK;
}


/************************************************************************
 *	Vektor.cpp			    V e k t o r A n g l e						*
 *	Eingabe: Vektor2	[CVektor&]										*
 *	return : Winkel zwischen den Vektoren this und V2 in RAD			*
 ************************************************************************/
double CVektor::VektorAngle (CVektor& V2)
{
double help;

help = this->SkalarProdukt(V2)/this->Abs()/V2.Abs();
if (help > 1.) help = 1.;
if (help < -1.) help = -1.;
return acos(help);
}

/************************************************************************
 *  Vektor.cpp				G r o s s k r e i s							*
 *	Input:  vLocB:	Vektor, building angle Wmax with this vector.		*
 *			w2:		angle btn 0 and Wmax								*
 *	Return:		Vektor building angle w2 with this vector.				*
 ************************************************************************/
CVektor CVektor::GrossKreis (CVektor& vLocB, double w2)
{
CVektor vX, N, S;
double nxy, w1, a;
double x1, y1;

N.VektorProdukt (*this, vLocB);	/* Koordinaten der Normalen auf */
					/* der Ebene, die durch die	*/
					/* beiden Ortsvektoren aufge-	*/
					/* spannt wird.			*/

					/* S = Schnittlinie zwischen	*/
					/* Ebene A-B und Aequatorebene	*/
nxy  = sqrt(N.x * N.x + N.y * N.y);
S.x = -N.y/nxy;				/* cos (winkel)			*/
S.y =  N.x/nxy;				/* sin (winkel)			*/
S.z =  0.;				/* winkel: zw. (1,0,0) und S	*/

w1 = this->VektorAngle (S);		/* Winkel zwischen Aequator-	*/
if (z < 0.) w1 *=	-1;		/* ebene und Ort A		*/

a = w1 + w2;				/* Winkel zwischen Aequator-	*/
					/* ebene und Ort X		*/

x1 = cos(a);				/* Koordinaten relativ zur	*/
y1 = N.z/N.Abs() * sin(a);	/* Schnittlinie von der		*/
					/* Ebene A-B und der 		*/
					/* Aequatorebene.		*/
vX.x = x1 * S.x  -  y1 * S.y;
vX.y = x1 * S.y  +  y1 * S.x;
if (N.z != 0.)	vX.z = y1 * nxy/N.z;
		else	vX.z = sin(a);

return vX;
}


/************************************************************************
 *							   Vektor Produkt							*
 *	Eingabe: Vektor 1 und Vektor 2	[CVektor&]							*
 ************************************************************************/
void CVektor::VektorProdukt (CVektor& V1, CVektor& V2)
{
x = V1.y * V2.z  -  V1.z * V2.y;
y = V1.z * V2.x  -  V1.x * V2.z;
z = V1.x * V2.y  -  V1.y * V2.x;
}

/************************************************************************
 *							 M u l t  M a t r i x  3					*
 *  Multipliziert die 3-spaltige Matrix V1,V2,V3 mit Vektor B1			*
 ************************************************************************/
void CVektor::MultMatrix3 (CVektor& V1, CVektor& V2, CVektor& V3, CVektor& B1)
{
x = B1.x * V1.x +  B1.y * V2.x + B1.z * V3.x;
y = B1.x * V1.y +  B1.y * V2.y + B1.z * V3.y;
z = B1.x * V1.z +  B1.y * V2.z + B1.z * V3.z;
}

/************************************************************************
 *							  R o t a t e  X Y							*
 *  Rotiert den Vektor "V1"  "a" Grad [rad] um den Punkt "P"			*
 ************************************************************************/
void CVektor::RotateXY (CVektor& V1, CVektor& P, double a)
{
x = cos(a) * V1.x - sin(a) * V1.y +
			(P.x - P.x * cos(a) + P.y * sin(a));
y = sin(a) * V1.x + cos(a) * V1.y +
			(P.y - P.x * sin(a) - P.y * cos(a));
z = V1.z;
}


/************************************************************************
 *  Vektor.cpp				    R o t a t e  X Y						*
 *  rRotated: this Vektor, um "a" Grad [rad] um den Punkt "P" rotiert	*
 ************************************************************************/
CVektor CVektor::RotateXY (CVektor& P, double a)
{
CVektor V1= *this;

CVektor rRotated (cos(a) * x - sin(a) * y +
				(P.x - P.x * cos(a) + P.y * sin(a)),

				sin(a) * x + cos(a) * y +
				(P.y - P.x * sin(a) - P.y * cos(a)),

				z);
return rRotated;
}

/************************************************************************
 *								  R o t a t e							*
 *  Rotiert this Vektor "a" Grad [rad] um den Punkt "P"					*
 ************************************************************************/
void CVektor::Rotate (CVektor& P, double a)
{
CVektor V1= *this;

x = cos(a) * V1.x - sin(a) * V1.y +
			(P.x - P.x * cos(a) + P.y * sin(a));
y = sin(a) * V1.x + cos(a) * V1.y +
			(P.y - P.x * sin(a) - P.y * cos(a));
z = V1.z;
}


/************************************************************************
 *						C i r c l e I n t e r s e c t i o n s	 		*
 *  Calculates intersections btn two circles of the same radius			*
 *  CA: Center of first circle, CB: Center of second circle				*
 ************************************************************************/
BOOL CVektor::CircleIntersections (CVektor& CA, CVektor& CB, double radius,
								   CVektor* ptIntersect1, CVektor* ptIntersect2)
{
	BOOL bIntersectionsAvailable = FALSE;

	CVektor RVab;			// direction from center A to center B
	RVab = CB - CA;

	double dDistBntCenters = RVab.Abs();

	CVektor M;				// middle of line btn CA and CB
	M = CA + RVab * 0.5;

	if (dDistBntCenters <= 2* radius)
	{			// intersections are available

		if (dDistBntCenters == 2*radius)
		{		// only one intersection at middle of CA-CB available
			*ptIntersect1 = M;
			*ptIntersect2 = M;
			bIntersectionsAvailable = TRUE;
		}
		else
		{		// two intersections available
			double dHalfDist = dDistBntCenters / 2;
			double dDistFromMiddleToIntersection = 
				sqrt(radius*radius - dHalfDist*dHalfDist);

			CVektor RVmc;			// direction from M to intersection
			RVmc.x = CB.y - CA.y;	// RVab, rotated 90 degrees
			RVmc.y = CA.x - CB.x;

			CVektor NRVmc(RVmc);	// normalize RVmc
			NRVmc.Normalize();

			CVektor Vmc;			// vektor from M to intersections
			Vmc = NRVmc * dDistFromMiddleToIntersection;

			*ptIntersect1 = M + Vmc;
			*ptIntersect2 = M - Vmc;
			bIntersectionsAvailable = TRUE;
		}
	}
	else
	{		// radius too small
		*ptIntersect1 = M;
		*ptIntersect2 = M;
	}

	return bIntersectionsAvailable;
}
