/************************************************************************
 *			 			 A i r s p a c e . c p p						*
 ************************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "DimDoc.h"
#include "Airspace.h"
#include "LatLon.h"

#include "resource.h"

/*
GetColor defines the color to draw the different airspaces
CreatePen creates new CPen objects for the different airspaces

*/


extern CDimDoc* 		ptDim;      

/************************************************************************
 *  Airspace.cpp			   C A i r s p a c e   CONSTRUCTOR		    *
 ************************************************************************/
CAirspace::CAirspace ()
{			
	this->Init();
}
  
/************************************************************************
 *  Airspace.cpp			  C A i r s p a c e   CONSTRUCTOR		    *
 ************************************************************************/
CAirspace::CAirspace (CString szIdentifier)
{										
	this->Init();

				// e.g.: ED000027
	strncpy (m_szIdentifier, (LPCTSTR)szIdentifier, SIZEOF_ASP_IDENT);
	m_szIdentifier[SIZEOF_ASP_IDENT-1]=0;
}  


/************************************************************************
 *  Airspace.cpp			  C A i r s p a c e   CONSTRUCTOR		    *
 ************************************************************************/
CAirspace::CAirspace (CAirspace& SourceAirspace)
{ 
	*this = SourceAirspace;
}

/************************************************************************
 *  Airspace.cpp			~ C A i r s p a c e    DESTRUCTOR			*
 ************************************************************************/
CAirspace::~CAirspace()
{
	if (m_ptPoly != NULL)
	{
		delete m_ptPoly;
		m_ptPoly = NULL;
	}
}

/************************************************************************
 *  Airspace.cpp				operator=								*
 ************************************************************************/
const CAirspace& CAirspace::operator=(const CAirspace& Airspace)
{
	m_dLatMax				= Airspace.m_dLatMax;		// top left edge
	m_dLonMin				= Airspace.m_dLonMin;
	m_dLatMin				= Airspace.m_dLatMin;		// bottom right edge
	m_dLonMax				= Airspace.m_dLonMax;
	m_nSegmentCnt			= Airspace.m_nSegmentCnt;
	m_lFirstSegmentIndex	= Airspace.m_lFirstSegmentIndex;	// index of CAirspaceSegmentDoc
	m_InputDirection		= Airspace.m_InputDirection;

	m_cFlag					= Airspace.m_cFlag;

	memcpy (m_szIdentifier, Airspace.m_szIdentifier, SIZEOF_ASP_IDENT); 
	memcpy (m_szName, Airspace.m_szName, SIZEOF_ASP_NAME); 

	m_dFrequency1			= Airspace.m_dFrequency1;
	m_nFreqDim1				= Airspace.m_nFreqDim1;

	m_cType					= Airspace.m_cType;		// A, B, C, ...
	m_cClass				= Airspace.m_cClass;	// 01,01 - A, B, C, D, E, F, G OR SPACE

	m_cLevel				= Airspace.m_cLevel;	
	m_cDummy				= Airspace.m_cDummy;
	m_lUpperEffAltMSL		= Airspace.m_lUpperEffAltMSL;	
	m_lLowerEffAltMSL		= Airspace.m_lLowerEffAltMSL;
	m_nElevDim				= Airspace.m_nElevDim;

			// required to select region in dlg opened via context menu
	memcpy (m_szRegionName, Airspace.m_szRegionName, SIZEOF_ASP_REGION);  

	m_ptPoly = NULL;		// don't copy pointer

	return *this;
}

/********************************************************************************
 *  Airspace.cpp				o p e r a t o r  ! =							*
 ********************************************************************************/
BOOL CAirspace::operator !=(const CAirspace& Airspace)
{
	if (m_cFlag != Airspace.m_cFlag)
		return TRUE;

	if (strcmp(m_szIdentifier, Airspace.m_szIdentifier) != 0)
		return TRUE;

	if (strcmp (m_szName, Airspace.m_szName) != 0)
		return TRUE; 

	if (m_dFrequency1			!= Airspace.m_dFrequency1)
		return TRUE;
	if (m_nFreqDim1				!= Airspace.m_nFreqDim1)
		return TRUE;

	if (m_cType					!= Airspace.m_cType)
		return TRUE;
	if (m_cClass				!= Airspace.m_cClass)
		return TRUE;

	if (m_cLevel				!= Airspace.m_cLevel)
		return TRUE;	
	if (m_cDummy				!= Airspace.m_cDummy)
		return TRUE;
	if (m_lUpperEffAltMSL		!= Airspace.m_lUpperEffAltMSL)
		return TRUE;	
	if (m_lLowerEffAltMSL		!= Airspace.m_lLowerEffAltMSL)
		return TRUE;
	if (m_nElevDim				!= Airspace.m_nElevDim)
		return TRUE;

return false;
}

/************************************************************************
 *  Airspace.cpp		R e s e t B o r d e r A n d I n d e x			*
 ************************************************************************/
void CAirspace::ResetBorderAndIndex ()
{
	m_dLatMax = NO_KOORD;		// top left edge
	m_dLonMin = NO_KOORD;
	m_dLatMin = NO_KOORD;		// bottom right edge
	m_dLonMax = NO_KOORD;
	m_nSegmentCnt		 = 0;
	m_lFirstSegmentIndex = -1;	// index of CAirspaceSegmentDoc
	m_InputDirection = NO_INPUTDIRECTION;
	SetAntipod(FALSE);
}

/************************************************************************
 *  Airspace.cpp					 I n i t							*
 ************************************************************************/
void CAirspace::Init ()
{
	ResetBorderAndIndex();

	m_cFlag = 0;

	memset(m_szIdentifier, 0, SIZEOF_ASP_IDENT);
	memset(m_szName, 0, SIZEOF_ASP_NAME);

	m_dFrequency1	= NO_FREQ;
	m_nFreqDim1		= DIM_MHZ;
	m_cType = CAirspace::NO_TYPE;	// CAirspace::TYPE  (CTR, ADIZ, ...)
	m_cClass= CAirspace::NO_CLASS;	// CAirspace::CLASS (A, B, C, ...)
	m_cLevel= CAirspace::NO_LEVEL;	// CAirspace::LEVEL (HIGH, LOW, ...)
	m_cDummy=' ';
	m_lUpperEffAltMSL = NO_ALT;	// INDICATES THE HIGHEST ALTITUDE (CEILING) 
	m_lLowerEffAltMSL = NO_ALT;
	m_nElevDim = DIM_FEET;

	SetActivated(TRUE);
	SetUserDefined(FALSE);
	SetAntipod(FALSE);


	memset(m_szRegionName, 0, SIZEOF_ASP_REGION);
	m_ptPoly = NULL;
}


/************************************************************************
 *  Airspace.cpp				E x p a n d B o r d e r					*
 *  SetAntipod must be called before!!!									*
 *  Updates min max border values of airspace							*
 *  for antipod airspaces, m_dLatMax is > 180!!							*
 *  called in: CAirspaceDoc::OnInputDirectionFound						*
 ************************************************************************/
void CAirspace::ExpandBorder(BOOL bInit, double dLat, double dLon)
{
	if (this->IsAntipod())
	{						// add 360 to each west lon and look for min max
		if (dLon < 0)
			dLon += 360;		// all lon > 0
	}

	if (bInit)
	{				// initialize border with first position
		m_dLatMax = dLat;
		m_dLonMin = dLon;
		m_dLatMin = dLat;
		m_dLonMax = dLon;
	}
	else
	{
			// check simple border, stored in CAirspace
			// to quickly decide, whether the airspace must be drawn or not

		if (m_dLatMax < dLat)
			m_dLatMax = dLat;

		if (m_dLonMin > dLon)
			m_dLonMin = dLon;	

		if (m_dLatMin > dLat)
			m_dLatMin = dLat;

		if (m_dLonMax < dLon)
			m_dLonMax = dLon;
	}
}


/************************************************************************
 *  Airspace.cpp			  G e t T o p L e f t E d g e				*
 ************************************************************************/
BOOL CAirspace::GetTopLeftEdge (double* ptMaxLat, double* ptMinLon)
{
	BOOL bOK = FALSE;
	if (*ptMaxLat != NULL && *ptMinLon != NULL)
	{
		*ptMaxLat = m_dLatMax;
		*ptMinLon = m_dLonMin;
		bOK = TRUE;
	}
	return bOK;
}

/************************************************************************
 *  Airspace.cpp		 G e t B o t t o m R i g h t E d g e			*
 ************************************************************************/
BOOL CAirspace::GetBottomRightEdge (double* ptMinLat, double* ptMaxLon)
{
	BOOL bOK = FALSE;
	if (*ptMinLat != NULL && *ptMaxLon != NULL)
	{
		*ptMinLat = m_dLatMin;
		*ptMaxLon = m_dLonMax;
		bOK = TRUE;
	}
	return bOK;
}


/************************************************************************
 *  Airspace.cpp    		 	G e t I d e n t i f i e r				*
 ************************************************************************/
CString CAirspace::GetIdentifier() 
{ 
	return (CString)m_szIdentifier; 
}

/************************************************************************
 *  Airspace.cpp    		 	S e t I d e n t i f i e r				*
 ************************************************************************/
void CAirspace::SetIdentifier(CString szIdentifier) 
{ 
	StringToCharArray(szIdentifier, m_szIdentifier, SIZEOF_ASP_IDENT);
}

/************************************************************************
 *  Airspace.cpp    		 	S e t I d e n t i f i e r				*
 ************************************************************************/
void CAirspace::SetIdentifier(CString szCC, long lID) 
{ 
	CString szIdentifier;
	szIdentifier.Format("%s%05ld", (LPCTSTR)szCC, lID);
	StringToCharArray(szIdentifier, m_szIdentifier, SIZEOF_ASP_IDENT);
}


/************************************************************************
 *  Airspace.cpp    		 G e t C o u n t r y C o d e				*
 ************************************************************************/
CString CAirspace::GetCountryCode()
{
	CString szCC;
	CString szIdentifier = this->GetIdentifier();
	szCC = szIdentifier.Left(2);
	return szCC;
}

/************************************************************************
 *  Airspace.cpp    		 S e t C o u n t r y C o d e 				*
 ************************************************************************/
void CAirspace::SetCountryCode(CString szCC)
{
	long lID = this->GetID();
	this->SetIdentifier(szCC, lID);
}


/************************************************************************
 *  Airspace.cpp    				 G e t I D							*
 ************************************************************************/
long CAirspace::GetID()
{
	long lID;
	CString szID;
	CString szIdentifier = this->GetIdentifier();
	szID = szIdentifier.Right(szIdentifier.GetLength() - 2);
	lID = atol((LPCTSTR)szID);
	return lID;
}

/************************************************************************
 *  Airspace.cpp    				 S e t I D							*
 ************************************************************************/
void CAirspace::SetID(long lID)
{
	CString szCC = this->GetCountryCode();
	this->SetIdentifier(szCC, lID);
}

/************************************************************************
 *  Airspace.cpp    		 	G e t R e g i o n N a m e				*
 ************************************************************************/
CString CAirspace::GetRegionName()
{         
CString szRegionName = (CString)m_szRegionName;
return szRegionName;
}

/************************************************************************
 *  Airspace.cpp    		 	S e t R e g i o n N a m e 				*
 ************************************************************************/
void CAirspace::SetRegionName(CString szRegionName)
{    
	StringToCharArray(szRegionName, m_szRegionName, SIZEOF_ASP_REGION);
}             

/************************************************************************
 *  Airspace.cpp    		 	G e t N a m e							*
 ************************************************************************/
CString CAirspace::GetName() 
{ 
	return (CString)m_szName; 
}

/************************************************************************
 *  Airspace.cpp    		 	S e t N a m e							*
 ************************************************************************/
void CAirspace::SetName(CString szName) 
{ 
	StringToCharArray(szName, m_szName, SIZEOF_ASP_NAME);
}

/************************************************************************
 *  Airspace.cpp    		 G e t D i s p l a y N a m e				*
 ************************************************************************/
CString CAirspace::GetDisplayName()
{
	CString name;
	name = GetName();
	return name;
}

/************************************************************************
 *  Airspace.cpp    		 	S e t C l a s s							*
 ************************************************************************/
void CAirspace::SetClass(char cClass)
{
	switch (cClass)
	{
		case 'A':	m_cClass = CAirspace::A;	break;
		case 'B':	m_cClass = CAirspace::B;	break;
		case 'C':	m_cClass = CAirspace::C;	break;
		case 'D':	m_cClass = CAirspace::D;	break;
		case 'E':	m_cClass = CAirspace::E;	break;
		case 'F':	m_cClass = CAirspace::F;	break;
		case 'G':	m_cClass = CAirspace::G;	break;
		case '-':	m_cClass = CAirspace::NO_CLASS;	break;
		default: m_cClass = CAirspace::NO_CLASS; break;
	}
}

/************************************************************************
 *  Airspace.cpp    		 	S e t S e v e l							*
 ************************************************************************/
void CAirspace::SetLevel(char cLevel)
{
	switch (cLevel)
	{
		case 'B': m_cLevel = CAirspace::HIGHLOW;	break; 
		case 'H': m_cLevel = CAirspace::HIGH;		break; 
		case 'L': m_cLevel = CAirspace::LOW;		break; 
		default:  m_cLevel = CAirspace::NO_LEVEL;	break;
	}
}

/************************************************************************
 *  Airspace.cpp    		 	S e t T y p e							*
 ************************************************************************/
void CAirspace::SetType (short nType)
{				// Values from DAFIF:
				//	01 - ADVISORY AREA (ADA) OR (UDA)
				//	02 - AIR DEFENSE IDENTIFICATION ZONE (ADIZ)
				//	03 - AIR ROUTE TRAFFIC CONTROL CENTER 		(ARTCC)
				//	04 - AREA CONTROL CENTER (ACC)
				//	05 - BUFFER ZONE (BZ)
				//	06 - CONTROL AREA (CTA) (UTA)
				//		  SPECIAL RULES AREA (SRA, U.K. ONLY)
				//	07 - CONTROL ZONE (CTLZ)
				//		  SPECIAL RULES ZONE (SRZ, U.K.  ONLY) 		MILITARY AERODROME TRAFFIC ZONE 		(MATZ, U.K. ONLY)
				//	08 - FLIGHT INFORMATION REGION (FIR)
				//	09 - OCEAN CONTROL AREA (OCA)
				//	10 - RADAR AREA
				//	11 - TERMINAL CONTROL AREA (TCA) OR (MTCA)
				//	12 - UPPER FLIGHT INFORMATION REGION (UIR) 
	if (1 <= nType && nType <= 12)
		 m_cType = (CAirspace::TYPE)(nType - 1);		// convert to enum, starting with 0
	else m_cType = CAirspace::NO_TYPE;
}


/************************************************************************
 *  Airspace.cpp    	 		S e t F l a g B i t						*
 ************************************************************************/
void CAirspace::SetFlagBit (unsigned char FAR* lpFlag, short BitMask, BOOL bSet)
{					/* BitMask: 1, 2, 4, 8, 16, ...	*/
	if (bSet)   *lpFlag = *lpFlag | (unsigned char)BitMask;
		else    *lpFlag = *lpFlag & (unsigned char)(~BitMask);	/* ALT 126: ~	*/
}

/************************************************************************
 *  Airspace.cpp    		 	S e t A c t i v a t e d 				*
 ************************************************************************/
void CAirspace::SetActivated(BOOL bActivated)
{   
	SetFlagBit (&m_cFlag, ASP_ACTIVATED, bActivated);
}

/************************************************************************
 *  Airspace.cpp    		 	I s A c t i v a t e d					*
 ************************************************************************/
BOOL CAirspace::IsActivated(void)
{ 
	BOOL bActivated = ((m_cFlag & ASP_ACTIVATED) == ASP_ACTIVATED);
	return bActivated;
}

/************************************************************************
 *  Airspace.cpp    	 	S e t U s e r D e f i n e d 				*
 ************************************************************************/
void CAirspace::SetUserDefined(BOOL bUserDefined)
{   
	SetFlagBit (&m_cFlag, ASP_USERDEFINED, bUserDefined);
}

/************************************************************************
 *  Airspace.cpp    		I s U s e r D e f i n e d 					*
 ************************************************************************/
BOOL CAirspace::IsUserDefined(void)
{ 
	BOOL bUserDefined = ((m_cFlag & ASP_USERDEFINED) == ASP_USERDEFINED);
	return bUserDefined;
}

/************************************************************************
 *  Airspace.cpp    	 	S e t A n t i p o d			 				*
 ************************************************************************/
void CAirspace::SetAntipod(BOOL bAntipod)
{   
	SetFlagBit (&m_cFlag, ASP_ANTIPOD, bAntipod);
}

/************************************************************************
 *  Airspace.cpp    		I s A n t i p o d		 					*
 ************************************************************************/
BOOL CAirspace::IsAntipod(void)
{ 
	BOOL bAntipod = ((m_cFlag & ASP_ANTIPOD) == ASP_ANTIPOD);
	return bAntipod;
}

/************************************************************************
 *  Airspace.cpp    		 	G e t L a b e l T e x t 				*
 ************************************************************************/
CString CAirspace::GetLabelText()
{
	CString szLabelText;

	short nClassID = (short)this->GetClass();
	if (nClassID != 255)
	{
		CString szClassID;
		szClassID.LoadString(IDS_ASPCLASS01 + nClassID);
		szLabelText = "[";
		szLabelText += szClassID;	
		szLabelText += "]";
	}

	CString szAltInfo = this->GetAltText();

	BOOL bTest = FALSE;
	if (bTest)	// for test: replace altitude range with name of airspace
		szAltInfo = this->GetDisplayName();	
	
	if (szAltInfo.GetLength() > 0)
	{
		szLabelText += " ";
		szLabelText += szAltInfo;
	}

	return szLabelText;
}

/************************************************************************
 *  Airspace.cpp    		 	G e t A l t T e x t						*
 ************************************************************************/
CString CAirspace::GetAltText()
{
	CString szAltInfo;

	CString szDim;
	ptDim->GetDimCstring(&szDim, m_nElevDim);

	if (m_lLowerEffAltMSL != NO_ALT && m_lUpperEffAltMSL != NO_ALT)
	{
		szAltInfo.Format((LPCTSTR)"%ld - %ld %s", 
			m_lLowerEffAltMSL, m_lUpperEffAltMSL, (LPCTSTR)szDim);
	}
	else
	{
		if (m_lLowerEffAltMSL != NO_ALT)
			szAltInfo.Format((LPCTSTR)"FL100 - %ld %s", 
				m_lLowerEffAltMSL, (LPCTSTR)szDim);
		if (m_lUpperEffAltMSL != NO_ALT)
			szAltInfo.Format((LPCTSTR)"%ld %s MSL", 
				m_lUpperEffAltMSL, (LPCTSTR)szDim);
	}

	switch (m_cType)
	{

	case CAirspace::ADVISORY_AREA:			// ADVISORY AREA
		break;

	case CAirspace::ADIZ:					// AIR DEFENSE IDENTIFICATION ZONE
		break;

	case CAirspace::AIR_RTE_TRAFFIC_CTR:	// AIR ROUTE TRAFFIC CONTROL CENTER
		break;

	case CAirspace::AREA_CTR:				// AREA CONTROL CENTER
		break;
	
	case CAirspace::BUFFER_ZONE:			// BUFFER ZONE
		break;

	case CAirspace::CONTROL_AREA:			// CONTROL AREA

		switch (m_cClass)
		{
		case CAirspace::A:
			break;
		case CAirspace::B:
			break;
		case CAirspace::C:
			break;
		case CAirspace::D:
			break;
		case CAirspace::E:
			break;
		case CAirspace::F:
			break;
		case CAirspace::G:
			break;
		default:
			break;

		}
		break;

	case CAirspace::CONTROL_ZONE:			// CONTROL ZONE
		break;

	case CAirspace::FIR:					// FLIGHT INFORMATION REGION
		break;

	case CAirspace::OCEAN_CTR:				// OCEAN CONTROL AREA
		break;

	case CAirspace::RADAR_AREA:				// RADAR AREA
		break;

	case CAirspace::TCR:					// TERMINAL CONTROL AREA
		break;

	case CAirspace::UPPER_FIR:				// UPPER FLIGHT INFORMATION REGION
		break;

	case CAirspace::NO_TYPE:
		break;
	}

	return szAltInfo;
}


/************************************************************************
 *  Airspace.cpp    	C o u r s e T o T e x t A n g l e				*
 *  Input: course from start position to end position of one segment	*
 *  Output: course to rotate label text to be parallel to segment		*
 ************************************************************************/
double CAirspace::CourseToTextAngle(double dCourse)
{
	double dAngle;

	if (dCourse > 180)
		dCourse -= 180;

	dAngle = dCourse - 90;
	return dAngle;
}

/************************************************************************
 *  Airspace.cpp    	C o u r s e T o L a b e l P o s i t i o n		*
 *  Input: course from start position to end position of one segment	*
 *  Output: course from center of segment to position of label			*
 ************************************************************************/
double CAirspace::CourseToLabelPosition(double dCourse)
{
	double dAngle=dCourse;

	if (this->m_InputDirection == CAirspace::CLOCKWISE)
		dAngle = dCourse + 90;
	if (this->m_InputDirection == CAirspace::COUNTERCLOCKWISE)
		dAngle = dCourse - 90;

	if (dAngle < 0)
		dAngle += 360;
	if (dAngle > 360)
		dAngle -= 360;

	return dAngle;
}

 
/************************************************************************
 *  Airspace.cpp    		 	G e t C o l o r							*
 ************************************************************************/
COLORREF CAirspace::GetColor()
{
	COLORREF crColor;

	switch (m_cType)
	{
	case CAirspace::ADVISORY_AREA:			// ADVISORY AREA
		crColor = RGB (0, 0, 0);
		break;

	case CAirspace::ADIZ:					// AIR DEFENSE IDENTIFICATION ZONE
		crColor = RGB (0, 128, 0);
		break;

	case CAirspace::AIR_RTE_TRAFFIC_CTR:	// AIR ROUTE TRAFFIC CONTROL CENTER
		crColor = RGB (255, 0, 255);
		break;

	case CAirspace::AREA_CTR:				// AREA CONTROL CENTER
		crColor = RGB (0, 0, 128);
		break;
	
	case CAirspace::BUFFER_ZONE:			// BUFFER ZONE
		crColor = RGB (128, 128, 128);
		break;

	case CAirspace::CONTROL_AREA:			// CONTROL AREA

		switch (m_cClass)
		{
		case CAirspace::A:
			crColor = RGB (0, 0, 0);
			break;
		case CAirspace::B:
			crColor = RGB (0, 0, 0);
			break;
		case CAirspace::C:
			crColor = RGB (0, 255, 0);
			break;
		case CAirspace::D:
			crColor = RGB (0, 255, 0);
			break;
		case CAirspace::E:
			crColor = RGB (255, 0, 0);
			break;
		case CAirspace::F:
			crColor = RGB (0, 0, 0);
			break;
		case CAirspace::G:
			crColor = RGB (0, 0, 0);
			break;
		default:
			crColor = RGB (0, 0, 0);
			break;

		}
		break;

	case CAirspace::CONTROL_ZONE:			// CONTROL ZONE
		crColor = RGB (255, 0, 0);
		break;

	case CAirspace::FIR:					// FLIGHT INFORMATION REGION
		crColor = RGB (0, 128, 128);
		break;

	case CAirspace::OCEAN_CTR:				// OCEAN CONTROL AREA
		crColor = RGB (0, 0, 255);
		break;

	case CAirspace::RADAR_AREA:				// RADAR AREA
		crColor = RGB (128, 128, 0);
		break;

	case CAirspace::TCR:					// TERMINAL CONTROL AREA
		crColor = RGB (0, 255, 0);
		break;

	case CAirspace::UPPER_FIR:				// UPPER FLIGHT INFORMATION REGION
		crColor = RGB (0, 220, 220);
		break;

	case CAirspace::NO_TYPE:
		crColor = RGB (255, 255, 255);
		break;
	}

//	crColor = RGB (0, 0, 0);				// for b&w docu only
	return crColor;
}

/************************************************************************
 *  Airspace.cpp    		 	C r e a t e P e n						*
 ************************************************************************/
CPen* CAirspace::CreatePen(CDC* pDC)
{
	CPen* ptPen = NULL;
	COLORREF crColor = this->GetColor();

//	#define PS_SOLID            0
//#define PS_DASH             1       /* -------  */
//#define PS_DOT              2       /* .......  */
//#define PS_DASHDOT          3       /* _._._._  */
//#define PS_DASHDOTDOT       4       /* _.._.._  */

	short nControlZonePenSize = 6;

	switch (m_cType)
	{

	case CAirspace::ADVISORY_AREA:			// ADVISORY AREA
		ptPen = new CPen(PS_SOLID, 10, crColor);
		break;

	case CAirspace::ADIZ:					// AIR DEFENSE IDENTIFICATION ZONE
		ptPen = new CPen(PS_SOLID, 10, crColor);
		break;

	case CAirspace::AIR_RTE_TRAFFIC_CTR:	// AIR ROUTE TRAFFIC CONTROL CENTER
		ptPen = new CPen(PS_SOLID, 10, crColor);
		break;

	case CAirspace::AREA_CTR:				// AREA CONTROL CENTER
		ptPen = new CPen(PS_DASH, 1, crColor);
		break;
	
	case CAirspace::BUFFER_ZONE:			// BUFFER ZONE
		ptPen = new CPen(PS_SOLID, 10, crColor);
		break;

	case CAirspace::CONTROL_AREA:			// CONTROL AREA
	//	ptPen = new CPen(PS_DOT, 1, crColor);
		ptPen = new CPen(PS_SOLID, nControlZonePenSize, crColor);
		break;

	case CAirspace::CONTROL_ZONE:			// CONTROL ZONE
		ptPen = new CPen(PS_SOLID, nControlZonePenSize, crColor);
		break;

	case CAirspace::FIR:					// FLIGHT INFORMATION REGION
		ptPen = new CPen(PS_SOLID, nControlZonePenSize, crColor);
		break;

	case CAirspace::OCEAN_CTR:				// OCEAN CONTROL AREA
		ptPen = new CPen(PS_SOLID, nControlZonePenSize, crColor);
		break;

	case CAirspace::RADAR_AREA:				// RADAR AREA
		ptPen = new CPen(PS_SOLID, nControlZonePenSize, crColor);
		break;

	case CAirspace::TCR:					// TERMINAL CONTROL AREA
	//	ptPen = new CPen(PS_DOT, 1, crColor);
		ptPen = new CPen(PS_SOLID, nControlZonePenSize, crColor);
		break;

	case CAirspace::UPPER_FIR:				// UPPER FLIGHT INFORMATION REGION
		ptPen = new CPen(PS_SOLID, 10, crColor);
		break;

	case CAirspace::NO_TYPE:
		ptPen = new CPen(PS_SOLID, 10, crColor);
		break;
	}

	return ptPen;
}

/************************************************************************
 *  Airspace.cpp    		 	C h a r T o F r e q D i m				*
 ************************************************************************/
short CAirspace::CharToFreqDim(char cDim)
{
	short nFreqDim=0;
	switch(cDim)
	{
		case 'K': nFreqDim = DIM_KHZ;	break;
		case 'M': nFreqDim = DIM_MHZ;	break;
	}

	return nFreqDim;
}

/************************************************************************
 *  Airspace.cpp    		 S e t F r e q u e n c y					*
 ************************************************************************/
void CAirspace::SetFrequency(float fFreq, short nFreqDim) 
{ 
	m_dFrequency1 = fFreq;
	m_nFreqDim1 = nFreqDim;
}

/************************************************************************
 *  Airspace.cpp    		G e t F r e q u e n c y						*
 ************************************************************************/
void CAirspace::GetFrequency(float* ptFreq, short* ptFreqDim) 
{ 
	*ptFreq		= m_dFrequency1;
	*ptFreqDim	= m_nFreqDim1;
}

/************************************************************************
 *  Airspace.cpp    		S t r i n g T o C h a r A r r a y			*
 *  Copies CString to char array. If CString is longer than array,		*
 *	only nArrayLen-1 characters will be copied!							*
 ************************************************************************/
void CAirspace::StringToCharArray(CString szText, char* ptText, short nArrayLen)
{
	short nLen = szText.GetLength();
	if (nLen < nArrayLen)
	{
		strcpy(ptText, (LPCTSTR)szText);
	}
	else
	{			// cut part of input parameter
		strncpy(ptText, (LPCTSTR)szText, nArrayLen-1);
		ptText[nArrayLen-1] = 0;
	}
}


/************************************************************************
 *  Airspace.cpp    		L o x o D i s t								*
 *  Returns dist from lat lon specified to middle of this airspace!		*
 ************************************************************************/
double CAirspace::LoxoDist (double dLat, double dLon, double* ptCourse)
{
	double dDist;

	double dCenterLat = (m_dLatMin + m_dLatMax)/2;
	double dCenterLon = (m_dLonMin + m_dLonMax)/2;

	CLatLon llCenter(dCenterLat, dCenterLon);
	dDist = llCenter.LoxoDist(dLat, dLon, ptCourse);

	return dDist;
}


/************************************************************************
 *  Airspace.cpp		B o r d e r C o n t a i n s						*
 *  returns TRUE, if fLat fLon is within border of this airspace		*
 ************************************************************************/
BOOL CAirspace::BorderContains (double dLat, double dLon)
{	
	BOOL bFound = FALSE;

	if (IsAntipod())
	{
		if (dLon < 0)		// west longitude
			dLon += 360;	// right lon of antipod border is > 180 !!
	}

	if ((m_dLatMax >= dLat && dLat > m_dLatMin) &&
		 (m_dLonMin <= dLon && dLon < m_dLonMax) )
	{
		bFound = IsActivated();		// TRUE only, if activated!!
	}

	return bFound;
}

/************************************************************************
 *  Airspace.cpp  			I s I n R e c t								*
 ************************************************************************/
BOOL CAirspace::IsInRect(DRECT rLatLon, double dLat, double dLon)
{
	BOOL bFound = FALSE;

	if ((rLatLon.top >= dLat && dLat > rLatLon.bottom) &&
		 (rLatLon.left <= dLon && dLon < rLatLon.right) )
	{
		bFound = TRUE;
	}

	return bFound;
}


/************************************************************************
 *  Airspace.cpp    		 	I s I n R e c t 						*
 ************************************************************************/
BOOL CAirspace::IsInRect(DRECT rLatLon)
{
	BOOL bInRect = FALSE;


						// get temp rLatLon to compare it with airspace border
	DRECT rTmpLatLon;
	rTmpLatLon = rLatLon;

	if (rLatLon.left < -180)
	{		// antipod map lon: -206 ... -138
		if (m_dLonMax > 0)
		{		// east lon airspace
			rTmpLatLon.left += 360;
			rTmpLatLon.right += 360;
		}
	}

	if (rLatLon.right > 180)
	{		// antipod map lon: 154 ... 222
		if (m_dLonMin < 0)
		{		// west lon airspace
			rTmpLatLon.left -= 360;
			rTmpLatLon.right -= 360;
		}
	}


	if (this->IsAntipod())
	{			// antipod airspace, m_dLonMax > 180
		if (0 > rLatLon.left && rLatLon.left >= -180 )
		{		// std. west rLatLon, all lon values > -180
			rTmpLatLon.left += 360;
			rTmpLatLon.right += 360;
		}
	}


		// test, if at least one point of rLatLon is within this airspaces borders
	if (!bInRect && BorderContains(rTmpLatLon.top, rTmpLatLon.left))	// TL of rLatLon
		bInRect = TRUE;
	if (!bInRect && BorderContains(rTmpLatLon.top, rTmpLatLon.right))	// TR of rLatLon
		bInRect = TRUE;
	if (!bInRect && BorderContains(rTmpLatLon.bottom, rTmpLatLon.right))	// BR of rLatLon
		bInRect = TRUE;
	if (!bInRect && BorderContains(rTmpLatLon.bottom, rTmpLatLon.left))	// BL of rLatLon
		bInRect = TRUE;

		//  test if at least one of this regions borders is within rLatLon
	if (!bInRect && IsInRect (rTmpLatLon, m_dLatMax, m_dLonMin))		// TL of region
		bInRect = TRUE;
	if (!bInRect && IsInRect (rTmpLatLon, m_dLatMax, m_dLonMax))		// TR of region
		bInRect = TRUE;
	if (!bInRect && IsInRect (rTmpLatLon, m_dLatMin, m_dLonMax))		// BR of region
		bInRect = TRUE;
	if (!bInRect && IsInRect (rTmpLatLon, m_dLatMin, m_dLonMin))		// BL of region
		bInRect = TRUE;

	if (bInRect)
	{
		bInRect = IsActivated();		// TRUE only, if activated!!
	}

	return bInRect;
}

/************************************************************************
 *  Airspace.cpp    		 	S e r i a l i z e						*
 ************************************************************************/
void CAirspace::Serialize(CArchive& ar, short nVersion)
{ 
	if (ar.IsStoring())
	{
		// TODO: add storing code here 
		ar << m_dLatMax;		// top left edge
		ar << m_dLonMin;
		ar << m_dLatMin;		// bottom right edge
		ar << m_dLonMax;
		ar << m_nSegmentCnt;
		ar << m_lFirstSegmentIndex;	// index of CAirspaceSegmentDoc
		ar << m_InputDirection;

		ar << (BYTE)m_cFlag;  		

		ar.Write (m_szIdentifier, SIZEOF_ASP_IDENT);	// e.g.: ED000027

		ar.Write(m_szName, SIZEOF_ASP_NAME);

		ar << m_dFrequency1;
		ar << m_nFreqDim1;
		ar << (BYTE)m_cType;			// CAirspace::TYPE  (CTR, ADIZ, ...)
		ar << (BYTE)m_cClass;			// CAirspace::CLASS (A, B, C, ...)
		ar << (BYTE)m_cLevel;			// CAirspace::LEVEL (HIGH, LOW, ...)
		ar << (BYTE)m_cDummy;
		ar << m_lUpperEffAltMSL;	// INDICATES THE HIGHEST ALTITUDE (CEILING) 
		ar << m_lLowerEffAltMSL;
		ar << m_nElevDim;
	}
	else
	{       
		// TODO: add loading code here    		
		BYTE	Byte; 
		int		nInpDir;
		
		ar >> m_dLatMax;		// top left edge
		ar >> m_dLonMin;
		ar >> m_dLatMin;		// bottom right edge
		ar >> m_dLonMax;
		ar >> m_nSegmentCnt;
		ar >> m_lFirstSegmentIndex;	// index of CAirspaceSegmentDoc
		ar >> nInpDir;	m_InputDirection = (INPUTDIRECTION)nInpDir;

		ar >> Byte;		m_cFlag = (unsigned char)Byte;

		ar.Read (m_szIdentifier, SIZEOF_ASP_IDENT);   // e.g.: ED000027
		ar.Read (m_szName, SIZEOF_ASP_NAME);

		ar >> m_dFrequency1;
		ar >> m_nFreqDim1; 
		ar >> Byte;		m_cType = (TYPE)Byte;			// CAirspace::TYPE  (CTR, ADIZ, ...)
		ar >> Byte;		m_cClass = (CLASS)Byte;			// CAirspace::CLASS (A, B, C, ...)
		ar >> Byte;		m_cLevel = (LEVEL)Byte;			// CAirspace::LEVEL (HIGH, LOW, ...)
		ar >> Byte;		m_cDummy = (unsigned char)Byte; 

		ar >> m_lUpperEffAltMSL;	// INDICATES THE HIGHEST ALTITUDE (CEILING) 
		ar >> m_lLowerEffAltMSL;
		ar >> m_nElevDim;

	//	OemToAnsi(m_szIdentifier, m_szIdentifier);
	}
} 
 