// AirspaceDoc.cpp: Implementierungsdatei
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "pf.h"

#include "DimDoc.h"
#include "AirspaceDoc.h"
#include "..\CPPTOOLS\Polygon.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


extern BOOL bDemo;

// SetSegmentDataValid should be set to FALSE, after a CAirspaceSegment was changed or added.
// Functions providing CAirspace Border or SegmentIndex should check the valid flag
// If it is not TRUE, then CAirspaceDoc::UpdateSegmentBorderAndIndex must be called

// UpdateSegmentBorderAndIndex creates a rectangular border around each airspace and
//   determines the direction of input.

// GetAirspaceAt calls CreatePolygonFor to create a temporarily used polygon. The pointer to
// that polygon is hold (but not serialized) in CAirspace  

// Airspace border may contain MaxLon > 180 for antipod airspaces!!
// AirspaceDocument border may contain MaxLon > 180 if there are airspaces around lon 180!!

/////////////////////////////////////////////////////////////////////////////
// CAirspaceDoc

IMPLEMENT_DYNCREATE(CAirspaceDoc, CDocument)

extern CDimDoc*		ptDim;

CAirspaceDoc::CAirspaceDoc()
{
	m_bOpenSubDocuments = TRUE;
	m_bConverted = FALSE;
	m_szExt = GetExt();
	m_dMaxLat1 = NO_KOORD;		// top left edge
	m_dMinLon1 = NO_KOORD;
	m_dMinLat1 = NO_KOORD;		// bottom right edge
	m_dMaxLon1 = NO_KOORD;

	m_ptSegmentDoc = new CAirspaceSegmentDoc();

	this->InitHeader();
}

BOOL CAirspaceDoc::OnNewDocument()
{
	if (!CDocument::OnNewDocument())
		return FALSE;

	if (!m_ptSegmentDoc->OnNewDocument())
		return FALSE;

	return TRUE;
}


CAirspaceDoc::~CAirspaceDoc()
{
	if (m_ptSegmentDoc != NULL)
	{
		delete m_ptSegmentDoc;
		m_ptSegmentDoc = NULL;
	}

	this->DeleteArrayOf (&m_Airspaces);
}

/************************************************************************
 *  AirspaceDoc.cpp				G e t E x t 							*
 ************************************************************************/
CString CAirspaceDoc::GetExt()
{
	CString szExt(".spa");
	return szExt;
}


BEGIN_MESSAGE_MAP(CAirspaceDoc, CDocument)
	//{{AFX_MSG_MAP(CAirspaceDoc)
		// HINWEIS - Der Klassen-Assistent fgt hier Zuordnungsmakros ein und entfernt diese.
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// Diagnose CAirspaceDoc

#ifdef _DEBUG
void CAirspaceDoc::AssertValid() const
{
	CDocument::AssertValid();
}

void CAirspaceDoc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG

/************************************************************************
 *  AirspaceDoc.cpp				operator=								*
 ************************************************************************/
const CAirspaceDoc& CAirspaceDoc::operator=(const CAirspaceDoc& AspDoc)
{
	memcpy (&m_Header, &(AspDoc.m_Header), sizeof(AIRSPACEHEADER)); 
	m_bConverted = AspDoc.m_bConverted;
	m_szExt = AspDoc.m_szExt;

	m_dMinLon1 = AspDoc.m_dMaxLat1;		// top left edge
	m_dMinLon1 = AspDoc.m_dMinLon1;
	m_dMinLat1 = AspDoc.m_dMinLat1;		// bottom right edge
	m_dMaxLon1 = AspDoc.m_dMaxLon1;

	DeleteArrayOf(&m_Airspaces);
	
	int i;
	for (i=0; i<AspDoc.m_Airspaces.GetSize(); i++)
	{
		CAirspace* ptAirspaceOld;
		ptAirspaceOld = (CAirspace*)(AspDoc.m_Airspaces.GetAt(i));
		if (ptAirspaceOld != NULL)
		{					// create memory for copied elements
			CAirspace* ptAirspaceNew = 
				new CAirspace(*ptAirspaceOld);
			if (ptAirspaceNew != NULL)
				m_Airspaces.Add (ptAirspaceNew);    // add new element 
		}	  
	}

	*m_ptSegmentDoc = *AspDoc.m_ptSegmentDoc;

	CString szTitle = AspDoc.GetTitle();
	SetTitle(szTitle);
	return *this;
}

/************************************************************************
 *  AirspaceDoc.cpp  		 G e t A c t R e g i o n N a m e 			*
 ************************************************************************/
CString CAirspaceDoc::GetActRegionName ()
{
	CString szRegion;
	short nDot;
	CString szNameExt = this->GetTitle();

	nDot = szNameExt.Find ('.');
	if (nDot >= 0)
		 szRegion = szNameExt.Left(nDot);			// for Windows NT only!!!
	else szRegion = szNameExt;

	return szRegion;
}

/************************************************************************
 *  AirspaceDoc.cpp  		 T r y T o G e t A i r s p a c e P t r 		*
 ************************************************************************/
BOOL CAirspaceDoc::TryToGetAirspacePtr(CAirspace** ptptAirspace, CString szIdentifier)
{
	BOOL bAvailable = FALSE;
	
	if (this->GetAirspacePtr(szIdentifier, ptptAirspace))
	{	
		bAvailable = TRUE;
	}
	else
	{			// new airspace, create new entry
		*ptptAirspace = new CAirspace(szIdentifier);
		if (*ptptAirspace != NULL)
		{
			if (this->AddSortedAirspacePtr(*ptptAirspace))
				bAvailable = TRUE;
		}
	}
	return bAvailable;
}


/************************************************************************
 *  AirspaceDoc.cpp			D e l e t e A r r a y O f 					*
 ************************************************************************/
void CAirspaceDoc::DeleteArrayOf (CObArray* ptArray)
{
int i, nEntryCnt;	
											
nEntryCnt = ptArray->GetSize();
for (i=0; i<nEntryCnt; i++)
	{
	CAirspace* ptEntry;
	if ((ptEntry = (CAirspace*)ptArray->GetAt(i)) != NULL)
		{
	//	TRACE ("Deleting Airspace %d\n", i);
		delete ptEntry;				// delete original element
		}
	}
ptArray->RemoveAll();
}  

/************************************************************************
 *  AirspaceDoc.cpp		D e l e t e C o n t e n t s						*
 *  Called from: OnFileNew and OnFileOpen								*
 ************************************************************************/
void CAirspaceDoc::DeleteContents()
{
this->InitHeader();
this->DeleteArrayOf (&m_Airspaces);

m_ptSegmentDoc->DeleteContents();
}


/************************************************************************
 *  AirspaceDoc.cpp    	 	S e t A n t i p o d			 				*
 ************************************************************************/
void CAirspaceDoc::SetAntipod(BOOL bAntipod)
{   
	SetFlagBit (&m_Header.cFlags, ASD_ANTIPOD, bAntipod);
}

/************************************************************************
 *  AirspaceDoc.cpp    		I s A n t i p o d		 					*
 ************************************************************************/
BOOL CAirspaceDoc::IsAntipod(void)
{ 
	BOOL bAntipod = ((m_Header.cFlags & ASD_ANTIPOD) == ASD_ANTIPOD);
	return bAntipod;
}

/************************************************************************
 *  AirspaceDoc.cpp		O n O p e n D o c u m e n t						*
 ************************************************************************/
BOOL CAirspaceDoc::OnOpenDocument(LPCTSTR pszPathName)
{
	BOOL bOpened = FALSE;
	this->SetPathName (pszPathName);

	bOpened = CDocument::OnOpenDocument(pszPathName); 

	if (bOpened)
	{
		if (m_bConverted)
		{
			if (this->OnSaveDocument (pszPathName, FALSE))		// don't confirm
			{
			m_bConverted = FALSE;
//			if (ptInit->IsSaveSignal())
//				MessageBeep(MB_OK);	
			}
		}
	}

	if (m_bOpenSubDocuments)
	{
		// even if main document isn't present at the moment, create new child doc!!

							// separate path from fullPath
		CString szFullPath = this->GetPathName();
		short	nNameIndex = szFullPath.ReverseFind('\\') + 1;
		CString szPath = szFullPath.Left (nNameIndex);  /* "D:\Code\Cpp_Win\PF40\"	*/

		CString szRegionName = this->GetActRegionName();

		m_ptSegmentDoc->OnNewDocument(); 			// deletes old structures   
		CString szRegion = szRegionName + m_ptSegmentDoc->GetExt();
		m_ptSegmentDoc->SetTitle(szRegion);
		m_ptSegmentDoc->OnOpenDocument (szPath + m_ptSegmentDoc->GetTitle());
	}

	m_bOpenSubDocuments = TRUE;

	return bOpened;
}


/************************************************************************
 *  AirspaceDoc.cpp		O n S a v e D o c u m e n t						*
 ************************************************************************/
BOOL CAirspaceDoc::OnSaveDocument(const char* pszPathName, BOOL bConfirm)
{
	BOOL bSaved = FALSE;
	BOOL bDoSave= FALSE;
	this->SetPathName (pszPathName);

	if (bConfirm)
	{
		CString szFileExt;		// ask only once:

		if(m_ptSegmentDoc!=NULL) 
		{						// ask to save airspace segment doc
			if(m_ptSegmentDoc->IsModified())
				szFileExt = m_ptSegmentDoc->GetTitle();
		}

		if (szFileExt.IsEmpty())
		{						// ask to save airspace doc
			if (this->IsModified())
				szFileExt = this->GetTitle();
		}

		if (!szFileExt.IsEmpty())
		{				// only ask, if something was changed!!
			CString szText;
			szText.Format(IDF_SAVE_FILE, (LPCTSTR)szFileExt);
			bDoSave = (AfxMessageBox ((LPCSTR)szText, MB_YESNO) == IDYES);
		}
	}
	else
	{
		bDoSave = TRUE;
	}

	if (bDoSave)
	{
	//	if (bDemo)
	//	{
	//		AfxMessageBox (IDS_SAVE_DISABLED);
	//		return bSaved;
	//	}
	//	else
		{
			if (m_ptSegmentDoc!=NULL) // && segments changed
				if (!this->IsSegmentDataValid())
					this->UpdateSegmentBorderAndIndex(m_ptSegmentDoc);

			bSaved = CDocument::OnSaveDocument(pszPathName); 



										// separate path from fullPath
			CString szFullPath = this->GetPathName();
			short	nNameIndex = szFullPath.ReverseFind('\\') + 1;
			CString szPath = szFullPath.Left (nNameIndex);  /* "D:\Code\Cpp_Win\PF40\"	*/

			CString szRegionName = this->GetActRegionName();
			CString szTitle = szRegionName + m_ptSegmentDoc->GetExt();
			m_ptSegmentDoc->SetTitle(szTitle);
		
			if (bSaved && m_ptSegmentDoc!=NULL) // && segments changed
				m_ptSegmentDoc->OnSaveDocument (szPath + m_ptSegmentDoc->GetTitle(), FALSE); 

		}
	}        
	else
	{   
		if (m_ptSegmentDoc!=NULL)
			m_ptSegmentDoc->SetModifiedFlag(FALSE);
		this->SetModifiedFlag (FALSE);
	}

return bSaved;
}


/************************************************************************
 *  AirspaceDoc.cpp 			C o m p a r e							*
 ************************************************************************/
int CAirspaceDoc::Compare(CAirspace* ptTestEntry, CAirspace* ptNewEntry)
{
        								// get new name to insert 
	CString	szNewName = ptNewEntry->GetIdentifier();
      
    int iCmp = Compare(ptTestEntry, szNewName);

	return iCmp;
}

/************************************************************************
 *  AirspaceDoc.cpp 			C o m p a r e							*
 ************************************************************************/
int CAirspaceDoc::Compare(CAirspace* ptTestEntry, CString szIdentifier)
{
    CString szTestName = ptTestEntry->GetIdentifier();
    int iCmp = szIdentifier.CompareNoCase ((LPCTSTR)szTestName);

	return iCmp;
}

/************************************************************************
 *  AirspaceDoc.cpp 		 G e t S o r t e d I n d e x 				*
 ************************************************************************/
short CAirspaceDoc::GetSortedIndex (CObArray& Airspaces, CAirspace* ptAirspace)
{
BOOL	bInsAfter;  
long	IndMin, IndMax, i, OldTestInd;

IndMin  = 0;
IndMax  = Airspaces.GetSize();		/* legal ind: 0...cnt-1		*/
i = (IndMin + IndMax) / 2;				/* i always < cnt !!	*/

if (Airspaces.GetSize() > 0)
   {
   do	{  
        CAirspace* ptTestEntry = (CAirspace*)m_Airspaces.GetAt((int)i);
        
		int iCmp = Compare(ptTestEntry, ptAirspace);
	    if (iCmp >=0)	bInsAfter = TRUE;
	    		else	bInsAfter = FALSE;

		if (bInsAfter)	IndMin = i;
				else	IndMax = i;
				
		OldTestInd = i;
		i = (IndMin + IndMax) / 2;
	
		} while (i != OldTestInd);

   if (bInsAfter) i++;
   }                     
   
return (short)i;
}
 
/************************************************************************
 *  AirspaceDoc.cpp	  		 I d e n t i f i e r T o I n d e x 			*
 *	Finds index of Airspace with specified identifer					*
 *	This function works only, if Airspaces are sorted by identifier!!!	*
 ************************************************************************/
short CAirspaceDoc::IdentifierToIndex (CString szIdentifier)
{
	short	nIndex = -1;
	long	IndMin, IndMax, i, OldTestInd;

	IndMin  = 0;
	IndMax  = m_Airspaces.GetSize();		/* legal ind: 0...cnt-1		*/
	i = (IndMin + IndMax) / 2;				/* i always < cnt !!		*/

	if (m_Airspaces.GetSize() > 0)
	{
		do	
		{  
			CAirspace* ptTestEntry = (CAirspace*)m_Airspaces.GetAt((int)i);

			int iCmp = Compare(ptTestEntry, szIdentifier);
			if (iCmp >=0)	IndMin = i;
	    			else	IndMax = i;

			if (iCmp == 0)
				nIndex = (short)i;					// store found index

			OldTestInd = i;
			i = (IndMin + IndMax) / 2;
		
		} while ((i != OldTestInd) && (nIndex == -1));

   }                     
 
return nIndex;
}

/************************************************************************
 *  AirspaceDoc.cpp		A d d S o r t e d A i r s p a c e P t r			*
 ************************************************************************/
BOOL CAirspaceDoc::AddSortedAirspacePtr (CAirspace* ptAirspace)
{
	BOOL bAdded = FALSE;
	int nOldCnt = m_Airspaces.GetSize();
	
	short nIndex = this->GetSortedIndex (m_Airspaces, ptAirspace);
	m_Airspaces.InsertAt(nIndex, ptAirspace);	
	bAdded = (nOldCnt < m_Airspaces.GetSize());
	this->SetAirspaceCnt(m_Airspaces.GetSize());

	return bAdded;
}

/************************************************************************
 *  AirspaceDoc.cpp		G e t A i r s p a c e P t r						*
 *  Purpose: returns pointer to specified Airspace						*
 ************************************************************************/
BOOL CAirspaceDoc::GetAirspacePtr (short nIndex, CAirspace** ptptAirspace)
{
	BOOL bDone = FALSE;
	if (nIndex >= 0 && nIndex < m_Airspaces.GetSize())
	{
		CAirspace* ptAirspace=NULL;
		ptAirspace = (CAirspace*)m_Airspaces.GetAt(nIndex);
		if (ptAirspace != NULL)
		{
			*ptptAirspace = ptAirspace;
			bDone = TRUE;
		}
	}

	return bDone;
}


/************************************************************************
 *  AirspaceDoc.cpp			G e t A i r s p a c e P t r					*
 ************************************************************************/
BOOL CAirspaceDoc::GetAirspacePtr (CString szIdentifier, CAirspace** ptptAirspace)
{	
	BOOL bDone = FALSE;
	short nIndex = -1;
	*ptptAirspace = NULL;

	nIndex = this->IdentifierToIndex(szIdentifier);
	if (nIndex >= 0)
	{
		bDone = this->GetAirspacePtr(nIndex, ptptAirspace);

	}
	else
	{
	//	AfxMessageBox ("CAirspaceDoc::GetAirspacePtr bad identifier: " + szIdentifier);
		bDone = FALSE;
	}
	return bDone;
}


/************************************************************************
 *  AirspaceDoc.cpp			D e l e t e A i r s p a c e P t r			*
 *  Purpose: deletes an Airspace entry									*
 *		don't forget to delete the corresponding AirspaceSegments		*
 ************************************************************************/
BOOL CAirspaceDoc::DeleteAirspacePtr (short nIndex)
{
	BOOL bDone = FALSE;

	CAirspace* ptAirspace = NULL;
	if (this->GetAirspacePtr(nIndex, &ptAirspace))
	{
		delete ptAirspace;
		m_Airspaces.RemoveAt(nIndex);
		bDone = TRUE;
	}

	return bDone;
}

/************************************************************************
 *  AirspaceDoc.cpp			D e l e t e A i r s p a c e P t r			*
 *  Purpose: deletes an Airspace entry, specified by szIdentifier		*
 *		don't forget to delete the corresponding AirspaceSegments		*
 ************************************************************************/
BOOL CAirspaceDoc::DeleteAirspacePtr (CString szIdentifier)
{
	BOOL bDone = FALSE;

	short nIndex = this->IdentifierToIndex(szIdentifier);
	if (nIndex >= 0)
	{
		bDone = DeleteAirspacePtr(nIndex);
	}
	return bDone;
}

/****************************************************************************
 *	AirspaceDoc.cpp			G e t M a x S e g m e n t C e n t e r			*
 *  looks for longest linear segment which can be used to draw label		*
 ****************************************************************************/
CLatLon CAirspaceDoc::GetMaxSegmentCenter(CAirspace* ptAirspace, long lStartIndex, double* ptMaxSegCourse)
{
	CLatLon llMaxSegCenter;

	CString szIdentifier;
	short i = (short)lStartIndex;
	BOOL bSearch = (i>=0);		// segments available for current airspace

	double dMaxSegmentLen = 0;		// to find best segment for label
	CLatLon llMaxSegmentPt1;
	CLatLon llMaxSegmentPt2;
	double dMaxSegmentCourse=0;
	int		nMaxSegmentIndex = 0;
	short nSegmentCnt = 0;		
	
//	if (ptAirspace->GetDisplayName().CompareNoCase("FRANKFURT SCTR C")==0)
//		nSegmentCnt=0;

	while (bSearch)
	{      
		CAirspaceSegment* ptAirspaceSegment = NULL;
		
		if (m_ptSegmentDoc->GetAirspaceSegmentPtr (i, &ptAirspaceSegment))
		{
							// segment of same airspace identifier?
			if (i == lStartIndex)
			{
				szIdentifier = ptAirspaceSegment->GetIdentifier();
			}
			else
			{
				bSearch = (szIdentifier.Compare(ptAirspaceSegment->GetIdentifier()) == 0);
			}

			if (bSearch)
			{
				double dLat, dLon;
				
				if (ptAirspaceSegment->GetShape() == CAirspaceSegment::LINE ||
					ptAirspaceSegment->GetShape() == CAirspaceSegment::IRREGULAR ||
					ptAirspaceSegment->GetShape() == CAirspaceSegment::GREATCIRCLE)
				{
								// try to get all LP values
					dLat = ptAirspaceSegment->GetLat1();
					dLon = ptAirspaceSegment->GetLon1();

					if (dLat != NO_KOORD && dLon != NO_KOORD)
					{
						CLatLon AspPt1(dLat, dLon);

						dLat = ptAirspaceSegment->GetLat2();
						dLon = ptAirspaceSegment->GetLon2();
						CLatLon AspPt2(dLat, dLon);

						if (dLat != NO_KOORD && dLon != NO_KOORD)
						{
							if (i==lStartIndex)
							{			// initialize max segment params
								dMaxSegmentLen = AspPt1.LoxoDist(AspPt2, &dMaxSegmentCourse);
								llMaxSegmentPt1 = AspPt1;
								llMaxSegmentPt2 = AspPt2;
								nMaxSegmentIndex = nSegmentCnt;
							}
							else
							{		// test next segment
								double dCourse;
								double dDist = AspPt1.LoxoDist(AspPt2, &dCourse);	
								if (dDist > dMaxSegmentLen)
								{
									dMaxSegmentLen = dDist;
									dMaxSegmentCourse = dCourse;
									llMaxSegmentPt1 = AspPt1;
									llMaxSegmentPt2 = AspPt2;
									nMaxSegmentIndex = nSegmentCnt;
								}
							}

							nSegmentCnt++;
						}
					
					}
				} // LINE, IRREGULAR, GREATCIRCLE
	
			}

			i++;
		}
		else
		{
			bSearch = FALSE;
		}

		if (!bSearch)
		{				// last segment was checked
			if (dMaxSegmentLen > 0)
			{
			
				llMaxSegCenter=CLatLon::PointOnLoxoLegAtDist(dMaxSegmentLen/2, llMaxSegmentPt1, llMaxSegmentPt2);
				*ptMaxSegCourse = dMaxSegmentCourse;
			}
		}
	}


	return llMaxSegCenter;
}


//     IDS_SPA_VERS            "PF SPA 1.00  "

/************************************************************************
 *  AirspaceDoc.cpp  		 	 H e a d e r T o V e r s 				*
 ************************************************************************/
short CAirspaceDoc::HeaderToVers (char* ptHeader)
{  
short nVersion = 0;
CString szTestVers; 

szTestVers.LoadString (IDS_SPA_VERS);
if (szTestVers.Compare (ptHeader) == 0)
	nVersion = ACT_SPA_VERS;

return nVersion;
} 
     
/************************************************************************
 *  AirspaceDoc.cpp  	 		V e r s T o H e a d e r 				*
 ************************************************************************/
void CAirspaceDoc::VersToHeader (LPSTR lpHeader, short nVersion)
{      
CString szVers;
switch (nVersion)
	{
	case ACT_SPA_VERS:	
			 szVers.LoadString (IDS_SPA_VERS);	break; 
	default: szVers.Empty();	break;
    }

strcpy (lpHeader, (LPCTSTR)szVers);
}      

/************************************************************************
 *  AirspaceDoc.cpp			I n i t H e a d e r							*
 ************************************************************************/
void CAirspaceDoc::InitHeader ()
{  
CString		szVers;

szVers.LoadString (IDS_SPA_VERS);
strcpy (m_Header.szVersion, (LPCTSTR)szVers);
m_Header.cFlags		= 0;
m_Header.cDummy		= 0;
m_Header.nActIndex	= -1;
m_Header.nEntryCnt	= 0; 

SetSegmentDataValid(FALSE);
}

/************************************************************************
 *  AirspaceDoc.cpp    	 		S e t F l a g B i t						*
 ************************************************************************/
void CAirspaceDoc::SetFlagBit (unsigned char* lpFlag, short BitMask, BOOL bSet)
{					/* BitMask: 1, 2, 4, 8, 16, ...	*/
if (bSet)   *lpFlag = *lpFlag | (unsigned char)BitMask;
    else    *lpFlag = *lpFlag & (unsigned char)(~BitMask);	/* ALT 126: ~	*/
}

/************************************************************************
 *  AirspaceDoc.cpp    	S e t S e g m e n t D a t a V a l i d			*
 ************************************************************************/
void CAirspaceDoc::SetSegmentDataValid(BOOL bValid)
{   
SetFlagBit (&m_Header.cFlags, ASP_SEGMENT_DATA_VALID, bValid);
}

/************************************************************************
 *  AirspaceDoc.cpp    	I s S e g m e n t D a t a V a l i d 			*
 ************************************************************************/
BOOL CAirspaceDoc::IsSegmentDataValid(void)
{ 
BOOL bValid = ((m_Header.cFlags & ASP_SEGMENT_DATA_VALID) == ASP_SEGMENT_DATA_VALID);
return bValid;
}

/************************************************************************
 *  AirspaceDoc.cpp		O n I n p u t D i r e c t i o n F o u n d		*
 *  checks for antipod airspace and expands airpace border				*
 *  uses dLastAirspaceMinLon to check for airspaces around lon 180		*
 ************************************************************************/
void CAirspaceDoc::OnInputDirectionFound(CAirspace* ptAirspace, BOOL* ptFirstAirspace,
									double dLastAirspaceMinLon,
									short nSegmentCnt, int nInpDir)
{
					// save input direction of last airspace
	ptAirspace->SetInputDirection(nInpDir);
					// save number of segments of last airspace
	ptAirspace->SetSegmentCnt(nSegmentCnt);




				// check if airspace does cross lon 180 to set antipod flag of airspace
				// create temporary polygon to test for a n t i p o d  koords
	CPolygon* ptPoly = CreatePolygonFor(ptAirspace);	// real lon. For Antipod, left > right!!
	if (ptPoly != NULL)
	{
		BOOL bAntipod = FALSE;
		double dLon1 = NO_KOORD;
		double dLon2 = NO_KOORD;
		int i;
		for (i=0; i<ptPoly->GetSize() && !bAntipod; i++)
		{
			CVektor* ptV = ptPoly->GetVektor(i);
			dLon2 = ptV->X();
		
			if (i>0)
			{
				if ((dLon1>0 && dLon2<0) || (dLon1<0 && dLon2>0))	// one lon West, one lon East
				{			// may be antipod
					bAntipod = ((fabs(dLon1) + fabs(dLon2)) > 180);	// around 180 ?
				}
			}

			dLon1 = dLon2;
		}


		ptAirspace->SetAntipod(bAntipod);		// store result in airspace


		BOOL bInit = TRUE;				// use polygon to get min max border
		for (i=0; i<ptPoly->GetSize(); i++)
		{
			CVektor* ptV = ptPoly->GetVektor(i);		// real lon. For Antipod, left > right!!
			double dLon = ptV->X();
			double dLat = ptV->Y();
			
			ptAirspace->ExpandBorder(bInit, dLat, dLon);
			bInit = FALSE;
		}

		delete ptPoly;
	}


				// check for airspaces around lon 180 to set antipod flag of document
	if (*ptFirstAirspace)
	{
		SetAntipod (ptAirspace->IsAntipod());
		*ptFirstAirspace = FALSE;
	}
	else
	{	
		if (!this->IsAntipod())
		{
			BOOL bAntipod = FALSE;

			if (ptAirspace->IsAntipod())
			{
				bAntipod = TRUE;
			}
			else
			{	// compare min longitude border values of last two airspaces
				double dLon1 = dLastAirspaceMinLon;
				double dLon2 = ptAirspace->GetMinLon();

				if ((dLon1>0 && dLon2<0) || (dLon1<0 && dLon2>0))	// one lon West, one lon East
				{			// may be antipod
					bAntipod = ((fabs(dLon1) + fabs(dLon2)) > 180);	// around 180 ?
				}
			}

			if (bAntipod)
				SetAntipod(bAntipod);
		}
	}
}

/************************************************************************
 *  AirspaceDoc.cpp			E x p a n d B o r d e r						*
 *  SetAntipod must be called before!!!									*
 *  expands border around airspaces covered by this document			*
 *  for an antipod d o c u m e n t ,									*
 *		lon of all west airspaces will be increased by 360				*
 *		so, m_dMaxLon1 may be > 180										*
 ************************************************************************/
void CAirspaceDoc::ExpandBorder(CAirspace* ptAirspace, 
									 BOOL* ptFirstAirspaceOfDoc)
{
	double dAspLatMax = ptAirspace->GetMaxLat();
	double dAspLonMax = ptAirspace->GetMaxLon();
	double dAspLatMin = ptAirspace->GetMinLat();
	double dAspLonMin = ptAirspace->GetMinLon();

	if (dAspLatMax != NO_KOORD && dAspLonMax != NO_KOORD &&
		dAspLatMin != NO_KOORD && dAspLonMin != NO_KOORD)
	{
		if (this->IsAntipod())
		{
			if (dAspLonMin < 0)
			{
				dAspLonMin += 360;
				dAspLonMax += 360;
			}
		}

		if (*ptFirstAirspaceOfDoc)
		{
			m_dMaxLat1 = dAspLatMax;		// top left edge
			m_dMinLon1 = dAspLonMin;
			m_dMinLat1 = dAspLatMin;		// bottom right edge
			m_dMaxLon1 = dAspLonMax;

			*ptFirstAirspaceOfDoc = FALSE;
		}
		else
		{
			if (m_dMaxLat1 < dAspLatMax)
				m_dMaxLat1 = dAspLatMax;

			if (m_dMinLon1 > dAspLonMin)
				m_dMinLon1 = dAspLonMin;

			if (m_dMinLat1 > dAspLatMin)
				m_dMinLat1 = dAspLatMin;

			if (m_dMaxLon1 < dAspLonMax)
				m_dMaxLon1 = dAspLonMax;
		}
	}
}


/************************************************************************
 *  AirspaceDoc.cpp		UpdateSegmentBorderAndIndex						*
 *  loops through all segments and adjusts the corresponding airspaces	*
 *  Call this AFTER you have modified a number of AirspaceSegments		*
 *  but BEFORE you use the border or segment index of CAirspace			*
 ************************************************************************/
void CAirspaceDoc::UpdateSegmentBorderAndIndex(CAirspaceSegmentDoc* ptAspSegDoc)
{
	short nSegmentIndex;
	BOOL bIsFirstPolyPointOfAirspace = FALSE;
	BOOL nSegmentCnt=0;
	CString szLastAirspaceID = "";

	if (!ptAspSegDoc->IsModified())
		return;

	CAirspace* ptAirspace = NULL;		// the current airspace object
	BOOL	bAspAvailable = FALSE;
	CPolygon Polygon;

	int i=0;				// first, reset FirstSegmentIndex of all airspaces
	for (i=0; i<m_Airspaces.GetSize(); i++)
	{
		CAirspace* ptAirspace;
		ptAirspace = (CAirspace*)m_Airspaces.GetAt(i);
		if (ptAirspace != NULL)
		{
			ptAirspace->ResetBorderAndIndex(); 
		}
	}

	BOOL bFirstAirspace = TRUE;
	double dLastAirspaceMinLon = NO_KOORD;


								// now, get new FirstSegmentIndex and input direction for each airspace,
								// which has segments defined
	for (nSegmentIndex=0; nSegmentIndex<ptAspSegDoc->GetAirspaceSegmentCnt(); nSegmentIndex++)
	{						// for all airspace segments:
		CAirspaceSegment* ptAirspaceSegment;
		if (ptAspSegDoc->GetAirspaceSegmentPtr(nSegmentIndex, &ptAirspaceSegment))
		{
			CString szAirspaceID = ptAirspaceSegment->GetIdentifier();

			if (szAirspaceID.Compare(szLastAirspaceID) != 0)
			{						// new airspace identifier found		
				if (ptAirspace != NULL)
				{	
					this->OnInputDirectionFound(ptAirspace, &bFirstAirspace, dLastAirspaceMinLon,
						nSegmentCnt, Polygon.GetInputDirection());

					dLastAirspaceMinLon = ptAirspace->GetMinLon();
					Polygon.Reset();
					nSegmentCnt=0;
				}

				bIsFirstPolyPointOfAirspace = TRUE;
				szLastAirspaceID = szAirspaceID;
				bAspAvailable = FALSE;	// we need a new airspace object

									// try to get the airspace from doc
				if (this->GetAirspacePtr(szAirspaceID, &ptAirspace))
				{
					bAspAvailable = TRUE;
				}
				else
				{									// no airspace for this segment found
					ptAirspace = new CAirspace();	// create a new airspace
					if (ptAirspace != NULL)
					{
						ptAirspace->SetIdentifier(szAirspaceID);

						// if required, initialize new airspace here...

													// add new airspace to doc
						bAspAvailable = this->AddSortedAirspacePtr(ptAirspace);
					}
				}
			}


			if (bAspAvailable)
			{		// airspace available
						// set first SegmentIndex, stored in CAirspace
						// to quickly find the segments of an airspace
				if (bIsFirstPolyPointOfAirspace)
				{
					ptAirspace->SetFirstSegmentIndex(nSegmentIndex);
					bIsFirstPolyPointOfAirspace = FALSE;
				}

				nSegmentCnt++;

				CAirspaceSegment::SHAPE shape = ptAirspaceSegment->GetShape();

				
				if (shape == CAirspaceSegment::CIRCLE)
				{		// this segment sets whole border and polygon!!
					double dLat = ptAirspaceSegment->GetGeodeticLat0();
					double dLon = ptAirspaceSegment->GetGeodeticLon0();
					double dRad = ptAirspaceSegment->GetRadius();

					if (dLat != NO_KOORD && dLon != NO_KOORD)
					{
											// calc rect around circle
						CLatLon llCenter(dLat, dLon);
						CLatLon llTopLeft, llBottomRight;

						double dDiagonal = dRad * 1.4142;	// to get bounding rect
						llTopLeft.FromRelKoords(llCenter, 315, dDiagonal);
						llBottomRight.FromRelKoords(llCenter, 135, dDiagonal);

						Polygon.AddVektor(llTopLeft.GetLon(), llTopLeft.GetLat());
						Polygon.AddVektor(llBottomRight.GetLon(), llTopLeft.GetLat());
						Polygon.AddVektor(llBottomRight.GetLon(), llBottomRight.GetLat());
						Polygon.AddVektor(llTopLeft.GetLon(), llBottomRight.GetLat());
					}
				}

				if ((shape == CAirspaceSegment::GREATCIRCLE) ||
					(shape == CAirspaceSegment::IRREGULAR) ||
				    (shape == CAirspaceSegment::LINE))
				{
					double dLat = ptAirspaceSegment->GetLat1();
					double dLon = ptAirspaceSegment->GetLon1();

					if (dLat != NO_KOORD && dLon != NO_KOORD)
					{
						Polygon.AddVektor(dLon, dLat);
					}
				}

				if ((shape == CAirspaceSegment::ARCLEFT)||
					(shape == CAirspaceSegment::ARCRIGHT))
				{			// for arc, copy arc start point into polygon
					double dLat = ptAirspaceSegment->GetLat1();
					double dLon = ptAirspaceSegment->GetLon1();

					if (dLat != NO_KOORD && dLon != NO_KOORD)
					{
						Polygon.AddVektor(dLon, dLat);
					}
				}
			}
		} // good airspace segment
	}  // check next segment index
	

	if (ptAirspace != NULL)
	{					// update data of last airspace
		this->OnInputDirectionFound(ptAirspace, &bFirstAirspace, dLastAirspaceMinLon,
			nSegmentCnt, Polygon.GetInputDirection());
	}

		// now we know, whether this airspace doc contains airspaces around lon 180.
	bFirstAirspace = TRUE;
	for (i=0; i<m_Airspaces.GetSize(); i++)
	{
		CAirspace* ptAirspace;
		ptAirspace = (CAirspace*)m_Airspaces.GetAt(i);
		if (ptAirspace != NULL)
		{
			this->ExpandBorder(ptAirspace, &bFirstAirspace);
		}
	}
		

	SetSegmentDataValid(TRUE);
}


/************************************************************************
 *  AirspaceDoc.cpp				C i r c l e T o P o l y					*
 *  calculates circle polygon points via CLatLon::FromRelKoords			*
 ************************************************************************/
void CAirspaceDoc::CircleToPoly(CPolygon* ptPoly,
						 double dCenterX, double dCenterY, double dRad_NM)
{
	double dArcAngle = 360;
	short nSteps = (short)(dArcAngle * 12/180);
	
	CLatLon llCenter(dCenterY, dCenterX);

	for (int i=0; i<nSteps; i++)
	{
		double dPolyAngle = dArcAngle * i/nSteps;
		short nRWK = (short)(dPolyAngle + 0.5);

		CLatLon llArcPoint;
		llArcPoint.FromRelKoords(llCenter, nRWK, dRad_NM);
		ptPoly->AddVektor(llArcPoint.GetLon(), llArcPoint.GetLat());
	}
}

/************************************************************************
 *  AirspaceDoc.cpp					A r c T o P o l y					*
 *  calculates arc polygon points via CLatLon::FromRelKoords			*
 *  Don't use CVektor::ArcToPoly here!! That func works linear!!		*
 ************************************************************************/
void CAirspaceDoc::ArcToPoly(CPolygon* ptPoly, double dStartX, double dStartY, 
						 double dEndX, double dEndY,
						 double dCenterX, double dCenterY, double dRad_NM,
						 BOOL bClockWiseArc)
{
	CVektor vCenter(dCenterX, dCenterY);
	CVektor vStart(dStartX, dStartY);
	CVektor vEend(dEndX, dEndY);

	CVektor vRNorth(0,1,0);

	CVektor vRstart = vStart - vCenter;
	CVektor vRend = vEend - vCenter;
	
	double dRad1 = vRNorth.VektorAngle(vRstart);
	if (vRstart.x < 0) dRad1=2*pi-dRad1;
	double dA1 = 180*dRad1/pi;

	double dRad2 = vRNorth.VektorAngle(vRend);
	if (vRend.x < 0) dRad2=2*pi-dRad2;
	double dA2 = 180*dRad2/pi;


	if (dA2 < dA1)
		dA2 += 360;

	double dArcAngle = dA2-dA1;
	if (!bClockWiseArc)
		dArcAngle = 360 - dArcAngle;	// get polygon points of other side

	CLatLon llCenter(dCenterY, dCenterX);
	ptPoly->AddVektor(vStart);

	double dRotateAngle, dPolyAngle;
	short nRWK;
	short nSteps = (short)(dArcAngle * 12/180);
	for (int i=1; i<nSteps; i++)
	{
		dRotateAngle = dArcAngle * i/nSteps;
		if (bClockWiseArc)	dPolyAngle = dA1 + dRotateAngle;
			else			dPolyAngle = dA1 - dRotateAngle;

		nRWK = (short)(dPolyAngle + 0.5);

		CLatLon llArcPoint;
		llArcPoint.FromRelKoords(llCenter, nRWK, dRad_NM);
		ptPoly->AddVektor(llArcPoint.GetLon(), llArcPoint.GetLat());
	}
}

/************************************************************************
 *  AirspaceDoc.cpp			C r e a t e P o l y g o n F o r				*
 *  reads segments of ptAirspaces and stores them in ptPoly				*
 ************************************************************************/
CPolygon* CAirspaceDoc::CreatePolygonFor(CAirspace* ptAirspace)
{
	CPolygon* ptPoly = NULL;

			// use segment doc to generate polygon
	CAirspaceSegmentDoc* ptAspSegDoc = this->GetSegmentDocPtr();
	int nStartIndex = (int)ptAirspace->GetFirstSegmentIndex();

	int inpDir = ptAirspace->GetInputDirection();

	short i = (short)nStartIndex;

	int nCnt = ptAirspace->GetSegmentCnt();
	if (nCnt > 0)
	{
		ptPoly = new CPolygon;

		for (i=nStartIndex; i<(nStartIndex + nCnt); i++)
		{
			CAirspaceSegment* ptAirspaceSegment = NULL;
			if (ptAspSegDoc->GetAirspaceSegmentPtr (i, &ptAirspaceSegment))
			{		// add a polygon segment
				double dLat1, dLon1, dLat2, dLon2, dLatCenter, dLonCenter, dRad, dRad_NM;
				BOOL bClockWiseArc;

				CAirspaceSegment::SHAPE shape = ptAirspaceSegment->GetShape();
				switch (shape)
				{
				case CAirspaceSegment::CIRCLE:
					dLatCenter = ptAirspaceSegment->GetGeodeticLat0();
					dLonCenter = ptAirspaceSegment->GetGeodeticLon0();
					dRad = ptAirspaceSegment->GetRadius();
					dRad_NM = ptDim->ConvertDist(dRad, ptAirspaceSegment->GetDistDim(), DIM_NM);
					this->CircleToPoly(ptPoly, dLonCenter, dLatCenter, dRad_NM);
					break;
				case CAirspaceSegment::LINE:
				case CAirspaceSegment::IRREGULAR:
				case CAirspaceSegment::GREATCIRCLE:
					dLat1 = ptAirspaceSegment->GetLat1();
					dLon1 = ptAirspaceSegment->GetLon1();

							// store start point only
					ptPoly->AddVektor(dLon1, dLat1);
					break;

				case CAirspaceSegment::ARCLEFT:		// L: COUNTERCLOCKWISE ARC
					bClockWiseArc = FALSE;
					dLat1 = ptAirspaceSegment->GetLat1();
					dLon1 = ptAirspaceSegment->GetLon1();
					dLat2 = ptAirspaceSegment->GetLat2();
					dLon2 = ptAirspaceSegment->GetLon2();
					dLatCenter = ptAirspaceSegment->GetGeodeticLat0();
					dLonCenter = ptAirspaceSegment->GetGeodeticLon0();
					dRad = ptAirspaceSegment->GetRadius();
					dRad_NM = ptDim->ConvertDist(dRad, ptAirspaceSegment->GetDistDim(), DIM_NM);

					// stores first point and calculated arc points, but NOT last point
					if (inpDir == CAirspace::COUNTERCLOCKWISE)
						this->ArcToPoly(ptPoly, dLon1, dLat1, dLon2, dLat2, 
							dLonCenter, dLatCenter, dRad_NM, bClockWiseArc);

					if (inpDir == CAirspace::CLOCKWISE)
						this->ArcToPoly(ptPoly, dLon2, dLat2, dLon1, dLat1, 
							dLonCenter, dLatCenter, dRad_NM, bClockWiseArc);
					break;	

				case CAirspaceSegment::ARCRIGHT:	// R: CLOCKWISE ARC
					bClockWiseArc = TRUE;
					dLat1 = ptAirspaceSegment->GetLat1();
					dLon1 = ptAirspaceSegment->GetLon1();
					dLat2 = ptAirspaceSegment->GetLat2();
					dLon2 = ptAirspaceSegment->GetLon2();

					dLatCenter = ptAirspaceSegment->GetGeodeticLat0();
					dLonCenter = ptAirspaceSegment->GetGeodeticLon0();
					dRad = ptAirspaceSegment->GetRadius();
					dRad_NM = ptDim->ConvertDist(dRad, ptAirspaceSegment->GetDistDim(), DIM_NM);

					// stores first point and calculated arc points, but NOT last point
					if (inpDir == CAirspace::CLOCKWISE)
						this->ArcToPoly(ptPoly, dLon1, dLat1, dLon2, dLat2, 
							dLonCenter, dLatCenter, dRad_NM, bClockWiseArc);
										
					if (inpDir == CAirspace::COUNTERCLOCKWISE)
						this->ArcToPoly(ptPoly, dLon2, dLat2, dLon1, dLat1, 
							dLonCenter, dLatCenter, dRad_NM, bClockWiseArc);

					break;

				default:
					break;
				}
			}
		}
	}

	return ptPoly;
}

/************************************************************************
 *  AirspaceDoc.cpp			G e t A i r s p a c e A t					*
 *  returns airspaces at given dLat/dLon								*
 ************************************************************************/
BOOL CAirspaceDoc::GetAirspaceAt(double dLat, double dLon, CAirspace** ptptAirspace, 
								 short* ptNextAirspaceIndex)
{
	BOOL bFound = FALSE;

	short nIndex = *ptNextAirspaceIndex;
	int i;
	for (i=nIndex; i<m_Airspaces.GetSize() && !bFound; i++)
	{
		CAirspace* ptAirspace=NULL;
		ptAirspace = (CAirspace*)m_Airspaces.GetAt(i);
		if (ptAirspace != NULL)
		{
			if (ptAirspace->BorderContains(dLat, dLon))
			{				// airspace is near dLat/dLon
 
				CPolygon* ptPoly = ptAirspace->GetPolyPtr();
				if (ptPoly == NULL)
				{
					ptPoly = CreatePolygonFor(ptAirspace);
					if (ptPoly != NULL)
					{
						if (ptAirspace->IsAntipod())
						{		// add 360 to west lon: all lon > 0
							for (int j=0; j<ptPoly->GetSize(); j++)
							{
								CVektor* ptV = ptPoly->GetVektor(j);
								double dLon = ptV->X();
								
								if (dLon < 0)
									dLon += 360;
								ptV->Set(dLon, ptV->Y());
							}
						}

						if (ptPoly->GetSize() > 2)
						{			// good polygon created, hold it in ptAirspace
							ptAirspace->SetPolyPtr(ptPoly);
						}
						else 
						{			// bad polygon, delete it
							delete ptPoly;
							ptPoly = NULL;
						}
					}
				}

				if (ptPoly != NULL)
				{
					// use polygon to test for dLat dLon...
					int inpDir = ptAirspace->GetInputDirection();
					if (ptPoly->IsInside(dLon, dLat, inpDir))
					{
						*ptptAirspace = ptAirspace;
						(*ptptAirspace)->SetRegionName(this->GetActRegionName());

						*ptNextAirspaceIndex = i+1;
						bFound = TRUE;
					}
				}
			}
		}
	}

	if (!bFound && i==m_Airspaces.GetSize())
	{		// last airspace of this region checked, change to next region
		*ptNextAirspaceIndex = 0;		// start with first airspace of next region
	}

	return bFound;
}




/************************************************************************
 *  AirspaceDoc.cpp			G e t A i r s p a c e I n R e c t			*
 *  returns airspaces at given rLatLon									*
 ************************************************************************/
BOOL CAirspaceDoc::GetAirspaceInRect(DRECT rLatLon, CAirspace** ptptAirspace, 
								 short* ptNextRegIndex, short* ptNextAirspaceIndex)
{
	BOOL bFound = FALSE;

	short nIndex = *ptNextAirspaceIndex;
  int i;
	for (i=nIndex; i<m_Airspaces.GetSize() && !bFound; i++)
	{
		CAirspace* ptAirspace=NULL;
		ptAirspace = (CAirspace*)m_Airspaces.GetAt(i);
		if (ptAirspace != NULL)
		{
	//		CString szAspName = ptAirspace->GetName();	// for debugging only
	//			if (ptAirspace->GetIdentifier().CompareNoCase("NZ00045") == 0)
	//				int s=1;
		
			if (ptAirspace->IsInRect(rLatLon))
			{
				*ptptAirspace = ptAirspace;
				(*ptptAirspace)->SetRegionName(this->GetActRegionName());

				*ptNextAirspaceIndex = i+1;
				bFound = TRUE;
			}
		}
	}

	if (!bFound && i==m_Airspaces.GetSize())
	{		// last airspace of this region checked, change to next region
		(*ptNextRegIndex)++;
		*ptNextAirspaceIndex = 0;		// start with first airspace of next region
	}

	return bFound;
}

/************************************************************************
 *  AirspaceDoc.cpp	  			 GetNewAirspaceID						*
 *  reads list of airspaces to determine the next free identifier		*
 *  for	specified country code szCC										*
 ************************************************************************/
long CAirspaceDoc::GetNewAirspaceID(CString szCC) 
{
	long lMaxID = -1;
	CAirspace* ptAirspace;
	short nCnt = this->GetAirspaceCnt();
	for (int i=0; i<nCnt; i++)
	{
		ptAirspace = NULL;
		if (this->GetAirspacePtr(i, &ptAirspace))
		{
			if(ptAirspace->GetCountryCode().CompareNoCase(szCC) == 0)
			{					// country code specified found
				long lIdentifier = ptAirspace->GetID();
				if (lMaxID == -1)
				{
					lMaxID = lIdentifier;
				}
				else
				{
					if (lIdentifier > lMaxID)
						lMaxID = lIdentifier;
				}
			}
		}
	}

	lMaxID++;

	return lMaxID;
}

/************************************************************************
 *  AirspaceDoc.cpp		C o p y U s e r A i r s p a c e s				*
 *  copies userdefined airspaces from ptSrcAspDoc to this doc			*
 *	called from UpdatDlg.cpp											*
 ************************************************************************/
short CAirspaceDoc::CopyUserAirspaces(CAirspaceDoc* ptSrcAspDoc)
{			
	short nAspCopied = 0;

	if(ptSrcAspDoc != NULL)
	{				
		CAirspaceSegmentDoc* ptSrcAspSegDoc = ptSrcAspDoc->GetSegmentDocPtr();
		if(ptSrcAspSegDoc != NULL)
		{

			for (int i=0; i<ptSrcAspDoc->GetAirspaceCnt(); i++)
			{							// for all source airspaces
				CAirspace* ptSrcAirspace=NULL;
				if (ptSrcAspDoc->GetAirspacePtr(i, &ptSrcAirspace))
				{
					if (ptSrcAirspace->IsUserDefined())
					{			// user defined airspace found, copy it

						CAirspace* ptAirspace = new CAirspace;
						if (ptAirspace != NULL)
						{
							*ptAirspace = *ptSrcAirspace;	// copy airspace

							CString szCountryCode = ptSrcAirspace->GetCountryCode();
							long lID = GetNewAirspaceID(szCountryCode);
							ptAirspace->SetID(lID);

								// store new segment in doc
							if (this->AddSortedAirspacePtr(ptAirspace))
							{
								this->SetModifiedFlag(TRUE);

								short nCnt = ptSrcAirspace->GetSegmentCnt();
								if (nCnt > 0)
								{
									short nStartIndex = (short)ptSrcAirspace->GetFirstSegmentIndex();
									CString szAspIdent = ptAirspace->GetIdentifier();

									short nSegCopied = m_ptSegmentDoc->CopyAirspaceSegments(szAspIdent, nStartIndex, nCnt, ptSrcAspSegDoc); 
									if (nSegCopied == nCnt)
										nAspCopied++;
								}
							} // ptAirspace added
						} // good ptAirspace created


					} // userdefined

				} // good ptSrcAirspace
			} // for all source airspaces
		}
	}
	return nAspCopied;
}


/************************************************************************
 *  AirspaceDoc.cpp  	 		S e r i a l i z e		 				*
 ************************************************************************/
void CAirspaceDoc::Serialize(CArchive& ar)
{
	short i, nVersion;

	if (ar.IsStoring())
	{
		// ZU ERLEDIGEN: Code zum Speichern hier einfgen
		nVersion = ACT_SPA_VERS;  
		VersToHeader ((LPSTR)&m_Header.szVersion, nVersion);	
		this->SetAirspaceCnt (m_Airspaces.GetSize());	// put elev cnt into header

		ar.Write (&m_Header, sizeof (AIRSPACEHEADER));   
		
		ar << m_dMaxLat1;		// top left edge
		ar << m_dMinLon1;
		ar << m_dMinLat1;		// bottom right edge
		ar << m_dMaxLon1;

		for (i=0; i<m_Airspaces.GetSize(); i++)
		{
			CAirspace* ptAirspace;
			ptAirspace = (CAirspace*)m_Airspaces.GetAt(i);
			if (ptAirspace != NULL)
				ptAirspace->Serialize (ar);  
		}      
		m_bConverted = FALSE;
	}
	else
	{
		m_bConverted = FALSE;

		// ZU ERLEDIGEN: Code zum Laden hier einfgen
		ar.Read (&m_Header, sizeof (AIRSPACEHEADER));
		nVersion = this->HeaderToVers (m_Header.szVersion);  


		ar >> m_dMaxLat1;		// top left edge
		ar >> m_dMinLon1;
		ar >> m_dMinLat1;		// bottom right edge
		ar >> m_dMaxLon1;

		short nCnt = this->GetAirspaceCnt ();
	
		for (i=0; i<nCnt; i++)
		{
			CAirspace* ptAirspace = new CAirspace; 
			if (ptAirspace != NULL)
			{
				ptAirspace->Serialize (ar, nVersion);
				m_Airspaces.Add (ptAirspace);       	// add new Element        
			}
		} 

		if (this->GetActIndex()==-1 && nCnt>0)
			this->SetActIndex(0);		// to select 1. entry in dialog list

	}
}

/////////////////////////////////////////////////////////////////////////////
// Befehle CAirspaceDoc 
