/************************************************************************
 *			 		 A i r s p a c e S e g m e n t . c p p				*
 ************************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "DimDoc.h"
#include "AirspaceSegment.h"

/* stores all parameters of an airspace segment. A segment is either	*/
/* a line, a circle or an arc											*/


/************************************************************************
 *  AirspaceSegment.cpp	  C A i r s p a c e S e g m e n t   CONSTRUCTOR	*
 ************************************************************************/
CAirspaceSegment::CAirspaceSegment ()
{		
	this->Init();
}
  
/************************************************************************
 *  AirspaceSegment.cpp	 C A i r s p a c e S e g m e n t   CONSTRUCTOR 	*
 ************************************************************************/
CAirspaceSegment::CAirspaceSegment (CString szIdentifier, long lSegNum)
{										
	this->Init();

				// e.g.: ED000027
	strncpy (m_szIdentifier, (LPCTSTR)szIdentifier, SIZEOF_ASP_IDENT);
	m_szIdentifier[SIZEOF_ASP_IDENT-1]=0;
	m_lSegmentNumber = lSegNum;
}  

/************************************************************************
 *  AirspaceSegment.cpp	 C A i r s p a c e S e g m e n t CONSTRUCTOR	*
 ************************************************************************/
CAirspaceSegment::CAirspaceSegment (CAirspaceSegment& SourceAirspaceSegment)
{ 
	*this = SourceAirspaceSegment;
}

/************************************************************************
 *  AirspaceSegment.cpp	 ~ C A i r s p a c e S e g m e n t   DESTRUCTOR	*
 ************************************************************************/
CAirspaceSegment::~CAirspaceSegment()
{

}

/************************************************************************
 *  AirspaceSegment.cpp				operator=							*
 ************************************************************************/
const CAirspaceSegment& CAirspaceSegment::operator=(const CAirspaceSegment& AirspaceSegment)
{
	memcpy (m_szIdentifier, AirspaceSegment.m_szIdentifier, SIZEOF_ASP_IDENT); 

	m_lSegmentNumber		= AirspaceSegment.m_lSegmentNumber;
	m_cShape				= AirspaceSegment.m_cShape;
	m_cDummy				= AirspaceSegment.m_cDummy;

	m_dLatitude1			= AirspaceSegment.m_dLatitude1;
	m_dLongitude1			= AirspaceSegment.m_dLongitude1;
	m_dLatitude2			= AirspaceSegment.m_dLatitude2;
	m_dLongitude2			= AirspaceSegment.m_dLongitude2;

	m_dGeodeticLatitude0	= AirspaceSegment.m_dGeodeticLatitude0;	
	m_dGeodeticLongitude0	= AirspaceSegment.m_dGeodeticLongitude0;
	m_dRadius1				= AirspaceSegment.m_dRadius1;	
	m_nDistDim				= AirspaceSegment.m_nDistDim;

	return *this;
}


/********************************************************************************
 *  AirspaceSegment.cpp				o p e r a t o r  ! =						*
 ********************************************************************************/
BOOL CAirspaceSegment::operator !=(const CAirspaceSegment& AirspaceSegment)
{
	double dEps = 0.001;
	if (strcmp(m_szIdentifier, AirspaceSegment.m_szIdentifier) != 0)
		return TRUE;


	if (m_lSegmentNumber		!= AirspaceSegment.m_lSegmentNumber)
		return TRUE;
	if (m_cShape				!= AirspaceSegment.m_cShape)
		return TRUE;
	if (m_cDummy				!= AirspaceSegment.m_cDummy)
		return TRUE;

	if (fabs(m_dLatitude1 - AirspaceSegment.m_dLatitude1) > dEps)
		return TRUE;
	if (fabs(m_dLongitude1 - AirspaceSegment.m_dLongitude1) > dEps)
		return TRUE;	
	if (fabs(m_dLatitude2 - AirspaceSegment.m_dLatitude2) > dEps)
		return TRUE;
	if (fabs(m_dLongitude2 - AirspaceSegment.m_dLongitude2) > dEps)
		return TRUE;
	
	if (fabs(m_dGeodeticLatitude0 - AirspaceSegment.m_dGeodeticLatitude0) > dEps)
		return TRUE;
	if (fabs(m_dGeodeticLongitude0 - AirspaceSegment.m_dGeodeticLongitude0) > dEps)
		return TRUE;
	if (m_dRadius1				!= AirspaceSegment.m_dRadius1)
		return TRUE;
	if (m_nDistDim				!= AirspaceSegment.m_nDistDim)
		return TRUE;

return FALSE;
}

/************************************************************************
 *  AirspaceSegment.cpp			     I n i t	 						*
 ************************************************************************/
void CAirspaceSegment::Init ()
{
	memset(m_szIdentifier, 0, SIZEOF_ASP_IDENT);
	m_lSegmentNumber = -1;
	m_cShape = CAirspaceSegment::LINE;
	m_cDummy = ' ';
	m_dLatitude1 = NO_KOORD;
	m_dLongitude1 = NO_KOORD;
	m_dLatitude2 = NO_KOORD;
	m_dLongitude2 = NO_KOORD;

	m_dGeodeticLatitude0 = NO_KOORD;
	m_dGeodeticLongitude0 = NO_KOORD;
	m_dRadius1 = NO_DIST;	 
	m_nDistDim = DIM_NM;

	m_lUndefined = -10000;
}

/************************************************************************
 *  AirspaceSegment.cpp			  S e t S h a p e 						*
 *  used to convert from DAFIF, see AirspaceDoc::OnParseDafifAirspace	*
 ************************************************************************/
void CAirspaceSegment::SetShape(char cShape)
{
	switch(cShape)
	{
	case 'A':
		m_cShape = CAirspaceSegment::POINT;			// POINT (WITHOUT RADIUS OR BEARING)
		break;

	case 'B':
		m_cShape = CAirspaceSegment::GREATCIRCLE;	// GREAT CIRCLE
		break;

	case 'C':
		m_cShape = CAirspaceSegment::CIRCLE;		// CIRCLE
		break;

	case 'G':
		m_cShape = CAirspaceSegment::IRREGULAR;		// GENERALIZED
		break;

	case 'H':
		m_cShape = CAirspaceSegment::LINE;			// RHUMB LINE
		break;

	case 'L':
		m_cShape = CAirspaceSegment::ARCLEFT;		// COUNTERCLOCKWISE ARC
		break;

	case 'R':
		m_cShape = CAirspaceSegment::ARCRIGHT;		// CLOCKWISE ARC
		break;

	}
}


/************************************************************************
 *  AirspaceSegment.cpp	 S e t I d e n t i f i e r						*
 ************************************************************************/
void CAirspaceSegment::SetIdentifier (CString szIdentifier)
{										
				// e.g.: ED000027
	strncpy (m_szIdentifier, (LPCTSTR)szIdentifier, SIZEOF_ASP_IDENT);
	m_szIdentifier[SIZEOF_ASP_IDENT-1]=0;
}  

/************************************************************************
 *  AirspaceSegment.cpp    		 G e t D i s p l a y N a m e			*
 ************************************************************************/
CString CAirspaceSegment::GetDisplayName()
{
	CString name;
	name = m_szIdentifier;
	return name;
}

/************************************************************************
 *  AirspaceSegment.cpp			     G e t L a t 1 						*
 ************************************************************************/
double CAirspaceSegment::GetLat1 ()
{
return m_dLatitude1;
}

/************************************************************************
 *  AirspaceSegment.cpp			    G e t L o n 1						*
 ************************************************************************/
double CAirspaceSegment::GetLon1 ()
{
return m_dLongitude1;
}     

/************************************************************************
 *  AirspaceSegment.cpp			     G e t L a t 2 						*
 ************************************************************************/
double CAirspaceSegment::GetLat2 ()
{
return m_dLatitude2;
}

/************************************************************************
 *  AirspaceSegment.cpp			    G e t L o n 2						*
 ************************************************************************/
double CAirspaceSegment::GetLon2 ()
{
return m_dLongitude2;
}     

/************************************************************************
 *  AirspaceSegment.cpp			    D r a w 							*
 ************************************************************************/
void CAirspaceSegment::Draw(CDC* pDC, long lX1, long lY1, long lX2, long lY2, 
							long lX0, long lY0, long lRadius)
{
	short err = 0;

	switch(m_cShape)
	{
	case CAirspaceSegment::POINT:			// POINT (WITHOUT RADIUS OR BEARING)
		break;

	case CAirspaceSegment::GREATCIRCLE:		// GREAT CIRCLE, like RHUMB LINE
		if (lX1 != m_lUndefined && lY1 != m_lUndefined && 
			lX2 != m_lUndefined && lY2 != m_lUndefined)
			DrawLine(pDC, lX1,lY1, lX2,lY2);
		else 
			err = 1;
		break;

	case CAirspaceSegment::CIRCLE:			// CIRCLE
		if (lX0 != m_lUndefined && lY0 != m_lUndefined && lRadius != m_lUndefined)
			DrawCircle(pDC, lX0,lY0, lRadius);
		else 
			err = 1;
		break;

	case CAirspaceSegment::IRREGULAR:		// GENERALIZED, like RHUMB LINE
		if (lX1 != m_lUndefined && lY1 != m_lUndefined && 
			lX2 != m_lUndefined && lY2 != m_lUndefined)
			DrawLine(pDC, lX1,lY1, lX2,lY2);
		else 
			err = 1;
		break;

	case CAirspaceSegment::LINE:			// RHUMB LINE
		if (lX1 != m_lUndefined && lY1 != m_lUndefined && 
			lX2 != m_lUndefined && lY2 != m_lUndefined)
			DrawLine(pDC, lX1,lY1, lX2,lY2);
		else 
			err = 1;
		break;

	case CAirspaceSegment::ARCLEFT:			// COUNTERCLOCKWISE ARC
		if (lX1 != m_lUndefined && lY1 != m_lUndefined && 
			lX2 != m_lUndefined && lY2 != m_lUndefined && 
			lX0 != m_lUndefined && lY0 != m_lUndefined && lRadius != m_lUndefined)
			DrawCounterClockwiseArc(pDC, lX1,lY1, lX2,lY2, lX0,lY0, lRadius);
		else 
			err = 1;
		break;

	case CAirspaceSegment::ARCRIGHT:		// CLOCKWISE ARC
		if (lX1 != m_lUndefined && lY1 != m_lUndefined && 
			lX2 != m_lUndefined && lY2 != m_lUndefined && 
			lX0 != m_lUndefined && lY0 != m_lUndefined && lRadius != m_lUndefined)
			DrawClockwiseArc(pDC, lX1,lY1, lX2,lY2, lX0,lY0, lRadius);
		else 
			err = 1;
		break;
	}
}

/************************************************************************
 *  AirspaceSegment.cpp			    D r a w L i n e						*
 ************************************************************************/
void CAirspaceSegment::DrawLine(CDC* pDC, long lX1, long lY1, long lX2, long lY2)
{
	pDC->MoveTo(lX1, lY1);
	pDC->LineTo(lX2, lY2);
}

/************************************************************************
 *  AirspaceSegment.cpp			D r a w C i r c l e				 		*
 ************************************************************************/
void CAirspaceSegment::DrawCircle(CDC* pDC, long lX0, long lY0, long lRadius)
{
	CRect rEllipse;

	rEllipse.SetRect (lX0-lRadius, lY0+lRadius, lX0+lRadius, lY0-lRadius);		// L, T, R, B

	CBrush* ptOldBrush = (CBrush*)pDC->SelectStockObject (NULL_BRUSH);
	pDC->Ellipse(rEllipse);
	pDC->SelectObject (ptOldBrush);

}

/************************************************************************
 *  AirspaceSegment.cpp			D r a w C l o c k w i s e A r c 		*
 ************************************************************************/
void CAirspaceSegment::DrawCounterClockwiseArc(CDC* pDC,
							long lX1, long lY1, long lX2, long lY2, 
							long lX0, long lY0, long lRadius)
{
	CRect rEllipse;

	rEllipse.SetRect (lX0-lRadius, lY0+lRadius, lX0+lRadius, lY0-lRadius);		// L, T, R, B
	CBrush* ptOldBrush = (CBrush*)pDC->SelectStockObject (NULL_BRUSH);

	CPoint pStart, pEnd;
	pStart.x = lX1;
	pStart.y = lY1;
	pEnd.x = lX2;
	pEnd.y = lY2;

	pDC->Arc(rEllipse, pStart, pEnd);
	pDC->SelectObject (ptOldBrush);
}

/************************************************************************
 *  AirspaceSegment.cpp			D r a w C l o c k w i s e A r c 		*
 ************************************************************************/
void CAirspaceSegment::DrawClockwiseArc(CDC* pDC,
							long lX1, long lY1, long lX2, long lY2, 
							long lX0, long lY0, long lRadius)
{
	CRect rEllipse;

	rEllipse.SetRect (lX0-lRadius, lY0+lRadius, lX0+lRadius, lY0-lRadius);		// L, T, R, B
	CBrush* ptOldBrush = (CBrush*)pDC->SelectStockObject (NULL_BRUSH);

	CPoint pStart, pEnd;
	pStart.x = lX2;
	pStart.y = lY2;
	pEnd.x = lX1;
	pEnd.y = lY1;

	pDC->Arc(rEllipse, pStart, pEnd);
	pDC->SelectObject (ptOldBrush);
}


/************************************************************************
 *  AirspaceSegment.cpp    		S e r i a l i z e						*
 ************************************************************************/
void CAirspaceSegment::Serialize(CArchive& ar, short nVersion)
{ 
	if (ar.IsStoring())
	{
		// TODO: add storing code here 

	//	AnsiToOem(m_szIdentifier, m_szIdentifier);
		ar.Write (m_szIdentifier, SIZEOF_ASP_IDENT);	// e.g.: ED000027
	//	OemToAnsi(m_szIdentifier, m_szIdentifier);

		ar << m_lSegmentNumber;
		ar << (BYTE)m_cShape;
		ar << (BYTE)m_cDummy;
		ar << m_dLatitude1;
		ar << m_dLongitude1;
		ar << m_dLatitude2;
		ar << m_dLongitude2;

		ar << m_dGeodeticLatitude0;
		ar << m_dGeodeticLongitude0;
		ar << m_dRadius1;
		ar << m_nDistDim;
	}
	else
	{       
		// TODO: add loading code here    		
		BYTE	Byte; 

		ar.Read (m_szIdentifier, SIZEOF_ASP_IDENT);   // e.g.: ED000027

		ar >> m_lSegmentNumber;
		ar >> Byte;		m_cShape = (CAirspaceSegment::SHAPE)Byte;
		ar >> Byte;		m_cDummy = (unsigned char)Byte; 
		ar >> m_dLatitude1;
		ar >> m_dLongitude1;
		ar >> m_dLatitude2;
		ar >> m_dLongitude2;

		ar >> m_dGeodeticLatitude0;
		ar >> m_dGeodeticLongitude0;
		ar >> m_dRadius1;	 
		ar >> m_nDistDim;

	//	OemToAnsi(m_szIdentifier, m_szIdentifier);
	}
} 