// AirspaceSegmentDlg.cpp: Implementierungsdatei
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "pf.h"

#include "InitDoc.h"
#include "DimDoc.h"    
#include "RteView.h"

#include "AirspaceSegmentDoc.h"
#include "AirspaceSegmentDlg.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
// Dialogfeld CAirspaceSegmentDlg 

extern CDimDoc*		ptDim;


CAirspaceSegmentDlg::CAirspaceSegmentDlg(CWnd* pParent, CAirspaceSegmentDoc* ptDoc, 
										 CString szAirspaceName,
										 CString szAirspaceIdentifier,
										 long lStartIndex)
	: CDlgTool(CAirspaceSegmentDlg::IDD, pParent)
{
	//{{AFX_DATA_INIT(CAirspaceSegmentDlg)
	m_nListIndex = -1;
	m_nFormIndex = -1;
	m_nRadDimIndex = -1;
	m_szIdentifier = _T("");
	m_lSegNum = 0;
	m_fRadius = 0;
	m_dLat0 = 0.0;
	m_dLon0 = 0.0;
	m_dLat1 = 0.0;
	m_dLon1 = 0.0;
	m_dLat2 = 0.0;
	m_dLon2 = 0.0;
	m_szAirspaceName = _T("");
	//}}AFX_DATA_INIT

	m_ptDoc = ptDoc;
	m_szAirspaceName = szAirspaceName;
	m_szIdentifier = szAirspaceIdentifier;
	m_ptWayDoc = NULL;			// for importing airspace from route

	ResetDlgEntries(szAirspaceIdentifier, lStartIndex);
	m_bShowFirstIntersection = TRUE;
}


CAirspaceSegmentDlg::~CAirspaceSegmentDlg()
{
}

void CAirspaceSegmentDlg::DoDataExchange(CDataExchange* pDX)
{
	this->ActivateDimGroups();

	CDlgTool::DoDataExchange(pDX);
	//{{AFX_DATA_MAP(CAirspaceSegmentDlg)
	DDX_Text(pDX, IDC_SEGAIRSPACE, m_szAirspaceName);
	DDX_LBIndex(pDX, IDC_SEGLIST, m_nListIndex);
	DDX_CBIndex(pDX, IDC_SEGFORM, m_nFormIndex);
	DDX_CBIndex(pDX, IDC_SEGRADDIM, m_nRadDimIndex);
	DDX_Text(pDX, IDC_SEGNUM, m_lSegNum);
	DDX_Dist(pDX, IDC_SEGRADIUS, m_fRadius);
	DDX_Latitude (pDX, IDC_SEGLAT0, m_dLat0);
	DDX_Longitude(pDX, IDC_SEGLON0, m_dLon0);
	DDX_Latitude (pDX, IDC_SEGLAT1, m_dLat1);
	DDX_Longitude(pDX, IDC_SEGLON1, m_dLon1);
	DDX_Latitude (pDX, IDC_SEGLAT2, m_dLat2);
	DDX_Longitude(pDX, IDC_SEGLON2, m_dLon2);
	//}}AFX_DATA_MAP
}

/*
	DDX_Dist(pDX, IDC_SEGRADIUS, m_fRadius);
	DDX_Text (pDX, IDC_SEGLAT0, m_dLat0);
	DDX_Text(pDX, IDC_SEGLON0, m_dLon0);
	DDX_Latitude (pDX, IDC_SEGLAT1, m_dLat1);
	DDX_Text(pDX, IDC_SEGLON1, m_dLon1);
	DDX_Text (pDX, IDC_SEGLAT2, m_dLat2);
	DDX_Text(pDX, IDC_SEGLON2, m_dLon2);

	DDX_Dist(pDX, IDC_SEGRADIUS, m_fRadius);
    DDV_DistOptional(pDX, m_fRadius, (float)0.1, (float)50.);
	DDX_Latitude (pDX, IDC_SEGLAT0, m_dLat0);
	DDX_Longitude(pDX, IDC_SEGLON0, m_dLon0);
	DDX_Latitude (pDX, IDC_SEGLAT1, m_dLat1);
	DDX_Longitude(pDX, IDC_SEGLON1, m_dLon1);
	DDX_Latitude (pDX, IDC_SEGLAT2, m_dLat2);
	DDX_Longitude(pDX, IDC_SEGLON2, m_dLon2);

  */

BEGIN_MESSAGE_MAP(CAirspaceSegmentDlg, CDlgTool)
	//{{AFX_MSG_MAP(CAirspaceSegmentDlg)
	ON_BN_CLICKED(IDC_SEGFROMROUTE, OnSegmentFromRoute)
	ON_BN_CLICKED(IDC_SEGINS, OnInsert)
	ON_BN_CLICKED(IDC_SEGCHANGE, OnChange)
	ON_BN_CLICKED(IDC_SEGDEL, OnDelete)
	ON_BN_CLICKED(IDOK, OnDone)
	ON_LBN_SELCHANGE(IDC_SEGLIST, OnSelChangeList)
	ON_BN_CLICKED(IDC_SEGCALC_CENTER, OnCalculateCenter)
	ON_CBN_SELCHANGE(IDC_SEGFORM, OnSelChangeSegForm)
	ON_WM_HELPINFO()
	ON_BN_CLICKED(IDC_SEGCOPY, OnCopyEndToStart)
	ON_BN_CLICKED(IDC_SEGTOROUTE, OnSegmentToRoute)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/************************************************************************
 *  AirspaceSegmentDlg.cpp	  	E n a b l e F o r E n t r y I n L i s t	*
 ************************************************************************/
void CAirspaceSegmentDlg::EnableForEntryInList(BOOL bEntryInList)
{
	if (bEntryInList)
	{
		this->DisableControl (IDC_SEGCHANGE, FALSE);  
		this->DisableControl (IDC_SEGDEL, FALSE);  
	}
	else
	{			// all airspaces deleted:
		this->DisableControl (IDC_SEGCHANGE, TRUE);  
		this->DisableControl (IDC_SEGDEL, TRUE);  
	}
}


/************************************************************************
 *  AirspaceSegmentDlg.cpp	   SetEndpointOfSegments					*
 *  if end2 of an segment was changed, end1 of the next segment should  *
 *  be changed also. If end1 of an segment was changed, end2 of the		*
 *  previous segment should be changed also!							*
 *																		*
 *  dOldLat, dOldLon: old position of modified segment or				*
 *						NO_KOORD, if not modified!						*
 *  SetEndpointOfSegments is called twice: once bEnd1=TRUE, once FALSE: *
 *  TRUE: end1 was modified, look for old position at end2 of other seg *
 *  FALSE: end2 was modified, look for old position at end1 of other seg*
 *  dNewLat, dNewLon: new position of modified segment					*
 *  lSegmentId: id of modified segment. Dont modify this segment here	*
 ************************************************************************/
void CAirspaceSegmentDlg::SetEndpointOfSegments(double dOldLat, double dOldLon, BOOL bEnd1, 
								double dNewLat, double dNewLon, long lSegmentId)
{
	short i = (short)m_lStartIndex;
	short nPrevSegIndex = -1;
	short nChangedSegIndex = -1;
	short nNextSegIndex = -1;
	short nLastSegIndex = -1;

	BOOL bNext = (i>=0);		// segments available for current airspace

	while (bNext)
	{					// get index of changed segment
		CAirspaceSegment* ptAirspaceSegment = NULL;
		
		if (m_ptDoc->GetAirspaceSegmentPtr (i, &ptAirspaceSegment))
		{
						// segment of same airspace identifier?
			bNext = (m_szIdentifier.Compare(ptAirspaceSegment->GetIdentifier()) == 0);
			if (bNext)
			{				// set nChangedSegIndex and nLastSegIndex here...
				if (ptAirspaceSegment->GetSegmentNumber() == lSegmentId)
				{		// the changed segment
					nChangedSegIndex = i;
				}

				nLastSegIndex = i;
			} // bNex
			i++;
		}
		else
		{
			bNext = FALSE;
		}
	}


	if (nLastSegIndex == (short)m_lStartIndex)
	{
		return;		// nothing to do
	}

							// set nPrevSegIndex and nNextSegIndex 

	if (nLastSegIndex == (short)m_lStartIndex + 1)
	{		// only 2 segments available
		if (nChangedSegIndex == (short)m_lStartIndex)
			nNextSegIndex = nChangedSegIndex +1;
		else
			nPrevSegIndex = nChangedSegIndex -1;
	}
	else
	{		// at least 3 segments available					
		nNextSegIndex = nChangedSegIndex + 1;
		nPrevSegIndex = nChangedSegIndex - 1;

		if (nNextSegIndex > nLastSegIndex)
			nNextSegIndex = (short)m_lStartIndex;

		if (nPrevSegIndex < (short)m_lStartIndex)
			nPrevSegIndex = nLastSegIndex;	
	}

	if (bEnd1)
	{		// may be required to modify end2 of previous segment		
		CAirspaceSegment* ptAirspaceSegment = NULL;
		
		if (nPrevSegIndex > -1)
		{
			if (m_ptDoc->GetAirspaceSegmentPtr (nPrevSegIndex, &ptAirspaceSegment))
			{
				if (dNewLat != NO_KOORD)
					if (ptAirspaceSegment->GetLat2() == dOldLat)
						ptAirspaceSegment->SetLat2(dNewLat);
				if (dNewLon != NO_KOORD)
					if (ptAirspaceSegment->GetLon2() == dOldLon)
						ptAirspaceSegment->SetLon2(dNewLon);
			}
		}
	}
	else
	{		// may be required to modify end1 of next segment
		CAirspaceSegment* ptAirspaceSegment = NULL;
		
		if (nNextSegIndex > -1)
		{
			if (m_ptDoc->GetAirspaceSegmentPtr (nNextSegIndex, &ptAirspaceSegment))
			{
				if (dNewLat != NO_KOORD)
					if (ptAirspaceSegment->GetLat1() == dOldLat)
						ptAirspaceSegment->SetLat1(dNewLat);

				if (dNewLon != NO_KOORD)
					if (ptAirspaceSegment->GetLon1() == dOldLon)
						ptAirspaceSegment->SetLon1(dNewLon);
			}
		}
	}
}



/************************************************************************
 *  AirspaceSegmentDlg.cpp	   	O n S e g m e n t C h a n g e d			*
 *  oldSegment: segment before it was changed							*
 *  newSegment: segment after it was changed
 ************************************************************************/
void CAirspaceSegmentDlg::OnSegmentChanged (CAirspaceSegment oldSegment, CAirspaceSegment newSegment)
{
		// are endpoints of segment changed? Then modify same endpoints of other segments!
	for (int i=0; i<2; i++)
	{			// for all endpoints of newSegment
		double dOldLat, dOldLon, dNewLat, dNewLon;
		BOOL bEnd1;

		if (i==0)
		{
			dOldLat = oldSegment.GetLat1();
			dOldLon = oldSegment.GetLon1();
			dNewLat = newSegment.GetLat1();
			dNewLon = newSegment.GetLon1();
			bEnd1 = TRUE;
		}
		else
		{
			dOldLat = oldSegment.GetLat2();
			dOldLon = oldSegment.GetLon2();
			dNewLat = newSegment.GetLat2();
			dNewLon = newSegment.GetLon2();
			bEnd1 = FALSE;
		}

		double dEps = 0.001;		// see CAirspaceSegment::operator !=

		if (fabs(dOldLat - dNewLat) <= dEps)
		{		// lat unchanged	
			dNewLat = NO_KOORD;
		}

		if (fabs(dOldLon - dNewLon) <= dEps)
		{		// lon unchanged	
			dNewLon = NO_KOORD;
		}

		if (dNewLat != NO_KOORD || dNewLon != NO_KOORD)  // at least one value changed:
			SetEndpointOfSegments(dOldLat, dOldLon, bEnd1, 
				dNewLat, dNewLon, newSegment.GetSegmentNumber());

	} // next side of segment
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp	  		G e t L i s t E n t r y				*
 ************************************************************************/
CString CAirspaceSegmentDlg::GetListEntry (CAirspaceSegment* ptAirspaceSegment)
{
	CString szListEntry;
	szListEntry.Format("%ld", ptAirspaceSegment->GetSegmentNumber());
	return szListEntry;
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp	   I n s e r t L i s t E n t r y 			*
 *  Adds new list entry to list and										*
 *  adds new ptAirspaceSegment to m_ListObjects							*
 ************************************************************************/
void CAirspaceSegmentDlg::InsertListEntry(CAirspaceSegment* ptAirspaceSegment, short nListIndex) 
{		
	CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);    

	CString szListEntry;
	szListEntry = GetListEntry(ptAirspaceSegment);

	m_ListObjects.SetAt(szListEntry, ptAirspaceSegment);
	ptList->InsertString(nListIndex, (LPCTSTR)szListEntry);  
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp	   	C h a n g e L i s t E n t r y 			*
 *  If list entry has changed, the list entry will be updated			*
 *  m_ListObjects remains unchanged										*
 ************************************************************************/
void CAirspaceSegmentDlg::ChangeListEntry(CAirspaceSegment* ptAirspaceSegment, short nListIndex) 
{
	CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);    
	CString szOldName;
	int nOldIndex = nListIndex;
	ptList->GetText(nOldIndex, szOldName);
			
	CString szListEntry;
	szListEntry = GetListEntry(ptAirspaceSegment);

	if (szListEntry.CompareNoCase((LPCTSTR)szOldName) != 0)
	{                		// change listbox entry	
		ptList->DeleteString(nOldIndex);
		ptList->InsertString(nListIndex, (LPCTSTR)szListEntry);  
	}   
}


/************************************************************************
 *  AirspaceSegmentDlg.cpp		 D e l e t e L i s t E n t r y 			*
 *  gets text to be deleted from list,									*
 *  removes pointer to already deleted memory from m_ListObjects		*
 *  and deletes old entry from list										*
 ************************************************************************/
void CAirspaceSegmentDlg::DeleteListEntry(short nListIndex) 
{
	CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);    

	CString szListEntry;
	ptList->GetText(nListIndex, szListEntry);

	m_ListObjects.RemoveKey(szListEntry);	// CAirspaceSegment* must be already deleted
	ptList->DeleteString(nListIndex);
}


/************************************************************************
 *  AirspaceSegmentDlg.cpp	  			F i l l L i s t					*
 ************************************************************************/
short CAirspaceSegmentDlg::FillList ()
{
	CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);    
	ptList->ResetContent();
	m_ListObjects.RemoveAll();

	CString szListEntry;
	short i = (short)m_lStartIndex;
	BOOL bNext = (i>=0);		// segments available for current airspace

	while (bNext)
	{      
		CAirspaceSegment* ptAirspaceSegment = NULL;
		
		if (m_ptDoc->GetAirspaceSegmentPtr (i, &ptAirspaceSegment))
		{
			if (i == m_lStartIndex)	// store identifier of airspace
				m_szIdentifier = ptAirspaceSegment->GetIdentifier();

						// segment of same airspace identifier?
			bNext = (m_szIdentifier.Compare(ptAirspaceSegment->GetIdentifier()) == 0);

			if (bNext)
			{
				szListEntry = GetListEntry(ptAirspaceSegment);
				m_ListObjects.SetAt(szListEntry, ptAirspaceSegment);
				ptList->AddString((LPCTSTR)szListEntry);  
			}

			i++;
		}
		else
		{
			bNext = FALSE;
		}
	}

	if (ptList->GetCount() > 0)
		 m_nListIndex = 0;
	else m_nListIndex = -1;

	ptList->SetCurSel(m_nListIndex);

	return m_nListIndex;
}   

/************************************************************************
 *  AirspaceSegmentDlg.cpp	  		F i l l S h a p e P U 				*
 ************************************************************************/
void CAirspaceSegmentDlg::FillShapePU ()
{      
	CComboBox* ptShape = (CComboBox*)GetDlgItem (IDC_SEGFORM);

	CString szListEntry;
	short i;

	for (i=IDS_ASGSHAPE01; i<=IDS_ASGSHAPE07; i++)
	{
		szListEntry.LoadString (i);  
		ptShape->AddString((LPCTSTR)szListEntry);  
	}
} 

/////////////////////////////////////////////////////////////////////////////
// Behandlungsroutinen fr Nachrichten CAirspaceSegmentDlg 

BOOL CAirspaceSegmentDlg::OnInitDialog() 
{
	m_nListIndex = FillList();
	FillShapePU();

	EnableForEntryInList(m_nListIndex >= 0);

	this->AddDimGroup (IDC_SEGRADIUS, IDC_SEGRADDIM);
	this->FillDimPU (IDC_SEGRADDIM, DISTANCE);

	CDlgTool::OnInitDialog();
	
	// TODO: Zustzliche Initialisierung hier einfgen

	if (m_nListIndex >= 0)
	{
		SetSelectedData(m_nListIndex);
	}
	EnableCalcSegmentsFromRte(m_nFormIndex);


	return TRUE;  // return TRUE unless you set the focus to a control
	              // EXCEPTION: OCX-Eigenschaftenseiten sollten FALSE zurckgeben
}


/************************************************************************
 *  AirspaceSegmentDlg.cpp			ResetDlgEntries						*
 *  resets all dlg entries												*
 ************************************************************************/
void CAirspaceSegmentDlg::ResetDlgEntries(CString szIdentifier, long lStartIndex)
{
//	m_szIdentifier = szIdentifier;
	m_lStartIndex = lStartIndex;

	m_nListIndex = -1;

	m_nFormIndex = (int)CAirspaceSegment::LINE;
	m_nRadDimIndex = ptDim->GetPUIndex (DISTANCE, DIM_NM);
	m_lSegNum = 10;
	m_fRadius = NO_DIST;
	m_dLat0 = NO_KOORD;
	m_dLon0 = NO_KOORD;
	m_dLat1 = NO_KOORD;
	m_dLon1 = NO_KOORD;
	m_dLat2 = NO_KOORD;
	m_dLon2 = NO_KOORD;

	m_ptAirspaceSegment=NULL;
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp			EnableCalcSegmentsFromRte			*
 *  enables or disables state of button	IDC_SEGFROMROUTE				*
 ************************************************************************/
void CAirspaceSegmentDlg::EnableCalcSegmentsFromRte(short nFormIndex) 
{
	short nWptCnt = 0;
	if (m_ptWayDoc != NULL)
	{				// get number of waypoints of actual route
		nWptCnt = m_ptWayDoc->GetWayPointCnt();
	}


				// each segment form requires different number of points
	CAirspaceSegment::SHAPE shape = (CAirspaceSegment::SHAPE)nFormIndex;

	BOOL bCalcSegmentsFromRte = TRUE;

	switch (shape)
	{
		case CAirspaceSegment::POINT:	// POINT (WITHOUT RADIUS OR BEARING)
			break;

		case CAirspaceSegment::GREATCIRCLE:	// GREAT CIRCLE
			bCalcSegmentsFromRte = (nWptCnt > 1);
			break;

		case CAirspaceSegment::CIRCLE:		// CIRCLE
			bCalcSegmentsFromRte = (nWptCnt > 0);
			break;

		case CAirspaceSegment::IRREGULAR:	// GENERALIZED
			bCalcSegmentsFromRte = (nWptCnt > 1);
			break;

		case CAirspaceSegment::LINE:		// RHUMB LINE
			bCalcSegmentsFromRte = (nWptCnt > 1);
			break;

		case CAirspaceSegment::ARCLEFT:		// COUNTERCLOCKWISE ARC
			bCalcSegmentsFromRte = (nWptCnt > 1);
			break;

		case CAirspaceSegment::ARCRIGHT:	// CLOCKWISE ARC
			bCalcSegmentsFromRte = (nWptCnt > 1);
			break;
	}

	DisableControl(IDC_SEGFROMROUTE, !bCalcSegmentsFromRte);
}


/************************************************************************
 *  AirspaceSegmentDlg.cpp			CheckForParamsRequired				*
 *  shows warn message, if params are missing for specified shape		*
 ************************************************************************/
BOOL CAirspaceSegmentDlg::CheckForParamsRequired(short nFormIndex) 
{
	BOOL bOK = TRUE;

	CAirspaceSegment::SHAPE shape = (CAirspaceSegment::SHAPE)nFormIndex;
	CString szShape;
	szShape.LoadString (IDS_ASGSHAPE01 + nFormIndex);  


	if (shape == CAirspaceSegment::POINT)
	{

	}


	if (shape == CAirspaceSegment::GREATCIRCLE ||
		shape == CAirspaceSegment::IRREGULAR ||
		shape == CAirspaceSegment::LINE)
	{
		if (m_dLat1 == NO_KOORD || m_dLon1 == NO_KOORD ||
			m_dLat2 == NO_KOORD || m_dLon2 == NO_KOORD)
		{
			CString szText;
			szText.Format(IDF_PARAMSREQ_LINE, (LPCTSTR)szShape);
			AfxMessageBox (szText, MB_OK);
			bOK = FALSE;
		}

	}

	if (shape == CAirspaceSegment::CIRCLE)
	{
		if (m_dLat0 == NO_KOORD || m_dLon0 == NO_KOORD ||
			 m_fRadius == NO_DIST)
		{
			CString szText;
			szText.Format(IDF_PARAMSREQ_CIRCLE, (LPCTSTR)szShape);
			AfxMessageBox (szText, MB_OK);
			bOK = FALSE;
		}

	}

	if (shape == CAirspaceSegment::ARCLEFT ||
		shape == CAirspaceSegment::ARCRIGHT)
	{
		if (m_dLat1 == NO_KOORD || m_dLon1 == NO_KOORD ||
			m_dLat2 == NO_KOORD || m_dLon2 == NO_KOORD ||
			m_dLat0 == NO_KOORD || m_dLon0 == NO_KOORD ||
			m_fRadius == NO_DIST)
		{
			CString szText;
			szText.Format(IDF_PARAMSREQ_ARC, (LPCTSTR)szShape);
			AfxMessageBox (szText, MB_OK);
			bOK = FALSE;
		}
	}


	if (bOK)
	{				// enough params for specified form,
		if (shape == CAirspaceSegment::POINT ||
			shape == CAirspaceSegment::GREATCIRCLE ||
			shape == CAirspaceSegment::IRREGULAR ||
			shape == CAirspaceSegment::LINE)
		{			// segement cannot display arc
			if (m_dLat0 != NO_KOORD && m_dLon0 != NO_KOORD &&
				m_fRadius != NO_DIST)
			{		// but arc parameters set
				CString szText;
				szText.Format(IDF_BADFORMFORARC, (LPCTSTR)szShape);
				AfxMessageBox (szText, MB_OK);
				bOK = FALSE;
			}
		}
	}
	return bOK;
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp			DeleteAllSegments					*
 *  deletes all segments of current airspace							*
 ************************************************************************/
void CAirspaceSegmentDlg::DeleteAllSegments() 
{
	CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);  
	if (ptList != NULL)
	{
		short nSegCnt = ptList->GetCount();
		for (int i=0; i<nSegCnt; i++)
		{						// always delete entry at start index
			short nIndex = (short)m_lStartIndex;
			short nSelIndex = 0;

			if (m_ptDoc->DeleteAirspaceSegmentPtr(nIndex))
			{					// segment deleted from doc
				m_ptDoc->SetModifiedFlag(TRUE);
				DeleteListEntry(nSelIndex);
			}
		}
	}
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp			ActRteWptToCircle					*
 *  reads act wpt of current route and converts it to circle segment	*
 *  initial radius will be set to 5 NM.									*
 ************************************************************************/
void CAirspaceSegmentDlg::ActRteWptToCircle(short nWptCnt) 
{
	CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);  	

	short nSelIndex = -1;

	short nActInd = m_ptWayDoc->GetActInd();
	if (nActInd < 0 || nActInd >= nWptCnt)
	{
		nActInd = 0;
	}

	double dLat = NO_KOORD;
	double dLon = NO_KOORD;

	CWayPoint WayPoint;
	if (m_ptWayDoc->GetWayPointPtr(nActInd, &WayPoint))
	{
		dLat = WayPoint.GetLat();
		dLon = WayPoint.GetLon();
	}

	if (dLat != NO_KOORD && dLon != NO_KOORD)
	{
		CAirspaceSegment* ptAirspaceSegment = new CAirspaceSegment;
		ptAirspaceSegment->SetIdentifier(m_szIdentifier);
		ptAirspaceSegment->SetGeodeticLat0(dLat);
		ptAirspaceSegment->SetGeodeticLon0(dLon);
		ptAirspaceSegment->SetRadius(5);
		ptAirspaceSegment->SetDistDim(DIM_NM);
		ptAirspaceSegment->SetShape(CAirspaceSegment::CIRCLE);

		long lNewAspSegNum = 10;
		ptAirspaceSegment->SetSegmentNumber(lNewAspSegNum);

								// store new segment in doc
		long lAbsIndex = m_ptDoc->AddSortedAirspaceSegmentPtr(ptAirspaceSegment);
		if (lAbsIndex >= 0)
		{
			if (m_lStartIndex == -1)
				m_lStartIndex = lAbsIndex;
	
			m_ptDoc->SetModifiedFlag(TRUE);
			nSelIndex++;
			m_nListIndex = nSelIndex;		// move selection on new entry
			InsertListEntry(ptAirspaceSegment, m_nListIndex);

			ptList->SetCurSel(m_nListIndex);
					// m_ptAirspaceSegment yet points on selected segment
			SetSelectedData(m_nListIndex);
					// now m_ptAirspaceSegment points on new segment!
		}
		else
		{
			delete ptAirspaceSegment;
			ptAirspaceSegment = NULL;
		}

	} // good dLat, dLon
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp			RouteToSegments						*
 *  reads entries of current route and converts waypoints to segments	*
 ************************************************************************/
void CAirspaceSegmentDlg::RouteToSegments(short nWptCnt) 
{
	double dLat0 = NO_KOORD;
	double dLon0 = NO_KOORD;
	double dLatLast, dLonLast;

					// get position, where to insert new segments
	CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);  	
	short nSelIndex = ptList->GetCurSel ();

	for (int i=0; i<=nWptCnt; i++)
	{
		double dLat = NO_KOORD;
		double dLon = NO_KOORD;

		if (i<nWptCnt)
		{			// use next way point
			CWayPoint WayPoint;
			if (m_ptWayDoc->GetWayPointPtr(i, &WayPoint))
			{
				dLat = WayPoint.GetLat();
				dLon = WayPoint.GetLon();
			}
		}
		else
		{			// use first way point
			dLat = dLat0;
			dLon = dLon0;
		}


		if (dLat != NO_KOORD && dLon != NO_KOORD)
		{
			if (i==0)
			{		// store first point
				dLat0 = dLat;
				dLon0 = dLon;
			}
			else
			{	// use last and current position for segment
				CAirspaceSegment* ptAirspaceSegment = new CAirspaceSegment;
				ptAirspaceSegment->SetIdentifier(m_szIdentifier);
				ptAirspaceSegment->SetLat1(dLatLast);
				ptAirspaceSegment->SetLon1(dLonLast);
				ptAirspaceSegment->SetLat2(dLat);
				ptAirspaceSegment->SetLon2(dLon);
				ptAirspaceSegment->SetShape(CAirspaceSegment::LINE);

				long lNewAspSegNum = GetNewSegmentNumber(nSelIndex);
				ptAirspaceSegment->SetSegmentNumber(lNewAspSegNum);

										// store new segment in doc
				long lAbsIndex = m_ptDoc->AddSortedAirspaceSegmentPtr(ptAirspaceSegment);
				if (lAbsIndex >= 0)
				{
					if (m_lStartIndex == -1)
						m_lStartIndex = lAbsIndex;
			
					m_ptDoc->SetModifiedFlag(TRUE);
					nSelIndex++;
					m_nListIndex = nSelIndex;		// move selection on new entry
					InsertListEntry(ptAirspaceSegment, m_nListIndex);

					if (i == nWptCnt)
					{		// last inserted segment
						ptList->SetCurSel(m_nListIndex);
								// m_ptAirspaceSegment yet points on selected segment
						SetSelectedData(m_nListIndex);
								// now m_ptAirspaceSegment points on new segment!
					}
				}
				else
				{
					delete ptAirspaceSegment;
					ptAirspaceSegment = NULL;
				}
			}

			dLatLast = dLat;
			dLonLast = dLon;
		} // good dLat, dLon
	}
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp			OnSegmentFromRoute					*
 *  if user really wants to create new segments,						*
 *		delete all old segmentes										*
 *		read entries of current route and converts wpts to segments		*
 ************************************************************************/
void CAirspaceSegmentDlg::OnSegmentFromRoute() 
{
	// TODO: Code fr die Behandlungsroutine der Steuerelement-Benachrichtigung hier einfgen
	if (m_ptWayDoc != NULL)
	{
		BOOL bRouteOK = FALSE;
		short nWptCnt = m_ptWayDoc->GetWayPointCnt();

		CAirspaceSegment::SHAPE shape = (CAirspaceSegment::SHAPE)m_nFormIndex;
		if (shape == CAirspaceSegment::CIRCLE)
		{
			bRouteOK = (nWptCnt > 0);
		}
		else
		{
			bRouteOK = (nWptCnt > 1);
		}

		if (bRouteOK)
		{
			BOOL bNewSegments = TRUE;

			CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);  	
			if (ptList != NULL && ptList->GetCount() > 0)
			{			// ask user to delete segments of current airspace
				CString szText;
				szText.Format(IDF_DELETE_SEGMENTS, (LPCTSTR)m_szAirspaceName);
				if (AfxMessageBox (szText, MB_YESNO) != IDYES)
					bNewSegments = FALSE;
			}

			if (bNewSegments)
			{						// delete old segments
				DeleteAllSegments();

									// reset entries of dlg
				CString szIdentifier("");
				long lStartIndex = -1;
				ResetDlgEntries(szIdentifier,lStartIndex);

									// create new segments
				if (shape == CAirspaceSegment::CIRCLE)
						ActRteWptToCircle(nWptCnt);
				else	RouteToSegments(nWptCnt);
			}
		}
	}
}


/************************************************************************
 *  AirspaceSegmentDlg.cpp			OnSegmentToRoute					*
 *  if user really wants to create a new route,							*
 *		delete all old waypoints										*
 *		read entries of current airspace and converts segments to wpts	*
 ************************************************************************/
void CAirspaceSegmentDlg::OnSegmentToRoute() 
{
	if (m_ptWayDoc != NULL)
	{
		CAirspaceSegment::SHAPE shape = (CAirspaceSegment::SHAPE)m_nFormIndex;
		if (shape == CAirspaceSegment::CIRCLE)
		{				// a circle segment cannot be converted to a route!
			AfxMessageBox (IDS_NOROUTEFROM_CIRCLE);
			return;
		}
		else
		{
			if (m_ptWayDoc->IsModified())
			{				// ask user to save current route
				CString szPath = m_ptWayDoc->GetPathName();
				m_ptWayDoc->OnSaveDocument(szPath, TRUE);
			}
		}


		m_ptWayDoc->NoHomeBase();
		m_ptWayDoc->OnNewDocument();
		m_ptWayDoc->SetTitle((LPCTSTR)m_szIdentifier);

								// for all segments of list
		CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);  
		if (ptList != NULL)
		{
			short nSegCnt = ptList->GetCount();
			for (int i=0; i<nSegCnt; i++)
			{						// always delete entry at start index
				short nIndex = (short)m_lStartIndex;

				CAirspaceSegment* ptAirspaceSegment = NULL;
				if (m_ptDoc->GetAirspaceSegmentPtr(nIndex+i, &ptAirspaceSegment))
				{					
					CWayPoint WayPt;
					CString szName;
					double dLat = NO_KOORD;
					double dLon = NO_KOORD;

					ptList->GetText(i, szName);
					dLat = ptAirspaceSegment->GetLat1();
					dLon = ptAirspaceSegment->GetLon1();

					if (dLat != NO_KOORD && dLon != NO_KOORD)
					{
						WayPt.SetLat(dLat);
						WayPt.SetLon(dLon);
						WayPt.SetAltDim(DIM_FEET);
						WayPt.SetPlannedAlt(5000);
						WayPt.SetCategory(WP_USER);
						WayPt.SetName(szName);
						WayPt.SetIndicator("");
						WayPt.SetFrequency(NO_FREQ);
						WayPt.SetRange(0);
						WayPt.SetAltitude(0);	

						m_ptWayDoc->AppendWpt(WayPt);
						m_ptWayDoc->SetModifiedFlag();
					}
				}
			}
		}
	}
}


/************************************************************************
 *  AirspaceSegmentDlg.cpp			GetNewSegmentNumber					*
 *  reads entries of list to determine the next free segment number		*
 *  if required, the numbers of the existing segments will be increased	*
 ************************************************************************/
long CAirspaceSegmentDlg::GetNewSegmentNumber(short nSelIndex)
{
	long lNewAspSegNum = 10;

	if (nSelIndex >= 0)
	{			// an entry was slected
				// get number of selected segment
		CAirspaceSegment* ptSelAirspaceSegment = GetSelectedObject(nSelIndex); 
		if (ptSelAirspaceSegment != NULL)
		{
			long lSelAspSegNum = ptSelAirspaceSegment->GetSegmentNumber();

							// look for next entry
			CAirspaceSegment* ptNextAirspaceSegment = GetSelectedObject(nSelIndex+1); 
			if (ptNextAirspaceSegment != NULL)
			{				// next entry available
				long lNextAspSegNum = ptNextAirspaceSegment->GetSegmentNumber();

				if (lNextAspSegNum > lSelAspSegNum + 1)
				{				//4>2+1
					lNewAspSegNum = (lNextAspSegNum + lSelAspSegNum) / 2;
				}
				else
				{	// too less space btn numbers, increase next numbers by 10
					int nSegmentCnt = m_ListObjects.GetCount();
					for (int i=nSegmentCnt; i>nSelIndex; i--)
					{
						CAirspaceSegment* ptAirspaceSegment = GetSelectedObject(i); 
						if (ptAirspaceSegment != NULL)
						{
							long lAspSegNum = ptAirspaceSegment->GetSegmentNumber();
							ptAirspaceSegment->SetSegmentNumber(lAspSegNum + 10);
						}
					}
					lNewAspSegNum = lSelAspSegNum + 10;
				}
			} // ptNextAirspaceSegment
			else
			{		// last segment was selected
				lNewAspSegNum = lSelAspSegNum + 10;
			}
		} // ptSelAirspaceSegment
	}

	return lNewAspSegNum;
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp			O n I n s e r t						*
 ************************************************************************/
void CAirspaceSegmentDlg::OnInsert() 
{
	CAirspaceSegment* ptAirspaceSegment = new CAirspaceSegment;
	if(this->GetData(ptAirspaceSegment))
	{			// get new entry

		CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);  
		
		short nSelIndex = ptList->GetCurSel ();

		long lNewAspSegNum = GetNewSegmentNumber(nSelIndex);
		ptAirspaceSegment->SetSegmentNumber(lNewAspSegNum);

						// store new segment in doc
		long lAbsIndex = m_ptDoc->AddSortedAirspaceSegmentPtr(ptAirspaceSegment);
		if (lAbsIndex >= 0)
		{
			if (m_lStartIndex == -1)
				m_lStartIndex = lAbsIndex;

			m_ptDoc->SetModifiedFlag(TRUE);

			m_nListIndex = nSelIndex+1;		// move selection on new entry
			InsertListEntry(ptAirspaceSegment, m_nListIndex);
			ptList->SetCurSel(m_nListIndex);
						// m_ptAirspaceSegment yet points on selected segment
			SetSelectedData(m_nListIndex);
						// now m_ptAirspaceSegment points on new segment!
			EnableForEntryInList(m_nListIndex >= 0);
		}
	}
	else
	{		// data not valid, delete airspace segment
		delete ptAirspaceSegment;
	}
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp			O n C h a n g e						*
 ************************************************************************/
void CAirspaceSegmentDlg::OnChange() 
{
	// TODO: Code fr die Behandlungsroutine der Steuerelement-Benachrichtigung hier einfgen
	if (m_ptAirspaceSegment != NULL)
	{
		CAirspaceSegment AirspaceSegment;
		if(this->GetData(&AirspaceSegment))
		{			// save old entry
			if (AirspaceSegment != *m_ptAirspaceSegment)
			{	
				OnSegmentChanged(*m_ptAirspaceSegment, AirspaceSegment);
							// to change record, update CONTENT of m_ptAirspaceSegment
				*m_ptAirspaceSegment = AirspaceSegment;
				m_ptDoc->SetModifiedFlag(TRUE);

				ChangeListEntry(m_ptAirspaceSegment, m_nListIndex);
			}
		}	
	}
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp			O n D e l e t e						*
 ************************************************************************/
void CAirspaceSegmentDlg::OnDelete() 
{
	// TODO: Code fr die Behandlungsroutine der Steuerelement-Benachrichtigung hier einfgen
	CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);  
		
	short nSelIndex = ptList->GetCurSel ();			
	short nIndex = (short)m_lStartIndex + nSelIndex;

	if (m_ptDoc->DeleteAirspaceSegmentPtr(nIndex))
	{					// segment deleted from doc
		m_ptDoc->SetModifiedFlag(TRUE);
			
		DeleteListEntry(nSelIndex);
		if (nSelIndex > 0)
			nSelIndex--;
		if (nSelIndex == m_ListObjects.GetCount())
			nSelIndex--;

		// m_ptAirspaceSegment yet points on selected segment
		if (nSelIndex >=0 && nSelIndex < m_ListObjects.GetCount())
		{				// now selection points to next entry
			m_nListIndex = nSelIndex;
			SetSelectedData(nSelIndex);
					// now m_ptAirspaceSegment points on next segment!
		}
		else
		{			// all entries deleted
			CString szIdentifier("");
			long lStartIndex = -1;
			ResetDlgEntries(szIdentifier, lStartIndex);

			EnableForEntryInList(m_nListIndex >= 0);

			this->UpdateData(FALSE);
		}
	}
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp			O n D o n e							*
 ************************************************************************/
void CAirspaceSegmentDlg::OnDone() 
{
	// TODO: Code fr die Behandlungsroutine der Steuerelement-Benachrichtigung hier einfgen
	if (m_ptAirspaceSegment != NULL)
	{
		CAirspaceSegment AirspaceSegment;
		if(this->GetData(&AirspaceSegment))
		{			// save old entry
			if (AirspaceSegment != *m_ptAirspaceSegment)
			{	
				OnSegmentChanged(*m_ptAirspaceSegment, AirspaceSegment);

							// to change record, update CONTENT of m_ptAirspaceSegment
				*m_ptAirspaceSegment = AirspaceSegment;
				m_ptDoc->SetModifiedFlag(TRUE);

				ChangeListEntry(m_ptAirspaceSegment, m_nListIndex);
			}
		CDialog::OnOK();
		}	
	}
	else
	{
		CDialog::OnOK();
	}
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp		G e t S e l e c t e d O b j e c t 		*
 *  returns object pointer from m_ListObjects or NULL,					*
 *  if nListIndex is out of range										*
 ************************************************************************/
CAirspaceSegment* CAirspaceSegmentDlg::GetSelectedObject(short nListIndex)
{
	CAirspaceSegment* ptAirspaceSegment = NULL;
	CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);    

	if (nListIndex >= 0 && nListIndex < ptList->GetCount())
	{
		CString szListEntry;
		ptList->GetText(nListIndex, szListEntry);
		if (szListEntry.GetLength() > 0)
		{
			ptAirspaceSegment = (CAirspaceSegment*)this->m_ListObjects[szListEntry];
		}
	}

	return ptAirspaceSegment;
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp	 	S e t S e l e c t e d D a t a  			*
 *  Stores pointer of selected object in m_ptAirspaceSegment			*
 *  and uptates dialog items											*
 ************************************************************************/
BOOL CAirspaceSegmentDlg::SetSelectedData(short nListIndex)
{
	BOOL bNewDataSet = FALSE;
	
		// to select a new record, store new pointer in m_ptAirspaceSegment
	m_ptAirspaceSegment = GetSelectedObject(nListIndex);
	if (m_ptAirspaceSegment != NULL)
	{
		SetData(m_ptAirspaceSegment);
		bNewDataSet = TRUE;
	}
	return bNewDataSet;
}


/************************************************************************
 *  AirspaceSegmentDlg.cpp	  		S e t D a t a  						*
 ************************************************************************/
void CAirspaceSegmentDlg::SetData(CAirspaceSegment* ptAirspaceSegment)
{
	m_lSegNum = ptAirspaceSegment->GetSegmentNumber();
	m_fRadius = (float)ptAirspaceSegment->GetRadius();

	m_nRadDimIndex = ptDim->GetPUIndex (DISTANCE, ptAirspaceSegment->GetDistDim());

	m_dLat0 = ptAirspaceSegment->GetGeodeticLat0();
	m_dLon0 = ptAirspaceSegment->GetGeodeticLon0();
	m_dLat1 = ptAirspaceSegment->GetLat1();
	m_dLon1 = ptAirspaceSegment->GetLon1();

	m_dLat2 = ptAirspaceSegment->GetLat2();
	m_dLon2 = ptAirspaceSegment->GetLon2();

	CAirspaceSegment::SHAPE shape = ptAirspaceSegment->GetShape();	// shape to index
	m_nFormIndex = (int)shape;

	UpdateData(FALSE);
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp	  		G e t D a t a  						*
 ************************************************************************/
BOOL CAirspaceSegmentDlg::GetData (CAirspaceSegment* ptAirspaceSegment)
{       
	BOOL bOK = FALSE;

	if (this->UpdateData(TRUE))
	{
		ptAirspaceSegment->SetIdentifier(m_szIdentifier);
		ptAirspaceSegment->SetSegmentNumber(m_lSegNum);
		ptAirspaceSegment->SetRadius(m_fRadius);

		ptAirspaceSegment->SetDistDim(ptDim->GetPUDim (DISTANCE, m_nRadDimIndex));

		ptAirspaceSegment->SetGeodeticLat0(m_dLat0);
		ptAirspaceSegment->SetGeodeticLon0(m_dLon0);
		ptAirspaceSegment->SetLat1(m_dLat1);
		ptAirspaceSegment->SetLon1(m_dLon1);

		ptAirspaceSegment->SetLat2(m_dLat2);
		ptAirspaceSegment->SetLon2(m_dLon2);

		CAirspaceSegment::SHAPE shape = (CAirspaceSegment::SHAPE)m_nFormIndex;
		ptAirspaceSegment->SetShape(shape);

		bOK = CheckForParamsRequired(m_nFormIndex);
	}
	return bOK;
}


/************************************************************************
 *  AirspaceSegmentDlg.cpp	  	O n S e l C h a n g e L i s t 			*
 ************************************************************************/
void CAirspaceSegmentDlg::OnSelChangeList() 
{
		// TODO: Code fr die Behandlungsroutine der Steuerelement-Benachrichtigung hier einfgen
	CListBox* ptList = (CListBox*)GetDlgItem (IDC_SEGLIST);    
	short nNewIndex = ptList->GetCurSel ();
	
	if (m_nListIndex != nNewIndex)
	{
		short nOldIndex = m_nListIndex;
		CAirspaceSegment AirspaceSegment;
		if (this->GetData(&AirspaceSegment))
		{			// save old entry
			if (AirspaceSegment != *m_ptAirspaceSegment)
			{	
				OnSegmentChanged(*m_ptAirspaceSegment, AirspaceSegment);
					// to change record, update CONTENT of m_ptAirspaceSegment, NOT pointer!!
				*m_ptAirspaceSegment = AirspaceSegment;
				m_ptDoc->SetModifiedFlag(TRUE);
			}

					// show selected entry
			if (SetSelectedData(nNewIndex))
				m_nListIndex = nNewIndex;
		}      
		else
		{				// bad input, reset selection to bad entry 
			ptList->SetCurSel (nOldIndex);	
			m_nListIndex = nOldIndex;
		}
	}
	else
	{					// show selected entry without saving current record (UNDO)
		SetSelectedData(nNewIndex);
	}
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp	  	O n S e l C h a n g e S e g F o r m		*
 ************************************************************************/
void CAirspaceSegmentDlg::OnSelChangeSegForm() 
{
	CComboBox* ptShape = (CComboBox*)GetDlgItem (IDC_SEGFORM);
	short nNewIndex = ptShape->GetCurSel ();
	
	if (m_nFormIndex != nNewIndex)
	{
		m_nFormIndex = nNewIndex;
		EnableCalcSegmentsFromRte(nNewIndex);
	}
}


/************************************************************************
 *  AirspaceSegmentDlg.cpp	  	O n C a l c u l a t e C e n t e r 		*
 *  if two end points of a segment are given, this method calculates	*
 *  the centre of an arc, used to connect that end points.				*
 *  If a radius is defined and large enough, it will be used.			*
 *  Otherwhise a radius will be set to the half of the segment distance *
 ************************************************************************/
void CAirspaceSegmentDlg::OnCalculateCenter() 
{
	// TODO: Code fr die Behandlungsroutine der Steuerelement-Benachrichtigung hier einfgen
	if (this->UpdateData(TRUE))
	{					
		if (m_dLat1 != NO_KOORD && m_dLon1 != NO_KOORD &&
			m_dLat2 != NO_KOORD && m_dLon2 != NO_KOORD)
		{
							// use route view to convert lat/lon to linear map values
			CRouteView RouteView;

			CMap*	ptMercator = RouteView.GetMapPtr();
			if (ptMercator != NULL)
			{
				DRECT rMinLatLon;
				rMinLatLon.bottom	= m_dLat1 - 1;
				rMinLatLon.top		= m_dLat1 + 1;
				rMinLatLon.left		= m_dLon1 - 1;
				rMinLatLon.right	= m_dLon1 + 1;

				CRect rMaxLP;
				rMaxLP.bottom		= -2000;
				rMaxLP.top			= 0;
				rMaxLP.left			= 0;
				rMaxLP.right		= 2000;

				BOOL bPrinter = FALSE;
				ptMercator->Calculate(rMinLatLon, rMaxLP, bPrinter);

				if (m_fRadius == NO_DIST)
					m_fRadius = 0;		// force to define radius

				double dLat1LP;		// convert lat/lon and dRadius to LP
				double dLon1LP;		
				short nDistDim = ptDim->GetPUDim (DISTANCE, m_nRadDimIndex);
				long lRadius = RouteView.DistanceToLP(m_dLat1, m_dLon1, &dLon1LP, &dLat1LP, 
											m_fRadius, nDistDim);

				double dLat2LP;		// convert lat2/lon2 to LP
				double dLon2LP;			
				ptMercator->XYtoLP(m_dLon2, m_dLat2, &dLon2LP, &dLat2LP);

				CVektor CA(dLon1LP, dLat1LP);
				CVektor CB(dLon2LP, dLat2LP);

									// use linear vektor method now:
				CVektor VI1;
				CVektor VI2;
				if (!CVektor::CircleIntersections(CA, CB, lRadius, &VI1, &VI2))
				{					// radius was too small
									// calculate new radius
					CLatLon llA(m_dLat1, m_dLon1);
					CLatLon llB(m_dLat2, m_dLon2);
					double dDist = llA.LoxoDist(llB) / 2; 
					m_fRadius = (float)ptDim->ConvertDist(dDist, DIM_NM, nDistDim);
				}

				if (m_bShowFirstIntersection)
				{			// convert linear map values to lat/lon
					ptMercator->LPtoXY(VI1.x, VI1.y, &m_dLon0, &m_dLat0);
					m_bShowFirstIntersection = FALSE;
				}
				else
				{			// convert linear map values to lat/lon
					ptMercator->LPtoXY(VI2.x, VI2.y, &m_dLon0, &m_dLat0);
					m_bShowFirstIntersection = TRUE;
				}
					
				this->UpdateData(FALSE);	// show results
			} // good ptMercator

		} // good lat/lon
	} // good Update
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp	  		O n C o p y E n d T o S t a r t		*
 ************************************************************************/
void CAirspaceSegmentDlg::OnCopyEndToStart() 
{
	CString szLat2;
	CString szLon2;

	GetText(IDC_SEGLAT2, szLat2);
	GetText(IDC_SEGLON2, szLon2);

	if (szLat2.GetLength() > 0 && szLon2.GetLength() > 0)
	{			// co-ords of end point may the available
		if (AfxMessageBox (IDS_COPYENDPOINT, MB_YESNO) == IDYES)
		{		// user wants to copy them to start point
			SetText(IDC_SEGLAT1, szLat2);
			SetText(IDC_SEGLON1, szLon2);
			SetText(IDC_SEGLAT2, "");	// clear end point
			SetText(IDC_SEGLON2, "");

			CWnd* ptWnd = this->GetDlgItem (IDC_SEGLAT2);	
			ptWnd->SetFocus ();
		}
	}
}

/************************************************************************
 *  AirspaceSegmentDlg.cpp	  		O n H e l p I n f o					*
 ************************************************************************/
BOOL CAirspaceSegmentDlg::OnHelpInfo(HELPINFO* pHelpInfo) 
{
	// TODO: Add your message handler code here and/or call default
	
if (pHelpInfo->iContextType == HELPINFO_WINDOW)
	{
	AfxGetApp()->WinHelp (pHelpInfo->dwContextId, HELP_CONTEXTPOPUP);
	}

return TRUE;
}


