/************************************************************************
 *  						C a l c . c p p	  							*
 ************************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "math.h"							// for: fabs

#include "PF.h"
#include "InitDoc.h"         

#include "WayDoc.h"
#include "PlaneDoc.h"
#include "MetDoc.h"
#include "Calc.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

extern CDimDoc* 		ptDim;
extern CMetDoc*			ptMet;
extern CCalcDoc*		ptCalc;


/////////////////////////////////////////////////////////////////////////////
// CStructList

/************************************************************************
 *  StructLi.cpp				CStructList					Constructor	*
 ************************************************************************/
template <class Type> 
CStructList<Type>::CStructList()
{

}


/************************************************************************
 *	StructLi.cpp				CStructList  Destructor					*
 ************************************************************************/
template <class Type>
CStructList<Type>::~CStructList()
{ 
this->RemoveAll();
}


/************************************************************************
 *  StructLi.cpp					G e t S i z e		 				*
 ************************************************************************/
template <class Type>
long CStructList<Type>::GetSize ()
{
long lSize = m_Array.GetSize();
return lSize;
}

/************************************************************************
 *  StructLi.cpp						A d d			 				*
 ************************************************************************/
template <class Type>
BOOL CStructList<Type>::Add (Type Entry)
{
BOOL	bAdded = FALSE;
Type*	ptParams = NULL;

ptParams = new Type;	   		// generate new Resolution struct
if (ptParams != NULL)
	{
//	TRACE ("CStructList::Add new Type\n");
	*ptParams = Entry;
	m_Array.Add (ptParams);
	bAdded = TRUE;
	}	

return bAdded;
}

/************************************************************************
 *  StructLi.cpp				I n s e r t A t			 				*
 ************************************************************************/
template <class Type>
BOOL CStructList<Type>::InsertAt (long i, Type Entry)
{
BOOL	bAdded = FALSE;
Type*	ptParams = NULL;

ptParams = new Type;	   		// generate new Resolution struct
if (ptParams != NULL)
	{
//	TRACE ("CStructList::InsertAt new Type %d\n", i);
	*ptParams = Entry;
	m_Array.InsertAt (i, ptParams);
	bAdded = TRUE;
	}	

return bAdded;
}

/************************************************************************
 *  StructLi.cpp						S e t A t		 				*
 ************************************************************************/
template <class Type>
BOOL CStructList<Type>::SetAt (long i, Type Entry)
{
BOOL	bOK = FALSE;
Type*	ptParams = NULL;

if (i>=0 && i<m_Array.GetSize())
	{
	Type* ptParams = (Type*)m_Array.GetAt (i);
	if (ptParams != NULL)
		{
		*ptParams = Entry;
		bOK = TRUE;
		}	
	}
return bOK;
}

/************************************************************************
 *  StructLi.cpp					R e m o v e A t						*
 ************************************************************************/
template <class Type>
BOOL CStructList<Type>::RemoveAt (long i)
{
BOOL	bOK = FALSE;
Type*	ptParams = NULL;

if (i>=0 && i<m_Array.GetSize())
	{
	Type* ptParams = (Type*)m_Array.GetAt (i);
	if (ptParams != NULL)
		{
//		TRACE ("CStructList::RemoveAt %d\n", i);
		delete ptParams; 
		m_Array.RemoveAt (i);
		bOK = TRUE;
		}	
	}
return bOK;
}
 
/************************************************************************
 *  StructLi.cpp						G e t A t						*
 ************************************************************************/
template <class Type>
BOOL CStructList<Type>::GetAt (long i, Type* ptStruct)
{
BOOL	bOK = FALSE;

if (i>=0 && i<m_Array.GetSize())
	{
	Type* ptParams = (Type*)m_Array.GetAt (i);
	if (ptParams != NULL)
		{
		*ptStruct	= *ptParams;
		bOK = TRUE;
		}	
	}
return bOK;
}

/************************************************************************
 *  StructLi.cpp						G e t A t						*
 ************************************************************************/
template <class Type>
Type CStructList<Type>::GetAt (long i)
{
Type Entry;
Type*	ptParams = NULL;

memset (&Entry, 0, sizeof (Type));

if (i>=0 && i<m_Array.GetSize())
	{
	Type* ptParams = (Type*)m_Array.GetAt (i);
	if (ptParams != NULL)
		{
		Entry = *ptParams;
		}	
	}
return Entry;
}



/************************************************************************
 *  StructLi.cpp				R e m o v e A l l		 				*
 ************************************************************************/
template <class Type>
void CStructList<Type>::RemoveAll ()
{
int i, nEntryCnt;	
											
nEntryCnt = m_Array.GetSize();
for (i=0; i<nEntryCnt; i++)
	{
	Type* ptEntry = NULL;
	ptEntry = (Type*)m_Array.GetAt(i);
	if (ptEntry != NULL)
		{
//		TRACE ("CStructList::RemoveAll %d\n", i);
		delete ptEntry;				// delete original element
		}
	}
m_Array.RemoveAll();
}



/////////////////////////////////////////////////////////////////////////////
// CCalculate
#define pi 3.1415926

/************************************************************************
 *  Calc.cpp			Geschwindigkeit ueber Grund						*
 ************************************************************************/
double Vg (int WiRi, int Vw, double Ve, double RWK)
{
double p,Ww;

Ww = (double)WiRi*pi/DB180;
p  = (double)Vw * cos(RWK*pi/DB180-Ww);

return -p + sqrt(p*p + Ve*Ve - (double)Vw*(double)Vw);
}

/************************************************************************
 *  Calc.cpp					Luv - Winkel							*
 ************************************************************************/
double Luv (int WiRi, int Vw, int Ve, double RWK, double Vgr)
{
double VeRi, Ww, Luv, help;

				/* Richtung der Eigengeschwindigkeit	*/
Ww = (double)WiRi*pi/DB180;

help = (Vgr*cos(RWK*pi/DB180) + (double)Vw*cos(Ww)) / (double)Ve;
if (help > 1.) help = 1.;
if (help < -1.) help = -1.;
VeRi = acos(help);
if ( (Vgr*sin(RWK*pi/DB180) + Vw*sin(Ww)) < 0.) VeRi=2*pi-VeRi;

Luv = VeRi*DB180/pi - RWK;		/* in degree		*/
if (Luv > DB180) Luv -= DB360;
if (Luv < -DB180) Luv += DB360;
return Luv;
}


/************************************************************************
 *  Calc.cpp				CCalculate				Constructor		*
 ************************************************************************/
CCalculate::CCalculate(CWayDoc* ptWayDoc)
{ 
m_ptPlane	= NULL;
m_dTotalDist = m_dTotalTime = m_dFuel = 0.;

m_ptWayDoc	= ptWayDoc;	 
}


/************************************************************************
 *  Calc.cpp			~CCalculate						Destructor		*
 ************************************************************************/
CCalculate::~CCalculate()
{
} 

/************************************************************************
 *  Calc.cpp					A c t i v a t e							*
 ************************************************************************/
void CCalculate::Activate (CPlane* ptPlane)
{
m_ptPlane	= ptPlane; 
m_bChanged = m_ptWayDoc->IsModified();

										// initialization
m_dTotalDist = m_dTotalTime = m_dFuel = 0.;
m_Legs.RemoveAll();
m_AlternateLegs.RemoveAll();

memset (&m_Cons, 0, sizeof (CONSUMTYPE));

											// if Insert UpDown:
											// InsertUpDown may have changed ActAlt	  and
	if (!m_ptWayDoc->IsUpDown())			// CWayDoc::InsertUpDown has already called SetPlaneSpeed
		this->SetPlaneSpeed (*ptPlane);		// SetPlaneSpeed sets ActAlt = PlAlt

						// important here to calculate temporarily new altitudes
	if (m_ptWayDoc->IsElev() || m_ptWayDoc->IsSemi()) 
		{
		if (m_ptWayDoc->IsUpDown())	// if (Elev || Semi) AND UpDown activated:
			{
			m_ptWayDoc->UpdateUpDown();	// update Elev, Semi and UpDown
			}
		else{
			if (m_ptWayDoc->IsElev())
				m_ptWayDoc->DoCalcElev();	// calculate Elev only

			if (m_ptWayDoc->IsSemi())
				this->InsertSemi ();		// calculate Semi only
			}
		}

}

/************************************************************************
 *  Calc.cpp					D e A c t i v a t e						*
 ************************************************************************/
void CCalculate::DeActivate ()
{
m_ptWayDoc->SetModifiedFlag(m_bChanged);
}


/************************************************************************
 *  Calc.cpp				S e t P l a n e	S p e e d					*
 ************************************************************************/
void CCalculate::SetPlaneSpeed(CPlane& Plane)
{
	short		i;
	CWayPoint	Wpt;
	double		fConvert;


	short nSpeedDim  = Plane.GetSpeedDim();

	short nCnt= m_ptWayDoc->GetWayPointCnt ();
	for (i=0; i<nCnt; i++)
    {
		m_ptWayDoc->GetWayPointPtr (i, &Wpt);

										/* copy PlAlt [x] into ActAlt [ft]	*/
		long lActAlt_ft = (long)ptDim->ConvertDist (Wpt.GetPlannedAlt(),
													Wpt.GetAltDim(), 
													DIM_FEET);
		Wpt.SetActAlt_ft(lActAlt_ft);

										// get plane speed at this way point
		float fCruiseSpd = Plane.GetCruiseSpeed();
		if (Wpt.IsUp())
			fCruiseSpd = Plane.GetClimbSpeed ();
		if (Wpt.IsDown())
			fCruiseSpd = Plane.GetDescSpeed ();

		fConvert = ptDim->ConvertSpeed (fCruiseSpd, nSpeedDim, DIM_KT);
		Wpt.SetActIAS_kt((short)(fConvert + 0.5));

		m_ptWayDoc->ChangeWpt (i, Wpt);
    }

	if (m_ptWayDoc->HasAlternates())
	{
		for (i=1; i<=2; i++)
		{
			if (m_ptWayDoc->GetAlternateWayPointPtr(i, &Wpt))
			{						
				float fCruiseSpd = Plane.GetCruiseSpeed();

				fConvert = ptDim->ConvertSpeed (fCruiseSpd, nSpeedDim, DIM_KT);
				Wpt.SetActIAS_kt((short)(fConvert + 0.5));
				m_ptWayDoc->SetAlternateWayPointPtr(i, Wpt);// store plane speed in Alternate Wpt
			}
		}
	}
}


/************************************************************************
 *  Calc.cpp			I n s e r t    O r t h o d r o m e				*
 ************************************************************************/
void CCalculate::InsertOrtho ()
{
CWayPoint	WayA, WayB, WayX;
CVektor A,B, X;			/* euklidische Koordinaten	*/
double wab, wax, dw;
double DIST, RWK, fActivDist;
int i,j, nOrtho;

/* activate calculation for distances > "fActivDist" NM (default: 1000 NM)*/

short nOrthoDim = ptCalc->GetOrthoDim();
long lOrthoDiff = ptCalc->GetOrthoDiff();
fActivDist = ptDim->ConvertDist (lOrthoDiff, nOrthoDim, DIM_NM);

i=1;

while (i < m_ptWayDoc->GetWayPointCnt ())
	{
	m_ptWayDoc->GetWayPointPtr (i-1, &WayA);
	m_ptWayDoc->GetWayPointPtr (i, &WayB);

									/* get distance between A and B */
    DIST = WayA.Distance (WayB, &RWK);

    if (DIST > fActivDist)			/* NM			*/
		{
		nOrtho = (int)(DIST/fActivDist);	/* Anzahl der Grosskreis-Punkte	*/
						/* A,B:to calculate insert point*/
		WayA.ToXYZ (&A);		   /* Koord Ort A	*/
		WayB.ToXYZ (&B);		   /* Koord Ort B	*/
		
		wab = A.VektorAngle (B);
		dw = wab / (nOrtho+1);
		
		for (j = 0; j<nOrtho; j++)
			{
			wax = (j+1) * dw;		/* X : Ort nach Winkel wax	*/ 
			X = A.GrossKreis (B, wax);

			WayX.FirstInit();
			WayX.FromXYZ (X);

			WayX.UseLatLonName();
			WayX.SetOrtho(TRUE);


								/* copy ActAlt [ft] into PlAlt [x]	*/
			long lActAlt_x = (long)ptDim->ConvertDist (WayA.GetActAlt_ft(),
												DIM_FEET,
												WayA.GetAltDim());
			WayX.SetPlannedAlt (lActAlt_x);
			WayX.SetAltDim (WayA.GetAltDim());

			WayX.SetActAlt_ft(  WayA.GetActAlt_ft());
			WayX.SetActIAS_kt(  WayA.GetActIAS_kt());
			WayX.SetVariation ( WayA.GetVariation());

			m_ptWayDoc->InsertWpt (i, WayX);
			i++;		/* effectiv: i+=2, see next i++!!	*/
			}
		}
		i++;
	} /* while i<cnt...	*/
}

/************************************************************************
 *  Calc.cpp				 D e l e t e O r t h o						*
 ************************************************************************/
void CCalculate::DeleteOrtho ()
{
CWayPoint	theWay;

int i=1;						/* i=0 contains start field	*/
while (i < m_ptWayDoc->GetWayPointCnt ())
	{
	m_ptWayDoc->GetWayPointPtr (i, &theWay);

	if (theWay.IsOrtho())
		m_ptWayDoc->DeleteWpt (i);
	else	i++;
	}
}

/************************************************************************
 *  Calc.cpp		I n s e r t    S e m i	C i r c u l a r				*
 ************************************************************************/
void CCalculate::InsertSemi ()
{
CWayPoint	WayA, WayB;
int	i;
long	lNewAlt_ft, lActivLevel_ft;
double	fConvert, DIST, RWK, MC;

/* activate semi circular rules above "lActivLevel_ft" (default: 2500 ft)*/
long lSemiStart = ptCalc->GetSemiStart();
short	nSemiDim= ptCalc->GetSemiDim();
fConvert = ptDim->ConvertDist (lSemiStart, nSemiDim, DIM_FEET);
lActivLevel_ft = (long)(fConvert + 0.5);

i=1;
short nCnt= m_ptWayDoc->GetWayPointCnt ();
while (i < nCnt)
	{
   	m_ptWayDoc->GetWayPointPtr (i-1, &WayA);
	m_ptWayDoc->GetWayPointPtr (i, &WayB);

	if (WayB.GetActAlt_ft() == NO_ALT || WayA.GetActAlt_ft() == NO_ALT)
		return;

					/* get distance between A and B */
	DIST = WayA.Distance(WayB, &RWK);

    if (WayB.GetActAlt_ft() > lActivLevel_ft)	    /* ft			*/
		{
		MC = RWK - WayB.GetVariation();	/* missweisender Kurs (magn.c.)	*/
		if (MC >= 0. && MC <= 179.)
				lNewAlt_ft = 3500L;		/* 1. Halbkreis			*/
		else	lNewAlt_ft = 2500L;		/* 2. Halbkreis			*/

		while (lNewAlt_ft < WayB.GetActAlt_ft()) lNewAlt_ft += 2000L;

		WayB.SetActAlt_ft(lNewAlt_ft);
		m_ptWayDoc->ChangeWpt (i, WayB);
		}
   i++;
   } /* while i<nCnt...	*/
}

/************************************************************************
 *  wNavFunc.c		       I n s e r t U p D o w n						*
 ************************************************************************/
BOOL CCalculate::InsertUpDown (CPlane* ptPlane, CMetDoc* ptMet)
{ 
BOOL bChanged = FALSE;
float	VrFl;				/* Geschw. im Reiseflug [kt]	*/
float	VstFl;				/* Geschw. im Steigflug [kt]	*/
float	Vst;				/* Steiggeschw. [ft/min]		*/

CWayPoint	WayA, WayB, WayX;
CVektor		A,B, X;			/* euklidische Koordinaten		*/
double wax, DIST;
int i;						/* index						*/
float  VsiFl, Vsi;			/* Vsinkfl, Sinkgeschw. 		*/
double dH, EPS_H;			/* Hoehendifferenz A,B [ft]		*/
double Dc, EPS_D;			/* Dist to fly in const. height */
double Dsteig, Dsink;		/* Strecke zum Steigen, Sinken	*/ 
double RWK, Vgr;

		/* calculate if dH >= EPS_H feet (default: 1500 ft)	*/
long lUpDownDiff = ptCalc->GetUpDownDiff ();
short nUpDownDim = ptCalc->GetUpDownDim();
EPS_H = ptDim->ConvertDist (lUpDownDiff, nUpDownDim, DIM_FEET);
EPS_D = 0.1;				/* insert or reduce if Dc>0.1NM	*/

ptPlane->GetHorzSpeed_kt (&VstFl, &VrFl, &VsiFl);   /* in kt	    */
ptPlane->GetVertSpeed_ftpm (&Vst, &Vsi);			/* in ft/min	*/

i=1;
while (i < m_ptWayDoc->GetWayPointCnt ())
    {
	m_ptWayDoc->GetWayPointPtr (i-1, &WayA);
	m_ptWayDoc->GetWayPointPtr (i, &WayB);

	if (WayB.GetActAlt_ft() == NO_ALT || WayA.GetActAlt_ft() == NO_ALT)
		return bChanged;

    dH = (double) (WayB.GetActAlt_ft() - WayA.GetActAlt_ft());		/* ft		*/

    if (fabs(dH) >= EPS_H)		/* moving up or down			*/
		{						/* get distance between A and B */
		DIST = WayA.Distance(WayB, &RWK);
								/* define WayB.W_Dir, .W_Speed	*/
		ptMet->SetWind (&WayB, WayA.GetActAlt_ft());
								/* A,B:to calculate insert point*/
		WayA.ToXYZ(&A);							/* Koord Ort A	*/
		WayB.ToXYZ(&B);							/* Koord Ort B	*/
		
		if (dH > 0.)
			{				/* moving up !!			*/
			Vgr = Vg(WayB.GetWindDir(), WayB.GetWindSpeed_kt(), VstFl, RWK);
			Dsteig = Vgr * dH/(double)Vst/60; 	
			Dc  = DIST - Dsteig;
			if (fabs(Dc) > EPS_D)
 				{
				if (Dc > 0.)
					{						/* I N S E R T			*/
					wax = Dsteig/60;		/* X : Ort nach Winkel wax	*/ 
					X = A.GrossKreis (B, wax*pi/DB180);		
					WayX.FirstInit ();
					WayX.FromXYZ (X);

					WayX.UseLatLonName();
					WayX.SetUp(TRUE);

									/* copy ActAlt [ft] into PlAlt [x]	*/
					long lActAlt_x = (long)ptDim->ConvertDist (WayB.GetActAlt_ft(),
														DIM_FEET,
														WayB.GetAltDim());
					WayX.SetPlannedAlt (lActAlt_x);
					WayX.SetAltDim (WayB.GetAltDim());

					WayX.SetActAlt_ft(  WayB.GetActAlt_ft());
					WayX.SetActIAS_kt((short)VstFl);
					WayX.SetVariation ( WayB.GetVariation());

					m_ptWayDoc->InsertWpt (i, WayX);
					i++;		/* effectiv: i+=2, see next i++!!	*/
					}
				else{			/* reduce height of B		*/
					WayB.SetActAlt_ft (WayA.GetActAlt_ft() + (long)(60L * Vst*DIST/VstFl));
					WayB.SetActIAS_kt ((short)VstFl);
					m_ptWayDoc->ChangeWpt (i, WayB);
					}	/* Dc > 0	*/
				bChanged = true;
				}		/* Dc > EPS_D	*/
			}
		else{							/* (dH < 0): moving down !!		*/
			Vgr = Vg(WayB.GetWindDir(), WayB.GetWindSpeed_kt(), VsiFl, RWK);
			Dsink = Vgr * dH/(double)Vsi/60;
			Dc = DIST - Dsink;
			if (fabs(Dc) > EPS_D)
				{
				if (Dc > 0.)
					{			/* I N S E R T			*/
					WayB.SetActIAS_kt((short)VsiFl);
					m_ptWayDoc->ChangeWpt (i, WayB);
					wax = Dsink/60;		/* X : Ort nach Winkel wax	*/ 
					X = B.GrossKreis (A, wax*pi/DB180);


					WayX.FirstInit ();
					WayX.FromXYZ (X);

					WayX.UseLatLonName();
					WayX.SetDown(TRUE);

									/* copy ActAlt [ft] into PlAlt [x]	*/
					long lActAlt_x = (long)ptDim->ConvertDist (WayA.GetActAlt_ft(),
														DIM_FEET,
														WayA.GetAltDim());
					WayX.SetPlannedAlt (lActAlt_x);
					WayX.SetAltDim (WayA.GetAltDim());
					WayX.SetActAlt_ft(WayA.GetActAlt_ft());

					WayX.SetActIAS_kt((short)VsiFl);
					WayX.SetVariation (WayA.GetVariation());

					m_ptWayDoc->InsertWpt (i, WayX);
					i++;		/* effectiv: i+=2, see next i++!!	*/
					}
				else{			/* reduce height of A		*/
					WayA.SetActAlt_ft(WayB.GetActAlt_ft() - (long)(60L * Vsi*DIST/VsiFl));
					WayA.SetActIAS_kt((short)VsiFl);
					m_ptWayDoc->ChangeWpt (i-1, WayA);
					if (i>1)
					   {			/* del inserted positions and	*/
					   this->DeleteUpDown ();	/* calculate from beginning	*/
					   i = 0;		/* effectiv: i=1, see next i++!	*/
					   }
					}	/* Dc > 0	*/
				bChanged = TRUE;
				}		/* Dc > EPS_D	*/
			}	/* move up or down	*/
		}	/* dh >= EPS_H	*/
	i++;
	}
	return bChanged;
}

/************************************************************************
 *  Calc.cpp				   D e l e t e U p D o w n					*
 ************************************************************************/
void CCalculate::DeleteUpDown ()
{
CWayPoint	theWay;

int i=1;						/* i=0 contains start field	*/
while (i < m_ptWayDoc->GetWayPointCnt ())
	{
	m_ptWayDoc->GetWayPointPtr (i, &theWay);

	if (theWay.IsUpDown())
		m_ptWayDoc->DeleteWpt (i);
	else	i++;
	}

}

/************************************************************************
 *  Calc.cpp				C o u r s e T i m e							*
 ************************************************************************/
void CCalculate::CourseTime()
{
	long	ActAlt0;
	double	Lat0, Lon0;

	short i;
	short nCnt= m_ptWayDoc->GetWayPointCnt ();
	for (i=0; i<nCnt; i++)
    {
		CWayPoint Wpt;
		if (m_ptWayDoc->GetWayPointPtr (i, &Wpt))
		{								// add std way points in m_Legs
			if (i > 0)
			{
				ptMet->SetWind (&Wpt, ActAlt0);
				m_ptWayDoc->ChangeWpt (i, Wpt);			// store Wind Data in Wpt

				LEGTYPE Leg;
				Leg = this->FlugDaten (Lat0, Lon0, &Wpt);
				m_Legs.Add(Leg);
			}

			Lat0 = Wpt.GetLat();				/* save 1. coordinates [degree]	*/
			Lon0 = Wpt.GetLon();
			ActAlt0 = Wpt.GetActAlt_ft();		/* save old height to calc. wind*/
		}
	}
}


/************************************************************************
 *  Calc.cpp		A l t e r n a t e C o u r s e T i m e				*
 *  calls FlugDaten, which increases m_dTotalDist, m_dTotalTime			*
 *			and creates LEGTYPE for each alternate						*
 ************************************************************************/
void CCalculate::AlternateCourseTime()
{
	if (m_ptWayDoc->HasAlternates())
	{									// add alternates in m_AlternateLegs
		long	ActAlt0;
		double	Lat0, Lon0;


		short nCnt= m_ptWayDoc->GetWayPointCnt ();
		CWayPoint Wpt;
		if (m_ptWayDoc->GetWayPointPtr (nCnt-1, &Wpt))
		{				// get last regular way point
			Lat0 = Wpt.GetLat();				/* save 1. coordinates [degree]	*/
			Lon0 = Wpt.GetLon();
			ActAlt0 = Wpt.GetActAlt_ft();		/* save old height to calc. wind*/
		}


		for (int i=1; i<=2; i++)
		{
			CWayPoint Wpt;
			if (m_ptWayDoc->GetAlternateWayPointPtr(i, &Wpt))
			{
				ptMet->SetWind (&Wpt, ActAlt0);
				m_ptWayDoc->SetAlternateWayPointPtr(i, Wpt);// store Wind Data in Alternate Wpt

				LEGTYPE Leg;
				Leg = this->FlugDaten (Lat0, Lon0, &Wpt);
				m_AlternateLegs.Add(Leg);

				Lat0 = Wpt.GetLat();				/* save 1. coordinates [degree]	*/
				Lon0 = Wpt.GetLon();
				ActAlt0 = Wpt.GetActAlt_ft();		/* save old height to calc. wind*/
			}
		}
	}
}



/************************************************************************
 *  Calc.cpp				F l u g D a t e n 							*
 *  Exact values in: dDistLeg_NM, dTimeLeg_min, dVgr, dMC, dRWK			*
 *  Rounded values in: dDistLeg, dTimeLeg, m_dTotalDist, m_dTotalTime	*
 *			dWCA, dMH													*
 ************************************************************************/
LEGTYPE CCalculate::FlugDaten(double dLatOld, double dLonOld, CWayPoint* ptWptNew)
{ 
CLatLon OldPos (dLatOld, dLonOld);
LEGTYPE Leg;


Leg.dDistLeg_NM = OldPos.LoxoDist (ptWptNew->GetLat(), ptWptNew->GetLon(), (double FAR*)&Leg.dRWK);
Leg.dDistLeg    = floor (Leg.dDistLeg_NM + 0.5);	
m_dTotalDist	+= Leg.dDistLeg;

Leg.dVgr	    = Vg (ptWptNew->GetWindDir(),
					ptWptNew->GetWindSpeed_kt(),
					ptWptNew->GetActIAS_kt(), Leg.dRWK);
Leg.dWCA	    = floor (Luv( ptWptNew->GetWindDir(), 
							ptWptNew->GetWindSpeed_kt(), 
							ptWptNew->GetActIAS_kt(), Leg.dRWK, Leg.dVgr) + 0.5);
					/* Flugzeit in Minuten		*/
Leg.dTimeLeg_min= 60 * Leg.dDistLeg_NM / Leg.dVgr;		// exact value

double dRoundedTime_min = 60 * Leg.dDistLeg / Leg.dVgr;
Leg.dTimeLeg    = floor (dRoundedTime_min + 1);	/* +1: aufrunden*/
m_dTotalTime	+= Leg.dTimeLeg; 

Leg.dMC			= Leg.dRWK - ptWptNew->GetVariation();	/* missweisender Kurs	*/
Leg.dMH 	    = floor (Leg.dMC + Leg.dWCA + 0.5);
if (Leg.dMH < 0.) Leg.dMH += DB360;		/* missweisender Steuerkurs	*/

return Leg;
}


/************************************************************************
 *  Calc.cpp					G e t A t				 				*
 *  Input: Index of rte waypoint: first wpt=0 has no leg info!!			*
 ************************************************************************/
LEGTYPE CCalculate::GetAt (short nIndex)
{
	LEGTYPE Leg;

	_fmemset (&Leg, 0, sizeof (LEGTYPE));

	nIndex--;				// there is no entry for first waypoint!!

	if (nIndex >= 0 && nIndex < m_Legs.GetSize())
	{
		Leg = m_Legs.GetAt (nIndex);
	}

	return Leg;
}

/************************************************************************
 *  Calc.cpp				G e t A l t e r n a t e A t					*
 *  Input: Index of alternate waypoint: may be 0 or 1					*
 ************************************************************************/
LEGTYPE CCalculate::GetAlternateAt (short nIndex)
{
	LEGTYPE Leg;

	_fmemset (&Leg, 0, sizeof (LEGTYPE));

	if (nIndex >= 0 && nIndex < m_AlternateLegs.GetSize())
	{
		Leg = m_AlternateLegs.GetAt (nIndex);
	}

	return Leg;
}


/************************************************************************
 *  Calc.cpp			    WeightAndBalance							*
 ************************************************************************/
void CCalculate::WeightAndBalance (CPlane* ptPlane,
						CLoadDoc* ptLoad, float* ptActUseFuel_kg)
{
float	fRow_kg[PL_MAX_ARM], fLug_kg[PL_MAX_ARM];

				/* defines fRow_kg, fLug_kg		*/
ptLoad->ConvertLoading (fRow_kg, fLug_kg, PL_MAX_ARM);

				/* defines ptActUseFuel_kg		*/
m_Cons.fTotal_kg = ptPlane->GetActMass_kg (fRow_kg, fLug_kg,
									ptLoad, ptActUseFuel_kg);

m_Cons.fTotal_mkg = ptPlane->GetActTorq_mkg (fRow_kg, fLug_kg, ptActUseFuel_kg);

				/* aktuelle Schwerpunktlage		*/
m_Cons.fCoG_m	= m_Cons.fTotal_mkg/m_Cons.fTotal_kg;

m_Cons.fCoG_m = (float)floor(m_Cons.fCoG_m * 100 + .5)/100;
}

/************************************************************************
 *  Calc.cpp				ActualizeTimeAndFuel						*
 ************************************************************************/
void CCalculate::ActualizeTimeAndFuel (float fFuel_l, float fActCons)
{   
	if (fFuel_l > m_Cons.fExtra_l)
		fFuel_l = m_Cons.fExtra_l;	    /* we use last fuel !!	*/
	m_Cons.fPowered_min += (fFuel_l * 60 / fActCons);
	m_Cons.fExtra_l -= fFuel_l;
}


/************************************************************************
 *  Calc.cpp				C a l c u l a t e F u e l					*
 ************************************************************************/
BOOL CCalculate::CalculateFuel (CPlane* ptPlane, float* ptActUseFuel_kg)
{
	short	i;
	float	fTotalUseFuel_kg;
	float	fFuel_l, fActCons;

					/* total mass of all usable fuel in kg	*/
	fTotalUseFuel_kg = (float)0;
	for (i=0; i<PL_MAX_TANK; i++)
		fTotalUseFuel_kg += *(ptActUseFuel_kg + i);

					/* total vol. of all usable fuel in l	*/
	m_Cons.fExtra_l = fTotalUseFuel_kg / (float)0.72;

	m_Cons.fEnRoute_l = (float)0;
	m_Cons.fPowered_min = (float)0;

	this->CourseTime();

	long	lActAlt0;
	short nCnt= m_ptWayDoc->GetWayPointCnt ();
	for (i=0; i<nCnt; i++)			
    {
		CWayPoint Wpt;
		if (m_ptWayDoc->GetWayPointPtr (i, &Wpt))
		{
			if (i>0)
			{					
				LEGTYPE Leg = this->GetAt (i);	// get calculated leg

								/* calculate actual fuel consumption    */
				fActCons = ptPlane->GetActCons (m_ptWayDoc->IsUpDown(), Wpt.GetActAlt_ft(), lActAlt0);
				if (fActCons == NO_FUEL)
					return FALSE;

							/* calculate used fuel [l]			*/
				fFuel_l = fActCons * (float)Leg.dTimeLeg/60;
				m_Cons.fEnRoute_l += fFuel_l;	    /* required fuel		*/
				this->ActualizeTimeAndFuel (fFuel_l, fActCons);
			}

			lActAlt0 = Wpt.GetActAlt_ft();		/* save old height to calc. wind*/
		}
	}

				    /*	wind and fuel			*/

	m_Cons.fEnRoute_NM  = m_dTotalDist;
	m_Cons.fEnRoute_min = m_dTotalTime;
	m_Cons.fEnRoute_l   = (float)floor(m_Cons.fEnRoute_l * 100 + .5)/100;


	if (m_ptWayDoc->HasAlternates())
	{
		m_Cons.fAlternateRoute_l	= 0;
		m_Cons.fAlternateRoute_min	= 0;

		AlternateCourseTime();		// increases m_dTotalDist, m_dTotalTime for alternates


		for (int i=0; i<2; i++)
		{			
			m_Cons.forAlternate[i].fRoute_NM	= 0;
			m_Cons.forAlternate[i].fRoute_min	= 0;
			m_Cons.forAlternate[i].fRoute_l		= 0;

			CWayPoint Wpt;				// alternate waypoint array starts with 1
			if (m_ptWayDoc->GetAlternateWayPointPtr(i+1, &Wpt))
			{							// alternate leg array starts with 0
				LEGTYPE Leg = this->GetAlternateAt (i);	// get calculated leg of alternate
										/* calculate actual fuel consumption    */
				fActCons = ptPlane->GetActCons (m_ptWayDoc->IsUpDown(), Wpt.GetActAlt_ft(), lActAlt0);
				if (fActCons == NO_FUEL)
					return FALSE;

							/* calculate used fuel [l]			*/
				fFuel_l = fActCons * (float)Leg.dTimeLeg/60;

				m_Cons.forAlternate[i].fRoute_min	= Leg.dTimeLeg;
				m_Cons.forAlternate[i].fRoute_l		= fFuel_l;
				m_Cons.forAlternate[i].fRoute_NM	= Leg.dDistLeg;
				m_Cons.forAlternate[i].LandLoc		= (CLocation)Wpt;

				this->ActualizeTimeAndFuel(fFuel_l, fActCons);

				m_Cons.fAlternateRoute_l	+= fFuel_l;	    /* required fuel		*/
				m_Cons.fAlternateRoute_min	+= Leg.dTimeLeg;

				lActAlt0 = Wpt.GetActAlt_ft();		/* save old height to calc. wind*/
			}
		}
	}

	return TRUE;
}

/************************************************************************
 *  Calc.cpp			  R e q u i r e d R W Y							*
 ************************************************************************/
void CCalculate::RequiredRWY (CPlane* ptPlane, CMetDoc* ptMet)
{
short	nRwyLen;
double	PressAlt, F, SumRoll, SumObst;
double	std, diff, RwySlope;
double	fMinAlt_ft, fRwyLen_m, fElevDiff_m, fTemp, fQNH;

					/* S t a r t		*/
if (ptMet->GetPressDim1()==DIM_INCHHG) fQNH = (double)ptMet->GetQNH1()/100;
								 else  fQNH = (double)ptMet->GetQNH1();
fQNH = ptDim->ConvertPress (fQNH, ptMet->GetPressDim1(), DIM_HPA);
fTemp = ptDim->ConvertTemp ((double)ptMet->GetTemp1(), ptMet->GetTempDim1(), DIM_C);

m_Cons.nStartRoll_m = 0;
m_Cons.nStartObst_m = 0;
if ( (m_Cons.StartLoc.GetAltitude() != NO_ALT) && (m_Cons.StartLoc.GetRwyLen(&nRwyLen)) )
    {
    ptPlane->GetStartBasis_m (m_Cons.fTotal_kg, &SumRoll, &SumObst);

    fMinAlt_ft = ptDim->ConvertDist ((double)m_Cons.StartLoc.GetAltitude(), m_Cons.StartLoc.GetElevDim(), DIM_FEET);
    PressAlt = fMinAlt_ft + (1013 - fQNH)*30.;		/*30 ft/hPa*/
    if (PressAlt >  3000.) F=0.18;	    /* Hoehenzuschalg	    */
    if (PressAlt <= 3000.) F=0.13;
    if (PressAlt <= 1000.) F=0.10;
    SumRoll += SumRoll * F * PressAlt/1000;
    SumObst += SumObst * F * PressAlt/1000;

    std = 15 - 2 * PressAlt/1000;	    /* -2 Grad C / 1000 ft  */
    diff = fTemp - std;
    SumRoll += SumRoll * (0.01 * diff);     /* Tempzuschlag 1%/Grd  */
    SumObst += SumObst * (0.01 * diff);     /* Tempzuschlag 1%/Grd  */
					    /* Slope of runway [%]  */

    fElevDiff_m = ptDim->ConvertDist ((double)m_Cons.StartLoc.GetElevDiff(), m_Cons.StartLoc.GetDiffDim(), DIM_METER);
    fRwyLen_m	= ptDim->ConvertDist ((double)nRwyLen, m_Cons.StartLoc.GetLenDim(), DIM_METER);
    RwySlope = 100 * fElevDiff_m/fRwyLen_m;
    SumRoll += SumRoll * 0.1 * RwySlope;    /* +10% / 1% Steigung   */
    SumObst += SumObst * 0.1 * RwySlope;    /* +10% / 1% Steigung   */

    if (m_Cons.StartLoc.IsGras())
		{
		SumRoll += SumRoll * .2;	    /* +20% wegen Grasbahn	*/
		SumObst += SumObst * .2;	    /* +20% wegen Grasbahn	*/
		}
    if (ptMet->IsWet1())
		{
		SumRoll += SumRoll * .30;	     /* +30% feucht-weich    */
		SumObst += SumObst * .30;	     /* +30% feucht-weich    */
		}
    m_Cons.nStartRoll_m = (int)SumRoll;
    m_Cons.nStartObst_m = (int)SumObst;
    }

						/* L a n d u n g	*/
if (ptMet->GetPressDim2()==DIM_INCHHG) fQNH = (double)ptMet->GetQNH2()/100;
							 	 else  fQNH = (double)ptMet->GetQNH2();
fQNH = ptDim->ConvertPress (fQNH, ptMet->GetPressDim2(), DIM_HPA);
fTemp = ptDim->ConvertTemp ((double)ptMet->GetTemp2(), ptMet->GetTempDim2(), DIM_C);

CalcLandingDistances(m_Cons.LandLoc, m_Cons.fLandMass_kg,
					 fTemp, fQNH, ptPlane,
					 &m_Cons.nLandRoll_m, &m_Cons.nLandObst_m);


	for (int i=0; i<2; i++)
	{		
		float fBaseMass;
		if (i==0)	fBaseMass = m_Cons.fLandMass_kg;
			else	fBaseMass = m_Cons.forAlternate[i-1].fLandMass_kg;

									/* Vol [l] * 0.72 kg/l	*/
		float fFuel_kg = m_Cons.forAlternate[i].fRoute_l * (float)0.72;
		m_Cons.forAlternate[i].fLandMass_kg = fBaseMass - fFuel_kg;

		CalcLandingDistances(m_Cons.forAlternate[i].LandLoc, 
					m_Cons.forAlternate[i].fLandMass_kg,
					 fTemp, fQNH, ptPlane,
					 &m_Cons.forAlternate[i].nLandRoll_m, 
					 &m_Cons.forAlternate[i].nLandObst_m);
	}
}

/************************************************************************
 *  Calc.cpp		    C a l c L a n d i n g D i s t a n c e s			*
 ************************************************************************/
void CCalculate::CalcLandingDistances(CLocation& Loc, float fLandMass_kg,
									  double fTemp, double fQNH, CPlane* ptPlane,
									 int* ptLandRoll_m, int* ptLandObst_m)
{
	short	nRwyLen;
	double	PressAlt, F, SumRoll, SumObst;
	double	std, diff, RwySlope;
	double	fMinAlt_ft, fRwyLen_m, fElevDiff_m;

	if ( (Loc.GetAltitude() != NO_ALT) && (Loc.GetRwyLen(&nRwyLen)) )
    {
		ptPlane->GetLandBasis_m (fLandMass_kg, &SumRoll, &SumObst);

		fMinAlt_ft = ptDim->ConvertDist ((double)Loc.GetAltitude(), Loc.GetElevDim(), DIM_FEET);
		PressAlt = fMinAlt_ft + (1013 - fQNH)*30.;		/*30 ft/hPa*/
		if (PressAlt >  3000.) F=0.027;	    /* Hoehenzuschalg	    */
		if (PressAlt <= 3000.) F=0.025;
		if (PressAlt <= 1000.) F=0.022;
		SumRoll += SumRoll * F * PressAlt/1000;
		SumObst += SumObst * F * PressAlt/1000;

		std = 15 - 2 * PressAlt/1000;	    /* -2 Grad C / 1000 ft  */
		diff = fTemp - std;
		SumRoll += SumRoll * (0.0022 * diff);   /* Zuschlag 0.22%/Grd   */
		SumObst += SumObst * (0.0022 * diff);   /* Zuschlag 0.22%/Grd   */

							/* Slope of runway [%]  */
		fElevDiff_m = ptDim->ConvertDist ((double)Loc.GetElevDiff(), Loc.GetDiffDim(), DIM_METER);
		fRwyLen_m	= ptDim->ConvertDist ((double)nRwyLen, Loc.GetLenDim(), DIM_METER);
		RwySlope = 100 * fElevDiff_m/fRwyLen_m;
		SumRoll += SumRoll * 0.1 * RwySlope;    /* +10% / 1% Geflle    */
		SumObst += SumObst * 0.1 * RwySlope;    /* +10% / 1% Geflle    */

		if (Loc.IsGras())
			{
			SumRoll += SumRoll * .2;	    /* +20% wegen Grasbahn  */
			SumObst += SumObst * .2;	    /* +20% wegen Grasbahn  */
			}
		if (ptMet->IsWet2())
			{
			SumRoll += SumRoll * .30;	     /* +30% feucht-weich    */
			SumObst += SumObst * .30;	     /* +30% feucht-weich    */
			}
		*ptLandRoll_m = (int)SumRoll;
		*ptLandObst_m = (int)SumObst;
    }
}



/************************************************************************
 *  Calc.cpp		    I s L o a d i n g F o r P l a n e O K			*
 ************************************************************************/
BOOL CCalculate::IsLoadingForPlaneOK(CLoadDoc* ptLoad, CPlane* ptPlane)
{
	BOOL bLoadingOK = TRUE;

						// for every weight entry, there must be an arm value!!
	short	i;
	float fNoArm = NO_ARM;

	float	fRow_kg[PL_MAX_ARM], fLug_kg[PL_MAX_ARM];

					/* defines fRow_kg, fLug_kg		*/
	ptLoad->ConvertLoading (fRow_kg, fLug_kg, PL_MAX_ARM);

	m_Cons.nBadRowLoad = 0;
	m_Cons.nBadLugLoad = 0;
	m_Cons.nBadFuelLoad = 0;

					/* check crew loading 		*/
	for (i=0; i<PL_MAX_ARM && m_Cons.nBadRowLoad==0; i++)
	{
		if (*(fRow_kg + i) != 0)
		{
			if (ptPlane->m_Arms.fRowArm[i] == fNoArm)
			{
				m_Cons.nBadRowLoad = i+1;
				bLoadingOK = FALSE;
			}
		}
	}

					/* check luggage loading 		*/
	for (i=0; i<PL_MAX_ARM && m_Cons.nBadLugLoad==0; i++)
	{
		if (*(fLug_kg + i) != 0)
		{
			if (ptPlane->m_Arms.fLugArm[i] == fNoArm)
			{
				m_Cons.nBadLugLoad = i+1;
				bLoadingOK = FALSE;
			}
		}
	}
	

					/* check fuel loading 		*/
	for (i=0; i<PL_MAX_TANK && m_Cons.nBadFuelLoad==0; i++)
	{
		if (ptLoad->GetTankFakt(i) > 0)
		{
			if (ptPlane->m_Tank.fArm[i] == fNoArm)
			{
				m_Cons.nBadFuelLoad = i+1;
				bLoadingOK = FALSE;
			}
		}
	}

	return bLoadingOK;

}


/************************************************************************
 *  Calc.cpp		       D o C o n s u m C a l c						*
 ************************************************************************/
BOOL CCalculate::DoConsumCalc (CPlane* ptPlane, CMetDoc* ptMet, CLoadDoc* ptLoad)
{
CWayPoint	firstWay, lastWay;
float		fActUseFuel_kg[PL_MAX_TANK];
double		fRwyLen_m;

if (!IsLoadingForPlaneOK(ptLoad, ptPlane))
	return FALSE;


	/* defines m_Cons.fTotal_kg, ->fTotal_mkg, ->fCoG_m and
		   fActUseFuel_kg[i]		*/
this->WeightAndBalance (ptPlane, ptLoad, fActUseFuel_kg);


m_Cons.bNoFuel = !this->CalculateFuel (ptPlane, fActUseFuel_kg);
  

m_Cons.fAvailFuel_l = (float)0;
if (!m_Cons.bNoFuel)     /* converts fActUseFuel_kg to ->fAvailFuel_l	*/
    {		       /* defines m_Cons.fTotalFuel_l, m_Cons.fFillDegr_Pc	*/
    m_Cons.fAvailFuel_l = ptPlane->GetFuelVol_l (fActUseFuel_kg,
		       m_Cons.fTotalFuel_l, m_Cons.fFillDegr_Pc);
    }

m_Cons.bBadBalance = ! ptPlane->BalanceOK (m_Cons.fTotal_kg, m_Cons.fCoG_m,
				 &m_Cons.fMinCoG_m, &m_Cons.fMaxCoG_m);

if (!m_Cons.bNoFuel)
    {
    float fFuel_l, fClimbCons, fCruiseCons, fDescCons;

    ptPlane->GetCons_l (&fClimbCons, &fCruiseCons, &fDescCons);
       
    m_Cons.nTaxi_min    = ptCalc->GetTaxiTime(); 	/* 10 min for taxiing				*/
    fFuel_l = fCruiseCons * m_Cons.nTaxi_min/60;
    m_Cons.fTaxi_l = fFuel_l;		    /* required fuel					*/
    this->ActualizeTimeAndFuel (fFuel_l, fCruiseCons);
                  
   	m_Cons.nDepArv_min  = ptCalc->GetDepArvTime();	/* 10 min for departure and arrival	*/
    fFuel_l = fCruiseCons * m_Cons.nDepArv_min/60;
    m_Cons.fDepArv_l = fFuel_l;		  /* required fuel					*/
    this->ActualizeTimeAndFuel (fFuel_l, fCruiseCons);
    
    m_Cons.fTaxiDepArv_l = (float)floor(10*(m_Cons.fTaxi_l + m_Cons.fDepArv_l)+.5)/10;	
    
    m_Cons.nSafeRes_min = ptCalc->GetSafeResTime();	/* 30 min for safety			*/
    fFuel_l = (float)floor(fCruiseCons * m_Cons.nSafeRes_min * 10/60 + .5)/10;
    m_Cons.fSafeRes_l = fFuel_l;		    /* required fuel		*/
    this->ActualizeTimeAndFuel (fFuel_l, fCruiseCons);


				/* Extrakraftstoff			*/
    m_Cons.nExtra_min = (int) (60. * m_Cons.fExtra_l / fCruiseCons);
    m_Cons.fPowered_min += m_Cons.nExtra_min;


				/* Erforderlicher Treibstoff fr	*/
				/* Erforderliche sichere Flugzeit	*/
    m_Cons.nReq_min = (int)(m_Cons.fEnRoute_min + m_Cons.fAlternateRoute_min + m_Cons.nDepArv_min + m_Cons.nTaxi_min);

						// Rundung
	long lAltRte10_l  = (long)((10*m_Cons.fAlternateRoute_l) + 0.5);
	long lRes10_l		= (long)((10*(m_Cons.fDepArv_l + m_Cons.fTaxi_l) + 0.5));
	float fAltRte_l = (float)lAltRte10_l/10;
	float fRes_l	= (float)lRes10_l/10;

	m_Cons.fReq_l   = m_Cons.fEnRoute_l + fAltRte_l + fRes_l;
	m_Cons.nSafe_min = m_Cons.nReq_min + m_Cons.nSafeRes_min;
    m_Cons.fSafe_l   = m_Cons.fReq_l + m_Cons.fSafeRes_l;

		/* Verfgbare sichere Flugzeit mit Extrakraftstoff	*/
    m_Cons.fMaxSafe_l   = m_Cons.fAvailFuel_l - m_Cons.fSafeRes_l;
    m_Cons.nMaxSafe_min = (int)(m_Cons.fPowered_min - m_Cons.nSafeRes_min + 0.5);
    if (m_Cons.fMaxSafe_l   < 0) m_Cons.fMaxSafe_l	 = 0;
    if (m_Cons.nMaxSafe_min < 0) m_Cons.nMaxSafe_min = 0;
    }

				    /* Landing Mass			*/
m_Cons.fLandMass_kg = m_Cons.fTotal_kg -		/* Vol [l] * 0.72 kg/l	*/
		    (m_Cons.fEnRoute_l + m_Cons.fDepArv_l + m_Cons.fTaxi_l) * (float)0.72;


if (m_Cons.fAvailFuel_l < m_Cons.fSafe_l)   /*	W a r n u n g e n	*/
    {				    /* are all tanks filled?		*/
    short i;
    BOOL    bAllFilled = TRUE;
    for (i=0; i<ptPlane->GetTankCnt (); i++)
	{
	if (m_Cons.fFillDegr_Pc[i] != (float)100)
	    bAllFilled = FALSE;
	}

    if (bAllFilled) m_Cons.bBadFuel1 = TRUE;
	else m_Cons.bBadFuel2 = TRUE;
    }

					/* Start- und Landestrecke	*/
m_ptWayDoc->GetWayPointPtr (0, &firstWay);
m_Cons.bBadElev1 = (firstWay.GetAltitude() == NO_ALT);
m_Cons.StartLoc = (CLocation)firstWay;

short nCnt= m_ptWayDoc->GetWayPointCnt ();
m_ptWayDoc->GetWayPointPtr (nCnt-1, &lastWay);
m_Cons.bBadElev2 = (lastWay.GetAltitude() == NO_ALT);
m_Cons.LandLoc = (CLocation)lastWay;


RequiredRWY (ptPlane, ptMet);

short nRwyLen;
m_Cons.bBadStart = TRUE;
if (m_Cons.StartLoc.GetRwyLen(&nRwyLen))
	{
	fRwyLen_m = ptDim->ConvertDist ((double)nRwyLen,
						m_Cons.StartLoc.GetLenDim(), DIM_METER);
	if (m_Cons.nStartObst_m > 0)
		m_Cons.bBadStart = (m_Cons.nStartObst_m > fRwyLen_m);
	}

m_Cons.bBadLand = TRUE;
if (m_Cons.LandLoc.GetRwyLen(&nRwyLen))
	{
	fRwyLen_m = ptDim->ConvertDist ((double)nRwyLen,
						m_Cons.LandLoc.GetLenDim(), DIM_METER);
	if (m_Cons.nLandObst_m > 0)
		m_Cons.bBadLand = (m_Cons.nLandObst_m > fRwyLen_m);
	}
					/* flight costs 	    */         
m_Cons.fCostDepArv = ptPlane->GetCost((short)(m_Cons.fEnRoute_min + m_Cons.nDepArv_min));
m_Cons.fCostTaxi = ptPlane->GetCost ((short)(m_Cons.fEnRoute_min + m_Cons.nDepArv_min + m_Cons.nTaxi_min));

return TRUE;
}
 
 
/************************************************************************
 *  Calc.cpp		       G e t T o t a l E E T						*
 *  Purpose: converts 130 min for flightplan into 210 (2:10)			*
 ************************************************************************/
UINT CCalculate::GetTotalEET ()
{
	short hour = 0;
	short min=0;
	short sec = 0;
	CDimDoc::MinutesToHMS(m_Cons.fEnRoute_min, &hour, &min, &sec);

	UINT uTime = 100*hour + min;
	return uTime;
}


/************************************************************************
 *  Calc.cpp		       G e t E n d u r a n c e						*
 *  Purpose: converts 130 min for flightplan into 210 (2:10)			*
 ************************************************************************/
UINT CCalculate::GetEndurance ()
{
short nEndurance = 	m_Cons.nMaxSafe_min + m_Cons.nSafeRes_min;

short nHours = nEndurance/60;
short nMinutes = nEndurance - 60*nHours;

UINT uTime = 100*nHours + nMinutes;
return uTime;
}
 
/////////////////////////////////////////////////////////////////////////////
// CCalculate commands
