/************************************************************************
 *  							E l e v . c p p	  						*
 ************************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "math.h"					// floor

#include "pf.h"

#include "InitDoc.h"
#include "LatLon.h"
#include "Elev.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

extern CInitDoc* 	ptInit;
extern CDimDoc*		ptDim;

/////////////////////////////////////////////////////////////////////////////
// CElev

IMPLEMENT_SERIAL(CElev, CObject, 0 /* schema number*/ )

/************************************************************************
 *  Elev.cpp			 	 C E l e v					Constructor		*
 ************************************************************************/
CElev::CElev(LPELEV lpElev)
{ 
if (lpElev == NULL)
	{       
	this->FirstInit();
	} 
else{
    this->ReadFromPtr (lpElev);
    }
}

CElev::CElev(CElev& SourceElev)
{ 
*this = SourceElev;
}



/************************************************************************
 *  Elev.cpp			 ~ C E l e v					Destructor		*
 ************************************************************************/
CElev::~CElev()
{
} 


void CElev::GetPtr(LPELEV lpElev)
{
this->WriteToPtr (lpElev);
}  

void CElev::SetPtr (LPELEV lpElev)
{
this->ReadFromPtr(lpElev);
}  

/************************************************************************
 *  Elev.cpp				operator=									*
 ************************************************************************/
const CElev& CElev::operator=(const CElev& Elev)
{
m_PosTL		= Elev.m_PosTL;
m_nSouth_min= Elev.m_nSouth_min;
m_nEast_min	= Elev.m_nEast_min;
m_lMaxElev	= Elev.m_lMaxElev;
m_nElevDim	= Elev.m_nElevDim;

return *this;
}

/************************************************************************
 *  Elev.cpp					F i r s t I n i t						*
 ************************************************************************/
void CElev::FirstInit()
{
	// m_PosTL
m_nSouth_min	= 30;
m_nEast_min		= 30;
m_lMaxElev		= NO_ALT;
m_nElevDim		= DIM_FEET;
}

/************************************************************************
 *  Elev.cpp				R e a d F r o m P t r						*
 ************************************************************************/
void CElev::ReadFromPtr(LPELEV lpElev)
{
	m_PosTL.SetLatLon(lpElev->fLat_TL, lpElev->fLon_TL);
	m_nSouth_min	= lpElev->nSouth_min;
	m_nEast_min		= lpElev->nEast_min;
	m_lMaxElev		= lpElev->lMaxElev;
	m_nElevDim		= lpElev->nElevDim;
}

/************************************************************************
 *  Elev.cpp				W r i t e T o P t r							*
 ************************************************************************/
void CElev::WriteToPtr(LPELEV lpElev) const
{
	lpElev->fLat_TL = (float)m_PosTL.GetLat();
	lpElev->fLon_TL = (float)m_PosTL.GetLon();
	lpElev->nSouth_min = m_nSouth_min;
	lpElev->nEast_min  = m_nEast_min;

	lpElev->lMaxElev= m_lMaxElev;
	lpElev->nElevDim= m_nElevDim;
}


/************************************************************************
 *  Elev.cpp    		 		 S e r i a l i z e						*
 ************************************************************************/
void CElev::Serialize(CArchive& ar, short nReadVers)
{ 
if (ar.IsStoring())
	{					// TODO: add storing code here 
	ELEVTYPE Elev;
	this->WriteToPtr(&Elev);
	ar.Write (&Elev, sizeof(ELEVTYPE)); 
	}
else{ 					// TODO: add loading code here    	    
 	if (nReadVers == ACT_ELEV_VERS)
	    {     
		ELEVTYPE Elev;
 		ar.Read (&Elev, sizeof(ELEVTYPE)); 
		this->ReadFromPtr(&Elev);
	    }
	}
} 


/************************************************************************
 *  Location.cpp    		 	C r e a t e P t r 						*
 ************************************************************************/
LPELEV CElev::CreatePtr ()
{
LPELEV lpElev = new ELEVTYPE;

_fmemset (lpElev, 0, sizeof (ELEVTYPE));

WriteToPtr(lpElev);

return lpElev;
}

/************************************************************************
 *  Elev.cpp    		 		I s E q u a l							*
 ************************************************************************/
BOOL CElev::IsEqual(const CElev& Elev)
{ 
if (!m_PosTL.IsEqual((CLatLon)(Elev.m_PosTL)))	return FALSE;
if (m_nSouth_min != Elev.m_nSouth_min)			return FALSE;
if (m_nEast_min  != Elev.m_nEast_min)			return FALSE;
if (m_lMaxElev != Elev.m_lMaxElev)				return FALSE;
if (m_nElevDim != Elev.m_nElevDim)				return FALSE;

return TRUE;
}      

/************************************************************************
 *  Elev.cpp    		 		I s E q u a l							*
 ************************************************************************/
BOOL CElev::IsEqual(LPELEV lpTestElev)
{
CElev TestElev (lpTestElev);
return (this->IsEqual (TestElev));
}      

/************************************************************************
 *  Elev.cpp    				A d d S o u t h							*
 ************************************************************************/
double CElev::AddSouth(double dLatTop, short nSouth_min)
{
	double dLat = dLatTop + (double)nSouth_min/60;
	if (dLat >  90) dLat =  90;
	if (dLat < -90) dLat = -90;
	return dLat;
}

/************************************************************************
 *  Elev.cpp    				A d d E a s t							*
 ************************************************************************/
double CElev::AddEast(double dLonLeft, short nEast_min)
{
	double dLon = dLonLeft + (double)nEast_min/60;
	if (dLon > 180)	dLon -= 360;
	if (dLon < -180)dLon += 360;
	return dLon;
}

/************************************************************************
 *  Elev.cpp    				G e t P o s B R							*
 ************************************************************************/
CLatLon CElev::GetPosBR()
{
	double dLat = CElev::AddSouth(m_PosTL.GetLat(), - m_nSouth_min);
	double dLon = CElev::AddEast (m_PosTL.GetLon(), m_nEast_min);

	CLatLon PosBR (dLat, dLon);
	return PosBR;
}

/************************************************************************
 *  Elev.cpp    		 	G e t L i s t E n t r y						*
 ************************************************************************/
CString CElev::GetListEntry()
{
	CString szEntry;
	CString szPosTL;

	m_PosTL.CreateLatLonStr(&szPosTL);

	// 	IDF_ELEVENTRY			"%s (%d/%d)"

	szEntry.Format (IDF_ELEVENTRY, szPosTL, m_nSouth_min, m_nEast_min);
	return szEntry;
}
   
/************************************************************************
 *  Elev.cpp    		 	C o m p a r e								*
 *  Purpose: Compares position LL with PosTL-PosBR:						*
 *	returns  0, if this area covers position LL							*
 *	returns  1, if this area is north or at least east of position LL	*
 *  returns -1, if this area is south or at least west of position LL	*
 ************************************************************************/
int CElev::Compare(const CLatLon& LL, BOOL bFrameIsInside)
{ 
	int nCmp = 0;
	BOOL bInside = FALSE;
	CLatLon PosBR = this->GetPosBR();
		
//	double dLonLeft = m_PosTL.GetLon();
//	double dLonRight= PosBR.GetLon();
//	double dLon = LL.GetLon();	
	
	double dLonLeft = m_PosTL.GetLon();
	double dLonRight = dLonLeft + CLatLon::FlightAngle(dLonLeft, PosBR.GetLon());
	double dLon = dLonLeft + CLatLon::FlightAngle(dLonLeft, LL.GetLon());
	
								// dLonLeft=170 dLonRight=-160 dLon=-150
	if (dLonRight < dLonLeft)	// dLonRight = -160
	{				// elev covers Lat=180
		dLonRight += 360;		// dLonRight = 200

		if (dLon < 0)			// dLon = -150
			dLon += 360;		// dLon = 210;
	}							// dLonLeft=170 dLonRight=200 dLon = 210


	if (bFrameIsInside)
	{
		bInside =	(dLonLeft	<= dLon) &&
					(dLon		<= dLonRight) &&
					(PosBR.GetLat()	<= LL.GetLat()) &&
					(LL.GetLat()	<= m_PosTL.GetLat());
	}
	else
	{			// frame is outside
		bInside =	(dLonLeft	< dLon) &&
					(dLon		< dLonRight) &&
					(PosBR.GetLat()	< LL.GetLat()) &&
					(LL.GetLat()	< m_PosTL.GetLat());
	}

	if (bInside)
	{
		nCmp = 0;
	}
	else
	{
		if (dLon <= dLonLeft)
		{				// elev area is east of position LL
			nCmp = (LL.GetLat() < m_PosTL.GetLat())? 1 : -1;				
		}
		else
		{				// elev area is north of positin LL
			nCmp = (LL.GetLat() < PosBR.GetLat())? 1 : -1;	
		}
	}

	return nCmp;
}

/************************************************************************
 *  Elev.cpp    		 	C o m p a r e								*
 *  Purpose: Compares Area PosTL-PosBR:									*
 *	returns  0, if this area is part of this test area					*
 *	returns  1, if this area is north or at least east of test area		*
 *  returns -1, if this area is south or at least west of test area		*
 ************************************************************************/
int CElev::Compare(const CElev& Elev)
{ 
	int nCmp = 0;
	CLatLon PosBR = this->GetPosBR();

	double dLatTop		= m_PosTL.GetLat();
	double dLonLeft		= m_PosTL.GetLon();
	double dLatBottom	= PosBR.GetLat();
	double dLonRight	= PosBR.GetLon();

	double dTestLatTop = Elev.m_PosTL.GetLat();
	double dTestLonLeft = Elev.m_PosTL.GetLon();
	double dTestLatBottom = CElev::AddSouth(dTestLatTop,-Elev.m_nSouth_min);
	double dTestLonRight = CElev::AddEast (dTestLonLeft, Elev.m_nEast_min);

	BOOL bSameArea = (dLatTop == dTestLatTop && 
					dLonLeft == dTestLonLeft &&
					dLatBottom == dTestLatBottom && 
					dLonRight == dTestLonRight);
	if (!bSameArea)
	{
			// dLonLeft=170 dLonRight=-160 dTestLonLeft=-150 dTestLonRight=-140
		if (dLonRight < dLonLeft)	// dLonRight = -160
		{				// elev covers Lat=180
			dLonRight += 360;		// dLonRight = 200

			if (dTestLonLeft < 0)	// dTestLonLeft = -150
				dTestLonLeft += 360;// dTestLonLeft = 210;

			if (dTestLonRight < 0)	// dTestLonRight = -140
				dTestLonRight += 360;// dTestLonRight = 220;

		}	// dLonLeft=170 dLonRight=200 dTestLonLeft = 210 dTestLonRight=220

		BOOL bFrameIsInside = FALSE;		// allow to touch 
		CLatLon ElevPosTR(dTestLatTop,	  dTestLonRight);
		CLatLon ElevPosBR(dTestLatBottom, dTestLonRight);
		CLatLon ElevPosBL(dTestLatBottom, dTestLonLeft);

		if ((this->Compare(Elev.m_PosTL, bFrameIsInside) != 0) &&
			(this->Compare(ElevPosTR, bFrameIsInside) != 0) &&
			(this->Compare(ElevPosBR, bFrameIsInside) != 0) &&
			(this->Compare(ElevPosBL, bFrameIsInside) != 0))
		{		// all corners of Elev outside or on frame of this?
			double dTestLatCenter = dTestLatBottom + (dTestLatTop - dTestLatBottom)/2;

			if (dTestLonRight <= dLonLeft)
			{			// this elev area is east of test area
				nCmp = (dTestLatCenter < dLatTop)? 1 : -1;
			}
			else
			{			// this elev area is north of test area
				nCmp = (dTestLatCenter < dLatBottom)? 1 : -1;
			}
		}
		else
		{	// test area at least partially covers this area
			nCmp = 0;
		}
	}


	return nCmp;
}


/************************************************************************
 *  Elev.cpp    				L a t T o Y								*
 ************************************************************************/
double CElev::LatToY (double dLat)
{
	double help = tan (dLat*pi/360. + pi/4); 	// for x>0, tan(x) is negative
	double fRadY = log (fabs(help));           // therefore use Absol...
	return fRadY;
}

/************************************************************************
 *  Elev.cpp    				L o n T o X 							*
 ************************************************************************/
double CElev::LonToX (double dLon)
{
return dLon*pi/180.;
}


/************************************************************************
 *  Elev.cpp    				Y t o L a t	 							*
 ************************************************************************/
double CElev::YtoLat (double dY)
{
	return 360. * (atan (exp(dY)) - pi/4) / pi;
}

/************************************************************************
 *  Elev.cpp    				X t o L o n 							*
 ************************************************************************/
double CElev::XtoLon(double dX)
{
	return dX * 180./pi;
}

/************************************************************************
 *  Elev.cpp    				I s L o n B t n 						*
 ************************************************************************/
BOOL CElev::IsLonBtn (double dMinLon, double dTestLon, double dMaxLon)
{
	BOOL bOnBorder;

	double dElev1Lon = dMinLon;
	double dElev2Lon = dElev1Lon + CLocation::FlightAngle(dMinLon, dMaxLon);
	BOOL bAntipodElev = (dElev2Lon > 180);

	if (bAntipodElev)
	{
		if (dTestLon < 0) dTestLon += 360;
	}
	else
	{
		if (dTestLon > 180) dTestLon -= 360;
	}

	bOnBorder = (dElev1Lon <= dTestLon) &&
				(dTestLon <= dElev2Lon);
	return bOnBorder;
}

/************************************************************************
 *  Elev.cpp    		 	C r o s s E a s t							*
 *  Purpose: defines ptLatCross,										*
 *	latitude at which route crosses east border line of elev area.		*
 *	returns  TRUE, if route enters or leaves elev area				 	*
 *  returns FALSE, if route crosses border north or south of elev area	*
 ************************************************************************/
BOOL CElev::CrossEast(CLatLon& Wpt1, CLatLon& Wpt2, double* ptLatCross)
{ 
	BOOL bOnBorder = FALSE;
	BOOL bOnLeg = FALSE;

										// check for lon > 180
	double dWpt1Lon = Wpt1.GetLon();
	double dWpt2Lon = dWpt1Lon + CLocation::FlightAngle(Wpt1.GetLon(), Wpt2.GetLon());
	BOOL bAntipodLeg = (dWpt2Lon > 180);

	CVektor vRteSt(LonToX(dWpt1Lon), LatToY(Wpt1.GetLat()));
	CVektor vRteRi(LonToX(dWpt2Lon)		- LonToX(dWpt1Lon),
				   LatToY(Wpt2.GetLat())- LatToY(Wpt1.GetLat()));

	CLatLon PosBR = this->GetPosBR ();					// right border of elev

	double dElevLon = PosBR.GetLon();
	if (bAntipodLeg && dElevLon < 0)
		dElevLon += 360;	
	
	CVektor vElevSt(LonToX(dElevLon), LatToY(PosBR.GetLat()));
	CVektor vElevRi(0, 1);

	CVektor vRes (vElevSt - vRteSt);

	// LGS2:	this vorinitialisiert mit Ergebnis-Vektor
	// Eingabe: Vektor X, Vektor Y, this = Ergebnisvektor
	// Ausgabe: Vektor aus den Komponenten X und Y in "this"

	if (vRes.LGS2 (vRteRi, -vElevRi))
	{				// route not parallel to western elev border
		CVektor vCross;
		vCross = vRteSt + vRteRi * vRes.X();
//		vCross = vElevSt + vElevRi * vRes.Y();
		*ptLatCross = YtoLat(vCross.Y());

					// is new point on border of elev
		bOnBorder = (PosBR.GetLat() <= *ptLatCross) &&
					(*ptLatCross <= m_PosTL.GetLat());

					// is new point on leg of route
		double dLatMin = (Wpt1.GetLat() < Wpt2.GetLat())? Wpt1.GetLat() : Wpt2.GetLat();
		double dLatMax = (Wpt1.GetLat() < Wpt2.GetLat())? Wpt2.GetLat() : Wpt1.GetLat();
		bOnLeg = (dLatMin <= *ptLatCross) &&
				 (*ptLatCross <= dLatMax);
	}

	return bOnBorder && bOnLeg;
}


/************************************************************************
 *  Elev.cpp    		 	C r o s s W e s t							*
 *  Purpose: defines ptLatCross,										*
 *	latitude at which route crosses west border line of elev area.		*
 *	returns  TRUE, if route enters or leaves elev area				 	*
 *  returns FALSE, if route crosses border north or south of elev area	*
 ************************************************************************/
BOOL CElev::CrossWest(CLatLon& Wpt1, CLatLon& Wpt2, double* ptLatCross)
{ 
	BOOL bOnBorder = FALSE;
	BOOL bOnLeg = FALSE;

										// check for lon > 180
	double dWpt1Lon = Wpt1.GetLon();
	double dWpt2Lon = dWpt1Lon + CLocation::FlightAngle(Wpt1.GetLon(), Wpt2.GetLon());
	BOOL bAntipodLeg = (dWpt2Lon > 180);

	CVektor vRteSt(LonToX (dWpt1Lon), LatToY(Wpt1.GetLat()));
	CVektor vRteRi(LonToX (dWpt2Lon)      - LonToX (dWpt1Lon),
				   LatToY (Wpt2.GetLat()) - LatToY (Wpt1.GetLat()));

	double dElevLon = m_PosTL.GetLon();
	if (bAntipodLeg && dElevLon < 0)
		dElevLon += 360;

	CVektor vElevSt(LonToX (dElevLon), LatToY (m_PosTL.GetLat()));	// left border of elev
	CVektor vElevRi(0, 1);

	CVektor vRes (vElevSt - vRteSt);

	// LGS2:	this vorinitialisiert mit Ergebnis-Vektor
	// Eingabe: Vektor X, Vektor Y, this = Ergebnisvektor
	// Ausgabe: Vektor aus den Komponenten X und Y in "this"

	if (vRes.LGS2 (vRteRi, -vElevRi))
	{				// route not parallel to western elev border
		CVektor vCross;
		vCross = vRteSt + vRteRi * vRes.X();
//		vCross = vElevSt + vElevRi * vRes.Y();
		*ptLatCross = YtoLat (vCross.Y());

					// is new point on border of elev
		CLatLon PosBR = this->GetPosBR ();
		bOnBorder = (PosBR.GetLat() <= *ptLatCross) &&
					(*ptLatCross <= m_PosTL.GetLat());

					// is new point on leg of route
		double dLatMin = (Wpt1.GetLat() < Wpt2.GetLat())? Wpt1.GetLat() : Wpt2.GetLat();
		double dLatMax = (Wpt1.GetLat() < Wpt2.GetLat())? Wpt2.GetLat() : Wpt1.GetLat();
		bOnLeg = (dLatMin <= *ptLatCross) &&
				 (*ptLatCross <= dLatMax);

	}

	return bOnBorder && bOnLeg;
}

/************************************************************************
 *  Elev.cpp    		 	C r o s s N o r t h							*
 *  Purpose: defines ptLonCross,										*
 *	longitude at which route crosses north border line of elev area.	*
 *	returns  TRUE, if route enters or leaves elev area				 	*
 *  returns FALSE, if route crosses border east or west of elev area	*
 ************************************************************************/
BOOL CElev::CrossNorth(CLatLon& Wpt1, CLatLon& Wpt2, double* ptLonCross)
{ 
	BOOL bOnBorder = FALSE;
	BOOL bOnLeg = FALSE;

										// check for lon > 180
	double dWpt1Lon = Wpt1.GetLon();
	double dWpt2Lon = dWpt1Lon + CLocation::FlightAngle(Wpt1.GetLon(), Wpt2.GetLon());

	CVektor vRteSt(LonToX (dWpt1Lon), LatToY (Wpt1.GetLat()));
	CVektor vRteRi(LonToX (dWpt2Lon)		- LonToX (dWpt1Lon),
				   LatToY (Wpt2.GetLat())	- LatToY (Wpt1.GetLat()));

	CVektor vElevSt(LonToX (m_PosTL.GetLon()), LatToY (m_PosTL.GetLat()));	// top border of elev
	CVektor vElevRi(1, 0);

	CVektor vRes (vElevSt - vRteSt);

	// LGS2:	this vorinitialisiert mit Ergebnis-Vektor
	// Eingabe: Vektor X, Vektor Y, this = Ergebnisvektor
	// Ausgabe: Vektor aus den Komponenten X und Y in "this"

	if (vRes.LGS2 (vRteRi, -vElevRi))
	{				// route not parallel to western elev border
		CVektor vCross;
		vCross = vRteSt + vRteRi * vRes.X();
//		vCross = vElevSt + vElevRi * vRes.Y();
		*ptLonCross = XtoLon (vCross.X());

					// is new point on border of elev
		CLatLon PosBR = this->GetPosBR ();
		bOnBorder = this->IsLonBtn (m_PosTL.GetLon(), *ptLonCross, PosBR.GetLon());

					// is new point on leg of route
		double	dLonMin = (dWpt1Lon < dWpt2Lon)? dWpt1Lon : dWpt2Lon;
		double	dLonMax = (dWpt1Lon < dWpt2Lon)? dWpt2Lon : dWpt1Lon;
		bOnLeg = (dLonMin <= *ptLonCross) &&
				 (*ptLonCross <= dLonMax);
	}

	return bOnBorder && bOnLeg;
}

/************************************************************************
 *  Elev.cpp    		 	C r o s s S o u t h							*
 *  Purpose: defines ptLonCross,										*
 *	longitude at which route crosses south border line of elev area.	*
 *	returns  TRUE, if route enters or leaves elev area				 	*
 *  returns FALSE, if route crosses border east or west of elev area	*
 ************************************************************************/
BOOL CElev::CrossSouth(CLatLon& Wpt1, CLatLon& Wpt2, double* ptLonCross)
{ 
	BOOL bOnBorder = FALSE;
	BOOL bOnLeg = FALSE;

										// check for lon > 180
	double dWpt1Lon = Wpt1.GetLon();
	double dWpt2Lon = dWpt1Lon + CLocation::FlightAngle(Wpt1.GetLon(), Wpt2.GetLon());

	CVektor vRteSt(LonToX (dWpt1Lon), LatToY (Wpt1.GetLat()));
	CVektor vRteRi(LonToX (dWpt2Lon)		- LonToX (dWpt1Lon),
				   LatToY (Wpt2.GetLat())	- LatToY (Wpt1.GetLat()));

	CLatLon PosBR = this->GetPosBR ();			// bottom border of elev
	CVektor vElevSt(LonToX (m_PosTL.GetLon()), LatToY (PosBR.GetLat()));
	CVektor vElevRi(1, 0);

	CVektor vRes (vElevSt - vRteSt);

	// LGS2:	this vorinitialisiert mit Ergebnis-Vektor
	// Eingabe: Vektor X, Vektor Y, this = Ergebnisvektor
	// Ausgabe: Vektor aus den Komponenten X und Y in "this"

	if (vRes.LGS2 (vRteRi, -vElevRi))
	{				// route not parallel to western elev border
		CVektor vCross;
		vCross = vRteSt + vRteRi * vRes.X();
//		vCross = vElevSt + vElevRi * vRes.Y();
		*ptLonCross = XtoLon (vCross.X());

					// is new point on border of elev
		CLatLon PosBR = this->GetPosBR ();
		bOnBorder = this->IsLonBtn (m_PosTL.GetLon(), *ptLonCross, PosBR.GetLon());

					// is new point on leg of route
		double	dLonMin = (dWpt1Lon < dWpt2Lon)? dWpt1Lon : dWpt2Lon;
		double	dLonMax = (dWpt1Lon < dWpt2Lon)? dWpt2Lon : dWpt1Lon;
		bOnLeg = (dLonMin <= *ptLonCross) &&
				 (*ptLonCross <= dLonMax);
	}

	return bOnBorder && bOnLeg;
}

/************************************************************************
 *  Elev.cpp    			 C r e a t e F o n t						*
 ************************************************************************/
BOOL CElev::CreateFont (CFont* ptFont, char* szName, long lHeight, BOOL bBold)
{    
BOOL	bOK = FALSE;
LOGFONT lfNewFont;

memset (&lfNewFont, 0, sizeof (LOGFONT));
lfNewFont.lfHeight = lHeight;
if (bBold) lfNewFont.lfWeight = FW_BOLD;
lfNewFont.lfCharSet = DEFAULT_CHARSET;
strcpy (lfNewFont.lfFaceName, szName);

bOK = ptFont->CreateFontIndirect ((LOGFONT FAR*)&lfNewFont);    
return bOK;
}


/****************************************************************************
 *	Elev.cpp						D r a w V a l u e						*
 ****************************************************************************/
void CElev::DrawValue (CDC* pDC, long lTop, long lLeft, long lBottom, long lRight) 
{
	long lElevSizeX = lRight - lLeft;
	long lElevSizeY = lBottom - lTop;
	long lCenterX = lLeft + (long)(0.55*lElevSizeX);
	long lTopOfText = lTop + (long)(0.4*lElevSizeY);


	long lLargeFontSize = (long)(-lElevSizeY)/6;
	long lSmallFontSize = (long)(0.8 * lLargeFontSize);

	
	short nTopOfSmall = (short)(lTopOfText - (lLargeFontSize - lSmallFontSize)/3);
		
	long lElev = (long)ptDim->ConvertDist(m_lMaxElev, m_nElevDim, DIM_FEET);


	CString szElev;								// lElev = 12345 ft
	float fElev = (float)lElev / 100;			// fElev = 123.45
	lElev = (long)(fElev + 0.5);				// lElev = 123
	szElev.Format ("%02ld", lElev);				// szElev= "123"
	short nLen = szElev.GetLength();
	CString szLarge(szElev.Left(nLen-1));		// szLarge = "12"
	CString szSmall(szElev.Right(1));			// szSmall = "3"


	int OldBkMode = pDC->SetBkMode (TRANSPARENT);		// transparent
	COLORREF OldTextCol = pDC->GetTextColor();		

	CColorDoc* ptColor = ptInit->GetColorPtr();
	COLORREF ElevCol = ptColor->GetColElevValue();
	pDC->SetTextColor(ElevCol);

	CFont BoldFont;
												// draw large part of elevation
	this->CreateFont (&BoldFont, "Arial", lLargeFontSize, TRUE);	// bBold=TRUE
	CFont* ptOldFont = (CFont*)pDC->SelectObject (&BoldFont);

	pDC->SetTextAlign (TA_RIGHT | TA_TOP);
	pDC->TextOut ((short)lCenterX, (short)lTopOfText, szLarge);

	pDC->SelectObject (ptOldFont);
	BoldFont.DeleteObject ();

												// draw small part of elevation
	this->CreateFont (&BoldFont, "Arial", lSmallFontSize, TRUE);	// bBold=TRUE
	pDC->SelectObject (&BoldFont);

	pDC->SetTextAlign (TA_LEFT | TA_TOP);	
	pDC->TextOut ((short)lCenterX, nTopOfSmall, szSmall);	

	pDC->SelectObject (ptOldFont);
	BoldFont.DeleteObject ();

	pDC->SetTextColor(OldTextCol);				// reset old text color
	pDC->SetBkMode (OldBkMode);					// reset background mode
}

/////////////////////////////////////////////////////////////////////////////
// CElev commands
