/************************************************************************
 *  						E l e v A r r a y . c p p	  				*
 ************************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "math.h"					// floor
#include "pf.h"

#include "InitDoc.h"				// DRECT

#include "ElevArray.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif


/////////////////////////////////////////////////////////////////////////////
// CElevArray

IMPLEMENT_SERIAL(CElevArray, CObject, 0 /* schema number*/ )

/************************************************************************
 *  Elev.cpp			 	 C E l e v					Constructor		*
 ************************************************************************/
CElevArray::CElevArray()
{ 
	this->Reset();
}

CElevArray::CElevArray(CElevArray& SourceElev)
{ 
*this = SourceElev;
}



/************************************************************************
 *  Elev.cpp			 ~ C E l e v					Destructor		*
 ************************************************************************/
CElevArray::~CElevArray()
{
	if (m_ptElevLine != NULL)
	{
		delete [] m_ptElevLine;			// CRASH!!!! nach Hinzufgen: copy constructor!!!
		m_ptElevLine	= NULL;
		m_nElevLineLen	= 0;
	}

	this->Close();
} 


/************************************************************************
 *  Elev.cpp				operator=									*
 ************************************************************************/
const CElevArray& CElevArray::operator=(const CElevArray& Elev)
{
	m_szName			= Elev.m_szName;
	m_szPath			= Elev.m_szPath;
	m_bActivated		= Elev.m_bActivated;

	m_fLonMin           = Elev.m_fLonMin;
	m_fLonMax           = Elev.m_fLonMax;
	m_fLatMax           = Elev.m_fLatMax;
	m_fLatMin           = Elev.m_fLatMin;

	m_nRows				= Elev.m_nRows;
	m_nColumns			= Elev.m_nColumns;
	m_fGridSize			= Elev.m_fGridSize;
	m_nElevMin_m		= Elev.m_nElevMin_m;
	m_nElevMax_m		= Elev.m_nElevMax_m;
	m_nElevMissingFlag	= Elev.m_nElevMissingFlag;
	m_ByteOrder			= Elev.m_ByteOrder;
	m_bOpened			= Elev.m_bOpened;

	m_ptFile			= NULL;		// don't copy pointers
									// deleting copy may delete original ptr!!
	m_lLastOffset		= Elev.m_lLastOffset;
	m_nLastElev_m		= Elev.m_nLastElev_m;

	if (Elev.m_nElevLineLen > 0)
	{					// copy m_ptElevLine
		m_ptElevLine = new short[Elev.m_nElevLineLen];
		
		if (m_ptElevLine != NULL)
		{
			m_nElevLineLen = Elev.m_nElevLineLen;
			for (short i=0; i<m_nElevLineLen; i++)
			{
				*(m_ptElevLine + i) = *(Elev.m_ptElevLine + i);
			}
		}
	}
	else
	{					// reset m_ptElevLine
		m_nElevLineLen		= 0;
		m_ptElevLine		= NULL;
	}


return *this;
}

/************************************************************************
 *  Elev.cpp					R e s e t								*
 ************************************************************************/
void CElevArray::Reset()
{
	m_szName			= "";
	m_szPath			= "";
	m_bActivated		= TRUE;

	m_fLonMin           = 0.00000000000;
	m_fLonMax           = 0.00000000000;
	m_fLatMax           = 0.00000000000;
	m_fLatMin           = 0.00000000000;

	m_nRows				= 2400;
	m_nColumns			= 2400;
	m_fGridSize			= 0.00833333333;		// (60-40)/2400 oder (20-0)/2400
	m_nElevMin_m		= 0;
	m_nElevMax_m		= 0;
	m_nElevMissingFlag	= 0;
	m_ByteOrder			= LITTLE_ENDIAN;
	m_bOpened			= FALSE;


	m_ptFile			= NULL;
	m_lLastOffset		= -1;
	m_nLastElev_m		= 0;

	m_nElevLineLen		= 0;
	m_ptElevLine		= NULL;
}


/************************************************************************
 *  Elev.cpp    		R e n a m e L y b i a T o L i b y a				*
 ************************************************************************/
BOOL CElevArray::RenameLybiaToLibya()
{
	BOOL bRenamed = FALSE;

	short nSlash = m_szPath.ReverseFind('\\');
	if (nSlash > 0)
	{
		short nCount = m_szPath.GetLength() - nSlash - 1;
		CString szFileExt(m_szPath.Right(nCount));


		if (szFileExt.CompareNoCase("Lybia.bin") == 0)
		{
			CString szNewPath;
			szNewPath = m_szPath.Left(nSlash+1);
			szNewPath += "Libya.bin";

//			CString szMessage;
//			szMessage.Format("nCount=%d, szFileExt=%s newPath=%s", nCount, (LPCTSTR)szFileExt, (LPCTSTR)szNewPath);
//			AfxMessageBox(szMessage);

			TRY
			{    
				CFile::Rename((LPCTSTR)m_szPath, (LPCTSTR)szNewPath);
				m_szPath = szNewPath;
			}
			CATCH( CFileException, e )
			{
				AfxMessageBox("Libya.bin not found!");
			}
			END_CATCH

			bRenamed = TRUE;
		}
	}

	return bRenamed;
}

/************************************************************************
 *  Elev.cpp    		 		 S e r i a l i z e						*
 ************************************************************************/
void CElevArray::Serialize(CArchive& ar, short nReadVers)
{ 
	int nVersion;

//	if (m_szName.CompareNoCase("Zentral Europa") == 0)
//		int a=1;

	if (ar.IsStoring())
	{					// TODO: add storing code here 
		nVersion = ACT_ELEV_ARRAY_VERS;  
		ar << (WORD)nVersion;

		ar << m_szName;
		ar << m_szPath;			// path to *.bin file
		ar << m_bActivated;

		ar << m_fLonMin;
		ar << m_fLonMax;
		ar << m_fLatMax;
		ar << m_fLatMin;

		ar << (WORD)m_nRows;
		ar << (WORD)m_nColumns;
		ar << m_fGridSize;
		ar << (WORD)m_nElevMin_m;
		ar << (WORD)m_nElevMax_m;
		ar << (WORD)m_nElevMissingFlag;
		ar << (WORD)m_ByteOrder;
	}
	else
	{ 					// TODO: add loading code here    	  
		WORD	Word;
		ar >> Word;		nVersion = (short)Word;

 		if (nVersion == ACT_ELEV_ARRAY_VERS)
		{   
			ar >> m_szName;
			ar >> m_szPath;			// path to *.bin file
			ar >> m_bActivated;

			ar >> m_fLonMin;
			ar >> m_fLonMax;
			ar >> m_fLatMax;
			ar >> m_fLatMin;

			ar >> Word;		m_nRows = (short)Word;
			ar >> Word;		m_nColumns = (short)Word;
			ar >> m_fGridSize;
			ar >> Word;		m_nElevMin_m = (short)Word;
			ar >> Word;		m_nElevMax_m = (short)Word;
			ar >> Word;		m_nElevMissingFlag = (short)Word;
			ar >> Word;		m_ByteOrder = (BYTEORDER)Word;

			RenameLybiaToLibya();		// rename Lybia.bin to Libya.bin

			// SRTM:
			if(m_szName.CompareNoCase("A_SRTMTest") == 0 ||
				m_szName.CompareNoCase("A_SRTMTest1") == 0)
				m_ByteOrder = BIG_ENDIAN;

		}
		else
		{
			this->Reset();
		}
	}
} 


/************************************************************************
 *  Elev.cpp    		 		I s E q u a l							*
 ************************************************************************/
BOOL CElevArray::IsEqual(const CElevArray& Elev)
{ 
	if (m_szName.Compare(Elev.m_szName) != 0) return FALSE;
	if (m_szPath.CompareNoCase(Elev.m_szPath) != 0) return FALSE;
	if (m_bActivated != Elev.m_bActivated) return FALSE;

	if (m_fLonMin != Elev.m_fLonMin) return FALSE;
	if (m_fLonMax != Elev.m_fLonMax) return FALSE;
	if (m_fLatMax != Elev.m_fLatMax) return FALSE;
	if (m_fLatMin != Elev.m_fLatMin) return FALSE;

	if (m_nRows != Elev.m_nRows) return FALSE;
	if (m_nColumns != Elev.m_nColumns) return FALSE;

	if (m_fGridSize != Elev.m_fGridSize) return FALSE;
	if (m_nElevMin_m != Elev.m_nElevMin_m) return FALSE;
	if (m_nElevMax_m != Elev.m_nElevMax_m) return FALSE;
	if (m_nElevMissingFlag != Elev.m_nElevMissingFlag) return FALSE;
	if (m_ByteOrder != Elev.m_ByteOrder) return FALSE;

	// not required to compare:
	//m_ptFile
	//m_lLastOffset
	//m_nLastElev_m
	// m_nElevLineLen
	// m_ptElevLine
	
return TRUE;
}      

/****************************************************************************
 *	ElevArray.cpp				I s A n t i p o d 							*
 *  Purpose: returns TRUE, if elev array shows right and left border of map *
 *				and right map side is > 180 degrees							*
 ****************************************************************************/
BOOL CElevArray::IsAntipod(DRECT rMap)
{
	BOOL bAntipod = FALSE;

	short nVis = 0;

	double dTestLon = rMap.left;
	if ((m_fLonMin <= dTestLon) && (dTestLon <= m_fLonMax))
		nVis++;

	dTestLon = rMap.right - 360;
	if ((m_fLonMin <= dTestLon) && (dTestLon <= m_fLonMax))
		nVis++;

	bAntipod = (nVis == 2);
 
	return bAntipod; 
} 

/****************************************************************************
 *	ElevArray.cpp			N e w S h o r t B y t e O r d e r				*
 ****************************************************************************/
short CElevArray::NewShortByteOrder (short nSource)
{
short nDest;
short	VARlen, i;

VARlen = 2;
for (i=0; i<VARlen; i++)
	*((char*)&nDest + VARlen - 1 - i) = *((char*)&nSource + i); 
return nDest;
}

/************************************************************************
 *  ElevArray.cpp  				G e t E l e v _ m						*
 ************************************************************************/
BOOL CElevArray::GetElev_m (double fLat, double fLon, short* ptElev)
{	
	BOOL bFound = FALSE;

	short nElev_m = m_nElevMissingFlag;

	short nRow=-1;
	short nColumn=-1;
	long  lOffs = -1;

	if (this->HasElevFor(fLat, fLon))
	{
		nRow	= (short)(m_nRows   /(m_fLatMin - m_fLatMax) * (fLat - m_fLatMax));
		nColumn = (short)(m_nColumns/(m_fLonMax - m_fLonMin ) * (fLon - m_fLonMin));

		lOffs = (nRow * m_nColumns + nColumn) * sizeof (short);
	}


	if (lOffs >= 0)
	{					// try to read counter from file
		CFile f;
		CFileException e;

		if(f.Open((LPCTSTR)m_szPath, CFile::modeRead, &e ) )
		{
			f.Seek(lOffs, CFile::begin);
			f.Read ((void*)&nElev_m, sizeof (short));

			if (m_ByteOrder	== BIG_ENDIAN)
			{					// PC needs LITTLE_ENDIAN
				nElev_m = this->NewShortByteOrder(nElev_m);
			}

			f.Close();

			if (nElev_m != m_nElevMissingFlag)
			{
				*ptElev = nElev_m;
				bFound = TRUE;
			}
		}
		else
		{
		#ifdef _DEBUG
		   afxDump << "File could not be opened " << e.m_cause << "\n";
		#endif
		}
	}

	return bFound;
}

/************************************************************************
 *  ElevArray.cpp  				H a s E l e v F o r						*
 ************************************************************************/
BOOL CElevArray::HasElevFor (double fLat, double fLon)
{	
	BOOL bFound = FALSE;

	if ((m_fLatMax >= fLat && fLat > m_fLatMin) &&
		 (m_fLonMin <= fLon && fLon < m_fLonMax) )
	{
		bFound = m_bActivated;		// TRUE only, if activated!!
	}

	return bFound;
}


/************************************************************************
 *  ElevArray.cpp  					O p e n								*
 ************************************************************************/
BOOL CElevArray::Open ()
{	
	CFileException e;

	if (m_ptFile == NULL)
	{
		m_ptFile = new CFile();
		if (m_ptFile != NULL)
		{
			if(m_ptFile->Open((LPCTSTR)m_szPath, CFile::modeRead, &e ) )
			{
				m_lLastOffset = -1;
				m_bOpened = TRUE;			
			}
			else
			{
			   m_bOpened = FALSE;

			#ifdef _DEBUG
			   afxDump << "File could not be opened " << e.m_cause << "\n";
			#endif
			}
		}

	}

	return m_bOpened;
}

/************************************************************************
 *  ElevArray.cpp				G e t R o w								*
 ************************************************************************/
short CElevArray::GetRow (double fLat)
{
	return (short)(m_nRows/(m_fLatMin - m_fLatMax) * (fLat - m_fLatMax));
}


/************************************************************************
 *  ElevArray.cpp				G e t C o l u m n						*
 ************************************************************************/
short CElevArray::GetColumn (double fLon)
{
	return (short)(m_nColumns/(m_fLonMax - m_fLonMin ) * (fLon - m_fLonMin));
}


/************************************************************************
 *  ElevArray.cpp				R e a d E l e v _ m						*
 ************************************************************************/
BOOL CElevArray::ReadElev_m (double fLat, double fLon, short* ptElev)
{
	BOOL bRead = FALSE;

	if(m_ptFile != NULL && m_bOpened)
	{
		short nElev_m = m_nElevMissingFlag;
		short nRow	= this->GetRow(fLat);
		short nColumn = this->GetColumn(fLon);

		long lOffs = (nRow * m_nColumns + nColumn) * sizeof (short);

		if (lOffs != m_lLastOffset)
		{
			m_ptFile->Seek(lOffs, CFile::begin);
			m_ptFile->Read ((void*)&nElev_m, sizeof (short));

			if (m_ByteOrder	== BIG_ENDIAN)
			{					// PC needs LITTLE_ENDIAN
				nElev_m = this->NewShortByteOrder(nElev_m);
			}

			m_lLastOffset = lOffs;
			m_nLastElev_m = nElev_m;
		}
		else
		{			// lOffs == m_lLastOffset
			nElev_m = m_nLastElev_m;
		}

		if (nElev_m != m_nElevMissingFlag)
		{
			*ptElev = nElev_m;
			bRead = TRUE;
		}

	}
		

	return bRead;
}


/************************************************************************
 *  ElevArray.cpp				R e a d E l e v L i n e					*
 ************************************************************************/
BOOL CElevArray::ReadElevLine (double fLat, double fLon, short nElevCnt)
{
	BOOL bRead = FALSE;

	if(m_ptFile != NULL && m_bOpened)
	{
		short nRow	= this->GetRow(fLat);		// 1144
		short nColumn = this->GetColumn(fLon);	// 892

		long lOffs = (nRow * m_nColumns + nColumn) * sizeof (short);

		if (lOffs != m_lLastOffset)
		{
			if (m_ptElevLine != NULL)
			{
				delete [] m_ptElevLine;
				m_ptElevLine = NULL;
				m_nElevLineLen = 0;
			}

			m_nElevLineLen = nElevCnt;
			m_ptElevLine = new short[nElevCnt];
			
			if (m_ptElevLine != NULL)
			{
				m_ptFile->Seek(lOffs, CFile::begin);
				m_ptFile->Read ((void*)m_ptElevLine, nElevCnt * sizeof (short));

				if (m_ByteOrder	== BIG_ENDIAN)
				{					// PC needs LITTLE_ENDIAN
					for (int i=0; i<nElevCnt; i++)
						*(m_ptElevLine+i) = this->NewShortByteOrder(*(m_ptElevLine+i));
				}

				bRead = TRUE;
				m_lLastOffset = lOffs;
			}
		}
		else
		{			// lOffs == m_lLastOffset
			// leave m_ptElevLine unchanged and use it again
			bRead = TRUE;
		}
	}
		
	return bRead;
}



/****************************************************************************
 *	ElevArray.cpp			G e t L o n F o r E l e v A r e a				*
 *  Purpose: shift lon from antipod map into range of elev area				*
 *  Input: lon used to visualize map										*
 *	Return: for dLon =-180 and elev from 150 ... 180 => 180					*
 ****************************************************************************/										 
double CElevArray::GetLonForElevArea (double dLon)
{
	short nOffset = 0;

	double dEps = 1e-10;

	if (dLon < m_fLonMin - dEps)
		nOffset = 360;
	if (dLon > m_fLonMax + dEps)
		nOffset = -360;

	return dLon + nOffset;
}



// loop von ImageRowStart bis image row end und dann passende elevs aus ElevArray lesen

/************************************************************************
 *  ElevArray.cpp			G e n e r a t e I m a g e L i n e			*
 *  Purpose: reads all elevations between dStartLon and dEndLon			*
 *		uses read elevations (m_ptElevLine)								*
 *		do define image line between dStartLon and dEndLon				*
 ************************************************************************/
void CElevArray::GenerateImageLine (LPSTR lpData, long lFileRowBytes,
									short nLine, short nImageRowStart, short nImageRowEnd,
									double dActLat, double dStartLon, double dEndLon)
{
								// convert map lon to original lon
	dStartLon = this->GetLonForElevArea (dStartLon);
	dEndLon = this->GetLonForElevArea (dEndLon);

								// get range of columns in ElevArray to cover 
								// area between dStartLon and dEndLon
	short nStartColumn = GetColumn (dStartLon);		// 884	
	short nEndColumn = GetColumn (dEndLon);			// 1189

								// number of elev array columns to read
	short nColumnCnt = nEndColumn - nStartColumn + 1;	// 306	

	double dImageToElevFakt = (double)(nEndColumn - nStartColumn) / (nImageRowEnd - nImageRowStart);

	if (ReadElevLine (dActLat, dStartLon, nColumnCnt))
	{						// m_ptElevLine and m_nElevLineLen is now defined!!

							// for every line of 1-2-4-8-bit-pixels:
		for (int j=nImageRowStart; j < nImageRowEnd; j++)	// j=884 to 1189
		{
			unsigned char pixel = (unsigned char)(0);	// undefined area

							// convert pixel x-position into ElevArray Column index
			short nActColumn = (short)((double)(j-nImageRowStart) * dImageToElevFakt);

			short nElev_m = *(m_ptElevLine+nActColumn);	// 133

			// SRTM:		// make everything below msl blue
			//if (nElev_m < 0)
			//	nElev_m = m_nElevMissingFlag;

			if (nElev_m != m_nElevMissingFlag)
			{
				if (nElev_m < 0) nElev_m = 0;			// elevation below sea level
													// other elevation:
				short nByteVal = 254 * nElev_m / 8000 + 1;
				if (nByteVal > 255) nByteVal = 255;

				pixel = (unsigned char)nByteVal;
			}
			// visible width of image: nEndColumn - nImageRowStart
			// internal width of image: lFileRowBytes (lFileRowBytes MODULO 4 = 0)
			// visible width may be smaller than internal width ( 0 < diff < 4)

			*(lpData + nLine * lFileRowBytes + j) = (char)pixel;
		}		
	}
}


/************************************************************************
 *  ElevArray.cpp						C l o s e						*
 ************************************************************************/
void CElevArray::Close ()
{
	if (m_ptFile != NULL)
	{	
		if (m_bOpened)
		{
			m_ptFile->Close();
			m_bOpened = FALSE;
		}

		delete m_ptFile;
		m_ptFile = NULL;
	}
}

/////////////////////////////////////////////////////////////////////////////
// CElevArray commands
