// ElevView.cpp : implementation of the CElevView class
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "math.h"							// for: atan
#include "pf.h"

#include "InitDoc.h"  

#include "TrackDoc.h"			// for TimeScaleTextX only
#include "MetDoc.h"
#include "PlaneDoc.h"
#include "Calc.h"			// this statement must be after PlaneDoc.h

#include "WayDoc.h"
#include "ElevDoc.h"
#include "ElevView.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

extern CInitDoc* 		ptInit;
extern CFontDoc* 		ptFontTool;
extern CDimDoc* 		ptDim;      
extern CPlaneDoc*		ptPlaneDoc;
extern CElevDoc*		ptElevDoc;

extern CLoadDoc*		ptLoad; 
//extern CCalcDoc*		ptCalc;
extern CMetDoc*			ptMet;


extern BOOL bMMText;

// Draws elevation below and airspace around current flight route
// IsAirspaceToDraw specifies type of airspaces to be drawn

/****************************************************************************
 *	ElevView.cpp			E l e v S c a l e T e x t Y 					*
 ****************************************************************************/
short ElevScaleTextY (double dYval, double dYGrid, CString* ptScale)
{
short	nLen;
char	szScale[32];							
CString	szScaleFormat;
CString szDim;

szScaleFormat.LoadString (IDF_TRACK_SCALE);
ptDim->GetDimCstring (&szDim, ptDim->Altitude());
nLen = sprintf (szScale, szScaleFormat, (long)dYval, (LPCTSTR)szDim);

*ptScale = (CString)szScale;
return nLen;
}

/****************************************************************************
 *  ElevView.cpp					E l e v M a p X 						*
 ****************************************************************************/
double ElevMapX (double dXval)
{
return dXval;
}

/****************************************************************************
 *  ElevView.cpp					E l e v M a p Y 						*
 ****************************************************************************/
double ElevMapY (double dYval)
{
return dYval;
}



/////////////////////////////////////////////////////////////////////////////
// CElevView

IMPLEMENT_DYNCREATE(CElevView, CTextViewTool)

BEGIN_MESSAGE_MAP(CElevView, CTextViewTool)
	//{{AFX_MSG_MAP(CElevView)
	ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
	//}}AFX_MSG_MAP
	// Standard printing commands
	ON_COMMAND(ID_FILE_PRINT, CTextViewTool::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CTextViewTool::OnFilePrintPreview)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CElevView construction/destruction

/************************************************************************
 *  ElevView.cpp   				C E l e v V i e w 			Constructor	*
 ************************************************************************/
CElevView::CElevView()
{
	// TODO: add construction code here
ptFontTool->GetParams (m_FontName, &m_wPointSize);

m_CharX = m_LineY = 0;
m_BCharX = m_BLineY = 0;

CreateBoldFont(&m_BoldFont);
CreateNewFont(&m_PlainFont);

m_ptColor = ptInit->GetColorPtr();

m_ptAltGraph = new CMap (ElevMapX, ElevMapY, 
						 ElevMapX, ElevMapY,
						TimeScaleTextX, ElevScaleTextY, 
						ANISOTROPIC);
m_bInitialView = TRUE;

}

/************************************************************************
 *  ElevView.cpp   			~ C E l e v V i e w 			Destructor	*
 ************************************************************************/
CElevView::~CElevView()
{
m_PlainFont.DeleteObject ();
m_BoldFont.DeleteObject ();

delete m_ptAltGraph;

}

/************************************************************************
 *  ElevView.cpp				U p d a t e F o n t s 					*
 ************************************************************************/
void CElevView::UpdateFonts (CDC* pDC)
{
ptFontTool->GetParams (m_FontName, &m_wPointSize);
m_lfHeight = this->Y_PointToLP (pDC, m_wPointSize);  

m_PlainFont.DeleteObject ();
CreateNewFont(&m_PlainFont);
this->GetCharSize (pDC, &m_PlainFont, &m_CharX, &m_LineY); 

m_BoldFont.DeleteObject ();
CreateBoldFont(&m_BoldFont);
this->GetCharSize (pDC, &m_BoldFont, &m_BCharX, &m_BLineY); 
}

/************************************************************************
 *  ElevView.cpp			O n I n i t i a l U p d a t e				*
 ************************************************************************/
void CElevView::OnInitialUpdate()
{
	CTextViewTool::OnInitialUpdate();
	// TODO: calculate the total size of this view

CDC* pDC = GetDC();    
pDC->SetMapMode(MM_LOMETRIC);

this->UpdateFonts (pDC);

ReleaseDC(pDC);     
}

/************************************************************************
 *  ElevView.cpp   				  S e t X T a b s 						*
 ************************************************************************/
void CElevView::SetXTabs (CDC* pDC, CWayDoc* ptWayDoc, int CharX)
{
int X_TAB, LogPixXToUse, LogPixYToUse;
GetLogPixToUse(pDC, (int*)&LogPixXToUse, (int*)&LogPixYToUse);   

X_TAB = 4*CharX;
m_dX = LogPixXToUse - 2*X_TAB;
m_dY = LogPixYToUse;
m_Xleft  = X_TAB;
m_Xright = m_dX;
}


/************************************************************************
 *  ElevView.cpp				P r e p a r e X T a b s					* 
 ************************************************************************/
void CElevView::PrepareXTabs (CDC* pDC)
{
CWayDoc* pDoc = GetDocument();

if (pDoc != NULL)
	{	        
	pDC->SetMapMode(MM_LOMETRIC);
	this->UpdateFonts (pDC);					// defines m_CharX
	CFont* ptOldFont = (CFont*)pDC->SelectObject (&m_BoldFont);		
    this->SetXTabs (pDC, pDoc, m_CharX);
	pDC->SelectObject (ptOldFont);
	}
}

/************************************************************************
 *  ElevView.cpp					O n U p d a t e 					* 
 *  Called on FileNew, FileOpen, Doc.UpdateAllViews 					*
 ************************************************************************/
void CElevView::OnUpdate (CView* pSender, LPARAM lHint, CObject* pHint)
{
CWayDoc* pDoc = GetDocument();
ASSERT_VALID(pDoc);  

if (pDoc != NULL)
	{	        
	CDC* pDC = GetDC();    
	this->PrepareXTabs(pDC);
	ReleaseDC(pDC);     
	
	CTextViewTool::OnUpdate (pSender, lHint, pHint);
	}	 
}

/************************************************************************
 *  ElevView.cpp   			 O n P r e p a r e D C						*
 ************************************************************************/
void CElevView::OnPrepareDC(CDC* pDC, CPrintInfo* pInfo) 
{
	// TODO: Add your specialized code here and/or call the base class

CWayDoc* pDoc = GetDocument();
ASSERT_VALID(pDoc);  

if (pDoc != NULL)
	{	        
	m_bForPrinter = pDC->IsPrinting();	  // important to get correct update after printing
	this->PrepareXTabs(pDC);
	}
		
if (!m_bForPrinter)
	CTextViewTool::OnPrepareDC(pDC, pInfo);	 // sets bContinuePrinting = TRUE if SetMaxPage is set
}

/************************************************************************
 *  ElevView.cpp				G e t M a x R e c t L P					*
 ************************************************************************/
void CElevView::GetMaxRectLP (CDC* pDC, CWnd* ptWnd, LPRECT lpRectMax, BOOL bPrinter)
{
					 //normal: CapsX = 52, LineY = 60
if (bPrinter)
    {
    int Width_pix, Height_lines;
    POINT   Offset, Pixels;

    Width_pix = pDC->GetDeviceCaps (HORZRES);	/* 960 pixels	*/
    Height_lines = pDC->GetDeviceCaps (VERTRES);	/* 841 lines	*/

    Pixels.x = Width_pix;
    Pixels.y = Height_lines;
	
	pDC->DPtoLP((LPPOINT)&Pixels, 1);   /* W=2032, H=-2967	    */
    Offset.x = (int)(0.05* Pixels.x);	    /* start 05% of pagewidth */
    Offset.y = (int)(0.08* Pixels.y);	    /* start 08% of pageheight*/
    Pixels.x = (int)(0.9 * Pixels.x);	    /* use 90% of pagewidth */
    Pixels.y = (int)(0.85* Pixels.y);	    /* use 85% of pageheight*/
    SetRect (lpRectMax, Offset.x, Offset.y, 
						Pixels.x, Pixels.y);
    }
else{					    /* use actual window size*/
    ptWnd->GetClientRect (lpRectMax);
	pDC->DPtoLP((LPPOINT)lpRectMax, 2);
    }
}


/************************************************************************
 *  ElevView.cpp			G e t U s e a b l e R e c t L P				*
 ************************************************************************/
CRect CElevView::GetUseableRectLP (CDC* pDC)
{
CRect rView, rBorderLP;

if (m_bForPrinter && pDC != NULL)
	{
	this->GetMaxRectLP (pDC, this, rView, m_bForPrinter);
	short nFrameX = (short)((double)(rView.right - rView.left) / 50);
	short nFrameY = (short)((double)(rView.bottom - rView.top) / 70);

	rBorderLP.top	= rView.top		+ 3*nFrameY;
	rBorderLP.left	= rView.left	+ 6*nFrameX;
	rBorderLP.right	= rView.right	- 0*nFrameX;
	rBorderLP.bottom = rView.bottom	- 1*nFrameY;
	}
else{
	CClientDC dc (this);
	dc.SetMapMode (MM_LOMETRIC);

	GetClientRect (rView);
	dc.DPtoLP((LPPOINT)&rView, 2);
	rBorderLP.top	= rView.top		+ 4*m_LineY;
	rBorderLP.left	= rView.left	+ 4*m_CharX;
	rBorderLP.right	= rView.right	- 4*m_CharX;
	rBorderLP.bottom = rView.bottom	- 2*m_LineY;
	}  

return rBorderLP;
}

/****************************************************************************
 *	ElevView.cpp			G e t T i m e F o r D i s t _ m i n 			*
 *	Purpose:  calc time in same way as in CCalculate::FlugDaten				*
 *	to avoid rounding errors btn Leg.dTimeLeg								*
 ****************************************************************************/
double CElevView::GetTimeForDist_min(double dDist_NM, double dLeg_NM, double dLeg_min)
{
	double dDist_min;

	dDist_min = dLeg_min * dDist_NM / dLeg_NM;
	return dDist_min;
}

/************************************************************************
 *  ElevView.cpp	   			L i n e A n g l e _ d e g 				*
 *  Purpose: calculates course-like angle of line via logical coords	*
 ************************************************************************/
double CElevView::LineAngle_deg (long LPx0, long LPy0, long LPx1, long LPy1)
{
	double dAngle_deg;

	if (LPx0 == LPx1)
	{							// no x-difference
		if (LPy1 > LPy0) dAngle_deg = 0;
			else		 dAngle_deg = 180;	
	}
	else
	{
		if (LPy0 == LPy1)
		{				// no y-difference
		if (LPx1 > LPx0) dAngle_deg = 90;
			else		 dAngle_deg = 270;	// shouldn't happen for elev view
		}
		else
		{
			if (LPx1 > LPx0)
			{
				double dAngle_rad = atan ((double)(LPy1-LPy0)/(LPx1-LPx0));
				dAngle_deg = 90 - (180 * dAngle_rad / pi);
			}
			else
			{
				double dAngle_rad = atan ((double)(LPy1-LPy0)/(LPx1-LPx0));
				dAngle_deg = 270 - (180 * dAngle_rad / pi);
			}
		}
	}

return dAngle_deg;
}


/************************************************************************
 *  ElevView.cpp	   			D r a w L a b e l 						*
 ************************************************************************/
void CElevView::DrawLabel (CDC* pDC, long LPx1, long LPy1, double TextAngle, long LineY, 
				BOOL bRightAlign, CString szLoc, short nSizeLP, BOOL bActWpt)
{	
long	LPx2, LPy2, TextVektLen, dY, dYtext;
double	Rad;

TextVektLen = 2*LineY;			

Rad = pi * TextAngle / 180;
if (bMMText) LPx2 = LPx1 + (long)(sin (Rad) * TextVektLen);
	else	 LPx2 = LPx1 - (long)(sin (Rad) * TextVektLen);
LPy2 = LPy1 - (long)(cos (Rad) * TextVektLen);


CPen DashPen (PS_DASH, 1, RGB (0, 0, 0));
CBrush* ptOldBrush;
CPen* ptOldPen = (CPen*)pDC->SelectObject (&DashPen);

						 				
CBrush brBackGround (bActWpt? m_ptColor->GetColScMapSel() : m_ptColor->GetColScMapLabel());
ptOldBrush = (CBrush*)pDC->SelectObject (&brBackGround);

int OldBkMode = pDC->SetBkMode (TRANSPARENT);

dY	= LineY / 3;							/* see: RectAlignXXXX		*/
dYtext = LineY / 6;

pDC->MoveTo (LPx1, LPy1);
pDC->LineTo (LPx2, LPy2+dY);											

COLORREF	OldTextCol = pDC->GetTextColor();
pDC->SetTextColor(RGB (0, 0, 0));			// switch to black text color
if (bRightAlign)	
	{										/* true: show frame			*/
	CMap::RectAlignRight (pDC, LPx2, LPy2+dY, LineY, nSizeLP, TRUE);	
	pDC->SetTextAlign (TA_RIGHT | TA_TOP);
	pDC->TextOut (LPx2, LPy2+dYtext, szLoc);

	}
else{										/* true: show frame			*/
	CMap::RectAlignLeft (pDC, LPx2, LPy2+dY, LineY, nSizeLP, TRUE);	
	pDC->SetTextAlign (TA_LEFT | TA_TOP);
	pDC->TextOut (LPx2, LPy2+dYtext, szLoc);

	}
pDC->SetTextColor(OldTextCol);				// switch back to colored map text
pDC->MoveTo (LPx1, LPy1);
		
pDC->SetBkMode (OldBkMode);
//pDC->SelectStockObject (WHITE_BRUSH);
pDC->SelectObject (ptOldBrush);
pDC->SelectObject (ptOldPen);
}	


/****************************************************************************
 *	ElevView.cpp			I s A i r s p a c e T o D r a w					*
 ****************************************************************************/
BOOL CElevView::IsAirspaceToDraw(CAirspace* ptAirspace)
{
	BOOL bDraw = FALSE;

	CAirspace::TYPE aspType = ptAirspace->GetType();
	CAirspace::CLASS aspClass = ptAirspace->GetClass();

	if (aspType == CAirspace::TCR)
		bDraw = TRUE;			// draw every terminal control area

	if (aspType == CAirspace::CONTROL_ZONE)
		bDraw = TRUE;			// draw every control zone

	if (aspType == CAirspace::CONTROL_AREA)
	{							// draw control areas of class A, B, C and D
		if (aspClass == CAirspace::A ||
			aspClass == CAirspace::B ||
			aspClass == CAirspace::C ||
			aspClass == CAirspace::D)
			bDraw = TRUE;
	}

	return bDraw;
}

/****************************************************************************
 *	ElevView.cpp					D r a w E l e v I C A O					*
 ****************************************************************************/
void CElevView::DrawElevICAO (CDC* pDC, CCalculate* ptLegs, BOOL bAntipod)
{

	CWayDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	if (pDoc != NULL)
	{
		long		lTotalTime_min;
		short		i, nCnt;
		short		nAltDim;
		CLatLon		LLold;
		CWayPoint	WayPt;

		nCnt	= pDoc->GetWayPointCnt();
		nAltDim = DIM_FEET;

		DRECT rAltTime = m_ptAltGraph->GetBorder();			// min and max x/y-Values

	//	m_ptAltGraph->Clip(TRUE);			/* draw on map only		*/

		lTotalTime_min = 0;

		for (i=0; i<nCnt; i++)
		{
			LEGTYPE Leg = ptLegs->GetAt (i);	// get calculated leg

			long lAlt, lStartOfLegTime_min;
			if (pDoc->GetWayPointPtr (i, &WayPt))
			{
				CLatLon LLact(WayPt.GetLat(), WayPt.GetLon());

				lAlt = WayPt.GetActAlt_ft();
				lStartOfLegTime_min = lTotalTime_min;

									// evtl. dist auf X-AchseLeg.dDistLeg, DIM_NM
				lTotalTime_min += (long)Leg.dTimeLeg; // rounded(!) time on X-axis, at least 1 minute
				lAlt = (long)ptDim->ConvertDist (lAlt, nAltDim, ptDim->Altitude());


				if (i>0)	
				{
					if (ptElevDoc != NULL)
					{
						m_ptColor->SelectElevPen (pDC, TRUE);


						short nStartInd, nEndInd;		// Elev Index
						if (ptElevDoc->GetMinMaxIndex (LLold, LLact, &nStartInd, &nEndInd))
						{
							int j;
							BOOL bAdded = FALSE;

							for (j=nStartInd; j<=nEndInd; j++)
							{							// for all Elevs around actual leg
								if (ptElevDoc->AddElevChanges(LLold, LLact, j))
								{
									bAdded = TRUE;
								}
							}

							if (bAdded)
							{				// change color for drawing elevation
								CBrush* ptOldBrush;
								CBrush brBackGround (m_bForPrinter? RGB (220, 220, 220) :
																	m_ptColor->GetColElevArea());
								ptOldBrush = (CBrush*)pDC->SelectObject (&brBackGround);

								for (j=0; j<ptElevDoc->GetElevChangeCnt(); j++)
								{		// get all strored elev changes for actual leg in sorted order	
									ELEVPOS	From, To;
									double	dTime_min;
									long	lElev_ft, lTime_sec, LPx, LPy;

									if (ptElevDoc->GetElevChange(j, &From, &To, &lElev_ft))
									{
												// convert distances to relative times since start
										if (Leg.dVgr > 0)
										{						// draw elev rect
											CRect rElev;
											// lElev_ft is valid from From.dDist_NM to To.dDist_NM

											lAlt = 0;			// bottom left
											if (From.dDist_NM==0) dTime_min = 0;		// start of leg
												else dTime_min = this->GetTimeForDist_min (From.dDist_NM, Leg.dDistLeg_NM, Leg.dTimeLeg);

											lTime_sec = (long)((lStartOfLegTime_min + dTime_min) * 60);

											if (lAlt < rAltTime.bottom)	lAlt = (long)rAltTime.bottom;
											m_ptAltGraph->XYtoLP (lTime_sec, lAlt, &LPx, &LPy);											
											rElev.left = LPx;
											rElev.bottom = LPy;

																// top right
											lAlt = (long)ptDim->ConvertDist (lElev_ft, DIM_FEET, ptDim->Altitude());
											dTime_min = this->GetTimeForDist_min (To.dDist_NM, Leg.dDistLeg_NM, Leg.dTimeLeg);
											lTime_sec = (long)((lStartOfLegTime_min + dTime_min) * 60);
										
											if (lAlt > rAltTime.top)	lAlt = (long)rAltTime.top;
											if (lAlt < rAltTime.bottom) lAlt = (long)rAltTime.bottom;
											m_ptAltGraph->XYtoLP (lTime_sec, lAlt, &LPx, &LPy);
											rElev.right = LPx;
											rElev.top = LPy;

											pDC->Rectangle(&rElev);
										}
									}
								}

								ptElevDoc->RemoveElevChanges();
							}
						}

						m_ptColor->SelectElevPen (pDC, FALSE);
					} // ptElevDoc != NUL
				}  // i > 0

				LLold = LLact;
			}	// if (GetWayPointPtr...
		} // for all waypoints

//		m_ptAltGraph->Clip(FALSE);
	}
}

/****************************************************************************
 *	ElevView.cpp					D r a w E l e v 1 k m					*
 ****************************************************************************/
void CElevView::DrawElev1km (CDC* pDC, CCalculate* ptLegs, BOOL bAntipod)
{
	CWayDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	if (pDoc != NULL)
	{
		long		LPx0, LPy0, LPx1, LPy1, lTotalTime_min;
		short		i, nCnt;
		short		nAltDim;
		CLatLon		LLold;
		CWayPoint	WayPt;

		LPx1 = LPy1 = 0;
		nCnt	= pDoc->GetWayPointCnt();
		nAltDim = DIM_FEET;

		CElevArrayDoc* ptElevArrayDoc = ptInit->GetElevArrayDocPtr();

		DRECT rAltTime = m_ptAltGraph->GetBorder();			// min and max x/y-Values

	//	m_ptAltGraph->Clip(TRUE);			/* draw on map only		*/

		lTotalTime_min = 0;

		for (i=0; i<nCnt; i++)
		{
			LEGTYPE Leg = ptLegs->GetAt (i);	// get calculated leg

			long lAlt, lStartOfLegTime_min;
			if (pDoc->GetWayPointPtr (i, &WayPt))
			{
				CLatLon LLact(WayPt.GetLat(), WayPt.GetLon());

				lAlt = WayPt.GetActAlt_ft();
				lStartOfLegTime_min = lTotalTime_min;

									// evtl. dist auf X-AchseLeg.dDistLeg, DIM_NM
				lTotalTime_min += (long)Leg.dTimeLeg; // rounded(!) time on X-axis, at least 1 minute
				lAlt = (long)ptDim->ConvertDist (lAlt, nAltDim, ptDim->Altitude());
				m_ptAltGraph->XYtoLP (lTotalTime_min*60, lAlt, &LPx1, &LPy1);


				if (i>0)	
				{
					if (ptElevArrayDoc != NULL)
					{				// for elevation profile OR airspaces
						long DPx0, DPx1, DPx;
						long LPx, LPy, LPbottom;

						m_ptColor->SelectElevArrayPen (pDC, TRUE);

												// get LP bottom, used to draw elev lines
						long lTopAlt = (long)rAltTime.top;
						long lBaseAlt = (long)rAltTime.bottom;
						m_ptAltGraph->XYtoLP (0, lBaseAlt, &LPx, &LPy);											
						LPbottom = LPy;

												// convert LPx0 and LPx1 to device pixels

						POINT DP;
						DP.x = LPx0;
						DP.y = 0;
						pDC->LPtoDP(&DP);
						DPx0 = DP.x;	

						DP.x = LPx1;
						pDC->LPtoDP(&DP);
						DPx1 = DP.x;


						if (Leg.dTimeLeg > 0)
						{
									// Leg.dTimeLeg was rounded up, so we have to reduce speed
									// this is required to stop for loop exactly at end of actual leg
							double dLegDist_NM = LLold.LoxoDist(LLact);
							double dRoundedLegTime_h	= Leg.dTimeLeg / 60;
							double dRoundedSpeed_kt = dLegDist_NM / dRoundedLegTime_h;

							for (DPx = DPx0; DPx<DPx1; DPx++)
							{						// for every device pixel of current leg:
								POINT LP;
								LP.x = DPx;
								LP.y = 0;
								pDC->DPtoLP(&LP);	// get LP.x for next pixel
								

													// get actual time_sec
								double dTime_sec, dDummyAlt;
								m_ptAltGraph->LPtoXY(LP.x, LP.y, &dTime_sec, &dDummyAlt);	// !!! BAD

								double dLegTime_sec = (double)(dTime_sec - lStartOfLegTime_min*60);
								if (dLegTime_sec < 0) dLegTime_sec = 0;
								double dLegTime_h = dLegTime_sec / 3600;

												
									// get dist from LPx after lTime_sec
								double dDist_NM = dRoundedSpeed_kt * dLegTime_h;

												// get Lat Lon of leg after dLegTime_sec
								CLatLon LL = CLatLon::PointOnLoxoLegAtDist (dDist_NM, LLold, LLact);

								// here we know the corresponding LatLon position

								long lTerrainAlt = NO_ALT;
												
								short nElev_m;		// get elev_m for LatLon
								if (ptElevArrayDoc->ArrayReadElev_m (LL.GetLat(), LL.GetLon(), &nElev_m))
								{
												// get LPy for elev_m
									lTerrainAlt = (long)ptDim->ConvertDist (nElev_m, DIM_METER, ptDim->Altitude());
									if (lTerrainAlt > rAltTime.top)	lTerrainAlt = (long)rAltTime.top;
									if (lTerrainAlt < rAltTime.bottom) lTerrainAlt = (long)rAltTime.bottom;
									m_ptAltGraph->XYtoLP ((long)dTime_sec, lTerrainAlt, &LPx, &LPy);

												// draw line from LPy to LPbottom
									pDC->MoveTo (LPx, LPy);			
									pDC->LineTo (LPx, LPbottom);
								}
							} // for all DPx...

						} // Leg.dTimeLeg > 0

						m_ptColor->SelectElevArrayPen (pDC, FALSE);

					} // ptElevArrayDoc != NULL

				}  // i > 0

				LPx0 = LPx1;
				LPy0 = LPy1;

				LLold = LLact;
			}	// if (GetWayPointPtr...
		} // for all waypoints

		if (ptElevArrayDoc != NULL)
			ptElevArrayDoc->ArrayClose();

//		m_ptAltGraph->Clip(FALSE);
	}
}

/****************************************************************************
 *	ElevView.cpp				D r a w A i r s p a c e s					*
 ****************************************************************************/
void CElevView::DrawAirspaces (CDC* pDC, CCalculate* ptLegs, BOOL bAntipod)
{

	CWayDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

		// TODO: add draw code for native data here            

	if (pDoc != NULL)
	{
		long		LPx0, LPy0, LPx1, LPy1, lTotalTime_min;
		short		i, nCnt;
		short		nAltDim;
		CLatLon		LLold;
		CWayPoint	WayPt;


		CElevArrayDoc* ptElevArrayDoc = ptInit->GetElevArrayDocPtr();
		CAirspaceRegionsDoc* ptAirspaceRegionsDoc = ptInit->GetAirspaceRegionsDocPtr();
		if (ptAirspaceRegionsDoc == NULL)
			return;


		LPx1 = LPy1 = 0;
		nCnt	= pDoc->GetWayPointCnt();
		nAltDim = DIM_FEET;

		DRECT rAltTime = m_ptAltGraph->GetBorder();			// min and max x/y-Values

		CBorder routeBorder;			// get border around route
		BOOL bShowNav = FALSE;
		pDoc->GetMapBorder (&routeBorder, bShowNav);
		DRECT rLatLon = routeBorder.GetLatLonRect();


		long lTerrainAlt = NO_ALT;


		short nRegionIndex = 0;
		while (ptAirspaceRegionsDoc->OpenAirspaceRegionForRect(rLatLon, &nRegionIndex))
		{
		//	m_ptAltGraph->Clip(TRUE);			/* draw on map only		*/

			lTotalTime_min = 0;

			for (i=0; i<nCnt; i++)
			{
				LEGTYPE Leg = ptLegs->GetAt (i);	// get calculated leg

				long lAlt, lStartOfLegTime_min;
				if (pDoc->GetWayPointPtr (i, &WayPt))
				{
					CLatLon LLact(WayPt.GetLat(), WayPt.GetLon());

					lAlt = WayPt.GetActAlt_ft();
					lStartOfLegTime_min = lTotalTime_min;

										// evtl. dist auf X-AchseLeg.dDistLeg, DIM_NM
					lTotalTime_min += (long)Leg.dTimeLeg; // rounded(!) time on X-axis, at least 1 minute
					lAlt = (long)ptDim->ConvertDist (lAlt, nAltDim, ptDim->Altitude());
					m_ptAltGraph->XYtoLP (lTotalTime_min*60, lAlt, &LPx1, &LPy1);


					if (i>0)	
					{

						if (ptElevArrayDoc != NULL || ptAirspaceRegionsDoc != NULL)
						{				// for elevation profile OR airspaces
							long DPx0, DPx1, DPx;
							long LPx, LPy, LPbottom;

							m_ptColor->SelectElevArrayPen (pDC, TRUE);

													// get LP bottom, used to draw elev lines
							long lTopAlt = (long)rAltTime.top;
							long lBaseAlt = (long)rAltTime.bottom;
							m_ptAltGraph->XYtoLP (0, lBaseAlt, &LPx, &LPy);											
							LPbottom = LPy;

													// convert LPx0 and LPx1 to device pixels

							POINT DP;
							DP.x = LPx0;
							DP.y = 0;
							pDC->LPtoDP(&DP);
							DPx0 = DP.x;	

							DP.x = LPx1;
							pDC->LPtoDP(&DP);
							DPx1 = DP.x;


							if (Leg.dTimeLeg > 0)
							{
										// Leg.dTimeLeg was rounded up, so we have to reduce speed
										// this is required to stop for loop exactly at end of actual leg
								double dLegDist_NM = LLold.LoxoDist(LLact);
								double dRoundedLegTime_h	= Leg.dTimeLeg / 60;
								double dRoundedSpeed_kt = dLegDist_NM / dRoundedLegTime_h;

								for (DPx = DPx0; DPx<DPx1; DPx++)
								{						// for every device pixel of current leg:
									POINT LP;
									LP.x = DPx;
									LP.y = 0;
									pDC->DPtoLP(&LP);	// get LP.x for next pixel
									

														// get actual time_sec
									double dTime_sec, dDummyAlt;
									m_ptAltGraph->LPtoXY(LP.x, LP.y, &dTime_sec, &dDummyAlt);	// !!! BAD

									double dLegTime_sec = (double)(dTime_sec - lStartOfLegTime_min*60);
									if (dLegTime_sec < 0) dLegTime_sec = 0;
									double dLegTime_h = dLegTime_sec / 3600;

													
										// get dist from LPx after lTime_sec
									double dDist_NM = dRoundedSpeed_kt * dLegTime_h;

													// get Lat Lon of leg after dLegTime_sec
									CLatLon LL = CLatLon::PointOnLoxoLegAtDist (dDist_NM, LLold, LLact);

									// here we know the corresponding LatLon position


										// get airspace info for LatLon
									short nAirspaceIndex = 0;

									CAirspace* ptAirspace = NULL;
									while (ptAirspaceRegionsDoc->GetAirspaceAt(LL.GetLat(), LL.GetLon(),
															&ptAirspace, &nAirspaceIndex))
									{
										// if lat lon is within an airspace
										long lUpperAlt = ptAirspace->GetUpperAlt();
										long lLowerAlt = ptAirspace->GetLowerAlt();
										short nAltDim;
										
										BOOL bDraw = IsAirspaceToDraw(ptAirspace);
										if (bDraw && (lUpperAlt != NO_ALT || lLowerAlt != NO_ALT))
										{
											// at least one alt given

											if (lUpperAlt == NO_ALT)
											{
												lUpperAlt = 10000;		// FL 100
												nAltDim = DIM_FEET;
											}
											else
											{
												nAltDim = ptAirspace->GetElevDim();
											}

											lUpperAlt = (long)ptDim->ConvertDist (lUpperAlt, nAltDim, ptDim->Altitude());
											if (lUpperAlt > lTopAlt)
												lUpperAlt = lTopAlt;


											if (lLowerAlt == NO_ALT)
											{		
												BOOL bIgnoreGround = TRUE;
												if (bIgnoreGround)
												{		// ignore ground, call DrawAirspaces before DrawElev1km
													lLowerAlt = lBaseAlt;
												}
												else
												{		// get elev_m for current position
													short nElev_m;	
													if (ptElevArrayDoc->ArrayReadElev_m (LL.GetLat(), LL.GetLon(), &nElev_m))
													{
																	// get LPy for elev_m
														lTerrainAlt = (long)ptDim->ConvertDist (nElev_m, DIM_METER, ptDim->Altitude());
														if (lTerrainAlt > rAltTime.top)	lTerrainAlt = (long)rAltTime.top;
														if (lTerrainAlt < rAltTime.bottom) lTerrainAlt = (long)rAltTime.bottom;
													}

													if (lTerrainAlt != NO_ALT)
														lLowerAlt = lTerrainAlt;		// GND
													else		// no ConvertDist required!!
														lLowerAlt = lBaseAlt;
												}
											}
											else
											{
												nAltDim = ptAirspace->GetElevDim();
												lLowerAlt = (long)ptDim->ConvertDist (lLowerAlt, nAltDim, ptDim->Altitude());
											}

											if (lLowerAlt < lTopAlt)
											{			// only if bottom of airspace is visible:
													// draw points of top and bottom of airspace
												COLORREF crColor = ptAirspace->GetColor();

												CPen* ptPen = new CPen(PS_SOLID, 8, crColor);
												CPen* ptOldPen = (CPen*)pDC->SelectObject (ptPen);

												m_ptAltGraph->XYtoLP ((long)dTime_sec, lUpperAlt, &LPx, &LPy);
										//		pDC->SetPixel(LPx, LPy, crColor);
												pDC->MoveTo(LPx, LPy);

												m_ptAltGraph->XYtoLP ((long)dTime_sec, lLowerAlt, &LPx, &LPy);
										//		pDC->SetPixel(LPx, LPy, crColor);
												pDC->LineTo(LPx, LPy);

												pDC->SelectObject(ptOldPen);
												delete ptPen;
											}
										} // bDraw
									} // while GetAirspaceAt
								} // for all DPx...

							} // Leg.dTimeLeg > 0

							m_ptColor->SelectElevArrayPen (pDC, FALSE);

						} // ptElevArrayDoc != NULL

					}  // i > 0

					LPx0 = LPx1;
					LPy0 = LPy1;

					LLold = LLact;
				}	// if (GetWayPointPtr...
			} // for all waypoints
//			m_ptAltGraph->Clip(FALSE);

		} // while airspace regions available for current route

		if (ptElevArrayDoc != NULL)
			ptElevArrayDoc->ArrayClose();
	}
}


/****************************************************************************
 *	ElevView.cpp				D r a w R o u t e							*
 ****************************************************************************/
void CElevView::DrawRoute (CDC* pDC, CCalculate* ptLegs, BOOL bAntipod)
{
	CWayDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);
	if (pDoc != NULL)
	{
		long		LPx0, LPy0, LPx1, LPy1, lTotalTime_min;
		short		i, nCnt;
		short		nAltDim;
		CLatLon		LLold;
		CWayPoint	WayPt;
		CPen*		ptPlAltPen;
		CPen*		ptOldPen;

		LPx1 = LPy1 = 0;
		nCnt	= pDoc->GetWayPointCnt();
		nAltDim = DIM_FEET;

		ELEVPROFIL*	ptElevProfil = new ELEVPROFIL[nCnt];	// allocate array for line angles

		if (m_bForPrinter)
		{
			ptPlAltPen = new CPen (PS_SOLID, m_ptColor->GetMapLineSize(), RGB(0,0,0));
			ptOldPen = (CPen*)pDC->SelectObject (ptPlAltPen);
		}
		else
		{
			m_ptColor->SelectRoutePen (pDC, TRUE);
		}

	//	m_ptAltGraph->Clip(TRUE);			/* draw on map only		*/

		lTotalTime_min = 0;

		for (i=0; i<nCnt; i++)
		{
			LEGTYPE Leg = ptLegs->GetAt (i);	// get calculated leg

			long lAlt, lStartOfLegTime_min;
			if (pDoc->GetWayPointPtr (i, &WayPt))
			{
				CLatLon LLact(WayPt.GetLat(), WayPt.GetLon());

				lAlt = WayPt.GetActAlt_ft();
				lStartOfLegTime_min = lTotalTime_min;

									// evtl. dist auf X-AchseLeg.dDistLeg, DIM_NM
				lTotalTime_min += (long)Leg.dTimeLeg; // rounded(!) time on X-axis, at least 1 minute
				lAlt = (long)ptDim->ConvertDist (lAlt, nAltDim, ptDim->Altitude());
				m_ptAltGraph->XYtoLP (lTotalTime_min*60, lAlt, &LPx1, &LPy1);

				(ptElevProfil+i)->lX = LPx1;
				(ptElevProfil+i)->lY = LPy1;
				(ptElevProfil+i)->dAngle_deg = 0;	// for index=0 dAngle_deg is unused!

				if (i>0)	
				{
					(ptElevProfil+i)->dAngle_deg = this->LineAngle_deg (LPx0, LPy0, LPx1, LPy1);

					pDC->MoveTo (LPx0, LPy0);				// draw planned alt
					pDC->LineTo (LPx1, LPy1);
				}  // i > 0

				LPx0 = LPx1;
				LPy0 = LPy1;

				LLold = LLact;
			}	// if (GetWayPointPtr...
		} // for all waypoints


//		m_ptAltGraph->Clip(FALSE);


		if (m_bForPrinter)
		{			
			pDC->SelectObject (ptOldPen);
			delete ptPlAltPen;
		}
		else
		{
			m_ptColor->SelectRoutePen (pDC, FALSE);
		}

		for (i=0; i<nCnt; i++)		// show label activated
		{												/* name of waypoint		*/
			double	TextAngle;
			BOOL	bRightAlign;
			short	nSizeLP;
			CString	szNameForRte; 
			BOOL	bShowActWpt = FALSE;

			if (pDoc->GetWayPointPtr (i, &WayPt))
			{
				TextAngle = ptElevDoc->LabelDirection (ptElevProfil, i, nCnt, &bRightAlign);
				nSizeLP = WayPt.GetTextWidth(pDC);
				szNameForRte = WayPt.GetNameForRte ();
				this->DrawLabel (pDC, (ptElevProfil+i)->lX, (ptElevProfil+i)->lY, TextAngle,
					m_LineY, bRightAlign, szNameForRte, nSizeLP, bShowActWpt);
			}
		}

		delete [] ptElevProfil;		// delete array of ELEVPROFIL structs, used for LabelDirection
	}
}


/************************************************************************
 *  ElevView.cpp    			 D r a w S k y							*
 ************************************************************************/
void CElevView::DrawSky (CDC* pDC, CRect& rMapLP)
{
	CBrush* ptOldBrush;					// activate blue sky for grid background
	CBrush brBackGround (m_ptColor->GetColSkyArea());
	CPen* ptOldPen;
	CPen SkyPen (PS_SOLID, 1, m_ptColor->GetColSkyArea());
	if (!m_bForPrinter)
	{
		ptOldBrush = (CBrush*)pDC->SelectObject (&brBackGround);
		ptOldPen = (CPen*)pDC->SelectObject (&SkyPen);
	}
	pDC->Rectangle(rMapLP);

	if (!m_bForPrinter) 
	{
		pDC->SelectObject (ptOldBrush);	// deactivate blue color
		pDC->SelectObject (ptOldPen);
	}
}

/************************************************************************
 *  ElevView.cpp    		 D r a w R o u t e E l e v					*
 ************************************************************************/
BOOL CElevView::DrawRouteElev (CDC* pDC, CWayDoc* ptWayDoc, 
					    int* ptY, BOOL bDraw)
{
//BOOL	bEndOfPage = FALSE;
int		X, Y;
CString szText;
CFont* ptOldFont;
CRect	rMapLP;

ActivateDrawing (bDraw);

ptOldFont = (CFont*)pDC->SelectObject (&m_PlainFont);
Y = m_LineY;
  
CRect rView = this->GetUseableRectLP (pDC);


CPlane	ActPlane;
CCalculate* ptLegs = NULL;
if (ptPlaneDoc != NULL)
	{
	short nIndex = ptPlaneDoc->GetActIndex();
	if (!ptPlaneDoc->GetPlanePtr(nIndex, &ActPlane))
		return TRUE;		// bEndOfPage = TRUE;
	}

					/* do prepare calculations		*/
ptLegs = ptWayDoc->GetCalculationPtr();
ptLegs->Activate(&ActPlane);

if (ptLegs->DoConsumCalc (&ActPlane, ptMet, ptLoad))
{

	long lMinX, lMaxX, lMinY, lMaxY;
	short	nGridX, nGridY;

	DRECT rMinMap;

			// needs CONSUMTYPE* ptC = ptLegs->GetConsPtr();
	ptWayDoc->GetElevMapBorders (&lMinX, &lMaxX, &lMinY, &lMaxY, &nGridX, &nGridY);
	rMinMap.top		= lMaxY;
	rMinMap.bottom  = lMinY;
	rMinMap.left	= lMinX;
	rMinMap.right	= lMaxX;

	m_rMap = rMinMap;

	m_ptAltGraph->CalculateGraph (pDC, m_rMap, rView, m_bForPrinter, nGridX, nGridY);


	rMapLP = m_ptAltGraph->GetGridRect();	// returns m_rGridLP
	if (m_bForPrinter)
		{
		m_ptAltGraph->SetTitle (ptWayDoc->GetTitle());
		}

	this->DrawSky (pDC, rMapLP);	// draw background of grid

	m_ptAltGraph->DrawGrid (pDC, m_CharX, m_LineY, &X, &Y);

	this->DrawElevICAO (pDC, ptLegs, FALSE);	 // bAntipod = FALSE;
	this->DrawAirspaces (pDC, ptLegs, FALSE);	 // bAntipod = FALSE;	
	this->DrawElev1km (pDC, ptLegs, FALSE);	 // bAntipod = FALSE;
						// draws altitude or speed with virtual function
	this->DrawRoute (pDC, ptLegs, FALSE);	 // bAntipod = FALSE;


	pDC->SelectObject (ptOldFont);

}
ptLegs->DeActivate();








return FALSE;
}

/////////////////////////////////////////////////////////////////////////////
// CElevView drawing


/************************************************************************
 *  ElevView.cpp   			  P r i n t P a g e							*
 ************************************************************************/
void CElevView::PrintPage (CDC* pDC, int nPageToPrint)
{
CWayDoc* pDoc = GetDocument();
ASSERT_VALID(pDoc);

int	Y;
int	X_TAB;
int	nPage = 1;

if (pDoc != NULL)
	{
	BOOL bDraw;

	do	{
		X_TAB = 4*m_CharX;
		Y = 3*m_LineY;

		bDraw = (nPage == nPageToPrint);
		m_bMore = DrawRouteElev (pDC, pDoc, &Y, bDraw);
		nPage++;
		} while (!bDraw);
	}    
}                                  

/************************************************************************
 *  ElevView.cpp   					  O n D r a w						*
 ************************************************************************/
void CElevView::OnDraw(CDC* pDC)
{
	CWayDoc* pDoc = GetDocument();
	ASSERT_VALID(pDoc);

	// TODO: add draw code for native data here            

int	Y = m_LineY;

if (pDoc != NULL)
	{
	BOOL	bMore = FALSE;

	bMore = DrawRouteElev (pDC, pDoc, &Y, m_bDraw);
	}	
}


/////////////////////////////////////////////////////////////////////////////
// CElevView printing


/************************************************************************
 *  ElevView.cpp   		  O n P r e p a r e P r i n t i n g				*
 ************************************************************************/
BOOL CElevView::OnPreparePrinting(CPrintInfo* pInfo)
{			 // called before the display of the print dialog
	// default preparation
m_bMore = TRUE;	               

UINT nMinPage = 1;						// Number of the first page of the document
UINT nMaxPage = 1;						// Number of the last page of the document
pInfo->SetMinPage (nMinPage);
pInfo->SetMaxPage (nMaxPage);

return DoPreparePrinting(pInfo);
}

/************************************************************************
 *  ElevView.cpp  			O n B e g i n P r i n t i n g				*
 ************************************************************************/
void CElevView::OnBeginPrinting(CDC* pDC, CPrintInfo* pInfo)
{			   // called after the print dialog exists
	// TODO: add extra initialization before printing
CWayDoc* pDoc = GetDocument();
ASSERT_VALID(pDoc);  

if (pDoc != NULL)
	{	        
//	pDC->SetMapMode(MM_LOMETRIC);

//	this->UpdateFonts (pDC);					// defines m_CharX

//    this->SetXTabs (pDC, pDoc, m_CharX);
	}
							// 2. possibility to set the document length 
}

/************************************************************************
 *  ElevView.cpp   		 		 O n P r i n t							*
 ************************************************************************/
void CElevView::OnPrint(CDC* pDC, CPrintInfo* pInfo) 
{
	// TODO: Add your specialized code here and/or call the base class
this->PrintPage (pDC, pInfo->m_nCurPage);
}

/************************************************************************
 *  ElevView.cpp  			O n E n d P r i n t i n g					*
 ************************************************************************/
void CElevView::OnEndPrinting(CDC* /*pDC*/, CPrintInfo* /*pInfo*/)
{
	// TODO: add cleanup after printing
}

/////////////////////////////////////////////////////////////////////////////
// CElevView diagnostics

#ifdef _DEBUG
void CElevView::AssertValid() const
{
	CTextViewTool::AssertValid();
}

void CElevView::Dump(CDumpContext& dc) const
{
	CTextViewTool::Dump(dc);
}

CWayDoc* CElevView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CWayDoc)));
	return (CWayDoc*)m_pDocument;
}

#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CElevView message handlers

void CElevView::OnEditCopy() 
{
	// TODO: Add your command handler code here
this->CopyToClipboard();		
}
