/*********************************************************************
 *						G a r m i n L i n k 1 E v . c p p		     *
 *********************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "pf.h"
#include "math.h"			// fabs

#include "DimDoc.h"  
#include "Location.h"			// defines INDICATOR_SIZE, WP_USER
#include "GpsDevice.h"
#include "GarminLink1Ev.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

extern CDimDoc*			ptDim;

/************************************************************************
 *  GarminEv.cpp				C G a r m i n E v e n t		CONSTRUCTOR	*
 ************************************************************************/
CGarminLink1Event::CGarminLink1Event()
{
	m_nFunc = 0;
	m_dwDataLen = 0;
	m_ptData = NULL;

	m_uCheckSum = 0;
	m_uFill = 0;
}

/************************************************************************
 *  GarminEv.cpp				C G a r m i n E v e n t		CONSTRUCTOR	*
 ************************************************************************/
CGarminLink1Event::CGarminLink1Event(CGarminLink1Event& SourceEvt)
{ 
*this = SourceEvt;
}

/************************************************************************
 *  GarminEv.cpp			~ C G a r m i n	E v e n t		DESTRUCTOR	*
 ************************************************************************/
CGarminLink1Event::~CGarminLink1Event()
{
if (m_ptData != NULL)
	{
	delete [] m_ptData;
	m_ptData = NULL;
	}
}

/************************************************************************
 *  GarminEv.cpp				o p e r a t o r  = 						*
 ************************************************************************/
CGarminLink1Event& CGarminLink1Event::operator =(const CGarminLink1Event& evt)
{
	if (m_ptData != NULL)
	{			// delete old data
		delete [] m_ptData;
		m_ptData = NULL;
		m_dwDataLen = 0;
	}

    m_nFunc	= evt.m_nFunc;
    m_dwDataLen	= evt.m_dwDataLen;
	
	m_ptData = (unsigned char*)new unsigned char[m_dwDataLen];
	memcpy(m_ptData, evt.m_ptData, m_dwDataLen);
   
    m_uCheckSum = evt.m_uCheckSum;
    m_uFill		= evt.m_uFill;

	return *this;
}

/************************************************************************
 *  GarminEv.cpp				S e t D a t a L e n						*
 ************************************************************************/
BOOL CGarminLink1Event::SetDataLen (DWORD dwLen)
{
	m_dwDataLen	= dwLen;
	m_ptData = (unsigned char*)new unsigned char[m_dwDataLen];
	return (m_ptData != NULL);
}

/************************************************************************
 *  GarminEv.cpp				S e t D a t a							*
 ************************************************************************/
void CGarminLink1Event::SetData (unsigned char* ptData)
{
	_fmemcpy (m_ptData, ptData, m_dwDataLen);
}


/************************************************************************
 *  GarminEv.cpp			    G e t C h e c k S u m					*
 ************************************************************************/
short CGarminLink1Event::GetCheckSum (unsigned char* lpOut, short nFirstByte, short nLastByte)
{
short	nSum = 0;
short	i;

for (i=nFirstByte; i<=nLastByte; i++)	// build check sum
	nSum += *(lpOut + i);
return nSum;
}

/************************************************************************
 *  GarminEv.cpp				C h e c k S u m O K						*
 ************************************************************************/
BOOL CGarminLink1Event::CheckSumOK()
{
return (m_uCheckSum == m_uFill);
}




/************************************************************************
 *  GarminEv.cpp			S p l i t A c k n o w l e d g e				*
 ************************************************************************/
short CGarminLink1Event::SplitAcknowledge ()
{
short nFunc = 0;

if (m_ptData != NULL)
    {
    nFunc = (short)*(m_ptData);
    }

return nFunc;
}


/************************************************************************
 *  GarminEv.cpp			O n R e a d P r o d u c t					*
 ************************************************************************/
BOOL CGarminLink1Event::OnReadProduct (Product_Data_Type* ptProd, CFile* ptFile)
{
	BOOL bRead = FALSE;
	long lOffset = 0;

	if (m_dwDataLen > 0)
	{		
		char	szBuffer[256];
		short	nBuffLen;
		

		ptProd->product_ID = *((short*)(m_ptData + lOffset));
		lOffset += 2;

		ptProd->software_version = *((short*)(m_ptData + lOffset));
		lOffset += 2;


		nBuffLen = sprintf (szBuffer, "\r\nProdID: %d Software: %d\r\nProdukt-Info:\r\n",
			ptProd->product_ID, ptProd->software_version);
		if (ptFile != NULL)
			ptFile->Write ((const void*)szBuffer, (UINT)nBuffLen);

	
		CStringArray prodInfos;

		do
		{
			char szText[256];
			SplitLimZeroTermString (&lOffset, szText, 256);
			CString prodInfo(szText);
			prodInfos.Add(prodInfo);

			if (ptProd->product_description.GetLength() == 0)
				ptProd->product_description = prodInfo;

			nBuffLen = sprintf (szBuffer, "%s\r\n", szText);
			if (ptFile != NULL)
				ptFile->Write ((const void*)szBuffer, (UINT)nBuffLen);

		} while ((DWORD)lOffset < m_dwDataLen);

		bRead = TRUE;
	}
	

	return bRead;
}


/************************************************************************
 *  GarminEv.cpp			S p l i t N u m b O f C m d s				*
 ************************************************************************/
short CGarminLink1Event::SplitNumbOfCmds(CFile* ptFile)
{
	short nNumbOfCmds = 0;

	if (m_ptData != NULL)
    {
		nNumbOfCmds = *((short*)m_ptData);

		
		if (ptFile != NULL)
		{
			long    lLen;
			char	szBuffer[256];
			lLen = sprintf (szBuffer, "\r\nNumber of records=%d\r\n", nNumbOfCmds);
			if (ptFile != NULL)
				ptFile->Write ((const void*)szBuffer, (UINT)lLen);
		}
    }

	return nNumbOfCmds;
}


/************************************************************************
 *  GarminEv.cpp		S p l i t L i m Z e r o T e r m S t r i n g		*
 ************************************************************************/
short CGarminLink1Event::SplitLimZeroTermString(long* ptOffset, char* ptText, short nMaxLen)
{
	short j=0;
	unsigned char ch;
    do	{
    	ch = (unsigned char)*((unsigned char*)m_ptData + (*ptOffset)++); 
   		*(ptText + j++) = (char)ch;  
    	} while (ch > 0 && j < nMaxLen);

	if (j==nMaxLen)
		*(ptText + nMaxLen-1) = 0;

	return j;
}

/************************************************************************
 *  GarminEv.cpp		S p l i t D 2 0 0 R o u t e H e a d e r			*
 ************************************************************************/
short CGarminLink1Event::SplitD200RouteHeader()
{
short nNumbOfRte = 0;

if (m_ptData != NULL)
    {
    nNumbOfRte = (short)*((unsigned char*)m_ptData);
    }

return nNumbOfRte;
}

/************************************************************************
 *  GarminEv.cpp		S p l i t D 2 0 1 R o u t e H e a d e r			*
 ************************************************************************/
short CGarminLink1Event::SplitD201RouteHeader(char* szName)
{
short nNumbOfRte = 0;

if (m_ptData != NULL)
    {
    short	i, j;
    BOOL	bStop = FALSE;
    
    i=0;                  
    j=0;
    nNumbOfRte = (short)*((unsigned char*)m_ptData + i++);               
    do	{
    	unsigned char ch = (unsigned char)*((unsigned char*)m_ptData + i++); 
    	if (ch==0x5F)
    		{
    		*szName = 0;
    		bStop=TRUE;
    		}       
    	else{
    		*(szName + j++) = (char)ch;  
    		}
    	if (ch == 0)
    		bStop = TRUE;
    	} while (!bStop);
    }

return nNumbOfRte;
}

/************************************************************************
 *  GarminEv.cpp		S p l i t D 2 0 2 R o u t e H e a d e r			*
 ************************************************************************/
short CGarminLink1Event::SplitD202RouteHeader(char* szName)
{
	short nStrLen=0;

	if (m_ptData != NULL)
    {
    short	i;
    BOOL	bStop = FALSE;
    
    i=0;                  
    nStrLen=0;
    do	{
    	unsigned char ch = (unsigned char)*((unsigned char*)m_ptData + i++); 
    	if (ch==0x5F)
    		{
    		*szName = 0;
    		bStop=TRUE;
    		}       
    	else{
    		*(szName + nStrLen++) = (char)ch;  
    		}
    	if (ch == 0)
    		bStop = TRUE;
    	} while (!bStop);
    }

return nStrLen;
}

/************************************************************************
 *  GarminEv.cpp		S p a c e T e r m i n a t e d T e x t			*
 ************************************************************************/
void CGarminLink1Event::SpaceTerminatedText (char* szText, short nBuffLen, 
							unsigned char* lpData, short nMaxData)
{       
short	i, nDataLen, nCopy;
BOOL	bEnd = FALSE;
  
memset (szText, 0, nBuffLen); 			// initialize text buffer

nDataLen = nMaxData;      				// get data length
for (i=nMaxData-1; i>=0 && !bEnd; i--)
	if (*(lpData+i) == ' ')	nDataLen--;
			else			bEnd = TRUE;
			
nCopy = (nDataLen > nBuffLen-1)? nBuffLen-1 : nDataLen;

for (i=0; i<nCopy; i++)
	*(szText+i) = *(lpData+i);
}

/************************************************************************
 *  GarminEv.cpp			T e x t T o L o w e r						* 
 *  Purpose: Convert string ABCD EFG to Abcd Efg						*
 ************************************************************************/
void CGarminLink1Event::TextToLower (char* szText)
{
short 	i, nLen;     
char	ch;
BOOL	bUpper = TRUE;

nLen = strlen (szText);

for (i=0; i<nLen; i++)
	{
	ch = *(szText + i);
	if (!bUpper)
		*(szText + i) = tolower(ch); 
	bUpper = (ch == ' ');
	}
}   

/************************************************************************
 *  GarminEv.cpp			S p l i t W r i t e W p t					*
 ************************************************************************/
BOOL CGarminLink1Event::SplitWriteWpt (CFile* ptFile,
		    char* szIndic, char* szName, double* ptLat, double* ptLon,
		    short* ptCC, unsigned char* ptCategory)
{
BOOL	bOK = FALSE;
long	lLat, lLon, lTime;
char	szFacility[31];

if (m_ptData != NULL)
    {
	char	szBuffer[256];
    long    lOffs = 0;
    long    lLen;

    this->SpaceTerminatedText (szIndic, INDICATOR_SIZE, m_ptData+lOffs, 6);
    lOffs += 6;

    *(m_ptData + lOffs) = 0;		// ignore first byte of long
    lLat = *((long*)(m_ptData + lOffs));
    *ptLat = (double)lLat/KOORD_FAKT;
    lOffs += 4;

    *(m_ptData + lOffs) = 0;		// ignore first byte of long
    lLon = *((long*)(m_ptData + lOffs));
    *ptLon = (double)lLon/KOORD_FAKT;
    lOffs += 4;

    lTime = *((unsigned long*)(m_ptData + lOffs));
    lOffs += 4;
         
	if (m_dwDataLen >= 124)
		{              
		lOffs = 62;
		this->SpaceTerminatedText (szFacility, 31, m_ptData+lOffs, 30);

		lOffs = 92;
		this->SpaceTerminatedText (szName, SIZEOF_ORT, m_ptData+lOffs, 24);
        this->TextToLower (szName);
        
		lOffs = 120;
		*ptCC = *((short*)(m_ptData+lOffs));
	
		lOffs = 123;
		*ptCategory = *((unsigned char*)(m_ptData+lOffs));
		}
         
    lLen = sprintf (szBuffer, "\r\nCity=%s Facility=%s Indic=%s, Lat=%f, Lon=%f\r\nCC=%d, Cat=%d, Time=%ld\r\n",
	       		szName, szFacility, szIndic, *ptLat, *ptLon, *ptCC, (short)(*ptCategory),
				lTime);
    if (ptFile != NULL)
		ptFile->Write ((const void*)szBuffer, (UINT)lLen);
    bOK = TRUE;
    }

return bOK;
}



/************************************************************************
 *  GarminEv.cpp			I s V a l u e N o t 1 E 2 5					*
 ************************************************************************/
BOOL CGarminLink1Event::IsValueNot1E25(float fValue)
{
	BOOL bValueOK = TRUE;
	double fDiff = fabs(fValue - (float)(1.00000e+025));
	bValueOK = (fDiff > 0.000001);	// 1 EXP 25
	return bValueOK;
}

/************************************************************************
 *  GarminEv.cpp		G a r m i n T o S y s t e m T i m e 			*
 ************************************************************************/
unsigned long CGarminLink1Event::GarminToSystemTime (unsigned long lGarminTime)
{
	unsigned long lTimeDiff = 7304 * (24*3600);
	unsigned long lSystemTime = lGarminTime + lTimeDiff;
	return lSystemTime;
}


/************************************************************************
 *  GarminEv.cpp		S p l i t D 3 0 0 T r a c k P o i n t			*
 ************************************************************************/
BOOL CGarminLink1Event::SplitD300TrackPoint (CFile* ptFile,
		    double* ptLat, double* ptLon, unsigned long* ptTime, BOOL* ptStart)
{
BOOL	bOK = FALSE;
D300_Trk_Point_Type	GarminWpt;

if (m_ptData != NULL)
    {
	char	szBuffer[256];
    long    lLen;

	short nBlkSize = sizeof (D300_Trk_Point_Type);

												// read garmin data struct
	memcpy (&GarminWpt, m_ptData, nBlkSize);
												// convert data to preflight format
	*ptLat = (double)GarminWpt.posn.lat/KOORD_FAKT;
	*ptLon = (double)GarminWpt.posn.lon/KOORD_FAKT;
 
	*ptTime = this->GarminToSystemTime (GarminWpt.time);
	*ptStart = GarminWpt.new_trk;
   
    lLen = sprintf (szBuffer, "\r\nLat=%f, Lon=%f Time=%ld, bStart=%d\r\n",
	       		*ptLat, *ptLon, *ptTime, *ptStart);
    if (ptFile != NULL)
		ptFile->Write ((const void*)szBuffer, (UINT)lLen);
    bOK = TRUE;
    }

return bOK;
}

/************************************************************************
 *  GarminEv.cpp		S p l i t D 3 0 1 T r a c k P o i n t			*
 ************************************************************************/
BOOL CGarminLink1Event::SplitD301TrackPoint (CFile* ptFile,
		    double* ptLat, double* ptLon, unsigned long* ptTime, 
			double* ptAlt_ft, BOOL* ptAltOK, BOOL* ptStart)
{
BOOL	bOK = FALSE;
D301_Trk_Point_Type	GarminWpt;

*ptLat = NO_KOORD;
*ptLon = NO_KOORD;
*ptTime = 0;
*ptAlt_ft = 0;
*ptAltOK = FALSE;
*ptStart = FALSE;

if (m_ptData != NULL)
    {
	char	szBuffer[256];
    long    lLen;

	short nBlkSize = sizeof (D301_Trk_Point_Type);

												// read garmin data struct
	memcpy (&GarminWpt, m_ptData, nBlkSize);
												// convert data to preflight format
	*ptLat = (double)GarminWpt.posn.lat/KOORD_FAKT;
	*ptLon = (double)GarminWpt.posn.lon/KOORD_FAKT;
 
	*ptTime = this->GarminToSystemTime (GarminWpt.time);

	*ptAlt_ft = -1000;
	*ptAltOK = IsValueNot1E25(GarminWpt.alt);
	if (*ptAltOK)
	{							// gps sends altitude in meters					
		*ptAlt_ft = ptDim->ConvertDist (GarminWpt.alt, DIM_METER, DIM_FEET);
	}

	double dDepth=0;				// unused!!
	BOOL bDepthOK = IsValueNot1E25(GarminWpt.dpth);
	if (bDepthOK)
	{							// gps sends depth in meters					
		dDepth = ptDim->ConvertDist (GarminWpt.dpth, DIM_METER, DIM_FEET);
	}

	*ptStart = GarminWpt.new_trk;

         
    lLen = sprintf (szBuffer, "\r\nLat=%f, Lon=%f Time=%ld, Alt_ft=%f\r\nDepth_ft=%f, bStart=%d\r\n",
	       		*ptLat, *ptLon, *ptTime, *ptAlt_ft, dDepth, *ptStart);
    if (ptFile != NULL)
		ptFile->Write ((const void*)szBuffer, (UINT)lLen);
    bOK = TRUE;
    }

return bOK;
}

/************************************************************************
 *  GarminEv.cpp		S p l i t D 3 0 2 T r a c k P o i n t			*
 ************************************************************************/
BOOL CGarminLink1Event::SplitD302TrackPoint (CFile* ptFile,
		    double* ptLat, double* ptLon, unsigned long* ptTime, 
			double* ptAlt_ft, BOOL* ptAltOK, 
			double* ptTemp_C, BOOL* ptTempOK, BOOL* ptStart)
{
BOOL	bOK = FALSE;
D302_Trk_Point_Type	GarminWpt;

*ptLat = NO_KOORD;
*ptLon = NO_KOORD;
*ptTime = 0;
*ptAlt_ft = 0;
*ptAltOK = FALSE;
*ptTemp_C = 0;
*ptTempOK = FALSE;
*ptStart = FALSE;

if (m_ptData != NULL)
    {
	char	szBuffer[256];
    long    lLen;

	short nBlkSize = sizeof (D302_Trk_Point_Type);

												// read garmin data struct
	memcpy (&GarminWpt, m_ptData, nBlkSize);
												// convert data to preflight format
	*ptLat = (double)GarminWpt.posn.lat/KOORD_FAKT;
	*ptLon = (double)GarminWpt.posn.lon/KOORD_FAKT;
 
	*ptTime = this->GarminToSystemTime (GarminWpt.time);

	*ptAlt_ft = -1000;
	*ptAltOK = IsValueNot1E25(GarminWpt.alt);
	if (*ptAltOK)
	{							// gps sends altitude in meters					
		*ptAlt_ft = ptDim->ConvertDist (GarminWpt.alt, DIM_METER, DIM_FEET);
	}

	double dDepth=0;				// unused!!
	BOOL bDepthOK = IsValueNot1E25(GarminWpt.dpth);
	if (bDepthOK)
	{							// gps sends depth in meters					
		dDepth = ptDim->ConvertDist (GarminWpt.dpth, DIM_METER, DIM_FEET);
	}

	*ptTempOK = IsValueNot1E25(GarminWpt.temp);
	if (*ptTempOK)
	{
		*ptTemp_C = GarminWpt.temp;
	}


	*ptStart = GarminWpt.new_trk;

         
    lLen = sprintf (szBuffer, "\r\nLat=%f, Lon=%f Time=%ld, Alt_ft=%f\r\nDepth_ft=%f, Temp=%f, bStart=%d\r\n",
	       		*ptLat, *ptLon, *ptTime, *ptAlt_ft, dDepth, *ptTemp_C, *ptStart);
    if (ptFile != NULL)
		ptFile->Write ((const void*)szBuffer, (UINT)lLen);
    bOK = TRUE;
    }

return bOK;
}

/************************************************************************
 *  GarminEv.cpp		S p l i t D 3 0 3 T r a c k P o i n t			*
 ************************************************************************/
BOOL CGarminLink1Event::SplitD303TrackPoint (CFile* ptFile,
		    double* ptLat, double* ptLon, unsigned long* ptTime, 
			double* ptAlt_ft, BOOL* ptAltOK, BYTE* ptHeartRate, BOOL* ptFirstStopTrk, BOOL* ptStartLap)
{
	BOOL	bOK = FALSE;
	D303_Trk_Point_Type	GarminWpt;

	*ptLat = NO_KOORD;
	*ptLon = NO_KOORD;
	*ptTime = 0;
	*ptAlt_ft = 0;
	*ptAltOK = FALSE;
	*ptHeartRate = 0;
	*ptFirstStopTrk = FALSE;
	*ptStartLap = FALSE;

	if (m_ptData != NULL)
    {
		char	szBuffer[256];
		long    lLen;

		short nBlkSize = sizeof (D303_Trk_Point_Type);

													// read garmin data struct
		memcpy (&GarminWpt, m_ptData, nBlkSize);

		*ptAltOK = IsValueNot1E25(GarminWpt.alt);
		BOOL bHeartRateValid = (GarminWpt.heart_rate == 0);

		BOOL bInvalPos;
		bInvalPos = (GarminWpt.posn.lat == 0x7FFFFFFF) && (GarminWpt.posn.lon = 0x7FFFFFFF);
		if (bInvalPos)
		{	//	Two consecutive track points with invalid position, invalid altitude, 
			//	and invalid heart rate indicate a pause in track point recording 
			//  during the time between the two points.
			if (!(*ptAltOK) && !bHeartRateValid)
			{
				if (*ptFirstStopTrk)
				{			// last track was already invalid, this is really the end
					*ptStartLap = TRUE;
					*ptFirstStopTrk = FALSE;
					bOK = TRUE;
				}
				else
				{
					*ptFirstStopTrk = TRUE;
				}
			}
			else
			{		// some values are valid, reset bool
				*ptFirstStopTrk = FALSE;
			}
		}
		else
		{										// convert data to preflight format
			*ptLat = (double)GarminWpt.posn.lat/KOORD_FAKT;
			*ptLon = (double)GarminWpt.posn.lon/KOORD_FAKT;
 
			*ptTime = this->GarminToSystemTime (GarminWpt.time);
			
			*ptAlt_ft = -1000;
			if (*ptAltOK)
			{							// gps sends altitude in meters					
				*ptAlt_ft = ptDim->ConvertDist (GarminWpt.alt, DIM_METER, DIM_FEET);
			}

			*ptHeartRate = GarminWpt.heart_rate;

			*ptStartLap = FALSE;

         
			lLen = sprintf (szBuffer, "\r\nLat=%f, Lon=%f Time=%ld, Alt_ft=%f\r\nHeartRate=%d\r\n",
	       				*ptLat, *ptLon, *ptTime, *ptAlt_ft, *ptHeartRate);
			if (ptFile != NULL)
				ptFile->Write ((const void*)szBuffer, (UINT)lLen);
			bOK = TRUE;
		}
	}
return bOK;
}

/************************************************************************
 *  GarminEv.cpp		S p l i t D 3 0 4 T r a c k P o i n t			*
 ************************************************************************/
BOOL CGarminLink1Event::SplitD304TrackPoint (CFile* ptFile,
		    double* ptLat, double* ptLon, unsigned long* ptTime, 
			double* ptAlt_ft, BOOL* ptAltOK, 
			double* ptDist_m, BOOL* ptDistOK, 
			BYTE* ptHeartRate, BYTE* ptCadence, BOOL* ptFirstStopTrk, BOOL* ptStartLap)
{
	BOOL	bOK = FALSE;
	D304_Trk_Point_Type	GarminWpt;

	*ptLat = NO_KOORD;
	*ptLon = NO_KOORD;
	*ptTime = 0;
	*ptAlt_ft = 0;
	*ptAltOK = FALSE;
	*ptDist_m = 0;
	*ptDistOK = FALSE;
	*ptHeartRate = 0;
	*ptCadence = 0;
	*ptFirstStopTrk = FALSE;
	*ptStartLap = FALSE;


	if (m_ptData != NULL)
    {
		char	szBuffer[256];
		long    lLen;

		short nBlkSize = sizeof (D304_Trk_Point_Type);

													// read garmin data struct
		memcpy (&GarminWpt, m_ptData, nBlkSize);

		*ptAltOK = IsValueNot1E25(GarminWpt.alt);
		BOOL bHeartRateValid = (GarminWpt.heart_rate != 0);

		BOOL bInvalPos;
		bInvalPos = (GarminWpt.posn.lat == 0x7FFFFFFF) && (GarminWpt.posn.lon = 0x7FFFFFFF);
		if (bInvalPos)
		{	//	Two consecutive track points with invalid position, invalid altitude, 
			//	and invalid heart rate indicate a pause in track point recording 
			//  during the time between the two points.
			if (!(*ptAltOK) && !bHeartRateValid)
			{
				if (*ptFirstStopTrk)
				{			// last track was already invalid, this is really the end
					*ptStartLap = TRUE;
					*ptFirstStopTrk = FALSE;
					bOK = TRUE;
				}
				else
				{
					*ptFirstStopTrk = TRUE;
				}
			}
			else
			{		// some values are valid, reset bool
				*ptFirstStopTrk = FALSE;
			}
		}
		else
		{
													// convert data to preflight format
			*ptLat = (double)GarminWpt.posn.lat/KOORD_FAKT;
			*ptLon = (double)GarminWpt.posn.lon/KOORD_FAKT;
 
			*ptTime = this->GarminToSystemTime (GarminWpt.time);

			if (*ptAltOK)
			{							// gps sends altitude in meters					
				*ptAlt_ft = ptDim->ConvertDist (GarminWpt.alt, DIM_METER, DIM_FEET);
			}


			*ptDist_m =-1;
			*ptDistOK = (GarminWpt.distance != (float)(1.00000e+025));	// 1 EXP 25
			if (*ptDistOK)
			{							// gps sends depth in meters	
				*ptDist_m = GarminWpt.distance;
			//	*ptDist_m = ptDim->ConvertDist (GarminWpt.distance, DIM_METER, DIM_METER);
			}


	//		BOOL bHeartRateOK = (GarminWpt.heart_rate != NULL);
			*ptHeartRate = GarminWpt.heart_rate;

	//		BOOL bCadenceOK = (GarminWpt.cadence != 0xFF);
			*ptCadence = GarminWpt.cadence;

			*ptStartLap = FALSE;

         
			lLen = sprintf (szBuffer, "\r\nLat=%f, Lon=%f Time=%ld, Alt_ft=%f\r\nDist_m=%f, heartRate=%d, Cadence=%d, Sensor=%d\r\n",
	       				*ptLat, *ptLon, *ptTime, *ptAlt_ft, *ptDist_m, *ptHeartRate, *ptCadence, GarminWpt.sensor);
			if (ptFile != NULL)
				ptFile->Write ((const void*)szBuffer, (UINT)lLen);
			bOK = TRUE;
		}
    }

return bOK;
}

/************************************************************************
 *  GarminEv.cpp		S p l i t D 3 1 0 T r a c k H e a d e r			*
 ************************************************************************/
BOOL CGarminLink1Event::SplitD310TrackHeader(CFile* ptFile, BOOL* ptDisplay, char* ptColor, CString* ptHeader)
{
	BOOL	bOK = FALSE;
	D310_Trk_Hdr_Type header;

	if (m_ptData != NULL)
    {
		char	szBuffer[256];
		long    lOffs = 0;
		long    lLen;

		header.dspl = m_ptData[lOffs++];
		header.color = m_ptData[lOffs++];
		lLen = this->SplitLimZeroTermString(&lOffs, header.trk_ident, 52);
		bOK = (lLen > 0);

		*ptDisplay = (header.dspl == 1);
		*ptColor = (char)header.color;
		*ptHeader = header.trk_ident;

		lLen = sprintf (szBuffer, "\r\nbDisplay=%d Color=%d, TrkIdent=%s\r\n",
	       		header.dspl, (int)header.color, header.trk_ident);

		if (ptFile != NULL)
			ptFile->Write ((const void*)szBuffer, (UINT)lLen);
	}

	return bOK;
}

/************************************************************************
 *  GarminEv.cpp		S p l i t D 3 1 1 T r a c k H e a d e r			*
 ************************************************************************/
BOOL CGarminLink1Event::SplitD311TrackHeader(CFile* ptFile, short* ptIndex)
{
	BOOL	bOK = FALSE;
	D311_Trk_Hdr_Type header;

	if (m_ptData != NULL)
    {
		char	szBuffer[256];
		long    lLen;
		long    lOffs = 0;

		header.index = m_ptData[lOffs++];

		*ptIndex = header.index;

		lLen = sprintf (szBuffer, "\r\nIndex=%d\r\n", header.index);

		if (ptFile != NULL)
			ptFile->Write ((const void*)szBuffer, (UINT)lLen);
		bOK = TRUE;
	}

	return bOK;
}

/************************************************************************
 *  GarminEv.cpp		S p l i t D 3 1 2 T r a c k H e a d e r			*
 ************************************************************************/
BOOL CGarminLink1Event::SplitD312TrackHeader(CFile* ptFile, BOOL* ptDisplay, char* ptColor, CString* ptHeader)
{
	BOOL	bOK = FALSE;
	D312_Trk_Hdr_Type header;

	if (m_ptData != NULL)
    {
		char	szBuffer[256];
		long    lOffs = 0;
		long    lLen;

		header.dspl = m_ptData[lOffs++];
		header.color = m_ptData[lOffs++];
		lLen = this->SplitLimZeroTermString(&lOffs, header.trk_ident, 52);
		bOK = (lLen > 0);

		*ptDisplay = (header.dspl == 1);
		*ptColor = (char)header.color;
		*ptHeader = header.trk_ident;

		lLen = sprintf (szBuffer, "\r\nbDisplay=%d Color=%d, TrkIdent=%s\r\n",
	       		header.dspl, (int)header.color, header.trk_ident);

		if (ptFile != NULL)
			ptFile->Write ((const void*)szBuffer, (UINT)lLen);
	}

	return bOK;

}


/////////////////////////////////////////////////////////////////////////////
// CGarminLink1Event diagnostics

#ifdef _DEBUG
void CGarminLink1Event::AssertValid() const
{
	CObject::AssertValid();
}

void CGarminLink1Event::Dump(CDumpContext& dc) const
{
	CObject::Dump(dc);
}

#endif //_DEBUG

IMPLEMENT_DYNAMIC(CGarminLink1Event, CObject)
