/************************************************************************
 *  					G a r m i n W p t . c p p	  					*
 ************************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "math.h"					// log, exp

#include "DimDoc.h"			// NO_KOORD, NO_ALT, ...
#include "GpsDevice.h"
#include "GarminProtocols.h"		// D100_Wpt
#include "GarminWpt.h"



/************************************************************************
 *  GarminWpt.cpp			 C G a r m i n W p t 		Constructor		*
 ************************************************************************/
CGarminWpt::CGarminWpt(CGpsDevice* ptGpsDevice)
{ 
	m_ptGpsDevice = ptGpsDevice;

    m_dLat 	= NO_KOORD;
    m_dLon	= NO_KOORD;
    m_szName	= "";
    m_szIdent	= "";
    m_nCategory	= 0;
    m_dAlt_m	= NO_ALT;
    m_szCC		= "";
	m_bStoreAsUserWpt = FALSE;
}


/************************************************************************
 *  GarminWpt.cpp			 ~ C G a r m i n W p t		Destructor		*
 ************************************************************************/
CGarminWpt::~CGarminWpt()
{
} 


/************************************************************************
 *  GarminWpt.cpp				S e t R t e W p t						*
 ************************************************************************/
void CGarminWpt::SetRteWpt (double dLat, double dLon, CString& szName, 
			CString& szIndicator, short nCategory, double dAlt_m, short nCC, BOOL bStoreAsUserWpt)
{
	if (szIndicator.GetLength() > 0)
		m_szIdent = szIndicator;
	else
		m_szIdent = szName;

	m_dLat		= dLat;
	m_dLon		= dLon;
	m_szName	= szName;
	m_nCategory	= nCategory;
	m_dAlt_m	= dAlt_m;

	char szTemp[3];
	memset (szTemp, 0, 3);
    *((short*)(szTemp)) = nCC;		// copy short into first 2 bytes of m_szCC
	m_szCC = szTemp;

	m_bStoreAsUserWpt = bStoreAsUserWpt;
}

/************************************************************************
 *  GarminWpt.cpp				G e t R t e W p t						*
 ************************************************************************/
void CGarminWpt::GetRteWpt (double* ptLat, double* ptLon, CString* ptName, 
			CString* ptIndicator, short* ptCategory, double* ptAlt_m, short* ptCC)
{
	*ptLat		= m_dLat;
	*ptLon		= m_dLon;
	*ptName		= m_szName;
	*ptIndicator= m_szIdent;
	*ptCategory	= m_nCategory;
	*ptAlt_m	= m_dAlt_m;

	char szTemp[3];
	strcpy(szTemp, (LPCTSTR)m_szCC);
    *ptCC = *((short*)(szTemp));	// convert first 2 bytes of m_szCC to short
}


/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 0 0 W p t T y p e					page 31	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD100WptType(unsigned char* ptData, long lLen)
{	// GPS 38, GPS 40, GPS 45, GPS 75, GPS II
    BOOL		bOK = false;
    D100_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D100_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;
        CString szComment;

 		m_szIdent = this->SplitSpaceTerminatedText (ptData, &lOffset, 6);
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        GarminWpt.unused = m_ptGpsDevice->SplitLong (ptData, &lOffset);
        szComment = this->SplitSpaceTerminatedText (ptData, &lOffset, 40);

        bOK = true;
    }
    return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 0 1 W p t T y p e					page 31	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD101WptType(unsigned char* ptData, long lLen)
{	// GPSMAP210, GPSMAP 220
    BOOL		bOK = false;
    D101_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D101_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;
        CString szComment;

		m_szIdent = this->SplitSpaceTerminatedText (ptData, &lOffset, 6);
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        GarminWpt.unused = m_ptGpsDevice->SplitLong (ptData, &lOffset);
        szComment = this->SplitSpaceTerminatedText (ptData, &lOffset, 40);
        GarminWpt.dst = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
	
        GarminWpt.smbl = *(ptData + lOffset);		// symbol id, see pg. 32
		lOffset += 1;
        
        m_szName = m_szIdent;
        bOK = true;
    }
    return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 0 2 W p t T y p e					page 32	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD102WptType(unsigned char* ptData, long lLen)
{		// GPSMAP175, GPSMAP 210, GPSMAP220
    BOOL		bOK = false;
    D102_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D102_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;
        CString szComment;
        
		m_szIdent = this->SplitSpaceTerminatedText (ptData, &lOffset, 6);
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        GarminWpt.unused = m_ptGpsDevice->SplitLong (ptData, &lOffset);
        szComment = this->SplitSpaceTerminatedText (ptData, &lOffset, 40);
        GarminWpt.dst = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
        GarminWpt.smbl = m_ptGpsDevice->SplitShort (ptData, &lOffset);
        
        m_szName = m_szIdent;
        bOK = true;
    }
    return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 0 3 W p t T y p e					page 32	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD103WptType(unsigned char* ptData, long lLen)
{		// GPS 12, GPS 12 XL, GPS 48, GPS II Plus
    BOOL		bOK = false;
    D103_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D103_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;
        CString szComment;

		m_szIdent = this->SplitSpaceTerminatedText (ptData, &lOffset, 6);
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        GarminWpt.unused = m_ptGpsDevice->SplitLong (ptData, &lOffset);
        szComment = this->SplitSpaceTerminatedText (ptData, &lOffset, 40);
	
        GarminWpt.smbl = *(ptData + lOffset);		// symbol id, see pg. 36
		lOffset += 1;
        
        GarminWpt.dspl = *(ptData + lOffset);		//  display symbol, see pg. 36
		lOffset += 1;
        
        m_szName = m_szIdent;
        bOK = true;
    }
    return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 0 4 W p t T y p e					page 33	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD104WptType(unsigned char* ptData, long lLen)
{	// GPS III
    BOOL		bOK = false;
    D104_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D104_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;
        CString szComment;
        
		m_szIdent = this->SplitSpaceTerminatedText (ptData, &lOffset, 6);
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        GarminWpt.unused = m_ptGpsDevice->SplitLong (ptData, &lOffset);
        szComment = this->SplitSpaceTerminatedText (ptData, &lOffset, 40);
        GarminWpt.dst = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
        GarminWpt.smbl = m_ptGpsDevice->SplitShort (ptData, &lOffset);

        GarminWpt.dspl = *(ptData + lOffset);		// display symbol, see pg. 37
		lOffset += 1;
        
        m_szName = m_szIdent;
        bOK = true;
    }
    return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 0 5 W p t T y p e					page 33	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD105WptType(unsigned char* ptData, long lLen)
{		// StreetPilot (user waypoints)
    BOOL		bOK = false;
    D105_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D105_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;
        
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        GarminWpt.smbl = m_ptGpsDevice->SplitShort (ptData, &lOffset);
        GarminWpt.wpt_ident = m_ptGpsDevice->SplitCString(ptData, &lOffset);
       
        m_szIdent = GarminWpt.wpt_ident;
        m_szName  = m_szIdent;
        bOK = true;
    }
    return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 0 6 W p t T y p e					page 33	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD106WptType(unsigned char* ptData, long lLen)
{		// StreetPilot (route waypoints)
    BOOL		bOK = false;
    D106_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D106_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;
        
        GarminWpt.wpt_class = *(ptData + lOffset);
		lOffset += 1;
        
	//	byte subclass[18];		/* subclass 18 */
		memcpy (&GarminWpt.subclass, (ptData + lOffset), 18);
		lOffset += 18;

		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        GarminWpt.smbl = m_ptGpsDevice->SplitShort (ptData, &lOffset);
        GarminWpt.wpt_ident = m_ptGpsDevice->SplitCString(ptData, &lOffset);
        GarminWpt.lnk_ident = m_ptGpsDevice->SplitCString(ptData, &lOffset);
       
        m_szIdent= GarminWpt.wpt_ident;
        m_szName = GarminWpt.lnk_ident;

        bOK = true;
    }
    return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 0 7 W p t T y p e					page 34	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD107WptType(unsigned char* ptData, long lLen)
{	// GPS 12CX
    BOOL		bOK = false;
    D107_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D107_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;
        CString szComment;

		m_szIdent = this->SplitSpaceTerminatedText (ptData, &lOffset, 6);
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        GarminWpt.unused = m_ptGpsDevice->SplitLong (ptData, &lOffset);
        szComment = this->SplitSpaceTerminatedText (ptData, &lOffset, 40);
        
        GarminWpt.smbl = *(ptData + lOffset);		// symbol id, see pg. 36
		lOffset += 1;
        GarminWpt.dspl = *(ptData + lOffset);		// symbol id, see pg. 36
		lOffset += 1;
       
        GarminWpt.dst = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
	
        GarminWpt.color = *(ptData + lOffset);		// waypoint color
        lOffset += 1;
       
        m_szName = m_szIdent;
        bOK = true;
    }
    return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 0 8 W p t T y p e					page 34	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD108WptType(unsigned char* ptData, long lLen)
{
    BOOL		bOK = false;
    D108_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D108_Wpt_Type));
   
    if (ptData != NULL)
    {
        long    lOffset = 0;

		GarminWpt.wpt_class = *(ptData + lOffset++);/* class (see below) 		*/
		GarminWpt.color = *(ptData + lOffset++);	/* color (see below) 		*/
		GarminWpt.dspl = *(ptData + lOffset++);		/* display options (see below) 	*/
		GarminWpt.attr = *(ptData + lOffset++);		/* attributes (see below) 	*/
        
        GarminWpt.smbl = m_ptGpsDevice->SplitShort (ptData, &lOffset); /* waypoint symbol */
	
	//	byte subclass[18];		/* subclass 18 */
		memcpy (&GarminWpt.subclass, (ptData + lOffset), 18);
		lOffset += 18;
		
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        
                                        /* altitude in meters			*/
        GarminWpt.alt = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
                                   		/* depth in meters				*/
        GarminWpt.dpth = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
                                        /* proximity distance in meters */
        GarminWpt.dist = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 


		memcpy (&GarminWpt.state, (ptData + lOffset), 2);
		lOffset += 2;
		memcpy (&GarminWpt.cc, (ptData + lOffset), 2);
		lOffset += 2;
		
	//	char state[2];			/* state 2 */
	//	char cc[2];			/* country code 2 */

		
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.ident, 51);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.comment, 51);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.facility, 31);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.city, 25);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.addr, 51);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.cross_road, 51);

                                          // convert data to standard format
        m_szName	= GarminWpt.facility;
        m_szIdent	= GarminWpt.ident;
        BOOL bGpsToStd = true;
        m_nCategory	= ConvertD108WptClass(bGpsToStd, GarminWpt.wpt_class);
        
        if (GarminWpt.alt != 0x0000004A)	// 1 EXP 25
            m_dAlt_m	= (double)GarminWpt.alt;
            
        SetCC(GarminWpt.cc);

//	TextToLower (szName);

        bOK = true;
    }

return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 0 9 W p t T y p e					page 36	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD109WptType(unsigned char* ptData, long lLen)
{
    BOOL		bOK = false;
    D109_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D109_Wpt_Type));
   
    if (ptData != NULL)
    {
        long    lOffset = 0;

		GarminWpt.dtyp = *(ptData + lOffset++);			/* data packet type (0x01 for D109) 	*/
		GarminWpt.wpt_class = *(ptData + lOffset++);	/* class (see below) 		*/

		short nColor, nDspl;
		GarminWpt.dspl_color = *(ptData + lOffset++);	/* color see pg. 39 		*/
		SplitD109DsplColor(GarminWpt.dspl_color, &nColor, &nDspl);
		// nColor may be 0,1,2,3,...,15 (see page 39) or 0x1f, see pg. 55
		// nDspl may be 0,1,2, see page 36

		GarminWpt.attr = *(ptData + lOffset++);			/* attributes (see below) 	*/
        
        GarminWpt.smbl = m_ptGpsDevice->SplitShort (ptData, &lOffset); /* waypoint symbol */
	
	//	byte subclass[18];		/* subclass 18 */
		memcpy (&GarminWpt.subclass, (ptData + lOffset), 18);
		lOffset += 18;
		
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        
                                        /* altitude in meters  */
        GarminWpt.alt = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
                                   	/* depth in meters  */
        GarminWpt.dpth = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
                                        /* proximity distance in meters  */
        GarminWpt.dist = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 


		memcpy (&GarminWpt.state, (ptData + lOffset), 2);
		lOffset += 2;
		memcpy (&GarminWpt.cc, (ptData + lOffset), 2);
		lOffset += 2;
				
	//	char state[2];			/* state 2 */
	//	char cc[2];			/* country code 2 */

        GarminWpt.ete = m_ptGpsDevice->SplitLong(ptData, &lOffset);

	
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.ident, 51);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.comment, 51);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.facility, 31);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.city, 25);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.addr, 51);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.cross_road, 51);

                                             // convert data to standard format
        m_szName	= GarminWpt.facility;
        m_szIdent	= GarminWpt.ident;
        BOOL bGpsToStd = true;
        m_nCategory	= ConvertD109WptClass(bGpsToStd, GarminWpt.wpt_class);
        
        if (GarminWpt.alt != 0x0000004A)	// 1 EXP 25
            m_dAlt_m	= (double)GarminWpt.alt;
            
        SetCC(GarminWpt.cc);

//	TextToLower (szName);

        bOK = true;
    }

return bOK;

}


/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 1 0 W p t T y p e					page 37	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD110WptType(unsigned char* ptData, long lLen)
{
    BOOL		bOK = false;
    D110_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D110_Wpt_Type));
   
    if (ptData != NULL)
    {
        long    lOffset = 0;

		GarminWpt.dtyp = *(ptData + lOffset++);			/* data packet type (0x01 for D110) 	*/
		GarminWpt.wpt_class = *(ptData + lOffset++);	/* class (see below) 		*/

		short nColor, nDspl;
		GarminWpt.dspl_color = *(ptData + lOffset++);	/* color see pg. 39 		*/
		SplitD109DsplColor(GarminWpt.dspl_color, &nColor, &nDspl);
		// nColor may be 0,1,2,3,...,15 (see page 39) or 0x1f, see pg. 55
		// nDspl may be 0,1,2, see page 36

		GarminWpt.attr = *(ptData + lOffset++);			/* attributes (see below) 	*/
        
        GarminWpt.smbl = m_ptGpsDevice->SplitShort (ptData, &lOffset); /* waypoint symbol */
	
	//	byte subclass[18];		/* subclass 18 */
		memcpy (&GarminWpt.subclass, (ptData + lOffset), 18);
		lOffset += 18;
		
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        
                                        /* altitude in meters  */
        GarminWpt.alt = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
                                   	/* depth in meters  */
        GarminWpt.dpth = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
                                        /* proximity distance in meters  */
        GarminWpt.dist = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 


		memcpy (&GarminWpt.state, (ptData + lOffset), 2);
		lOffset += 2;
		memcpy (&GarminWpt.cc, (ptData + lOffset), 2);
		lOffset += 2;
				
	//	char state[2];			/* state 2 */
	//	char cc[2];			/* country code 2 */

        GarminWpt.ete = m_ptGpsDevice->SplitLong(ptData, &lOffset);

		GarminWpt.temp = m_ptGpsDevice->SplitFloat(ptData, &lOffset);
		GarminWpt.time = (Time_Type)m_ptGpsDevice->SplitULong(ptData, &lOffset);
		GarminWpt.wpt_cat = m_ptGpsDevice->SplitUShort(ptData, &lOffset);
	
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.ident, 51);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.comment, 51);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.facility, 31);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.city, 25);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.addr, 51);
		this->SplitLimZeroTermString(ptData, &lOffset, GarminWpt.cross_road, 51);

                                             // convert data to standard format
        m_szName	= GarminWpt.facility;
        m_szIdent	= GarminWpt.ident;
        BOOL bGpsToStd = true;
        m_nCategory	= ConvertD110WptClass(bGpsToStd, GarminWpt.wpt_class);
        
        if (GarminWpt.alt != 0x0000004A)	// 1 EXP 25
            m_dAlt_m	= (double)GarminWpt.alt;
            
        SetCC(GarminWpt.cc);

//	TextToLower (szName);

        bOK = true;
    }

return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 5 0 W p t T y p e					page 40	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD150WptType(unsigned char* ptData, long lLen)
{		// GPS 150, GPS 155, GNC 250, GNC 300
    BOOL		bOK = false;
    D150_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D150_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;

        CString szComment, szCity, szState;

		m_szIdent = this->SplitSpaceTerminatedText (ptData, &lOffset, 6);

		memcpy (GarminWpt.cc, (ptData + lOffset), 2);
		lOffset += 2;

        GarminWpt.wpt_class = *(ptData + lOffset);	
		lOffset += 1;

		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        m_dAlt_m = m_ptGpsDevice->SplitShort (ptData, &lOffset);
        szCity = this->SplitSpaceTerminatedText (ptData, &lOffset, 24);
        szState = this->SplitSpaceTerminatedText (ptData, &lOffset, 2);
        m_szName = this->SplitSpaceTerminatedText (ptData, &lOffset, 30);
        szComment = this->SplitSpaceTerminatedText (ptData, &lOffset, 40);

        BOOL bGpsToStd = true;
        m_nCategory	= ConvertD150WptClass(bGpsToStd, GarminWpt.wpt_class);
        SetCC(GarminWpt.cc);

        if (m_nCategory != CGarminWpt::AIRPORT)
            m_dAlt_m = NO_ALT;

        bOK = true;
    }
    return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 5 1 W p t T y p e					page 41	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD151WptType(unsigned char* ptData, long lLen)
{	// GPS 55 AVD, GPS 89
    BOOL		bOK = false;
    D151_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D151_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;

        CString szComment, szCity, szState;

		m_szIdent = this->SplitSpaceTerminatedText (ptData, &lOffset, 6);
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        GarminWpt.unused = m_ptGpsDevice->SplitLong (ptData, &lOffset);
        szComment = this->SplitSpaceTerminatedText (ptData, &lOffset, 40);
        GarminWpt.dst = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
        m_szName = this->SplitSpaceTerminatedText (ptData, &lOffset, 30);
        szCity = this->SplitSpaceTerminatedText (ptData, &lOffset, 24);
        szState = this->SplitSpaceTerminatedText (ptData, &lOffset, 2);
        m_dAlt_m = m_ptGpsDevice->SplitShort (ptData, &lOffset);

		memcpy (GarminWpt.cc, (ptData + lOffset), 2);
		lOffset += 2;
	
        GarminWpt.unused2 = *(ptData + lOffset);	
		lOffset += 1;
	
        GarminWpt.wpt_class = *(ptData + lOffset);	
		lOffset += 1;
        
        BOOL bGpsToStd = true;
        m_nCategory	= ConvertD151WptClass(bGpsToStd, GarminWpt.wpt_class);
        SetCC(GarminWpt.cc);

        if (m_nCategory != CGarminWpt::AIRPORT)
            m_dAlt_m = NO_ALT;

        bOK = true;
    }
    return bOK;
}


/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 5 2 W p t T y p e					page 41	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD152WptType(unsigned char* ptData, long lLen)
{	// GPS 90, GPS 95 AVD, GPS95 XL and GPSCOM 190
    BOOL		bOK = false;
    D152_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D152_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;

        CString szComment, szCity, szState;

		m_szIdent = this->SplitSpaceTerminatedText (ptData, &lOffset, 6);
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        GarminWpt.unused = m_ptGpsDevice->SplitLong (ptData, &lOffset);
        szComment = this->SplitSpaceTerminatedText (ptData, &lOffset, 40);
        GarminWpt.dst = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
        m_szName = this->SplitSpaceTerminatedText (ptData, &lOffset, 30);
        szCity = this->SplitSpaceTerminatedText (ptData, &lOffset, 24);
        szState = this->SplitSpaceTerminatedText (ptData, &lOffset, 2);
        m_dAlt_m = m_ptGpsDevice->SplitShort (ptData, &lOffset);

		memcpy (GarminWpt.cc, (ptData + lOffset), 2);
		lOffset += 2;
	
        GarminWpt.unused2 = *(ptData + lOffset);	
		lOffset += 1;
	
        GarminWpt.wpt_class = *(ptData + lOffset);	
		lOffset += 1;
        
        BOOL bGpsToStd = true;
        m_nCategory	= ConvertD152WptClass(bGpsToStd, GarminWpt.wpt_class);
        SetCC(GarminWpt.cc);

        if (m_nCategory != CGarminWpt::AIRPORT)
            m_dAlt_m = NO_ALT;

        bOK = true;
    }
    return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 5 4 W p t T y p e					page 42	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD154WptType(unsigned char* ptData, long lLen)
{
    BOOL		bOK = false;
    D154_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D154_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;

        CString szComment, szCity, szState;

		m_szIdent = this->SplitSpaceTerminatedText (ptData, &lOffset, 6);
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        GarminWpt.unused = m_ptGpsDevice->SplitLong (ptData, &lOffset);
        szComment = this->SplitSpaceTerminatedText (ptData, &lOffset, 40);
        GarminWpt.dst = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
        m_szName = this->SplitSpaceTerminatedText (ptData, &lOffset, 30);
        szCity = this->SplitSpaceTerminatedText (ptData, &lOffset, 24);
        szState = this->SplitSpaceTerminatedText (ptData, &lOffset, 2);
        m_dAlt_m = m_ptGpsDevice->SplitShort (ptData, &lOffset);

		memcpy (GarminWpt.cc, (ptData + lOffset), 2);
		lOffset += 2;
	
        GarminWpt.unused2 = *(ptData + lOffset);	
		lOffset += 1;
	
        GarminWpt.wpt_class = *(ptData + lOffset);	
		lOffset += 1;
        
        GarminWpt.smbl = m_ptGpsDevice->SplitShort (ptData, &lOffset);	// Symbol_Type


        BOOL bGpsToStd = true;
        m_nCategory	= ConvertD154WptClass(bGpsToStd, GarminWpt.wpt_class);
        SetCC(GarminWpt.cc);

        if (m_nCategory != CGarminWpt::AIRPORT)
            m_dAlt_m = NO_ALT;

        bOK = true;
    }
    return bOK;
}

/********************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 5 5 W p t T y p e					page 43	*
 ********************************************************************************/
BOOL CGarminWpt::SplitD155WptType(unsigned char* ptData, long lLen)
{
    BOOL		bOK = false;
    D155_Wpt_Type	GarminWpt;
    memset (&GarminWpt, 0, sizeof(D155_Wpt_Type));
    
    if (ptData != NULL)
    {
        long    lOffset = 0;

        CString szComment, szCity, szState;

		m_szIdent = this->SplitSpaceTerminatedText (ptData, &lOffset, 6);
		SplitSemicircle(ptData, &lOffset, &m_dLat, &m_dLon);
        GarminWpt.unused = m_ptGpsDevice->SplitLong (ptData, &lOffset);
        szComment = this->SplitSpaceTerminatedText (ptData, &lOffset, 40);
        GarminWpt.dst = m_ptGpsDevice->SplitFloat (ptData, &lOffset); 
        m_szName = this->SplitSpaceTerminatedText (ptData, &lOffset, 30);
        szCity = this->SplitSpaceTerminatedText (ptData, &lOffset, 24);
        szState = this->SplitSpaceTerminatedText (ptData, &lOffset, 2);
        m_dAlt_m = m_ptGpsDevice->SplitShort (ptData, &lOffset);

		memcpy (GarminWpt.cc, (ptData + lOffset), 2);
		lOffset += 2;
	
        GarminWpt.unused2 = *(ptData + lOffset);	
		lOffset += 1;
	
        GarminWpt.wpt_class = *(ptData + lOffset);	
		lOffset += 1;
        
        GarminWpt.smbl = m_ptGpsDevice->SplitShort (ptData, &lOffset);	// Symbol_Type

        GarminWpt.dspl = *(ptData + lOffset);	
		lOffset += 1;

        BOOL bGpsToStd = true;
        m_nCategory	= ConvertD155WptClass(bGpsToStd, GarminWpt.wpt_class);
        SetCC(GarminWpt.cc);

        if (m_nCategory != CGarminWpt::AIRPORT)
            m_dAlt_m = NO_ALT;

        bOK = true;
    }
    return bOK;
}



/********************************************************************************
 *  GarminWpt.cpp				S p l i t R t e W p t							*
 ********************************************************************************/
BOOL CGarminWpt::SplitRteWpt(unsigned char* ptData, long lLen, short nWptProt)
{			
	short nLen = 0;

	if (nWptProt == D100_Wpt)
		nLen = SplitD100WptType(ptData, lLen);
	if (nWptProt == D101_Wpt)
		nLen = SplitD101WptType(ptData, lLen);
	if (nWptProt == D102_Wpt)
		nLen = SplitD102WptType(ptData, lLen);
	if (nWptProt == D103_Wpt)
		nLen = SplitD103WptType(ptData, lLen);
	if (nWptProt == D104_Wpt)
		nLen = SplitD104WptType(ptData, lLen);
	if (nWptProt == D105_Wpt)
		nLen = SplitD105WptType(ptData, lLen);
	if (nWptProt == D106_Wpt)
		nLen = SplitD106WptType(ptData, lLen);
	if (nWptProt == D107_Wpt)
		nLen = SplitD107WptType(ptData, lLen);
	if (nWptProt == D108_Wpt)
		nLen = SplitD108WptType(ptData, lLen);
	if (nWptProt == D109_Wpt)
		nLen = SplitD109WptType(ptData, lLen);
	if (nWptProt == D110_Wpt)
		nLen = SplitD110WptType(ptData, lLen);

	if (nWptProt == D150_Wpt)
		nLen = SplitD150WptType(ptData, lLen);
	if (nWptProt == D151_Wpt)
		nLen = SplitD151WptType(ptData, lLen);
	if (nWptProt == D152_Wpt)
		nLen = SplitD152WptType(ptData, lLen);

	if (nWptProt == D154_Wpt)
		nLen = SplitD154WptType(ptData, lLen);
	if (nWptProt == D155_Wpt)
		nLen = SplitD155WptType(ptData, lLen);

	return nLen;
}


/********************************************************************************
 *  GarminWpt.cpp		C o n v e r t D 1 0 8 W p t C l a s s 					*
 ********************************************************************************/
short CGarminWpt::ConvertD108WptClass(BOOL bGpsToStd, short srcClass)
{
// The enumerated values for the "wpt_class" member of the D108_Wpt_Type are defined as follows:

    enum D108_Class
	{
	USER_WPT = 0x00,	/* User waypoint */
	AVTN_APT_WPT = 0x40,	/* Aviation Airport waypoint */
	AVTN_INT_WPT = 0x41,	/* Aviation Intersection waypoint */
	AVTN_NDB_WPT = 0x42,	/* Aviation NDB waypoint */
	AVTN_VOR_WPT = 0x43,	/* Aviation VOR waypoint */
	AVTN_ARWY_WPT = 0x44,	/* Aviation Airport Runway waypoint */
	AVTN_AINT_WPT = 0x45,	/* Aviation Airport Intersection */
	AVTN_ANDB_WPT = 0x46,	/* Aviation Airport NDB waypoint */
	MAP_PNT_WPT = 0x80,		/* Map Point waypoint */
	MAP_AREA_WPT = 0x81,	/* Map Area waypoint */
	MAP_INT_WPT = 0x82,		/* Map Intersection waypoint */
	MAP_ADRS_WPT = 0x83,	/* Map Address waypoint */
	MAP_LABEL_WPT = 0x84,	/* Map Label Waypoint */
	MAP_LINE_WPT = 0x85,	/* Map Line Waypoint */
	};


    short nDestClass=0;
    
    if (bGpsToStd)
    {		// convert GPS Format into standard format
        switch(srcClass)
        {
            case USER_WPT:		/* User waypoint */
                nDestClass = CGarminWpt::USER;
                break;
            case AVTN_APT_WPT:		/* Aviation Airport waypoint */
                nDestClass = CGarminWpt::AIRPORT;
                break;
            case AVTN_INT_WPT:		/* Aviation Intersection waypoint */
                nDestClass = CGarminWpt::INTERSECTION;
                break;
            case AVTN_NDB_WPT:		/* Aviation NDB waypoint */
                nDestClass = CGarminWpt::NDB;
                break;
            case AVTN_VOR_WPT:		/* Aviation VOR waypoint */
                 nDestClass = CGarminWpt::VOR;
                break;
            case AVTN_ARWY_WPT:		/* Aviation Airport Runway waypoint */
                nDestClass = CGarminWpt::AIRPORT;
                break;
            case AVTN_AINT_WPT:		/* Aviation Airport Intersection */
                nDestClass = CGarminWpt::INTERSECTION;
                break;
            case AVTN_ANDB_WPT:		/* Aviation Airport NDB waypoint */
                nDestClass = CGarminWpt::NDB;
                break;
            case MAP_PNT_WPT:		/* Map Point waypoint */
                nDestClass = CGarminWpt::USER;
                break;
            case MAP_AREA_WPT:		/* Map Area waypoint */
                nDestClass = CGarminWpt::USER;
                break;
            case MAP_INT_WPT:		/* Map Intersection waypoint */
                nDestClass = CGarminWpt::INTERSECTION;
                break;
            case MAP_ADRS_WPT:		/* Map Address waypoint */
                 nDestClass = CGarminWpt::USER;
                break;
            case MAP_LABEL_WPT:		/* Map Label Waypoint */
                nDestClass = CGarminWpt::USER;
                break;
            case MAP_LINE_WPT:		/* Map Line Waypoint */
                nDestClass = CGarminWpt::USER;
                break;
        }
    
    }
    else
    {		// convert standard format into GPS format
		if (m_bStoreAsUserWpt)
			srcClass = CGarminWpt::USER;

        switch(srcClass)
        {
            case CGarminWpt::AIRPORT:
                nDestClass = AVTN_APT_WPT;
                break;
            case CGarminWpt::REPORT:
                nDestClass = USER_WPT;
                break;
            case CGarminWpt::INTERSECTION:
                nDestClass = AVTN_AINT_WPT;
                break;
            case CGarminWpt::NDB:
                nDestClass = AVTN_ANDB_WPT;
                break;
            case CGarminWpt::VOR:
                nDestClass = AVTN_VOR_WPT;
                break;
            case CGarminWpt::USER:
                nDestClass = USER_WPT;
                break;
            case CGarminWpt::SAIL:
                nDestClass = AVTN_APT_WPT;
                break;
            case CGarminWpt::HELI:
                nDestClass = AVTN_APT_WPT;
                break;
        }
    }
    
    return nDestClass;
}

/********************************************************************************
 *  GarminWpt.cpp		C o n v e r t D 1 0 9 W p t C l a s s 					*
 ********************************************************************************/
short CGarminWpt::ConvertD109WptClass(BOOL bGpsToStd, short srcClass)
{
	return ConvertD108WptClass(bGpsToStd, srcClass);
}

/********************************************************************************
 *  GarminWpt.cpp		C o n v e r t D 1 1 0 W p t C l a s s 					*
 ********************************************************************************/
short CGarminWpt::ConvertD110WptClass(BOOL bGpsToStd, short srcClass)
{
// The enumerated values for the "wpt_class" member of the D108_Wpt_Type are defined as follows:

    enum D110_Class
	{
		USER_WPT = 0x00, /* user waypoint */
		AVTN_APT_WPT = 0x40, /* aviation airport waypoint */
		AVTN_INT_WPT = 0x41, /* aviation intersection waypoint */
		AVTN_NDB_WPT = 0x42, /* aviation NDB waypoint */
		AVTN_VOR_WPT = 0x43, /* aviation VOR waypoint */
		AVTN_ARWY_WPT = 0x44, /* aviation airport runway waypoint */
		AVTN_AINT_WPT = 0x45, /* aviation airport intersection */
		AVTN_ANDB_WPT = 0x46, /* aviation airport ndb waypoint */
		MAP_PNT_WPT = 0x80, /* map point waypoint */
		MAP_AREA_WPT = 0x81, /* map area waypoint */
		MAP_INT_WPT = 0x82, /* map intersection waypoint */
		MAP_ADRS_WPT = 0x83, /* map address waypoint */
		MAP_LINE_WPT = 0x84, /* map line waypoint */
	};

    short nDestClass=0;
    
    if (bGpsToStd)
    {		// convert GPS Format into standard format
        switch(srcClass)
        {
            case USER_WPT:		/* User waypoint */
                nDestClass = CGarminWpt::USER;
                break;
            case AVTN_APT_WPT:		/* Aviation Airport waypoint */
                nDestClass = CGarminWpt::AIRPORT;
                break;
            case AVTN_INT_WPT:		/* Aviation Intersection waypoint */
                nDestClass = CGarminWpt::INTERSECTION;
                break;
            case AVTN_NDB_WPT:		/* Aviation NDB waypoint */
                nDestClass = CGarminWpt::NDB;
                break;
            case AVTN_VOR_WPT:		/* Aviation VOR waypoint */
                 nDestClass = CGarminWpt::VOR;
                break;
            case AVTN_ARWY_WPT:		/* Aviation Airport Runway waypoint */
                nDestClass = CGarminWpt::AIRPORT;
                break;
            case AVTN_AINT_WPT:		/* Aviation Airport Intersection */
                nDestClass = CGarminWpt::INTERSECTION;
                break;
            case AVTN_ANDB_WPT:		/* Aviation Airport NDB waypoint */
                nDestClass = CGarminWpt::NDB;
                break;
            case MAP_PNT_WPT:		/* Map Point waypoint */
                nDestClass = CGarminWpt::USER;
                break;
            case MAP_AREA_WPT:		/* Map Area waypoint */
                nDestClass = CGarminWpt::USER;
                break;
            case MAP_INT_WPT:		/* Map Intersection waypoint */
                nDestClass = CGarminWpt::INTERSECTION;
                break;
            case MAP_ADRS_WPT:		/* Map Address waypoint */
                 nDestClass = CGarminWpt::USER;
                break;
            case MAP_LINE_WPT:		/* Map Line Waypoint */
                nDestClass = CGarminWpt::USER;
                break;
        }
    
    }
    else
    {		// convert standard format into GPS format
 		if (m_bStoreAsUserWpt)
			srcClass = CGarminWpt::USER;

		switch(srcClass)
        {
            case CGarminWpt::AIRPORT:
                nDestClass = AVTN_APT_WPT;
                break;
            case CGarminWpt::REPORT:
                nDestClass = USER_WPT;
                break;
            case CGarminWpt::INTERSECTION:
                nDestClass = AVTN_AINT_WPT;
                break;
            case CGarminWpt::NDB:
                nDestClass = AVTN_ANDB_WPT;
                break;
            case CGarminWpt::VOR:
                nDestClass = AVTN_VOR_WPT;
                break;
            case CGarminWpt::USER:
                nDestClass = USER_WPT;
                break;
            case CGarminWpt::SAIL:
                nDestClass = AVTN_APT_WPT;
                break;
            case CGarminWpt::HELI:
                nDestClass = AVTN_APT_WPT;
                break;
        }
    }
    
    return nDestClass;
}

/********************************************************************************
 *  GarminWpt.cpp		C o n v e r t D 1 5 0 W p t C l a s s 					*
 ********************************************************************************/
short CGarminWpt::ConvertD150WptClass(BOOL bGpsToStd, short srcClass)
{
    //The enumerated values for the "wpt_class" member of the D150_Wpt_Type are shown below:
	enum D150_Class
	{
	apt_wpt_class = 0, /* airport waypoint class */
	int_wpt_class = 1, /* intersection waypoint class */
	ndb_wpt_class = 2, /* NDB waypoint class */
	vor_wpt_class = 3, /* VOR waypoint class */
	usr_wpt_class = 4, /* user defined waypoint class */
	rwy_wpt_class = 5, /* airport runway threshold waypoint class */
	aint_wpt_class = 6, /* airport intersection waypoint class */
	locked_wpt_class = 7 /* locked waypoint class */
	};


    short nDestClass=0;
    
    if (bGpsToStd)
    {		// convert GPS Format into standard format
        switch(srcClass)
        {
            case apt_wpt_class:		/* airport waypoint class */
                nDestClass = CGarminWpt::AIRPORT;
                break;
            case int_wpt_class:		/* intersection waypoint class */
                nDestClass = CGarminWpt::INTERSECTION;
                break;
            case ndb_wpt_class:		/* NDB waypoint class */
                nDestClass = CGarminWpt::NDB;
                break;
            case vor_wpt_class:		/* VOR waypoint class */
                nDestClass = CGarminWpt::VOR;
                break;
            case usr_wpt_class:		/* user defined waypoint class */
                nDestClass = CGarminWpt::USER;
                break;
            case rwy_wpt_class:		/* airport runway threshold waypoint class */
            case aint_wpt_class:	/* airport intersection waypoint class */
            case locked_wpt_class:	/* locked waypoint class */
                nDestClass = CGarminWpt::USER;
                break;
        }   
    }
    else
    {		// convert standard format into GPS format
		if (m_bStoreAsUserWpt)
			srcClass = CGarminWpt::USER;

		switch(srcClass)
        {
			case CGarminWpt::AIRPORT:
                nDestClass = apt_wpt_class;
				break;
			case CGarminWpt::REPORT:
	            nDestClass = usr_wpt_class;
				break;
			case CGarminWpt::INTERSECTION:
                nDestClass = int_wpt_class;
				break;
			case CGarminWpt::NDB:
                nDestClass = ndb_wpt_class;
				break;
			case CGarminWpt::VOR:
                nDestClass = vor_wpt_class;
				break;
			case CGarminWpt::USER:
                nDestClass = usr_wpt_class;
				break;
			case CGarminWpt::SAIL:
                nDestClass = apt_wpt_class;
				break;
			case CGarminWpt::HELI:
                nDestClass = apt_wpt_class;
				break;
        }
    }
    
    return nDestClass;
}




/********************************************************************************
 *  GarminWpt.cpp			C o n v e r t D 1 5 1 W p t C l a s s 				*
 ********************************************************************************/
short CGarminWpt::ConvertD151WptClass(BOOL bGpsToStd, short srcClass)
{
        //The enumerated values for the "wpt_class" member of the D151_Wpt_Type are shown below:
    enum D151_Class
    {
        apt_wpt_class = 0, 		/* airport waypoint class */
        vor_wpt_class = 1, 		/* VOR waypoint class */
        usr_wpt_class = 2, 		/* user defined waypoint class */
        locked_wpt_class = 3 		/* locked waypoint class */
    };

    short nDestClass=0;
    
    if (bGpsToStd)
    {		// convert GPS Format into standard format
        switch(srcClass)
        {
            case apt_wpt_class:		/* airport waypoint class */
                nDestClass = CGarminWpt::AIRPORT;
                break;
            case vor_wpt_class:		/* VOR waypoint class */
                nDestClass = CGarminWpt::VOR;
                break;
            case usr_wpt_class:		/* user defined waypoint class */
                nDestClass = CGarminWpt::USER;
                break;
            case locked_wpt_class:	/* locked waypoint class */
                nDestClass = CGarminWpt::USER;
                break;
        }
    
    }
    else
    {		// convert standard format into GPS format
		if (m_bStoreAsUserWpt)
			srcClass = CGarminWpt::USER;

        switch(srcClass)
        {
			case CGarminWpt::AIRPORT:
                nDestClass = apt_wpt_class;
		break;
			case CGarminWpt::REPORT:
                nDestClass = usr_wpt_class;
		break;
			case CGarminWpt::INTERSECTION:
                nDestClass = usr_wpt_class;
		break;
			case CGarminWpt::NDB:
                nDestClass = usr_wpt_class;
		break;
			case CGarminWpt::VOR:
                nDestClass = vor_wpt_class;
		break;
			case CGarminWpt::USER:
                nDestClass = usr_wpt_class;
		break;
			case CGarminWpt::SAIL:
                nDestClass = apt_wpt_class;
		break;
			case CGarminWpt::HELI:
                nDestClass = apt_wpt_class;
		break;

        }
    }
    
    return nDestClass;
}

/********************************************************************************
 *  GarminWpt.cpp		C o n v e r t D 1 5 2 W p t C l a s s					*
 ********************************************************************************/
short CGarminWpt::ConvertD152WptClass(BOOL bGpsToStd, short srcClass)
{
        //The enumerated values for the "wpt_class" member of the D151_Wpt_Type are shown below:
    enum D152_Class
    {
    apt_wpt_class = 0,		 /* airport waypoint class */
    int_wpt_class = 1, 		/* intersection waypoint class */
    ndb_wpt_class = 2, 		/* NDB waypoint class */
    vor_wpt_class = 3, 		/* VOR waypoint class */
    usr_wpt_class = 4, 		/* user defined waypoint class */
    locked_wpt_class = 5 	/* locked waypoint class */
    };

    short nDestClass=0;
    
    if (bGpsToStd)
    {		// convert GPS Format into standard format
        switch(srcClass)
        {
            case apt_wpt_class:		/* airport waypoint class */
                nDestClass = CGarminWpt::AIRPORT;
                break;
            case int_wpt_class:	 	/* intersection waypoint class */
                nDestClass = CGarminWpt::INTERSECTION;
                break;
            case ndb_wpt_class:	 	/* NDB waypoint class */
                nDestClass = CGarminWpt::NDB;
                break;
            case vor_wpt_class:	 	/* VOR waypoint class */
                nDestClass = CGarminWpt::VOR;
                break;
            case usr_wpt_class:	 	/* user defined waypoint class */
                nDestClass = CGarminWpt::USER;
                break;
            case locked_wpt_class:	/* locked waypoint class */
                nDestClass = CGarminWpt::USER;
                break;
         }
    }
    else
    {		// convert standard format into GPS format
		if (m_bStoreAsUserWpt)
			srcClass = CGarminWpt::USER;

        switch(srcClass)
        {
			case CGarminWpt::AIRPORT:
                nDestClass = apt_wpt_class;
				break;
			case CGarminWpt::REPORT:
                nDestClass = usr_wpt_class;
				break;
			case CGarminWpt::INTERSECTION:
                nDestClass = int_wpt_class;
				break;
			case CGarminWpt::NDB:
                nDestClass = ndb_wpt_class;
				break;
			case CGarminWpt::VOR:
                nDestClass = vor_wpt_class;
				break;
			case CGarminWpt::USER:
                nDestClass = usr_wpt_class;
				break;
			case CGarminWpt::SAIL:
                nDestClass = apt_wpt_class;
				break;
			case CGarminWpt::HELI:
                nDestClass = apt_wpt_class;
				break;
        }
    }
    
    return nDestClass;
}

/********************************************************************************
 *  GarminWpt.cpp		C o n v e r t D 1 5 4 W p t C l a s s			pg: 42	*
 ********************************************************************************/
short CGarminWpt::ConvertD154WptClass(BOOL bGpsToStd, short srcClass)
{
	//The enumerated values for the "wpt_class" member of the D154_Wpt_Type are shown below:
	enum D154_Class
	{
	apt_wpt_class = 0, /* airport waypoint class */
	int_wpt_class = 1, /* intersection waypoint class */
	ndb_wpt_class = 2, /* NDB waypoint class */
	vor_wpt_class = 3, /* VOR waypoint class */
	usr_wpt_class = 4, /* user defined waypoint class */
	rwy_wpt_class = 5, /* airport runway threshold waypoint class */
	aint_wpt_class = 6, /* airport intersection waypoint class */
	andb_wpt_class = 7, /* airport NDB waypoint class */
	sym_wpt_class = 8, /* user defined symbol-only waypoint class */
	locked_wpt_class = 9 /* locked waypoint class */
	};

    short nDestClass=0;
    
    if (bGpsToStd)
    {		// convert GPS Format into standard format
        switch(srcClass)
        {
           case apt_wpt_class:		/* airport waypoint class */
                nDestClass = CGarminWpt::AIRPORT;
                break;
           case int_wpt_class:		/* intersection waypoint class */
                nDestClass = CGarminWpt::INTERSECTION;
                break;
           case ndb_wpt_class:		/* NDB waypoint class */
                nDestClass = CGarminWpt::NDB;
                break;
           case vor_wpt_class:		/* VOR waypoint class */
                nDestClass = CGarminWpt::VOR;
                break;
           case usr_wpt_class:		/* user defined waypoint class */
                nDestClass = CGarminWpt::USER;
                break;
           case rwy_wpt_class:		/* airport runway threshold waypoint class */
           case aint_wpt_class:		/* airport intersection waypoint class */
           case andb_wpt_class:		/* airport NDB waypoint class */
           case sym_wpt_class:		/* user defined symbol-only waypoint class */
           case locked_wpt_class:	/* locked waypoint class */
                nDestClass = CGarminWpt::USER;
                break;
         }
    }
    else
    {		// convert standard format into GPS format
		if (m_bStoreAsUserWpt)
			srcClass = CGarminWpt::USER;
	
        switch(srcClass)
        {
			case CGarminWpt::AIRPORT:
                nDestClass = apt_wpt_class;
				break;
			case CGarminWpt::REPORT:
                nDestClass = usr_wpt_class;
				break;
			case CGarminWpt::INTERSECTION:
                nDestClass = int_wpt_class;
				break;
			case CGarminWpt::NDB:
                nDestClass = ndb_wpt_class;
				break;
			case CGarminWpt::VOR:
                nDestClass = vor_wpt_class;
				break;
			case CGarminWpt::USER:
                nDestClass = usr_wpt_class;
				break;
			case CGarminWpt::SAIL:
                nDestClass = apt_wpt_class;
				break;
			case CGarminWpt::HELI:
                nDestClass = apt_wpt_class;
				break;
        }
    }
    
    return nDestClass;
}


/********************************************************************************
 *  GarminWpt.cpp		C o n v e r t D 1 5 5 W p t C l a s s			pg: 43	*
 ********************************************************************************/
short CGarminWpt::ConvertD155WptClass(BOOL bGpsToStd, short srcClass)
{
	//The enumerated values for the "wpt_class" member of the D155_Wpt_Type are shown below:
	enum D155_Class
	{
	apt_wpt_class = 0, /* airport waypoint class */
	int_wpt_class = 1, /* intersection waypoint class */
	ndb_wpt_class = 2, /* NDB waypoint class */
	vor_wpt_class = 3, /* VOR waypoint class */
	usr_wpt_class = 4, /* user defined waypoint class */
	locked_wpt_class = 5 /* locked waypoint class */
	};

    short nDestClass=0;
    
    if (bGpsToStd)
    {		// convert GPS Format into standard format
        switch(srcClass)
        {
           case apt_wpt_class:		/* airport waypoint class */
                nDestClass = CGarminWpt::AIRPORT;
                break;
           case int_wpt_class:		/* intersection waypoint class */
                nDestClass = CGarminWpt::INTERSECTION;
                break;
           case ndb_wpt_class:		/* NDB waypoint class */
                nDestClass = CGarminWpt::NDB;
                break;
           case vor_wpt_class:		/* VOR waypoint class */
                nDestClass = CGarminWpt::VOR;
                break;
           case usr_wpt_class:		/* user defined waypoint class */
                nDestClass = CGarminWpt::USER;
                break;
           case locked_wpt_class:	/* locked waypoint class */
                nDestClass = CGarminWpt::USER;
                break;
         }
    }
    else
    {		// convert standard format into GPS format
		if (m_bStoreAsUserWpt)
			srcClass = CGarminWpt::USER;

        switch(srcClass)
        {
			case CGarminWpt::AIRPORT:
                nDestClass = apt_wpt_class;
				break;
			case CGarminWpt::REPORT:
                nDestClass = usr_wpt_class;
				break;
			case CGarminWpt::INTERSECTION:
                nDestClass = int_wpt_class;
				break;
			case CGarminWpt::NDB:
                nDestClass = ndb_wpt_class;
				break;
			case CGarminWpt::VOR:
                nDestClass = vor_wpt_class;
				break;
			case CGarminWpt::USER:
                nDestClass = usr_wpt_class;
				break;
			case CGarminWpt::SAIL:
                nDestClass = apt_wpt_class;
				break;
			case CGarminWpt::HELI:
                nDestClass = apt_wpt_class;
				break;
        }
    }
    
    return nDestClass;
}

/********************************************************************************
 *  GarminWpt.cpp		C a t e g o r y T o D 1 0 8 S y m b o l					*
 ********************************************************************************/
short CGarminWpt::CategoryToD108Symbol(short nCategory)
{
    short nSymbol=18;		 // see pages 33, 34
    switch(nCategory)
    {
        case CGarminWpt::AIRPORT:
            nSymbol = 16384;			// sym_airport
            break;
        case CGarminWpt::REPORT:
            nSymbol = 18;				// sym_wpt_dot
            break;
        case CGarminWpt::INTERSECTION:
            nSymbol = 16385;			// sym_int
            break;
        case CGarminWpt::NDB:
            nSymbol = 16386;			// sym_ndb
            break;
        case CGarminWpt::VOR:
            nSymbol = 16387;			// sym_vor
            break;
        case CGarminWpt::USER:
            nSymbol = 18;				// sym_wpt_dot
            break;
        case CGarminWpt::SAIL:
            nSymbol = 16393;			// sym_glider
            break;
        case CGarminWpt::HELI:
            nSymbol = 16388;			// sym_heliport
            break;
    }
    
    return nSymbol;
}

/********************************************************************************
 *  GarminWpt.cpp		C a t e g o r y T o D 1 0 9 S y m b o l					*
 ********************************************************************************/
short CGarminWpt::CategoryToD109Symbol(short nCategory)
{
    return CategoryToD108Symbol(nCategory);
}

/********************************************************************************
 *  GarminWpt.cpp		C a t e g o r y T o D 1 5 4 S y m b o l					*
 ********************************************************************************/
short CGarminWpt::CategoryToD154Symbol(short nCategory)
{
    return CategoryToD108Symbol(nCategory);
}

/********************************************************************************
 *  GarminWpt.cpp		C a t e g o r y T o D 1 5 5 S y m b o l					*
 ********************************************************************************/
short CGarminWpt::CategoryToD155Symbol(short nCategory)
{
    return CategoryToD108Symbol(nCategory);
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 0 0 W p t T y p e		page 35	*
 ************************************************************************/
short CGarminWpt::PrepareD100WptType(unsigned char* ptData)
{	// GPS 38, GPS 40, GPS 45, GPS 75, GPS II
    long lOffset = 0;
    
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szIdent, 6);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);		// semicircle
    m_ptGpsDevice->PrepareLong(ptData, &lOffset, 0);				// longword unused
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 40);// no comment
        
    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 0 1 W p t T y p e		page 35	*
 ************************************************************************/
short CGarminWpt::PrepareD101WptType(unsigned char* ptData)
{	// GPSMAP 210 GPSMAP 220 (both prior to version 4.00)
    long lOffset = 0;
    
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szIdent, 6);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);		// semicircle
    m_ptGpsDevice->PrepareLong(ptData, &lOffset, 0);				// longword unused
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 40);// no comment
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, 0);				// float dst

	byte smbl = 18;										// sym_wpt_dot see pg. 33
    *(ptData + lOffset++) = smbl;						// byte sbml
        
    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 0 2 W p t T y p e		page 36	*
 ************************************************************************/
short CGarminWpt::PrepareD102WptType(unsigned char* ptData)
{	// GPSMAP 175, GPSMAP 210, GPSMAP 220
    long lOffset = 0;
    
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szIdent, 6);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);		// semicircle
    m_ptGpsDevice->PrepareLong(ptData, &lOffset, 0);				// longword unused
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 40);// no comment
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, 0);				// float dst

	Symbol_Type smbl = 18;									// sym_wpt_dot see pg. 33
    m_ptGpsDevice->PrepareShort(ptData, &lOffset, smbl);			// Symbol_Type smbl
        
    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 0 3 W p t T y p e		page 36	*
 ************************************************************************/
short CGarminWpt::PrepareD103WptType(unsigned char* ptData)
{	// GPS 12, GPS 12XL, GPS 48, GPS II Plus
// The enumerated values for the "dspl" member of the D103_Wpt_Type are shown below:

enum
{
	dspl_name = 0,				/* Display symbol with waypoint name */
	dspl_none = 1,				/* Display symbol by itself */
	dspl_cmnt = 2				/* Display symbol with comment */
};

    long lOffset = 0;
    
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szIdent, 6);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);		// semicircle
    m_ptGpsDevice->PrepareLong(ptData, &lOffset, 0);				// longword unused
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 40);// no comment
 
	byte smbl = 18;										// sym_wpt_dot see pg. 33
    *(ptData + lOffset++) = smbl;						// byte sbml

	byte dspl = dspl_name;							
    *(ptData + lOffset++) = dspl;						// byte dspl
        
    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 0 4 W p t T y p e		page 37	*
 ************************************************************************/
short CGarminWpt::PrepareD104WptType(unsigned char* ptData)
{	// GPS III
// The enumerated values for the "dspl" member of the D104_Wpt_Type are shown below:

enum D105_Smbl
{
	dspl_smbl_none = 0,		/* Display symbol by itself */
	dspl_smbl_only = 1,		/* Display symbol by itself */
	dspl_smbl_name = 3,		/* Display symbol with waypoint name */
	dspl_smbl_cmnt = 5,		/* Display symbol with comment */
};

    long lOffset = 0;
    
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szIdent, 6);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);		// semicircle
    m_ptGpsDevice->PrepareLong(ptData, &lOffset, 0);				// longword unused
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 40);// no comment
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, 0);				// float dst

	Symbol_Type smbl = 18;									// sym_wpt_dot see pg. 33
    m_ptGpsDevice->PrepareShort(ptData, &lOffset, smbl);			// Symbol_Type smbl
 
	byte dspl = dspl_smbl_name;							
    *(ptData + lOffset++) = dspl;						// byte dspl
        
    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 0 5 W p t T y p e		page 37	*
 ************************************************************************/
short CGarminWpt::PrepareD105WptType(unsigned char* ptData)
{	// StreetPilot (user waypoints)
    long lOffset = 0;
    
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);		// semicircle

	Symbol_Type smbl = 18;									// sym_wpt_dot see pg. 33
    m_ptGpsDevice->PrepareShort(ptData, &lOffset, smbl);			// Symbol_Type smbl

	m_ptGpsDevice->PrepareCString(ptData, &lOffset, m_szIdent);
        
    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 0 6 W p t T y p e		page 37	*
 ************************************************************************/
short CGarminWpt::PrepareD106WptType(unsigned char* ptData)
{	// StreetPilot (route waypoints)
    long lOffset = 0;
    
	byte wpt_class = 0;		// 0: User, != 0 for other							
    *(ptData + lOffset++) = wpt_class;								// byte wpt_class

	this->PrepareSubClass (ptData, &lOffset);	// zeros for User wpts
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);	// semicircle

	Symbol_Type smbl = 18;									// sym_wpt_dot see pg. 33
    m_ptGpsDevice->PrepareShort(ptData, &lOffset, smbl);		// Symbol_Type smbl

	m_ptGpsDevice->PrepareCString(ptData, &lOffset, m_szIdent);	// char wpt_ident
	m_ptGpsDevice->PrepareCString(ptData, &lOffset, "");		// char lnk_ident
        
    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 0 7 W p t T y p e		page 38	*
 ************************************************************************/
short CGarminWpt::PrepareD107WptType(unsigned char* ptData)
{	// GPS 12CX
    long lOffset = 0;
    
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szIdent, 6);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);		// semicircle
    m_ptGpsDevice->PrepareLong(ptData, &lOffset, 0);				// longword unused
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 40);// no comment

	byte smbl = 18;										// sym_wpt_dot see pg. 33
    *(ptData + lOffset++) = smbl;						// byte sbml
    
	byte dspl = dspl_smbl_name;							
    *(ptData + lOffset++) = dspl;						// byte dspl
	
	m_ptGpsDevice->PrepareFloat(ptData, &lOffset, 0);	// float dst

 	byte color = clr_default;							
    *(ptData + lOffset++) = color;						// byte color
       
    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 0 8 W p t T y p e		page 39	*
 ************************************************************************/
short CGarminWpt::PrepareD108WptType(unsigned char* ptData)
{	// GPSMAP 162/168, eMap, GPSMAP 295

   enum D103_Smbl
    {
	dspl_name = 0,		/* Display symbol with waypoint name */
	dspl_none = 1,		/* Display symbol by itself */
	dspl_cmnt = 2,		/* Display symbol with comment */
    };


    long lOffset = 0;
    
    BOOL bGpsToStd = false;
    byte wpt_class = (byte)ConvertD108WptClass(bGpsToStd, m_nCategory);
    *(ptData + lOffset++) = wpt_class;			// byte wpt_class
    
    *(ptData + lOffset++) =  0xFF;			// byte color, Default_Color
    *(ptData + lOffset++) =  (byte)dspl_name;		// byte dspl, symbol with waypoint name 
    *(ptData + lOffset++) = 0x60;			// byte attr, see page 39
    
    Symbol_Type smbl =  CategoryToD108Symbol(m_nCategory);
    m_ptGpsDevice->PrepareShort(ptData, &lOffset, (short)smbl);// Symbol_Type smbl

    this->PrepareSubClass (ptData, &lOffset);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);	// semicircle
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, (float)m_dAlt_m);	// float alt

    float fUndef = 0x0000004A;	// 1 EXP 25
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, fUndef);	// float dpth
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, 0);		// float dist

    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 2);	// state

	if (wpt_class == USER_WPT)
		m_szCC="";						// replace "ED" with "" for user wpts

    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szCC, 2);	// country code


                                            // write variable strings of data struct
    this->PrepareLimZeroTermString(ptData, &lOffset, m_szIdent, 51);	// ident
    this->PrepareLimZeroTermString(ptData, &lOffset, "", 51);		// comment
    this->PrepareLimZeroTermString(ptData, &lOffset, m_szName, 31);	// facility
    this->PrepareLimZeroTermString(ptData, &lOffset, "", 25);		// city
    this->PrepareLimZeroTermString(ptData, &lOffset, "", 51);		// addr
    this->PrepareLimZeroTermString(ptData, &lOffset, "", 51);		// cross_road
        
    return (short)lOffset;
}


/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 0 9 W p t T y p e		page 55	*
 ************************************************************************/
short CGarminWpt::PrepareD109WptType(unsigned char* ptData)
{	// GPS 196
   enum D103_Smbl
    {
	dspl_name = 0,		/* Display symbol with waypoint name */
	dspl_none = 1,		/* Display symbol by itself */
	dspl_cmnt = 2,		/* Display symbol with comment */
    };


    long lOffset = 0;
    
    BOOL bGpsToStd = false;
    byte wpt_class = (byte)ConvertD109WptClass(bGpsToStd, m_nCategory);

    *(ptData + lOffset++) = 0x01;			// byte dtyp (0x01 for D109)
    *(ptData + lOffset++) = wpt_class;		// byte wpt_class
    *(ptData + lOffset++) = PrepareD109DsplColor(0x1F, dspl_name);	// byte dspl_color
    *(ptData + lOffset++) = 0x70;			// byte attr, must be 0x70 for D109
    
    Symbol_Type smbl =  CategoryToD109Symbol(m_nCategory);
    m_ptGpsDevice->PrepareShort(ptData, &lOffset, (short)smbl);// Symbol_Type smbl

    this->PrepareSubClass (ptData, &lOffset);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);	// semicircle
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, (float)m_dAlt_m);	// float alt

    float fUndef = 0x0000004A;	// 1 EXP 25
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, fUndef);	// float dpth
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, 0);		// float dist

    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 2);	// state

	if (wpt_class == USER_WPT)
		m_szCC="";						// replace "ED" with "" for user wpts

    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szCC, 2);	// country code

    m_ptGpsDevice->PrepareLong(ptData, &lOffset, 0);		// longword ete;

                                            // write variable strings of data struct
    this->PrepareLimZeroTermString(ptData, &lOffset, m_szIdent, 51);	// ident
    this->PrepareLimZeroTermString(ptData, &lOffset, "", 51);		// comment
    this->PrepareLimZeroTermString(ptData, &lOffset, m_szName, 31);	// facility
    this->PrepareLimZeroTermString(ptData, &lOffset, "", 25);		// city
    this->PrepareLimZeroTermString(ptData, &lOffset, "", 51);		// addr
    this->PrepareLimZeroTermString(ptData, &lOffset, "", 51);		// cross_road
        
    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 1 0 W p t T y p e		page 55	*
 ************************************************************************/
short CGarminWpt::PrepareD110WptType(unsigned char* ptData)
{	// Forerunner 205

	enum
	{
		clr_Black = 0,
		clr_Dark_Red = 1,
		clr_Dark_Green = 2,
		clr_Dark_Yellow = 3,
		clr_Dark_Blue = 4,
		clr_Dark_Magenta = 5,
		clr_Dark_Cyan = 6,
		clr_Light_Gray = 7,
		clr_Dark_Gray = 8,
		clr_Red = 9,
		clr_Green = 10,
		clr_Yellow = 11,
		clr_Blue = 12,
		clr_Magenta = 13,
		clr_Cyan = 14,
		clr_White = 15,
		clr_Transparent
	};

   enum D103_Smbl
    {
		dspl_name = 0,		/* Display symbol with waypoint name */
		dspl_none = 1,		/* Display symbol by itself */
		dspl_cmnt = 2,		/* Display symbol with comment */
    };


    long lOffset = 0;
    
    BOOL bGpsToStd = false;
    byte wpt_class = (byte)ConvertD110WptClass(bGpsToStd, m_nCategory);

    *(ptData + lOffset++) = 0x01;			// byte dtyp (0x01 for D109)
    *(ptData + lOffset++) = wpt_class;		// byte wpt_class
    *(ptData + lOffset++) = PrepareD109DsplColor(clr_Black, dspl_name);	// byte dspl_color
    *(ptData + lOffset++) = 0x80;			// byte attr, must be 0x80 for D110
    
    Symbol_Type smbl =  CategoryToD109Symbol(m_nCategory);
    m_ptGpsDevice->PrepareShort(ptData, &lOffset, (short)smbl);// Symbol_Type smbl

    this->PrepareSubClass (ptData, &lOffset);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);	// semicircle
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, (float)m_dAlt_m);	// float alt

    float fUndef = 0x0000004A;	// 1 EXP 25
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, fUndef);	// float dpth
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, 0);		// float dist

    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 2);	// state

	if (wpt_class == USER_WPT)
		m_szCC="";						// replace "ED" with "" for user wpts

    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szCC, 2);	// country code

    m_ptGpsDevice->PrepareLong(ptData, &lOffset, 0);		// longword ete;

	m_ptGpsDevice->PrepareFloat(ptData, &lOffset, fUndef);		// float temp
	m_ptGpsDevice->PrepareULong(ptData, &lOffset, 0xFFFFFFFF);	// Time_Type time
	m_ptGpsDevice->PrepareUShort(ptData, &lOffset, 0);			// uint16 wpt_cat

                                            // write variable strings of data struct
    this->PrepareLimZeroTermString(ptData, &lOffset, m_szIdent, 51);	// ident
    this->PrepareLimZeroTermString(ptData, &lOffset, "", 51);		// comment
    this->PrepareLimZeroTermString(ptData, &lOffset, m_szName, 31);	// facility
    this->PrepareLimZeroTermString(ptData, &lOffset, "", 25);		// city
    this->PrepareLimZeroTermString(ptData, &lOffset, "", 51);		// addr
    this->PrepareLimZeroTermString(ptData, &lOffset, "", 51);		// cross_road
        
    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 5 0 W p t T y p e		page 40	*
 ************************************************************************/
short CGarminWpt::PrepareD150WptType(unsigned char* ptData)
{	// GPS 150, GPS 155, GNC 250 and GNC 300
	enum D150_Class
	{
	apt_wpt_class = 0, /* airport waypoint class */
	int_wpt_class = 1, /* intersection waypoint class */
	ndb_wpt_class = 2, /* NDB waypoint class */
	vor_wpt_class = 3, /* VOR waypoint class */
	usr_wpt_class = 4, /* user defined waypoint class */
	rwy_wpt_class = 5, /* airport runway threshold waypoint class */
	aint_wpt_class = 6, /* airport intersection waypoint class */
	locked_wpt_class = 7 /* locked waypoint class */
	};
	
	BOOL bGpsToStd = false;
    byte wpt_class = (byte)ConvertD150WptClass(bGpsToStd, m_nCategory);

    long lOffset = 0;
    
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szIdent, 6);

	if (wpt_class == usr_wpt_class)
		m_szCC="";						// replace "ED" with "" for user wpts
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szCC, 2); // country code

    *(ptData + lOffset++) = wpt_class;								// byte wpt_class
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);		// semicircle
	m_ptGpsDevice->PrepareShort(ptData, &lOffset, (int)m_dAlt_m);	// short alt
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 24);// city
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 2); // state
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szName, 30); // facility
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 40);// no comment
        
    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 5 1 W p t T y p e		page 41	*
 ************************************************************************/
short CGarminWpt::PrepareD151WptType(unsigned char* ptData)
{
    enum D151_Class
    {
        apt_wpt_class = 0, 		/* airport waypoint class */
        vor_wpt_class = 1, 		/* VOR waypoint class */
        usr_wpt_class = 2, 		/* user defined waypoint class */
        locked_wpt_class = 3 		/* locked waypoint class */
    };

	BOOL bGpsToStd = false;
    byte wpt_class = (byte)ConvertD151WptClass(bGpsToStd, m_nCategory);

    long lOffset = 0;
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szIdent, 6);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);		// semicircle
    m_ptGpsDevice->PrepareLong(ptData, &lOffset, 0);				// longword unused
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 40);// no comment
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, 0);				// float dst
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szName, 30); // facility
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 24);// city
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 2); // state
	m_ptGpsDevice->PrepareShort(ptData, &lOffset, (int)m_dAlt_m);	// short alt

	if (wpt_class == usr_wpt_class)
		m_szCC="";						// replace "ED" with "" for user wpts

	this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szCC, 2); // country code

    *(ptData + lOffset++) = 0;						// char unused2
    *(ptData + lOffset++) = wpt_class;					// byte wpt_class
        
    return (short)lOffset;
}


/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 5 2 W p t T y p e		page 41	*
 ************************************************************************/
short CGarminWpt::PrepareD152WptType(unsigned char* ptData)
{
//The enumerated values for the "wpt_class" member of the D152_Wpt_Type are shown below:
	enum D152_Class
	{
	apt_wpt_class = 0, /* airport waypoint class */
	int_wpt_class = 1, /* intersection waypoint class */
	ndb_wpt_class = 2, /* NDB waypoint class */
	vor_wpt_class = 3, /* VOR waypoint class */
	usr_wpt_class = 4, /* user defined waypoint class */
	locked_wpt_class = 5 /* locked waypoint class */
	};

	
	BOOL bGpsToStd = false;
    byte wpt_class = (byte)ConvertD152WptClass(bGpsToStd, m_nCategory);

    long lOffset = 0;
    
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szIdent, 6);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);		// semicircle
    m_ptGpsDevice->PrepareLong(ptData, &lOffset, 0);				// longword unused
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 40);// no comment
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, 0);				// float dst
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szName, 30); // facility
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 24);// city
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 2); // state
	m_ptGpsDevice->PrepareShort(ptData, &lOffset, (int)m_dAlt_m);	// short alt

	if (wpt_class == usr_wpt_class)
		m_szCC="";						// replace "ED" with "" for user wpts

	this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szCC, 2); // country code

    *(ptData + lOffset++) = 0;										// char unused2
    *(ptData + lOffset++) = wpt_class;								// byte wpt_class
        
    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 5 4 W p t T y p e		page 42	*
 ************************************************************************/
short CGarminWpt::PrepareD154WptType(unsigned char* ptData)
{
	//The enumerated values for the "wpt_class" member of the D154_Wpt_Type are shown below:
	enum D154_Class
	{
	apt_wpt_class = 0, /* airport waypoint class */
	int_wpt_class = 1, /* intersection waypoint class */
	ndb_wpt_class = 2, /* NDB waypoint class */
	vor_wpt_class = 3, /* VOR waypoint class */
	usr_wpt_class = 4, /* user defined waypoint class */
	rwy_wpt_class = 5, /* airport runway threshold waypoint class */
	aint_wpt_class = 6, /* airport intersection waypoint class */
	andb_wpt_class = 7, /* airport NDB waypoint class */
	sym_wpt_class = 8, /* user defined symbol-only waypoint class */
	locked_wpt_class = 9 /* locked waypoint class */
	};

	BOOL bGpsToStd = false;
    byte wpt_class = (byte)ConvertD154WptClass(bGpsToStd, m_nCategory);

    long lOffset = 0;
    
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szIdent, 6);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);		// semicircle
    m_ptGpsDevice->PrepareLong(ptData, &lOffset, 0);				// longword unused
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 40);// no comment
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, 0);				// float dst
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szName, 30); // facility
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 24);// city
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 2); // state
	m_ptGpsDevice->PrepareShort(ptData, &lOffset, (int)m_dAlt_m);	// short alt

	if (wpt_class == usr_wpt_class)
		m_szCC="";						// replace "ED" with "" for user wpts

	this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szCC, 2); // country code

    *(ptData + lOffset++) = 0;										// char unused2
    *(ptData + lOffset++) = wpt_class;								// byte wpt_class
        
    Symbol_Type smbl =  CategoryToD154Symbol(m_nCategory);
    m_ptGpsDevice->PrepareShort(ptData, &lOffset, smbl);	// Symbol_Type smbl

    return (short)lOffset;
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 5 5 W p t T y p e		page 43	*
 ************************************************************************/
short CGarminWpt::PrepareD155WptType(unsigned char* ptData)
{
	//The enumerated values for the "dspl" member of the D155_Wpt_Type are shown below:
	enum D155_Smbl
	{
	dspl_smbl_only = 1, /* Display symbol by itself */
	dspl_smbl_name = 3, /* Display symbol with waypoint name */
	dspl_smbl_cmnt = 5, /* Display symbol with comment */
	};

	//The enumerated values for the "wpt_class" member of the D155_Wpt_Type are shown below:
	enum D155_Class
	{
	apt_wpt_class = 0, /* airport waypoint class */
	int_wpt_class = 1, /* intersection waypoint class */
	ndb_wpt_class = 2, /* NDB waypoint class */
	vor_wpt_class = 3, /* VOR waypoint class */
	usr_wpt_class = 4, /* user defined waypoint class */
	locked_wpt_class = 5 /* locked waypoint class */
	};
	
	BOOL bGpsToStd = false;
    byte wpt_class = (byte)ConvertD155WptClass(bGpsToStd, m_nCategory);

    long lOffset = 0;
    
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szIdent, 6);
    this->PrepareSemicircle(ptData, &lOffset, m_dLat, m_dLon);		// semicircle
    m_ptGpsDevice->PrepareLong(ptData, &lOffset, 0);				// longword unused
    this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 40);// no comment
    m_ptGpsDevice->PrepareFloat(ptData, &lOffset, 0);				// float dst
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szName, 30); // facility
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 24);// city
	this->PrepareFilteredSpaceTermString (ptData, &lOffset, "", 2); // state
	m_ptGpsDevice->PrepareShort(ptData, &lOffset, (int)m_dAlt_m);	// short alt

	if (wpt_class == usr_wpt_class)
		m_szCC="";						// replace "ED" with "" for user wpts

	this->PrepareFilteredSpaceTermString (ptData, &lOffset, m_szCC, 2); // country code

    *(ptData + lOffset++) = 0;										// char unused2
    *(ptData + lOffset++) = wpt_class;								// byte wpt_class
        
    Symbol_Type smbl =  CategoryToD155Symbol(m_nCategory);
    m_ptGpsDevice->PrepareShort(ptData, &lOffset, smbl);	// Symbol_Type smbl

    *(ptData + lOffset++) = dspl_smbl_name;					// byte dspl

    return (short)lOffset;
}


/************************************************************************
 *  GarminWpt.cpp		P r e p a r e R t e W p t						*
 ************************************************************************/
short CGarminWpt::PrepareRteWpt(unsigned char* ptData, short nWptProt)
{			
	short nLen = 0;

	if (nWptProt == D100_Wpt)
		nLen = PrepareD100WptType(ptData);
	if (nWptProt == D101_Wpt)
		nLen = PrepareD101WptType(ptData);
	if (nWptProt == D102_Wpt)
		nLen = PrepareD102WptType(ptData);
	if (nWptProt == D103_Wpt)
		nLen = PrepareD103WptType(ptData);
	if (nWptProt == D104_Wpt)
		nLen = PrepareD104WptType(ptData);
	if (nWptProt == D105_Wpt)
		nLen = PrepareD105WptType(ptData);
	if (nWptProt == D106_Wpt)
		nLen = PrepareD106WptType(ptData);
	if (nWptProt == D107_Wpt)
		nLen = PrepareD107WptType(ptData);
	if (nWptProt == D108_Wpt)
		nLen = PrepareD108WptType(ptData);
	if (nWptProt == D109_Wpt)
		nLen = PrepareD109WptType(ptData);
	if (nWptProt == D110_Wpt)
		nLen = PrepareD110WptType(ptData);

	if (nWptProt == D150_Wpt)
		nLen = PrepareD150WptType(ptData);
	if (nWptProt == D151_Wpt)
		nLen = PrepareD151WptType(ptData);
	if (nWptProt == D152_Wpt)
		nLen = PrepareD152WptType(ptData);

	if (nWptProt == D154_Wpt)
		nLen = PrepareD154WptType(ptData);
	if (nWptProt == D155_Wpt)
		nLen = PrepareD155WptType(ptData);

	return nLen;
}

/****************************************************************************************
 *  GarminWpt.cpp		P r e p a r e L i m Z e r o T e r m S t r i n g					*
 ****************************************************************************************/
void CGarminWpt::PrepareLimZeroTermString(unsigned char* lpOut, long* ptOffs, 
                                        CString szText, short nMaxLen)
{
    unsigned char ch;
    short i=0;
    BOOL bToUpper=true;
    
	short nTextLen = szText.GetLength();

    while (i < nTextLen && i < nMaxLen)	
    {
		ch = szText.GetAt(i++);
        *(lpOut + (*ptOffs)++) = FilterTextByte(ch, bToUpper);
    }

    *(lpOut + (*ptOffs)++) = 0;
}

/****************************************************************************************
 *  GarminWpt.cpp		S p l i t L i m Z e r o T e r m S t r i n g						*
 ****************************************************************************************/
short CGarminWpt::SplitLimZeroTermString(unsigned char* ptData, long* ptOffset, char* ptText, short nMaxLen)
{
	short j=0;
	unsigned char ch;
    do	{
    	ch = (unsigned char)*(ptData + (*ptOffset)++); 
   		*(ptText + j++) = (char)ch;  
    	} while (ch > 0 && j < nMaxLen);

	if (j==nMaxLen)
		*(ptText + nMaxLen-1) = 0;

	return j;
}

/****************************************************************************************
 *  GarminWpt.cpp		P r e p a r e S u b C l a s s				*
 ****************************************************************************************/
void CGarminWpt::PrepareSubClass (unsigned char* ptData, long* ptOffs)
{
    m_ptGpsDevice->PrepareShort(ptData, ptOffs, 0);		// nDummy
    m_ptGpsDevice->PrepareLong(ptData, ptOffs, 0);		// lDummy
    m_ptGpsDevice->PrepareULong(ptData, ptOffs, 0xFFFFFFFF);	// ulDummy1
    m_ptGpsDevice->PrepareULong(ptData, ptOffs, 0xFFFFFFFF);	// ulDummy2
    m_ptGpsDevice->PrepareULong(ptData, ptOffs, 0xFFFFFFFF);	// ulDummy3
}



/****************************************************************************************
 *  GarminWpt.cpp		    	F i l t e r T e x t B y t e								*
 *  Input: szName, ...																	*
 ****************************************************************************************/
unsigned char CGarminWpt::FilterTextByte(unsigned char inChar, BOOL bToUpper)
{
    unsigned char ch = inChar;
    
    if (bToUpper)
    {
        if (ch == 0xa7) ch = 'S';	// convert "" to "S"
        if (ch == (int)'+') ch = 0x20;  // convert "+" to space char
            
                                        // use upper case letters only
        if (!isupper (ch))
                ch = toupper (ch);

        if (ch==(unsigned char)'') ch = 'A';		// convert '' to A
        if (ch==(unsigned char)'') ch = 'O';		// convert '' to O
        if (ch==(unsigned char)'') ch = 'U';		// convert '' to U

        if (ch==(unsigned char)'') ch = 'A';		// convert '' to A
        if (ch==(unsigned char)'') ch = 'O';		// convert '' to O        
        if (ch==(unsigned char)'') ch = 'U';		// convert '' to U

        if (ch==(unsigned char)'') ch = 'E';		// convert '' to E        
        if (ch==(unsigned char)'') ch = 'E';		// convert '' to E
	}
    else
    {
        if (ch == 0xa7) ch = 's';	// convert "" to "s"
        if (ch == (int)'+') ch = 0x20;  // convert "+" to space char
                
        if (ch==(unsigned char)'') ch = 'A';		// convert '' to A
        if (ch==(unsigned char)'') ch = 'O';		// convert '' to O
        if (ch==(unsigned char)'') ch = 'U';		// convert '' to U
        
        if (ch==(unsigned char)'') ch = 'a';		// convert '' to a
        if (ch==(unsigned char)'') ch = 'o';		// convert '' to o        
        if (ch==(unsigned char)'') ch = 'u';		// convert '' to u

        if (ch==(unsigned char)'') ch = 'e';		// convert '' to e        
        if (ch==(unsigned char)'') ch = 'e';		// convert '' to e
	}
    
    return ch;
}

/****************************************************************************************
 *  GarminWpt.cpp		    	F i l t e r T e x t B y t e s							*
 *  Input: szName, ...																	*
 ****************************************************************************************/
void CGarminWpt::FilterTextBytes (CString& szName, BOOL bToUpper)
{
    short i;
    int	ch = 0;
    short nBytes = szName.GetLength();
                                                
    for (i=0; i<nBytes; i++)			// check "nBytes" bytes of szName
    {
        ch = szName.GetAt(i);
        
        ch = FilterTextByte((unsigned char)ch, bToUpper);
        
	szName.SetAt(i, ch);
    }
}

/****************************************************************************************
 *  GarminWpt.cpp		  C o p y S p a c e T e r m S t r i n g			*
 *  Input: szName = "EDFE", Output: ptData="EDFE  ", if nBytes = 6 (no terminating 0)	*
 ****************************************************************************************/
void CGarminWpt::PrepareSpaceTermString (unsigned char* ptData, long* ptOffs, 
                                        CString szSource, short nBytes)
{
    short i;
    BOOL	bEnd = false;
    int	ch = 0;
    short nStrLen = szSource.GetLength();
                                                // use upper case letters only
    for (i=0; i<nBytes; i++)			// fill "nBytes" bytes of ptData
    {
	if (!bEnd)
        {
            if (i < nStrLen)
            {					// chars available
                ch = szSource.GetAt(i);
                                            // stop copy on first space char
                bEnd = (ch == 0x20);
            }
            else
            {
                bEnd = true;
            }
        }
		
	if (bEnd)
        {
            ch = 0x20;
        }
        
	*(ptData + *ptOffs + i) = ch;
    }
    
    *ptOffs += nBytes;
}

/****************************************************************************************
 *  GarminWpt.cpp		P r e p a r e F i l t e r e d S p a c e T e r m S t r i n g		*
 *  Input: szName = "EDFE", Output: ptData="EDFE  ", if nBytes = 6 (no terminating 0)	*
 ****************************************************************************************/
void CGarminWpt::PrepareFilteredSpaceTermString (unsigned char* ptData, long* ptOffs,
                                CString szSource, short nBytes)
{
    BOOL bToUpper = true;
    FilterTextBytes (szSource, bToUpper);
    PrepareSpaceTermString (ptData, ptOffs, szSource, nBytes);
}

/****************************************************************************************
 *  GarminWpt.cpp	S p l i t S p a c e T e r m i n a t e d T e x t			*
 ****************************************************************************************/
CString CGarminWpt::SplitSpaceTerminatedText (unsigned char* lpData, long* ptOffset, short nMaxData)
{       
    CString 	szText;
    short	i, nDataLen;
    BOOL	bEnd = FALSE;
        
    szText = "";
    
    nDataLen = nMaxData;      				// get data length
    for (i=nMaxData-1; i>=0 && !bEnd; i--)
    {
	if (*(lpData + *ptOffset +i) == ' ')	nDataLen--;
                                    else	bEnd = TRUE;
    }
			
    for (i=0; i<nDataLen; i++)
	szText += *(lpData + *ptOffset +i);
        
    *ptOffset += nMaxData;
    
    return szText;
}


/********************************************************************************
 *  GarminWpt.cpp			S p l i t S e m i c i r c l e						*
 ********************************************************************************/
void CGarminWpt::SplitSemicircle(unsigned char* ptData, long* ptOffset, double* ptLat, double* ptLon)
{    
    Semicircle_Type semicircle;
    
    semicircle.lat = m_ptGpsDevice->SplitLong (ptData, ptOffset);
    semicircle.lon = m_ptGpsDevice->SplitLong (ptData, ptOffset); 

            
    double dFakt;
    dFakt = exp(log((double)2) * 31);		// 2 exp 31
    
    *ptLat = semicircle.lat * (180 / dFakt);
    *ptLon = semicircle.lon * (180 / dFakt);
}



/********************************************************************************
 *  GarminWpt.cpp			P r e p a r e S e m i c i r c l e					*
 ********************************************************************************/
void CGarminWpt::PrepareSemicircle(unsigned char* ptData, long* ptOffset, double dLat, double dLon)
{
    Semicircle_Type semicircle;
    
    double dFakt;
    dFakt = exp(log((double)2) * 31);		// 2 exp 31

    semicircle.lat = (long)(dLat * dFakt/180);
    semicircle.lon = (long)(dLon * dFakt/180);

    //semicircle.lat = (long)(dLat * KOORD_FAKT);
    //semicircle.lon = (long)(dLon * KOORD_FAKT);
    
    m_ptGpsDevice->PrepareLong(ptData, ptOffset, semicircle.lat);
    m_ptGpsDevice->PrepareLong(ptData, ptOffset, semicircle.lon);
}

/************************************************************************
 *  GarminWpt.cpp		S y s t e m T o G a r m i n T i m e 			*
 ************************************************************************/
unsigned long CGarminWpt::SystemToGarminTime (unsigned long lSystemTime)
{
	unsigned long lTimeDiff = 7304 * (24*3600);
	unsigned long lGarminTime = lSystemTime - lTimeDiff;
	return lGarminTime;
}

/************************************************************************
 *  GarminWpt.cpp					G e t C C							*
 ************************************************************************/
void CGarminWpt::GetCC (char* szCC)		// 2 byte array only!! 
{
	szCC[0] = m_szCC.GetAt(0);
	szCC[1] = m_szCC.GetAt(1);			// no terminating 0!!
}

/************************************************************************
 *  GarminWpt.cpp					S e t C C							*
 ************************************************************************/
void CGarminWpt::SetCC (char* szCC)		// 2 byte array only!! 
{
	m_szCC = "";
	m_szCC += szCC[0];
	m_szCC += szCC[1];					// no terminating 0!!
}

/************************************************************************
 *  GarminWpt.cpp		P r e p a r e D 1 0 9 D s p l C o l o r	page 55	*
 ************************************************************************/
byte CGarminWpt::PrepareD109DsplColor(short nColor, short nDspl)
{
	byte dspl_color = 0;
	byte color = (byte)nColor;	//			CCCCC000
	byte displ = (byte)nDspl;	//			DD000000

	displ = displ >> 5;			// Bits:	01234567
	dspl_color = color | displ;	// Values:	CCCCCDD0

	return dspl_color;
}

/************************************************************************
 *  GarminWpt.cpp		S p l i t D 1 0 9 D s p l C o l o r		page 55	*
 ************************************************************************/
void CGarminWpt::SplitD109DsplColor(byte dspl_color, short* ptColor, short* ptDspl)
{
	byte color_mask = (byte)0xF8;		// 11111000	= 0xF8
	byte displ_mask = (byte)0x06;		// 00000110 = 0x06

	*ptColor = dspl_color & color_mask;
	*ptDspl  = (dspl_color & displ_mask) << 5;
}
