#ifndef __GARMIN_WPT__      
#define __GARMIN_WPT__

// GarminWpt.h
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de


/**************** Basic Types *************************************/
typedef unsigned char	byte;
typedef unsigned short	uint16;
typedef unsigned long	longword;
typedef unsigned char	boolean;

#define KOORD_FAKT ((float)11930464.7)	// 2exp31 / 180 

typedef struct
	{
	long	lat;		// latitude in semicircles 
	long	lon;		// longitude in semicircles
	}  Semicircle_Type;

// degrees = semicircles * (180 / 2 exp31)
// semicircles = degrees * (2 exp31 / 180)


typedef struct
	{
	double	lat;		// latitude in radians
	double	lon;		// longitude in radians
	}  Radian_Type;

// degrees = radians * (180/pi)
// radians = degrees = (pi/180)


typedef short	Symbol_Type;
typedef longword	Time_Type;

typedef struct
{
	short nDummy;
	long  lDummy;
	unsigned long ulDummy1;
	unsigned long ulDummy2;
	unsigned long ulDummy3;
} SubClassType;




/******************************** D100_Wpt_Type	****************************/

//Example products: GPS 38, GPS 40, GPS 45, GPS 75 and GPS II.
typedef struct
{
	char ident[6];				/* identifier */
	Semicircle_Type posn;		/* position */
	longword unused;			/* should be set to zero */
	char cmnt[40];				/* comment */
} D100_Wpt_Type;

/******************************** D101_Wpt_Type	****************************/

//Example products: GPSMAP 210 and GPSMAP 220 (both prior to version 4.00).
typedef struct
{
	char ident[6];				/* identifier */
	Semicircle_Type posn;		/* position */
	longword unused;			/* should be set to zero */
	char cmnt[40];				/* comment */
	float dst;					/* proximity distance (meters) */
	byte smbl;					/* symbol id */
} D101_Wpt_Type;

/******************************** D102_Wpt_Type	****************************/

//Example products: GPSMAP 175, GPSMAP 210 and GPSMAP 220.
typedef struct
{
	char ident[6];				/* identifier */
	Semicircle_Type posn;		/* position */
	longword unused;			/* should be set to zero */
	char cmnt[40];				/* comment */
	float dst;					/* proximity distance (meters) */
	Symbol_Type smbl;			/* symbol id */
} D102_Wpt_Type;

/******************************** D103_Wpt_Type	****************************/
//Example products: GPS 12, GPS 12 XL, GPS 48 and GPS II Plus.

// The enumerated values for the "smbl" member of the D103_Wpt_Type are shown below:
enum
{
	smbl_dot = 0,				/* dot symbol */
	smbl_house = 1,				/* house symbol */
	smbl_gas = 2,				/* gas symbol */
	smbl_car = 3,				/* car symbol */
	smbl_fish = 4,				/* fish symbol */
	smbl_boat = 5,				/* boat symbol */
	smbl_anchor = 6,			/* anchor symbol */
	smbl_wreck = 7,				/* wreck symbol */
	smbl_exit = 8,				/* exit symbol */
	smbl_skull = 9,				/* skull symbol */
	smbl_flag = 10,				/* flag symbol */
	smbl_camp = 11,				/* camp symbol */
	smbl_circle_x = 12,			/* circle with x symbol */
	smbl_deer = 13,				/* deer symbol */
	smbl_1st_aid = 14,			/* first aid symbol */
	smbl_back_track = 15		/* back track symbol */
};


typedef struct
{
	char ident[6];				/* identifier */
	Semicircle_Type posn;		/* position */
	longword unused;			/* should be set to zero */
	char cmnt[40];				/* comment */
	byte smbl;					/* symbol id */
	byte dspl;					/* display option */
} D103_Wpt_Type;


/******************************** D104_Wpt_Type	****************************/
//Example products: GPS III.

// The enumerated values for the "dspl" member of the D104_Wpt_Type are shown below:

enum
{
	dspl_smbl_none = 0,		/* Display symbol by itself */
	dspl_smbl_only = 1,		/* Display symbol by itself */
	dspl_smbl_name = 3,		/* Display symbol with waypoint name */
	dspl_smbl_cmnt = 5,		/* Display symbol with comment */
};


typedef struct
{
	char ident[6];			/* identifier */
	Semicircle_Type posn;	/* position */
	longword unused;		/* should be set to zero */
	char cmnt[40];			/* comment */
	float dst;				/* proximity distance (meters) */
	Symbol_Type smbl;		/* symbol id */
	byte dspl;				/* display option */
} D104_Wpt_Type;



/******************************** D105_Wpt_Type	****************************/
// Example products: StreetPilot (user waypoints).

typedef struct
{
	Semicircle_Type posn;	/* position */
	Symbol_Type smbl;		/* symbol id */
    CString wpt_ident;	/* null-terminated string 	*/
} D105_Wpt_Type;


/******************************** D106_Wpt_Type	****************************/

//Example products: StreetPilot (route waypoints).

typedef struct
{
	byte wpt_class;			/* class */
	byte subclass[13];		/* subclass */
	Semicircle_Type posn;	/* position */
	Symbol_Type smbl;		/* symbol id */
	CString	wpt_ident;		/* null-terminated string 	*/
	CString lnk_ident;		/* null-terminated string 	*/
} D106_Wpt_Type;

//The enumerated values for the "wpt_class" member of the D106_Wpt_Type are as follows:
//Zero: indicates a user waypoint ("subclass" is ignored).
//Non-zero: indicates a non-user waypoint ("subclass" must be valid).

/******************************** D107_Wpt_Type	****************************/
//Example products: GPS 12CX.


//The enumerated values for the "smbl" member of the D107_Wpt_Type are the same as the the "smbl" member of
//the D103_Wpt_Type.
//The enumerated values for the "dspl" member of the D107_Wpt_Type are the same as the the "dspl" member of the
//D103_Wpt_Type.

//The enumerated values for the "color" member of the D107_Wpt_Type are shown below:
enum
{
	clr_default = 0,		/* Default waypoint color */
	clr_red = 1,			/* Red */
	clr_green = 2,			/* Green */
	clr_blue = 3			/* Blue */
};

typedef struct
{
	char ident[6];			/* identifier */
	Semicircle_Type posn;	/* position */
	longword unused;		/* should be set to zero */
	char cmnt[40];			/* comment */
	byte smbl;				/* symbol id */
	byte dspl;				/* display option */
	float dst;				/* proximity distance (meters) */
	byte color;				/* waypoint color */
} D107_Wpt_Type;



/******************************** D108_Wpt_Type	****************************/

// The enumerated values for the "wpt_class" member of the D108_Wpt_Type are defined as follows:

typedef enum
	{
	USER_WPT = 0x00,		/* User waypoint */
	AVTN_APT_WPT = 0x40,	/* Aviation Airport waypoint */
	AVTN_INT_WPT = 0x41,	/* Aviation Intersection waypoint */
	AVTN_NDB_WPT = 0x42,	/* Aviation NDB waypoint */
	AVTN_VOR_WPT = 0x43,	/* Aviation VOR waypoint */
	AVTN_ARWY_WPT = 0x44,	/* Aviation Airport Runway waypoint */
	AVTN_AINT_WPT = 0x45,	/* Aviation Airport Intersection */
	AVTN_ANDB_WPT = 0x46,	/* Aviation Airport NDB waypoint */
	MAP_PNT_WPT = 0x80,		/* Map Point waypoint */
	MAP_AREA_WPT = 0x81,	/* Map Area waypoint */
	MAP_INT_WPT = 0x82,		/* Map Intersection waypoint */
	MAP_ADRS_WPT = 0x83,	/* Map Address waypoint */
	MAP_LABEL_WPT = 0x84,	/* Map Label Waypoint */
	MAP_LINE_WPT = 0x85,	/* Map Line Waypoint */
	} WPT_CLASS_TYPE;


// The "color" member can be one of the following values:
enum {	Black,		Dark_Red,		Dark_Green, Dark_Yellow,
		Dark_Blue,	Dark_Magenta,	Dark_Cyan,	Light_Gray,
		Dark_Gray,	Red,			Green,		Yellow,
		Blue,		Magenta,		Cyan,		White,
		Default_Color = 0xFF };




typedef struct
	{
	byte wpt_class;			/* class (see below) 1 */
	byte color;				/* color (see below) 1 */
	byte dspl;				/* display options (see below) 1 */
	byte attr;				/* attributes (see below) 1 */
	Symbol_Type smbl;		/* waypoint symbol 2 */
//	byte subclass[18];		/* subclass 18 */
	SubClassType	subclass;
	Semicircle_Type posn;	/* 32 bit semicircle 8 */
	float alt;				/* altitude in meters 4 */
	float dpth;				/* depth in meters 4 */
	float dist;				/* proximity distance in meters 4 */
	char state[2];			/* state 2 */
	char cc[2];				/* country code 2 */
	char ident[52];			/* variable length string 1-51 */
	char comment[52];		/* waypoint user comment 1-51 */
	char facility[32];		/* facility name 1-31 */
	char city[26];			/* city name 1-25 */
	char addr[52];			/* address number 1-51 */
	char cross_road[52];	/* intersecting road label 1-51 */
	} D108_Wpt_Type;



/******************************** D109_Wpt_Type	****************************/
//Example products: GPS196.


    typedef struct                      /*                                 size */
        {
        byte            dtyp;           /* data packet type (0x01 for D109)1    */
        byte            wpt_class;      /* class                           1    */
        byte            dspl_color;     /* display & color (see below)     1    */
        byte            attr;           /* attributes (0x70 for D109)      1    */
        Symbol_Type     smbl;           /* waypoint symbol                 2    */
//	byte subclass[18];		/* subclass 18 */
		SubClassType	subclass;
        Semicircle_Type posn;           /* 32 bit semicircle               8    */
        float           alt;            /* altitude in meters              4    */
        float           dpth;           /* depth in meters                 4    */
        float           dist;           /* proximity distance in meters    4    */
        char            state[2];       /* state                           2    */
        char            cc[2];          /* country code                    2    */
        longword        ete;            /* outbound link ete in seconds    4    */
	char ident[52];			/* variable length string 1-51 */
	char comment[52];		/* waypoint user comment 1-51 */
	char facility[32];		/* facility name 1-31 */
	char city[26];			/* city name 1-25 */
	char addr[52];			/* address number 1-51 */
	char cross_road[52];	/* intersecting road label 1-51 */
	} D109_Wpt_Type;

//All fields are defined the same as D108 except as noted below.

//dtyp - Data packet type, must be 0x01 for D109.

//dsp_color - The 'dspl_color' member contains three fields; bits 0-4 specify
//the color, bits 5-6 specify the waypoint display attribute and bit 7 is unused
//and must be 0. Color values are as specified for D108 except that the default
//value is 0x1f. Display attribute values are as specified for D108.

//attr - Attribute. Must be 0x70 for D109.

//ete - Estimated time en route in seconds to next waypoint. Default value is
//0xffffffff.



/******************************** D110_Wpt_Type	****************************/
//Example products: Forerunner 205.

typedef struct
{
	byte dtyp;				/* data packet type (0x01 for D110) */
	byte wpt_class;		/* class */
	byte dspl_color;		/* display & color (see below) */
	byte attr;				/* attributes (0x80 for D110) */
	Symbol_Type smbl;		/* waypoint symbol */
//	uint8 subclass[18];		/* subclass */
	SubClassType	subclass;
    Semicircle_Type posn;   /* 32 bit semicircle               8    */
	float alt;				/* altitude in meters */
	float dpth;				/* depth in meters */
	float dist;				/* proximity distance in meters */
	char state[2];			/* state */
	char cc[2];				/* country code */
    longword  ete;          /* outbound link ete in seconds    4    */
	float temp;				/* temperature */
	Time_Type time;			/* timestamp */
	uint16 wpt_cat;			/* category membership */

	char ident[52];			/* variable length string 1-51 */
	char comment[52];		/* waypoint user comment 1-51 */
	char facility[32];		/* facility name 1-31 */
	char city[26];			/* city name 1-25 */
	char addr[52];			/* address number 1-51 */
	char cross_road[52];	/* intersecting road label 1-51 */
} D110_Wpt_Type;




/******************************** D150_Wpt_Type	****************************/
//Example products: GPS 150, GPS 155, GNC 250 and GNC 300.


	typedef struct
	{
	char ident[6];			/* identifier */
	char cc[2];				/* country code */
	byte wpt_class;			/* class */
	Semicircle_Type posn;	/* position */
	int alt;				/* altitude (meters) */
	char city[24];			/* city */
	char state[2];			/* state */
	char name[30];			/* facility name */
	char cmnt[40];			/* comment */
	} D150_Wpt_Type;



/******************************** D151_Wpt_Type	****************************/
// Example products: GPS 55 AVD, GPS 89.


	typedef struct
	{
	char ident[6];			/* identifier */
	Semicircle_Type posn;	/* position */
	longword unused;		/* should be set to zero */
	char cmnt[40];			/* comment */
	float dst;				/* proximity distance (meters) */
	char name[30];			/* facility name */
	char city[24];			/* city */
	char state[2];			/* state */
	int alt;				/* altitude (meters) */
	char cc[2];				/* country code */
	char unused2;			/* should be set to zero */
	byte wpt_class;			/* class */
	} D151_Wpt_Type;


/******************************** D152_Wpt_Type	****************************/
// Example products: GPS 90, GPS 95 AVD, GPS 95 XL and GPSCOM 190.


	typedef struct
	{
	char ident[6];			/* identifier */
	Semicircle_Type posn;	/* position */
	longword unused;		/* should be set to zero */
	char cmnt[40];			/* comment */
	float dst;				/* proximity distance (meters) */
	char name[30];			/* facility name */
	char city[24];			/* city */
	char state[2];			/* state */
	int alt;				/* altitude (meters) */
	char cc[2];				/* country code */
	char unused2;			/* should be set to zero */
	byte wpt_class;			/* class */
	} D152_Wpt_Type;


/******************************** D154_Wpt_Type	****************************/
//Example products: GPSMAP 195.


	typedef struct
	{
	char ident[6];			/* identifier */
	Semicircle_Type posn;	/* position */
	longword unused;		/* should be set to zero */
	char cmnt[40];			/* comment */
	float dst;				/* proximity distance (meters) */
	char name[30];			/* facility name */
	char city[24];			/* city */
	char state[2];			/* state */
	int alt;				/* altitude (meters) */
	char cc[2];				/* country code */
	char unused2;			/* should be set to zero */
	byte wpt_class;			/* class */
	Symbol_Type smbl;		/* symbol id */
	} D154_Wpt_Type;



/******************************** D155_Wpt_Type	****************************/

// Example products: GPS III Pilot.



	typedef struct
	{
	char ident[6];			/* identifier */
	Semicircle_Type posn;	/* position */
	longword unused;		/* should be set to zero */
	char cmnt[40];			/* comment */
	float dst;				/* proximity distance (meters) */
	char name[30];			/* facility name */
	char city[24];			/* city */
	char state[2];			/* state */
	int alt;				/* altitude (meters) */
	char cc[2];				/* country code */
	char unused2;			/* should be set to zero */
	byte wpt_class;			/* class */
	Symbol_Type smbl;		/* symbol id */
	byte dspl;				/* display option */
	} D155_Wpt_Type;


/***********************************************************************************/


/******************************** D300_Trk_Point_Type	****************************/


typedef struct
{
	Semicircle_Type		posn;				/* position					*/
	longword			time;				/* time						*/
	boolean				new_trk;			/* new track segment?		*/
} D300_Trk_Point_Type;


typedef struct
{
	Semicircle_Type		posn;				/* position					*/
	longword			time;				/* time						*/
	float				alt;				/* altitude in meters		*/
	float				dpth;				/* depth in meters			*/
	boolean				new_trk;			/* new track segment?		*/
} D301_Trk_Point_Type;

typedef struct
{
	Semicircle_Type		posn; /* position */
	longword			time; /* time */
	float				alt; /* altitude in meters */
	float				dpth; /* depth in meters */
	float				temp; /* temp in degrees C */
	boolean				new_trk; /* new track segment? */
} D302_Trk_Point_Type;

typedef struct
{
	Semicircle_Type		posn; /* position */
	longword			time; /* time */
	float				alt; /* altitude in meters */
	byte				heart_rate; /* heart rate in beats per minute */
} D303_Trk_Point_Type;

typedef struct
{
	Semicircle_Type		posn; /* position */
	longword			time; /* time */
	float				alt; /* altitude in meters */
	float				distance; /* distance traveled in meters. See below. */
	byte				heart_rate; /* heart rate in beats per minute */
	byte				cadence; /* in revolutions per minute */
	boolean				sensor; /* is a wheel sensor present? */
} D304_Trk_Point_Type;


/******************************** RteLinkData	****************************/

typedef enum
	{
	line = 0,
	link = 1,
	net = 2,
	direct = 3,
	snap = 0xFF
	} LinkClassType;

typedef struct
	{
	short			link_class;	
	SubClassType	subclass;
	char			ident[52];
	} RteLinkType;


#define SIZEOF_CC 3

class CGpsDevice;
class CGarminWpt : public CObject
{
public:
	enum {  AIRPORT = 1,
			REPORT =2,
			INTERSECTION =3,
			NDB	 =4,
			VOR	 =5,
			USER =6,
			SAIL	=7,
			HELI =8 };

private:
	double	m_dLat;
	double  m_dLon;
	CString m_szName;
	CString m_szIdent;
	short	m_nCategory;
	double	m_dAlt_m;
    CString	m_szCC;
	BOOL	m_bStoreAsUserWpt;

	CGpsDevice*	m_ptGpsDevice;


// Implementation
public:
	CGarminWpt(CGpsDevice* ptGpsDevice);		
	~CGarminWpt();     
	

	void SetRteWpt (double dLat, double dLon, CString& szName, 
			CString& szIndicator, short nCategory, double dAlt_m, short nCC, BOOL bStoreAsUserWpt);
	void GetRteWpt (double* ptLat, double* ptLon, CString* ptName, 
			CString* ptIndicator, short* ptCategory, double* ptAlt_m, short* ptCC);

	BOOL SplitRteWpt(unsigned char* ptData, long lLen, short nWptProt);
	short PrepareRteWpt(unsigned char* ptData, short nWptProt);

private:
	BOOL SplitD100WptType(unsigned char* ptData, long lLen);
    BOOL SplitD101WptType(unsigned char* ptData, long lLen);
    BOOL SplitD102WptType(unsigned char* ptData, long lLen);
    BOOL SplitD103WptType(unsigned char* ptData, long lLen);
    BOOL SplitD104WptType(unsigned char* ptData, long lLen);
    BOOL SplitD105WptType(unsigned char* ptData, long lLen);
    BOOL SplitD106WptType(unsigned char* ptData, long lLen);
    BOOL SplitD107WptType(unsigned char* ptData, long lLen);
    BOOL SplitD108WptType(unsigned char* ptData, long lLen);
    BOOL SplitD109WptType(unsigned char* ptData, long lLen);
    BOOL SplitD110WptType(unsigned char* ptData, long lLen);
    BOOL SplitD150WptType(unsigned char* ptData, long lLen);
    BOOL SplitD151WptType(unsigned char* ptData, long lLen);
    BOOL SplitD152WptType(unsigned char* ptData, long lLen);
    BOOL SplitD154WptType(unsigned char* ptData, long lLen);
    BOOL SplitD155WptType(unsigned char* ptData, long lLen);


private:
    short ConvertD108WptClass(BOOL bGpsToStd, short srcClass);
	short ConvertD109WptClass(BOOL bGpsToStd, short srcClass);
	short ConvertD110WptClass(BOOL bGpsToStd, short srcClass);
	short ConvertD150WptClass(BOOL bGpsToStd, short srcClass);
    short ConvertD151WptClass(BOOL bGpsToStd, short srcClass);
	short ConvertD152WptClass(BOOL bGpsToStd, short srcClass);
	short ConvertD154WptClass(BOOL bGpsToStd, short srcClass);
	short ConvertD155WptClass(BOOL bGpsToStd, short srcClass);

    short CategoryToD108Symbol(short nCategory);
    short CategoryToD109Symbol(short nCategory);
	short CategoryToD154Symbol(short nCategory);
	short CategoryToD155Symbol(short nCategory);


private:
	short PrepareD100WptType(unsigned char* ptData);
	short PrepareD101WptType(unsigned char* ptData);
	short PrepareD102WptType(unsigned char* ptData);
	short PrepareD103WptType(unsigned char* ptData);
	short PrepareD104WptType(unsigned char* ptData);
	short PrepareD105WptType(unsigned char* ptData);
	short PrepareD106WptType(unsigned char* ptData);
	short PrepareD107WptType(unsigned char* ptData);
	short PrepareD108WptType(unsigned char* ptData);
	short PrepareD109WptType(unsigned char* ptData);
	short PrepareD110WptType(unsigned char* ptData);

	short PrepareD150WptType(unsigned char* ptData);
	short PrepareD151WptType(unsigned char* ptData);
	short PrepareD152WptType(unsigned char* ptData);

	short PrepareD154WptType(unsigned char* ptData);
	short PrepareD155WptType(unsigned char* ptData);

private:
    void PrepareLimZeroTermString(unsigned char* lpOut, long* ptOffs, 
                                            CString szText, short nMaxLen);
    short SplitLimZeroTermString(unsigned char* ptData, long* ptOffset, 
                                            char* ptText, short nMaxLen);
    void PrepareSubClass (unsigned char* ptData, long* ptOffs);

    unsigned char FilterTextByte(unsigned char inChar, BOOL bToUpper);
    void FilterTextBytes (CString& szName, BOOL bToUpper);
    void PrepareSpaceTermString (unsigned char* ptData, long* ptOffs, 
                                    CString szSource, short nBytes);
    void PrepareFilteredSpaceTermString (unsigned char* ptData, long* ptOffs,
                            CString szSource, short nBytes);
    CString SplitSpaceTerminatedText (unsigned char* lpData, long* ptOffset, short nMaxData);

    void SplitSemicircle(unsigned char* ptData, long* ptOffset, double* ptLat, double* ptLon);
    void PrepareSemicircle(unsigned char* ptData, long* ptOffset, double dLat, double dLon);

	unsigned long SystemToGarminTime (unsigned long lSystemTime);

	void GetCC (char* szCC);
	void SetCC (char* szCC);

	byte PrepareD109DsplColor(short nColor, short nDspl);
	void SplitD109DsplColor(byte dspl_color, short* ptColor, short* ptDspl);

};

#endif
