// InterMap.cpp: Implementierungsdatei
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include <math.h>					// fabs

#include "pf.h"
#include "MainFrm.h"

#include "InitDoc.h"  
#include "MetDoc.h"			    
#include "LocDoc.h"
#include "PlaneDoc.h"
#include "AirspaceDoc.h"
#include "AirspaceDlg.h"

#include "Calc.h"			// this statement must be after PlaneDoc.h

#include "PlAltDlg.h"
#include "FlPlDoc.h"

#include "WayDoc.h"
#include "BearDlg.h"			// for DistAddSorted

#include "..\CPPTOOLS\Vektor.h"                                       
#include "Symbols.h"

#include "InterMap.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

extern CInitDoc* 		ptInit;
extern CDimDoc* 		ptDim;   
extern CPlaneDoc*		ptPlaneDoc;
extern CLocDoc*			ptLocDoc;
extern CFlightPlanDoc*	ptFlPlDoc;
extern CScanMapDoc*		ptScMpDoc;
extern BOOL bMMText;

extern BOOL				bDemo;


#define MOVE_TIMER 4

/////////////////////////////////////////////////////////////////////////////
// CInteractiveMap

IMPLEMENT_DYNAMIC(CInteractiveMap, CObject)


/************************************************************************
 *	InterMap.cpp			C I n t e r a c t i v e M a p				*
 ************************************************************************/										 
CInteractiveMap::CInteractiveMap(CWnd* ptWnd)
{
	m_ptWnd = ptWnd;		// m_ptWnd: CMapView or CScanMapView
	m_ptWayDoc = NULL;

	m_dLat = 0.;
	m_dLon = 0.;

	m_ptLaLoList = ptInit->GetLaLoListPtr();
	m_ptMapDoc	= ptInit->GetMapDocPtr();
	m_ptMapDoc->SetLatLon (NO_KOORD, NO_KOORD);		// used to generate status bar entry

	m_ptWayMenu = NULL;				// appears after right mouse button down
	m_ptMapMenu = NULL;				// appears after STRG and right mouse button down

								// for interactively manipulation of rte
	m_ptMapLocs = new CObArray;			// array of CLocation*
	m_ptMapNavs = new CObArray;			// array of CLocation*
	m_ptMapAirspaces = new CObArray;	// array of CAirspace*
	m_ptRteWpts = new CObArray;			// array of CWayPoint*

	m_bMapLocs = FALSE;
	m_bMapNavs = FALSE;
	m_bRteWpts = FALSE;

	m_uTimer = 0;
}

/************************************************************************
 *	InterMap.cpp			~ C I n t e r a c t i v e M a p	 			*
 ************************************************************************/										 
CInteractiveMap::~CInteractiveMap()
{
m_ptMapDoc->SetLatLon (NO_KOORD, NO_KOORD);		// used to generate status bar entry

DeleteArrayOfLocs(m_ptMapLocs);
DeleteArrayOfLocs(m_ptMapNavs);
DeleteArrayOfAirspaces(m_ptMapAirspaces);
DeleteArrayOfWpts(m_ptRteWpts);
delete m_ptMapLocs;
delete m_ptMapNavs;
delete m_ptMapAirspaces;
delete m_ptRteWpts;
}


/************************************************************************
 *  InterMap.cpp	  	 			S e t L a t L o n 					*
 ************************************************************************/
void CInteractiveMap::SetLatLon	(double dLat, double dLon)
{
if (dLon >  180.) dLon -= 360;
if (dLon < -180.) dLon += 360;

m_dLat = dLat;
m_dLon = dLon;

m_ptMapDoc->SetLatLon (dLat, dLon);		// used to generate status bar entry
}


/************************************************************************
 *  InterMap.cpp	  	 S t a r t P o s C h a n g e d T i m e r		*
 ************************************************************************/
void CInteractiveMap::StartPosChangedTimer(short nTenthSec)
{
	this->StopPosChangedTimer();

	short nMilliSec = nTenthSec * 100;
	CWnd* ptParentWnd = m_ptWnd->GetParent();	  
	m_uTimer = ptParentWnd->SetTimer (MOVE_TIMER, nMilliSec, NULL);
}

/************************************************************************
 *  InterMap.cpp	  	 S t o p P o s C h a n g e d T i m e r			*
 ************************************************************************/
void CInteractiveMap::StopPosChangedTimer()
{
	if (m_uTimer != 0)
	{
		CWnd* ptParentWnd = m_ptWnd->GetParent();	  
		ptParentWnd->KillTimer(m_uTimer);
		m_uTimer = 0;
	}
}


/************************************************************************
 *  InterMap.cpp	  	 G e t I n i t i a l P l a n n e d A l t 		*
 *  Try to get planned altitude from previous waypoint or				*
 *	Try to get safe altitude from actual location or					*
 *  Try to get elevation of actual location from elevarray				*
 ************************************************************************/
long CInteractiveMap::GetInitialPlannedAlt(CLocation* ptLoc, short nActWptIndex, short nAltDim)
{
	long lPlannedAlt = NO_ALT;
	long lSafeAlt = NO_ALT;

	if (nActWptIndex > 0)		// get planned altitude from last rte wpt
	{
		CWayPoint WptOfRte;
		if (m_ptWayDoc->GetWayPointPtr (nActWptIndex, &WptOfRte))
		{
			lPlannedAlt = WptOfRte.GetPlannedAlt();
			short nWptAltDim = WptOfRte.GetAltDim();
			if (nWptAltDim != nAltDim)
			{
				double fConvert;
				fConvert = ptDim->ConvertDist((double)lPlannedAlt, nWptAltDim, nAltDim);
				lPlannedAlt = (long)(fConvert + 0.5);
			}
		}	
	}

										// try to get safe alt from ptLoc
	lSafeAlt = ptLoc->GetSafeAlt_MSL(nAltDim);
	if (lSafeAlt == NO_ALT)
	{							// no elevation saved in ptLoc
		CElevArrayDoc* ptElevArrayDoc = ptInit->GetElevArrayDocPtr();
		if (ptElevArrayDoc != NULL)
		{						// get Elev from ElevArray and calc safe alt again
			short nElevFrGrid_m;
			if (ptElevArrayDoc->GetElev_m((CLatLon)*ptLoc, &nElevFrGrid_m))
			{
				double fConvert;				
				short	nElevDim = ptLoc->GetElevDim();
				fConvert = ptDim->ConvertDist((double)nElevFrGrid_m, DIM_METER, nElevDim);
								// define elevation of ptLoc
				ptLoc->SetAltitude ((long)(fConvert + 0.5));
				lSafeAlt = ptLoc->GetSafeAlt_MSL(nAltDim);
			}
		}
	}



	if (lPlannedAlt == NO_ALT)	
	{		
		if (lSafeAlt == NO_ALT)
		{
			// no initial planned altitude for CPlAltDlg
		}
		else
		{	// use lSafeAlt as initial altitude
			lPlannedAlt = lSafeAlt;
		}
	}
	else
	{					
		if (lSafeAlt == NO_ALT)
		{
			// use lPlannedAlt as initial altitude
		}
		else
		{	// use safe alt only if planned alt is too low
			if (lSafeAlt > lPlannedAlt)
				lPlannedAlt = lSafeAlt;
		}
	}

	return lPlannedAlt;			// may be NO_ALT

}


/////////////////////////////////////////////////////////////////////////////
// Behandlungsroutinen fr Nachrichten CInteractiveMap 

/************************************************************************
 *  InterMap.cpp	  	 O n M e n u M e E d i t M a p L o c			*
 ************************************************************************/
void CInteractiveMap::OnMenuMeEditMapLoc(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
short	nIndex = nID - ID_ME_LOCEDIT_FIRST;

CLocation* ptLoc = (CLocation*)m_ptMapLocs->GetAt(nIndex);
if (ptLoc != NULL)
	{
	BOOL bShowLoc = TRUE;
	if (ptLocDoc->Change(m_ptWnd, ptLoc, bShowLoc))
		{
		ptLocDoc->SetModifiedFlag(TRUE);
		m_ptWnd->Invalidate(FALSE);
		}
	}
}


/************************************************************************
 *  InterMap.cpp	  	O n M e n u M e E d i t A i r s p a c e 		*
 ************************************************************************/
void CInteractiveMap::OnMenuMeEditAirspace(UINT nID, CWayDoc* ptWayDoc) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
	short	nIndex = nID - ID_ME_AIRSPACEEDIT_FIRST;

	m_ptWayDoc = ptWayDoc;

	CAirspace* ptAirspace = (CAirspace*)m_ptMapAirspaces->GetAt(nIndex);
	if (ptAirspace != NULL)
	{				// m_ptWnd: CMapView or CScanMapView
		CAirspaceDlg AirspaceDlg(m_ptWnd, ptAirspace, TRUE);
		AirspaceDlg.SetWayDocPtr(ptWayDoc);
		int RetVal = AirspaceDlg.DoModal();
		
		if (RetVal == IDOK)
		{
		}

		m_ptWnd->InvalidateRect(NULL); 
	}
}



/************************************************************************
 *  InterMap.cpp	  	 	O n M e n u A p p M a p L o c				*
 ************************************************************************/
void CInteractiveMap::OnMenuAppMapLoc(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
short	nIndex = nID - ID_WY_APP_FIRST;

CLocation* ptLoc = (CLocation*)m_ptMapLocs->GetAt(nIndex);
if (ptLoc != NULL)
	{
	if (m_ptWayDoc != NULL)
		{	
		short nActWptIndex = m_ptWayDoc->GetWayPointCnt() - 1;
		if (m_ptWayDoc->SameLocAround (nActWptIndex, *ptLoc))
			{	// new way point too close at last way point!!
			}
		else{
			CWayPoint WayPt;

			short	nAltDim = ptDim->Altitude();
			long	lPlannedAlt = this->GetInitialPlannedAlt (ptLoc, nActWptIndex, nAltDim);


			WayPt.SetLoc (*ptLoc);
			WayPt.SetPlannedAlt (lPlannedAlt);	
			WayPt.SetAltDim (nAltDim);

			CPlAltDlg PlannedAltBox (m_ptWnd, &WayPt);
			if (PlannedAltBox.DoModal () == IDOK)
				{
				if (m_ptWayDoc->AppendWpt (WayPt))
					{
					if (m_ptWayDoc->IsUpDown())	// if UpDown activated:
						{
						m_ptWayDoc->UpdateUpDown();
						}

					m_ptWayDoc->SetActInd (nActWptIndex+1);
					ptInit->SetLatLonToCenter(WayPt.GetLat(), WayPt.GetLon());
					m_ptWayDoc->SetModifiedFlag(TRUE);
					ptFlPlDoc->SetUpdateRequired (TRUE);
					m_ptWnd->Invalidate(FALSE);
					}
				}
			}
		if (bDemo) m_ptWayDoc->SetModifiedFlag(FALSE);
		}
	}
}

/************************************************************************
 *  InterMap.cpp	  	 	O n M e n u I n s M a p L o c				*
 ************************************************************************/
void CInteractiveMap::OnMenuInsMapLoc(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
short	nIndex = nID - ID_WY_INS_FIRST;

CLocation* ptLoc = (CLocation*)m_ptMapLocs->GetAt(nIndex);
if (ptLoc != NULL)
	{
	if (m_ptWayDoc != NULL)
		{	 
		short nActWptIndex = m_ptWayDoc->GetActInd();
		if (nActWptIndex >=0 && nActWptIndex <m_ptWayDoc->GetWayPointCnt())
			{
			if (m_ptWayDoc->SameLocAround (nActWptIndex, *ptLoc))
				{

				}
			else{
				CWayPoint WayPt;
				short	nAltDim = ptDim->Altitude();
				long	lPlannedAlt = this->GetInitialPlannedAlt (ptLoc, nActWptIndex, nAltDim);


				WayPt.SetLoc (*ptLoc);
				WayPt.SetPlannedAlt (lPlannedAlt);	
				WayPt.SetAltDim (nAltDim);

				CPlAltDlg PlannedAltBox (m_ptWnd, &WayPt);
				if (PlannedAltBox.DoModal () == IDOK)
					{
					if (m_ptWayDoc->InsertWpt (nActWptIndex+1, WayPt))
						{
						if (m_ptWayDoc->IsUpDown())	// if UpDown activated:
							{
							m_ptWayDoc->UpdateUpDown();
							}
						m_ptWayDoc->SetActInd (nActWptIndex+1);
						ptInit->SetLatLonToCenter(WayPt.GetLat(), WayPt.GetLon());
						m_ptWayDoc->SetModifiedFlag(TRUE);
						ptFlPlDoc->SetUpdateRequired (TRUE);
						m_ptWnd->Invalidate(FALSE);
						}
					}
				}
			}
		if (bDemo) m_ptWayDoc->SetModifiedFlag(FALSE);
		}
	}
}

/************************************************************************
 *  InterMap.cpp	  	 	O n M e n u B e a r M a p N a v				*
 ************************************************************************/
void CInteractiveMap::OnMenuBearMapNav(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
short	nIndex = nID - ID_WY_BEAR_FIRST;

CLocation* ptNavLoc = (CLocation*)m_ptMapNavs->GetAt(nIndex);
if (ptNavLoc != NULL)
	{
	if (m_ptWayDoc != NULL)
		{	 
		short nActWptIndex = m_ptWayDoc->GetActInd();
		if (nActWptIndex >= 0)
			{
			CWayPoint WptOfRte;
			if (m_ptWayDoc->GetWayPointPtr (nActWptIndex, &WptOfRte))
				{
				short i;
				BOOL bDeleted = FALSE;
				BOOL bAdded = FALSE;

				for (i=0; i<2; i++)
					{							// delete bearing?
					if (WptOfRte.HasBearing (i))
						{
						CString szNav = WptOfRte.GetBearingName(i); 
						if (szNav.Compare (ptNavLoc->GetName()) == 0)
							{
							WptOfRte.DeleteBearing(i);
							bDeleted = TRUE;
							}
						}
					}

				if (!bDeleted)
					{
					short nBearInd=-1;	// get bearing index 0 or 1
					if (!WptOfRte.HasBearing (0))
						{
						nBearInd = 0;
						}
					else{
						if (!WptOfRte.HasBearing (1))
							nBearInd = 1;
						}

					if (nBearInd > -1)
						{
						WptOfRte.CalculateBearing (nBearInd, *ptNavLoc);
						bAdded = TRUE;
						}
					}

				if (bDeleted || bAdded)
					{
					if (m_ptWayDoc->ChangeWpt (nActWptIndex, WptOfRte))
						{
						m_ptWayDoc->SetModifiedFlag(TRUE);
						m_ptWnd->Invalidate(FALSE);
						}
					}
				}	
			}
		if (bDemo) m_ptWayDoc->SetModifiedFlag(FALSE);
		}
	}
}

/************************************************************************
 *  InterMap.cpp	  	 	O n M e n u D e l R t e W p t				*
 ************************************************************************/
void CInteractiveMap::OnMenuDelRteWpt(UINT nID)
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
	short	nIndex = nID - ID_WY_DEL_FIRST;

	CWayPoint* ptWpt = (CWayPoint*)m_ptRteWpts->GetAt(nIndex);
	if (ptWpt != NULL)
	{
		if (m_ptWayDoc != NULL)
		{	 
			short i;
			BOOL	bFound = FALSE;
			short	nWptCnt = m_ptWayDoc->GetWayPointCnt();

			for (i=0; i<nWptCnt && !bFound; i++)
			{
				CWayPoint WptOfRte;
				if (m_ptWayDoc->GetWayPointPtr (i, &WptOfRte))
				{
					if (WptOfRte.IsEqual(*ptWpt))
					{
						bFound = TRUE;
						m_ptWayDoc->DeleteWpt (i);

						m_ptWayDoc->SetModifiedFlag(TRUE);
						ptFlPlDoc->SetUpdateRequired (TRUE);


						BOOL bActWptVisible = FALSE;
						
						// is actual waypoint visible on the current map?
						CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex());
						if (ptMap != NULL)
						{
							CLatLon llActWpt;
							m_ptWayDoc->GetActLatLonPtr(&llActWpt);
							bActWptVisible = ptMap->IsLatLonInRect(llActWpt.GetLat(), llActWpt.GetLon());
						}
						
						if(bActWptVisible)
						{		// stay on current map
							m_ptWnd->Invalidate(FALSE);
						}
						else
						{		// look for new map
							CMainFrame* ptMainFrame = (CMainFrame*)m_ptWnd->GetParent();	  
							ptMainFrame->ShowActWaypoint(m_ptWayDoc);
						}
					}
				}
			}
			if (bDemo) m_ptWayDoc->SetModifiedFlag(FALSE);
		}
	}
}

/************************************************************************
 *  InterMap.cpp	  	 	O n M e n u A c t R t e W p t				*
 ************************************************************************/
void CInteractiveMap::OnMenuActRteWpt(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
short	nIndex = nID - ID_WY_ACT_FIRST;

CWayPoint* ptWpt = (CWayPoint*)m_ptRteWpts->GetAt(nIndex);
if (ptWpt != NULL)
	{
	if (m_ptWayDoc != NULL)
		{	 
		short i;
		BOOL	bFound = FALSE;

		for (i=0; i<m_ptWayDoc->GetWayPointCnt() && !bFound; i++)
			{
			CWayPoint WptOfRte;
			if (m_ptWayDoc->GetWayPointPtr (i, &WptOfRte))
				{
				if (WptOfRte.IsEqual(*ptWpt))
					{
					short nActWptIndex = m_ptWayDoc->GetActInd();
					bFound = TRUE;
					if (nActWptIndex != i)
						{
						m_ptWayDoc->SetActInd (i);
						m_ptWayDoc->SetModifiedFlag(TRUE);
						ptFlPlDoc->SetUpdateRequired (TRUE);
						m_ptWnd->Invalidate(FALSE);
						}
					}
				}
			}
		if (bDemo) m_ptWayDoc->SetModifiedFlag(FALSE);
		}
	}
}

/************************************************************************
 *  InterMap.cpp	  	 	O n M e n u P l A l t R t e W p t			*
 ************************************************************************/
void CInteractiveMap::OnMenuPlAltRteWpt(UINT nID)
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
short	nIndex = nID - ID_WY_PLALT_FIRST;

CWayPoint* ptWpt = (CWayPoint*)m_ptRteWpts->GetAt(nIndex);
if (ptWpt != NULL)
	{
	if (m_ptWayDoc != NULL)
		{	 
		short i;
		BOOL	bFound = FALSE;

		for (i=0; i<m_ptWayDoc->GetWayPointCnt() && !bFound; i++)
			{
			CWayPoint WayPt;
			if (m_ptWayDoc->GetWayPointPtr (i, &WayPt))
				{
				if (WayPt.IsEqual(*ptWpt))
					{
					bFound = TRUE;

					CPlAltDlg PlannedAltBox (m_ptWnd, &WayPt);
					if (PlannedAltBox.DoModal () == IDOK)
						{
						if (m_ptWayDoc->ChangeWpt (i, WayPt))
							{
							if (m_ptWayDoc->IsUpDown())	// if UpDown activated:
								{
								m_ptWayDoc->UpdateUpDown();
								}
							m_ptWayDoc->SetModifiedFlag(TRUE);
							ptFlPlDoc->SetUpdateRequired (TRUE);
							m_ptWnd->Invalidate(FALSE);
							}
						}
					}
				}
			}
		if (bDemo) m_ptWayDoc->SetModifiedFlag(FALSE);
		}
	}
}

/************************************************************************
 *  InterMap.cpp	  	 	I s M o u s e O n L e g						*
 ************************************************************************/
BOOL CInteractiveMap::IsMouseOnLeg (CWayPoint& WayPtA, CWayPoint& WayPtB, double* ptDist)
{														// dist in X,Y,Z - values
BOOL bOnLeg = FALSE;

CLatLon LatLonA (WayPtA.GetLat(), WayPtA.GetLon()); 
CLatLon LatLonB (WayPtB.GetLat(), WayPtB.GetLon()); 
CLatLon LatLonP (m_dLat, m_dLon); 

CVektor vA, vB,vP, vR, vAP, vS, vSP;
double	dFaktB, dFaktS;

LatLonA.ToXYZ (&vA);
LatLonB.ToXYZ (&vB);
LatLonP.ToXYZ (&vP);

vR = vB - vA;
vAP = vP - vA;

if (vR.Abs() > 0)
	{
	dFaktB = 1.;	// vB = vA + vR * dFaktB
	dFaktS = vR.SkalarProdukt (vAP) / vR.SkalarProdukt(vR);
	vS = vA + vR * dFaktS;

	vSP = vP - vS;
	*ptDist = vSP.Abs();
	bOnLeg = ((1 >= dFaktS) && (dFaktS >= 0));
	}

return bOnLeg;
}


/************************************************************************
 *  InterMap.cpp	  	 	D i s t M o u s e T o W p t					*
 ************************************************************************/
double CInteractiveMap::DistMouseToWpt (CWayPoint& WayPt)
{
double dDist;	   // in X,Y,Z - values

CLatLon LatLon (WayPt.GetLat(), WayPt.GetLon()); 
CLatLon LatLonP (m_dLat, m_dLon); 

CVektor vA, vP, vAP;

LatLon.ToXYZ (&vA);
LatLonP.ToXYZ (&vP);

vAP = vP - vA;
dDist = vAP.Abs();
return dDist;
}

/************************************************************************
 *  InterMap.cpp    	 	O n S h o w S c a n n e d M a p				*
 ************************************************************************/
void CInteractiveMap::OnShowScannedMap (double dLat, double dLon, CWnd* ptParentWnd, WORD wCmdID)
{
short i, nActIndex;
double dMinDist;

nActIndex = -1;

for (i=0; i<ptScMpDoc->GetMapCnt(); i++)
	{
	CScannedMap* ptMap = ptScMpDoc->GetMapPtr (i);
	if (ptMap != NULL)
		{
		long lSizeX, lSizeY;
		double dTestLat, dTestLon, dDist, dCourse;
		CLatLon llMapCenter, llMouse;

								// get center of scanned map
		ptMap->GetScaledSizeLP (&lSizeX, &lSizeY);
		if (!bMMText) lSizeY = -lSizeY;
		ptMap->LPtoLatLon (lSizeX/2, lSizeY/2, &dTestLat, &dTestLon);
		llMapCenter.SetLatLon (dTestLat, dTestLon);

		llMouse.SetLatLon (dLat, dLon);


		dDist = llMouse.LoxoDist (llMapCenter, &dCourse);

		if (i==0)
			{
			dMinDist = dDist;
			nActIndex = i;
			}
		else{
			if (dDist < dMinDist) 
				{
				dMinDist = dDist;
				nActIndex = i;
				}
			}
		}
	}

if (nActIndex >= 0)
	{
	CString szText;
//	szText=CTextViewTool::Format(IDF_SWITCH_TO_MAP, 
//								ptScMpDoc->GetMapName(nActIndex));
	szText.Format(IDF_SWITCH_TO_MAP, 
								ptScMpDoc->GetMapName(nActIndex));
	if (AfxMessageBox ((LPCSTR)szText, MB_YESNO) == IDYES) 
		{
		ptScMpDoc->ActivateMap (nActIndex);

		ptScMpDoc->SetAutoSelect(FALSE);		// avoid activating other map to view route

		WPARAM wParam = MAKELONG (wCmdID, 0);
		ptParentWnd->SendMessage(WM_COMMAND, wParam, 0);
		}
	}
}


/************************************************************************
 *  InterMap.cpp	  	 	O n A c t i v a t e L e g					*
 *  Purpose: searches leg next to mouse. 								*
 *			On last leg, last wpt is checked also						*
 ************************************************************************/
void CInteractiveMap::OnActivateLeg(CWayDoc* ptDoc) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
short	i, nActInd;
double	dDistFromMouse;			
double	dMinDist;

CWayPoint WayPtA, WayPtB;
BOOL	bFound = FALSE;

m_ptWayDoc = ptDoc;

if (m_ptWayDoc != NULL)
	{	
	BOOL bCalcTemp = FALSE;
	CPlane	ActPlane;			// calculate for up down points	
	CCalculate* ptLegs = NULL;

	if (bCalcTemp)
		{
		if (ptPlaneDoc != NULL)
			{
			short nIndex = ptPlaneDoc->GetActIndex();
			if (ptPlaneDoc->GetPlanePtr(nIndex, &ActPlane))
				{
				ptLegs = m_ptWayDoc->GetCalculationPtr();
				ptLegs->Activate(&ActPlane);
				}
			}						// end calculation for up down points
		}

	short nCnt = m_ptWayDoc->GetWayPointCnt();
	if (nCnt > 1)
		{
		for (i=0; i<nCnt-1; i++)
			{
			if (m_ptWayDoc->GetWayPointPtr (i,   &WayPtA) &&
				m_ptWayDoc->GetWayPointPtr (i+1, &WayPtB) )
				{						// check distance from each leg
				if (IsMouseOnLeg (WayPtA, WayPtB, &dDistFromMouse))
					{
					if (bFound)
						{
						if (dDistFromMouse < dMinDist)
							{
							dMinDist = dDistFromMouse;
							nActInd = i;
							}
						}
					else{
						bFound = TRUE;
						dMinDist = dDistFromMouse;
						nActInd = i;
						}
					}
				else{
					if (i==nCnt-2)		  // if mouse not on last leg:
						{				  // check distance to last waypoint
						dDistFromMouse = DistMouseToWpt (WayPtB);
						if (bFound)
							{
							if (dDistFromMouse < dMinDist)
								{			  // mark last waypoint only
								dMinDist = dDistFromMouse;
								nActInd = i+1;
								}
							}
						else{
							bFound = TRUE;
							dMinDist = dDistFromMouse;
							nActInd = i+1;
							}
						}
					}
				}
			}
		}



	if (bFound)
		{
		m_ptWayDoc->SetActInd (nActInd);
		m_ptWnd->Invalidate(FALSE);
		}

	if (ptLegs != NULL)
		ptLegs->DeActivate ();
	}
}

/************************************************************************
 *  InterMap.cpp	  	 	O n U p d a t e X X X						*
 ************************************************************************/

void CInteractiveMap::OnUpdateWyApp(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
pCmdUI->Enable(m_bMapLocs);	
}

void CInteractiveMap::OnUpdateWyIns(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
BOOL bDisabled = TRUE;

if (m_ptWayDoc != NULL)
	{	 
	short nActWptIndex = m_ptWayDoc->GetActInd();
	bDisabled = (nActWptIndex==-1 || !m_bMapLocs);
	}

pCmdUI->Enable(!bDisabled);
}

void CInteractiveMap::OnUpdateWyBear(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
BOOL bDisabled = TRUE;
BOOL bAllBearings = FALSE;

if (m_ptWayDoc != NULL)
	{	 
	short nActWptIndex = m_ptWayDoc->GetActInd();
	if (nActWptIndex > -1)
		{
		CWayPoint WayPt;
		if (m_ptWayDoc->GetWayPointPtr (nActWptIndex, &WayPt))
			{
			bAllBearings = (WayPt.HasBearing(0) && WayPt.HasBearing(1));
			}
		}
	bDisabled = (nActWptIndex==-1 || !m_bMapNavs || bAllBearings);
	}
 
pCmdUI->Enable(!bDisabled);
}

void CInteractiveMap::OnUpdateWyDel(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
pCmdUI->Enable(m_bRteWpts);
}

void CInteractiveMap::OnUpdateWyAct(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
pCmdUI->Enable(m_bRteWpts);
}

void CInteractiveMap::OnUpdateWyPlAlt(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
pCmdUI->Enable(m_bRteWpts);
}

void CInteractiveMap::OnUpdateWyLocEdit(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
pCmdUI->Enable(m_bMapLocs);	
}


/************************************************************************
 *  InterMap.cpp			D e l e t e A r r a y O f L o c s			*
 ************************************************************************/
void CInteractiveMap::DeleteArrayOfLocs (CObArray* ptArray)
{
int i, nEntryCnt;	
											
nEntryCnt = ptArray->GetSize();
for (i=0; i<nEntryCnt; i++)
	{
	CLocation* ptEntry;
	if ((ptEntry = (CLocation*)ptArray->GetAt(i)) != NULL)
		{
		TRACE ("Deleting Loc/Nav of InterMap = %d\n", i);
		delete ptEntry;				// delete original element
		}
	}
ptArray->RemoveAll();
}

/************************************************************************
 *  InterMap.cpp	D e l e t e A r r a y O f A i r s p a c e s			*
 ************************************************************************/
void CInteractiveMap::DeleteArrayOfAirspaces (CObArray* ptArray)
{
int i, nEntryCnt;	
											
nEntryCnt = ptArray->GetSize();
for (i=0; i<nEntryCnt; i++)
	{
	CAirspace* ptEntry;
	if ((ptEntry = (CAirspace*)ptArray->GetAt(i)) != NULL)
		{
		TRACE ("Deleting Airspace of InterMap = %d\n", i);
		delete ptEntry;				// delete original element
		}
	}
ptArray->RemoveAll();
}

/************************************************************************
 *  InterMap.cpp			D e l e t e A r r a y O f W p t s			*
 ************************************************************************/
void CInteractiveMap::DeleteArrayOfWpts (CObArray* ptArray)
{
int i, nEntryCnt;	
											
nEntryCnt = ptArray->GetSize();
for (i=0; i<nEntryCnt; i++)
	{
	CWayPoint* ptEntry;
	if ((ptEntry = (CWayPoint*)ptArray->GetAt(i)) != NULL)
		{
		TRACE ("Deleting Wpts of InterMap = %d\n", i);
		delete ptEntry;				// delete original element
		}
	}
ptArray->RemoveAll();
}

/************************************************************************
 *  InterMap.cpp	  		 	S e a r c h M a p L o c s 				*
 ************************************************************************/
BOOL CInteractiveMap::SearchMapLocs(CObArray* ptMapLocs, CObArray* ptMapNavs) 
{
CLocation Loc;
DRECT	rLatLon;
short	i;
double	dDistFromMouse;				// dummy
CDWordArray	Distances;		// array of DWORDS for distances
CDWordArray	NavDists;		// array of DWORDS for nav aid distances
short	nActRegIndex = 0;
BOOL	bWpts = FALSE;

rLatLon = CLatLon::RadiusToDRECT (m_dLat, m_dLon, 0.25);	// degrees	

if (m_ptLaLoList->PrepareGetLoc (rLatLon))
	{
	while (m_ptLaLoList->GetLocationInRect (&Loc, rLatLon, &nActRegIndex))
		{
		double dKurs;

		dDistFromMouse = Loc.LoxoDist (m_dLat, m_dLon, &dKurs);
		CBearDlg::DistAddSorted (&Distances, &i, dDistFromMouse);

		CLocation* ptLoc = new CLocation(Loc);
		ptMapLocs->InsertAt (i, ptLoc);

		if (m_ptWayDoc != NULL)
			{	 
			double dDistFromActWpt;
			short nActWptIndex = m_ptWayDoc->GetActInd();
			if (nActWptIndex > -1)
				{
				CWayPoint WayPt;
				if (m_ptWayDoc->GetWayPointPtr (nActWptIndex, &WayPt))
					{		// is Loc a facility near to WayPt?
					if (Loc.IsFacilityInRange(WayPt, &dDistFromActWpt))
						{
						CBearDlg::DistAddSorted (&NavDists, &i, dDistFromMouse);
						ptLoc = new CLocation(Loc);
						ptMapNavs->InsertAt (i, ptLoc);
						}
					}
				}
			}
		}
	}
m_bMapNavs  = (ptMapNavs->GetSize() > 0);

						// put actual lat lon at first position
CLocation* ptUsersLoc = new CLocation();
ptUsersLoc->SetLatLon (m_dLat, m_dLon);
ptUsersLoc->UseLatLonName(); 
if (ptMapLocs->GetSize() > 0)
	{					// use nearest known map location to get variation  
	CLocation* ptNearestLoc = (CLocation*)ptMapLocs->GetAt(0);
	if (ptNearestLoc != NULL)
		{
		ptUsersLoc->SetVariation (ptNearestLoc->GetVariation());
		}
	}
ptMapLocs->InsertAt (0, ptUsersLoc);
m_bMapLocs = (ptMapLocs->GetSize() > 0);
bWpts = (ptMapLocs->GetSize() > 0);
 
return bWpts;
}

/************************************************************************
 *  InterMap.cpp	  	S e a r c h M a p A i r s p a c e s 			*
 ************************************************************************/
BOOL CInteractiveMap::SearchMapAirspaces(CObArray* ptMapAirspaces) 
{
	CAirspace* ptAirspace;
	DRECT	rLatLon;
	short	i;
	double	dDistFromMouse;				// dummy
	CDWordArray	Distances;		// array of DWORDS for distances
	CDWordArray	NavDists;		// array of DWORDS for nav aid distances
	short	nActRegIndex = 0;
	short	nActAirspaceIndex = 0;		// region relative index!!
	BOOL	bAirspaces = FALSE;

	rLatLon = CLatLon::RadiusToDRECT (m_dLat, m_dLon, 0.25);	// degrees	

	CAirspaceRegionsDoc* ptAspRegionsDoc = ptInit->GetAirspaceRegionsDocPtr();


	while (ptAspRegionsDoc->GetAirspaceInRect (rLatLon, &ptAirspace, &nActRegIndex, &nActAirspaceIndex))
	{
		double dKurs;

		dDistFromMouse = ptAirspace->LoxoDist (m_dLat, m_dLon, &dKurs);
		CBearDlg::DistAddSorted (&Distances, &i, dDistFromMouse);

		CAirspace* ptMapAirspace = new CAirspace(*ptAirspace);
		ptMapAirspaces->InsertAt (i, ptMapAirspace);
	}


	bAirspaces = (ptMapAirspaces->GetSize() > 0);
 
	return bAirspaces;
}


/************************************************************************
 *  InterMap.cpp	  	 	S e a r c h R t e W p t s 					*
 ************************************************************************/
BOOL CInteractiveMap::SearchRteWpts(CObArray* ptRteWpts) 
{
CWayPoint WayPt;
DRECT	rLatLon;
short	i = 0;
double	dDistFromMouse;				// dummy
CDWordArray	Distances;		// array of DWORDS for distances
BOOL	bWpts = FALSE;

if (m_ptWayDoc != NULL)
	{	
	rLatLon = WayPt.RadiusToDRECT (m_dLat, m_dLon, 0.25);	// degrees	
 
	for (i=0; i<m_ptWayDoc->GetWayPointCnt(); i++)
		{
		if (m_ptWayDoc->GetWayPointPtr (i, &WayPt))
			{
			if ((rLatLon.bottom <= WayPt.GetLat() && WayPt.GetLat() <= rLatLon.top) &&
				(rLatLon.left <= WayPt.GetLon() && WayPt.GetLon() <= rLatLon.right) )
				{
				short	nIndex;
				double	dKurs;
				dDistFromMouse = WayPt.LoxoDist (m_dLat, m_dLon, &dKurs);
				CBearDlg::DistAddSorted (&Distances, &nIndex, dDistFromMouse);

				CWayPoint* ptWpt = new CWayPoint(WayPt);
				ptRteWpts->InsertAt (nIndex, ptWpt);
				}
			}
		}
	}

bWpts = (ptRteWpts->GetSize() > 0);
m_bRteWpts  = (ptRteWpts->GetSize() > 0);

return bWpts;
}

/************************************************************************
 *  InterMap.cpp	  	 	N e w M a p L o c s M e n u					*
 ************************************************************************/
void CInteractiveMap::NewMapLocsMenu(CMenu* ptMenu, short nSubMenuIndex, 
								long lFirstID, long lLastID, CObArray* ptMapLocs) 
{
CMenu* ptSubMenu = ptMenu->GetSubMenu(nSubMenuIndex);

if (ptSubMenu != NULL)
	{
	short i;
	short	nMaxCnt = (short)(lLastID - lFirstID + 1);
	short nCnt = ptSubMenu->GetMenuItemCount();

	for (i=nCnt-1; i>=0; i--)
		ptSubMenu->RemoveMenu (i, MF_BYPOSITION);

	for (i=0; i<ptMapLocs->GetSize() && i<nMaxCnt; i++)
		{ 
		CLocation* ptLoc = (CLocation*)ptMapLocs->GetAt(i);
		if (ptLoc != NULL)
			{ 
			ptSubMenu->AppendMenu (MF_ENABLED | MF_STRING, 
							(UINT)(lFirstID + i), 
							(LPCTSTR)ptLoc->GetName());  
			}
		}
	}
}

/************************************************************************
 *  InterMap.cpp	  	 N e w M a p A i r s p a c e s M e n u			*
 ************************************************************************/
void CInteractiveMap::NewMapAirspacesMenu(CMenu* ptMenu, short nSubMenuIndex, 
								long lFirstID, long lLastID, CObArray* ptMapAirspaces) 
{
CMenu* ptSubMenu = ptMenu->GetSubMenu(nSubMenuIndex);

if (ptSubMenu != NULL)
	{
	short i;
	short	nMaxCnt = (short)(lLastID - lFirstID + 1);
	short nCnt = ptSubMenu->GetMenuItemCount();

	for (i=nCnt-1; i>=0; i--)
		ptSubMenu->RemoveMenu (i, MF_BYPOSITION);

	for (i=0; i<ptMapAirspaces->GetSize() && i<nMaxCnt; i++)
		{ 
		CAirspace* ptAirspace = (CAirspace*)ptMapAirspaces->GetAt(i);
		if (ptAirspace != NULL)
			{ 
			ptSubMenu->AppendMenu (MF_ENABLED | MF_STRING, 
							(UINT)(lFirstID + i), 
							(LPCTSTR)ptAirspace->GetName());  
			}
		}
	}
}

/************************************************************************
 *  InterMap.cpp	  		N e w R t e W p t s M e n u					*
 ************************************************************************/
void CInteractiveMap::NewRteWptsMenu(CMenu* ptMenu, short nSubMenuIndex, 
								long lFirstID, long lLastID, CObArray* ptRteWpts) 
{
CMenu* ptSubMenu = ptMenu->GetSubMenu(nSubMenuIndex);

if (ptSubMenu != NULL)
	{
	short i;
	short	nMaxCnt = (short)(lLastID - lFirstID + 1);
	short nCnt = ptSubMenu->GetMenuItemCount();

	for (i=nCnt-1; i>=0; i--)
		ptSubMenu->RemoveMenu (i, MF_BYPOSITION);

	for (i=0; i<ptRteWpts->GetSize() && i<nMaxCnt; i++)
		{ 
		CWayPoint* ptWpt = (CWayPoint*)ptRteWpts->GetAt(i);
		if (ptWpt != NULL)
			{ 
			ptSubMenu->AppendMenu (MF_ENABLED | MF_STRING, 
							(UINT)(lFirstID + i), 
							(LPCTSTR)ptWpt->GetName());  
			}
		}
	}
}

/************************************************************************
 *  InterMap.cpp	  		S h o w W a y P o p u p M e n u				*
 ************************************************************************/
void CInteractiveMap::ShowWayPopupMenu (CPoint point, CWayDoc* ptWayDoc)
{
BOOL	bMapWpt, bRteWpt;
short	nWayMenuID = IDR_WAY;

m_ptWayDoc = ptWayDoc;

DeleteArrayOfLocs(m_ptMapLocs);					// delete contents of old arrays
DeleteArrayOfLocs(m_ptMapNavs);
DeleteArrayOfWpts(m_ptRteWpts);
				    
bMapWpt = this->SearchMapLocs (m_ptMapLocs, m_ptMapNavs);	
bRteWpt = this->SearchRteWpts (m_ptRteWpts);

if (bMapWpt && !bRteWpt)
	nWayMenuID = IDR_WAY_MAP;
if (!bMapWpt && bRteWpt)
	nWayMenuID = IDR_WAY_RTE;
if (bMapWpt && bRteWpt)
	nWayMenuID = IDR_WAY_MAPRTE;

		
CMenu Menu;
if (Menu.LoadMenu(nWayMenuID))
	m_ptWayMenu = Menu.GetSubMenu(0);


if (m_ptWayMenu != NULL)
	{
	if (bMapWpt)
		{				
		this->NewMapLocsMenu (m_ptWayMenu, 0, 
							ID_WY_INS_FIRST, ID_WY_INS_LAST, m_ptMapLocs);
		this->NewMapLocsMenu (m_ptWayMenu, 1, 
							ID_WY_APP_FIRST, ID_WY_APP_LAST, m_ptMapLocs);
		this->NewMapLocsMenu (m_ptWayMenu, 2, 
							ID_WY_BEAR_FIRST, ID_WY_BEAR_LAST, m_ptMapNavs);
		}

	if (bRteWpt)
		{
		this->NewRteWptsMenu (m_ptWayMenu, 3, 
							ID_WY_DEL_FIRST, ID_WY_DEL_LAST, m_ptRteWpts);
		this->NewRteWptsMenu (m_ptWayMenu, 4, 
							ID_WY_ACT_FIRST, ID_WY_ACT_LAST, m_ptRteWpts);
		this->NewRteWptsMenu (m_ptWayMenu, 5, 
							ID_WY_PLALT_FIRST, ID_WY_PLALT_LAST, m_ptRteWpts);
		}

	m_ptWayMenu->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON, 
									point.x, point.y, m_ptWnd);
	}

}



/************************************************************************
 *  CInteractiveMap.cpp	  		CheckSymbolsMenu 						*
 ************************************************************************/
void CInteractiveMap::CheckSymbolsMenu (long lRsrcID, BOOL bCheck)
{
UINT uActiv = (bCheck)? MF_CHECKED : MF_UNCHECKED;
if (m_ptMapMenu != NULL)	// set check mark at new plane
    m_ptMapMenu->CheckMenuItem ((UINT)(lRsrcID), uActiv | MF_BYCOMMAND);
}

/************************************************************************
 *  CInteractiveMap.cpp	  		S h o w M a p P o p u p M e n u			*
 ************************************************************************/
void CInteractiveMap::ShowMapPopupMenu (CPoint point)
{
	BOOL	bMapWpt;
	BOOL	bMapAirspaces;

	DeleteArrayOfLocs(m_ptMapLocs);					// delete contents of old arrays
	DeleteArrayOfLocs(m_ptMapNavs);
	DeleteArrayOfAirspaces(m_ptMapAirspaces);
						
	bMapWpt = this->SearchMapLocs (m_ptMapLocs, m_ptMapNavs);	
	bMapAirspaces = this->SearchMapAirspaces(m_ptMapAirspaces);

	CMenu Menu;
	if (Menu.LoadMenu(IDR_MAP))
		m_ptMapMenu = Menu.GetSubMenu(0);


	if (m_ptMapMenu != NULL)
	{
		short i;
		for (i=0; i<FLAG_CNT; i++)
			CheckSymbolsMenu (ID_ME_AIRPORT+i, m_ptMapDoc->IsActivated (SHOW_APT+i));

		if (bMapWpt)
			this->NewMapLocsMenu (m_ptMapMenu, 7, 
						ID_ME_LOCEDIT_FIRST, ID_ME_LOCEDIT_LAST, m_ptMapLocs);

	//	if (bMapAirspaces)
			this->NewMapAirspacesMenu (m_ptMapMenu, 8, 
						ID_ME_AIRSPACEEDIT_FIRST, ID_ME_AIRSPACEEDIT_LAST, m_ptMapAirspaces);

		m_ptMapMenu->TrackPopupMenu(TPM_LEFTALIGN | TPM_RIGHTBUTTON, 
										point.x, point.y, m_ptWnd);
	}
}