// LaLoList.cpp : implementation of the CLaLoList class
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "pf.h"
#include "math.h"		// fabs

#include "InitDoc.h"	 //includes "LaLoList.h"
#include "DimDoc.h"

#include "LocDoc.h"
#include "RegioDlg.h"				// dialog to set regions to use

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

extern CInitDoc* 	ptInit;              

 
/************************************************************************
 *  LaLoList.cpp				L a L o C m p							*
 ************************************************************************/
int LaLoCmp (void FAR* elem1, void FAR* elem2)
{
int	cmpVal;
float	p_1, p_2;

p_1 = ((LALOINDEXTYPE FAR*)elem1)->fLat;
p_2 = ((LALOINDEXTYPE FAR*)elem2)->fLat;
if (p_1 == p_2)
	{
	p_1 = ((LALOINDEXTYPE FAR*)elem1)->fLon;
	p_2 = ((LALOINDEXTYPE FAR*)elem2)->fLon;
	if (p_1 == p_2)
		{
		cmpVal = 0;
		}
	else{
		cmpVal = (p_1 < p_2)? -1 : 1;
		}
	}
else{
	cmpVal = (p_1 < p_2)? -1 : 1;
	}

return cmpVal;
} /* End of LaLoCmp() */ 
 


/////////////////////////////////////////////////////////////////////////////
// CLaLoList
IMPLEMENT_SERIAL(CLaLoList, CObject, 0 /* schema number*/ )

/////////////////////////////////////////////////////////////////////////////
// CLaLoList construction/destruction

CLaLoList::CLaLoList()
{
	// TODO: add one-time construction code here  
m_ptRegDrawn		= NULL;
m_ptLaLoList		= NULL;
m_lLaLoListMax		= 0; 
m_lLaLoListIndex	= 0;

m_nAddRegionIndex	= -1; 
m_szActRegion		= "";
m_ptDoc = 0; 
m_EPS_wi = (float)0.01;		/* Genauigkeit zum Winkelvergleich	*/

}

CLaLoList::~CLaLoList()
{   
this->Free();
}


/************************************************************************
 *  LaLoList.cpp					A l l o c							*
 ************************************************************************/
BOOL CLaLoList::Alloc (long lCnt)
{
BOOL bOK = TRUE;

this->Free();

if (lCnt > 0)
	{
	m_ptLaLoList = new LALOINDEXTYPE[lCnt];   
	if (m_ptLaLoList != NULL)  
		{
		m_lLaLoListMax = lCnt;  
		m_lLaLoListIndex = 0;
		}    
	else{
		bOK = FALSE;
		}
	}

return bOK;
}

/************************************************************************
 *  LaLoList.cpp					F r e e								*
 ************************************************************************/
void CLaLoList::Free ()
{
if (m_lLaLoListMax > 0)
	{
	delete [] m_ptLaLoList;
	m_ptLaLoList 	= NULL;
	m_lLaLoListMax  = 0; 
	m_lLaLoListIndex = 0;
	}

if (m_ptRegDrawn != NULL)
	{
	delete [] m_ptRegDrawn;
	m_ptRegDrawn = NULL;
	}
}


/************************************************************************
 *  LaLoList.cpp				S e t L a L o E n t r y					*
 ************************************************************************/
void CLaLoList::SetLaLoEntry (LALOINDEXTYPE* ptSL, float fLat, float fLon,
					short nCategory, short nIndex, short nRegIndex)
{
ptSL->fLat			= fLat;
ptSL->fLon			= fLon;
ptSL->nCategory		= nCategory;
ptSL->nIndex		= nIndex;
ptSL->nRegionIndex	= nRegIndex;
}

/************************************************************************
 *  LaLoList.cpp				A d d E n t r y							*
 ************************************************************************/
void CLaLoList::AddEntry (CLocation* ptLocation, short nSubListIndex)
{
if (m_nAddRegionIndex != -1)
	{
	LALOINDEXTYPE* ptEntry;

	float fLat = (float)ptLocation->GetLat(); 
	float fLon = (float)ptLocation->GetLon(); 
	short nCategory = ptLocation->GetCategory();

	if (m_lLaLoListIndex < m_lLaLoListMax)
		{
		if (m_ptLaLoList != NULL)
			{
			if (fLat != NO_KOORD && fLon != NO_KOORD)
				{
				ptEntry = m_ptLaLoList + m_lLaLoListIndex;     
				this->SetLaLoEntry (ptEntry, fLat, fLon,
											   nCategory,
											   nSubListIndex,
											   m_nAddRegionIndex);
				m_lLaLoListIndex++;
				}
			}
		}
	else{
		AfxMessageBox ("LaLoList-Array limit reached");
		}
	}
}

/************************************************************************
 *  LaLoList.cpp					S o r t								*
 ************************************************************************/
void CLaLoList::Sort ()
{
if (m_lLaLoListIndex > 1 )
    {
     if (m_ptLaLoList != NULL)
		{
		lqsort((void FAR*)m_ptLaLoList, m_lLaLoListIndex,
		       (unsigned)sizeof (LALOINDEXTYPE), LaLoCmp);
		}
    }
}   


/************************************************************************
 *  LaLoList.cpp		    	G e t I n d e x R a n g e				*
 ************************************************************************/
BOOL CLaLoList::GetIndexRange (float fLat, float fLon, long* ptFirst, long* ptLast)
{
BOOL bFound = FALSE;
BOOL bStop = FALSE;
long lCnt = this->GetSize();
long i;

for (i=0; i<lCnt && !bStop; i++)
	{
	LALOINDEXTYPE* ptEntry=NULL;
	ptEntry = this->GetEntryPtr (i);
	if (ptEntry != NULL)
		{
		if (fabs (fLat - ptEntry->fLat) <= m_EPS_wi)
			{									/* Latitude OK	*/
			if (fabs (fLon - ptEntry->fLon) <= m_EPS_wi)
				{								/* Longitude OK	*/
				if (!bFound)
					{
					bFound = TRUE;
					*ptFirst = *ptLast = i;
					}
				else{
					*ptLast = i;		
					}
				}
			else{
				if (bFound)
					bStop = TRUE;
				}
			}
		}
	}
return bFound;
}

/************************************************************************
 *  LaLoList.cpp	  		 L a t L o n C m p							*
 ************************************************************************/
short CLaLoList::LatLonCmp (float fLat1, float fLon1, 
							float fLat2, float fLon2)
{
short nCmp;

float fLatDiff = fLat1 - fLat2;
if (fabs (fLatDiff) <= m_EPS_wi)
	{									/* Latitude OK	*/
	float fLonDiff = fLon1 - fLon2;
	if (fabs (fLonDiff) <= m_EPS_wi)
		{								/* Longitude OK	*/
		nCmp = 0;
		}
	else{			// Lat in limit, now check exact:
		if (fLatDiff == 0)	nCmp = (fLonDiff < 0)? -1 : 1; 
					else	nCmp = (fLatDiff < 0)? -1 : 1; 
		}
	}
else{	// bad Lat, check Lat:
	nCmp = (fLatDiff < 0)? -1 : 1; 
	}

return nCmp;
}

/************************************************************************
 *  LaLoList.cpp	  		 G e t S o r t e d I n d e x 				*
 ************************************************************************/
BOOL CLaLoList::GetSortedInd (float fLat, float fLon, long* ptIndex)
{
BOOL	bFound = FALSE;
BOOL	bUpperIntervall; 
long	lMinIndex, lMaxIndex, i, OldTestInd;

lMinIndex  = 0;
lMaxIndex  = this->GetSize();			/* legal ind: 0...cnt-1		*/
i = (lMinIndex + lMaxIndex) / 2;		/* i always < cnt !!		*/

if (lMaxIndex > 0)
	{
	int iCmp=1;

	do	{  
		LALOINDEXTYPE* ptEntry=NULL;
		ptEntry = this->GetEntryPtr (i);
		if (ptEntry != NULL)
			{
			iCmp = LatLonCmp (fLat, fLon, ptEntry->fLat, ptEntry->fLon);

			if (iCmp >=0)	bUpperIntervall = TRUE;
	    			else	bUpperIntervall = FALSE;

			if (bUpperIntervall)	lMinIndex = i;
							else	lMaxIndex = i;
					
			OldTestInd = i;
			i = (lMinIndex + lMaxIndex) / 2;
			}
		} while (i != OldTestInd);

	if (iCmp == 0)
		{
		*ptIndex = i;
		bFound = TRUE;
		}
	}                     
   
return bFound;
}


/************************************************************************
 *  LaLoList.cpp	  		 G e t S t a r t I n d e x 					*
 *  Purpose: Returns index of entry for lat lon values equal or larger	*
 *			 than lower left map area									*
 ************************************************************************/
BOOL CLaLoList::GetStartIndex (DRECT& rLatLon, long* ptIndex)
{
BOOL	bFound = FALSE;
LALOINDEXTYPE* ptEntry=NULL;
long	i;
long lCnt = this->GetSize();
 
for (i=0; i<lCnt && !bFound; i++)
	{
	ptEntry = NULL;
	ptEntry = this->GetEntryPtr (i);
	if (ptEntry != NULL)
		{
		if (CLatLon::IsLatLonInRect (ptEntry->fLat, ptEntry->fLon, rLatLon))			
			{  
			*ptIndex = i;
			bFound = TRUE;
			}
		}
	}

return bFound;
}

/************************************************************************
 *  LaLoList.cpp	  		 G e t E n d I n d e x 						*
 *  Purpose: Returns index of entry for lat lon values equal or larger	*
 *			 than upper right map area									*
 ************************************************************************/
BOOL CLaLoList::GetEndIndex (DRECT& rLatLon, long* ptIndex)
{
BOOL	bFound = FALSE;
LALOINDEXTYPE* ptEntry=NULL;
long	i;
long lCnt = this->GetSize();

for (i=lCnt-1; i>=0 && !bFound; i--)
	{
	ptEntry = NULL;
	ptEntry = this->GetEntryPtr (i);
	if (ptEntry != NULL)
		{
		if (CLatLon::IsLatLonInRect (ptEntry->fLat, ptEntry->fLon, rLatLon))			
			{  
			*ptIndex = i;
			bFound = TRUE;
			}
		}
	}

return bFound;
}

/****************************************************************************
 *	LaLoList.cpp				I n d e x T o R a n g e						*
 ****************************************************************************/
void CLaLoList::IndexToRange (long* ptFirstInd, long lIndex, long* ptLastInd)
{
BOOL bSame;
LALOINDEXTYPE* ptEntry=NULL;
short nCmp;
long	j;						
float fLat, fLon;
long lCnt = this->GetSize();

*ptFirstInd = *ptLastInd = lIndex;

ptEntry = this->GetEntryPtr (lIndex);
if (ptEntry != NULL)
	{
	fLat = ptEntry->fLat;
	fLon = ptEntry->fLon;
	
	bSame = TRUE;				// get first index with same lat lon	
	for (j=lIndex-1; j>=0 && bSame; j--)
		{
		ptEntry = this->GetEntryPtr (j);
		if (ptEntry != NULL)
			{
			nCmp = LatLonCmp (fLat, fLon, ptEntry->fLat, ptEntry->fLon);
			bSame = (nCmp == 0);
			if (bSame) (*ptFirstInd)--;
			}
		}
	
	bSame = TRUE;				// get last index with same lat lon	
	for (j=lIndex+1; j<lCnt && bSame; j++)
		{
		ptEntry = this->GetEntryPtr (j);
		if (ptEntry != NULL)
			{
			nCmp = LatLonCmp (fLat, fLon, ptEntry->fLat, ptEntry->fLon);
			bSame = (nCmp == 0);
			if (bSame) (*ptLastInd)++;
			}
		}
	}
}


/************************************************************************
 *  LaLoList.cpp		    G e t S o r t e d I n d e x R a n g e		*
 ************************************************************************/
BOOL CLaLoList::GetSortedIndexRange (float fLat, float fLon, long* ptFirst, long* ptLast)
{
BOOL bFound = FALSE;
long lIndex;
if (this->GetSortedInd (fLat, fLon, &lIndex))
	{
	this->IndexToRange (ptFirst, lIndex, ptLast);	
	bFound = TRUE;
	}
return bFound;
}


/************************************************************************
 *  LaLoList.cpp		    		G e t L o c							*
 ************************************************************************/
BOOL CLaLoList::GetLoc (CLocation* ptLoc, long lIndex, short* ptActRegInd)
{
BOOL	bOK = FALSE;
long lCnt = this->GetSize();

if (lIndex >=0 && lIndex < lCnt)
	{
	LALOINDEXTYPE* ptEntry=NULL;
	ptEntry = this->GetEntryPtr (lIndex);
	if (ptEntry != NULL)
		{
		*ptActRegInd = ptEntry->nRegionIndex;

		if (*ptActRegInd >=0 && *ptActRegInd < m_ptRegions->GetSize())
			{
			if (m_ptDoc->GetRegionIndex() != *ptActRegInd)
				{			// load required region
				this->LoadRegion (*ptActRegInd);
				}
			*ptLoc = m_ptDoc->GetLocation (ptEntry->nCategory, ptEntry->nIndex);
			ptLoc->SetRegionName(m_szActRegion);	// to be transferred into KoordDlg
			bOK = (ptLoc != NULL);
			}
		}
	}
return bOK;
}


/************************************************************************
 *  LaLoList.cpp		    	L o a d R e g i o n 					*
 ************************************************************************/
BOOL CLaLoList::LoadRegion (short nRegIndex)
{
	BOOL bLoaded = FALSE;
	CRegion* ptRegion = (CRegion*)m_ptRegions->GetAt(nRegIndex);
	if (ptRegion != NULL)
	{
		m_szActRegion = ptRegion->GetName();
		m_ptDoc->OnNewDocument(); 			// deletes old structures   
		m_ptDoc->SetRegionIndex (nRegIndex);
		m_ptDoc->SetPathName(m_szLocPath + m_szActRegion + (CString)".krd");
		bLoaded = m_ptDoc->OnOpenDocument (m_ptDoc->GetPathName());
	}

	return bLoaded;
}

/************************************************************************
 *  LaLoList.cpp		    S e t A c t R e g I n d e x 				*
 *  Unused
 ************************************************************************/
short CLaLoList::SetActRegIndex (CLocation& Loc)
{
BOOL bFound = FALSE;
float	fLat;
float	fLon;
long lFirstIndex, lLastIndex;
short	nActRegIndex = -1;

fLat = (float)Loc.GetLat();
fLon = (float)Loc.GetLon();

if (this->GetIndexRange (fLat, fLon, &lFirstIndex, &lLastIndex))
	{
	long i;

	for (i=lFirstIndex; i<=lLastIndex && !bFound; i++)
		{
		CLocation FoundLoc;
		this->GetLoc (&FoundLoc, i, &nActRegIndex);	// sets nActRegIndex!!
		bFound = Loc.IsEqual (FoundLoc);
		}
	}
return nActRegIndex;
}

/************************************************************************
 *  LaLoList.cpp		   P r e p a r e G e t L o c					*
 *  Purpose: Used to initialize GetLocationInRect						*
 ************************************************************************/
BOOL CLaLoList::PrepareGetLoc (DRECT& rLatLon)
{
BOOL bOK, bStart, bEnd;

bOK = FALSE;
bStart = this->GetStartIndex (rLatLon, &m_lStartIndex);
bEnd = this->GetEndIndex (rLatLon, &m_lEndIndex);

								// store initial region index
m_nActRegion = m_ptDoc->GetRegionIndex();

m_bInitRegionUsed = FALSE;		// used to decide wether to switch back 
								// to initial region or not


if (bStart && bEnd)
	{
	short nRegCnt = m_ptRegions->GetSize();
	if (m_ptRegDrawn != NULL)
		{
		delete [] m_ptRegDrawn;
		m_ptRegDrawn = NULL;
		}
	m_ptRegDrawn = new BOOL[nRegCnt];
	memset (m_ptRegDrawn, 0, nRegCnt*sizeof (BOOL));

	m_bSearchNewRegion = TRUE;
	m_lNextIndex = m_lStartIndex;
	bOK = TRUE;
	}
return bOK;
}


/************************************************************************
 *  LaLoList.cpp		    G e t L o c a t i o n I n R e c t			*
 *  Purpose: To find locations for map drawing.							*
 *	Call PrepareGetLoc to initialize:									*
 *		m_lStartIndex, m_lEndIndex: Range of visible entries			*
 *		m_ptRegDrawn: Array of BOOL to mark a region as drawn			*
 *		m_bSearchNewRegion: Cleared to switch to a new region			*
 *		m_lNextIndex: Next index of actual loop							*
 *  Um einen extrem hufigen Wechsel zwischen den Regionen zu vermeiden,*
 *  wird die Liste der Orte, die auf der Karte sichtbar sind, N mal		*
 *  durchlaufen. Bei jedem Durchgang werden nur Orte aus e i n e r		*
 *  Region gezeichnet. Sobld alle Orte aus der aktuellen Region in die	*
 *  Karte eingetragen sind, beginnt ein neuer Durchgang. Wenn dabei 	*
 *  Orte aus Regionen gefunden werden, die schon gezeichnet wurden,		*
 *  mu ein erneutes Zeichnen verhindert werden. Dazu dient das BOOL	*
 *  Array, das fr jede mgliche Region einen Eintrag anbietet.	Sobald	*
 *	ein Ort aus einer Region gefunden wird, die noch nicht gezeichnet	*
 *  wurde, wird die Region gewechselt. Diese Aktion ist mit einem Datei-*
 *  zugriff verbunden. Die nchste Region wird also immer erst dann		*
 *  aktiviert, wenn alle notwendigen Eintrge aus der aktuellen Region	*
 *  entnommen wurden. Damit wird die Anzahl der Dateizugriffe minimiert!*
 ************************************************************************/
BOOL CLaLoList::GetLocationInRect (CLocation* ptLoc, DRECT& rLatLon, 
						short* ptActRegInd)
{
BOOL bFound = FALSE;

do	{
	long	i;
	for (i=m_lNextIndex; i<=m_lEndIndex && !bFound; i++)
		{									// for all enries around map area
		LALOINDEXTYPE* ptEntry=NULL;
		ptEntry = this->GetEntryPtr (i);
		if (ptEntry != NULL)
			{
			if (CLatLon::IsLatLonInRect (ptEntry->fLat, ptEntry->fLon, rLatLon))
				{							// new Entry in map area
				if (m_bSearchNewRegion)
					{						// cleared to draw new region
					if (*(m_ptRegDrawn + ptEntry->nRegionIndex) == FALSE)
						{					// new region was not drawn
						*(m_ptRegDrawn + ptEntry->nRegionIndex) = TRUE;
						*ptActRegInd = ptEntry->nRegionIndex;
						m_bSearchNewRegion = FALSE;
						}
					}

				if (!m_bSearchNewRegion)
					{			
					if (ptEntry->nRegionIndex == *ptActRegInd)
						{
						bFound = this->GetLoc (ptLoc, i, ptActRegInd);
						m_lNextIndex = i+1;
						}
					}
				}
			}
		}


	if (i>m_lEndIndex)
		{				// all available entries checked in act region
		if (m_bSearchNewRegion)
			{			// no undrawn entry found, stop loop
			if (m_ptRegDrawn != NULL)
				{
				delete [] m_ptRegDrawn;
				m_ptRegDrawn = NULL;
				}

			m_bSearchNewRegion = FALSE;
			}
		else{			// undrawn entry found, try to find next undrawn region
			m_bSearchNewRegion = TRUE;
			m_lNextIndex = m_lStartIndex;
			}
		}
	} while (m_bSearchNewRegion && !bFound);  // repeat for every region found


	if (bFound)
	{	// check if we can reactivate the initial region 
		// after looking for all locations in rLatLon
		if (*ptActRegInd == m_nActRegion)
			m_bInitRegionUsed = TRUE;
	}
	else
	{	// if initial region was visible in rLatLon, 
		// switch back to initially used region
		if (m_bInitRegionUsed)
			this->LoadRegion (m_nActRegion);
	}

return bFound;
}


/////////////////////////////////////////////////////////////////////////////
// CLaLoList commands
