/************************************************************************
 *  						L o c a t i o n . c p p	  					*
 ************************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "math.h"							// for: fabs

#include "resource.h"			// IDF_LATLON
#include "LatLon.h"
#include "DimDoc.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CLatLon

extern CDimDoc* ptDim;

IMPLEMENT_SERIAL(CLatLon, CObject, 0 /* schema number*/ )

#define pi 3.1415926

CLatLon::CLatLon()
{ 
m_dLat = NO_KOORD;	 
m_dLon = NO_KOORD;  
}

CLatLon::CLatLon(const CLatLon& Source)
{ 
m_dLat = Source.GetLat();
m_dLon = Source.GetLon();
}

CLatLon::CLatLon(double dLat, double dLon)
{ 
m_dLat = dLat;
m_dLon = dLon;
}



CLatLon::~CLatLon()
{
} 

/************************************************************************
 *  LatLon.cpp				operator=								*
 ************************************************************************/
const CLatLon& CLatLon::operator=(const CLatLon& Source)
{
m_dLat = Source.m_dLat;
m_dLon = Source.m_dLon;
return *this;
}

/************************************************************************
 *  LatLon.cpp					S e t L a t L o n						*
 ************************************************************************/
void CLatLon::SetLatLon(double dLat, double dLon)
{ 
m_dLat = dLat;
m_dLon = dLon;
}

/************************************************************************
 *  LatLon.cpp				C r e a t e L a t L o n S t r				*
 ************************************************************************/
void CLatLon::CreateLatLonStr(CString* ptLatLon, short nDest)
{
	short	Lat_G, Lat_M, Lat_D;
	short	Lon_G, Lon_M, Lon_D;
	double	fLat_M, fLon_M;

	double dLat = m_dLat;
	double dLon = m_dLon;

	Lat_D = 'N';
	if (dLat < 0.)	{ Lat_D = 'S'; dLat *= -1; }
	Lat_G = (short)dLat;
	fLat_M = (dLat - Lat_G)*60;
	Lat_M = (short)(fLat_M + 0.5 );

	if (nDest == LATLON_STAT)	
	{
		if (59.95 < fLat_M && fLat_M < 60.) 
			{Lat_G++; fLat_M=0;}
	}
	else
	{
		if (Lat_M == 60) 
			{Lat_G++; Lat_M=0;}	
	}

	Lon_D = 'E';

	if (dLon < 0.)	{ Lon_D = 'W'; dLon *= -1; }
	Lon_G = (short)dLon;
	fLon_M = (dLon - Lon_G)*60;
	Lon_M = (short)( fLon_M + 0.5 );

	if (nDest == LATLON_STAT)	
	{
		if (59.95 < fLon_M && fLon_M < 60.) 
			{Lon_G++; fLon_M = 0;}
	}
	else
	{
		if (Lon_M == 60) 
			{Lon_G++; Lon_M=0;}
	}

	CString szFormat;
	char	szBuffer[256]; 
	
	switch (nDest)
	{
	case LATLON_DDMM:
		szFormat.LoadString (IDF_LATLON);
		sprintf(szBuffer, (LPCTSTR)szFormat, Lat_G, Lat_M, Lat_D,
										Lon_G, Lon_M, Lon_D);
		break;
	case LATLON_FLPL:
		szFormat.LoadString (IDF_FLPL_LATLON);
		sprintf(szBuffer, (LPCTSTR)szFormat, Lat_G, Lat_M, Lat_D,
										Lon_G, Lon_M, Lon_D);
		break;
	case LATLON_STAT:
		szFormat.LoadString (IDF_STATUS_LATLON);
		sprintf(szBuffer, (LPCTSTR)szFormat, Lat_G, fLat_M, Lat_D,
										Lon_G, fLon_M, Lon_D);
		break;
	}

	*ptLatLon = szBuffer;
}

/************************************************************************
 *  LatLon.cpp					I s V a l i d							*
 ************************************************************************/
BOOL CLatLon::IsValid ()
{
BOOL	bValid = FALSE;     
					 
bValid = (	(m_dLat >= -90. && m_dLat <= 90.) &&
			(m_dLon >= -180.&& m_dLon <= 180.) ); 
return bValid;
}


/************************************************************************
 *  LatLon.cpp				I s N e a r								*
 ************************************************************************/
BOOL CLatLon::IsNear(CLatLon& TestLocation)
{
BOOL	bNear = FALSE;

if (fabs(TestLocation.GetLat() - this->GetLat()) < 1)
    {			    /* Breitengrad-Differenz < 60 NM	    */
    double  Lat_Average;
    Lat_Average = pi * (TestLocation.GetLat() + this->GetLat()) / 360;
    if (fabs(TestLocation.GetLon() - this->GetLon()) * cos(Lat_Average) < 1)
		{		    /* Lngengrad-Differenz < 60 NM	    */
		bNear = TRUE;
		}
    }
return bNear;
}


/************************************************************************
 *  LatLon.cpp				I s E q u a l								*
 ************************************************************************/
BOOL CLatLon::IsEqual(CLatLon& TestLocation)
{
short	nDeg, nTestDeg;
short	nMin, nTestMin;
short	nSec, nTestSec;
short	nDir, nTestDir;


						// compare latitudes
this->AngleToDisplay (this->GetLat(),		TRUE, &nDeg, &nMin, &nSec, &nDir);
this->AngleToDisplay (TestLocation.GetLat(),TRUE, &nTestDeg, &nTestMin, &nTestSec, &nTestDir);
if (nDeg != nTestDeg) return FALSE;
if (nMin != nTestMin) return FALSE;
if (nSec != nTestSec) return FALSE;
if (nDir != nTestDir) return FALSE;

						// compare longitudes
this->AngleToDisplay (this->GetLon(),		FALSE, &nDeg, &nMin, &nSec, &nDir);
this->AngleToDisplay (TestLocation.GetLon(),FALSE, &nTestDeg, &nTestMin, &nTestSec, &nTestDir);
if (nDeg != nTestDeg) return FALSE;
if (nMin != nTestMin) return FALSE;
if (nSec != nTestSec) return FALSE;
if (nDir != nTestDir) return FALSE;
	   
return TRUE;
}


/************************************************************************
 *  LatLon.cpp			RadiusToDRECT				*
 ************************************************************************/
DRECT CLatLon::RadiusToDRECT (double dLat, double dLon, double dDegrees)
{						// if nDegrees == 1:
DRECT rLatLon;			// rect around ActLoc, "radius" 60 NM
double rLat = pi * dLat / 360;	// angle in rad
double dLatRadius = dDegrees / cos(rLat);	// cos (60): 0.5
											// 2 degrees = 60 NM
rLatLon.left	= dLon - dLatRadius;
rLatLon.right	= dLon + dLatRadius;
rLatLon.top		= dLat + dDegrees;	// 1 degree = 60 NM
rLatLon.bottom	= dLat - dDegrees;
return rLatLon;
}

/************************************************************************
 *  LatLon.cpp				W i R a d Y 								*
 *	Return : Strecke vom Aequator zum betreffenden Breitengrad [Rad]	*
 ************************************************************************/
double CLatLon::WiRadY ()
{
double	help;
help = tan (m_dLat*pi/360. + pi/4); 	// for x>0, tan(x) is negative
help = log (fabs(help));           // therefore use fabs...
return help;
}

/************************************************************************
 *  Elev.cpp    				Y t o L a t	 							*
 ************************************************************************/
double CLatLon::YtoLat (double dY)
{
	return 360. * (atan (exp(dY)) - pi/4) / pi;
}

/************************************************************************
 *  Elev.cpp    				X t o L o n 							*
 ************************************************************************/
double CLatLon::XtoLon(double dX)
{
	return dX * 180./pi;
}

/************************************************************************
 *  LatLon.cpp					L o x o D i s t							*
 *		    Entfernung ueber Kursgleiche								*
 ************************************************************************/
double CLatLon::LoxoDist (double dLat, double dLon, double* ptCourse)
{
CLatLon Dest (dLat, dLon);
return this->LoxoDist (Dest, ptCourse);
}

/************************************************************************
 *  Location.cpp			F l i g h t A n g l e						*
 *  Same as in CBorder class											*
 ************************************************************************/
double CLatLon::FlightAngle (double A, double B)
{
double angle;

if (A < 0.) A += 360.;
if (B < 0.) B += 360.;
angle = B - A;
if (angle > 180.) angle -= 360.;
if (angle <-180.) angle += 360.;

return angle;
}

/************************************************************************
 *  LaLoList.cpp			  G e t M a p L o n 						*
 ************************************************************************/
BOOL CLatLon::GetMapLon(double dLonLeft, double dLon, double dLonRight, double* ptMapLon)
{
	BOOL	bLonOK = (dLonLeft<=dLon && dLon<=dLonRight);

	if (!bLonOK)
	{
		BOOL	bAntipod = (fabs(dLonLeft) > 180 || fabs(dLonRight) > 180);

		if (bAntipod)
		{
			double dTestLon = dLon+360;
			bLonOK = (dLonLeft<=dTestLon && dTestLon<=dLonRight);
			if (!bLonOK)
			{
				dTestLon = dLon-360;
				bLonOK = (dLonLeft<=dTestLon && dTestLon<=dLonRight);
			}

			dLon = dTestLon;
		}
	}

	*ptMapLon = dLon;
	return bLonOK;
}

/************************************************************************
 *  LaLoList.cpp		  I s L a t L o n I n R e c t 					*
 ************************************************************************/
BOOL CLatLon::IsLatLonInRect (double dLat, double dLon, DRECT& rLatLon, double* ptMapLon)
{
	BOOL bInRect = FALSE;
	
	double dMapLon;
	BOOL bLonOK = CLatLon::GetMapLon (rLatLon.left, dLon, rLatLon.right, &dMapLon);
	if (ptMapLon != NULL)
		*ptMapLon = dMapLon;

	if (bLonOK &&
		rLatLon.bottom<=dLat && dLat<=rLatLon.top)
		bInRect = TRUE;

return bInRect;
}




/************************************************************************
 *  LatLon.cpp					L o x o D i s t							*
 *		    Entfernung ueber Kursgleiche								*
 ************************************************************************/
double CLatLon::LoxoDist (CLatLon& Dest, double* ptCourse)
{
double DIST;
double NewLat, dLon, dWest, dCourse;

dLon = this->FlightAngle(m_dLon, Dest.GetLon());
if (fabs(dLon) < 1E-8) dLon = 0.;		// wg. Rundungsfehler	

NewLat = Dest.GetLat();

if (dLon == 0.)
    {
    dCourse = (m_dLat > NewLat)? DB180 : 0.;
    }
else{
    if (m_dLat != NewLat)	/* Kurs auf Loxodrome		*/
		{
		dCourse = atan ( dLon * pi/DB180	/
			(Dest.WiRadY () - this->WiRadY ()))  * DB180/pi;
		if (dCourse < 0.) dCourse += DB180;
		}
    else{
		dCourse = 90.;
		}					
    }

dWest = -dLon;			
if (dWest <= 0.) dWest += DB360;	/* Winkeldiff. in westl. Richtg.*/
if (dWest < DB180) dCourse += DB180;	/* Soll-Kurs von 180 - 360	*/

							/* Entfernung auf Loxodrome	*/
if (m_dLat != NewLat)		/* benutzt nicht gerundeten Kurs*/
		DIST = 60/cos (dCourse * pi/DB180) * (NewLat - m_dLat);
else	DIST = 60 * fabs(dLon) * cos (m_dLat * pi/DB180);

dCourse = floor (dCourse + 0.5);		/* Kurs auf- oder abrunden	*/

if (DIST < 0)
	{										
	DIST *= -1;
	dCourse += 180;
	if (dCourse > 360) dCourse -= 360;
	}

if (ptCourse != NULL)
	*ptCourse = dCourse;

return DIST;				/* Distanz nicht gerundet	*/
}

/****************************************************************************
 *	ElevView.cpp			PointOnLoxoLegAtDist							*
 *  Purpose: Calculates LatLon for point on leg, dDist NM after LL1			*
 ****************************************************************************/
CLatLon CLatLon::PointOnLoxoLegAtDist (double dDist, CLatLon& LL1, CLatLon& LL2)
{
	CLatLon LL;
		
	double dLon = CLatLon::FlightAngle(LL1.GetLon(), LL2.GetLon());
	if (fabs(dLon) < 1E-8) dLon = 0.;		// wg. Rundungsfehler	

	double X1 = LL1.GetLon() * pi/180;
	double X2 = (LL1.GetLon()+ dLon) * pi/180;

	double dDist12 = LL1.LoxoDist(LL2);

	if (dDist12 != 0)
	{
		double dX = dDist * (X2 - X1) / dDist12;
		dX = X1 + dX;

		double dY = dDist * (LL2.WiRadY() - LL1.WiRadY()) / dDist12;
		dY = LL1.WiRadY() + dY;
		
		LL.SetLat (CLatLon::YtoLat (dY));
		LL.SetLon (CLatLon::XtoLon (dX));
	}
	else
	{
		LL = LL1;
	}

	return LL;
}
                

/****************************************************************************
 *	ElevView.cpp			PointOnOrthoLegAtDist							*
 *  Purpose: Calculates LatLon for point on leg, dDist NM after LL1			*
 ****************************************************************************/
CLatLon CLatLon::PointOnOrthoLegAtDist (double dDist, CLatLon& LL1, CLatLon& LL2)
{
	CLatLon LLcrash;
	CVektor A, B, X;
	double dAngleAX;

	LL1.ToXYZ(&A);				/* Koordinaten von Ort A		*/
	LL2.ToXYZ(&B);				/* Koordinaten von Ort B		*/

	dAngleAX = (pi/180) * dDist/60;

	X = A.GrossKreis (B, dAngleAX);	/* X : Ort nach Winkel dAngleAX	*/ 

	LLcrash.FromXYZ (X);
	return LLcrash;
}

                  
/************************************************************************
 *  LatLon.cpp				A n g l e T o D M S							*
 ************************************************************************/
void CLatLon::AngleToDMS (double angle, BOOL bLat,
		 short* ptDegr, short* ptMin, short* ptSec, short* ptDir)
{
short	Grad, Min, Sec, Dir;

if (angle < 0.)	Dir = (bLat)? 'S': 'W';
		else	Dir = (bLat)? 'N': 'E';

if (angle < 0.) angle *=-1;
Grad	= (short)angle;
Min	= (short)( (angle - Grad)*60 );

Sec = (short)( ((angle - Grad)*60 - Min)*60 + 0.5 );
if (Sec == 60) { Min++; Sec=0;}
if (Min == 60) {Grad++; Min=0;}

*ptDegr = Grad;
*ptMin = Min;
*ptSec = Sec;
*ptDir = Dir;
}

/************************************************************************
 *  LatLon.cpp				A n g l e T o D i s p l a y					*
 ************************************************************************/
void CLatLon::AngleToDisplay (double angle, BOOL bLat,
		 short* ptDegr, short* ptCentMin, short* ptSec, short* ptDir)
{
short	Grad, Min, Sec, Dir;
LATLONFORM LatLonForm = ptDim->GetLatLonFormat();

if (angle < 0.)	Dir = (bLat)? 'S': 'W';
		else	Dir = (bLat)? 'N': 'E';

if (angle < 0.) angle *=-1;
Grad	= (short)angle;

if (LatLonForm == LLF_GGGMMSS)
	{
	Min	= (short)( (angle - Grad)*60 );

	Sec = (short)( ((angle - Grad)*60 - Min)*60 + 0.5 );
	if (Sec == 60) { Min++; Sec=0;}
	if (Min == 60) {Grad++; Min=0;}

	Min *= 100;						// convert to 1/100 minutes
	}
else{		// degree, 1/100 minutes
	Min	= (short)( (angle - Grad)*60 * 100 + 0.5);	// 1/100 minutes
	if (Min == 60 *100) {Grad++; Min=0;}

	Sec = 0;
	}

*ptDegr		= Grad;
*ptCentMin	= Min;
*ptSec		= Sec;
*ptDir		= Dir;
}	


/************************************************************************
 *  LatLon.cpp			      T o X Y Z									*
 *	Eingabe :	Breitengrad in Grad (suedliche Breite negativ!)			*
 *				Laengengrad in Grad (westliche Lnge negativ!)			*
 *	Ausgabe : Koordinaten des Ortes [CVektor]						*
 ************************************************************************/
void CLatLon::ToXYZ (CVektor* ptOrt)
{
double rLat, rLon;		/* Breiten- und Laengengrade [rad]	*/

rLat = m_dLat * pi/DB180;
rLon = m_dLon * pi/DB180;
ptOrt->x = cos(rLat) * sin(rLon);
ptOrt->y = cos(rLat) * cos(rLon);
ptOrt->z = sin(rLat); 
}


/************************************************************************
 *  LatLon.cpp			      F r o m X Y Z								*
 *	Eingabe : Koordinaten des Ortes [CVektor&]							*
 *	Ausgabe : Breitengrad in Grad (suedliche Breite negativ!)			*
 *		  Laengengrad in Grad (westliche Lnge negativ!)				*
 ************************************************************************/
void CLatLon::FromXYZ (CVektor& vLoc)
{
double rLat, rLon;		/* Breiten- und Laengengrade [rad]	*/ 

rLat = asin(vLoc.z);
if (fabs(vLoc.z) < 1.)
	{
	double help;
	help = vLoc.y/cos(rLat);
	if (help > 1.) help = 1.;
	if (help < -1.) help = -1.;
	rLon = acos(help);
	}
else{				/* Nord- oder Suedpol, Lat: undefiniert	*/
	rLon = 0.;
	}
if(vLoc.x < 0.) rLon *= -1;

m_dLat = rLat * DB180/pi;
m_dLon = rLon * DB180/pi;
}

/************************************************************************
 *  LatLon.cpp				D i s t a n c e								*
 *		kuerzeste Entfernung ueber Grosskreis							*
 ************************************************************************/
double CLatLon::Distance (CLatLon& Loc, double* ptKurs)
{
CVektor A,B, X, NordPol;		/* euklidische Koordinaten	*/
CVektor N1, N2;
double wab, DIST, dWest;

DIST = 0.; *ptKurs=0.;			/* Ort A = Ort B				*/
if (m_dLat == Loc.GetLat() &&
    m_dLon == Loc.GetLon()) return DIST;

this->ToXYZ(&A);				/* Koordinaten von Ort A		*/
Loc.ToXYZ(&B);					/* Koordinaten von Ort B		*/

wab = A.VektorAngle (B);		/* Winkel zwischen Ortsvektoren */
DIST = 60 * (wab * DB180/pi);	/* Dist von A nach B in NM		*/

N1.VektorProdukt (A,B);			/* Koordinaten der Normalen auf */
								/* der Ebene, die durch die		*/
								/* beiden Ortsvektoren aufge-	*/
								/* spannt wird.					*/

								/* mittlerer Kurs auf 1. Teil   */
X = A.GrossKreis (B, wab/2);	/* X : Ort nach Winkel wab/2	*/ 

NordPol.x = 0.;					/* Koordinaten der Normalen auf */
NordPol.y = 0.;					/* der Ebene, die durch Ort X,  */
NordPol.z = .99999;				/* dem Nordpol und dem Erd-		*/
N2.VektorProdukt (X, NordPol);	/* mittelpunkt verlaeuft.		*/
				
								/* Ergebnis immer zw. 0 u. 180! */
*ptKurs = N1.VektorAngle (N2) * DB180/pi;
dWest = m_dLon - Loc.GetLon();
if (dWest < 0.) dWest = DB360 + dWest;	/* Winkeldiff. in westl. Richtg.*/
if (dWest < DB180) *ptKurs = DB360 - *ptKurs;   /* Soll-Kurs von 180 - 360	*/

*ptKurs = floor (*ptKurs + 0.5);		/* Kurs auf- oder abrunden	*/

return DIST;
}


 
/************************************************************************
 *  LatLon.cpp				F r o m R e l K o o r d s					*
 *  Berechnet einen GrossKreis-Punkt, der von "BaseLoc" aus in einer	*
 *  Richtung von "nRWK" und einer Entfernung von "Dist" liegt.			*
 ************************************************************************/
void CLatLon::FromRelKoords (CLatLon& BaseLoc, short nRWK, double Dist)
{
CVektor A, P, B1, B2, X;		/* euklidische Koordinaten	*/
double	    RadLat, RadLon, RadWi;

BaseLoc.ToXYZ(&A);    /* Koordinaten von Ort A	*/

RadLat = BaseLoc.GetLat() * pi / DB180;
RadLon = BaseLoc.GetLon() * pi / DB180;

B1.x = 1	    * sin (nRWK * pi/DB180);
B1.y =-sin (RadLat) * cos (nRWK * pi/DB180);
B1.z = cos (RadLat) * cos (nRWK * pi/DB180);

P.x = P.y = P.z = 0;
//RotateXY ((LPVEK)&B1, (LPVEK)&P, -RadLon, (LPVEK)&B2);
B2 = B1.RotateXY (P, -RadLon);

RadWi = Dist * pi/DB180 / 60;		/* RadWi = angle (A, X)			*/
X = A.GrossKreis (B2, RadWi);		/* X : Ort nach Winkel RadWi	*/

this->FromXYZ(X);
}
 
/************************************************************************
*  LatLon.cpp    		 		 S e r i a l i z e						*
 ************************************************************************/
void CLatLon::Serialize(CArchive& ar)
{ 
if (ar.IsStoring())
	{		// TODO: add storing code here 
	ar << m_dLat;	
	ar << m_dLon;	
	}
else{       // TODO: add loading code here    
	ar >> m_dLat;	
	ar >> m_dLon;	
	}
} 
  
/////////////////////////////////////////////////////////////////////////////
// CLatLon commands
