/************************************************************************
 *  						L o c a t i o n . c p p	  					*
 ************************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "math.h"							// for: fabs
#include "pf.h"

#include "DimDoc.h"
#include "Location.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

extern CDimDoc* ptDim;

/////////////////////////////////////////////////////////////////////////////
// CLocation

IMPLEMENT_SERIAL(CLocation, CLatLon, 0 /* schema number*/ )

CLocation::CLocation(LPKOORD lpLocation)
{ 
if (lpLocation == NULL)
	{       
	this->FirstInit();
	} 
else{
    this->ReadFromPtr (lpLocation);
    }
}

CLocation::CLocation(CLocation& SourceLoc)
{ 
*this = SourceLoc;
}


CLocation::~CLocation()
{
} 

void CLocation::GetPtr(LPKOORD lpLocation)
{
this->WriteToPtr (lpLocation);
}  

void CLocation::SetPtr (LPKOORD lpLocation)
{
this->ReadFromPtr(lpLocation);
}  

/************************************************************************
 *  Location.cpp				operator=								*
 ************************************************************************/
const CLocation& CLocation::operator=(const CLocation& Loc)
{
_fmemcpy (m_szOrt, Loc.m_szOrt, SIZEOF_ORT); 


//*((CLatLon*)this) = (CLatLon)Loc;

((CLatLon*)this)->SetLat (Loc.m_dLat);
((CLatLon*)this)->SetLon (Loc.m_dLon); 
m_lMinAlt	= Loc.m_lMinAlt;  
m_nVAR		= Loc.m_nVAR;        
m_lFrq		= Loc.m_lFrq;
m_nRwyLen	= Loc.m_nRwyLen;      
m_nElevDiff	= Loc.m_nElevDiff;
m_cRange	= Loc.m_cRange;	
m_cFlag		= Loc.m_cFlag;
			
_fmemcpy (m_szIndicator, Loc.m_szIndicator, INDICATOR_SIZE);  

m_nElevDim	= Loc.m_nElevDim;     
m_nDistDim	= Loc.m_nDistDim;    
m_nLenDim	= Loc.m_nLenDim;     
	           
m_nCountryCode	= Loc.m_nCountryCode;
m_cType		= Loc.m_cType;	
m_cDummy	= Loc.m_cDummy;
m_nDiffDim  = Loc.m_nDiffDim;    
m_lDummy	= Loc.m_lDummy;

_fmemcpy (m_szRegionName, Loc.m_szRegionName, SIZEOF_REGION);  

return *this;
}

/************************************************************************
 *  Location.cpp    		S t r i n g T o C h a r A r r a y			*
 *  Copies CString to char array. If CString is longer than array,		*
 *	only nArrayLen-1 characters will be copied!							*
 ************************************************************************/
void CLocation::StringToCharArray(CString szText, char* ptText, short nArrayLen)
{
	short nLen = szText.GetLength();
	if (nLen < nArrayLen)
	{
		strcpy(ptText, (LPCTSTR)szText);
	}
	else
	{			// cut part of input parameter
		strncpy(ptText, (LPCTSTR)szText, nArrayLen-1);
		ptText[nArrayLen-1] = 0;
	}
}


/************************************************************************
 *  Location.cpp				F i r s t I n i t						*
 ************************************************************************/
void CLocation::FirstInit()
{
_fmemset (m_szOrt, 0, SIZEOF_ORT);  
m_lMinAlt	= NO_ALT;  
m_nVAR		= 0;        
m_lFrq		= (long)NO_FREQ;
m_nRwyLen	= NO_RWY;      
m_nElevDiff	= 0;
m_cRange	= 0;	
m_cFlag		= Country;
			
_fmemset (m_szIndicator, 0, INDICATOR_SIZE);  


if (ptDim != NULL)
	{
	m_nElevDim	= ptDim->Altitude();     
	m_nDistDim	= ptDim->Distance();    
	m_nLenDim	= ptDim->RwyLength();     
	m_nDiffDim  = ptDim->Altitude();    
	}
else{
	m_nElevDim	= 0;
	m_nDistDim	= 0;
	m_nLenDim	= 0;
	m_nDiffDim  = 0;    
	}           

m_nCountryCode	= 0;
SetCategory (WP_USER);
m_cDummy	= 0;
m_lDummy	= -1;

						// defined in BOOL CLaLoList::GetLoc
_fmemset (m_szRegionName, 0, SIZEOF_REGION);  
}


/************************************************************************
 *  Location.cpp				I s D e f i n e d 						*
 ************************************************************************/
BOOL CLocation::IsDefined()
{
	BOOL bDefined = TRUE;

	if ((strlen(m_szOrt) == 0) ||
		(this->GetLat() == NO_KOORD) ||
		(this->GetLon() == NO_KOORD))
		bDefined = FALSE;

	return bDefined;
}

/************************************************************************
 *  Location.cpp				R e s e t								*
 ************************************************************************/
void CLocation::Reset(CString szName, short nVar, BYTE btFlag)
{
this->FirstInit();

this->SetName (szName);
SetLat (NO_KOORD);
SetLon (NO_KOORD); 
m_nVAR		= nVar;        
m_cFlag		= btFlag;
}


/************************************************************************
 *  Location.cpp				ReadFromPtr							*
 ************************************************************************/
void CLocation::ReadFromPtr(LPKOORD lpLoc)
{
_fmemcpy (m_szOrt, lpLoc->szOrt, SIZEOF_ORT); 
SetLat (lpLoc->dLat);
SetLon (lpLoc->dLon); 
m_lMinAlt	= lpLoc->lMinAlt;  
m_nVAR		= lpLoc->nVAR;        
m_lFrq		= lpLoc->lFrq;
m_nRwyLen	= lpLoc->nRwyLen;      
m_nElevDiff	= lpLoc->nElevDiff;
m_cRange	= lpLoc->cRange;	
m_cFlag		= lpLoc->cFlag;
			
_fmemcpy (m_szIndicator, lpLoc->szIndicator, INDICATOR_SIZE);  

m_nElevDim	= lpLoc->nAltDim;     
m_nDistDim	= lpLoc->nDistDim;    
m_nLenDim	= lpLoc->nLenDim;     
	           
m_nCountryCode	= lpLoc->nCountryCode;
m_cType		= lpLoc->cType;	
m_cDummy	= lpLoc->cDummy;
m_nDiffDim  = lpLoc->nDiffDim;    
m_lDummy	= lpLoc->lDummy;
}

/************************************************************************
 *  Location.cpp				WriteToPtr							*
 ************************************************************************/
void CLocation::WriteToPtr(LPKOORD lpLoc) const
{
_fmemcpy (lpLoc->szOrt, m_szOrt, SIZEOF_ORT); 
lpLoc->dLat = GetLat ();
lpLoc->dLon	= GetLon (); 
lpLoc->lMinAlt	= m_lMinAlt;  
lpLoc->nVAR		= m_nVAR;        
lpLoc->lFrq		= m_lFrq;
lpLoc->nRwyLen	= m_nRwyLen;      
lpLoc->nElevDiff= m_nElevDiff;
lpLoc->cRange	= m_cRange;	
lpLoc->cFlag	= m_cFlag;
			
_fmemcpy (lpLoc->szIndicator, m_szIndicator, INDICATOR_SIZE);  

lpLoc->nAltDim	= m_nElevDim;     
lpLoc->nDistDim	= m_nDistDim;    
lpLoc->nLenDim	= m_nLenDim;     
	           
lpLoc->nCountryCode	= m_nCountryCode;
lpLoc->cType		= m_cType;	
lpLoc->cDummy		= m_cDummy;
lpLoc->nDiffDim		= m_nDiffDim;    
lpLoc->lDummy		= m_lDummy;
}


/************************************************************************
 *  Location.cpp			C r e a t e H o m e B a s e 		  		*
 ************************************************************************/
void CLocation::CreateHomeBase()
{
	strcpy (m_szOrt,"Frankfurt-Egelsb.");
	this->SetLat (49.96000);
    this->SetLon ( 8.64333);
    m_lMinAlt	= 385L;
    m_nVAR		= 0;
    m_lFrq		= 12172;	/* Frequenz * 100		*/
    m_nRwyLen	= 1166;
    m_nElevDiff = 0;
    this->SetRwyDir (9);
    m_cFlag	= Country;	/* SaveAlt: +500, no gras	*/

    strcpy (m_szIndicator, "EDFE");
    m_nElevDim   = DIM_FEET;
    m_nDistDim  = DIM_NM;
    m_nLenDim   = DIM_METER;
    m_cType	    = WP_AIRPORT;
	this->SetCountryCode ((CString)"ED");
    m_cDummy	= 0;
    m_nDiffDim	= m_nElevDim;
    m_lDummy    = -1;

}

/************************************************************************
 *  Location.cpp			N a v F r e q T o C a t 		 	 		*
 ************************************************************************/
short CLocation::NavFreqToCat (long lFreq)
{
short nCat = WP_USER;

if (lFreq >= 108000 && lFreq <= 117970)  		/* VOR 1/1000 MHz			*/
	nCat = WP_VOR;
		
if (lFreq >= 190000 && lFreq <= 1750000) 		/* NDB 1/1000 kHz			*/
	nCat = WP_NDB;
	
return nCat;
}


/************************************************************************
 *  Location.cpp			S e t T y p e A n d D u m m i e s			*
 ************************************************************************/
void CLocation::SetTypeAndDummies (long lFreq)
{
m_cType   = WP_USER;

if (lFreq >= 108000 && lFreq <= 117970)	    /* VOR 1/1000 MHz			*/
    m_cType	= WP_VOR;

if (lFreq >= 118000 && lFreq <= 136950)	    /* Airport 1/1000 MHz	    */
    {
    BOOL    bReportingPt = TRUE;
    short   i;

    for (i=0; i<4 && bReportingPt; i++)	    /* 4 upper case letters */
	if ((m_szOrt[i] < 'A') || (m_szOrt[i] > 'Z'))
	    bReportingPt = FALSE;
    if (m_szOrt[i] != '-')   /* and a '-' => reporting point */
	bReportingPt = FALSE;

    m_cType	= (bReportingPt)? WP_REPORT : WP_AIRPORT;
    }

if (lFreq >= 190000 && lFreq <= 1750000)	    /* NDB 1/1000 kHz	    */
    m_cType	= WP_NDB;

m_cDummy	= 0;
m_nDiffDim	= m_nElevDim;
m_lDummy	= -1;          
}


/************************************************************************
 *  Location.cpp			A d d D i g i t T o F r e q					*
 ************************************************************************/
void CLocation::AddDigitToFreq(long* ptFreq)
{
	long lFreq00 = *ptFreq;
	long lFreq000 = 0;

	if (lFreq00 >= 10800 &&				/* VOR 1/100 MHz	bis 11797   */
	    lFreq00 <= 13695)				/* Airport 1/100kHz	 ab 11800   */
    {	// aus xxxyy xxxyy0 machen
		lFreq000 = 10*lFreq00;

		char	szBuffer[16]; 
		int len = sprintf (szBuffer, "%ld", lFreq000);
		if (len == 6)
		{
			if (szBuffer[4] == '2')
			{	//     01234 
				// aus xxx22 xxx225 machen
				lFreq000 += 5;
			}
			else if (szBuffer[4] == '7')
			{	//     01234 
				// aus xxx77 xxx775 machen
				lFreq000 += 5;
			}									
		}
		
		*ptFreq = lFreq000;				/* Airport/VOR 1/1000 MHz	    */
	}
	else
	{
		*ptFreq = 10 * lFreq00;			/* NDB 1/1000 kHz				*/
	}
}

/************************************************************************
 *  Location.cpp    		 		 S e r i a l i z e					*
 ************************************************************************/
void CLocation::Serialize(CArchive& ar, short nReadVers, short nCC)
{ 
if (ar.IsStoring())
	{
		// TODO: add storing code here 
	AnsiToOem(m_szOrt, m_szOrt);
	ar.Write (m_szOrt, SIZEOF_ORT);
	OemToAnsi(m_szOrt, m_szOrt);		// this object may be used after it was stored!!

	CLatLon::Serialize (ar);

	ar << m_lMinAlt; 	
    ar << (WORD)m_nVAR;
	ar << m_lFrq;
    ar << (WORD)m_nRwyLen;  	
    ar << (WORD)m_nElevDiff;      	
    ar << (BYTE)m_cRange; 
    ar << (BYTE)m_cFlag;  		
    
	ar.Write (m_szIndicator, INDICATOR_SIZE);
    ar << (WORD)m_nElevDim; 
    ar << (WORD)m_nDistDim; 
    ar << (WORD)m_nLenDim;  
	
    ar << (WORD)m_nCountryCode;

    ar << (BYTE)m_cType;
    ar << (BYTE)m_cDummy;
	                                      
    ar << (WORD)m_nDiffDim;
	ar << m_lDummy;	
	}
	else
	{       
		// TODO: add loading code here    
	BYTE	Byte; 
	WORD	Word;

	ar.Read (m_szOrt, SIZEOF_ORT); 
	  
	CLatLon::Serialize (ar);

	ar >> m_lMinAlt; 								/* minimale Flughhe, Elevation 	*/
    ar >> Word;		m_nVAR = (short)Word;
	ar >> m_lFrq;
    ar >> Word;		m_nRwyLen = (short)Word;  	
    ar >> Word;		m_nElevDiff = (short)Word;  
	ar >> Byte;		m_cRange = (unsigned char)Byte;
	ar >> Byte;		m_cFlag = (unsigned char)Byte;


	if (nReadVers == 1)    
		{ 		// loading verion 1 database (struct KOORDTYPE1)
		*(m_szIndicator) = 0;
		m_nElevDim    = DIM_FEET;
		m_nDistDim   = DIM_NM;
		m_nLenDim    = DIM_METER;     
		this->SetTypeAndDummies (m_lFrq); 
		m_nCountryCode = nCC;

		if (this->IsAirport())	 
			this->SetRwyDir (NO_RWYDIR);
 	    }  
	    
	if (nReadVers == 2)
	    {						// reads struct KOORDTYPE2
		ar.Read (m_szIndicator, INDICATOR_SIZE);
	    ar >> Word;		m_nElevDim = (short)Word; 
		ar >> Word;		m_nDistDim = (short)Word; 
		ar >> Word;		m_nLenDim = (short)Word;   
		ar >> Word;		m_nDiffDim = (short)Word; // reads dummy!!
		this->SetTypeAndDummies (m_lFrq);	// sets m_nDiffDim here!!    
 		m_nCountryCode = nCC;

		if (this->IsAirport())	 
			this->SetRwyDir (NO_RWYDIR);
	    } 
	    
	if (nReadVers == 3)
	    {						// reads struct KOORDTYPE21
		ar.Read (m_szIndicator, INDICATOR_SIZE);
	    ar >> Word;		m_nElevDim = (short)Word; 
		ar >> Word;		m_nDistDim = (short)Word; 
		ar >> Word;		m_nLenDim = (short)Word;   
		ar >> Word;		m_nCountryCode = (short)Word;
		ar >> Byte;		m_cType = (unsigned char)Byte; 
		ar >> Byte;		m_cDummy = (unsigned char)Byte;
		                                      
		ar >> Word;		m_nDiffDim = (short)Word; 
		if (m_nDiffDim == -1) 
						m_nDiffDim = m_nElevDim;
		ar >> m_lDummy;

		if (this->IsAirport())	 
			this->SetRwyDir (NO_RWYDIR);
	    }

	if (nReadVers <= 4)
	{
		this->AddDigitToFreq(&m_lFrq);
	}

	if (nReadVers == 4 || nReadVers == ACT_KOORD_VERS)
	    {					// reads struct KOORDTYPE21 with RwyDir in Range
		ar.Read (m_szIndicator, INDICATOR_SIZE);
	    ar >> Word;		m_nElevDim = (short)Word; 
		ar >> Word;		m_nDistDim = (short)Word; 
		ar >> Word;		m_nLenDim = (short)Word;   
		ar >> Word;		m_nCountryCode = (short)Word;
		ar >> Byte;		m_cType = (unsigned char)Byte; 
		ar >> Byte;		m_cDummy = (unsigned char)Byte;
		ar >> Word;		m_nDiffDim = (short)Word; 
		if (m_nDiffDim == -1) 
						m_nDiffDim = m_nElevDim;
		ar >> m_lDummy;
	    }

	
	OemToAnsi(m_szOrt, m_szOrt);
	}
} 

/************************************************************************
 *  Location.cpp    		 	C r e a t e P t r 						*
 ************************************************************************/
LPKOORD CLocation::CreatePtr ()
{
LPKOORD lpLoc = new KOORDTYPE;

_fmemset (lpLoc, 0, sizeof (KOORDTYPE));

WriteToPtr(lpLoc);

return lpLoc;
}

/************************************************************************
 *  Location.cpp    		 	I s E q u a l							*
 ************************************************************************/
BOOL CLocation::IsEqual(LPKOORD lpK1, LPKOORD lpK2)
{    
//double	EPS_wi = 0.00001;			/* Genauigkeit zum Winkelvergleich	*/

if (_fstrcmp (lpK1->szOrt, lpK2->szOrt) != 0)	return FALSE; 
//if (fabs (lpK1->dLat - lpK2->dLat) > EPS_wi)	return FALSE;
//if (fabs (lpK1->dLon - lpK2->dLon) > EPS_wi)	return FALSE;       

CLatLon Pos1 (lpK1->dLat, lpK1->dLon);
CLatLon Pos2 (lpK2->dLat, lpK2->dLon);
if (!Pos1.IsEqual(Pos2))	return FALSE;


if (ptDim->CompareDist (lpK1->lMinAlt, lpK1->nAltDim, 
						lpK2->lMinAlt, lpK2->nAltDim, NO_ALT) != 0) return FALSE;
if (lpK1->nVAR != lpK2->nVAR)			return FALSE;
if (lpK1->lFrq != lpK2->lFrq)			return FALSE;  

if (ptDim->CompareDist (lpK1->nRwyLen, lpK1->nLenDim, 
						lpK2->nRwyLen, lpK2->nLenDim, NO_RWY) != 0) return FALSE;
if (ptDim->CompareDist (lpK1->nElevDiff, lpK1->nAltDim, 
						lpK2->nElevDiff, lpK2->nAltDim, NO_ALT) != 0) return FALSE;


if ((lpK1->cFlag & Country) 	!= (lpK2->cFlag & Country))	return FALSE;
if ((lpK1->cFlag & City) 		!= (lpK2->cFlag & City))	return FALSE;
if ((lpK1->cFlag & bigCity) 	!= (lpK2->cFlag & bigCity))	return FALSE;
if ((lpK1->cFlag & Gras) 		!= (lpK2->cFlag & Gras))	return FALSE;
 
if (_fstrcmp (lpK1->szIndicator, lpK2->szIndicator) != 0)	return FALSE; 
if (lpK1->nAltDim != lpK2->nAltDim)		return FALSE;
if (lpK1->nLenDim != lpK2->nLenDim)		return FALSE;

if (lpK1->nCountryCode != lpK2->nCountryCode)		return FALSE;
if (lpK1->cType != lpK2->cType)			return FALSE;	// type unchanged??
if (lpK1->cDummy != lpK2->cDummy)		return FALSE;
if (lpK1->nDiffDim != lpK2->nDiffDim)		return FALSE;
if (lpK1->lDummy != lpK2->lDummy)		return FALSE;   

if (this->IsAirport())		// check this after beeing sure, that type is unchanged
	{		// compare runway direction
	if (lpK1->cRange != lpK2->cRange) return FALSE;
	}
else{		// compare range
	if (lpK1->nDistDim != lpK2->nDistDim)	return FALSE;
	if (ptDim->CompareDist (lpK1->cRange, lpK1->nDistDim, 
						lpK2->cRange, lpK2->nDistDim, NO_LONG) != 0) return FALSE;
	}

return TRUE;
}

/************************************************************************
 *  Location.cpp    		 		I s E q u a l						*
 ************************************************************************/
BOOL CLocation::IsEqual(LPKOORD lpTestLocation)
{
CLocation TestLoc (lpTestLocation);
return (this->IsEqual (TestLoc));
}      

/************************************************************************
 *  Location.cpp    		 		I s R e l a t i v					*
 ************************************************************************/
BOOL CLocation::IsRelativ()
{
BOOL bRelativ = ((m_cFlag & Relativ) == Relativ);
return bRelativ;
}

/************************************************************************
 *  Location.cpp    		 		I s E q u a l						*
 ************************************************************************/
BOOL CLocation::IsEqual(const CLocation& Loc)
{
//double	EPS_wi = 0.00001;			/* Genauigkeit zum Winkelvergleich	*/

if (_fstrcmp (m_szOrt, Loc.m_szOrt) != 0)	return FALSE; 
//if (fabs (m_dLat - Loc.m_dLat) > EPS_wi)	return FALSE;
//if (fabs (m_dLon - Loc.m_dLon) > EPS_wi)	return FALSE;       

CLatLon Pos1 (m_dLat, m_dLon);
CLatLon Pos2 (Loc.m_dLat, Loc.m_dLon);
if (!Pos1.IsEqual(Pos2))	return FALSE;


if (ptDim->CompareDist (m_lMinAlt, m_nElevDim, 
						Loc.m_lMinAlt, Loc.m_nElevDim, NO_ALT) != 0) return FALSE;
if (m_nVAR != Loc.m_nVAR)			return FALSE;
if (m_lFrq != Loc.m_lFrq)			return FALSE;  

if (ptDim->CompareDist (m_nRwyLen, m_nLenDim, 
						Loc.m_nRwyLen, Loc.m_nLenDim, NO_RWY) != 0) return FALSE;
if (ptDim->CompareDist (m_nElevDiff, m_nElevDim, 
						Loc.m_nElevDiff, Loc.m_nElevDim, NO_ALT) != 0) return FALSE;
if (ptDim->CompareDist (m_cRange, m_nDistDim, 
						Loc.m_cRange, Loc.m_nDistDim, NO_LONG) != 0) return FALSE;

if ((m_cFlag & Country) 	!= (Loc.m_cFlag & Country))	return FALSE;
if ((m_cFlag & City) 		!= (Loc.m_cFlag & City))	return FALSE;
if ((m_cFlag & bigCity) 	!= (Loc.m_cFlag & bigCity))	return FALSE;
if ((m_cFlag & Gras) 		!= (Loc.m_cFlag & Gras))	return FALSE;
 
if (_fstrcmp (m_szIndicator, Loc.m_szIndicator) != 0)	return FALSE; 
if (m_nElevDim != Loc.m_nElevDim)		return FALSE;
if (m_nDistDim != Loc.m_nDistDim)	return FALSE;
if (m_nLenDim != Loc.m_nLenDim)		return FALSE;

if (m_nCountryCode != Loc.m_nCountryCode)		return FALSE;
if (m_cType != Loc.m_cType)			return FALSE;
if (m_cDummy != Loc.m_cDummy)		return FALSE;
if (m_nDiffDim != Loc.m_nDiffDim)		return FALSE;
if (m_lDummy != Loc.m_lDummy)		return FALSE;   

return TRUE;
}

/************************************************************************
 *  Location.cpp			F l i g h t A n g l e						*
 *  Same as in CBorder class											*
 ************************************************************************/
double CLocation::FlightAngle (double A, double B)
{
double angle;

if (A < 0.) A += 360.;
if (B < 0.) B += 360.;
angle = B - A;
if (angle > 180.) angle -= 360.;
if (angle <-180.) angle += 360.;
return angle;
}

/************************************************************************
 *  Location.cpp				I s N e a r								*
 ************************************************************************/
BOOL CLocation::IsNear(CLocation& TestLocation)
{
BOOL	bNear = FALSE;
short	nDegreeLimit = 2;
 
if (fabs(TestLocation.GetLat() - this->GetLat()) < 2)
    {			    /* Breitengrad-Differenz < 120 NM	    */
    double  Lat_Average;
    Lat_Average = pi * (TestLocation.GetLat() + this->GetLat()) / 360;

	double dLonDiff = CLocation::FlightAngle(TestLocation.GetLon(), this->GetLon());

    if (fabs(dLonDiff) * cos(Lat_Average) < 2)
 //   if (fabs(TestLocation.GetLon() - this->GetLon()) * cos(Lat_Average) < 2)
		{		    /* Lngengrad-Differenz < 120 NM	    */
		bNear = TRUE;
		}
    }
return bNear;
}

/************************************************************************
 *  Location.cpp				IsFacilityInRange				*
 ************************************************************************/
BOOL CLocation::IsFacilityInRange (CLocation& ActLoc, double* ptDist)
{
double	LimDist;
double	dKurs;

if (this->IsFacility())
    {				/* Funknavigations-Anlage		*/
	long	lRange;
	*ptDist = this->LoxoDist ((CLatLon)ActLoc, &dKurs);

	if (this->GetRange(&lRange))
		{
		LimDist = ptDim->ConvertDist((double)lRange,
				m_nDistDim, DIM_NM);
		}
	else{
		LimDist = 60.;		    /* NM	    */
		}

	if (*ptDist <= LimDist)
		return TRUE;
    }
return FALSE;
}

/************************************************************************
 *  Location.cpp				I s A i r p o r t						*
 ************************************************************************/
BOOL CLocation::IsAirport ()
{
BOOL	bAirport = FALSE;     
					 
bAirport = (this->GetCategory() == WP_AIRPORT ||
			this->GetCategory() == WP_SAIL); 

if (this->GetCategory() == WP_USER)
	{
	short nRwyLen;				// dummy 
	bAirport = this->GetRwyLen (&nRwyLen);
	}

return bAirport;
}

/************************************************************************
 *  Location.cpp				I s F a c i l i t y						*
 ************************************************************************/
BOOL CLocation::IsFacility ()
{
BOOL	bFacility = FALSE;     
					 
bFacility = ((this->GetCategory() == WP_NDB) || (this->GetCategory() == WP_VOR)); 
return bFacility;
}

/************************************************************************
 *  Location.cpp				I s V a l i d							*
 ************************************************************************/
BOOL CLocation::IsValid ()
{
BOOL	bValid = FALSE;     

bValid = CLatLon::IsValid();
					 
bValid = (	bValid &&
			(m_nVAR >= -180 && m_nVAR <= 180)); 
return bValid;
}


/************************************************************************
 *  Location.cpp    	 		S e t F l a g B i t						*
 ************************************************************************/
void CLocation::SetFlagBit (unsigned char FAR* lpFlag, short BitMask, BOOL bSet)
{					/* BitMask: 1, 2, 4, 8, 16, ...	*/
if (bSet)   *lpFlag = *lpFlag | (unsigned char)BitMask;
    else    *lpFlag = *lpFlag & (unsigned char)(~BitMask);	/* ALT 126: ~	*/
}

/************************************************************************
 *  Location.cpp    		 	G e t T e x t W i d t h					*
 ************************************************************************/
short CLocation::GetTextWidth(CDC* pDC)
{  
short	nTextLen;
CSize	TextSize;

if (this->IndicatorUsed())
			TextSize = pDC->GetTextExtent(m_szIndicator, strlen(m_szIndicator));
	else	TextSize = pDC->GetTextExtent(m_szOrt, strlen(m_szOrt));
nTextLen = (short)TextSize.cx;
return nTextLen;
}

/************************************************************************
 *  Location.cpp    		 	S e t R e g i o n N a m e 				*
 ************************************************************************/
void CLocation::SetRegionName(CString szRegionName)
{    
	StringToCharArray(szRegionName, m_szRegionName, SIZEOF_REGION);
}             

/************************************************************************
 *  Location.cpp    		 	G e t R e g i o n N a m e				*
 ************************************************************************/
CString CLocation::GetRegionName()
{         
CString szRegionName = (CString)m_szRegionName;
return szRegionName;
}
    
/************************************************************************
 *  Location.cpp    		 		S e t N a m e						*
 *  copies szName to lpName[SIZEOF_ORT]									*
 ************************************************************************/
void CLocation::SetName(LPSTR lpName, CString szName)
{    
	StringToCharArray(szName, lpName, SIZEOF_ORT);
}   

/************************************************************************
 *  Location.cpp    		 		 S e t N a m e						*
 ************************************************************************/
void CLocation::SetName(CString szOrt)
{    
	StringToCharArray(szOrt, m_szOrt, SIZEOF_ORT);
}


/************************************************************************
 *  Location.cpp    			G e t N a m e							*
 ************************************************************************/
CString CLocation::GetName()
{         
CString szOrt = (CString)m_szOrt;
return szOrt;
}

/************************************************************************
 *  Location.cpp    			 G e t N a m e							*
 ************************************************************************/
BOOL CLocation::GetName(LPSTR lpName)
{         
_fstrcpy (lpName, (LPSTR)m_szOrt);    
return (*m_szOrt > 0);
}


/************************************************************************
 *  Location.cpp    		 U s e L a t L o n N a m e					*
 ************************************************************************/
void CLocation::UseLatLonName()
{
CString szLatLon;
this->CreateLatLonStr(&szLatLon);
this->SetName(szLatLon);
}


/************************************************************************
 *  Location.cpp    		 		S e t I n d i c a t o r				*
 *  copies szIndicator to lpIndicator[INDICATOR_SIZE]					*
 ************************************************************************/
void CLocation::SetIndicator(LPSTR lpIndicator, CString szIndicator)
{    
	StringToCharArray(szIndicator, lpIndicator, INDICATOR_SIZE);
}      

/************************************************************************
 *  Location.cpp    		 		S e t I n d i c a t o r				*
 ************************************************************************/
void CLocation::SetIndicator(CString szIndicator)
{    
	StringToCharArray(szIndicator, m_szIndicator, INDICATOR_SIZE);
}             

/************************************************************************
 *  Location.cpp    		 		G e t I n d i c a t o r				*
 ************************************************************************/
CString CLocation::GetIndicator()
{         
CString szIndicator = (CString)m_szIndicator;
return szIndicator;
}

/************************************************************************
 *  Location.cpp    		 	 G e t I n d i c a t o r				*
 ************************************************************************/
BOOL CLocation::GetIndicator(LPSTR lpIndicator)
{         
_fstrcpy (lpIndicator, (LPSTR)m_szIndicator);
return (*m_szIndicator > 0);
}

/************************************************************************
 *  Location.cpp    		 	 S e t A l t i t u d e	 				*
 ************************************************************************/
void CLocation::SetAltitude(long lAlt)
{
m_lMinAlt = lAlt;
}

/************************************************************************
 *  Location.cpp    		 	G e t A l t i t u d e	 				*
 ************************************************************************/
BOOL CLocation::GetAltitude(long* ptAlt)
{
BOOL bOK = FALSE;  
*ptAlt = NO_ALT;
if (m_lMinAlt != NO_ALT)
	{
	*ptAlt  = m_lMinAlt;    /* minimale Flughhe, Elevation 	*/   
	bOK = TRUE;
	}
return bOK;        
}            

/************************************************************************
 *  Location.cpp    		 	S e t F r e q u e n c y					*
 ************************************************************************/
void CLocation::SetFrequency(float fFreq)
{
if (fFreq == NO_FREQ)
	{
	m_lFrq = 0L;
	}
else{
	m_lFrq = (long)(fFreq * 1000 +.5);
	}
}

/************************************************************************
 *  Location.cpp    		 	G e t F r e q u e n c y					*
 ************************************************************************/
BOOL CLocation::GetFrequency(float* ptFreq)
{ 
BOOL bOK = FALSE;  
*ptFreq = NO_FREQ;
if (m_lFrq > 0)
	{
	*ptFreq = (float)m_lFrq/1000; 
	bOK = TRUE;
	}
return bOK;        
}            


/************************************************************************
 *  Location.cpp    		 	S e t R a n g e							*
 ************************************************************************/
void CLocation::SetRange(long lRange)
{
if (lRange == NO_LONG)
	 m_cRange = (BYTE)0;
else m_cRange = (BYTE)lRange;
}	

/************************************************************************
 *  Location.cpp    		 		G e t R a n g e						*
 ************************************************************************/
BOOL CLocation::GetRange(long* ptRange)
{ 
BOOL bOK = FALSE;   
*ptRange = NO_LONG;
if (m_cRange > (BYTE)0)
	{
	*ptRange = (long)m_cRange; 
	bOK = TRUE;
	}
return bOK;        
}  

/************************************************************************
 *  Location.cpp    		 		G e t R a n g e						*
 ************************************************************************/
long CLocation::GetRange()
{ 
long lRange;

if (m_cRange > (BYTE)0)
	{
	lRange = (long)m_cRange; 
	}
else{			// use default value 60 NM
	lRange = (long)ptDim->ConvertDist(60, DIM_NM, m_nDistDim);
	}

return lRange;        
}  

/************************************************************************
 *  Location.cpp    		 	S e t R w y D i r						*
 ************************************************************************/
void CLocation::SetRwyDir(short nDir)
{ 
m_cRange = (BYTE)nDir;		//10 degrees
}      

/************************************************************************
 *  Location.cpp    		 		G e t R w y D i r					*
 *  Returns values from 0 to 17											*
 ************************************************************************/
short CLocation::GetRwyDir(void)
{ 
return (BYTE)m_cRange;			// 10 degrees        
}   

/************************************************************************
 *  Location.cpp    			S e t R w y L e n						*
 ************************************************************************/
void CLocation::SetRwyLen(short nRwyLen)
{
m_nRwyLen = nRwyLen;
}

/************************************************************************
 *  Location.cpp    			G e t R w y L e n						*
 ************************************************************************/
BOOL CLocation::GetRwyLen(short* ptRwyLen)
{ 
BOOL bOK = FALSE;     
*ptRwyLen = NO_RWY;
if (m_nRwyLen > 1)
	{
	*ptRwyLen = (short)m_nRwyLen; 
	bOK = TRUE;
	}
return bOK;        
} 
               
/************************************************************************
 *  Location.cpp    		 		S e t G r a s						*
 ************************************************************************/
void CLocation::SetGras(BOOL bGras)
{   
SetFlagBit (&m_cFlag, Gras, bGras);
}

/************************************************************************
 *  Location.cpp    		 		I s G r a s							*
 ************************************************************************/
BOOL CLocation::IsGras(void)
{ 
BOOL bGras = ((m_cFlag & Gras) == Gras);
return bGras;
}

/************************************************************************
 *  Location.cpp			 C h e c k L o n g L i m i t s	 			*
 ************************************************************************/
BOOL CLocation::CheckLongLimits(long lValue, long lMin, long lMax)
{
BOOL bOK = TRUE;

bOK = (lValue >= lMin && lValue <= lMax);
    
if (!bOK)
	{
	CString szMsg;
	szMsg.Format(IDF_IMP_LONGLIMITS, lValue, lMin, lMax);
	AfxMessageBox(szMsg);
	}
return bOK;
}                      

/************************************************************************
 *  Location.cpp		C h e c k D o u b l e L i m i t s	 			*
 ************************************************************************/
BOOL CLocation::CheckDoubleLimits(double dValue, double dMin, double dMax)
{
BOOL bOK = TRUE;

bOK = (dValue >= dMin && dValue <= dMax);
    
if (!bOK)
	{
	CString szMsg;
	szMsg.Format(IDF_IMP_DOUBLELIMITS, dValue, dMin, dMax);
	AfxMessageBox(szMsg);
	}
return bOK;
}                      


/*
    IDF_IMP_CONFIRM         "Mchten Sie Orte aus einer Text-Datei in die Region %s aufnehmen?"
	IDF_IMP_DELIMITER		"In Zeile %d mssen Sie ein Zeichen zur Trennung der Parameter festlegen, z.B. DELIMITER=;" 
	IDF_IMP_UNKNOWN			"In Zeile %d haben Sie in der %d. Spalte den unbekannten Parameter %s angekndigt."
	IDF_IMP_COLUMNS			"In Zeile %d haben Sie %d Parameter definiert, obwohl pro Ort hchstens %d Parameter vorgesehen sind."	
	IDF_IMP_TOKENS			"In Zeile %d haben Sie %d Parameter definiert, aber in Zeile 2 haben Sie %d Parameter angekndigt."
	IDF_IMP_PARAMS			"In Zeile %d wurden unzulssige Parameter gefunden!"	
	IDS_IMP_OK				"Das Importieren wurde erfolgreich abgeschlossen!"	
	IDS_IMP_CANCELED		"Das Importieren wurde abgebrochen!"
	IDF_IMP_LONGLIMITS		"Der Wert %ld sollte zwischen %ld und %ld liegen!"
	IDF_IMP_DOUBLELIMITS	"Der Wert %.2f sollte zwischen %.2f und %.2f liegen!"
	IDS_IMP_NAME			"Der Ortsname ist zu lang oder fehlt (Spalte: NAME)"
	IDS_IMP_INDIC			"Die Ortskennung ist zu lang (Spalte: IDENT)"
	IDS_IMP_LAT				"Ungltiger Breitengrad (Spalte: LAT)"
	IDS_IMP_LON				"Ungltiger Lngengrad (Spalte: LON)"
	IDS_IMP_TYPE			"Ungltige Angage zur Kategorie (SPALTE: TYPE)"
	IDS_IMP_ELEVDIM			"Verwenden Sie fr die Elevation die Einheit DIM_FEET oder DIM_METER (Spalte: ELEV_IN)"
	IDS_IMP_VAR				"Ungltige Variation (Spalte: VAR)"
	IDS_IMP_LENDIM			"Verwenden Sie fr die Bahnlnge die Einheit DIM_FEET oder DIM_METER (Spalte: RWYLEN_IN)"
	IDS_IMP_RWYDIR			"Ungltige Angabe zur Ausrichtung der Bahn. (Spalte: RWYDIR)"
	IDS_IMP_RANGEDIM		"Verwenden Sie fr die Reichweite die Einheit DIM_NM, DIM_KM oder DIM_STM. (Spalte: RANGE_IN)"
	IDS_IMP_SLOPEDIM		"Verwenden Sie fr die Neigung Bahn die Hhendifferenz in DIM_FEET oder DIM_METER. (Spalte: RWYSLOPE_IN)"
*/
/************************************************************************
 *  Location.cpp    		 		I s O K								*
 ************************************************************************/
BOOL CLocation::IsOK(void)
{ 
	BOOL bOK = TRUE;
	BOOL bValid;
	short	nLen;

	// COUNTRY
	// short	m_nCountryCode;

	// NAME
	nLen = strlen (m_szOrt);
	if (nLen < 1 || nLen >= SIZEOF_ORT)
	{
		AfxMessageBox (IDS_IMP_NAME);
		return FALSE;
	}

	//IDENT
	nLen = strlen (m_szIndicator);
	if (nLen >= INDICATOR_SIZE)
	{
		AfxMessageBox (IDS_IMP_INDIC);	
		return FALSE;
	}

	// LAT
	bValid = (GetLat() >= -90. && GetLat() <= 90.); 
	if (!bValid)
	{
		AfxMessageBox (IDS_IMP_LAT);
		return FALSE;
	}

	// LON
	bValid = (GetLon() >= -180. && GetLon() <= 180.); 
	if (!bValid)
	{
		AfxMessageBox (IDS_IMP_LON);
		return FALSE;
	}

	// TYPE				// "WP_AIRPORT", ..., WP_HELI
	if (m_cType < WP_AIRPORT || m_cType > WP_HELI)
	{
		AfxMessageBox (IDS_IMP_TYPE);
		return FALSE;
	}

	// VAR
	bValid = (m_nVAR >= -180.&& m_nVAR <= 180.); 
	if (!bValid)
	{
		AfxMessageBox (IDS_IMP_VAR);
		return FALSE;
	}

	// FREQ
	if (m_lFrq > NO_FREQ)
		{
		float fHelp = (float)m_lFrq / 100;
		if (!CheckDoubleLimits(fHelp, 108, 1750))
			return FALSE;
		}


	// ELEV_IN
	if (m_nElevDim != DIM_FEET && m_nElevDim != DIM_METER)
	{
		AfxMessageBox (IDS_IMP_ELEVDIM);
		return FALSE;
	}

	// ELEV
	if (m_lMinAlt != NO_ALT)
	{
		long lAlt_ft = (long)ptDim->ConvertDist(m_lMinAlt, m_nElevDim, DIM_FEET);
		if (!CheckLongLimits(lAlt_ft, -5000, 10000))
			return FALSE;
	}


	if (this->IsAirport())
	{
		// RWYLEN_IN
		if (m_nLenDim != DIM_FEET && m_nLenDim != DIM_METER)
		{
			AfxMessageBox (IDS_IMP_LENDIM);
			return FALSE;
		}

		// RWYLEN
		if (m_nRwyLen != NO_RWY)
		{
			long lLen_m = (long)ptDim->ConvertDist(m_nRwyLen, m_nLenDim, DIM_METER);
			if (!CheckLongLimits(lLen_m, 100, 5000))
				return FALSE;
		}

		// RWYDIR
		if (m_cRange < 1 || m_cRange > 36)
		{
			AfxMessageBox (IDS_IMP_RWYDIR);
			return FALSE;
		}

		// FEST

		// RWYSLOPE_IN
		if (m_nDiffDim != DIM_FEET && m_nDiffDim != DIM_METER)
		{
			AfxMessageBox (IDS_IMP_SLOPEDIM);
			return FALSE;
		}

		// RWYSLOPE
		if (m_nElevDiff > 0)
		{
			short nDiff_ft = (short)ptDim->ConvertDist(m_nElevDiff, m_nDiffDim, DIM_FEET);
			if (!CheckLongLimits(nDiff_ft, -250, 250))
				return FALSE;
		}
	}
	else
	{
		// RANGE_IN
		if (m_nDistDim != DIM_NM && m_nDistDim != DIM_KM && m_nDistDim != DIM_STM)
		{
			AfxMessageBox (IDS_IMP_RANGEDIM);
			return FALSE;
		}

		// RANGE		
		if (m_cRange > (BYTE)0)
		{
			long lRange_NM = (long)ptDim->ConvertDist(m_cRange, m_nDistDim, DIM_NM);
			if (!CheckLongLimits(lRange_NM, 0, 255))
				return FALSE;
		}
	}

return bOK;
}


/************************************************************************
 *  Location.cpp    		 	S e t U s e r C r e a t e d				*
 ************************************************************************/
void CLocation::SetUserCreated(BOOL bIndic)
{   
SetFlagBit (&m_cFlag, UserCreated, bIndic);
}

/************************************************************************
 *  Location.cpp    		 	I s U s e r C r e a t e d 				*
 ************************************************************************/
BOOL CLocation::IsUserCreated(void)
{ 
BOOL bIndic = ((m_cFlag & UserCreated) == UserCreated);
return bIndic;
}

/************************************************************************
 *  Location.cpp    		 	U s e I n d i c a t o r					*
 ************************************************************************/
void CLocation::UseIndicator(BOOL bIndic)
{   
SetFlagBit (&m_cFlag, NameIsIndic, bIndic);
}

/************************************************************************
 *  Location.cpp    		 	I n d i c a t o r U s e d 				*
 ************************************************************************/
BOOL CLocation::IndicatorUsed(void)	const
{ 
BOOL bIndic = ((m_cFlag & NameIsIndic) == NameIsIndic);
return bIndic;
}


/************************************************************************
 *  Location.cpp		M a r k N a m e F o r R t e						*
 ************************************************************************/
BOOL CLocation::MarkNameForRte (CString szNameForRte)
{ 
BOOL bUseIndic = (szNameForRte.CompareNoCase (m_szIndicator)==0); 
BOOL bUseName = (szNameForRte.CompareNoCase (m_szOrt)==0); 
this->UseIndicator (bUseIndic);
return (bUseIndic || bUseName);
}

/************************************************************************
 *  Location.cpp			G e t N a m e F o r R t e 					*
 ************************************************************************/
CString CLocation::GetNameForRte ()
{                            
if (this->IndicatorUsed())	return (CString)m_szIndicator;
					else	return (CString)m_szOrt;
}                      

/************************************************************************
 *  Location.cpp    	 		S e t E l e v D i f f					*
 ************************************************************************/
void CLocation::SetElevDiff(long lElevDiff)
{ 
if (lElevDiff == NO_ALT)
	{
	m_nElevDiff = (unsigned char)0;
	}
else{
    m_nElevDiff = (short)lElevDiff;
	}
}      

/************************************************************************
 *  Location.cpp    		 	G e t E l e v D i f f					*
 ************************************************************************/
BOOL CLocation::GetElevDiff(long* ptElevDiff)
{ 
BOOL bOK = FALSE; 
*ptElevDiff = NO_ALT;
if (m_nElevDiff > (unsigned char)0)
	{
	*ptElevDiff = (long)m_nElevDiff; 
	bOK = TRUE;
	}
return bOK;        
}      

/************************************************************************
 *  Location.cpp    		 	G e t E l e v D i f f					*
 ************************************************************************/
short CLocation::GetElevDiff()
{ 
long lElevDiff;
if (!this->GetElevDiff (&lElevDiff))
	{
	lElevDiff = 0;
	}
return (short)lElevDiff;        
}      

/************************************************************************
 *  Location.cpp    		S e t T o p o g r a p h y 					*
 ************************************************************************/
void CLocation::SetTopography(short nTopo)
{
SetFlagBit (&m_cFlag, Country, (nTopo == Country));
SetFlagBit (&m_cFlag, City, (nTopo == City));
SetFlagBit (&m_cFlag, bigCity, (nTopo == bigCity));
}
 
/************************************************************************
 *  Location.cpp    		 G e t T o p o g r a p h y 					*
 ************************************************************************/
short CLocation::GetTopography()
{ 
short nTopo = Country;
if (m_cFlag > (BYTE)0)
	{      
	if ((m_cFlag & Country) == Country)	nTopo = Country;
	if ((m_cFlag & City) == City)		nTopo = City;
	if ((m_cFlag & bigCity) == bigCity)	nTopo = bigCity;
	}
return nTopo;        
}       

/************************************************************************
 *  Location.cpp    		 S e t C o u n t r y C o d e				*
 ************************************************************************/
void CLocation::SetCountryCode(CString szCC)
{    
_fstrncpy ((LPSTR)&(m_nCountryCode), (LPCTSTR)szCC, 2);
}        

/************************************************************************
 *  Location.cpp    	 	G e t C o u n t r y C o d e					*
 ************************************************************************/
CString CLocation::GetCountryCode(void)
{  
char szCC[3];
_fstrncpy ((LPSTR)szCC, (LPSTR)&m_nCountryCode, 2);
szCC[2] = 0;
return (CString)szCC;        
}      


/************************************************************************
 *  Location.cpp    		 	G e t S a f e A l t _ M S L 			*
 *  Purpose: returns safe altitude specified dimension (nAltDim)		*
 ************************************************************************/
long CLocation::GetSafeAlt_MSL(short nAltDim)
{ 
long	lAlt = NO_ALT;
long	lAlt_GND, lAlt_MSL, lElev;
short	nSafeAlt_ft =0;

if (this->GetAltitude(&lElev))
	{
	if ((m_cFlag & Country) == Country)	nSafeAlt_ft = 500;
	if ((m_cFlag & City) == City)		nSafeAlt_ft = 1000;
	if ((m_cFlag & bigCity) == bigCity)	nSafeAlt_ft = 2000;

	lAlt_GND = (long)ptDim->ConvertDist((double)nSafeAlt_ft, DIM_FEET, m_nElevDim);
	lAlt_MSL = lElev + lAlt_GND;

	lAlt = (long)ptDim->ConvertDist(lAlt_MSL, m_nElevDim, nAltDim);
	}
return lAlt;
}      

/************************************************************************
 *  Location.cpp    		S t r i n g T o L a t L o n					*
 ************************************************************************/
BOOL CLocation::StringToLatLon(LPCTSTR szText, double* ptAngle)
{
	BOOL bOK = FALSE;

	*ptAngle	= 0;						/* initial value			*/

	short charcount = strlen (szText);
	if (charcount > 0)
	{
		BOOL	bNeg;
		short   Status, i, j;
		char	ch;
		char	szValue[18];
		double	Wi;
				   
		bNeg	= FALSE;
		Status	= 0;						/* get degrees				*/
		
		i=0;
		do	{
			j=0;
			do	{							/* get token string			*/
				ch = *(szText+i);
				if ((ch=='W' || ch=='w') ||
					(ch=='S' || ch=='s') )	/* direction E:+, W:-		*/
					{
					bNeg = TRUE;			/* N:+, S:-					*/
					ch   = 0;
					}
				if (ch== ',') ch='.';
				if (ch== ' ') ch=0;
				*(szValue+j) = ch;
				i++;
				j++;
				} while (ch != 0);	
			if (j>1)
				{
				Wi = atof (szValue);
			
				if (Status == 0) *ptAngle += Wi;
				if (Status == 1) *ptAngle += Wi/60;
				if (Status == 2) *ptAngle += Wi/3600;
				Status++;
				}
			} while (i<charcount);
		
		if (bNeg) (*ptAngle) *= -1;

		bOK = TRUE;
	}

	return bOK;
}


/************************************************************************
 *  Location.cpp    		S t r i n g T o C a t e g o r y				*
 ************************************************************************/
BOOL CLocation::StringToCategory (CString szCategory, short* ptCat)
{
	*ptCat = 0;

	if (szCategory.Compare("WP_AIRPORT") == 0) 
		*ptCat = WP_AIRPORT;
	if (szCategory.Compare("WP_REPORT") == 0) 
		*ptCat = WP_REPORT;
	if (szCategory.Compare("WP_INTERSECTION") == 0) 
		*ptCat = WP_INTERSECTION;
	if (szCategory.Compare("WP_NDB") == 0) 
		*ptCat = WP_NDB;
	if (szCategory.Compare("WP_VOR") == 0) 
		*ptCat = WP_VOR;
	if (szCategory.Compare("WP_USER") == 0) 
		*ptCat = WP_USER;
	if (szCategory.Compare("WP_SAIL") == 0) 
		*ptCat = WP_SAIL;
	if (szCategory.Compare("WP_HELI") == 0) 
		*ptCat = WP_HELI;

	return (*ptCat > 0);
}

/************************************************************************
 *  Location.cpp    		S t r i n g T o F r e q						*
 ************************************************************************/
float CLocation::StringToFreq (CString szFreq)
{
	char szBuff[64];
	short nLen = szFreq.GetLength();

	memcpy (szBuff, (LPCTSTR)szFreq, nLen);
	szBuff[nLen] = 0;

	for (int i=0; i<nLen; i++)
	{
		if (szBuff[i] == ',')
			szBuff[i] = '.';
	}

	float fFreq = (float)atof(szBuff);
	return fFreq;
}

/************************************************************************
 *  Location.cpp    		 	S e t C a t e g o r y					*
 ************************************************************************/
void CLocation::SetCategory(short nCat)
{ 
m_cType = (BYTE)nCat;
}      

/************************************************************************
 *  Location.cpp    		 	G e t C a t e g o r y					*
 ************************************************************************/
short CLocation::GetCategory(void)
{ 
return (short)m_cType;        
}      
 
   
/************************************************************************
 *  Location.cpp    		 GetCountryName 			*
 ************************************************************************/
CString CLocation::GetCountryName(void)
{ 
CString szName;
short	nOffset = -1;
short	nCC = m_nCountryCode;

											// Region ADRIA:
if (nCC == *((short*)"LD")) nOffset = 0;	// Croatia
if (nCC == *((short*)"LJ")) nOffset = 1;	// Slovenia
if (nCC == *((short*)"LQ")) nOffset = 2;	// Bosnien-Herzegovina
if (nCC == *((short*)"LW")) nOffset = 3;	// Macedonia
if (nCC == *((short*)"LY")) nOffset = 4;	// Yugoslavia

if (nCC == *((short*)"LO")) nOffset = 5;	// Austria

											// Region BENELUX:
if (nCC == *((short*)"EB")) nOffset = 6;	// Belgium
if (nCC == *((short*)"EH")) nOffset = 7;	// Netherlands
if (nCC == *((short*)"EL")) nOffset = 8;	// Luxembourg

											// Region CZECH SLOVAK
if (nCC == *((short*)"LK")) nOffset = 9;	// Czech Republic
if (nCC == *((short*)"LZ")) nOffset = 10;   // Slovakia
if (nCC == *((short*)"EK")) nOffset = 11;	// Denmark
if (nCC == *((short*)"EF")) nOffset = 12;   // Finland
if (nCC == *((short*)"LF")) nOffset = 13;	// France
if (nCC == *((short*)"ED")) nOffset = 14;	// Germany
if (nCC == *((short*)"LG")) nOffset = 15;	// Greece
if (nCC == *((short*)"LH")) nOffset = 16;	// Hungary
if (nCC == *((short*)"EI")) nOffset = 17;	// Ireland

											// Region ITALY
if (nCC == *((short*)"LI")) nOffset = 18;	// Italy
if (nCC == *((short*)"LM")) nOffset = 19;	// Malta

if (nCC == *((short*)"EN")) nOffset = 20;	// Norway
if (nCC == *((short*)"EP")) nOffset = 21;	// Poland
if (nCC == *((short*)"LP")) nOffset = 22;	// Portugal

											// Region SPAIN:
if (nCC == *((short*)"LE")) nOffset = 23;	// Spain
if (nCC == *((short*)"GC")) nOffset = 24;	// Gran Canaria
if (nCC == *((short*)"GE")) nOffset = 25;	// Melilla
if (nCC == *((short*)"ES")) nOffset = 26;	// Sweden
if (nCC == *((short*)"LS")) nOffset = 27;	// Switzerland
if (nCC == *((short*)"EG")) nOffset = 28;	// United Kingdom



if (nOffset >= 0)
		szName.LoadString (IDS_COUNTRY_00 + nOffset);
else	szName = this->GetCountryCode(); 

return szName;        
}      
                                  
/************************************************************************
 *  Location.cpp				 A n g l e T o D M S						*
 ************************************************************************/
void CLocation::AngleToDMS (double angle, BOOL bLat,
		 short* ptDegr, short* ptMin, short* ptSec, short* ptDir)
{
short	Grad, Min, Sec, Dir;

if (angle < 0.)	Dir = (bLat)? 'S': 'W';
	else	Dir = (bLat)? 'N': 'E';

if (angle < 0.) angle *=-1;
Grad	= (short)angle;
Min	= (short)( (angle - Grad)*60 );

Sec = (short)( ((angle - Grad)*60 - Min)*60 + 0.5 );
if (Sec == 60) { Min++; Sec=0;}
if (Min == 60) {Grad++; Min=0;}

*ptDegr = Grad;
*ptMin = Min;
*ptSec = Sec;
*ptDir = Dir;
}
  
/////////////////////////////////////////////////////////////////////////////
// CLocation commands
