// Map.cpp : implementation of the CNameList class
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include <afxpriv.h>	// for CPreviewDC class

#include <math.h>

#include "pf.h"

#include "InitDoc.h"	 //includes "NameList.h"
#include "DimDoc.h"

#include "LocDoc.h"
#include "Map.h"  

#include "..\CPPTOOLS\Vektor.h"
#include "Symbols.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

extern CInitDoc* 	ptInit;              
extern BOOL bMMText;


/////////////////////////////////////////////////////////////////////////////
// CBorder

/************************************************************************
 *  Map.cpp					C B o r d e r			Constructor			*
 ************************************************************************/
CBorder::CBorder()
{
	// TODO: add one-time construction code here 
	m_bBorder		= FALSE;
	m_bAroundWorld	= FALSE;
	m_oldLon		= 0;
	m_LeftAngle		= 0;
	m_RightAngle	= 0;
	m_AngleFromStart= 0;
	m_LonFirst		= 0;

	m_rBaseLatLon.top = m_rBaseLatLon.bottom = 0;
	m_rBaseLatLon.left = m_rBaseLatLon.right = 0;

	m_dCenterLat = m_dCenterLon = 0;
}

CBorder::~CBorder()
{   
}


/************************************************************************
 *  Map.cpp					F l i g h t A n g l e						*
 *  Same as in CLocation class											*
 ************************************************************************/
double CBorder::FlightAngle (double A, double B)
{
double angle;

if (A < 0.) A += 360.;
if (B < 0.) B += 360.;
angle = B - A;
if (angle > 180.) angle -= 360.;
if (angle <-180.) angle += 360.;
return angle;
}

/****************************************************************************
 *	Map.cpp						I n i t										*
 ****************************************************************************/										 
void CBorder::Init (double fLat, double fLon)
{
m_bBorder = FALSE;								/* all angles in degrees	*/
m_bAroundWorld	= FALSE;						// is world surrounded??
m_rBaseLatLon.bottom = m_rBaseLatLon.top = fLat;/* init vertical borders	*/
m_LonFirst = fLon;								/* init horizontal borders	*/
m_LeftAngle = m_RightAngle = m_AngleFromStart = 0.;
m_oldLon = fLon;

m_dCenterLat=fLat; 
m_dCenterLon=fLon; 
}


/****************************************************************************
 *	Map.cpp						E x p a n d									*
 ****************************************************************************/										 
void CBorder::Expand (double fLat, double fLon)
{										/* getting flight angle		*/
m_bBorder = TRUE;
m_AngleFromStart += FlightAngle (m_oldLon, fLon);	

if (m_AngleFromStart > 0.)
	{									/* always positive			*/
	if (m_AngleFromStart > m_RightAngle) 
		m_RightAngle = m_AngleFromStart;
	}
else{									/* always negative			*/
	if (m_AngleFromStart < m_LeftAngle) 
		m_LeftAngle = m_AngleFromStart;
	}

if (fabs(m_LeftAngle) + fabs(m_RightAngle) >= 360.)
	{							  // world surrounded!!
	m_bAroundWorld = TRUE;
	m_rBaseLatLon.left  = m_LonFirst - 180.;
	m_rBaseLatLon.right = m_LonFirst + 180.;
	}
else{							/* get horizontal borders	*/
	m_rBaseLatLon.left = m_LonFirst + m_LeftAngle;
	m_rBaseLatLon.right = m_LonFirst + m_RightAngle;
	}
										/* get vertical borders		*/
if (fLat > m_rBaseLatLon.top) m_rBaseLatLon.top = fLat;
if (fLat < m_rBaseLatLon.bottom) m_rBaseLatLon.bottom = fLat;

m_oldLon = fLon;
}

/****************************************************************************
 *	Map.cpp						F i n i s h									*
 ****************************************************************************/										 
void CBorder::Finish ()
{						
double dLat, dLon;

if (m_bBorder)
	{		
	if (!m_bAroundWorld)
		{						// expand 10% of area 
		dLat = (m_rBaseLatLon.top - m_rBaseLatLon.bottom) * 0.1;
		dLon = (m_rBaseLatLon.right - m_rBaseLatLon.left) * 0.1;

		m_rBaseLatLon.top		= m_rBaseLatLon.top		+ dLat;
		m_rBaseLatLon.bottom	= m_rBaseLatLon.bottom	- dLat;
		m_rBaseLatLon.left		= m_rBaseLatLon.left	- dLon;
		m_rBaseLatLon.right		= m_rBaseLatLon.right	+ dLon;

		if ((m_rBaseLatLon.right - m_rBaseLatLon.left) >= 360.)
			{							  // world surrounded!!
			m_rBaseLatLon.left = m_dCenterLon - 180.;
			m_rBaseLatLon.right = m_dCenterLon + 180.;
			}
		}
	}
else{					// generate area around single waypoint
	dLat = 0.5;	// 0.5 degrees radius
	dLon = 0.5;

	m_rBaseLatLon.top		= m_dCenterLat + dLat;
	m_rBaseLatLon.bottom	= m_dCenterLat - dLat;
	m_rBaseLatLon.left		= m_dCenterLon - dLon;
	m_rBaseLatLon.right		= m_dCenterLon + dLon;
	}


m_bBorder = TRUE;
}


/************************************************************************
 *  Map.cpp				C e n t e r B o r d e r F o r					*
 ************************************************************************/
DRECT CBorder::CenterBorderFor (double dCenterLat, double dCenterLon)
{
	m_dCenterLat=dCenterLat; 
	m_dCenterLon=dCenterLon; 

	double dLat = (m_rBaseLatLon.top - m_rBaseLatLon.bottom)/2;
	double dLon = (m_rBaseLatLon.right - m_rBaseLatLon.left)/2;

	m_rBaseLatLon.top		= m_dCenterLat + dLat;
	m_rBaseLatLon.bottom	= m_dCenterLat - dLat;
	m_rBaseLatLon.left		= m_dCenterLon - dLon;
	m_rBaseLatLon.right		= m_dCenterLon + dLon;

	return m_rBaseLatLon;
}

/************************************************************************
 *  Map.cpp					S e t B o r d e r F o r						*
 ************************************************************************/
void CBorder::SetBorderFor (double dLat, double dLon)
{
	this->Init (dLat, dLon);
	this->Finish ();
}


/////////////////////////////////////////////////////////////////////////////
// CMap
IMPLEMENT_SERIAL(CMap, CObject, 0 /* schema number*/ )



/************************************************************************
 *  Map.cpp						C M a p				Constructor			*
 ************************************************************************/
CMap::CMap()
{
	// TODO: add one-time construction code here 
m_bValid = FALSE; 
m_MapXtoLP = NULL;
m_MapYtoLP = NULL;
m_szTitle.Empty();
m_nMode = ISOTROPIC;
m_bAntipod		= FALSE;						// crossing 180
}

/************************************************************************
 *  Map.cpp						C M a p				Constructor			*
 ************************************************************************/
CMap::CMap(MapProcPtr MapXtoLP, MapProcPtr MapYtoLP, 
		   MapProcPtr MapLPtoX, MapProcPtr MapLPtoY,
			ScaleProcPtr TextX, ScaleProcPtr TextY, short nMode)
{
	// TODO: add one-time construction code here 
m_bValid = FALSE; 
m_bAntipod		= FALSE;						// crossing 180

m_MapXtoLP	= MapXtoLP;
m_MapYtoLP	= MapYtoLP;

m_MapLPtoX  = MapLPtoX;
m_MapLPtoY	= MapLPtoY;

m_nLPSKx = 0;
m_nLPSKy = 0;
							 	// set scale text functions
m_ScaleTextX = TextX;
m_ScaleTextY = TextY;

m_szTitle.Empty();
m_nMode = nMode;
m_ptColor = ptInit->GetColorPtr();
m_ptMapDoc = ptInit->GetMapDocPtr();
}			  

/************************************************************************
 *  Map.cpp						~ C M a p					Destructor	*
 ************************************************************************/
CMap::~CMap()
{   
}
 
/****************************************************************************
 *	Map.cpp						S e t T i t l e								*
 ****************************************************************************/										 
void CMap::SetTitle (CString szTitle)	// for printing only
{
m_szTitle = szTitle;
}



/****************************************************************************
 *	Map.cpp							X Y t o L P								*
 ****************************************************************************/										 
void CMap::XYtoLP (double dXval, double dYval, long* LPx, long* LPy)
{
double	dAct, dMin;

dAct = (m_MapXtoLP) (dXval);
dMin = (m_MapXtoLP) (m_rLatLon.left);
*LPx = (long)floor(m_LPx0 + m_dFaktX * (dAct - dMin));
		
dAct = (m_MapYtoLP) (dYval);
dMin = (m_MapYtoLP) (m_rLatLon.bottom);
*LPy = (bMMText)? (long)floor(m_LPy0 - m_dFaktY * (dAct - dMin)) :
			      (long)floor(m_LPy0 + m_dFaktY * (dAct - dMin));
}                                // Mac: ...m_LPy0 - m_dFaktY...
								 // PC:		m_LPy0 + m_dFaktY...


/****************************************************************************
 *	Map.cpp							L P t o X Y								*
 ****************************************************************************/										 
void CMap::LPtoXY (long LPx, long LPy, double* ptXval, double* ptYval)
{								// get latitude
double dMin, dAct;

dMin = (m_MapXtoLP) (m_rLatLon.left);
dAct = dMin + (LPx - m_LPx0) / m_dFaktX;

*ptXval = (m_MapLPtoX)(dAct);


dMin = (m_MapYtoLP) (m_rLatLon.bottom);
dAct = (bMMText)? dMin + (m_LPy0 - LPy) / m_dFaktY :
				  dMin + (LPy - m_LPy0) / m_dFaktY;
*ptYval = (m_MapLPtoY)(dAct);
}

/****************************************************************************
 *	Map.cpp							X Y t o L P								*
 *  Purpose: use to get exact LPx LPy from Lat Lon							*
 ****************************************************************************/										 
void CMap::XYtoLP (double dXval, double dYval, double* LPx, double* LPy)
{
double	dAct, dMin;

dAct = (m_MapXtoLP) (dXval);
dMin = (m_MapXtoLP) (m_rLatLon.left);
*LPx = m_LPx0 + m_dFaktX * (dAct - dMin);
		
dAct = (m_MapYtoLP) (dYval);
dMin = (m_MapYtoLP) (m_rLatLon.bottom);
*LPy = (bMMText)? m_LPy0 - m_dFaktY * (dAct - dMin) :
			      m_LPy0 + m_dFaktY * (dAct - dMin);
}                                // Mac: ...m_LPy0 - m_dFaktY...
								 // PC:		m_LPy0 + m_dFaktY...

/****************************************************************************
 *	Map.cpp							L P t o X Y								*
 *  Purpose: use to get lat lon from calculated LPx LPy						*
 ****************************************************************************/										 
void CMap::LPtoXY (double LPx, double LPy, double* ptXval, double* ptYval)
{								// get latitude
double dMin, dAct;

dMin = (m_MapXtoLP) (m_rLatLon.left);
dAct = dMin + (LPx - m_LPx0) / m_dFaktX;

*ptXval = (m_MapLPtoX)(dAct);


dMin = (m_MapYtoLP) (m_rLatLon.bottom);
dAct = (bMMText)? dMin + (m_LPy0 - LPy) / m_dFaktY :
				  dMin + (LPy - m_LPy0) / m_dFaktY;
*ptYval = (m_MapLPtoY)(dAct);
}


/****************************************************************************
 *	Map.cpp					G e t L a b e l S i z e							*
 ****************************************************************************/										 
void CMap::GetLabelSize (double dGridX, double dGridY,
						short* ptLabelLP_X, short* ptLabelLP_Y)
{
CString szText;
CSize	SizeMin, SizeMax;

if (m_ScaleTextX == NULL)
	{
	*ptLabelLP_X	= 0;
	}
else{								/* get max X-axis label length			*/
	(m_ScaleTextX) (m_rLatLon.left, dGridX, &szText); 
	SizeMin = m_pDC->GetTextExtent (szText, szText.GetLength());
	(m_ScaleTextX) (m_rLatLon.right, dGridX, &szText);
	SizeMax = m_pDC->GetTextExtent (szText, szText.GetLength());
	*ptLabelLP_X = (short)((SizeMin.cx > SizeMax.cx)? SizeMin.cx : SizeMax.cx);
	}

if (m_ScaleTextY == NULL)
	{
	*ptLabelLP_Y = 0;
	}
else{							/* get max Y-axis label length			*/
	(m_ScaleTextY) (m_rLatLon.bottom, dGridY, &szText);
	SizeMin = m_pDC->GetTextExtent (szText, szText.GetLength());
	(m_ScaleTextY) (m_rLatLon.top, dGridY, &szText);
	SizeMax = m_pDC->GetTextExtent (szText, szText.GetLength());
	*ptLabelLP_Y = (short)((SizeMin.cx > SizeMax.cx)? SizeMin.cx : SizeMax.cx);
	}

if (!bMMText) *ptLabelLP_Y = -(*ptLabelLP_Y);
}

/****************************************************************************
 *	Map.cpp						G e t G r i d X								*
 ****************************************************************************/										 
double CMap::GetGridX (DRECT& rLatLon)
{
double fGridX;
int Lat0, Lat1, Lon0, Lon1;						/* rounded degrees			*/

Lat0 = (int)floor (rLatLon.bottom);				/* rounded degrees			*/
Lat1 = (int)floor (rLatLon.top);				/* degrees (+-180 ... +-90)	*/
Lon0 = (int)floor (rLatLon.left);
Lon1 = (int)floor (rLatLon.right);

fGridX = 0.1;
if ( abs(Lat1-Lat0) >  1  ||  abs(Lon1-Lon0) >  1) fGridX = 0.5;
if ( abs(Lat1-Lat0) >  5  ||  abs(Lon1-Lon0) >  5) fGridX = 1.;
if ( abs(Lat1-Lat0) > 10  ||  abs(Lon1-Lon0) > 10) fGridX = 5.;
if ( abs(Lat1-Lat0) > 50  ||  abs(Lon1-Lon0) > 50) fGridX = 10.;
if ( abs(Lat1-Lat0) > 90  ||  abs(Lon1-Lon0) > 90) fGridX = 20.;

return fGridX;
}

/****************************************************************************
 *	Map.cpp					G e t R e q u i r e d S i z e L P 				*
 ****************************************************************************/										 
void CMap::GetRequiredSizeLP (DRECT& rLatLon, double* ptReqWidthLP, double* ptReqHeightLP)
{									// get required LP area
DRECT	rReqLP;

rReqLP.right = (m_MapXtoLP) (rLatLon.right);
rReqLP.left  = (m_MapXtoLP) (rLatLon.left);

rReqLP.top   = (m_MapYtoLP) (rLatLon.top);
rReqLP.bottom= (m_MapYtoLP) (rLatLon.bottom);

									// PC: negativ value, MAC: dMax-dMin 
								//						PC:dMin- dMax
*ptReqWidthLP =  rReqLP.right - rReqLP.left;
*ptReqHeightLP = (bMMText)? rReqLP.top - rReqLP.bottom :
							rReqLP.bottom - rReqLP.top;
}

/****************************************************************************
 *	Map.cpp						T r a n s f o r m							*
 *	Default: nLPSKx = nLPSKy = 0											*
 ****************************************************************************/										 
void CMap::Transform (DRECT rMap, CRect& rView, BOOL bPrinter, short LPSKx, short LPSKy)	
{
long	LPx1, LPy1;
double	MaxLPx, MaxLPy;				/* max. Breite u. Hhe der Karte		*/
double	dReqHeightLP, dReqWidthLP;	/* in Rad								*/
double	dX, dY;						/* Size of map in 1/10mm (logical)		*/


m_rLatLon	= rMap;
m_rMaxLP	= rView;
m_bPrinter  = bPrinter;
m_nLPSKx	= LPSKx;
m_nLPSKy	= LPSKy;

this->GetRequiredSizeLP (m_rLatLon, &dReqWidthLP, &dReqHeightLP);

										// available LP area
MaxLPx = (double)(m_rMaxLP.right - m_rMaxLP.left - m_nLPSKx);
MaxLPy = (double)(m_rMaxLP.bottom - m_rMaxLP.top - m_nLPSKy);

m_dFaktX = MaxLPx/dReqWidthLP;
m_dFaktY = MaxLPy/dReqHeightLP;

if (m_nMode == ANISOTROPIC)
	{
	dX = m_dFaktX * dReqWidthLP;
	dY = m_dFaktY * dReqHeightLP;
	}

	
if (m_nMode == ISOTROPIC)
	{								// find fitting side
	if (bPrinter)
		{
		m_bXfits = (bMMText)? !(m_dFaktX * dReqHeightLP > MaxLPy) :
							!(m_dFaktX * dReqHeightLP < MaxLPy);
		}
	else{									/* screens width > height				*/
		m_bXfits = (m_dFaktY * dReqWidthLP > MaxLPx);
		}

	if (bPrinter)
		{									/* Printers page Width < Height			*/
		dY = m_dFaktX * dReqHeightLP;		/* PC: negative val		*/
		if (bMMText)
			{
			if (dY > MaxLPy) dY = MaxLPy;     // Mac: dY > MaxLPy
			}
		else{
			if (dY < MaxLPy) dY = MaxLPy;	  // PC:  dY < MaxLPy
			}

		m_dFaktY = dY / dReqHeightLP;		/* positive val				*/
		
		m_dFaktX = m_dFaktY;
		dX = m_dFaktX * dReqWidthLP;
		}
	else{									/* screens width > height				*/
		dX	= m_dFaktY * dReqWidthLP;
		if (dX > MaxLPx)
			dX = MaxLPx;
		m_dFaktX = dX / dReqWidthLP;
		
		m_dFaktY = m_dFaktX;
		dY = m_dFaktY * dReqHeightLP;
		}
	}
	
							// always centered output	
m_LPx0 = m_rMaxLP.left   + m_nLPSKx  + (MaxLPx - dX)/2;
m_LPy0 = m_rMaxLP.bottom - m_nLPSKy  - (MaxLPy - dY)/2;
	
this->XYtoLP (m_rLatLon.right, m_rLatLon.top, &LPx1, &LPy1);
							// left, top, right, bottom
SetRect (&m_rGridLP, (int)m_LPx0, LPy1, LPx1, (int)m_LPy0);
}

/****************************************************************************
 *	Map.cpp						U s e F u l l V i e w						*
 ****************************************************************************/										 
DRECT CMap::UseFullView (BOOL bXfits, DRECT rMapMin, CRect rView)
{
DRECT	rMapBorder;
double	dRight, dLeft, dBottom, dTop;

this->LPtoXY (rView.left, rView.top, &dLeft, &dTop);
this->LPtoXY (rView.right, rView.bottom, &dRight, &dBottom);

rMapBorder = rMapMin;

if (bXfits)
	{					// try to expand latitude
	double dLimit = (m_dGridY >= 10.)? 80. : 85.;
	if (fabs(dTop)	< dLimit)   rMapBorder.top = dTop;
	if (fabs(dBottom) < dLimit) rMapBorder.bottom = dBottom;
	}
else{					// expand longitude, no limit!
	rMapBorder.right = dRight;
	rMapBorder.left	 = dLeft;	
	}

return rMapBorder;
}

/****************************************************************************
 *	Map.cpp						E x p a n d		 							*
 ****************************************************************************/										 
BOOL CMap::Expand (DRECT rMapMin, CRect& rMaxLP, BOOL bPrinter, DRECT* ptVirtMap)
{	
double	fGridX, fGridY;

m_bAntipod		= FALSE;						// crossing 180

fGridY = fGridX = GetGridX (rMapMin);		/* grid for minimal map rect	*/

if (rMapMin.right - rMapMin.left < fGridX) 
	{								// avoid widht = 0
	rMapMin.right+=fGridX;
	rMapMin.left -=fGridX;
	}
if (rMapMin.top - rMapMin.bottom < fGridY) 
	{								// avoid height = 0
	rMapMin.top+=fGridY;
	rMapMin.bottom -=fGridY;
	}

	if (fGridY >= 10.)
	{
		rMapMin.top		= min (rMapMin.top, 80.);
		rMapMin.bottom	= max (rMapMin.bottom, -80.);
	}
	if (fGridY <= 5.)
	{
		rMapMin.top		= min (rMapMin.top, 85.);
		rMapMin.bottom	= max (rMapMin.bottom, -85.);
	}

	if (rMapMin.top - rMapMin.bottom <= 0)
		return FALSE;



if ((rMapMin.right - rMapMin.left) > 360.)
	{
	double dCenterLon = (rMapMin.right + rMapMin.left)/2;

	rMapMin.left = dCenterLon - 180.;
	rMapMin.right = dCenterLon + 180.;
	}


this->Transform (rMapMin, rMaxLP, bPrinter);   // sets  m _ r L a t L o n  <----------------

*ptVirtMap = this->UseFullView(m_bXfits, m_rLatLon, rMaxLP);

return TRUE;
}

/****************************************************************************
 *	Map.cpp						C a l c u l a t e 							*
 ****************************************************************************/										 
void CMap::Calculate (DRECT rMap, CRect& rMaxLP, BOOL bPrinter)
{	
m_rLatLon	= rMap;
m_rMaxLP	= rMaxLP;
m_bAntipod		= FALSE;						// crossing 180


if ((rMap.right - rMap.left) > 360.)
	{											// world surrounded
	double dCenterLon = (m_rLatLon.right + m_rLatLon.left)/2;

	rMap.left = dCenterLon - 180.;
	rMap.right = dCenterLon + 180.;
	}

if (rMap.right > 180.)
{
	MessageBeep((UINT)-1);
	m_bAntipod = TRUE;
}

this->Transform (rMap, rMaxLP, bPrinter);

											/* grid for full map view	*/
m_dGridX = m_dGridY = GetGridX (m_rLatLon);

m_bValid = TRUE; 
}

/****************************************************************************
 *	Map.cpp					C a l c u l a t e  G r a p h					*
 ****************************************************************************/										 
void CMap::CalculateGraph (CDC* pDC, DRECT rMap, CRect& rMaxLP, BOOL bPrinter, short nGridX, short nGridY)
{	
m_rLatLon	= rMap;
m_rMaxLP	= rMaxLP;
m_bAntipod		= FALSE;						// crossing 180
m_pDC	= pDC;


m_dGridX = nGridX;
m_dGridY = nGridY;
this->GetLabelSize (m_dGridX, m_dGridY, &m_nXLabelLP, &m_nYLabelLP);

short nLPSKx = m_nXLabelLP;				/* Width of Y-axis in 1/10mm			*/
short nLPSKy = m_nYLabelLP;			/* Height of X-axis in 1/10mm			*/
this->Transform (rMap, rMaxLP, bPrinter, nLPSKx, nLPSKy);


m_bValid = TRUE; 
}


/****************************************************************************
 *	Map.cpp						G e t S c a l e								*
 ****************************************************************************/										 
long CMap::GetScale (long lHorzResLPI)	
{
long lScale, dPixX;
double fDist, mLat, dLon;

dPixX = m_rGridLP.right - m_rGridLP.left;		// visible X-pixels
dLon  = m_rLatLon.right - m_rLatLon.left;		// visible lon degrees
mLat  = (m_rLatLon.top + m_rLatLon.bottom)/2;	// average lat degree
fDist = dLon * 60 * cos (mLat*pi/180);			// visible horizontal distance

lScale = (long)(fDist * 185000 * lHorzResLPI / (dPixX * 2.54));
return lScale;
}

/****************************************************************************
 *	Map.cpp					G e t L a t L o n B o r d e r					*
 ****************************************************************************/										 
void CMap::GetLatLonBorder (DRECT* ptLatLon)
{
memcpy (ptLatLon, &m_rLatLon, sizeof (DRECT));
}


/****************************************************************************
 *	Map.cpp						S c r o l l B o r d e r						*
 ****************************************************************************/
void CMap::ScrollBorder (CPoint pActScroll, CPoint pInitScroll)
{
double	dRight, dLeft, dBottom, dTop;

long lHorz = pActScroll.x - pInitScroll.x;
long lVert = pActScroll.y - pInitScroll.y;
this->LPtoXY (	m_rMaxLP.left  + lHorz,
				m_rMaxLP.top   + lVert, &dLeft, &dTop);
this->LPtoXY (	m_rMaxLP.right + lHorz, 
				m_rMaxLP.bottom+ lVert, &dRight, &dBottom);


if (lVert != 0)
	{					// try to expand latitude
	double dLimit = (m_dGridY >= 10.)? 80. : 85.;
	if (fabs (dTop)    <= dLimit) m_rLatLon.top    = dTop;
	if (fabs (dBottom) <= dLimit) m_rLatLon.bottom = dBottom;
	}

if (lHorz != 0)
	{					// expand longitude, no limit
	m_rLatLon.right = dRight;  
	m_rLatLon.left  = dLeft;   		
	}
}

/****************************************************************************
 *	Map.cpp						R o u n d U p Y   							*
 ****************************************************************************/										 
double CMap::RoundUpY (double dLat, double dGridY)								
{
double dRounded;
BOOL bNegativ = (dLat < 0.);

short nCount = (short)(dLat/dGridY);
if (!bNegativ) nCount++;

dRounded = nCount*dGridY;
return dRounded;
}

/****************************************************************************
 *	Map.cpp						R o u n d D o w n X							*
 ****************************************************************************/										 
double CMap::RoundDownX (double dLon, double dGridX)								
{
double dRounded;
BOOL bNegativ = (dLon < 0.);

short nCount = (short)(dLon / dGridX);
if (bNegativ) nCount--;

dRounded = nCount * dGridX;

return dRounded;
}


/****************************************************************************
 *	Map.cpp						D r a w G r i d  							*
 ****************************************************************************/										 
void CMap::DrawGrid (CDC* pDC, int CharX, int LineY, int* ptX, int* ptY)
{
long	LPx0, LPy0, LPx1, LPy1, UsedXPos;
double	dLatStart, dLonStart, dYval, dXval;
double	EPS = 0.0001;
CString szScale;

m_bDraw = TRUE;
m_pDC	= pDC;
m_CharX  = CharX;
m_LineY = LineY;

this->GetLabelSize (m_dGridX, m_dGridY, &m_nXLabelLP, &m_nYLabelLP);

if(m_bPrinter)
	{
	LPx0 = (m_rGridLP.right + m_rGridLP.left)/2;
	LPy0 = m_rGridLP.top - 3*m_LineY/2;
	TextAlignCenter (m_pDC, LPx0, LPy0, m_szTitle);
	}

					/* draw horizontal grid of map	*/
if (m_nLPSKy == 0)
	 dLatStart = this->RoundUpY (m_rLatLon.bottom, m_dGridY);								
else dLatStart = m_rLatLon.bottom;
for (dYval = dLatStart; dYval <= m_rLatLon.top+EPS; dYval += m_dGridY)
	{
	this->XYtoLP (m_rLatLon.right, dYval, &LPx0, &LPy0);
	m_pDC->MoveTo (LPx0, LPy0);
	this->XYtoLP (m_rLatLon.left, dYval, &LPx1, &LPy1);
	m_pDC->LineTo (LPx1, LPy1);

	m_pDC->LineTo (LPx1-m_CharX/3, LPy1);
	}	
	  
						/* draw vertical grid of map	*/
dLonStart = this->RoundDownX (m_rLatLon.right, m_dGridX);
for (dXval = dLonStart; dXval >= m_rLatLon.left-EPS; dXval -= m_dGridX)
	{
	this->XYtoLP (dXval, m_rLatLon.top, &LPx0, &LPy0);
	m_pDC->MoveTo (LPx0, LPy0);
	this->XYtoLP (dXval, m_rLatLon.bottom, &LPx1, &LPy1);
	m_pDC->LineTo (LPx1, LPy1);
	
	if (dXval == dLonStart ) 
		{
		UsedXPos = LPx0 + m_nXLabelLP;		/* init UsedXPos	*/
		/* first max x-pos will be LPx0 + 2*m_CharX. So first output is		*/
		/* not restricted by UsedXPos!										*/
		}
	
				/* is right x-pos of new value < left x-pos of old value ?	*/
	if (LPx0+m_nXLabelLP/2 < UsedXPos )
		{
		long Y;

		UsedXPos = LPx0 - m_nXLabelLP/2;
		
		if (m_bPrinter)
			{			// put longitude values below map area
			Y= m_rGridLP.bottom+m_LineY;
			m_pDC->LineTo (LPx1, LPy1+m_LineY/3);
			}
		else{
			if (m_nLPSKy != 0)
					 Y = m_rGridLP.bottom+m_LineY;
				else Y = m_rGridLP.bottom-m_LineY;
			if (m_nLPSKy != 0) 	m_pDC->LineTo (LPx1, LPy1+m_LineY/3);
			}

		(ScaleProcPtr)(m_ScaleTextX) (dXval, m_dGridX, &szScale);
		this->TextAlignCenter (m_pDC, LPx0, Y, szScale);
		}
	}

for (dYval = dLatStart; dYval <= m_rLatLon.top+EPS; dYval += m_dGridY)
	{						// draw lat scale after grid is completed
	long lYpos; 
	this->XYtoLP (m_rLatLon.right, dYval, &LPx0, &LPy0);
	(ScaleProcPtr)(m_ScaleTextY) (dYval, m_dGridY, &szScale);

	lYpos = LPy0 - m_LineY/2;
	if (m_bPrinter)
		{					// draw Y-Scale left of left map border
		this->TextAlignRight (m_pDC, m_rGridLP.left-m_CharX, lYpos, szScale);
		}
	else{
		if (m_nLPSKx != 0)	// draw Y-Scale left of left map border
			 this->TextAlignRight (m_pDC, m_rGridLP.left-m_nLPSKx/2, lYpos, szScale);
		else this->TextAlignLeft (m_pDC, m_rGridLP.left, lYpos, szScale);
		}					// draw Y-Scale right of left map border
	}

							// draw frame for maps having scale outside of map area
//if (m_bPrinter || m_nLPSKx != 0)
if (m_bPrinter)
	{
	CPen FramePen (PS_SOLID, 2, RGB (0, 0, 0));
	CPen* ptOldPen = (CPen*)m_pDC->SelectObject (&FramePen);

	m_pDC->MoveTo (m_rGridLP.left, m_rGridLP.top);
	m_pDC->LineTo (m_rGridLP.right, m_rGridLP.top);
	m_pDC->LineTo (m_rGridLP.right, m_rGridLP.bottom);
	m_pDC->LineTo (m_rGridLP.left, m_rGridLP.bottom);
	m_pDC->LineTo (m_rGridLP.left, m_rGridLP.top);

	m_pDC->SelectObject (ptOldPen);
	}
}

/****************************************************************************
 *	Map.cpp						D r a w L o c a t i o n						*
 ****************************************************************************/
void CMap::DrawLocation (CLocation& Loc, CSymbols& Symbols)
{
double dLat = Loc.GetLat();
double dLon = Loc.GetLon();
double dMapLon;

if (CLatLon::IsLatLonInRect (dLat, dLon, m_rLatLon, &dMapLon))
	{  
	long	LPx0, LPy0;
	CString	szName;
	                 
	this->XYtoLP (dMapLon, dLat, &LPx0, &LPy0);

	if (Symbols.Quality() < VQ_MEDIUM)
		{
		szName = Loc.GetIndicator();
		if (szName.GetLength()==0)
			szName = Loc.GetName();
		}
	else{
		szName = Loc.GetName();
		}

	short nType = Loc.GetCategory ();
	short nVar  = Loc.GetVariation ();	
						   
	switch (nType)
		{
		case WP_AIRPORT:
			if (m_ptMapDoc->IsActivated (SHOW_APT))
				Symbols.APT_Draw (LPx0, LPy0, szName, nVar, Loc.GetRwyDir(), Loc.IsGras()); 
			break;
		case WP_REPORT:
			if (m_ptMapDoc->IsActivated (SHOW_REP))
				Symbols.REP_Draw (LPx0, LPy0, szName); 
			break;
		case WP_INTERSECTION:
			if (m_ptMapDoc->IsActivated (SHOW_INT))
				Symbols.INT_Draw (LPx0, LPy0, Loc.GetIndicator()); 
			break;
		case WP_NDB:
			if (m_ptMapDoc->IsActivated (SHOW_NDB))
				Symbols.NDB_Draw (LPx0, LPy0, Loc.GetIndicator()); 
			break;
		case WP_VOR:
			if (m_ptMapDoc->IsActivated (SHOW_VOR))
				Symbols.VOR_Draw (LPx0, LPy0, szName, Loc.GetIndicator(), nVar); 
			break;
		case WP_USER:
			if (m_ptMapDoc->IsActivated (SHOW_USR))
				Symbols.USR_Draw (LPx0, LPy0, szName); 
			break;
		case WP_SAIL:
			if (m_ptMapDoc->IsActivated (SHOW_SAIL))
				Symbols.SAIL_Draw (LPx0, LPy0, szName, Loc.IsGras()); 
			break;
		case WP_HELI:
			if (m_ptMapDoc->IsActivated (SHOW_HELI))
				Symbols.HELI_Draw (LPx0, LPy0, szName, Loc.IsGras()); 
			break;
		}
	}          
}

/************************************************************************
 *  MapView.cpp	   			G e t L a b e l S t a r t L P 				*
 ************************************************************************/
void CMap::GetLabelStartLP(long LPx1, long LPy1, double dAngle, long LineY,
								   long* ptLPx, long* ptLPy)
{
	long lTextVektLen;
	double dRad;

	lTextVektLen = 2*LineY;			

	dRad = pi * dAngle / 180;
	if (bMMText) *ptLPx = LPx1 + (long)(sin (dRad) * lTextVektLen);
		else	 *ptLPx = LPx1 - (long)(sin (dRad) * lTextVektLen);
	*ptLPy = LPy1 - (long)(cos (dRad) * lTextVektLen);
}

/************************************************************************
 *  MapView.cpp	       			D r a w L a b e l 						*
 ************************************************************************/
void CMap::DrawLabel (long LPx1, long LPy1, double TextAngle, long LineY, 
				BOOL bRightAlign, CString szLoc, short nSizeLP, BOOL bActWpt)
{	
long	LPx2, LPy2, dY, dYtext;

GetLabelStartLP(LPx1, LPy1, TextAngle, LineY, &LPx2, &LPy2);


CPen DashPen (PS_DASH, 1, RGB (0, 0, 0));
CBrush* ptOldBrush;
CPen* ptOldPen = (CPen*)m_pDC->SelectObject (&DashPen);

CBrush brBackGround (bActWpt? m_ptColor->GetColMapSel() : m_ptColor->GetColMapLabel());
ptOldBrush = (CBrush*)m_pDC->SelectObject (&brBackGround);


int OldBkMode = m_pDC->SetBkMode (TRANSPARENT);

dY	= LineY / 3;							/* see: RectAlignXXXX		*/
dYtext = LineY / 6;

m_pDC->MoveTo (LPx1, LPy1);
m_pDC->LineTo (LPx2, LPy2+dY);											

if (bRightAlign)	
	{										/* true: show frame			*/
	this->RectAlignRight (m_pDC, LPx2, LPy2+dY, LineY, nSizeLP, TRUE);	
	this->TextAlignRight (m_pDC, LPx2, LPy2+dYtext, szLoc);
	}
else{										/* true: show frame			*/
	this->RectAlignLeft (m_pDC, LPx2, LPy2+dY, LineY, nSizeLP, TRUE);	
	this->TextAlignLeft (m_pDC, LPx2, LPy2+dYtext, szLoc);
	}
		
m_pDC->MoveTo (LPx1, LPy1);

m_pDC->SetBkMode (OldBkMode);
//m_pDC->SelectStockObject (WHITE_BRUSH);
m_pDC->SelectObject (ptOldBrush);
m_pDC->SelectObject (ptOldPen);
}	

/************************************************************************
 *  Map.cpp					C l i p										*
 ************************************************************************/
void CMap::Clip(BOOL bOn)
{
CRect rClientLP;				// client rect in logical pixel
if (bOn)	rClientLP = m_rGridLP;
	else	rClientLP = m_rMaxLP;

CRect rClientDP;
rClientDP = rClientLP;	
m_pDC->LPtoDP (&rClientDP);		// client rect in device pixel


if (m_pDC->IsKindOf (RUNTIME_CLASS(CPreviewDC)))
	{							// adjust clip rect for preview
	CPreviewDC* pPrevDC = (CPreviewDC*)m_pDC;

	pPrevDC->PrinterDPtoScreenDP (&rClientDP.TopLeft());
	pPrevDC->PrinterDPtoScreenDP (&rClientDP.BottomRight());

					// now offset the result by teh viewport
					// of the print preview window:
	CPoint ptOrg;
	::GetViewportOrgEx (m_pDC->m_hDC, &ptOrg);
	rClientDP += ptOrg;
	}


CRgn rClip;			// create the clip region and select it:
rClip.CreateRectRgn(rClientDP.left, rClientDP.top,
					rClientDP.right,rClientDP.bottom);
m_pDC->SelectClipRgn (&rClip);
}

/************************************************************************
 *  Map.cpp				T e x t A l i g n L e f t						*
 ************************************************************************/
void CMap::TextAlignLeft (CDC* pDC, long X, long Y, CString szText)
{ 
if (m_bDraw)
	{
	pDC->SetTextAlign (TA_LEFT | TA_TOP);
	pDC->TextOut (X, Y, szText);
	}
}

/************************************************************************
 *  Map.cpp				T e x t A l i g n C e n t e r					*
 ************************************************************************/
void CMap::TextAlignCenter (CDC* pDC, long X, long Y, CString szText)
{ 
if (m_bDraw)
	{
	pDC->SetTextAlign (TA_CENTER | TA_TOP);
	pDC->TextOut (X, Y, szText);
	}
}

/************************************************************************
 *  Map.cpp				T e x t A l i g n R i g h t						*
 ************************************************************************/
void CMap::TextAlignRight (CDC* pDC, long X, long Y, CString szText)
{ 
if (m_bDraw)
	{
	pDC->SetTextAlign (TA_RIGHT | TA_TOP);
	pDC->TextOut (X, Y, szText);
	}
}

/************************************************************************
 *  wDrTools.c		R e c t A l i g n L e f t			*
 ************************************************************************/
void CMap::RectAlignLeft (CDC* pDC, int Xleft, int Ybot, int LineY,
				short nSizeLP, BOOL bFrame)
{
short	X0, Y0, X1, Y1;
short	dX;

dX	= abs(LineY) / 2;

X0	= Xleft - dX;
X1	= Xleft + nSizeLP + dX;
Y0	= Ybot + LineY;
Y1	= Y0 - (short)(1.4*LineY);

if (bFrame)
    {
    pDC->Rectangle (X0, Y0, X1+1, Y1+1); /* left, top, right, bottom	*/
    }
else{
    CBrush* ptBrush;
    CRect    rErase;
    rErase.SetRect(X0, Y0, X1+1, Y1+1); /* left, top, right, bottom	*/
    ptBrush = (CBrush*)GetStockObject (WHITE_BRUSH);
    pDC->FillRect (rErase, ptBrush);
    }
}

/************************************************************************
 *  wDrTools.c		R e c t A l i g n R i g h t			*
 ************************************************************************/
void CMap::RectAlignRight (CDC* pDC, int Xright, int Ybot, int LineY,
				short nSizeLP, BOOL bFrame)
{
short	X0, Y0, X1, Y1;
short	dX;

dX	= abs(LineY) / 2;

X0	= Xright - nSizeLP - dX;
X1	= Xright + dX;
Y0	= Ybot + LineY;
Y1	= Y0 - (short)(1.4*LineY);

if (bFrame)
    {
    pDC->Rectangle (X0, Y0, X1+1, Y1+1); /* left, top, right, bottom	*/
    }
else{
    CBrush* ptBrush;
    CRect    rErase;
    rErase.SetRect(X0, Y0, X1+1, Y1+1); /* left, top, right, bottom	*/
    ptBrush = (CBrush*)GetStockObject (WHITE_BRUSH);
    pDC->FillRect (rErase, ptBrush);
    }
}



