// MapView.cpp: Implementierungsdatei
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include <afxpriv.h>	// for CPreviewDC class

#include <math.h>

#include "pf.h"
#include "..\CPPTOOLS\TxVwTool.h"	  // for CTextViewTool::Format
#include "..\Cpptools\image.h"			// GetBmpBytesPerRow

#include "InitDoc.h"
#include "TrackDoc.h"
#include "MetDoc.h"			    
#include "LocDoc.h"
#include "PlaneDoc.h"
#include "PlAltDlg.h"

#include "ElevDoc.h"			// GetElevCnt		
#include "WayDoc.h"
#include "BearDlg.h"			// for DistAddSorted
#include "MapView.h"    

#include "InterMap.h"

#include "..\CPPTOOLS\Vektor.h"                                       
#include "Symbols.h"

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

extern CInitDoc* 		ptInit;
extern CFontDoc* 		ptFontTool;
extern CDimDoc* 		ptDim;      

extern CLocDoc*			ptLocDoc;
extern CScanMapDoc*		ptScMpDoc;
extern CElevDoc*		ptElevDoc;
extern CTrackDoc*		ptTrackDoc;

extern BOOL bMMText;

/************************************************************************
 *  wNavFunc.c				W i R a d Y 								*
 *	Eingabe: Breitengrad (Lat) in Grad									*
 *	Return : Strecke vom Aequator zum betreffenden Breitengrad [Rad]	*
 ************************************************************************/
double WiRadY (double Lat)
{
double	help;
help = tan (Lat*pi/360. + pi/4); 	// for x>0, tan(x) is negative
help = log (fabs(help));           // therefore use Absol...
return help;
}

/****************************************************************************
 *	mDraw.c					L o n S c a l e T e x t X						*
 ****************************************************************************/
short LonScaleTextX (double dXval, double dXGrid, CString* ptScale)
{
short	nLen;
char	mDir;	
char	szScale[32];							/* map direction			*/
CString	szScaleFormat;                                                    

if (dXGrid <= 0.5)					/* Koord format: %4.1f%c, else %3.0f%c	*/
	 szScaleFormat.LoadString (MAP_KOORD1);
else szScaleFormat.LoadString (MAP_KOORD0);

if (dXval < L0_LON) dXval += 360.;
if (dXval > L1_LON) dXval -= 360.;
mDir = ' '; mDir = (char)((dXval > 0.)? 'E' : 'W');

nLen = sprintf (szScale, (LPCTSTR)szScaleFormat, fabs(dXval), mDir);
*ptScale = (CString)szScale;
return nLen;
}      

/****************************************************************************
 *	mDraw.c					L a t S c a l e T e x t Y  						*
 ****************************************************************************/
short LatScaleTextY (double dYval, double dYGrid, CString* ptScale)
{
short	nLen;
char	mDir;									/* map direction			*/
char	szScale[32];							/* map direction			*/
CString	szScaleFormat;

if (dYGrid <= 0.5)					/* Koord format: %4.1f%c, else %3.0f%c	*/
	 szScaleFormat.LoadString (MAP_KOORD1);
else szScaleFormat.LoadString (MAP_KOORD0);

mDir = ' '; mDir = (char) ((dYval > 0.)? 'N' : 'S');

nLen = sprintf (szScale, (LPCTSTR)szScaleFormat, fabs(dYval), mDir);
*ptScale = (CString)szScale;
return nLen;
}

/****************************************************************************
 *	MapView.cpp					L o n M a p X t o L P						*
 ****************************************************************************/
double LonMapXtoLP (double dXval)
{
double fRadX;
fRadX = dXval*pi/180.;
return fRadX;
}

/****************************************************************************
 *	MapView.cpp					L a t M a p Y t o L P						*
 ****************************************************************************/
double LatMapYtoLP (double dYval)
{
double fRadY;
fRadY = WiRadY(dYval);
return fRadY;
}

/****************************************************************************
 *	MapView.cpp					L o n M a p L P t o X 						*
 ****************************************************************************/
double LonMapLPtoX (double dLPx)
{
	double dValue = dLPx * 180./pi;
	return dValue;
}

/****************************************************************************
 *	MapView.cpp					L a t M a p L P t o Y						*
 ****************************************************************************/
double LatMapLPtoY (double dLPy)
{
	double dValue = 360. * (atan (exp(dLPy)) - pi/4) / pi;	// 50

//double dValue = 360. * (atan (exp(dLPy)) + pi/4) / pi;	// 230
	return dValue;
}




/////////////////////////////////////////////////////////////////////////////
// CMapView

IMPLEMENT_DYNAMIC(CMapView, CBitmapView)

BEGIN_MESSAGE_MAP(CMapView, CBitmapView)
	//{{AFX_MSG_MAP(CMapView)
	ON_WM_RBUTTONDOWN()
	ON_WM_MOUSEMOVE()
	ON_COMMAND(ID_MP_REGION         , OnMenuRegion)
	ON_COMMAND(ID_ME_CENTER			, OnMenuCenter)
	ON_COMMAND(ID_ME_LARGER			, OnMenuLarger)
	ON_COMMAND(ID_ME_SMALLER		, OnMenuSmaller)
	ON_COMMAND(ID_ME_AIRPORT	    , OnMenuAPT)
	ON_COMMAND(ID_ME_REPORT	        , OnMenuREP)
	ON_COMMAND(ID_ME_INTERSECT	    , OnMenuINT)
	ON_COMMAND(ID_ME_NDB		    , OnMenuNDB)
	ON_COMMAND(ID_ME_VOR		    , OnMenuVOR)
	ON_COMMAND(ID_ME_USR		    , OnMenuUSR)
	ON_COMMAND(ID_ME_SAIL		    , OnMenuSAIL)
	ON_COMMAND(ID_ME_HELI		    , OnMenuHELI)
	ON_COMMAND(ID_ME_MAP		    , OnMenuMAP)
	ON_COMMAND(ID_ME_ELEV		    , OnMenuELEV)
	ON_COMMAND(ID_ME_AIRSPACE		, OnMenuAirspace)
	ON_COMMAND(ID_WY_INS			, OnMenuInsert)
	ON_COMMAND(ID_WY_APP			, OnMenuAppend)
	ON_COMMAND(ID_WY_BEAR			, OnMenuBear)
	ON_COMMAND(ID_WY_DEL			, OnMenuDelete)
	ON_COMMAND(ID_WY_ACT			, OnMenuActivate)
	ON_COMMAND(ID_WY_PLALT			, OnMenuPlAlt)

	ON_COMMAND_RANGE(ID_ME_LOCEDIT_FIRST, ID_ME_LOCEDIT_LAST, OnMenuMeEditMapLoc)
	ON_COMMAND_RANGE(ID_ME_AIRSPACEEDIT_FIRST, ID_ME_AIRSPACEEDIT_LAST, OnMenuMeEditAirspace)

	ON_COMMAND_RANGE(ID_WY_INS_FIRST, ID_WY_INS_LAST, OnMenuInsMapLoc)
	ON_COMMAND_RANGE(ID_WY_APP_FIRST, ID_WY_APP_LAST, OnMenuAppMapLoc)
	ON_COMMAND_RANGE(ID_WY_BEAR_FIRST, ID_WY_BEAR_LAST, OnMenuBearMapNav)
	ON_COMMAND_RANGE(ID_WY_DEL_FIRST, ID_WY_DEL_LAST, OnMenuDelRteWpt)
	ON_COMMAND_RANGE(ID_WY_ACT_FIRST, ID_WY_ACT_LAST, OnMenuActRteWpt)
	ON_COMMAND_RANGE(ID_WY_PLALT_FIRST, ID_WY_PLALT_LAST, OnMenuPlAltRteWpt)
	ON_UPDATE_COMMAND_UI(ID_WY_INS, OnUpdateWyIns)
	ON_UPDATE_COMMAND_UI(ID_WY_APP, OnUpdateWyApp)
	ON_UPDATE_COMMAND_UI(ID_WY_BEAR, OnUpdateWyBear)
	ON_UPDATE_COMMAND_UI(ID_WY_DEL, OnUpdateWyDel)
	ON_UPDATE_COMMAND_UI(ID_WY_ACT, OnUpdateWyAct)
	ON_UPDATE_COMMAND_UI(ID_WY_PLALT, OnUpdateWyPlAlt)
	ON_WM_CONTEXTMENU()
	ON_WM_SIZE()
	ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
	ON_WM_LBUTTONDOWN()
	ON_COMMAND(ID_TB_CENTER, OnTBMapCenter)
	ON_COMMAND(ID_TB_LARGER, OnMenuLarger)
	ON_COMMAND(ID_TB_SMALLER, OnMenuSmaller)
	ON_WM_RBUTTONUP()
	ON_UPDATE_COMMAND_UI(ID_ME_AIRPORT, OnUpdateAirport)
	ON_UPDATE_COMMAND_UI(ID_ME_NDB, OnUpdateNdb)
	ON_UPDATE_COMMAND_UI(ID_ME_INTERSECT, OnUpdateIntersect)
	ON_UPDATE_COMMAND_UI(ID_ME_REPORT, OnUpdateReport)
	ON_UPDATE_COMMAND_UI(ID_ME_VOR, OnUpdateVor)
	ON_UPDATE_COMMAND_UI(ID_ME_USR, OnUpdateUsr)
	ON_UPDATE_COMMAND_UI(ID_ME_SAIL, OnUpdateSail)
	ON_UPDATE_COMMAND_UI(ID_ME_HELI, OnUpdateHeli)
	ON_UPDATE_COMMAND_UI(ID_ME_MAP, OnUpdateMapFrames)
	ON_UPDATE_COMMAND_UI(ID_ME_ELEV, OnUpdateElev)
	ON_UPDATE_COMMAND_UI(ID_ME_AIRSPACE, OnUpdateAirspace)
	//}}AFX_MSG_MAP
	// Standard printing commands
	ON_COMMAND(ID_FILE_PRINT, CBitmapView::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CBitmapView::OnFilePrintPreview)
END_MESSAGE_MAP()


/************************************************************************
 *	MapView.cpp						C M a p V i e w						*
 ************************************************************************/										 
CMapView::CMapView()
{
	m_bMore = FALSE;
	m_bDraw = FALSE;
	m_dX	= 0;
	m_dY	= 0;
	m_Ybottom	= 0;
	m_bForPrinter = FALSE;
	m_dwLinesPerPage = 0;

	ptFontTool->GetParams (m_FontName, &m_wPointSize);

	m_ptLaLoList = ptInit->GetLaLoListPtr();
	m_ptMapDoc = ptInit->GetMapDocPtr();
	m_bTBMapCenter = FALSE;

	m_ptMercator = new CMap (LonMapXtoLP, LatMapYtoLP,
							 LonMapLPtoX, LatMapLPtoY,
							LonScaleTextX, LatScaleTextY, 
							ISOTROPIC);

	m_bInitialView = TRUE;
	m_bCenterView = FALSE;

	m_dActFakt = 1.;
	m_nIncreaseCnt = 0;
	m_bScrollBarsOn = FALSE;
	m_dSFx = m_dSFy = 0.5;

	m_ptInterMap = NULL;
	m_ptInterMap = new CInteractiveMap (this);

	m_ptImage = new CImage (IMG_BMP); 			// replace the view's DIB
	if (m_ptImage == NULL)
	{
		AfxMessageBox ("Bad new CImage");
	}

	m_Target = FOR_SCREEN;

	m_bSeaCT = TRUE;			// used to create new image for storing into file
	m_bFlightCT = TRUE;
	m_lElev_m = 0;
	m_rClientLP.SetRect(0,0,0,0);
}

/************************************************************************
 *	MapView.cpp						~ C M a p V i e w	 				*
 ************************************************************************/										 
CMapView::~CMapView()
{
delete m_ptMercator;

	if (m_ptInterMap != NULL)
		delete m_ptInterMap;

	if (m_ptImage != NULL)
		delete m_ptImage;

}


/************************************************************************
 *	MapView.cpp					ActivateInitialView	 					*
 *  Passt Graphik an Route an und zentriert llToCenter, falls definiert	*
 ************************************************************************/										 
void CMapView::ActivateInitialView()
{
	m_bInitialView = TRUE;

	m_dActFakt = 1.;
	m_nIncreaseCnt = 0;
	m_bScrollBarsOn = FALSE;
	m_dSFx = m_dSFy = 0.5;
}

void CMapView::ActivateCenterView()
{
	m_bCenterView = TRUE;
}

                   
/////////////////////////////////////////////////////////////////////////////
// Aus CTextViewTool

/************************************************************************
 *  TxVwTool.cpp			Y _ P o i n t T o L P						*
 ************************************************************************/
short CMapView::Y_PointToLP (CDC* pDC, WORD wPointSize)
{
short	yLogPixPerInch;
short	yDevicePixels;
POINT	LP;

//  H [mm] = H [pt] *  25.4/72 [mm/pt]
//  H [mm] = H [IN] * 25.4 [mm/IN]
//  DP = H [IN] * yLogPixPerInch

yLogPixPerInch = pDC->GetDeviceCaps (LOGPIXELSY);
yDevicePixels = (int)((double)wPointSize * yLogPixPerInch / 72);

LP.x = 0;
LP.y = yDevicePixels;
pDC->DPtoLP((LPPOINT) &LP, 1);
return (short)LP.y;
}

/************************************************************************
 *  TxVwTool.cpp    	    C r e a t e N e w F o n t					*
 ************************************************************************/
BOOL CMapView::CreateNewFont (CFont* ptFont)
{    
BOOL	bOK = FALSE;
LOGFONT lfNewFont;

memset (&lfNewFont, 0, sizeof (LOGFONT));
lfNewFont.lfHeight = m_lfHeight;
lfNewFont.lfCharSet = DEFAULT_CHARSET;
strcpy (lfNewFont.lfFaceName, m_FontName);

bOK = ptFont->CreateFontIndirect ((LOGFONT FAR*)&lfNewFont);    
return bOK;
}

/************************************************************************
 *  TxVwTool.c					C h a n g e F o n t						*
 ************************************************************************/
CFont* CMapView::ChangeFont (CDC* pDC, CFont* ptNewFont, int* ptCapsX, int* ptLineY)
{
TEXTMETRIC	tm;
CFont* ptOldFont;
//int	CharX;

ptOldFont = pDC->SelectObject (ptNewFont);
                  
pDC->GetTextMetrics (&tm);
//CharX = tm.tmAveCharWidth;
*ptCapsX = tm.tmMaxCharWidth;

if (bMMText)	*ptLineY = tm.tmHeight + tm.tmExternalLeading;
	else		*ptLineY = -(tm.tmHeight + tm.tmExternalLeading);

return ptOldFont;
}

/************************************************************************
 *  TxVwTool.cpp			C h o o s e F o n t 						*
 ************************************************************************/
void CMapView::ChooseFont()
{
LOGFONT lfNewFont;
memset (&lfNewFont, 0, sizeof (LOGFONT));

CDC* pDC = GetDC();
pDC->SetMapMode (MM_LOMETRIC);

lfNewFont.lfHeight = this->Y_PointToLP (pDC, m_wPointSize);
ReleaseDC(pDC);

lfNewFont.lfCharSet = DEFAULT_CHARSET;
strcpy (lfNewFont.lfFaceName, m_FontName);

CFontDialog FontDlg(&lfNewFont, CF_BOTH, NULL, this);
int RetVal = FontDlg.DoModal();	
switch (RetVal)
	{
	case IDOK:
	//	FontDlg.m_cf.lpLogFont;		//LPLOGFONT
		m_wPointSize = (WORD)FontDlg.m_cf.iPointSize/10;
		strcpy (m_FontName, lfNewFont.lfFaceName); 
		strcpy (m_FontName, FontDlg.m_cf.lpLogFont->lfFaceName); 
		this->Invalidate(FALSE);
		break;
	case IDCANCEL:
		DWORD dwError = CommDlgExtendedError();
		break;
	}
}


/************************************************************************
 *  MapView.cpp					T e x t R o t a t e						*
 *  fAngle  < 0, right end moves up										*
 *  fAngle == 0, horizontal												*
 *  fAngle  > 0, right end moves down									*
 ************************************************************************/
void CMapView::TextRotate (CDC* pDC, long lX, long lY, double dAngle, CString szText)
{ 
	 short nAngleTenths = (short)(dAngle*10);

     CFont fntRotated;

     LOGFONT logFont;
     memset( &logFont, 0, sizeof(logFont));

     _tcscpy(logFont.lfFaceName, _T("Arial"));
     logFont.lfHeight = m_lfHeight;     
     logFont.lfUnderline = false;
     logFont.lfItalic = false;
     logFont.lfWeight = FW_NORMAL;

     logFont.lfOrientation = nAngleTenths;		// 450: turn right 45 degrees 
     logFont.lfEscapement = nAngleTenths;
     VERIFY(fntRotated.CreateFontIndirect(&logFont));

     CFont* pOldFont = (CFont*)pDC->SelectObject(&fntRotated);
//	 int OldBkMode = pDC->SetBkMode (TRANSPARENT);

     pDC->TextOut(lX, lY, szText);
	 
//	 pDC->SetBkMode (OldBkMode);
     pDC->SelectObject(pOldFont);
}

/************************************************************************
 *  TxVwTool.cpp			T e x t A l i g n L e f t					*
 ************************************************************************/
void CMapView::TextAlignLeft (CDC* pDC, short nX, short nY, CString szText)
{ 
if (m_bDraw)
	{
	pDC->SetTextAlign (TA_LEFT | TA_TOP);
	pDC->TextOut (nX, nY, szText);
	}
}

/************************************************************************
 *  TxVwTool.cpp			T e x t A l i g n C e n t e r				*
 ************************************************************************/
void CMapView::TextAlignCenter (CDC* pDC, short nX, short nY, CString szText)
{ 
if (m_bDraw)
	{
	pDC->SetTextAlign (TA_CENTER | TA_TOP);
	pDC->TextOut (nX, nY, szText);
	}
}

/************************************************************************
 *  TxVwTool.cpp			R s r c A l i g n C e n t e r E x			*
 ************************************************************************/
void CMapView::RsrcAlignCenterEx (CDC* pDC, short nX, short nY, short nID, ...)
{
CString szFormat;
char	szBuffer[256];   

szFormat.LoadString (nID);

va_list ptArgs;
va_start(ptArgs, nID);
short nLen = vsprintf (szBuffer, (LPCTSTR)szFormat, ptArgs);
va_end( ptArgs );

if (m_bDraw)
	{
	pDC->SetTextAlign (TA_CENTER | TA_TOP);
	pDC->TextOut (nX, nY, szBuffer, nLen);
	}
}

/************************************************************************
 *  Map.cpp					Y t o L a t 								*
 ************************************************************************/
void CMapView::YtoLat (double* ptLat)
{
if (*ptLat > 90.)	*ptLat = 180. - *ptLat;
if (*ptLat < -90.)	*ptLat = -(180. + *ptLat);
}

/************************************************************************
 *  Map.cpp					X t o L o n 								*
 ************************************************************************/
void CMapView::XtoLon (double* ptLon)
{
//if (*ptLon > 180.)	*ptLon = *ptLon - 360.;
//if (*ptLon < -180.)	*ptLon = 360 + *ptLon;

	double dEps = 1e-10;				// Rundungsfehler !!

	double dDiff = *ptLon - 180.;		// for *ptLon = 180 => 2.8421709430404e-014
	if (dDiff > dEps) *ptLon = *ptLon - 360.;

	dDiff = -180. - *ptLon;
	if (dDiff > dEps)	*ptLon = 360 + *ptLon;
}

/************************************************************************
 *  Map.cpp					L P t o L a t L o n							*
 ************************************************************************/
void CMapView::LPtoLatLon (long lLPx, long lLPy, double* ptLat, double* ptLon)
{
	m_ptMercator->LPtoXY (lLPx, lLPy, ptLon, ptLat);
	XtoLon	(ptLon);
}

/************************************************************************
 *  Map.cpp					L P t o L a t L o n							*
 *  Purpose: use to get lat lon from calculated LPx LPy					*
 ************************************************************************/
void CMapView::LPtoLatLon (double lLPx, double lLPy, double* ptLat, double* ptLon)
{
	m_ptMercator->LPtoXY (lLPx, lLPy, ptLon, ptLat);
	XtoLon	(ptLon);
}


/************************************************************************
 *  Map.cpp					D P t o L a t L o n							*
 ************************************************************************/
void CMapView::DPtoLatLon (CDC* pDC, long lDPx, long lDPy, double* ptLat, double* ptLon)
{
							   // convert DC to LC
	CPoint PosLP (lDPx, lDPy);		// lDPx=616, lDPy=397
	pDC->DPtoLP(&PosLP);			// PosLP.x=1630, PosLP.y=-1050 

	m_ptMercator->LPtoXY (PosLP.x, PosLP.y, ptLon, ptLat);
	XtoLon	(ptLon);
}


/////////////////////////////////////////////////////////////////////////////
// Zeichnung CMapView 

void CMapView::OnDraw(CDC* pDC)
{

	// TODO: add draw code for native data here            
//int	CharX, LineY;

//CFont NewFont; 
//CreateNewFont(&NewFont);
//CFont* ptOldFont = this->ChangeFont (pDC, &NewFont, &CharX, &LineY); 

DWORD 	dwIndex = 0;
this->DrawMap (pDC, &dwIndex);

//pDC->SelectObject (ptOldFont);
//NewFont.DeleteObject ();
}



/////////////////////////////////////////////////////////////////////////////
// Diagnose CMapView

#ifdef _DEBUG
void CMapView::AssertValid() const
{
	CBitmapView::AssertValid();
}

void CMapView::Dump(CDumpContext& dc) const
{
	CBitmapView::Dump(dc);
}

CWayDoc* CMapView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CWayDoc)));
	return (CWayDoc*)m_pDocument;
}
#endif //_DEBUG



/************************************************************************
 *  MapView.cpp					G e t M a x R e c t L P					*
 ************************************************************************/
void CMapView::GetMaxRectLP (CDC* pDC, CWnd* ptWnd, LPRECT lpRectMax, BOOL bPrinter)
{
					 //normal: CapsX = 52, LineY = 60
if (bPrinter)
    {
    int Width_pix, Height_lines;
    POINT   Offset, Pixels;

    Width_pix = pDC->GetDeviceCaps (HORZRES);	/* 960 pixels	*/
    Height_lines = pDC->GetDeviceCaps (VERTRES);	/* 841 lines	*/

    Pixels.x = Width_pix;
    Pixels.y = Height_lines;
	
	pDC->DPtoLP((LPPOINT)&Pixels, 1);   /* W=2032, H=-2967	    */
    Offset.x = (int)(0.05* Pixels.x);	    /* start 05% of pagewidth */
    Offset.y = (int)(0.08* Pixels.y);	    /* start 08% of pageheight*/
    Pixels.x = (int)(0.9 * Pixels.x);	    /* use 90% of pagewidth */
    Pixels.y = (int)(0.85* Pixels.y);	    /* use 85% of pageheight*/
    SetRect (lpRectMax, Offset.x, Offset.y, 
						Pixels.x, Pixels.y);
    }
else{					    /* use actual window size*/
    ptWnd->GetClientRect (lpRectMax);
	pDC->DPtoLP((LPPOINT)lpRectMax, 2);
    }
}  

/************************************************************************
 *  MapView.cpp				G e t U s e a b l e R e c t L P				*
 ************************************************************************/
CRect CMapView::GetUseableRectLP (CDC* pDC)
{
CRect rView, rBorderLP;

if (m_bForPrinter && pDC != NULL)
	{
	this->GetMaxRectLP (pDC, this, rView, m_bForPrinter);
	short nFrameX = (short)((double)(rView.right - rView.left) / 50);
	short nFrameY = (short)((double)(rView.bottom - rView.top) / 70);

	rBorderLP.top	= rView.top		+ 3*nFrameY;
	rBorderLP.left	= rView.left	+ 6*nFrameX;
	rBorderLP.right	= rView.right	- 0*nFrameX;
	rBorderLP.bottom = rView.bottom	- 1*nFrameY;
	}
else{
	CClientDC dc (this);
	dc.SetMapMode(MM_LOMETRIC);

	GetClientRect (rView);
	dc.DPtoLP((LPPOINT)&rView, 2);
	rBorderLP.top	= rView.top		;
	rBorderLP.left	= rView.left	;
	rBorderLP.right	= rView.right	;
	rBorderLP.bottom = rView.bottom	;
	}  

return rBorderLP;
}

/************************************************************************
 *  MapView.cpp    		 	D r a w S c a n n e d M a p s				*
 ************************************************************************/
void CMapView::DrawScannedMaps (CDC* pDC, DRECT rLatLon)
{
short i, j;
CColorDoc* ptColor = ptInit->GetColorPtr();

for (i=0; i<ptScMpDoc->GetMapCnt(); i++)
	{
	CScannedMap* ptMap = ptScMpDoc->GetMapPtr (i);
	if (ptMap != NULL)
		{
		CLatLon MapTL = ptMap->GetBorderLatLon(CALPOS_TL);
		CLatLon MapBR = ptMap->GetBorderLatLon(CALPOS_BR);

		if (CLatLon::IsLatLonInRect (MapBR.GetLat(), MapBR.GetLon(), rLatLon) ||
			CLatLon::IsLatLonInRect (MapTL.GetLat(), MapTL.GetLon(), rLatLon))
			{
			long lXstart, lYstart, lXTopRight, lYTopRight, lX, lY;
			CLatLon llBorder;
								  
			CPen* ptNewPen = new CPen (PS_SOLID, 5, ptColor->GetColMapFrame ());
			CPen* ptOldPen = pDC->SelectObject (ptNewPen);
			for (j=0; j<4; j++)
				{
				llBorder = ptMap->GetBorderLatLon(j);

				double dMapLon;
				CLatLon::GetMapLon(rLatLon.left, llBorder.GetLon(), rLatLon.right, &dMapLon);

				m_ptMercator->XYtoLP (dMapLon, llBorder.GetLat(), &lX, &lY);

				if (j==0)
					{
					lXstart = lX;
					lYstart = lY;
					pDC->MoveTo(lX, lY);
					}
				else{
					if (j==1)
						{
						lXTopRight = lX;
						lYTopRight = lY;
						}
					pDC->LineTo(lX, lY);

					if (j==2)
						{		// bottom right: draw first part of X
						pDC->MoveTo (lXstart, lYstart);
						pDC->LineTo(lX, lY);
						}

					if (j==3)
						{		// bottom right: draw first part of X
						pDC->MoveTo (lXTopRight, lYTopRight);
						pDC->LineTo(lX, lY);
						}
					}
				}
			pDC->LineTo(lXstart, lYstart);
			pDC->SelectObject (ptOldPen);	 
			delete ptNewPen;
			}
		}
	}
}


/************************************************************************
 *  MapView.cpp    		 	D r a w D r a w E l e v A r e a s 			*
 ************************************************************************/
void CMapView::DrawElevAreas (CDC* pDC, DRECT rLatLon)
{
	short i;
	CColorDoc* ptColor = ptInit->GetColorPtr();

	if (ptElevDoc != NULL)
	{
		short nStartInd, nEndInd;		// Elev Index
		CLatLon Map_TL (rLatLon.top, rLatLon.left);
		CLatLon Map_BR (rLatLon.bottom, rLatLon.right);

		if (ptElevDoc->GetMinMaxIndex (Map_TL, Map_BR, &nStartInd, &nEndInd))
		{
			CPen* ptNewPen = new CPen (PS_SOLID, 3, ptColor->GetColElevValue ());
			CPen* ptOldPen = pDC->SelectObject (ptNewPen);

			for (i=nStartInd; i<=nEndInd; i++)
			{							// for all Elevs of actual mercator map
				CElev Elev;
				if (ptElevDoc->GetElevPtr (i, &Elev))
				{
					CLatLon ElevBR = Elev.GetPosBR();
					CLatLon ElevTL = Elev.GetPosTL();
			
					double dRightLon, dLeftLon;
					BOOL bRightIn = CLatLon::IsLatLonInRect (ElevBR.GetLat(), ElevBR.GetLon(), rLatLon, &dRightLon);
					BOOL bLeftIn = CLatLon::IsLatLonInRect (ElevTL.GetLat(), ElevTL.GetLon(), rLatLon, &dLeftLon);
									// use help bools to avoid undefined dRightLon or dLeftLon
					if (bRightIn || bLeftIn)	// if statement doesn`t execute 2. statement, if 1. is TRUE!!
					{
						long lXstart, lYstart, lX, lY;

						m_ptMercator->XYtoLP (dLeftLon, ElevTL.GetLat(), &lX, &lY);

						lXstart = lX;
						lYstart = lY;
						pDC->MoveTo(lX, lY);			// top left

						m_ptMercator->XYtoLP (dRightLon, ElevTL.GetLat(), &lX, &lY);
						pDC->LineTo (lX, lY);			// top right

						m_ptMercator->XYtoLP (dRightLon, ElevBR.GetLat(), &lX, &lY);
						pDC->LineTo (lX, lY);			// bottom right

						Elev.DrawValue (pDC, lYstart, lXstart, lY, lX);

						m_ptMercator->XYtoLP (dLeftLon, ElevBR.GetLat(), &lX, &lY);
						pDC->LineTo (lX, lY);			// bottom left

						pDC->LineTo(lXstart, lYstart);	// top left
					}
				}
			}


			pDC->SelectObject (ptOldPen);	 
			delete ptNewPen; 

		}
	}
}

/****************************************************************************
 *	MapView.cpp				D i s t a n c e T o L P 	 					*
 *  Purpose: Converts a distance, specified by value and dimension to LP	*
 *  LP values do depend on position on earth, specified by dLat, dLon.		*
 ****************************************************************************/
long CMapView::DistanceToLP(double dLat, double dLon, double* LPx, double* LPy,
							double dDist, short nDistDim)
{
	long lRadius = 0;

	m_ptMercator->XYtoLP  (dLon, dLat, LPx, LPy);	// center of arc
	double dDist_NM = ptDim->ConvertDist (dDist, nDistDim, DIM_NM);

								// calc LatLon, dDist_NM north of center
	CLatLon llCenter(dLat, dLon);
	CLatLon llNorth;
	short nRWK = 0;
	llNorth.FromRelKoords (llCenter, nRWK, dDist_NM);
	double dXn, dYn;
	m_ptMercator->XYtoLP  (llNorth.GetLon(), llNorth.GetLat(), &dXn, &dYn);	// center of arc

	lRadius = (long)fabs(dYn - *LPy);
	return lRadius;
}

/****************************************************************************
 *	MapView.cpp				D i s t a n c e T o L P 	 					*
 *  Purpose: Converts a distance, specified by value and dimension to LP	*
 *  LP values do depend on position on earth, specified by dLat, dLon.		*
 ****************************************************************************/
long CMapView::DistanceToLP(double dLat, double dLon, long* LPx, long* LPy,
							double dDist, short nDistDim)
{
	long lRadius = 0;

	m_ptMercator->XYtoLP  (dLon, dLat, LPx, LPy);	// center of arc
	double dDist_NM = ptDim->ConvertDist (dDist, nDistDim, DIM_NM);

								// calc LatLon, dDist_NM north of center
	CLatLon llCenter(dLat, dLon);
	CLatLon llNorth;
	short nRWK = 0;
	llNorth.FromRelKoords (llCenter, nRWK, dDist_NM);
	long lXn, lYn;
	m_ptMercator->XYtoLP  (llNorth.GetLon(), llNorth.GetLat(), &lXn, &lYn);	// center of arc

	lRadius = abs(lYn - *LPy);
	return lRadius;
}

/****************************************************************************
 *	MapView.cpp				D r a w S e g m e n t s							*
 *  Purpose: draws all segments of an airspace,								*
 *  dLonOffset (360, -360) is used to draw on antipod maps					*			
 ****************************************************************************/
BOOL CMapView::DrawSegments (CDC* pDC, DRECT rLatLon, CAirspace* ptAirspace,
							CAirspaceSegmentDoc* ptSegDoc, long lStartIndex, double dLonOffset)
{
	BOOL bCompleteDrawn = TRUE;
	BOOL bDrawn = FALSE;
	CString szIdentifier;
	CString szText;
	short i = (short)lStartIndex;
	BOOL bSegmentsAvailable = (i>=0);		// segments available for current airspace
	BOOL bAntipodAirspace = ptAirspace->IsAntipod();

	while (bSegmentsAvailable)
	{      
		CAirspaceSegment* ptAirspaceSegment = NULL;
		
		if (ptSegDoc->GetAirspaceSegmentPtr (i, &ptAirspaceSegment))
		{
							// segment of same airspace identifier?
			if (i == lStartIndex)
			{
				szIdentifier = ptAirspaceSegment->GetIdentifier();
			}
			else
			{
				bSegmentsAvailable = (szIdentifier.Compare(ptAirspaceSegment->GetIdentifier()) == 0);
			}

			if (bSegmentsAvailable)
			{
				BOOL bLP_OK = TRUE;
				double dLat, dLon;
				long lX1, lY1, lX2, lY2, lX0, lY0, lRadius;
				lX1 = lY1 = lX2 = lY2 = lX0 = lY0 = lRadius = ptAirspaceSegment->GetUndefinedLong();

								// try to get all LP values
				dLat = ptAirspaceSegment->GetLat1();
				dLon = ptAirspaceSegment->GetLon1();
				if (dLat != NO_KOORD && dLon != NO_KOORD)
				{
					if (bAntipodAirspace)
					{
						if (dLon < 0)
							dLon += 360;
					}
					dLon += dLonOffset;

					m_ptMercator->XYtoLP (dLon, dLat, &lX1, &lY1);

					if (lX1 <= -32768 || lX1 > 32768 ||
						lY1 <= -32768 || lY1 > 32768)
						bLP_OK = FALSE;
				}

				dLat = ptAirspaceSegment->GetLat2();
				dLon = ptAirspaceSegment->GetLon2();
				if (dLat != NO_KOORD && dLon != NO_KOORD)
				{
					if (bAntipodAirspace)
					{
						if (dLon < 0)
							dLon += 360;
					}
					dLon += dLonOffset;

					m_ptMercator->XYtoLP  (dLon, dLat, &lX2, &lY2);				

					if (lX2 <= -32768 || lX2 > 32768 ||
						lY2 <= -32768 || lY2 > 32768)
						bLP_OK = FALSE;
				}

				dLat = ptAirspaceSegment->GetGeodeticLat0();
				dLon = ptAirspaceSegment->GetGeodeticLon0();

				if (dLat != NO_KOORD && dLon != NO_KOORD)
				{				// center of arc defined							.
					if (bAntipodAirspace)
					{
						if (dLon < 0)
							dLon += 360;
					}
					dLon += dLonOffset;

								// convert lat/lon and dRadius to LP here..
					double dRadius = ptAirspaceSegment->GetRadius();
					short nDistDim = ptAirspaceSegment->GetDistDim();
					lRadius = this->DistanceToLP(dLat, dLon, &lX0, &lY0, dRadius, nDistDim);

					if (lX0 <= -32768 || lX0 > 32768 ||
						lY0 <= -32768 || lY0 > 32768)
						bLP_OK = FALSE;
				}
				else
				{
					lX0 = 0;
					lY0 = 0;
					lRadius = 0;
				}
					
				if (bLP_OK)	
				{	// draw segment here...
						ptAirspaceSegment->Draw(pDC, lX1,lY1, lX2,lY2, lX0,lY0,lRadius);
						bDrawn = TRUE;
				}
				else	
				{
					bCompleteDrawn = FALSE;
				}
			}

			i++;
		}
		else
		{
			bSegmentsAvailable = FALSE;
		}
	}

	return (bDrawn && bCompleteDrawn);
}

/****************************************************************************
 *	MapView.cpp				D r a w A i r s p a c e s  						*
 ****************************************************************************/
void CMapView::DrawAirspaces (CDC* pDC, DRECT rLatLon, long lMapScale, long LineY)
{

	CAirspaceRegionsDoc* ptAirspaceRegionsDoc = ptInit->GetAirspaceRegionsDocPtr();
	if (ptAirspaceRegionsDoc != NULL)
	{
		short nRegionIndex = 0;
		short nAirspaceIndex = 0;
		CAirspace* ptAirspace = NULL;

		while (ptAirspaceRegionsDoc->GetAirspaceInRect(rLatLon, &ptAirspace, 
			&nRegionIndex, &nAirspaceIndex))
		{
			if (ptAirspace != NULL)
			{
				CAirspaceDoc* ptAirspaceDoc = ptAirspaceRegionsDoc->GetDocument();
				CAirspaceSegmentDoc* ptSegDoc = ptAirspaceDoc->GetSegmentDocPtr();


				long lStartIndex = ptAirspace->GetFirstSegmentIndex();

			//	CColorDoc* ptColor = ptInit->GetColorPtr();


						// color depends on kind of airpace
				CPen* ptNewPen = ptAirspace->CreatePen(pDC);
				CPen* ptOldPen = (CPen*)pDC->SelectObject (ptNewPen);


				double dLonOffset = 0.;
				if (lStartIndex >= 0)
				{			// to calculate temp. long values for airpsaces on antipod maps
					if (rLatLon.left < -180)
					{		// antipod map lon: -206 ... -138
						if (ptAirspace->GetMaxLon() > 0)
						{		// east lon airpace
							dLonOffset = -360;
						}
					}

					if (rLatLon.right > 180)
					{		// antipod map lon: 154 ... 222
						if (ptAirspace->GetMinLon() < 0)
						{		// west lon airspace
							dLonOffset = 360.;
						}
					}
				}

				BOOL bCompleteDrawn = DrawSegments(pDC, rLatLon, ptAirspace, ptSegDoc, lStartIndex, dLonOffset);

				if (bCompleteDrawn && m_ptMapDoc->IsActivated(SHOW_LABEL))
				{
					double dMaxSegCourse=0;
					CLatLon llMaxSegCenter = ptAirspaceDoc->GetMaxSegmentCenter(ptAirspace, lStartIndex, &dMaxSegCourse);
					double dAngle = ptAirspace->CourseToTextAngle(dMaxSegCourse);
					double dLabelAngle = ptAirspace->CourseToLabelPosition(dMaxSegCourse);

					long lX, lY;
					CString szLabel = ptAirspace->GetLabelText();
					double dLon = llMaxSegCenter.GetLon();
					if (ptAirspace->IsAntipod())
					{
						if (dLon < 0)
							dLon += 360;
					}
					dLon += dLonOffset;
					m_ptMercator->XYtoLP  (dLon, llMaxSegCenter.GetLat(), &lX, &lY);


					long lLabelX = lX;
					long lLabelY = lY;
					m_ptMercator->GetLabelStartLP(lX, lY, dLabelAngle, LineY/2, &lLabelX, &lLabelY);

					this->TextRotate(pDC, lLabelX, lLabelY, dAngle, szLabel);
				}

				pDC->SelectObject (ptOldPen);	 
				delete ptNewPen;

			}
		}
	}
}


/****************************************************************************
 *	MapView.cpp				L e g I s A n t i p o d   						*
 ****************************************************************************/
BOOL CMapView::LegIsAntipod (double dLon0, double dFlightAngle)
{
	BOOL bAntipod = FALSE;
											// dLon0 begin of leg
	double dLon1 = dLon0 + dFlightAngle;	// dLon1 end of leg

	if (dLon1 > m_rVirtualLatLon.right)
	{							// end of leg east of visible area
		double dLonTest = dLon1 - 360;
		if (dLonTest > m_rVirtualLatLon.left)
			bAntipod = TRUE;	// end of leg visible on west side
	}

	if (dLon1 < m_rVirtualLatLon.left)
	{							// end of leg west of visible area
		double dLonTest = dLon1 + 360;
		if (dLonTest < m_rVirtualLatLon.right)
			bAntipod = TRUE;	// end of leg visible on east side

	}
	return bAntipod;
}

/****************************************************************************
 *	MapView.cpp				C h a n g e M a p S i d e  						*
 ****************************************************************************/
void CMapView::ChangeMapSide (CDC* pDC, long LPx0, long LPy0, 
						double dWptLat, double dWptLon, 
						CPoint* ptLeaveSideLP, CPoint* ptEnterSideLP)
{

	long LPx1, LPy1;
	int X1_Border, X2_Border, Y_Border;
	float Stg;

									/* get LP of next locus			*/
	m_ptMercator->XYtoLP (dWptLon, dWptLat, &LPx1, &LPy1);

						// get LP of virtual map border
	long lRightLP, lLeftLP, lDummy;
	m_ptMercator->XYtoLP (m_rVirtualLatLon.left, 0., &lLeftLP, &lDummy);
	m_ptMercator->XYtoLP (m_rVirtualLatLon.right, 0., &lRightLP, &lDummy);
	X1_Border = (dWptLon > m_rVirtualLatLon.right)? lRightLP : lLeftLP;
	X2_Border = (dWptLon > m_rVirtualLatLon.right)? lLeftLP : lRightLP;

	if (LPx0 == LPx1) Stg = (float)0;		/* Y_Border = LPy0		*/
		else	Stg = (float)(LPy1-LPy0) / (LPx1-LPx0);
		
	Y_Border = (int)(Stg * (X1_Border-LPx0) + LPy0);


	ptLeaveSideLP->x = X1_Border;
	ptLeaveSideLP->y = Y_Border;

	ptEnterSideLP->x = X2_Border;
	ptEnterSideLP->y = Y_Border;
}

/****************************************************************************
 *	ElevArrayDoc.cp			G e n e r a t e W h i t e I m a g e 			*
 *  Purpose: Returns Ptr to 2-dimensional byte array and					*
 ****************************************************************************/
void CMapView::GenerateWhiteImage(LPSTR lpData, long lHeight, long lFileRowBytes)
{
	short nLine;
	BOOL bForMac = FALSE;

					// read rows of bitmap and store it in lpData
	for (int i=0; i< lHeight; i++)
	{					
				// PC: start reading first row of pixels
				// Mac: start reading last row of pixels
		nLine = (short)(bForMac? i : lHeight - 1 - i);

								// for every line of 1-2-4-8-bit-pixels:
		for (int j=0; j < lFileRowBytes; j++)
		{
			unsigned char pixel = (unsigned char)(255.0);
			*(lpData + nLine * lFileRowBytes + j) = (char)pixel;
		}		
	} // for line...
}


/****************************************************************************
 *	ElevArrayDoc.cp			G e n e r a t e E l e v I m a g e				*
 *  Purpose: if visible, opens elev array and generates image from it		*
 ****************************************************************************/
void CMapView::GenerateElevImage(LPSTR	lpData, CElevArray* ptElevArray, 
							 DRECT rMap, DRECT rElevArray, CRect rClientLP,
							 double dXImageResPerClientRes, double dYImageResPerClientRes,
							 long lFileRowBytes, long lPixHeight)
{
	DRECT rVisible;				// visible lat/lon area of elev array in rMap

						// compare rElevArray for antipod map with rMap
	rVisible.top	= min(rMap.top, rElevArray.top);
	rVisible.left	= max(rMap.left, rElevArray.left);
	rVisible.bottom	= max(rMap.bottom, rElevArray.bottom);
	rVisible.right	= min(rMap.right, rElevArray.right);

						// rVisible for antipod map
	if ((rVisible.top > rVisible.bottom) &&		// compare Lat/Lon values
		(rVisible.right > rVisible.left))
	{		// good visible area:

		if (ptElevArray->Open())
		{							// convert visible Lat/Lon to LP
						
			DRECT drVisibleLP;
			double dVisLPx, dVisLPy;
						// use special version of XYtoLP to get double LP values !!
			m_ptMercator->XYtoLP (rVisible.left, rVisible.top, &dVisLPx, &dVisLPy);
			drVisibleLP.top = dVisLPy;			// 11367
			drVisibleLP.left = dVisLPx;			// -5024

						// use special version of XYtoLP to get double LP values !!
			m_ptMercator->XYtoLP (rVisible.right, rVisible.bottom, &dVisLPx, &dVisLPy);
			drVisibleLP.bottom = dVisLPy;		// -10075
			drVisibleLP.right = dVisLPx;			// 8485
			
		
									// copy all elevs of opened ElevArray into bitmap
			short nLine;
			BOOL bForMac = FALSE;
									// convert visible LP to pixel rows
			short nRowStart,nRowEnd;
			nRowStart = (short)((drVisibleLP.top   - rClientLP.top)	* dYImageResPerClientRes + 0.5);
			nRowEnd   = (short)((drVisibleLP.bottom- rClientLP.top) * dYImageResPerClientRes + 0.5);	// after round up, be larger than lPixHeight
			if (nRowEnd > (short)lPixHeight)		// check limit of nRowEnd!
				nRowEnd = (short)lPixHeight;

			for (int i=nRowStart; i < nRowEnd; i++)
			{		
							// PC: start reading first row of pixels
							// Mac: start reading last row of pixels
				nLine = (short)(bForMac? i : lPixHeight - i - 1);
				if (nLine < 0) nLine = -nLine;	// positive image line number

					// convert row index into double LPy value !!
				double dActLPy = drVisibleLP.top  + (double)(i - nRowStart) / dYImageResPerClientRes;

					// convert LPx/LPy of actual line into Lat/Lon 
				double dActLat;
				double dStartLon, dEndLon;						// dActLat=50.46
							// get special version of LPtoLatLon to use double LP
				LPtoLatLon(drVisibleLP.left, dActLPy, &dActLat, &dStartLon);// dStartLon=7.436
				LPtoLatLon(drVisibleLP.right, dActLPy, &dActLat, &dEndLon);	// dEndLon=9.849

				// dStartLon and dEndLon are original lat lon values (West: -179 .... 0)

				short nColumnStart,nColumnEnd;
				nColumnStart= (short)((drVisibleLP.left  - rClientLP.left) * dXImageResPerClientRes + 0.5);
				nColumnEnd	= (short)((drVisibleLP.right - rClientLP.left) * dXImageResPerClientRes + 0.5);	// after round up, be larger than lFileRowBytes
				if (nColumnEnd > (short)lFileRowBytes)			// check limit of nColumnEnd
					nColumnEnd = (short)lFileRowBytes;


				ptElevArray->GenerateImageLine (lpData, lFileRowBytes,
							nLine, 
							nColumnStart, nColumnEnd,		// horizontal range of final image
							dActLat, dStartLon, dEndLon);
			} // for line...

	//		ptElevArray->SetActivated(FALSE);
			ptElevArray->Close();

		} // elevarray open
	} // good visible area
}


/****************************************************************************
 *	MapView.cpp			G e t R e c t F o r A n t i p o d M a p				*
 *  Purpose: for maps showing lon 180, west lon must be > 180!				*
 *  Input: Original rect of elev array (-180 ... 180) and					*
 *			rect of visible map,											*
 *			may be around lon 180, then west lon are > 180					*
 *  Output: unchanged, if rElevArray is visible on rMap,					*
 *			left and right, each +360, if this increased range is visible	*
 *			left and richt, each -360, if this decreased range is visible	*
 *	Example: for rElevArray (150E, 180E) and map (-217, ... 80) new 		*
 *			 rElevArray is  -210, -180										*
 ****************************************************************************/										 
BOOL CMapView::GetRectForAntipodMap (DRECT* ptElevArray, DRECT rVisibleLatLon)
{
	BOOL bChanged = FALSE;
	double dEps = 1e-10;
	

	short nCnt = (short)(rVisibleLatLon.left - ptElevArray->right) / 360;

					// test if map is visible for offset -360
	double dRightElevArrayBorder = ptElevArray->right + nCnt * 360;
						// left and right border of map < -180?
	BOOL bAntipodMapIncreased = (rVisibleLatLon.right < -180.);

						// left elev array border seen on rVisibleLatLon??
	if (((rVisibleLatLon.left <= dRightElevArrayBorder - dEps) && (dRightElevArrayBorder <= rVisibleLatLon.right - dEps)) ||
		bAntipodMapIncreased)		// OR both map borders > 180
	{					// increase lat for elev arrays right of 180
		ptElevArray->left += nCnt * 360;
		ptElevArray->right += nCnt * 360;
		bChanged = TRUE;
	}


	if (!bChanged)
	{
		short nCnt = (short)(rVisibleLatLon.right - ptElevArray->left) / 360;

							// test if map is visible for offset +360
		double dLeftElevArrayBorder = ptElevArray->left + nCnt * 360;

							// left and right border of map > 180?
		bAntipodMapIncreased = (rVisibleLatLon.left > 180.);

							// left elev array border seen on rVisibleLatLon??
		if (((rVisibleLatLon.left <= dLeftElevArrayBorder - dEps) && (dLeftElevArrayBorder <= rVisibleLatLon.right - dEps)) ||
			bAntipodMapIncreased)		// OR both map borders > 180
		{					// increase lat for elev arrays right of 180
			ptElevArray->left += nCnt * 360;
			ptElevArray->right += nCnt * 360;
			bChanged = TRUE;
		}
	}

	return bChanged;
}

/****************************************************************************
 *	MapView.cpp				C r e a t e 2 5 6 I m a g e	L P					*
 *  Purpose: Returns Ptr to 2-dimensional byte array						*
 ****************************************************************************/
LPSTR CMapView::Create256ImageLP(CRect rClientLP, DRECT rMap, 
								long lPixWidth, long lPixHeight,
								double dXImageResPerClientRes, double dYImageResPerClientRes,
								WORD* ptBitCount, LONG* ptFileRowBytes)
{
	LPSTR	lpData = NULL;
	DWORD	dwMemSize;

	if (lPixHeight < 0) lPixHeight = -lPixHeight;

	*ptBitCount = 8;
										// get number of bytes per row
	*ptFileRowBytes = GetBmpBytesPerRow (lPixWidth, *ptBitCount);	// 1023


	dwMemSize = (DWORD)lPixHeight * *ptFileRowBytes;			// 638976
	lpData = (LPSTR) new char[dwMemSize];
	if (lpData != NULL)
	{
		this->GenerateWhiteImage(lpData, lPixHeight, *ptFileRowBytes);

		CElevArrayDoc* ptElevArrayDoc = ptInit->GetElevArrayDocPtr();

//		ptElevArrayDoc->ActivateAll (TRUE);

		CElevArray* ptElevArray = NULL;
		DRECT rElevArray;			// lat/lon borders of elev array
		for (short index=0; index<ptElevArrayDoc->GetElevCnt(); index++)
		{
			if(ptElevArrayDoc->GetElevArrayPtr(index, &ptElevArray))
			{							// for all elevation areas:
			//	if (ptElevArray->GetName().CompareNoCase("Sibirien (Ost)") == 0)
				if (ptElevArray->GetName().CompareNoCase("Zentral Europa") == 0)
		//		if (ptElevArray->GetName().CompareNoCase("Sd Europa") == 0)
					int debug=1;


				if (ptElevArray->IsActivated())
				{						// get original area lat lon
					rElevArray.top = ptElevArray->GetLatMax();		// 60
					rElevArray.left = ptElevArray->GetLonMin();		// 0
					rElevArray.bottom = ptElevArray->GetLatMin();	// 40
					rElevArray.right = ptElevArray->GetLonMax();	// 20

					double dEps = 1e-10;
					if (rElevArray.left < rMap.left - dEps)
					{			// left border of elev array outside of rMap
						if (ptElevArray->IsAntipod(rMap))
						{		// elev array is seen at left and RIGHT border


										// convert rElevArray for antipod map, if necessary.
							DRECT rRightElevArray = rElevArray;
										// rRightElevArray contains orginal lat lon
							this->GetRectForAntipodMap (&rRightElevArray, rMap);
										// now rRightElevArray containts map lon for map at left border
							rRightElevArray.left += 360;
							rRightElevArray.right += 360;
										// now rRightelevArray contains map lon for map at RIGHT border


							this->GenerateElevImage(lpData, ptElevArray, 
											rMap, rRightElevArray, rClientLP,
											dXImageResPerClientRes, dYImageResPerClientRes,
											*ptFileRowBytes, lPixHeight);
						}
					}

							// convert rElevArray for antipod map, if necessary.
					this->GetRectForAntipodMap (&rElevArray, rMap);

							// here rElevArray contains lat lon for antipod map
					this->GenerateElevImage(lpData, ptElevArray, 
										rMap, rElevArray, rClientLP,
										dXImageResPerClientRes, dYImageResPerClientRes,
										*ptFileRowBytes, lPixHeight);
				} // activated
			} // good ptElevArray
		} // for all elev arrays

//		ptElevArrayDoc->ActivateAll (TRUE);
		
	}
return lpData;
}

/************************************************************************
 *  MapView.cpp    		 	G e t M o v i n g M a p A l t _ m			*
 ************************************************************************/
BOOL CMapView::GetMovingMapAlt_m(long* ptAlt_m)
{
	BOOL bShowMovingMapAlt = FALSE;
	short nLastTrkIndex;

	CMovingMapDoc* ptMovingMap = ptInit->GetMovingMapDocPtr();
	if (ptMovingMap != NULL)
	{
		if (ptMovingMap->IsRecording())
		{	
			if (ptTrackDoc != NULL)
			{
				nLastTrkIndex = ptTrackDoc->GetCnt()-1;
				if (nLastTrkIndex > 0)
				{
					if (ptTrackDoc->GetAltitude(nLastTrkIndex, DIM_METER, ptAlt_m))
					{	
						bShowMovingMapAlt = TRUE;	
					}
				}
			}
		}
		else
		{
			if(ptMovingMap->IsJustFinished())
				bShowMovingMapAlt = TRUE;		// show image for last used elev
		}
	}
	else
	{
		*ptAlt_m = 0;
	}

	return bShowMovingMapAlt;
}


/************************************************************************
 *  MapView.cpp    		 	 S a v e I m a g e  						*
 ************************************************************************/
BOOL CMapView::SaveImage()
{
	BOOL bOK = TRUE;
	
		// must be called before DrawElevImage to define actual rLatLon
	DRECT rLatLon;
	m_ptMercator->GetLatLonBorder (&rLatLon);		// returns m_rLatLon

	CDC* pDC = NULL;				// no DC required to save image
	CRect rViewLP;

	CRect rLP;						// get size of image from actual window dimensions
	rLP = m_ptMercator->GetGridRect ();
	long lWidth = rLP.right - rLP.left;
	long lHeight = rLP.bottom - rLP.top;
	rViewLP.SetRect(0,0,lWidth, lHeight);
	
										// make always same image size:
//	rViewLP.SetRect(0,0,1500, -1000);	// rViewLP: logical coordinates x=1500, y=-1000

	if (m_bSeaCT || m_bFlightCT)		// must be called AFTER ScrollBorder!!
		bOK = this->DrawElevImage(pDC, rViewLP, rLatLon, m_lElev_m, m_bSeaCT);	

	return bOK;
}



/************************************************************************
 *  MapView.cpp    		 	D r a w E l e v I m a g e					*
 *  pDC = NULL for m_Target = FOR_FILE									*
 ************************************************************************/
BOOL CMapView::DrawElevImage(CDC* pDC, CRect rViewLP, DRECT rMap, long lElev_m, BOOL bSeaCT)
{
	BOOL bOK = TRUE;

	WORD wBitCount=0; 
	LONG lFileRowBytes=0;
	LPSTR lpData = NULL;
	LONG	lWidth;
	LONG	lHeight;
	CRect rClientLP;

	long lPixPerMeterX;
	long lPixPerMeterY;

	rClientLP = rViewLP;				// rViewLP: logical coordinates x=1902, y=-1341
	CPoint ImageLinesPerInch;
	CPoint ClientLinesPerInch;

	m_Target = FOR_SCREEN;
	if (m_ptImage->IsToSave())
		 m_Target = FOR_FILE;
	if (m_bForPrinter)
		 m_Target = FOR_PRINTER;


	switch (m_Target)
	{
	case FOR_FILE:
		ImageLinesPerInch.x = 300;			// high Resolution of image
		ImageLinesPerInch.y = 300;
		
		ClientLinesPerInch.x = 254;
		ClientLinesPerInch.y = -254;
											// use size of window for image
		rClientLP = m_rClientLP;
		break;
	case FOR_PRINTER:
		ImageLinesPerInch.x = 96;			// low Resolution of image
		ImageLinesPerInch.y = 96;
											// number of pixels per logical inch
		ClientLinesPerInch.x = pDC->GetDeviceCaps (LOGPIXELSX);		// PC: x=96 Mac: 72
		ClientLinesPerInch.y = pDC->GetDeviceCaps (LOGPIXELSY);		// PC: y=96 Mac: 72
		pDC->DPtoLP(&ClientLinesPerInch);	// number of LP per logical inch

		break;
	case FOR_SCREEN:						// number of pixels per logical inch
		ImageLinesPerInch.x = pDC->GetDeviceCaps (LOGPIXELSX);		// PC: x=96 Mac: 72
		ImageLinesPerInch.y = pDC->GetDeviceCaps (LOGPIXELSY);		// PC: y=96 Mac: 72
		
		ClientLinesPerInch = ImageLinesPerInch;
		pDC->DPtoLP(&ClientLinesPerInch);	// number of logical pixels per logical inch
											// 300/-300
		m_rClientLP = rClientLP;
		break;
	}

							// ImageResPerClientRes = ImageRes[dpi]/ClientRes[dpi];			
	double dXImageResPerClientRes = (double)ImageLinesPerInch.x/ClientLinesPerInch.x;
	double dYImageResPerClientRes = (double)ImageLinesPerInch.y/ClientLinesPerInch.y;

											// define size of image (OK)
	lWidth =  (long)((rClientLP.right - rClientLP.left) * dXImageResPerClientRes);	// 1023	
	lHeight = (long)((rClientLP.bottom - rClientLP.top) * dYImageResPerClientRes);	// -624		
	if (lHeight < 0) lHeight = -lHeight;	// make height positive		624

	if (m_ptImage != NULL)
	{
											// create memory for lpData and
											// define whole image area (lpData)
		lpData = this->Create256ImageLP(rClientLP, rMap, 
									lWidth, lHeight,
									dXImageResPerClientRes, dYImageResPerClientRes,
									&wBitCount, &lFileRowBytes);
											// lpData memory will be freed in m_ptImage!

		lPixPerMeterX = (long)(100 * ImageLinesPerInch.x / 2.54);	// 3779
		lPixPerMeterY = (long)(100 * ImageLinesPerInch.y / 2.54);	// 3779

		if (lpData != NULL &&
			m_ptImage->Generate(lHeight, lWidth, 
								wBitCount, lFileRowBytes, 
								lPixPerMeterX, lPixPerMeterY, 
								lpData))	// deletes old memory and stores new lpData
		{
 			long lStartX, lStartY;  
			CSize	bmSize;


										// put actual color table into image
			CElevColorDoc* ptColorTableDoc ;
			if (bSeaCT)
				ptColorTableDoc = ptInit->GetElevColorDocPtr(CElevColorDoc::SEA_BASED);
			else
				ptColorTableDoc = ptInit->GetElevColorDocPtr(CElevColorDoc::FLIGHT_BASED);

			if (ptColorTableDoc != NULL)
			{
				short nCnt=0;
				RGBQUAD* ptRGBQuads = ptColorTableDoc->CreateRGBQuads(&nCnt, lElev_m);
				if (ptRGBQuads != NULL)
				{
					m_ptImage->SetColorTable(nCnt, ptRGBQuads);
					delete [] ptRGBQuads;
				}
			}


			switch (m_Target)
			{
			case FOR_FILE:
				if (m_ptImage->IsToSave())
				{
					m_ptImage->SaveImage(m_ptImage->GetFileToSave());
					m_ptImage->EmptyFileToSave();	// to avoid saving image again
				}
				break;

			case FOR_PRINTER:
				{
				lStartX = rClientLP.left;
				lStartY = rClientLP.top;

				long lFieldHeight = (long)(rClientLP.top   - rClientLP.bottom);
				long lFieldWidth  = (long)(rClientLP.right - rClientLP.left);
				CSize sDraw(lFieldWidth, -lFieldHeight);
			
				CPoint pStart = CPoint(lStartX, lStartY);
				m_ptImage->Stretch (pDC, pStart, sDraw);  // calls StretchDIBits(...)
				}
				break;
			case FOR_SCREEN:
				lStartX = (long)(rClientLP.left);
				lStartY = (long)(rClientLP.top);
				
				m_ptImage->SetROPCode (SRCAND);	 // transparent
				m_ptImage->Display(pDC, CPoint(lStartX, lStartY));
				break;
			}

			// hier vielleicht zu frh!! Erst, wenn view gewechselt oder neu aufgebaut wird!
			m_ptImage->DisposeImagePtr();		// generate new image every update
		}
		else
		{	// bad image generate
			AfxMessageBox("Bad image generate");
		}	
	} // m_ptImage != NULL

	return bOK;
}


/************************************************************************
 *  MapView.cpp    		 	 D r a w M a p	 							*
 ************************************************************************/
BOOL CMapView::DrawMap (CDC* pDC, DWORD* ptIndex)
{
BOOL	bEndOfPage = TRUE;
int		X, Y, LineY, CharX;
CString szText;

BOOL bDraw = TRUE;			 
ActivateDrawing (bDraw);

CFont PlainFont, BoldFont;
  
CRect rViewLP = this->GetUseableRectLP (pDC);
m_ptMapDoc->SetAborted(FALSE);

if (m_bInitialView)
	{				 // to use full view on screen
	DRECT rMinMap = m_Border.GetLatLonRect();

							// center location stored in init doc
	CLatLon LLtoCenter = ptInit->GetLatLonToCenter();
	if(LLtoCenter.IsValid())
		{
		rMinMap = m_Border.CenterBorderFor(LLtoCenter.GetLat(), LLtoCenter.GetLon());
		}


	if (m_ptMercator->Expand (rMinMap, rViewLP, m_bForPrinter, &m_rVirtualLatLon))
		{
		m_rVisibleLatLon = m_rVirtualLatLon;
		m_sVirt = this->GetVirtSize (m_nIncreaseCnt, rViewLP);
		m_bInitialView = FALSE;
		}
	else{
		m_ptMapDoc->SetAborted(TRUE);
		WPARAM wParam = MAKELONG (ID_VIEW_LAST, 0);
		(this->GetParent())->PostMessage(WM_COMMAND, wParam, 0);
		return bEndOfPage;
		}
	}


if (m_bCenterView)
	{	
							// center location stored in init doc
	CLatLon LLtoCenter = ptInit->GetLatLonToCenter();
	if(LLtoCenter.IsValid())
		{
		DoCenterLatLon(LLtoCenter.GetLat(), LLtoCenter.GetLon());
		}
	m_bCenterView = FALSE;
	}

if (m_bForPrinter)	 // use full printer area
	{		   // dont store m_rVirtualLatLon while printing!!
	CRect rViewLP = this->GetUseableRectLP(pDC);
	CSize sVirt = this->GetVirtSize (m_nIncreaseCnt, rViewLP);
	DRECT rMinMap = this->GetVisibleMapRect (m_rVirtualLatLon, sVirt, m_dSFx, m_dSFy);

   	DRECT rMap;
	if (m_ptMercator->Expand (rMinMap, rViewLP, m_bForPrinter, &rMap))
		{
		m_ptMercator->Calculate (rMap, rViewLP, m_bForPrinter);
		}
	else{
		m_ptMapDoc->SetAborted(TRUE);

		if (pDC->IsKindOf (RUNTIME_CLASS(CPreviewDC)))
		{							// for preview

		}
		else
		{							// no preview
			WPARAM wParam = MAKELONG (ID_VIEW_LAST, 0);
			(this->GetParent())->PostMessage(WM_COMMAND, wParam, 0);	// Zeigt MsgBox
		}
		return bEndOfPage;
		}
	}
else{
	m_ptMercator->Calculate (m_rVisibleLatLon, rViewLP, m_bForPrinter);
	}

			// diese drei Zeilen wurden von vor "CRect rViewLP" verschoben	
CreateNewFont(&PlainFont);
CFont* ptOldFont = this->ChangeFont (pDC, &PlainFont, &CharX, &LineY); 
Y = LineY;

													
long lMapScale = 0;
							// calculates map scale: 1 : XX
CPoint pHorzResDPI;
pHorzResDPI.x = pDC->GetDeviceCaps (LOGPIXELSX);
pHorzResDPI.y = pDC->GetDeviceCaps (LOGPIXELSY);
pDC->DPtoLP (&pHorzResDPI);
lMapScale = m_ptMercator->GetScale (pHorzResDPI.x);	
														

if (!m_bForPrinter)
	m_ptMercator->ScrollBorder (this->GetScrollPosition(), 
								m_pInitScroll);

					// must be called before DrawElevImage to define actual rLatLon
DRECT rLatLon;
m_ptMercator->GetLatLonBorder (&rLatLon);		// returns m_rLatLon


									// check which color table to use
CColorDoc* ptColor = ptInit->GetColorPtr();
m_bSeaCT	= ptColor->IsActivateSeaCT();
m_bFlightCT	= ptColor->IsActivateFlightCT();

				// check whether selected image is possible to show
	if (GetMovingMapAlt_m(&m_lElev_m))
	{			// elevation of track is available
		if (m_bFlightCT)
			m_bSeaCT = FALSE;		// show high terrain
	}
	else
	{			// elevation of track is NOT available
		m_bFlightCT = FALSE;
	}


if (m_bSeaCT || m_bFlightCT)		// must be called AFTER ScrollBorder!!
	this->DrawElevImage(pDC, rViewLP, rLatLon, m_lElev_m, m_bSeaCT);	
// ELSE: Mac has to draw white background here...

if (m_bForPrinter)
	{
	CString	szTitle;                                                    

	szTitle.Format(IDF_MAPTITLE,		// "%s 1:%d"
				(LPCTSTR)m_szMapTitle, lMapScale);
	m_ptMercator->SetTitle (szTitle);
	}

m_ptMercator->DrawGrid (pDC, CharX, LineY, &X, &Y);


CSymbols MapSymbols (pDC, m_bForPrinter, abs(LineY), lMapScale);
CLocation Loc;
if (MapSymbols.Quality() > VQ_NONE)
	{
	short nActRegIndex = 0;
	if (m_ptLaLoList->PrepareGetLoc (rLatLon))
		{
		while (m_ptLaLoList->GetLocationInRect (&Loc, rLatLon, &nActRegIndex))
			{
			m_ptMercator->DrawLocation (Loc, MapSymbols);
			}
		}
	}

m_ptMercator->Clip(TRUE);

if (m_ptMapDoc->IsActivated(SHOW_MAP))
	this->DrawScannedMaps (pDC, rLatLon);

if (m_ptMapDoc->IsActivated(SHOW_ELEV))
	this->DrawElevAreas (pDC, rLatLon);

if (m_ptMapDoc->IsActivated(SHOW_AIRSPACE))
	this->DrawAirspaces(pDC, rLatLon, lMapScale, LineY);

m_ptMercator->Clip(FALSE);

					// draws route or track	with virtual function
this->DrawOnMap (pDC, FALSE);

pDC->SelectObject (ptOldFont);
PlainFont.DeleteObject ();    

return bEndOfPage;
}

/////////////////////////////////////////////////////////////////////////////
// Behandlungsroutinen fr Nachrichten CMapView 

/************************************************************************
 *  MapView.cpp   		  O n P r e p a r e P r i n t i n g				*
 ************************************************************************/
BOOL CMapView::OnPreparePrinting(CPrintInfo* pInfo)
{			 // called before the display of the print dialog
	// default preparation
m_bMore = TRUE;	               

UINT nMinPage = 1;						// Number of the first page of the document
UINT nMaxPage = 1;						// Number of the last page of the document
pInfo->SetMinPage (nMinPage);
pInfo->SetMaxPage (nMaxPage);

return DoPreparePrinting(pInfo);
}


/************************************************************************
 *  MapView.cpp	  		 	OnBeginPrinting 							*
 ************************************************************************/
void CMapView::OnBeginPrinting(CDC* pDC, CPrintInfo* pInfo) 
{
	// TODO: Speziellen Code hier einfgen und/oder Basisklasse aufrufen
	
	CBitmapView::OnBeginPrinting(pDC, pInfo);
}

/************************************************************************
 *  MapView.cpp	  		 	OnEndPrinting 							*
 ************************************************************************/
void CMapView::OnEndPrinting(CDC* pDC, CPrintInfo* pInfo) 
{
	// TODO: Speziellen Code hier einfgen und/oder Basisklasse aufrufen
	
CBitmapView::OnEndPrinting(pDC, pInfo);
}

/************************************************************************
 *  MapView.cpp	  		 	OnPrepareDC 							*
 ************************************************************************/
void CMapView::OnPrepareDC(CDC* pDC, CPrintInfo* pInfo) 
{
	// TODO: Speziellen Code hier einfgen und/oder Basisklasse aufrufen
CWayDoc* pDoc = GetDocument();
ASSERT_VALID(pDoc);

if (pDoc != NULL)
	{	
	m_bForPrinter = pDC->IsPrinting();	  // important to get correct update after printing

	pDC->SetMapMode (MM_LOMETRIC);

	ptFontTool->GetParams (m_FontName, &m_wPointSize);
	m_lfHeight = this->Y_PointToLP (pDC, m_wPointSize);

	m_pScrollBarSizeLP.x = GetSystemMetrics (SM_CXHSCROLL);
	m_pScrollBarSizeLP.y = GetSystemMetrics (SM_CYHSCROLL);
	pDC->DPtoLP (&m_pScrollBarSizeLP);


	CBitmapView::OnPrepareDC(pDC, pInfo);
	   					
	if (pInfo != NULL)
	    {
		UINT nPages = 1;

		pInfo->SetMaxPage (nPages);
		pInfo->m_nNumPreviewPages = nPages;
		pInfo->m_bContinuePrinting = (m_bMore || pInfo->m_bPreview);
		}    
		
 	} 

// if (!m_bForPrinter)	
//	CBitmapView::OnPrepareDC(pDC, pInfo);
}


/************************************************************************
 *  MapView.cpp	  		 	H i d e S c r o l l B a r s 				*
 ************************************************************************/
void CMapView::HideScrollBars()
{
CSize totalSize (12, 12);
CSize pageSize (totalSize.cx, totalSize.cy);		// for page scroll
CSize lineSize (totalSize.cx/4, totalSize.cy/4);		// line scroll
SetScrollSizes (MM_LOMETRIC, totalSize, pageSize, lineSize);

m_bScrollBarsOn = FALSE;
}

/************************************************************************
 *  MapView.cpp	  		 	OnInitialUpdate 							*
 ************************************************************************/
void CMapView::OnInitialUpdate() 
{

HideScrollBars ();

	// TODO: Speziellen Code hier einfgen und/oder Basisklasse aufrufen
CBitmapView::OnInitialUpdate();
}

/************************************************************************
 *  MapView.cpp	  		 	ScrollToScaledPos 							*
 ************************************************************************/
void CMapView::ScrollToScaledPos (CRect rView)
{
CPoint  pScrollPos;			// change scrollbars
pScrollPos.x = (long)(m_dSFx * (m_sVirt.cx - rView.Width()));
pScrollPos.y = (long)(m_dSFy * (m_sVirt.cy - rView.Height()));

CSize sVirt = m_sVirt;
CSize pageSize (rView.Width(), rView.Height());		// page scroll
sVirt.cy    = abs(sVirt.cy);				// to avoid negative scroll size
pageSize.cy = abs(pageSize.cy);				// to avoid negative scroll size
CSize lineSize (pageSize.cx/4, pageSize.cy/4);		// line scroll
SetScrollSizes (MM_LOMETRIC, sVirt, pageSize, lineSize);

if (pScrollPos.x < 0) pScrollPos.x = 0;		// avoid negative values
if (bMMText)
	{
	if (pScrollPos.y < 0) pScrollPos.y = 0;		// avoid negative values
	}
else{
	if (pScrollPos.y > 0) pScrollPos.y = 0;		// avoid positive values
	}
m_bScrollBarsOn = TRUE;
ScrollToPosition (pScrollPos);			// logical 
m_pInitScroll = pScrollPos;
}

/************************************************************************
 *  MapView.cpp	  		 	GetScrollFaktors 							*
 ************************************************************************/
void CMapView::GetScrollFaktors (double* ptSFx, double* ptSFy,
							CSize sVirt, CRect rView)
{
if (m_bScrollBarsOn)
	{
	CPoint pScroll = GetScrollPosition();  // logical coordinates
	*ptSFx = (double)pScroll.x / (sVirt.cx - rView.Width());
	*ptSFy = (double)pScroll.y / (sVirt.cy - rView.Height());
	}
else{								// center of virtual rect
	*ptSFx = 0.5;
	*ptSFy = 0.5;
	}
}


/************************************************************************
 *  MapView.cpp	  		 	SearchDataBase 								*
 ************************************************************************/
BOOL CMapView::SearchDataBase (float fLat, float fLon, CWayPoint* ptWayPt)
{
BOOL	bFound = FALSE;
long lFirstIndex, lLastIndex;
BOOL	bMultNames = FALSE;
if (m_ptLaLoList->GetSortedIndexRange (fLat, fLon, &lFirstIndex, &lLastIndex))
	{
	short nActRegIndex;		   // dummy for GetLoc
	bMultNames = (lFirstIndex != lLastIndex);

	if (bMultNames)
		{
		CLocation FoundLoc;
		m_ptLaLoList->GetLoc (&FoundLoc, lFirstIndex, &nActRegIndex);
		ptWayPt->SetLoc (FoundLoc);
		}
	else{
		CLocation FoundLoc;
		m_ptLaLoList->GetLoc (&FoundLoc, lFirstIndex, &nActRegIndex);
		ptWayPt->SetLoc (FoundLoc);
		}


	bFound = TRUE;
	}
return bFound;
}

/************************************************************************
 *  MapView.cpp	  		 	O n M o u s e M o v e 						*
 ************************************************************************/
void CMapView::OnMouseMove(UINT nFlags, CPoint point) 
{
	// TODO: Code fr die Behandlungsroutine fr Nachrichten hier einfgen und/oder Standard aufrufen
	m_pMouseLP = point;

	CClientDC dc (this);
	dc.SetMapMode (MM_LOMETRIC);
	dc.DPtoLP (&m_pMouseLP);

	if (!m_ptMercator->IsValid())
		return;
							

	if (m_bTBMapCenter)
	{
		::SetCursor (::LoadCursor(NULL, IDC_CROSS));
	}
	else
	{								// is moving map running?
		BOOL bRecording = FALSE;
		CMovingMapDoc* ptMovingMap = ptInit->GetMovingMapDocPtr();
		if (ptMovingMap != NULL)
		{
			bRecording = ptMovingMap->IsRecording();
		}

		if (!bRecording)	// this check is also required in CScMpView::OnMouseMove!!
		{					// mouse move handling, only if moving map is NOT running!
			CRect rGridDP = m_ptMercator->GetGridRect();   // logical pixels
			rGridDP.right	+= 1;
			rGridDP.bottom += 1;
			dc.LPtoDP (&rGridDP);						// device pixels

			if (rGridDP.PtInRect(point))  // device pixels always positive (normalized)
			{
				double dLat, dLon;
				m_ptMercator->LPtoXY (m_pMouseLP.x, m_pMouseLP.y, &dLon, &dLat);
				XtoLon	(&dLon);
				
				m_ptInterMap->SetLatLon (dLat, dLon);
				m_ptInterMap->StartPosChangedTimer(10);		// 10 1/10 sec = 1 sec
			}
		}
	}

	CBitmapView::OnMouseMove(nFlags, point);
}

/****************************************************************************
 *	MapView.cpp				G e t V i r t S i z e  							*
 ****************************************************************************/										 
CSize CMapView::GetVirtSize (short nIncreaseCnt, CRect rView)
{
CSize sVirtLP;

double dBaseScaleFakt = 1.4142;

if (nIncreaseCnt == 0)
	{
	m_dActFakt = 1.;
	}
else{
	if (nIncreaseCnt > 0)
			m_dActFakt = dBaseScaleFakt * nIncreaseCnt;
	else	m_dActFakt = -1./(dBaseScaleFakt * nIncreaseCnt);
	}

sVirtLP.cx = (long)(m_dActFakt * rView.Width());
sVirtLP.cy = (long)(m_dActFakt * rView.Height());

return sVirtLP;
}

/****************************************************************************
 *	MapView.cpp					O n M e n u R e g i o n 					*
 ****************************************************************************/										 
void CMapView::OnMenuRegion()
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
if (ptLocDoc != NULL)
	{	        
	BOOL	bUseAirports = TRUE;

	ptInit->ResetLatLonToCenter();		// don't center any wpt of route
	ptLocDoc->GetMapBorder (&m_Border, bUseAirports);
	this->ResetScroll ();

	this->Invalidate(FALSE);
	}
}

/****************************************************************************
 *	MapView.cpp					R e c t A r o u n d							*
 ****************************************************************************/										 
CRect CMapView::RectAround (CPoint pFix, CSize sVirtLP, double dFakt)
{
CRect rMapLP;

long lTop		= pFix.y;
long lBottom	= sVirtLP.cy - pFix.y;
long lLeft		= pFix.x;
long lRight		= sVirtLP.cx - pFix.x;

rMapLP.top		= pFix.y - (long)((double)lTop / dFakt);
rMapLP.bottom	= pFix.y + (long)((double)lBottom / dFakt);
rMapLP.right	= pFix.x + (long)((double)lRight / dFakt);
rMapLP.left		= pFix.x - (long)((double)lLeft / dFakt);

return rMapLP;
} 

/****************************************************************************
 *	MapView.cpp					GetVisibleMapRect					*
 ****************************************************************************/										 
DRECT CMapView::GetVisibleMapRect (DRECT rVirtMap, CSize sVirtLP, 
				double dSFx, double dSFy)
{
DRECT rMap;
CPoint pFix;
pFix.x = (long)(dSFx * sVirtLP.cx);
pFix.y = (long)(dSFy * sVirtLP.cy);

CRect rMapLP = this->RectAround (pFix, sVirtLP, m_dActFakt); 

								//  use new virtual map to ...		 
CRect rVirt;		 
rVirt.SetRect (0, 0, sVirtLP.cx, sVirtLP.cy);
m_ptMercator->Calculate (rVirtMap, rVirt, FALSE);// never for printer!!

								// get visible map area (lat lon)
m_ptMercator->LPtoXY (rMapLP.left,  rMapLP.top, &rMap.left, &rMap.top);
m_ptMercator->LPtoXY (rMapLP.right, rMapLP.bottom, &rMap.right, &rMap.bottom);

return rMap;
		// update view and new calculation must follow now
}


/****************************************************************************
 *	MapView.cpp						ResetScroll							*
 ****************************************************************************/										 
void CMapView::ResetScroll ()
{
m_bInitialView = true;

m_nIncreaseCnt = 0;
m_dActFakt = 1.;	
m_pInitScroll = CPoint(0,0);  	  
m_dSFx = m_dSFy = 0.5;	// weglassen, wenn die letzte Scrolleinstellung
						// fr neue Scrollbars genutzt werden soll!

HideScrollBars();
}

/****************************************************************************
 *	MapView.cpp						C e n t e r L P							*
 *  Redefines m_rVirtualLatLon and m_rVisibleLatLon
 ****************************************************************************/										 
void CMapView::CenterLP(CPoint PosLP) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
CRect rView = this->GetUseableRectLP();

CPoint pActScrollPos;
pActScrollPos.x = (long)(m_dSFx * (m_sVirt.cx - rView.Width()));
pActScrollPos.y = (long)(m_dSFy * (m_sVirt.cy - rView.Height()));

//CPoint pScroll = GetScrollPosition();

CPoint pVirtMouse;
pVirtMouse.x = PosLP.x + pActScrollPos.x; 
pVirtMouse.y = PosLP.y + pActScrollPos.y; 

m_dSFx = m_dSFy = 0.5;
						 
			 // Offset to move virtual map rect LP = VirtMouse - Virt Center
long lOffsX = pVirtMouse.x - (LONG)(m_dSFx * m_sVirt.cx);
long lOffsY = pVirtMouse.y - (LONG)(m_dSFy * m_sVirt.cy);

CRect rVirtNew;
rVirtNew.top	= 0			 + lOffsY;
rVirtNew.left	= 0			 + lOffsX;
rVirtNew.right	= m_sVirt.cx + lOffsX;
rVirtNew.bottom = m_sVirt.cy + lOffsY;


CRect rVirt;		//  use old virtual map to ...		 
rVirt.SetRect (0, 0, m_sVirt.cx, m_sVirt.cy);
m_ptMercator->Calculate (m_rVirtualLatLon, rVirt, m_bForPrinter);

					// get lat lon border of new virtual map
double dLeft, dTop, dRight, dBottom;
m_ptMercator->LPtoXY (rVirtNew.left,  rVirtNew.top, 
				&dLeft, &dTop);
m_ptMercator->LPtoXY (rVirtNew.right, rVirtNew.bottom, 
				&dRight, &dBottom);

m_rVirtualLatLon.top = dTop;		// set new map border
m_rVirtualLatLon.left= dLeft;
m_rVirtualLatLon.right = dRight;
m_rVirtualLatLon.bottom = dBottom;

if (m_nIncreaseCnt > 0)
	{
	this->ScrollToScaledPos (rView);
	}

m_rVisibleLatLon = this->GetVisibleMapRect (m_rVirtualLatLon, m_sVirt, m_dSFx, m_dSFy);
}

/****************************************************************************
 *	Map.cpp				G e t L o n F o r A n t i p o d M a p				*
 *  Purpose: for maps showing lon 180, west lon must be > 180!				*
 *  Input: Original lon (-180 ... 180)										*
 *	Return: for dLon = 170 and map from -150 ... 180 ... 200 => -170		*
 *			for dLon =-170 and map from -150 ... 180 ... 200 => -190		*
 ****************************************************************************/										 
double CMapView::GetLonForAntipodMap (double dLon)
{
	short nOffset = 0;
	
	BOOL	bAntipod = (fabs(m_rVirtualLatLon.left) > 180 || fabs(m_rVirtualLatLon.right) > 180);
	if (bAntipod)
	{
//		short nCnt = (short)(m_rVirtualLatLon.right - dLon) / 360;
		short nCnt=1;
		if (dLon < m_rVirtualLatLon.left)
			nOffset = nCnt * 360;
		if (dLon > m_rVirtualLatLon.right)
			nOffset = -nCnt * 360;
	}

	return dLon + nOffset;
}


/****************************************************************************
 *	MapView.cpp					D o C e n t e r L a t L o n					*
 ****************************************************************************/										 
void CMapView::DoCenterLatLon(double dLat, double dLon) 
{
	dLon = this->GetLonForAntipodMap(dLon);

	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
	long lX, lY;
	m_ptMercator->XYtoLP (dLon, dLat, &lX, &lY);
	CPoint PosLP (lX, lY);
	this->CenterLP (PosLP);

	ptInit->SetLatLonToCenter(dLat, dLon);
	m_ptInterMap->SetLatLon(dLat, dLon);	// show act pos in status bar
	m_ptInterMap->StartPosChangedTimer(1);	// 1/10 sec
}


/****************************************************************************
 *	MapView.cpp						D o C e n t e r 						*
 ****************************************************************************/										 
void CMapView::DoCenter(double dLat, double dLon) 
{
	DoCenterLatLon(dLat, dLon);
	this->Invalidate(FALSE);
}

/****************************************************************************
 *	MapView.cpp					S e t B o r d e r F o r						*
 ****************************************************************************/										 
void CMapView::SetBorderFor (double dLat, double dLon)
{
	m_Border.SetBorderFor (dLat, dLon);
}

/****************************************************************************
 *	MapView.cpp						OnMenuCenter							*
 *  Purpose: called after mouse klicked on position to center				*
 ****************************************************************************/										 
void CMapView::OnMenuCenter() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen

	double dLat, dLon;				// get and store centered position
	m_ptMercator->LPtoXY(m_pMouseLP.x, m_pMouseLP.y, &dLon, &dLat);
	ptInit->SetLatLonToCenter(dLat, dLon);

	this->CenterLP (m_pMouseLP);
	this->Invalidate(FALSE);
}


/****************************************************************************
 *	MapView.cpp						OnMenuLarger							*
 ****************************************************************************/										 
void CMapView::OnMenuLarger() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
CRect rView = this->GetUseableRectLP();

m_nIncreaseCnt++;			// calc new sVirt
m_sVirt = this->GetVirtSize (m_nIncreaseCnt, rView);

if (m_sVirt.cx < 32768 && m_sVirt.cy > -32768)
	{
	if (m_nIncreaseCnt > 0)
		{
		if (!m_bScrollBarsOn)
			{	// rView will get scrollbars: reduced area available
			rView.right  -= m_pScrollBarSizeLP.x;
			rView.bottom -= m_pScrollBarSizeLP.y;
			}
		this->ScrollToScaledPos (rView);
		}

	m_rVisibleLatLon = this->GetVisibleMapRect (m_rVirtualLatLon, m_sVirt, m_dSFx, m_dSFy);

	this->Invalidate(FALSE);
	}
else{				// limit of short reached: stop increasing map !!
				// Shorts are used in LPtoDP ((LPPOINT)...) in ScrollToScaledPos in SetScrollSizes!!
	m_nIncreaseCnt--;
	m_sVirt = this->GetVirtSize (m_nIncreaseCnt, rView);
	MessageBeep (100);
	}
}	

/****************************************************************************
 *	MapView.cpp						OnMenuSmaller							*
 ****************************************************************************/										 
void CMapView::OnMenuSmaller() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
CRect rView = this->GetUseableRectLP();

m_nIncreaseCnt--;			// calc new sVirt
m_sVirt = this->GetVirtSize (m_nIncreaseCnt, rView);

if (m_nIncreaseCnt > 0)
	{
	this->ScrollToScaledPos (rView);
	}
else{						// deactivate scrollbars
	HideScrollBars();
	m_pInitScroll = CPoint(0,0);  	  
	}

m_rVisibleLatLon = this->GetVisibleMapRect (m_rVirtualLatLon, m_sVirt, m_dSFx, m_dSFy);

this->Invalidate(FALSE);
}			

void CMapView::OnMenuAPT()
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptMapDoc->InvertActivateState(SHOW_APT);
this->Invalidate(FALSE);
}
 	
void CMapView::OnMenuREP() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptMapDoc->InvertActivateState(SHOW_REP);
this->Invalidate(FALSE);
}

void CMapView::OnMenuINT() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptMapDoc->InvertActivateState(SHOW_INT);
this->Invalidate(FALSE);
}

void CMapView::OnMenuNDB() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptMapDoc->InvertActivateState(SHOW_NDB);
this->Invalidate(FALSE);
}

void CMapView::OnMenuVOR() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptMapDoc->InvertActivateState(SHOW_VOR);
this->Invalidate(FALSE);
}

void CMapView::OnMenuUSR() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptMapDoc->InvertActivateState(SHOW_USR);
this->Invalidate(FALSE);
}

void CMapView::OnMenuSAIL() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptMapDoc->InvertActivateState(SHOW_SAIL);
this->Invalidate(FALSE);
}

void CMapView::OnMenuHELI() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptMapDoc->InvertActivateState(SHOW_HELI);
this->Invalidate(FALSE);
}


void CMapView::OnMenuMAP() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptMapDoc->InvertActivateState(SHOW_MAP);
this->Invalidate(FALSE);
}

void CMapView::OnMenuELEV() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptMapDoc->InvertActivateState(SHOW_ELEV);
this->Invalidate(FALSE);
}

void CMapView::OnMenuAirspace() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptMapDoc->InvertActivateState(SHOW_AIRSPACE);
this->Invalidate(FALSE);
}



/************************************************************************
 *  MapView.cpp	  		 	O n S i z e		 							*
 ************************************************************************/
void CMapView::OnSize(UINT nType, int cx, int cy) 
{
	CBitmapView::OnSize(nType, cx, cy);
	
	// TODO: Code fr die Behandlungsroutine fr Nachrichten hier einfgen
CRect rView = this->GetUseableRectLP();
m_sVirt = this->GetVirtSize (m_nIncreaseCnt, rView);

if (m_nIncreaseCnt > 0)
	this->ScrollToScaledPos (rView);

m_rVisibleLatLon = this->GetVisibleMapRect (m_rVirtualLatLon, m_sVirt, m_dSFx, m_dSFy);
}

/************************************************************************
 *  MapView.cpp	  		 	O n S c r o l l	 							*
 ************************************************************************/
BOOL CMapView::OnScroll(UINT nScrollCode, UINT nPos, BOOL bDoScroll) 
{
	// TODO: Speziellen Code hier einfgen und/oder Basisklasse aufrufen

CRect rView = this->GetUseableRectLP();
this->GetScrollFaktors (&m_dSFx, &m_dSFy, m_sVirt, rView);

return CBitmapView::OnScroll(nScrollCode, nPos, bDoScroll);
}


/************************************************************************
 *  MapView.cpp	  		 O n M e n u M e E d i t M a p L o c			*
 ************************************************************************/
void CMapView::OnMenuMeEditMapLoc(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuMeEditMapLoc (nID);
ActivateInitialView();
}

/************************************************************************
 *  MapView.cpp	  		 O n M e n u M e E d i t A i r s p a c e 		*
 ************************************************************************/
void CMapView::OnMenuMeEditAirspace(UINT nID) 
{
	CWayDoc* ptWayDoc = GetDocument();
	m_ptInterMap->OnMenuMeEditAirspace (nID, ptWayDoc);
	ActivateInitialView();
}

/************************************************************************
 *  MapView.cpp	  		 	O n M e n u I n s M a p L o c				*
 ************************************************************************/
void CMapView::OnMenuInsMapLoc(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuInsMapLoc (nID);
ActivateInitialView();
}

/************************************************************************
 *  MapView.cpp	  		 	O n M e n u A p p M a p L o c				*
 ************************************************************************/
void CMapView::OnMenuAppMapLoc(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuAppMapLoc (nID);
ActivateInitialView();
}

/************************************************************************
 *  MapView.cpp	  		 	O n M e n u B e a r M a p N a v				*
 ************************************************************************/
void CMapView::OnMenuBearMapNav(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuBearMapNav (nID);
}

/************************************************************************
 *  MapView.cpp	  		 	O n M e n u D e l R t e W p t				*
 ************************************************************************/
void CMapView::OnMenuDelRteWpt(UINT nID)
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuDelRteWpt (nID);
}

/************************************************************************
 *  MapView.cpp	  		 	O n M e n u A c t R t e W p t				*
 ************************************************************************/
void CMapView::OnMenuActRteWpt(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuActRteWpt (nID);
}

/************************************************************************
 *  MapView.cpp	  	 	O n M e n u P l A l t R t e W p t				*
 ************************************************************************/
void CMapView::OnMenuPlAltRteWpt(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuPlAltRteWpt (nID);
}



/************************************************************************
 *  MapView.cpp	  		 	O n M e n u X X X							*
 ************************************************************************/
void CMapView::OnMenuAppend() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
OnMenuAppMapLoc(ID_WY_APP_FIRST);  // use first and only entry
}

void CMapView::OnMenuInsert() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
OnMenuInsMapLoc(ID_WY_INS_FIRST);  // use first and only entry
}

void CMapView::OnMenuBear() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
OnMenuBearMapNav(ID_WY_BEAR_FIRST);  // use first and only entry
}

void CMapView::OnMenuDelete() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
OnMenuDelRteWpt(ID_WY_DEL_FIRST);  // use first and only entry
}

void CMapView::OnMenuActivate() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
OnMenuActRteWpt(ID_WY_ACT_FIRST);  // use first and only entry
}

void CMapView::OnMenuPlAlt() 
{
	// TODO: Add your command handler code here
OnMenuPlAltRteWpt(ID_WY_PLALT_FIRST);  // use first and only entry	
}



/************************************************************************
 *  MapView.cpp	  		 	O n U p d a t e X X X						*
 ************************************************************************/

void CMapView::OnUpdateWyApp(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
m_ptInterMap->OnUpdateWyApp(pCmdUI);	
}

void CMapView::OnUpdateWyIns(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
m_ptInterMap->OnUpdateWyIns(pCmdUI);	
}

void CMapView::OnUpdateWyBear(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
m_ptInterMap->OnUpdateWyBear(pCmdUI);	
}

void CMapView::OnUpdateWyDel(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
m_ptInterMap->OnUpdateWyDel(pCmdUI);	
}

void CMapView::OnUpdateWyAct(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
m_ptInterMap->OnUpdateWyAct(pCmdUI);	
}

void CMapView::OnUpdateWyPlAlt(CCmdUI* pCmdUI) 
{
	// TODO: Code fr die Befehlsbehandlungsroutine zum Aktualisieren der Benutzeroberflche hier einfgen
m_ptInterMap->OnUpdateWyPlAlt(pCmdUI);	
}


/************************************************************************
 *  MapView.cpp	  		 O n U p d a t e x x x 							*
 ************************************************************************/

void CMapView::OnUpdateAirport(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
pCmdUI->SetCheck (m_ptMapDoc->IsActivated(SHOW_APT));
}

void CMapView::OnUpdateNdb(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
pCmdUI->SetCheck (m_ptMapDoc->IsActivated(SHOW_NDB));
}

void CMapView::OnUpdateIntersect(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
pCmdUI->SetCheck (m_ptMapDoc->IsActivated(SHOW_INT));
}

void CMapView::OnUpdateReport(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
pCmdUI->SetCheck (m_ptMapDoc->IsActivated(SHOW_REP));
}

void CMapView::OnUpdateVor(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
pCmdUI->SetCheck (m_ptMapDoc->IsActivated(SHOW_VOR));
}

void CMapView::OnUpdateUsr(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
pCmdUI->SetCheck (m_ptMapDoc->IsActivated(SHOW_USR));
}

void CMapView::OnUpdateSail(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
pCmdUI->SetCheck (m_ptMapDoc->IsActivated(SHOW_SAIL));
}

void CMapView::OnUpdateHeli(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
pCmdUI->SetCheck (m_ptMapDoc->IsActivated(SHOW_HELI));
}

void CMapView::OnUpdateMapFrames(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
BOOL bEnable = FALSE;

if (ptScMpDoc != NULL)
	bEnable = (ptScMpDoc->GetMapCnt() >0);

pCmdUI->SetCheck (m_ptMapDoc->IsActivated(SHOW_MAP));
pCmdUI->Enable(bEnable);	
}

void CMapView::OnUpdateElev(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
	BOOL bEnable = FALSE;

	if (ptElevDoc != NULL)
		bEnable = (ptElevDoc->GetElevCnt() > 0);

	pCmdUI->SetCheck (m_ptMapDoc->IsActivated(SHOW_ELEV));
	pCmdUI->Enable(bEnable);	
}

void CMapView::OnUpdateAirspace(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
	BOOL bEnable = FALSE;

	CAirspaceRegionsDoc* ptAirspaceRegionsDoc = ptInit->GetAirspaceRegionsDocPtr();
	if (ptAirspaceRegionsDoc != NULL)
		bEnable = (ptAirspaceRegionsDoc->GetCnt() > 0);

	pCmdUI->SetCheck (m_ptMapDoc->IsActivated(SHOW_AIRSPACE));
	pCmdUI->Enable(bEnable);	
}



void CMapView::OnRButtonDown(UINT nFlags, CPoint point) 
{
	// TODO: Code fr die Behandlungsroutine fr Nachrichten hier einfgen und/oder Standard aufrufen

CBitmapView::OnRButtonDown(nFlags, point);
}


/************************************************************************
 *  MapView.cpp	  		 	O n C o n t e x t M e n u					*
 ************************************************************************/
void CMapView::OnContextMenu(CWnd* pWnd, CPoint point) 
{
	// TODO: Code fr die Behandlungsroutine fr Nachrichten hier einfgen

short nVirtKey;			// If the high-order bit is 1, the key is down; 
						//	otherwise, it is up.

nVirtKey = GetKeyState(VK_CONTROL); 
if ((nVirtKey & 0x8000) == 0x8000)
	{
	m_ptInterMap->ShowMapPopupMenu (point);
	}
else{
	m_ptInterMap->ShowWayPopupMenu (point, GetDocument());
	}
}

/************************************************************************
 *  MapView.cpp	  		 	O n E d i t C o p y							*
 ************************************************************************/
void CMapView::OnEditCopy() 
{
	// TODO: Add your command handler code here
this->CopyToClipboard();		
}

/************************************************************************
 *  MapView.cpp	  		 	O n L B u t t o n D o w n					*
 ************************************************************************/
void CMapView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	// TODO: Add your message handler code here and/or call default
m_pMouseLP = point;

CClientDC dc (this);
dc.SetMapMode (MM_LOMETRIC);
dc.DPtoLP (&m_pMouseLP);

//CPoint pScrollLP = this->GetScrollPosition();

	CBitmapView::OnLButtonDown(nFlags, point);

	if (m_bTBMapCenter)
	{
		m_bTBMapCenter = FALSE;
		this->OnMenuCenter();
	}
	else
	{
		if (m_ptMapDoc->IsActivated(SHOW_MAP))
		{
			double  dLat, dLon;
							   // get mouse position
			m_ptMercator->LPtoXY (m_pMouseLP.x, m_pMouseLP.y, &dLon, &dLat);
			XtoLon	(&dLon);

			m_ptInterMap->OnShowScannedMap (dLat, dLon, this->GetParent(), m_ptMapDoc->GetCmdID());
		}
		else
		{
			m_ptInterMap->OnActivateLeg (GetDocument());
		}
	}
}

/************************************************************************
 *  MapView.cpp	  		 	O n T B M a p C e n t e r					*
 ************************************************************************/
void CMapView::OnTBMapCenter() 
{
	// TODO: Add your command handler code here
m_bTBMapCenter = TRUE;	
}

