/************************************************************************
 *  							P l a n e . c p p	  					*
 ************************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "math.h"					// floor

#include "pf.h"

#include "InitDoc.h"
#include "Plane.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

extern CDimDoc*		ptDim;
extern CCalcDoc*	ptCalc;

/////////////////////////////////////////////////////////////////////////////
// CPlane

IMPLEMENT_SERIAL(CPlane, CObject, 0 /* schema number*/ )

/************************************************************************
 *  Plane.cpp			 	 C P l a n e				Constructor		*
 ************************************************************************/
CPlane::CPlane(LPPLANE lpPlane)
{ 
if (lpPlane == NULL)
	{       
	this->FirstInit();
	} 
else{
    this->ReadFromPtr (lpPlane);
    }
}

CPlane::CPlane(CPlane& SourcePlane)
{ 
*this = SourcePlane;
}



/************************************************************************
 *  Plane.cpp			 ~ C P l a n e					Destructor		*
 ************************************************************************/
CPlane::~CPlane()
{
} 


void CPlane::GetPtr(LPPLANE lpPlane)
{
this->WriteToPtr (lpPlane);
}  

void CPlane::SetPtr (LPPLANE lpPlane)
{
this->ReadFromPtr(lpPlane);
}  

/************************************************************************
 *  Plane.cpp				operator=									*
 ************************************************************************/
const CPlane& CPlane::operator=(const CPlane& Plane)
{
_fmemcpy (m_szName, Plane.m_szName, SIZEOF_PLANE); 

_fmemcpy (&m_Speed, &Plane.m_Speed, sizeof (SPEEDTYPE)); 
_fmemcpy (&m_StLa, &Plane.m_StLa, sizeof (SLTYPE)); 
_fmemcpy (&m_Arms, &Plane.m_Arms, sizeof (ARMSTYPE)); 
_fmemcpy (&m_Tank, &Plane.m_Tank, sizeof (TANKTYPE)); 
m_nCost = Plane.m_nCost;
_fmemcpy (&m_Lim, &Plane.m_Lim, sizeof (COGLIMTYPE)); 
_fmemcpy (&m_FlPl, &Plane.m_FlPl, sizeof (PL_FLPL_TYPE)); 
m_lDummy = Plane.m_lDummy;
m_nDummy = Plane.m_nDummy;

return *this;
}

/************************************************************************
 *  Plane.cpp					F i r s t I n i t						*
 ************************************************************************/
void CPlane::FirstInit()
{
short	i;
//float	fNoArm = GetUndefArm (ptDim->Arm());	//ptDim may be NULL during initialization
float fNoArm = NO_ARM;

_fmemset (m_szName, 0, SIZEOF_PLANE);

m_Speed.nDistDim	= DIM_NM;
m_Speed.nAltDim		= DIM_FEET;
m_Speed.nVolDim		= DIM_LITER;

m_Speed.fClimbSpd	= (float)0;
m_Speed.fCruiseSpd	= (float)0;
m_Speed.fDescSpd	= (float)0;

m_Speed.fClimbCons	= (float)0;
m_Speed.fCruiseCons	= (float)0;
m_Speed.fDescCons	= (float)0;

m_Speed.fClimbRate	= (float)0;
m_Speed.fDescRate	= (float)0;


FirstFlPlInit (&m_FlPl);


m_StLa.nMassDim		= DIM_KG;
m_StLa.nLenDim		= DIM_METER;
m_StLa.nStartCnt	= 0;
m_StLa.nLandCnt		= 0;
for (i=0; i<PL_MAX_STLA; i++)
    {
    m_StLa.fStartMass[i] = (float)NO_MASS;
    m_StLa.fStartRoll[i] = (float)NO_RWY;
    m_StLa.fStartObst[i] = (float)NO_RWY;

    m_StLa.fLandMass[i] = (float)NO_MASS;
    m_StLa.fLandRoll[i] = (float)NO_RWY;
    m_StLa.fLandObst[i] = (float)NO_RWY;
    }

//m_Arms.nArmDim	= ptDim->Arm();
m_Arms.nArmDim  = DIM_METER;
m_Arms.nRowCnt	= 0;
m_Arms.nLugCnt	= 0;
for (i=0; i<PL_MAX_ARM; i++)
    {
    m_Arms.fRowArm[i]	= fNoArm;
    m_Arms.fLugArm[i]	= fNoArm;
    }

m_Tank.nVolDim	= DIM_LITER;
//m_Tank.nArmDim	= ptDim->Arm();
m_Tank.nArmDim  = DIM_METER;
m_Tank.nCnt		= 0;
for (i=0; i<PL_MAX_TANK; i++)
    {
    m_Tank.fArm[i]	= fNoArm;
    m_Tank.fUse[i]	= NO_FUEL;
    m_Tank.fUnuse[i]	= NO_FUEL;
    }

m_nCost		= 0;

m_Lim.nMassDim	= DIM_KG;
//m_Lim.nArmDim 	= ptDim->Arm();
m_Lim.nArmDim		= DIM_METER;
m_Lim.fEmptyMass	= (float)NO_MASS;
m_Lim.fEmptyTorq	= (float)NO_MASS;
m_Lim.nCnt		= 0;
for (i=0; i<PL_MAX_LIM; i++)
    {
    m_Lim.fMass[i]		= (float)NO_MASS;
    m_Lim.fFrntArm[i]	= fNoArm;
    m_Lim.fBackArm[i]	= fNoArm;
    }

m_lDummy = -1;
m_nDummy = -1;
}


/************************************************************************
 *  Plane.cpp			 G e t U n d e f A r m							*
 ************************************************************************/
float CPlane::GetUndefArm (short nArmDim)
{
double fConvert;

fConvert = ptDim->ConvertDist (NO_ARM, DIM_METER, nArmDim);
return (float)((short)fConvert);
}

/************************************************************************
 *  Plane.cpp				R e a d F r o m P t r						*
 ************************************************************************/
void CPlane::ReadFromPtr(LPPLANE lpPlane)
{
_fmemcpy (m_szName, lpPlane->szName, SIZEOF_PLANE); 

_fmemcpy (&m_Speed, &lpPlane->Speed, sizeof (SPEEDTYPE)); 
_fmemcpy (&m_StLa, &lpPlane->StLa, sizeof (SLTYPE)); 
_fmemcpy (&m_Arms, &lpPlane->Arms, sizeof (ARMSTYPE)); 
_fmemcpy (&m_Tank, &lpPlane->Tank, sizeof (TANKTYPE)); 
m_nCost = lpPlane->nCost;
_fmemcpy (&m_Lim, &lpPlane->Lim, sizeof (COGLIMTYPE)); 
_fmemcpy (&m_FlPl, &lpPlane->FlPl, sizeof (PL_FLPL_TYPE)); 
m_lDummy = lpPlane->lDummy;
m_nDummy = lpPlane->nDummy;
}

/************************************************************************
 *  Plane.cpp				W r i t e T o P t r							*
 ************************************************************************/
void CPlane::WriteToPtr(LPPLANE lpPlane) const
{
_fmemcpy (lpPlane->szName, m_szName, SIZEOF_PLANE); 
_fmemcpy (&lpPlane->Speed,&m_Speed, sizeof (SPEEDTYPE)); 
_fmemcpy (&lpPlane->StLa, &m_StLa, sizeof (SLTYPE)); 
_fmemcpy (&lpPlane->Arms, &m_Arms, sizeof (ARMSTYPE)); 
_fmemcpy (&lpPlane->Tank, &m_Tank, sizeof (TANKTYPE)); 
lpPlane->nCost = m_nCost;
_fmemcpy (&lpPlane->Lim, &m_Lim, sizeof (COGLIMTYPE)); 
_fmemcpy (&lpPlane->FlPl, &m_FlPl, sizeof (PL_FLPL_TYPE)); 
lpPlane->lDummy = m_lDummy;
lpPlane->nDummy = m_nDummy;
}

/************************************************************************
 *  Plane.cpp		 C o p y P l a n e 1 T o P l a n e					*
 ************************************************************************/
void CPlane::CopyPlane1ToPlane (PLANETYPE1* ptOldPlane)
{
this->FirstInit ();

strcpy (m_szName, ptOldPlane->szKennz);

m_Speed.nDistDim	= DIM_NM;
m_Speed.nAltDim	= DIM_FEET;
m_Speed.nVolDim	= DIM_LITER;
m_Speed.fClimbSpd	= (float)ptOldPlane->V_Steigfl;
m_Speed.fCruiseSpd	= (float)ptOldPlane->V_Reisefl;
m_Speed.fDescSpd	= (float)ptOldPlane->V_Reisefl;

m_Speed.fClimbCons	= ptOldPlane->Verbr2;
m_Speed.fCruiseCons	= ptOldPlane->Verbr1;
m_Speed.fDescCons	= ptOldPlane->Verbr1;

m_Speed.fClimbRate	= (float)ptOldPlane->V_Steig;
m_Speed.fDescRate	= (float)-ptOldPlane->V_Steig;

m_StLa.nMassDim	= DIM_KG;
m_StLa.nLenDim	= DIM_METER;
m_StLa.nStartCnt	= 1;		    /* max: 2	*/
m_StLa.fStartMass[0]	= ptOldPlane->M_Ges1;
m_StLa.fStartRoll[0]	= (float)NO_RWY;
m_StLa.fStartObst[0]	= (float)ptOldPlane->Start;
m_StLa.nLandCnt	= 1;		    /* max: 2	*/
m_StLa.fLandMass[0]	= ptOldPlane->M_Ges1;
m_StLa.fLandRoll[0]	= (float)NO_RWY;
m_StLa.fLandObst[0]	= (float)ptOldPlane->Land;

m_Arms.nArmDim	= DIM_METER;
m_Arms.nRowCnt	= 2;		    /* max: 5	*/
m_Arms.fRowArm[0]	= ptOldPlane->H_Reihe1;
m_Arms.fRowArm[1]	= ptOldPlane->H_Reihe2;
m_Arms.nLugCnt	= 2;		    /* max: 5	 */
m_Arms.fLugArm[0]	= ptOldPlane->H_Gep1;
m_Arms.fLugArm[1]	= ptOldPlane->H_Gep2;

m_Tank.nVolDim	= DIM_LITER;
m_Tank.nArmDim	= DIM_METER;
m_Tank.nCnt		= 1;		    /* max: 3	*/
m_Tank.fArm[0]	= ptOldPlane->H_Kraft;
m_Tank.fUse[0]	= ptOldPlane->aKraft;
m_Tank.fUnuse[0]	= ptOldPlane->naKraft;

m_nCost		= 0;

m_Lim.nMassDim	= DIM_KG;
m_Lim.nArmDim 	= DIM_METER;
m_Lim.fEmptyMass	= ptOldPlane->M_leer;
m_Lim.fEmptyTorq	= ptOldPlane->MO_leer;
m_Lim.nCnt		= 2;			/* max: 2   */
m_Lim.fMass[0]	= ptOldPlane->M_Ges1;
m_Lim.fFrntArm[0]	= ptOldPlane->H_1Min;
m_Lim.fBackArm[0]	= ptOldPlane->H_1Max;
m_Lim.fMass[1]	= ptOldPlane->M_Ges0;
m_Lim.fFrntArm[1]	= ptOldPlane->H_0Min;
m_Lim.fBackArm[1]	= ptOldPlane->H_0Max;

m_lDummy = -1;
m_nDummy = -1;
}  

/************************************************************************
 *  Plane.cpp				F i r s t F l P l I n i t					*
 ************************************************************************/
void CPlane::FirstFlPlInit (PL_FLPL_TYPE* ptFlPl)
{
_fmemset (ptFlPl->szIdent, 0, PL_SIZE_IDENT);
_fmemset (ptFlPl->szPlaneType, 0, PL_SIZE_PLTYPE);
_fmemset (ptFlPl->szEquipment, 0, PL_SIZE_EQUIP);
_fmemset (ptFlPl->szPlaneCol, 0, PL_SIZE_PLCOL);
ptFlPl->nSSR = SS_NONE;			
ptFlPl->bEmRadioUHF = FALSE;
ptFlPl->bEmRadioVHF = FALSE;
ptFlPl->bEmRadioELBA= FALSE;
}

/************************************************************************
 *  Plane.cpp    		 	S e t S t a r t L a n d D i m				*
 ************************************************************************/
void CPlane::SetStartLandDim(short nMassDim, short nLenDim)
{
m_StLa.nMassDim= nMassDim;
m_StLa.nLenDim	= nLenDim;
}

/************************************************************************
 *  Plane.cpp    		 		S e t S t a r t D a t a					*
 ************************************************************************/
void CPlane::SetStartData(short nIndex, float fMass, float fRoll, float fObst)
{
m_StLa.fStartMass[nIndex]	= fMass;
m_StLa.fStartRoll[nIndex]	= fRoll;
m_StLa.fStartObst[nIndex]	= fObst;
}

/************************************************************************
 *  Plane.cpp    		 		S e t L a n d D a t a					*
 ************************************************************************/
void CPlane::SetLandData(short nIndex, float fMass, float fRoll, float fObst)
{
m_StLa.fLandMass[nIndex]	= fMass;
m_StLa.fLandRoll[nIndex]	= fRoll;
m_StLa.fLandObst[nIndex]	= fObst;
} 

/************************************************************************
 *  Plane.cpp    		 	G e t S t a r t L a n d D i m 				*
 ************************************************************************/
void CPlane::GetStartLandDim(short* ptMassDim, short* ptLenDim)
{
*ptMassDim = m_StLa.nMassDim;
*ptLenDim  = m_StLa.nLenDim;
}

/************************************************************************
 *  Plane.cpp    		 	G e t S t a r t D a t a						*
 ************************************************************************/
void CPlane::GetStartData(short nIndex, float* ptMass, float* ptRoll, float* ptObst)
{
*ptMass = m_StLa.fStartMass[nIndex];
*ptRoll = m_StLa.fStartRoll[nIndex];
*ptObst = m_StLa.fStartObst[nIndex];
}


/************************************************************************
 *  Plane.cpp    		 	G e t L a n d D a t a						*
 ************************************************************************/
void CPlane::GetLandData(short nIndex, float* ptMass, float* ptRoll, float* ptObst)
{
*ptMass = m_StLa.fLandMass[nIndex];
*ptRoll = m_StLa.fLandRoll[nIndex];
*ptObst = m_StLa.fLandObst[nIndex];
} 


/************************************************************************
 *  Plane.cpp    		 		 SetLeverArmDim				*
 ************************************************************************/
void CPlane::SetLeverArmDim(short nArmDim)
{
m_Arms.nArmDim= nArmDim;
}

/************************************************************************
 *  Plane.cpp    		 		SetRowArm				*
 ************************************************************************/
void CPlane::SetRowArm(short nIndex, float fArm)
{
m_Arms.fRowArm[nIndex] = fArm;
}

/************************************************************************
 *  Plane.cpp    		 		SetLugArm				*
 ************************************************************************/
void CPlane::SetLugArm(short nIndex, float fArm)
{
m_Arms.fLugArm[nIndex] = fArm;
}
						 
/************************************************************************
 *  Plane.cpp    		 		 GetLeverArmDim				*
 ************************************************************************/
short CPlane::GetLeverArmDim()
{
return m_Arms.nArmDim;
}

/************************************************************************
 *  Plane.cpp    		 		GetRowArm				*
 ************************************************************************/
float CPlane::GetRowArm(short nIndex)
{
return m_Arms.fRowArm[nIndex];
}

/************************************************************************
 *  Plane.cpp    		 		GetLugArm			*
 ************************************************************************/
float CPlane::GetLugArm(short nIndex)
{
return m_Arms.fLugArm[nIndex];
}


/************************************************************************
 *  Plane.cpp    		 		 SetTankDim								*
 ************************************************************************/
void CPlane::SetTankDim(short nVolDim, short nArmDim)
{
m_Tank.nVolDim= nVolDim;
m_Tank.nArmDim= nArmDim;
}

/************************************************************************
 *  Plane.cpp    		 		SetTankData				*
 ************************************************************************/
void CPlane::SetTankData(short nIndex, float fUse, float fUnuse, float fArm)
{
m_Tank.fUse[nIndex]	= fUse;
m_Tank.fUnuse[nIndex]	= fUnuse;
m_Tank.fArm[nIndex]	= fArm;
}

/************************************************************************
 *  Plane.cpp    		 		 GetTankDim								*
 ************************************************************************/
void CPlane::GetTankDim(short* ptVolDim, short* ptArmDim)
{
*ptVolDim = m_Tank.nVolDim;
*ptArmDim = m_Tank.nArmDim;
}

/************************************************************************
 *  Plane.cpp    		 		G e t T a n k D a t a					*
 ************************************************************************/
void CPlane::GetTankData(short nIndex, float* ptUse, float* ptUnuse, float* ptArm)
{
*ptUse	 = m_Tank.fUse[nIndex];
*ptUnuse = m_Tank.fUnuse[nIndex];
*ptArm	 = m_Tank.fArm[nIndex];
}

/************************************************************************
 *  Plane.cpp    		 		 S e r i a l i z e						*
 ************************************************************************/
void CPlane::Serialize(CArchive& ar, short nReadVers)
{ 
if (ar.IsStoring())
	{
		// TODO: add storing code here 
	AnsiToOem(m_szName, m_szName);
	ar.Write (m_szName, SIZEOF_PLANE);


	ar.Write (&m_Speed, sizeof(SPEEDTYPE)); 
	ar.Write (&m_StLa, sizeof(SLTYPE)); 
	ar.Write (&m_Arms, sizeof(ARMSTYPE)); 
	ar.Write (&m_Tank, sizeof(TANKTYPE)); 
	ar.Write (&m_nCost, sizeof(short)); 
	ar.Write (&m_Lim, sizeof(COGLIMTYPE)); 
	ar.Write (&m_FlPl, sizeof (PL_FLPL_TYPE));
	ar << m_lDummy;	
    ar << (WORD)m_nDummy;
	}
else{ 					// TODO: add loading code here    
	WORD	Word;

	if (nReadVers == 1)    
		{ 		// loading verion 1 database
        PLANETYPE1 OldPlane;
        ar.Read (&OldPlane, sizeof(PLANETYPE1)); 
 		this->CopyPlane1ToPlane (&OldPlane);
 		FirstFlPlInit (&m_FlPl);
	    }  

 	if (nReadVers == 2)
	    {     
	    ar.Read (m_szName, SIZEOF_PLANE);   
 		ar.Read (&m_Speed, sizeof(SPEEDTYPE)); 
		ar.Read (&m_StLa, sizeof(SLTYPE)); 
		ar.Read (&m_Arms, sizeof(ARMSTYPE)); 
		ar.Read (&m_Tank, sizeof(TANKTYPE)); 
		ar.Read (&m_nCost, sizeof(short)); 
		ar.Read (&m_Lim, sizeof(COGLIMTYPE)); 
		FirstFlPlInit (&m_FlPl);
		ar >> m_lDummy;
		ar >> Word;		m_nDummy = (short)Word;
	    }
	    
 	if (nReadVers == ACT_PLANE_VERS)
	    {     
	    ar.Read (m_szName, SIZEOF_PLANE);   
 		ar.Read (&m_Speed, sizeof(SPEEDTYPE)); 
		ar.Read (&m_StLa, sizeof(SLTYPE)); 
		ar.Read (&m_Arms, sizeof(ARMSTYPE)); 
		ar.Read (&m_Tank, sizeof(TANKTYPE)); 
		ar.Read (&m_nCost, sizeof(short)); 
		ar.Read (&m_Lim, sizeof(COGLIMTYPE)); 
		ar.Read (&m_FlPl, sizeof(PL_FLPL_TYPE)); 
		ar >> m_lDummy;
		ar >> Word;		m_nDummy = (short)Word;
	    }
	
	OemToAnsi(m_szName, m_szName);
	}
} 


/************************************************************************
 *  Location.cpp    		 	C r e a t e P t r 						*
 ************************************************************************/
LPPLANE CPlane::CreatePtr ()
{
LPPLANE lpPlane = new PLANETYPE;

_fmemset (lpPlane, 0, sizeof (PLANETYPE));

WriteToPtr(lpPlane);

return lpPlane;
}

/************************************************************************
 *  Plane.cpp    		 		I s E q u a l							*
 ************************************************************************/
BOOL CPlane::IsEqual(const CPlane& Plane)
{
												// compare Name
if (_fstrcmp (m_szName, Plane.m_szName) != 0)	return FALSE; 

if (memcmp (&m_Speed, &Plane.m_Speed, sizeof (SPEEDTYPE)) != 0) return FALSE;
if (memcmp (&m_StLa, &Plane.m_StLa, sizeof (SLTYPE)) != 0) return FALSE;
if (memcmp (&m_Arms, &Plane.m_Arms, sizeof (ARMSTYPE)) != 0) return FALSE;
if (memcmp (&m_Tank, &Plane.m_Tank, sizeof (TANKTYPE)) != 0) return FALSE;

if (m_nCost != Plane.m_nCost)			return FALSE;

if (memcmp (&m_Lim, &Plane.m_Lim, sizeof (COGLIMTYPE)) != 0) return FALSE;
if (memcmp (&m_FlPl, &Plane.m_FlPl, sizeof (PL_FLPL_TYPE)) != 0) return FALSE;

if (m_lDummy != Plane.m_lDummy)			return FALSE;
if (m_nDummy != Plane.m_nDummy)			return FALSE;

return TRUE;
}      

/************************************************************************
 *  Plane.cpp    		 		I s E q u a l							*
 ************************************************************************/
BOOL CPlane::IsEqual(LPPLANE lpTestPlane)
{
CPlane TestPlane (lpTestPlane);
return (this->IsEqual (TestPlane));
}      

   
/************************************************************************
 *  Plane.cpp    		 		 S e t N a m e							*
 ************************************************************************/
void CPlane::SetName(CString szName)
{    
_fstrcpy (m_szName, (LPCTSTR)szName); 
}

/************************************************************************
 *  Plane.cpp    		 		 S e t M a s s							*
 ************************************************************************/
void CPlane::SetMass(short nIndex, float fMass)
{    
m_Lim.fMass[nIndex] = fMass; 
}

/************************************************************************
 *  Plane.cpp    		 	S e t F r o n t A r m						*
 ************************************************************************/
void CPlane::SetFrontArm(short nIndex, float fFrntArm)
{    
m_Lim.fFrntArm[nIndex] = fFrntArm; 
}

/************************************************************************
 *  Plane.cpp    		 	S e t B a c k A r m							*
 ************************************************************************/
void CPlane::SetBackArm(short nIndex, float fBackArm)
{    
m_Lim.fBackArm[nIndex] = fBackArm; 
}

/************************************************************************
 *  Plane.cpp    		 		 G e t M a s s							*
 ************************************************************************/
float CPlane::GetMass(short nIndex)
{    
return m_Lim.fMass[nIndex]; 
}

/************************************************************************
 *  Plane.cpp    		 	G e t F r o n t A r m						*
 ************************************************************************/
float CPlane::GetFrontArm(short nIndex)
{    
return m_Lim.fFrntArm[nIndex]; 
}

/************************************************************************
 *  Plane.cpp    		 	G e t B a c k A r m							*
 ************************************************************************/
float CPlane::GetBackArm(short nIndex)
{    
return m_Lim.fBackArm[nIndex]; 
}

/************************************************************************
 *  Plane.cpp    		 		 G e t N a m e							*
 ************************************************************************/
CString CPlane::GetName()
{         
CString szName = (CString)m_szName;
return szName;
}

/************************************************************************
 *  Plane.cpp    		 		 G e t N a m e							*
 ************************************************************************/
/*BOOL CPlane::GetName(LPSTR lpName)
{         
_fstrcpy (lpName, (LPSTR)m_szName);    
return (*m_szName > 0);
}	*/



/************************************************************************
 *  Plane.cpp    		 		 G e t S p e e d D i m 					*
 ************************************************************************/
short CPlane::GetSpeedDim()
{
short nSpeedDim = ptDim->DistDimToSpeedDim(m_Speed.nDistDim);
return nSpeedDim;
}  

/************************************************************************
 *  Plane.cpp    		 		 G e t C l i m b D i m 					*
 ************************************************************************/
short CPlane::GetClimbDim()
{
short nSpeedDim = ptDim->AltDimToClimbDim(m_Speed.nAltDim);
return nSpeedDim;
}	  

/************************************************************************
 *  Plane.cpp    		 		 S e t S p e e d D i m 					*
 ************************************************************************/
void CPlane::SetSpeedDim(short nDim)
{
short nDistDim = ptDim->SpeedDimToDistDim (nDim);
m_Speed.nDistDim = nDistDim;
}

/************************************************************************
 *  Plane.cpp    		 		 S e t C l i m b D i m 					*
 ************************************************************************/
void CPlane::SetClimbDim(short nDim)
{
short nAltDim = ptDim->ClimbDimToAltDim (nDim);
m_Speed.nAltDim = nAltDim;
}

/************************************************************************
 *  Plane.cpp			G e t H o r z S p e e d _ k t					*
 ************************************************************************/
void CPlane::GetHorzSpeed_kt (float* ptVstFl, float* ptVrFl, float* ptVsiFl)
{
short nSpeedDim = ptDim->DistDimToSpeedDim (m_Speed.nDistDim);

*ptVstFl = (float)ptDim->ConvertSpeed (m_Speed.fClimbSpd,nSpeedDim, DIM_KT);
*ptVrFl = (float)ptDim->ConvertSpeed (m_Speed.fCruiseSpd,nSpeedDim, DIM_KT);
*ptVsiFl = (float)ptDim->ConvertSpeed (m_Speed.fDescSpd, nSpeedDim, DIM_KT);
}

/************************************************************************
 *  Plane.cpp			G e t V e r t S p e e d _ f t p m				*
 ************************************************************************/
void CPlane::GetVertSpeed_ftpm (float* ptVst, float* ptVsi)
{
short	nClimbDim = ptDim->AltDimToClimbDim(m_Speed.nAltDim); 

*ptVst = (float)ptDim->ConvertSpeed (m_Speed.fClimbRate,
			       nClimbDim, DIM_FTPM);
*ptVsi = (float)ptDim->ConvertSpeed (m_Speed.fDescRate,
			       nClimbDim, DIM_FTPM);
}

/************************************************************************
 *  Plane.cpp				G e t U s e a b l e F u e l _ k g			*
 ************************************************************************/
float CPlane::GetUseableFuel_kg (float* ptPerCent, short nTankInd)
{
double	fLiter;
float fFuelVol, fFuel_kg;
fFuel_kg = (float)0;

if (*(ptPerCent+nTankInd) != NO_PCENT &&
      m_Tank.fUse[nTankInd] != NO_FUEL &&
      m_Tank.fUnuse[nTankInd] != NO_FUEL)
    {			// see wActData.c: NewWeightOK
    short   nPerCent = (short)(100 * *(ptPerCent+nTankInd));
    fFuelVol = ((m_Tank.fUse[nTankInd] +
		m_Tank.fUnuse[nTankInd]) * nPerCent) / 100;
    fFuelVol -= m_Tank.fUnuse[nTankInd];

    fLiter = ptDim->ConvertVolume (fFuelVol, m_Tank.nVolDim, DIM_LITER);
						/* Vol [l] * 0.72 kg/l	*/
    fFuel_kg = (float)(fLiter*0.72);
    }

return fFuel_kg;
}


/************************************************************************
 *  Plane.cpp				G e t S t a r t B a s i s _ m				*
 ************************************************************************/
void CPlane::GetStartBasis_m (float fActMass_kg, double* ptRoll, double* ptObst)
{
float fRoll = (float)0;
float fObst = (float)0;
float fMinMass_kg, fMaxMass_kg;

fMinMass_kg = (float)ptDim->ConvertMass (m_StLa.fStartMass[1],
				  m_StLa.nMassDim, DIM_KG);
fMaxMass_kg = (float)ptDim->ConvertMass (m_StLa.fStartMass[0],
				  m_StLa.nMassDim, DIM_KG);

					/* dist. for max. mass defined	*/
if (m_StLa.fStartRoll[0] != NO_RWY)
    fRoll = m_StLa.fStartRoll[0];

					/* dist. for min. mass defined	*/
if (m_StLa.fStartRoll[1] != NO_RWY &&
    (fMaxMass_kg - fMinMass_kg) > 0.)
    {
    fRoll = (fActMass_kg - fMinMass_kg) *
	    (m_StLa.fStartRoll[0] - m_StLa.fStartRoll[1]) /
	    (float)(fMaxMass_kg - fMinMass_kg) + m_StLa.fStartRoll[1];
    }

*ptRoll = (float)ptDim->ConvertDist (fRoll, m_StLa.nLenDim, DIM_METER);

					/* dist. for max. mass defined	*/
if (m_StLa.fStartObst[0] != NO_RWY)
    fObst = m_StLa.fStartObst[0];

					/* dist. for min. mass defined	*/
if (m_StLa.fStartObst[1] != NO_RWY &&
    (fMaxMass_kg - fMinMass_kg) > 0.)
    {
    fObst = (fActMass_kg - fMinMass_kg) *
	    (m_StLa.fStartObst[0] - m_StLa.fStartObst[1]) /
	    (float)(fMaxMass_kg - fMinMass_kg) + m_StLa.fStartObst[1];
    }

*ptObst = (float)ptDim->ConvertDist (fObst, m_StLa.nLenDim, DIM_METER);
}

/************************************************************************
 *  Plane.cpp				G e t L a n d B a s i s _ m					*
 ************************************************************************/
void CPlane::GetLandBasis_m (float fActMass_kg, double* ptRoll, double* ptObst)
{
float fRoll = (float)0;
float fObst = (float)0;
float fMinMass_kg, fMaxMass_kg;

fMinMass_kg = (float)ptDim->ConvertMass (m_StLa.fLandMass[1],
				  m_StLa.nMassDim, DIM_KG);
fMaxMass_kg = (float)ptDim->ConvertMass (m_StLa.fLandMass[0],
				  m_StLa.nMassDim, DIM_KG);

					/* dist. for max. mass defined	*/
if (m_StLa.fLandRoll[0] != NO_RWY)
    fRoll = m_StLa.fLandRoll[0];

					/* dist. for min. mass defined	*/
if (m_StLa.fLandRoll[1] != NO_RWY &&
    (fMaxMass_kg - fMinMass_kg) > 0.)
    {
    fRoll = (fActMass_kg - fMinMass_kg) *
	    (m_StLa.fLandRoll[0] - m_StLa.fLandRoll[1]) /
	    (float)(fMaxMass_kg - fMinMass_kg) + m_StLa.fLandRoll[1];
    }

*ptRoll = (float)ptDim->ConvertDist (fRoll, m_StLa.nLenDim, DIM_METER);

					/* dist. for max. mass defined	*/
if (m_StLa.fLandObst[0] != NO_RWY)
    fObst = m_StLa.fLandObst[0];

					/* dist. for min. mass defined	*/
if (m_StLa.fLandObst[1] != NO_RWY &&
    (fMaxMass_kg - fMinMass_kg) > 0.)
    {
    fObst = (fActMass_kg - fMinMass_kg) *
	    (m_StLa.fLandObst[0] - m_StLa.fLandObst[1]) /
	    (float)(fMaxMass_kg - fMinMass_kg) + m_StLa.fLandObst[1];
    }

*ptObst = (float)ptDim->ConvertDist (fObst, m_StLa.nLenDim, DIM_METER);
}

/************************************************************************
 *  Plane.cpp					G e t C o n s _ l						*
 ************************************************************************/
void CPlane::GetCons_l (float* ptClimb, float* ptCruise, float* ptDesc)
{
*ptClimb  = (float)ptDim->ConvertVolume (m_Speed.fClimbCons,
				  m_Speed.nVolDim, DIM_LITER);
*ptCruise = (float)ptDim->ConvertVolume (m_Speed.fCruiseCons,
				  m_Speed.nVolDim, DIM_LITER);
*ptDesc   = (float)ptDim->ConvertVolume (m_Speed.fDescCons,
				  m_Speed.nVolDim, DIM_LITER);
}

/************************************************************************
 *  wNavFunc.c					G e t A c t C o n s						*
 ************************************************************************/
float CPlane::GetActCons (BOOL bUpDown, long lNewAlt_ft, long lOldAlt_ft)
{
float	fActCons;
long	dH, EPS_H;			/* Hoehendifferenz	*/
float	fClimbCons, fCruiseCons, fDescCons;

this->GetCons_l (&fClimbCons, &fCruiseCons, &fDescCons);

if (fCruiseCons == (float)0 || fClimbCons == (float)0)
    return NO_FUEL;

fActCons = fCruiseCons;

if (bUpDown)
    {			    /* check difference of altitudes:	*/
		/* UpDown, if |dH| >= EPS_H feet (default: 1500 ft)	*/
    EPS_H = (long)ptDim->ConvertDist ((double)ptCalc->GetUpDownDiff(), 
										ptCalc->GetUpDownDim(), DIM_FEET);
    dH = lNewAlt_ft - lOldAlt_ft;

    if (dH > EPS_H) fActCons = fClimbCons;
    if (dH < EPS_H) fActCons = fDescCons;
    }
return fActCons;
}

/************************************************************************
 *  Plane.cpp				G e t W a k e C a t e g o r y 				*
 ************************************************************************/
WAKETURBCAT CPlane::GetWakeCategory()
{
WAKETURBCAT Cat = WT_LIGHT;

float	fMax_kg = (float)ptDim->ConvertMass (m_Lim.fMass[0],
							m_Lim.nMassDim, DIM_KG);

if (fMax_kg > 7000)
	Cat = WT_MEDIUM;
	
if (fMax_kg >= 136000) 
	Cat = WT_HEAVY;

return Cat;
}

/************************************************************************
 *  Plane.cpp    		 		 S e t I d e n t						*
 ************************************************************************/
void CPlane::SetIdent(CString szIdent)
{    
_fstrcpy (m_FlPl.szIdent, (LPCTSTR)szIdent); 
}

/************************************************************************
 *  Plane.cpp    		 		 S e t T y p e							*
 ************************************************************************/
void CPlane::SetType(CString szType)
{    
_fstrcpy (m_FlPl.szPlaneType, (LPCTSTR)szType); 
}

/************************************************************************
 *  Plane.cpp    		 		 S e t E q u i p m e n t 				*
 ************************************************************************/
void CPlane::SetEquipment(CString szEqu)
{    
_fstrcpy (m_FlPl.szEquipment, (LPCTSTR)szEqu); 
}

/************************************************************************
 *  Plane.cpp    		 		 S e t C o l o r		 				*
 ************************************************************************/
void CPlane::SetColor(CString szColor)
{    
_fstrcpy (m_FlPl.szPlaneCol, (LPCTSTR)szColor); 
}

/************************************************************************
 *  Plane.cpp    		 		 G e t I d e n t						*
 ************************************************************************/
CString CPlane::GetIdent()
{         
CString szIdent = (CString)m_FlPl.szIdent;	   // PL_SIZE_IDENT
return szIdent;
}

/************************************************************************
 *  Plane.cpp    		 		 G e t T y p e							*
 ************************************************************************/
CString CPlane::GetType()
{         
CString szType = (CString)m_FlPl.szPlaneType;		//PL_SIZE_PLTYPE
return szType;
}

/************************************************************************
 *  Plane.cpp    		 		 G e t E q u i p m e n t				*
 ************************************************************************/
CString CPlane::GetEquipment()
{         
CString szEqu = (CString)m_FlPl.szEquipment;		//PL_SIZE_EQUIP
return szEqu;
}

/************************************************************************
 *  Plane.cpp    		 		 G e t C o l o r						*
 ************************************************************************/
CString CPlane::GetColor()
{         
CString szColor = (CString)m_FlPl.szPlaneCol;		//PL_SIZE_PLCOL
return szColor;
}

            
/************************************************************************
 *  Plane.cpp				G e t A c t M a s s _ k g					*
 ************************************************************************/
float CPlane::GetActMass_kg (float* ptRow_kg, float* ptLug_kg,
							 CLoadDoc* ptLoad, float* ptActUseFuel_kg)
{
BOOL	bOverloaded = FALSE;
short	i;
float	fAct_kg, fMax_kg;
BOOL	bMaxFuelMode = TRUE;
float	fTank_pc[PL_MAX_TANK];

fMax_kg = (float)ptDim->ConvertMass (m_Lim.fMass[0],
							m_Lim.nMassDim, DIM_KG);

				/* Calculate Mass:			*/
		/* Empty Mass includes	u n u s a b l e	 fuel + oil	*/
fAct_kg = (float)ptDim->ConvertMass (m_Lim.fEmptyMass,
							 m_Lim.nMassDim, DIM_KG);

				/* add crew and luggage 		*/
for (i=0; i<PL_MAX_ARM; i++)
    fAct_kg += (*(ptRow_kg + i) + *(ptLug_kg + i));
bOverloaded = (fAct_kg > fMax_kg);

for (i=0; i<PL_MAX_TANK; i++)
    {	 
    if (ptLoad->GetTankFakt(i) != NO_PCENT)	    /* check max fuel mode	*/
		bMaxFuelMode = FALSE;
    *(ptActUseFuel_kg + i) = (float)0;	/* reset tank mass values	*/
    }

for (i=0; i<PL_MAX_TANK; i++)
    {
    if (bMaxFuelMode)
		 fTank_pc[i] = (float)1;	    /* try to use all tanks	*/
    else fTank_pc[i] = ptLoad->GetTankFakt(i);
    }

for (i=0; i<PL_MAX_TANK && !bOverloaded; i++)
    {
    float fFuel_kg = this->GetUseableFuel_kg (fTank_pc, i);
    if (fAct_kg + fFuel_kg > fMax_kg)
		{
		bOverloaded = TRUE;
		fFuel_kg = fMax_kg - fAct_kg;
		}

    fAct_kg += fFuel_kg;
    *(ptActUseFuel_kg + i) = fFuel_kg;
    }

return fAct_kg;
}

/************************************************************************
 *  Plane.cpp			G e t A c t T o r q _ m k g						*
 ************************************************************************/
float CPlane::GetActTorq_mkg (float* ptRow_kg, float* ptLug_kg,
										float* ptActUseFuel_kg)
{
short	i;
float	fAct_xkg, fAct_mkg, fArm_m;
float	fNoArm = this->GetUndefArm (m_Arms.nArmDim);

fAct_xkg = (float)ptDim->ConvertMass (m_Lim.fEmptyTorq,
										m_Lim.nMassDim, DIM_KG);
fAct_mkg = (float)ptDim->ConvertDist (fAct_xkg, m_Lim.nArmDim, DIM_METER);

for (i=0; i<PL_MAX_ARM; i++)
    {
    if (m_Arms.fRowArm[i] != fNoArm)
		{
		fArm_m = (float)ptDim->ConvertDist (m_Arms.fRowArm[i], m_Arms.nArmDim, DIM_METER);
		fAct_mkg += (*(ptRow_kg+i) * fArm_m);
		}
    if (m_Arms.fLugArm[i] != fNoArm)
		{
		fArm_m = (float)ptDim->ConvertDist (m_Arms.fLugArm[i], m_Arms.nArmDim, DIM_METER);
		fAct_mkg += (*(ptLug_kg+i) * fArm_m);
		}
    }


fNoArm = this->GetUndefArm (m_Tank.nArmDim);
for (i=0; i<PL_MAX_TANK; i++)
    {
    if (m_Tank.fArm[i] != fNoArm)
		{
		fArm_m = (float)ptDim->ConvertDist (m_Tank.fArm[i], m_Tank.nArmDim, DIM_METER);
		fAct_mkg += (*(ptActUseFuel_kg+i) * fArm_m);
		}
    }

return fAct_mkg;
}

/************************************************************************
 *  Plane.cpp				G e t F u e l V o l _ l						*
 *  Purpose: defines "ptfTotalFuel_l" the whole tank vol of each tank	*
 *	     defines "ptFillDegr_Pc", fill degr. (0...100%) of each tank	*
 *  Returns "fSumFuel_l", sum of al usable fuel [l]						*
 ************************************************************************/
float CPlane::GetFuelVol_l (float* ptUseFuel_kg, float* ptfTotalFuel_l, float* ptFillDegr_Pc)
{
float	fSumFuel_l = (float)0;
short i;

for (i=0; i<PL_MAX_TANK; i++)
    {
	float fActUseFuel_l, fUseFuel_l, fUnuseFuel_l;
	*(ptfTotalFuel_l + i) = (float)0;
	*(ptFillDegr_Pc + i)  = 0;

	if (m_Tank.fUse[i] != NO_FUEL &&
		m_Tank.fUnuse[i] != NO_FUEL)
		{
						/* Mass[kg] / 0.72 kg/l */
		fActUseFuel_l = *(ptUseFuel_kg+i)/(float)0.72;

		fUseFuel_l = (float)ptDim->ConvertVolume (m_Tank.fUse[i],
										 m_Tank.nVolDim, DIM_LITER);
		fUnuseFuel_l = (float)ptDim->ConvertVolume (m_Tank.fUnuse[i],
										m_Tank.nVolDim, DIM_LITER);

		*(ptfTotalFuel_l + i) = fActUseFuel_l + fUnuseFuel_l;
		*(ptFillDegr_Pc + i) = 100 * *(ptfTotalFuel_l + i) /
					  (fUseFuel_l + fUnuseFuel_l);

		fSumFuel_l += fActUseFuel_l;
		}
    }        
fSumFuel_l = (float)floor(10*fSumFuel_l)/10;
return fSumFuel_l;
}


/************************************************************************
 *  Plane.cpp				G e t M a x T a n k V o l					*
 ************************************************************************/
void CPlane::GetMaxTankVol (double* ptUseable_l, double* ptUnuseable_l)
{
double dUseFuel_l = 0;
double dUnuseFuel_l = 0;

for (int i=0; i<PL_MAX_TANK; i++)
    {
	if (m_Tank.fUse[i] != NO_FUEL &&
		m_Tank.fUnuse[i] != NO_FUEL)
		{
		dUseFuel_l += ptDim->ConvertVolume (m_Tank.fUse[i],
										 m_Tank.nVolDim, DIM_LITER);
		dUnuseFuel_l += ptDim->ConvertVolume (m_Tank.fUnuse[i],
										m_Tank.nVolDim, DIM_LITER);
		}
    }        
*ptUseable_l = dUseFuel_l;
*ptUnuseable_l = dUnuseFuel_l;
}


/************************************************************************
 *  Plane.cpp				G e t C o G L i m i t s						*
 ************************************************************************/
void CPlane::GetCoGLimits (float* ptLimMass,
			float* ptLimFrntArm, float* ptLimBackArm, short nCnt)
{
short	i;

for (i=0; i<nCnt; i++)
    {
    *(ptLimMass + i) = (float)ptDim->ConvertMass (*(m_Lim.fMass + i),
			    m_Lim.nMassDim, DIM_KG);
    *(ptLimFrntArm + i) = (float)ptDim->ConvertDist (*(m_Lim.fFrntArm + i),
			    m_Lim.nArmDim, DIM_METER);
    *(ptLimBackArm + i) = (float)ptDim->ConvertDist (*(m_Lim.fBackArm + i),
			    m_Lim.nArmDim, DIM_METER);
    }
}

/************************************************************************
 *  Plane.cpp				B a l a n c e O K							*
 *	Eingabe:	Flugzeugdaten, aktuelle Gesamtmasse u. SP-Lage			*
 *	Ausgabe:	SP-Grenzwerte fuer die aktuelle Gesamtmasse				*
 *	Return:		FALSE, wenn SP-Lage ausserhalb der Grenzwerte			*		
 ************************************************************************/
BOOL CPlane::BalanceOK (float fAct_kg, double fCoG_m,
						double* ptMinCoG_m, double* ptMaxCoG_m)
{
float	MGesDiff;
float	fLimMass[PL_MAX_LIM];
float	fLimFrntArm[PL_MAX_LIM];
float	fLimBackArm[PL_MAX_LIM];

					/* in kg und Meter		*/
this->GetCoGLimits (fLimMass, fLimFrntArm, fLimBackArm, PL_MAX_LIM);

MGesDiff = fLimMass[0] - fLimMass[1];	 /* z.B.: 1089kg - 885kg	*/

if (MGesDiff > (float)0)
    {
    #define EPS_ARM_M ((double)0.005)

    if (fAct_kg < fLimMass[1])
		{				/* z.B.: akt. Masse < 885 kg	*/
		*ptMinCoG_m = fLimFrntArm[1];	   /* nutze Limits fuer 885 kg	   */
		*ptMaxCoG_m = fLimBackArm[1];
		}
    else{				/* berechne Limits		*/
		*ptMinCoG_m = (fLimFrntArm[0] - fLimFrntArm[1]) /
				MGesDiff * (fAct_kg - fLimMass[1]) + fLimFrntArm[1];
		*ptMaxCoG_m = (fLimBackArm[0] - fLimBackArm[1]) /
				MGesDiff * (fAct_kg - fLimMass[1]) + fLimBackArm[1];
		}

    if ((fCoG_m < *ptMinCoG_m - EPS_ARM_M) ||
	(fCoG_m > *ptMaxCoG_m + EPS_ARM_M))	/* bad balance !!*/
	return FALSE;
  }
return TRUE;
}

/************************************************************************
 *  Plane.cpp					G e t C o s t							*
 ************************************************************************/
float CPlane::GetCost (short nMinutes)
{
float fCost = (float)0;

if (m_nCost > 0)
    fCost = (float)m_nCost * nMinutes / 60;

return fCost;
}

/////////////////////////////////////////////////////////////////////////////
// CPlane commands
