#ifndef __PLANE__      
#define __PLANE__

// Plane.h
// Important Setting for PLANETYPE: 
// Project - Settings - C/C++ - Category: Code Generation - 
// Struct Member Alignment: 2 bytes
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#define MAX_PLANE_CNT 19

#define SIZEOF_PLANE	12

#define PL_MAX_STLA	2
#define PL_MAX_ARM	5
#define PL_MAX_TANK	3
#define PL_MAX_LIM	2

								// flight plan parameters...
#define PL_SIZE_EQUIP	8
#define PL_SIZE_IDENT	8
#define PL_SIZE_PLTYPE	8
#define PL_SIZE_PLCOL	128

typedef enum	{
				WT_HEAVY,
				WT_MEDIUM,
				WT_LIGHT
				} WAKETURBCAT;

typedef enum	{
				SS_NONE,
				SS_NOCODING,
				SS_2DIGITS_A,
				SS_4DIGITS_A,
				SS_MODE_A_C
				} SSRTYPE;


typedef struct
    {			    /* for flight plan		    */
	char		szIdent[PL_SIZE_IDENT];
	char		szPlaneType[PL_SIZE_PLTYPE];
	char		szEquipment[PL_SIZE_EQUIP];
	char		szPlaneCol[PL_SIZE_PLCOL];
	SSRTYPE		nSSR;			
	BOOL		bEmRadioUHF;
	BOOL		bEmRadioVHF;
	BOOL		bEmRadioELBA;
	} PL_FLPL_TYPE;

typedef struct
	{
	short	nDistDim;		/* for speed							*/
	short	nAltDim;		/* for rate of climb					*/
	short	nVolDim;
	float	fCruiseSpd;		/* Cruising speed kt					*/
	float	fCruiseCons;	/* Fuel consumption while cruising l/h	*/
	float	fClimbSpd;		/* Cruising speed while climbing kt		*/
	float	fClimbCons;		/* Fuel consumption while climbing l/h	*/
	float	fClimbRate;		/* Rate of climb ft/min					*/
	float	fDescSpd;		/* Cruising speed while descending kt	*/
	float	fDescCons;		/* Fuel consumtion while descending l/h */
	float	fDescRate;		/* Rate of descend ft/min				*/
	} SPEEDTYPE;

typedef struct
    {			    /* for Start/Land dialog:		    */
    short   nMassDim;	    /* for masses			    */
    short   nLenDim;	    /* for take-off and landing distances   */
    short   nStartCnt;	    /* Number of specified start distances  */
    float   fStartMass[PL_MAX_STLA];   /* Mass in kg for spec. dist.*/
    float   fStartRoll[PL_MAX_STLA];   /* take-off distance in m    */
    float   fStartObst[PL_MAX_STLA];   /* take-off dist 15 m obst.  */
    short   nLandCnt;		    /* Number of specified land distances   */
    float   fLandMass[PL_MAX_STLA]; /* Mass in kg for specified distances   */
    float   fLandRoll[PL_MAX_STLA]; /* Landing distance in m		    */
    float   fLandObst[PL_MAX_STLA]; /* landing distance over 15 m obstacle  */
    } SLTYPE;

typedef struct
	{			/* for Loararms	dialog:			*/
	short	nArmDim;	/* for lever arms			*/
	short	nRowCnt;	/* Number of specified rows of seats	*/
	float	fRowArm[PL_MAX_ARM];	/* Lever arms for seats in m	*/
	short	nLugCnt;	/* Number of specified luggage parts	*/
	float	fLugArm[PL_MAX_ARM];	/* Lever arms for luggage in m	*/
	} ARMSTYPE;

typedef struct
	{			/* for Fuel Capacity dialog:	    */
	short	nVolDim;	/* for consumption		    */
	short	nArmDim;	/* for lever arms		    */
	short	nCnt;		/* Number of specified fuel tanks   */
	float	fUse[PL_MAX_TANK];	/* Useable fuel in Liters   */
	float	fUnuse[PL_MAX_TANK];	/* Unuseable fuel in Liters */
	float	fArm[PL_MAX_TANK];	/* Lever arms for tanks in m*/
	} TANKTYPE;

typedef struct
	{
	short	nMassDim;	/* for masses				*/
	short	nArmDim;	/* for lever arms			*/
	float	fEmptyMass;	/* Empty mass kg			*/
	float	fEmptyTorq;	/* Empty torque in mkg			*/
	short	nCnt;		/* Number of specified CentOfGrav. lim. */
	float	fMass[PL_MAX_LIM];    /* Mass in kg for spec.d limits	*/
	float	fFrntArm[PL_MAX_LIM]; /* Front limit  center of gravity */
	float	fBackArm[PL_MAX_LIM]; /* Back limit center of gravity	*/
	} COGLIMTYPE;               
 
#define ACT_PLANE_VERS	3   

#pragma pack(2)

typedef struct
	{
	char	szKennz[SIZEOF_PLANE];	/* Kennzeichen			*/
	short	V_Reisefl;	/* Reisegeschwindigkeit kt		*/
	float	Verbr1;		/* Kraftstoffverbrauch im Reiseflug l/h	*/
	short	V_Steigfl;	/* Fluggeschwindigkeit beim Steigen kt	*/
	float	Verbr2;		/* Kraftstoffverbrauch im Steigflug l/h	*/
	short	V_Steig;	/* Steiggeschwindigkeit ft/min		*/
	short	dummy1;		/* Reserve				*/
	short	dummy2;		/* Reserve				*/
	short	Start;		/* Startstrecke (15m Hindernis) m	*/
	short	Land;		/* Landestrecke (15m Hindernis) m	*/
	float	dummy3;		/* Reserve				*/
	float	M_leer;		/* Leermasse kg				*/
	float	MO_leer;	/* Leermoment mkg			*/
	float	aKraft;		/* max. ausfliegbarer Kraftstoff l	*/
	float	naKraft;	/* nicht mausfliegbarer Kraftstoff l	*/
	float	H_Kraft;	/* Hebelarm zum Kraftstoff m		*/
	float	H_Reihe1;	/* Hebelarm zur 1. Sitzreihe m		*/
	float	H_Reihe2;	/* Hebelarm zur 2. Sitzreihe m		*/
	float	H_Gep1;		/* Hebelarm zum 1. Gepaeckraum m	*/
	float	H_Gep2;		/* Hebelarm zum 2. Gepaeckraum m	*/
	float	M_Ges1;		/* maximal zulaessige Gesamtmasse kg	*/
	float	H_1Min;		/* vordere SP-Lage bei voller Beladung	*/
	float	H_1Max;		/* hintere SP-Lage bei voller Beladung	*/
	float	M_Ges0;		/* 2. Masse zur Kontrolle der Beladung	*/
	float	H_0Min;		/* vordere SP-Lage bei 2. Masse		*/
	float	H_0Max;		/* hintere SP-Lage bei 2. Masse		*/
	} PLANETYPE1;

typedef PLANETYPE1 FAR* LPPLANE1;

typedef struct
	{
	char	szName[SIZEOF_PLANE];	/* Callsign			*/
	SPEEDTYPE   Speed;
	SLTYPE	    StLa;
	ARMSTYPE    Arms;
	TANKTYPE    Tank;
	short	    nCost;	/* Flight cost per hour 		*/
	COGLIMTYPE  Lim;
	long	    lDummy;
	short	    nDummy;
	} PLANETYPE2;

typedef PLANETYPE2 FAR* LPPLANE2;


typedef struct
	{
	char			szName[SIZEOF_PLANE];	/* Callsign	or Type		*/
	SPEEDTYPE		Speed;
	SLTYPE			StLa;
	ARMSTYPE		Arms;
	TANKTYPE		Tank;
	short			nCost;		// Flight cost per hour 		
	COGLIMTYPE		Lim;
	PL_FLPL_TYPE	FlPl;		// flight plan params
	long			lDummy;
	short			nDummy;
	} PLANETYPE;

typedef PLANETYPE FAR* LPPLANE;
#pragma pack() 
	
class CLoadDoc;	
class CPlane : public CObject
{
	DECLARE_SERIAL(CPlane)

// Attributes
public:    
	char			m_szName[SIZEOF_PLANE];	/* Callsign			*/
	SPEEDTYPE		m_Speed;
	SLTYPE			m_StLa;
	ARMSTYPE		m_Arms;
	TANKTYPE		m_Tank;
	short			m_nCost;	// Flight cost per hour 
	COGLIMTYPE		m_Lim;
	PL_FLPL_TYPE	m_FlPl;		// flight plan params
	long			m_lDummy;
	short			m_nDummy;


// Operations
private:

protected:
	void ReadFromPtr(LPPLANE lpPlane);
	void WriteToPtr(LPPLANE lpPlane) const;

// Implementation
public:
	CPlane(LPPLANE lpPlane = NULL);		
	CPlane(CPlane& SourcePlane);
	~CPlane();                                                

	void GetPtr(LPPLANE lpPlane);
	void SetPtr (LPPLANE lpPlane);
	const CPlane& operator=(const CPlane& Plane);
	void FirstInit();

	static float GetUndefArm (short nArmDim);
	void CopyPlane1ToPlane (PLANETYPE1* ptOldPlane);
	void FirstFlPlInit (PL_FLPL_TYPE* ptFlPl);

	void SetStartLandDim(short nMassDim, short nLenDim);
	void SetStartData(short nIndex, float fMass, float fRoll, float fObst);
	void SetLandData(short nIndex, float fMass, float fRoll, float fObst);
	void GetStartLandDim(short* ptMassDim, short* ptLenDim);
	void GetStartData(short nIndex, float* ptMass, float* ptRoll, float* ptObst);
	void GetLandData(short nIndex, float* ptMass, float* ptRoll, float* ptObst);

	void SetLeverArmDim(short nArmDim);
	void SetRowArm(short nIndex, float fArm);
	void SetLugArm(short nIndex, float fArm);
	short GetLeverArmDim();
	float GetRowArm(short nIndex);
	float GetLugArm(short nIndex);

	void SetTankDim(short nVolDim, short nArmDim);
	void SetTankData(short nIndex, float fUse, float fUnuse, float fArm);
	void SetTankCnt (short nCnt) { m_Tank.nCnt = nCnt; }
	void GetTankDim(short* ptVolDim, short* ptArmDim);
	void GetTankData(short nIndex, float* ptUse, float* ptUnuse, float* ptArm);
	float GetUseTankVol(short nIndex){return m_Tank.fUse[nIndex];}
	float GetUnuseTankVol(short nIndex){return m_Tank.fUnuse[nIndex];}
	float GetTankArm(short nIndex){return m_Tank.fArm[nIndex];}

	short GetTankCnt (){return m_Tank.nCnt;}

 	void Serialize(CArchive& ar, short nVersion = ACT_PLANE_VERS);
	LPPLANE CreatePtr ();
	BOOL IsEqual(const CPlane& ptTestPlane);
	BOOL IsEqual(LPPLANE lpTestPlane);

 	void SetName(CString szName);
	void SetCost(short nCost) { m_nCost = nCost; }
	void SetClimbSpeed (float fSpeed) { m_Speed.fClimbSpd = fSpeed; }
	void SetClimbCons (float fCons) { m_Speed.fClimbCons = fCons; }
	void SetClimbRate (float fSpeed) { m_Speed.fClimbRate = fSpeed; }
   	void SetCruiseSpeed (float fSpeed) { m_Speed.fCruiseSpd = fSpeed; }
	void SetCruiseCons (float fCons) { m_Speed.fCruiseCons = fCons; }
	void SetDescSpeed (float fSpeed) { m_Speed.fDescSpd = fSpeed; }
	void SetDescCons (float fCons) { m_Speed.fDescCons = fCons; }
	void SetDescRate (float fSpeed) { m_Speed.fDescRate = fSpeed; }

	void SetMass(short nIndex, float fMass);
	void SetFrontArm(short nIndex, float fFrntArm);
	void SetBackArm(short nIndex, float fBackArm);
 	void SetEmptyMass (float fMass) { m_Lim.fEmptyMass = fMass; }
	void SetEmptyTorq (float fTorq) { m_Lim.fEmptyTorq = fTorq; }
	void SetMassCnt (short nCnt) { m_Lim.nCnt = nCnt; }

	void SetSpeedDim(short nDim);
  	void SetVolDim(short nDim){m_Speed.nVolDim = nDim;}
	void SetClimbDim(short nDim);
  	void SetArmDim(short nDim){m_Lim.nArmDim = nDim;}
  	void SetMassDim(short nDim){m_Lim.nMassDim = nDim;}


 	CString GetName(); 
//	BOOL GetName(LPSTR lpName);
	short GetCost() { return m_nCost; }
	float GetClimbSpeed() { return m_Speed.fClimbSpd; }
	float GetClimbCons() { return m_Speed.fClimbCons; }
	float GetClimbRate() { return m_Speed.fClimbRate; }
	float GetCruiseSpeed() { return m_Speed.fCruiseSpd; }
	float GetCruiseCons() { return m_Speed.fCruiseCons; }
	float GetDescSpeed () { return m_Speed.fDescSpd; }
	float GetDescCons () { return m_Speed.fDescCons; }
	float GetDescRate () { return m_Speed.fDescRate; }


	float GetStartMass(short nIndex){return m_StLa.fStartMass[nIndex];}
	float GetStartRoll(short nIndex){return m_StLa.fStartRoll[nIndex];}
	float GetStartObst(short nIndex){return m_StLa.fStartObst[nIndex];}

	float GetLandMass(short nIndex){return m_StLa.fLandMass[nIndex];}
	float GetLandRoll(short nIndex){return m_StLa.fLandRoll[nIndex];}
	float GetLandObst(short nIndex){return m_StLa.fLandObst[nIndex];}


	void SetIdent(CString szIdent);
	void SetType(CString szType);
	void SetEquipment(CString szEqu);
	void SetColor(CString szColor);
	void SetSSRType (SSRTYPE Type) { m_FlPl.nSSR = Type; }			
	void SetEmRadioUHF (BOOL bSet) { m_FlPl.bEmRadioUHF = bSet; } 
	void SetEmRadioVHF (BOOL bSet) { m_FlPl.bEmRadioVHF = bSet; } 
	void SetEmRadioELBA (BOOL bSet) { m_FlPl.bEmRadioELBA = bSet; } 

	WAKETURBCAT	GetWakeCategory ();
	CString GetIdent();
	CString GetType();
	CString GetEquipment();
	CString GetColor();

	SSRTYPE GetSSRType () { return m_FlPl.nSSR; }			
	BOOL	HasEmRadioUHF () { return m_FlPl.bEmRadioUHF; }
	BOOL	HasEmRadioVHF () { return m_FlPl.bEmRadioVHF;  }
	BOOL	HasEmRadioELBA () { return m_FlPl.bEmRadioELBA;	}


	float GetMass(short nIndex);
	float GetFrontArm(short nIndex);
	float GetBackArm(short nIndex);
 	float GetEmptyMass() { return m_Lim.fEmptyMass; }
	float GetEmptyTorq() { return m_Lim.fEmptyTorq; }
	short GetMassCnt() { return m_Lim.nCnt; }


	short GetSpeedDim();
	short GetVolDim () { return m_Speed.nVolDim; }
	short GetClimbDim();
	short GetArmDim () { return m_Lim.nArmDim; }
	short GetMassDim() { return m_Lim.nMassDim; }

	void	GetHorzSpeed_kt (float* ptVstFl, float* ptVrFl, float* ptVsiFl);
	void	GetVertSpeed_ftpm (float* ptVst, float* ptVsi);
	float	GetUseableFuel_kg (float* ptPerCent, short nTankInd);
	void	GetStartBasis_m (float fActMass_kg, double* ptRoll, double* ptObst);
	void	GetLandBasis_m (float fActMass_kg, double* ptRoll, double* ptObst);
	void	GetCons_l (float* ptClimb, float* ptCruise, float* ptDesc);
	float	GetActCons (BOOL bUpDown, long lNewAlt_ft, long lOldAlt_ft);

	float	GetActMass_kg (float* ptRow_kg, float* ptLug_kg,
							 CLoadDoc* ptLoad, float* ptActUseFuel_kg);
	float	GetActTorq_mkg (float* ptRow_kg, float* ptLug_kg,
										float* ptActUseFuel_kg);
	float	GetFuelVol_l (float* ptUseFuel_kg, float* ptfTotalFuel_l, float* ptFillDegr_Pc);
	void	GetMaxTankVol (double* ptUseable_l, double* ptUnuseable_l);
	void	GetCoGLimits (float* ptLimMass, float* ptLimFrntArm, float* ptLimBackArm, short nCnt);
	BOOL	BalanceOK (float fAct_kg, double fCoG_m, double* ptMinCoG_m, double* ptMaxCoG_m);
	float	GetCost (short nMinutes);

};
#endif 

