/************************************************************************
 *  							S c M p D o c . c p p  					*
 ************************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "pf.h"

#include "InitDoc.h"	 //includes "NameList.h"

#include "WayDoc.h"
#include "TrackDoc.h"
#include "LamCalDlg.h"				// LAMBERT projection
#include "LinCalDlg.h"				// LINEAR projection
#include "ScMpView.h"				// for ScrollHorzMax
#include "ScMpDlg.h"				// dialog to set scan map values
#include "ScMpDoc.h"  

extern CInitDoc* 	ptInit;  
extern BOOL bDemo;

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CScanMapDoc
IMPLEMENT_DYNCREATE(CScanMapDoc, CDocument)

BEGIN_MESSAGE_MAP(CScanMapDoc, CDocument)
	//{{AFX_MSG_MAP(CScanMapDoc)
		// NOTE - the ClassWizard will add and remove mapping macros here.
		//    DO NOT EDIT what you see in these blocks of generated code!
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CScanMapDoc construction/destruction

/************************************************************************
 *  ScMpDoc.cpp	  		 	C S c a n M a p D o c			Constructor	*
 ************************************************************************/
CScanMapDoc::CScanMapDoc(CString szFileName)
{      
	CString szFullPath;

	m_ptDib = NULL;
	m_nActMapIndex = -1;
	szFullPath = ptInit->GetActualPath() + szFileName;   // C:\PF22\ScMapDoc.ini"

	BOOL bAddToMRU = FALSE;
	this->SetPathName (szFullPath, bAddToMRU);
	m_bShowCalibDlg = FALSE;
	m_bAutoSelect = TRUE;
	m_bAborted = FALSE;
	m_bImported = FALSE;
	m_bImport = FALSE;

	m_nIncreaseCnt = 0;

	m_dActFaktX = 1;
	m_dActFaktY = 1;
	m_ptScMpView = NULL;
}

/************************************************************************
 *  ScMpDoc.cpp	  			~ C S c a n M a p D o c			Destructor	*
 ************************************************************************/
CScanMapDoc::~CScanMapDoc()
{         
this->DeleteArrayOf (&m_ScannedMaps);

if (m_ptDib != NULL)
	{
	delete m_ptDib;
	m_ptDib = NULL;
	}
}
     
/************************************************************************
 *  ScMpDoc.cpp					G e t E x t 							*
 ************************************************************************/
CString CScanMapDoc::GetExt()
{
	CString szExt(".scm");
	return szExt;
}
  
/************************************************************************
 *  ScMpDoc.cpp				D e l e t e A r r a y O f 					*
 ************************************************************************/
void CScanMapDoc::DeleteArrayOf (CObArray* ptArray)
{
int i, nEntryCnt;	
											
nEntryCnt = ptArray->GetSize();
for (i=0; i<nEntryCnt; i++)
	{
	CObject* ptEntry;
	if ((ptEntry = ptArray->GetAt(i)) != NULL)
		{
		delete ptEntry;				// delete original element
		}
	}
ptArray->RemoveAll();
}



/************************************************************************
 *  ScMpDoc.cpp    		 O r i g i n a l T o S c a l e d L P			*
 *  Purpose: convert unscaled LP into LP on scaled map					* 
 ************************************************************************/
CSize CScanMapDoc::OriginalToScaledLP(CSize pOriginalLP)
{
	CPoint pointVar;
	pointVar.x = pOriginalLP.cx;
	pointVar.y = pOriginalLP.cy;

	pointVar = this->OriginalToScaledLP(pointVar);

	CSize sizeVar;
	sizeVar.cx = pointVar.x;
	sizeVar.cy = pointVar.y;
	return sizeVar;
}


/************************************************************************
 *  ScMpDoc.cpp    		 O r i g i n a l T o S c a l e d L P			*
 *  Purpose: convert unscaled LP into LP on scaled map					* 
 ************************************************************************/
CPoint CScanMapDoc::OriginalToScaledLP(CPoint pOriginalLP)
{
	CPoint pScaledLP;

	if (m_nIncreaseCnt==0)
	{		// map in original size: store act position in map
		pScaledLP = pOriginalLP;
	}
	else
	{		// map increased or decreased: calculate LP for scaled size
		pScaledLP.x = (int)((double)pOriginalLP.x*m_dActFaktX);
		pScaledLP.y = (int)((double)pOriginalLP.y*m_dActFaktY);
	}

	return pScaledLP;
}

/************************************************************************
 *  ScMpDoc.cpp			 S c a l e d T o O r i g i n a l L P			*
 *  Purpose: convert scaled LP into LP on unscaled map					* 
 ************************************************************************/
CPoint CScanMapDoc::ScaledToOriginalLP(CPoint pScaledLP)
{
	CPoint pOriginalLP;

	if (m_nIncreaseCnt==0)
	{		// map in original size: store act position in map
		pOriginalLP = pScaledLP;
	}
	else
	{		// map increased or decreased: calculate LP for original size
		pOriginalLP.x = (int)((double)pScaledLP.x/m_dActFaktX);
		pOriginalLP.y = (int)((double)pScaledLP.y/m_dActFaktY);
	}

	return pOriginalLP;
}

/****************************************************************************
 *	ScMpDoc.cpp				C h a n g e I n c r e a s e C n t 				*
 ****************************************************************************/										 
void CScanMapDoc::ChangeIncreaseCnt (short nDiff)
{
	m_nIncreaseCnt += nDiff;
}


/****************************************************************************
 *	ScMpDoc.cpp					C a l c A c t F a k t 						*
 ****************************************************************************/										 
BOOL CScanMapDoc::CalcActFakt (CSize cImage, CSize* ptVirtLP)
{
	BOOL	bOK = FALSE;
	CSize sVirtLP;

	double dBaseScaleFakt = 1.3;

	if (m_nIncreaseCnt == 0)
	{
		m_dActFaktX = 1.;
		m_dActFaktY = 1.;
	}
	else
	{
		if (m_nIncreaseCnt > 0)
		{
			m_dActFaktX = dBaseScaleFakt * m_nIncreaseCnt;
			m_dActFaktY = dBaseScaleFakt * m_nIncreaseCnt;
		}
		else
		{
			m_dActFaktX = -1./(dBaseScaleFakt * m_nIncreaseCnt);
			m_dActFaktY = -1./(dBaseScaleFakt * m_nIncreaseCnt);
		}
	}

	ptVirtLP->cx = (long)(m_dActFaktX * cImage.cx);
	ptVirtLP->cy = (long)(m_dActFaktY * cImage.cy);

	if (ptVirtLP->cx < 32768 && ptVirtLP->cy < 32768)	// both values positive!!
	{
		bOK = true;
	}

	return bOK;
}

/************************************************************************
 *  ScMpDoc.cpp				C o p y M a p s								*
 ************************************************************************/
void CScanMapDoc::CopyMaps (CObArray* ptDest, CObArray* ptSource)
{
short i;
for (i=0; i<ptSource->GetSize(); i++) 
	{	 
	CScannedMap* ptMap = NULL;
	ptMap = (CScannedMap*)ptSource->GetAt (i);    
	if (ptMap != NULL)
		{
		CScannedMap* lpCpyMap = new CScannedMap (*ptMap);  
		ptDest->Add (lpCpyMap);    
		}
	}
}        

/************************************************************************
 *  ScMpDoc.cpp				G e t D a t a B a s e 						*
 ************************************************************************/
void CScanMapDoc::GetDataBase (CObArray* ptMapList)
{    
                              	// copy database from Doc to Dlg
this->DeleteArrayOf (ptMapList);
this->CopyMaps (ptMapList, &m_ScannedMaps);   
}   

/************************************************************************
 *  ScMpDoc.cpp				S e t D a t a B a s e 						*
 ************************************************************************/
void CScanMapDoc::SetDataBase (CObArray* ptMapList)
{     
                          		 // copy database from Dlg to Doc
this->DeleteArrayOf (&m_ScannedMaps);
this->CopyMaps (&m_ScannedMaps, ptMapList);   
}


/************************************************************************
 *  ScMpDoc.cpp	  				O p e n B m p F i l e					*
 ************************************************************************/
BOOL CScanMapDoc::OpenBmpFile (CString szFile) 
{ 
BOOL bOpen = FALSE;
BOOL bRead = FALSE;

m_szMapFile = szFile;
       
BeginWaitCursor();		// display the hourglass cursor

CFile file;
if (file.Open((LPCTSTR)m_szMapFile, CFile::modeRead)) 
	{
	bOpen = TRUE;

	if (m_ptDib != NULL)
		{
		delete m_ptDib;
		m_ptDib = NULL;
		}

	m_ptDib = new CImage; 			// replace the view's DIB
	if (m_ptDib != NULL)
		{
		if (m_ptDib->Read(file)) 
			{
			bRead = TRUE;
			}
		}

	file.Close();
	}

EndWaitCursor();		// remove the hourglass cursor

if (!bOpen)
	{
	CString szMsg;
	szMsg.Format (IDF_FILE_NOT_FOUND, (LPCTSTR)m_szMapFile);
	AfxMessageBox (szMsg);
	}
else{
	if (!bRead)
		{
		CString szMsg;
		szMsg.Format(IDF_BAD_BMP, (LPCTSTR)m_szMapFile);
		AfxMessageBox(szMsg);
		}
	}

return (bOpen && bRead);
}


/************************************************************************
 *  ScMpDoc.cpp				A p p e n d M a p							*
 ************************************************************************/
BOOL CScanMapDoc::AppendMap (CScannedMap* ptMap)
{
BOOL bAppended = FALSE;
short nCnt = GetMapCnt();

if (ptMap != NULL)
	{  
	m_ScannedMaps.Add (ptMap);       	// add new Element  
	bAppended = (GetMapCnt() == nCnt+1);      
	}
return bAppended;
}

/************************************************************************
 *  ScMpDoc.cpp					A d d M a p								*
 ************************************************************************/
BOOL CScanMapDoc::AddMap (short nIndex, CScannedMap* ptMap)
{
BOOL bAppended = FALSE;
short nCnt = GetMapCnt();

if (ptMap != NULL)
	{  
	m_ScannedMaps.InsertAt (nIndex, ptMap);       	// add new Element  
	bAppended = (GetMapCnt() == nCnt+1);      
	}
return bAppended;
}

/************************************************************************
 *  ScMpDoc.cpp		 		 G e t S o r t e d I n d e x 				*
 ************************************************************************/
short CScanMapDoc::GetSortedIndex (CObArray& Maps, CScannedMap* ptMap)
{
BOOL	bInsAfter;  
long	IndMin, IndMax, i, OldTestInd;

        								// get new name to insert 
CString	szNewName = ptMap->GetName();

IndMin  = 0;
IndMax  = Maps.GetSize();		/* legal ind: 0...cnt-1		*/
i = (IndMin + IndMax) / 2;				/* i always < cnt !!	*/

if (Maps.GetSize() > 0)
   {
   do	{  
        CScannedMap* lpTestMap = (CScannedMap*)Maps.GetAt((int)i);
        CString szTestName = lpTestMap->GetName();
        
        int iCmp = szNewName.CompareNoCase ((LPCTSTR)szTestName);

	    if (iCmp >=0)	bInsAfter = TRUE;
	    		else	bInsAfter = FALSE;

		if (bInsAfter)	IndMin = i;
				else	IndMax = i;
				
		OldTestInd = i;
		i = (IndMin + IndMax) / 2;
	
		} while (i != OldTestInd);

   if (bInsAfter) i++;
   }                     
   
return (short)i;
}

/************************************************************************
 *  ScMpDoc.cpp	  		 I n s e r t M a p S o r t e d					*
 ************************************************************************/
short CScanMapDoc::InsertMapSorted(CScannedMap* ptMap)	
{	
short nNewIndex= -1;

nNewIndex = this->GetSortedIndex (m_ScannedMaps, ptMap); 

if (nNewIndex > 0)
	{
 	CScannedMap* ptPrevLoc = (CScannedMap*)m_ScannedMaps.GetAt(nNewIndex-1);
	CString szPrevName = ptPrevLoc->GetName();
	CString szNewName = ptMap->GetName();

	if (szNewName.Compare (szPrevName) == 0)
		{
		AfxMessageBox (IDS_SAME_NAME);
		return -1;
		}
	}

m_ScannedMaps.InsertAt (nNewIndex, ptMap);	// insert new
return nNewIndex;
}	


/************************************************************************
 *  ScMpDoc.cpp				C h a n g e M a p							*
 ************************************************************************/
BOOL CScanMapDoc::ChangeMap (short nIndex, CScannedMap* ptNewMap)
{
BOOL bChanged = FALSE;
short nCnt = GetMapCnt();

if (nIndex >= 0 && nIndex < m_ScannedMaps.GetSize())
	{
	CScannedMap* ptOldMap = (CScannedMap*)m_ScannedMaps.GetAt(nIndex);
	if (ptOldMap != NULL)
		{  
		delete ptOldMap;						// delete old map ptr
		}

	if (ptNewMap != NULL)
		{  
		m_ScannedMaps.SetAt (nIndex, ptNewMap);    // set new map ptr 
		bChanged = (GetMapCnt() == nCnt);      
		}
	}
else{
	AfxMessageBox ("CScanMapDoc::ChangeMap Bad index!");
	}
return bChanged;
}

/************************************************************************
 *  ScMpDoc.cpp					D e l e t e M a p						*
 ************************************************************************/
BOOL CScanMapDoc::DeleteMap (short nIndex)
{
BOOL bDeleted = FALSE;
short nCnt = GetMapCnt();

if (nIndex >= 0 && nIndex < m_ScannedMaps.GetSize())
	{
	CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(nIndex);
	if (ptMap != NULL)
		{  
		m_ScannedMaps.RemoveAt(nIndex);
		delete ptMap;
		bDeleted = (GetMapCnt() == nCnt-1);      
		}
	}
else{
	AfxMessageBox ("CScanMapDoc::DeleteMap Bad index!");
	}

return bDeleted;
}

/************************************************************************
 *  ScMpDoc.cpp				G e t M a p C n t 							*
 ************************************************************************/
short CScanMapDoc::GetMapCnt ()
{
return (short)m_ScannedMaps.GetSize();
}

/************************************************************************
 *  ScMpDoc.cpp				G e t M a p P t r							*
 ************************************************************************/
CScannedMap* CScanMapDoc::GetMapPtr (short nIndex)
{
CScannedMap* ptMap=NULL;
if (nIndex>=0  &&  nIndex < GetMapCnt())
	{
	ptMap = (CScannedMap*)m_ScannedMaps.GetAt (nIndex);  
	} 
return ptMap;      
}

/************************************************************************
 *  ScMpDoc.cpp					G e t M a p N a m e						*
 ************************************************************************/
CString CScanMapDoc::GetMapName (short nIndex)
{
CString szName;

CScannedMap* ptMap = this->GetMapPtr (nIndex);
if (ptMap != NULL)
	szName = ptMap->GetName();

return szName;
}

/************************************************************************
 *  ScMpDoc.cpp				M a p N a m e T o I n d e x					*
 ************************************************************************/
short CScanMapDoc::MapNameToIndex (CString szName)
{
short nIndex = -1;
short	i;

BOOL bFound = FALSE;

for (i=0; i<m_ScannedMaps.GetSize() && !bFound; i++)
	{
	CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i);
	if (ptMap != NULL)
		{	
		CString szTestName = ptMap->GetName();	
		bFound = (szTestName.Compare (szName) == 0);
		if (bFound)
			nIndex = i;
		}
	}
return nIndex;
}

/************************************************************************
 *  ScMpDoc.cpp				M a p P a t h T o I n d e x					*
 ************************************************************************/
short CScanMapDoc::MapPathToIndex (CString szPath)
{
short nIndex = -1;
short	i;

BOOL bFound = FALSE;

for (i=0; i<m_ScannedMaps.GetSize() && !bFound; i++)
	{
	CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i);
	if (ptMap != NULL)
		{	
		CString szTestPath = ptMap->GetPath();	
		bFound = (szTestPath.Compare (szPath) == 0);
		if (bFound)
			nIndex = i;
		}
	}
return nIndex;
}



/************************************************************************
 *  ScMpDoc.cpp   		 	I n c r e a s e C n t O K					*
 ************************************************************************/
BOOL CScanMapDoc::IncreaseCntOK (CScannedMap* ptMap)
{
	long lSizeX, lSizeY;
	
	ptMap->GetSizeLP(&lSizeX, &lSizeY);

	CSize sImage(lSizeX, lSizeY);
	CSize sVirtLP;

	return this->CalcActFakt (sImage, &sVirtLP);
}

/************************************************************************
 *  ScMpDoc.cpp   		 	A c t i v a t e M a p						*
 ************************************************************************/
BOOL CScanMapDoc::ActivateMap (short nActIndex, CSize* ptSizeTotal)
{
	BOOL bFound = FALSE;

	CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(nActIndex);
	if (ptMap != NULL && m_ptScMpView != NULL)
	{			// get path, open bmp file and set act index
		if (this->OpenBmpFile (ptMap->GetPath()))		// creates new m_ptDib!!
		{								// put new image ptr into scan map view
			m_ptScMpView->SetImagePtr(m_ptDib);

			this->SetActMapIndex (nActIndex);

			
			double dOldMapFaktX = ptMap->GetScaleFaktX();
			double dOldMapFaktY = ptMap->GetScaleFaktY();
			if (m_dActFaktX > dOldMapFaktX ||
				m_dActFaktY > dOldMapFaktY)
			{			// make sure, that this factor is not too large
				while (!this->IncreaseCntOK(ptMap))
					ChangeIncreaseCnt(-1);
			}

			long lSizeX, lSizeY;			// get size of next map
			ptMap->SetScaleFaktX(m_dActFaktX);	// put actual X scaling factor into new map
			ptMap->SetScaleFaktY(m_dActFaktY);	// put actual Y scaling factor into new map
			ptMap->GetScaledSizeLP (&lSizeX, &lSizeY);

			if (ptSizeTotal != NULL)
				{
				ptSizeTotal->cx = lSizeX;
				ptSizeTotal->cy = lSizeY;
				}

			bFound = TRUE;

		}
	}
	return bFound;
}

/************************************************************************
 *  ScMpDoc.cpp		A c t i v a t e S c M a p F o r L a t L o n			*
 *  Purpose: Activate scanned map for best view of specified location	*
 ************************************************************************/
BOOL CScanMapDoc::ActivateScMapForLatLon (CLatLon LatLon)
{					
	BOOL	bFound=FALSE;
	double	dMaxFakt=0;
	short i, nNewMapInd, nOldMapInd;

	short nCnt = GetMapCnt();

	nOldMapInd = this->GetActMapIndex();
	nNewMapInd = -1;

	long	lX_LP, lY_LP;		
	for (i=0; i<nCnt; i++)
	{
		CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i);
		if (ptMap != NULL)
		{
			if (ptMap->IsAutoSel() && ptMap->IsCalibrated())
			{
				ptMap->LatLonToLP (LatLon.GetLon(), LatLon.GetLat(), &lX_LP, &lY_LP);
				double dActFakt = ptMap->GetCenterFakt (lX_LP, lY_LP);

				if (dActFakt > dMaxFakt)
				{
					dMaxFakt = dActFakt;
					nNewMapInd = i;
				}

			} // AutoSel ?
		} // ptMap
	} // end for all maps

if (nNewMapInd >= 0)
	{	
	if (nOldMapInd == nNewMapInd)
		bFound = TRUE;
	else	
		bFound = this->ActivateMap (nNewMapInd);
	}
return bFound;
}




/************************************************************************
 *  ScMpDoc.cpp				G e t V A C I n d e x F o r					*
 ************************************************************************/
BOOL CScanMapDoc::GetVACIndexFor (CLatLon llAerodrome, short* ptVACIndex)
{
	BOOL	bFound		=FALSE;
	double	dMaxFakt	=0;
	short	nNewMapInd	= -1;

	short nCnt = GetMapCnt();
	for (int i=0; i<nCnt; i++)
	{
		CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i);
		if (ptMap != NULL)
		{
			if (ptMap->IsCalibrated() && ptMap->GetProjection() == LINEAR)
			{			// linear calibrated map
												// is visual approach chart for llAerodrome?
				long	lX_LP, lY_LP;		
				ptMap->LatLonToLP (llAerodrome.GetLon(), llAerodrome.GetLat(), &lX_LP, &lY_LP);
				double dActFakt = ptMap->GetCenterFakt (lX_LP, lY_LP);

				if (dActFakt > dMaxFakt)
				{
					dMaxFakt = dActFakt;
					nNewMapInd = i;
					bFound = TRUE;
				}
			}
		}
	}

	if (bFound && (ptVACIndex != NULL))
		*ptVACIndex = nNewMapInd;

	return bFound;
}

/************************************************************************
 *  ScMpDoc.cpp					I s O n V A C 							*
 ************************************************************************/
BOOL CScanMapDoc::IsOnVAC (double dLat, double dLon, CLatLon llAerodrome, short* ptVACIndex)
{
	BOOL bOnVAC = FALSE;

	short nNewMapInd = -1;
	if (this->GetVACIndexFor(llAerodrome, &nNewMapInd))
	{				// VAC for llAerodrome found

		CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(nNewMapInd);
		if (ptMap != NULL)
		{
			if (ptMap->IsLatLonVisible(dLat, dLon))
			{
				bOnVAC = TRUE;
				*ptVACIndex = nNewMapInd;
			}
		}
	}	

	return bOnVAC;
}

/************************************************************************
 *  ScMpDoc.cpp				A c t i v a t e N e a r e s t M a p			*
 ************************************************************************/
BOOL CScanMapDoc::ActivateNearestMap (CWayDoc* ptWayDoc)
{
BOOL	bFound=FALSE;
short i, nMaxFound, nNewMapInd, nOldMapInd;
short nCnt = GetMapCnt();

nOldMapInd = this->GetActMapIndex();
nNewMapInd = -1;
nMaxFound=0;
for (i=0; i<nCnt; i++)
	{
	CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i);
	if (ptMap != NULL)
		{
		if (ptMap->IsAutoSel() && ptMap->IsCalibrated())
			{
			short j, nActFound;

			nActFound=0;					// for all waypoints of route:
			for (j=0; j<ptWayDoc->GetWayPointCnt(); j++)
				{
				CWayPoint WayPt;
				if (ptWayDoc->GetWayPointPtr (j, &WayPt))
					{						// is actual WayPt on ptMap?? 
					if (ptMap->IsLatLonVisible(WayPt.GetLat(), WayPt.GetLon()))
						nActFound++;
					}
				}  // end for all way points of actual route

			if (nActFound > nMaxFound)
				{
				nMaxFound = nActFound;
				nNewMapInd = i;
				}
			} // AutoSel ?
		} // ptMap
	} // end for all maps

if (nNewMapInd >= 0)
	{	
	if (nOldMapInd == nNewMapInd)
		bFound = TRUE;
	else	
		bFound = this->ActivateMap (nNewMapInd);
	}
return bFound;
}

/************************************************************************
 *  ScMpDoc.cpp				A c t i v a t e N e a r e s t M a p			*
 ************************************************************************/
BOOL CScanMapDoc::ActivateNearestMap (CTrackDoc* ptTrackDoc)
{
BOOL	bFound=FALSE;
short i, nMaxFound, nNewMapInd, nOldMapInd;
short nCnt = GetMapCnt();


nOldMapInd = this->GetActMapIndex();
nNewMapInd = -1;
nMaxFound=0;
for (i=0; i<nCnt; i++)
	{
	CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i);
	if (ptMap != NULL)
		{
		if (ptMap->IsAutoSel() && ptMap->IsCalibrated())
			{
			short j, nActFound, nWptCnt, nStep;
			long lSizeX, lSizeY;
			CRect rMapLP;
											// get logical area of map
			ptMap->GetScaledSizeLP (&lSizeX, &lSizeY);
			rMapLP.SetRect (0,0, lSizeX, -lSizeY);

			nActFound=0;					// for all waypoints of track:
			nWptCnt = ptTrackDoc->GetCnt();
			nStep = (nWptCnt > 20)? nWptCnt / 20 : 1;
			for (j=0; j<nWptCnt; j+=nStep)
				{
				CTrackPoint TrackPt;
				double	dLat, dLon;
				long	lX_LP, lY_LP;
				if (ptTrackDoc->GetTrackPointPtr (j, &TrackPt))
					{						
					if (TrackPt.GetLatLon (&dLat, &dLon))
						{						// is actual TrackPt on ptMap?? 
						ptMap->LatLonToLP (dLon, dLat, &lX_LP, &lY_LP);
						if ((rMapLP.left <= lX_LP && lX_LP <= rMapLP.right) &&
							(rMapLP.top >= lY_LP && lY_LP >= rMapLP.bottom))
							nActFound++;
						}
					}
				}  // end for all way points of actual route

			if (nActFound > nMaxFound)
				{
				nMaxFound = nActFound;
				nNewMapInd = i;
				}
			}  // AutoSel?
		} // ptMap
	} // end for all maps

if (nNewMapInd >= 0)
	{	
	if (nOldMapInd == nNewMapInd)
		bFound = TRUE;
	else	
		bFound = this->ActivateMap (nNewMapInd);
	}
return bFound;
}

/************************************************************************
 *  ScMpDoc.cpp			S e a r c h N e x t M o v i n g M a p			*
 *  Purpose: searches map with max. free space in moving direction		*
 ************************************************************************/
BOOL CScanMapDoc::SearchNextMovingMap (CTrackDoc* ptTrackDoc, CSize* ptSizeTotal)
{
BOOL	bFound=false;
double	dMaxFreeFakt = 0;
short	nNewMapInd = -1;

int nCnt = GetMapCnt();

short	nOldMapInd = this->GetActMapIndex();

for (int i=0; i<nCnt; i++)
	{
	CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i);
	if (ptMap != NULL)
		{
		BOOL bLastVisible=false;
		CTrackPoint TrackPt;
		double dLat1, dLon1;
		long lLPx1, lLPy1;
		
		short nTrackCnt = ptTrackDoc->GetCnt();
		
		if (ptMap->IsAutoSel() && ptMap->IsCalibrated())
			{
			if (ptTrackDoc->GetTrackPointPtr (nTrackCnt-1, &TrackPt))
				{						
				if (TrackPt.GetLatLon (&dLat1, &dLon1))
					{						// is actual TrackPt on ptMap?? 					
					long lSizeX, lSizeY;
					CRect rMapLP;
											// get logical area of map
					ptMap->GetScaledSizeLP (&lSizeX, &lSizeY);
					rMapLP.SetRect (0,0, lSizeX, -lSizeY);					
										
					ptMap->LatLonToLP (dLon1, dLat1, &lLPx1, &lLPy1);
					
					bLastVisible = ((rMapLP.left <= lLPx1 && lLPx1 <= rMapLP.right) &&
									(rMapLP.top >= lLPy1 && lLPy1 >= rMapLP.bottom));						
					}
				}

			if (bLastVisible)
				{						// get previous point with Log Koord != Log koord of last point
				BOOL bTwoDifferentPoints = false;
				double dLat0, dLon0;
				long 	lLPx0, lLPy0;
				
				if (nTrackCnt > 1)
					{
					short j=nTrackCnt-2;
					do	{
						if (ptTrackDoc->GetTrackPointPtr (j--, &TrackPt))
							{
							if (TrackPt.GetLatLon (&dLat0, &dLon0))
								{						// is actual TrackPt on ptMap?? 					
								ptMap->LatLonToLP (dLon0, dLat0, &lLPx0, &lLPy0);
								bTwoDifferentPoints = ((lLPx0 != lLPx1) || (lLPy0 != lLPy1));
								}	
							}				
						} while (!bTwoDifferentPoints && j>=0);
					}
					
				if (bTwoDifferentPoints)
					{						// actual course available:
					double dFreeFakt = ptMap->GetFreeFakt (lLPx0, lLPy0, lLPx1, lLPy1);
					if (dFreeFakt > dMaxFreeFakt)
						{
						dMaxFreeFakt = dFreeFakt;
						nNewMapInd = i;
						}
					}
				else{					// no course available:
					nNewMapInd = i;		// only one track point: use this map
										// evtl. besser: suche karte, die LatLon mglichst in Mitte zeigt
					}
				}
			}
		} // ptMap != nil
	} // next map


if (nNewMapInd >= 0)
	{	
	if (nOldMapInd == nNewMapInd)
		{
		bFound = true;
		}
	else{	
		bFound = this->ActivateMap (nNewMapInd, ptSizeTotal);
		}

	}
return bFound;
}

/************************************************************************
 *  ScMpDoc.cpp   		 	I s G o o d C e n t e r M a p				*
 *  Purpose: returns TRUE, if llActCenter is visible on ptMap			*
 ************************************************************************/
BOOL CScanMapDoc::IsGoodCenterMap (CLatLon llActCenter, CScannedMap* ptMap)
{
	BOOL	bSelect = FALSE;
								// center of actual map should be seen on new map
	long LPx, LPy;
	ptMap->LatLonToLP (llActCenter.GetLon(), llActCenter.GetLat(), &LPx, &LPy);
	if (ptMap->IsInside(LPx, LPy))
	{					 
		bSelect = TRUE;
	}	
		 
	return bSelect;
}


/************************************************************************
 *  ScMpDoc.cpp   		 	I s G o o d N e x t M a p					*
 *  Purpose: looks for the closest map in a specified direction (nDir)	*
 ************************************************************************/
BOOL CScanMapDoc::IsGoodNextMap (CLatLon llActCenterSide, CLatLon llTestCenterSide,
							double  dActMapWidth,
							short nDir, CScannedMap* ptMap, short* ptMaxQual)
{
	BOOL	bSelect = FALSE;
	short nAngle		= 60;		
	short nActQuality	= 0;
	short nCourseQual	= 0;
	short nDistQual		= 0;
	double dDist;		// best: 0
	double dCourse;		// best: nDir
	double dTestMapWidth;
	short	nActCourseDiff;


								// get center lat lon of test map 
	CLatLon llCenter = ptMap->GetCenterLatLon();
								// get with [NM] of test map
	dTestMapWidth= ptMap->GetWidth_NM (llCenter);

								// dist from actual map side to test map side
	dDist = llActCenterSide.LoxoDist (llTestCenterSide, &dCourse);
	if (nDir == 0) if (dCourse > 180) dCourse -= 360;


								// check course quality

	nActQuality=0;
		
	nActCourseDiff = (short)(dCourse - nDir);
	nActCourseDiff = (nActCourseDiff>=0)? nActCourseDiff : -nActCourseDiff;


	short nMaxDir		= nDir+nAngle;
	short nMinDir		= nDir-nAngle;
	if (dCourse > nMinDir && dCourse < nMaxDir)
	{
		nCourseQual = 100 - nActCourseDiff*100/nAngle;
	}
	else
	{
		return bSelect;
	}

							// check distance quality
	nDistQual = (short)(dActMapWidth / dDist * 100);
	 


	nActQuality = nCourseQual + nDistQual;

	if (nActQuality > *ptMaxQual)
	{
		bSelect = TRUE;
		*ptMaxQual = nActQuality;
	}

	return bSelect;
}



/************************************************************************
 *  ScMpDoc.cpp   		 AssignIncreaseCountersFor						*
 *  Purpose: selects maps to be used for the position specified			*
 ************************************************************************/
BOOL CScanMapDoc::AssignIncreaseCountersFor(CDC* pDC, double dLat, double dLon)
{
	BOOL	bAssigned=FALSE;

	long lActMapScale;
	CPoint pHorzResDPI;
	double dActMapWidth;

	if (m_nActMapIndex < 0)
		return bAssigned;



	pHorzResDPI.x = pDC->GetDeviceCaps (LOGPIXELSX);
	pHorzResDPI.y = pDC->GetDeviceCaps (LOGPIXELSY);
	pDC->DPtoLP (&pHorzResDPI);



								// begin getting data of actual map
	CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(m_nActMapIndex);
	if (ptMap != NULL)
	{						// get center lat lon of test map 
		lActMapScale = ptMap->GetScale (pHorzResDPI.x);	
		CLatLon llCenter = ptMap->GetCenterLatLon();
							// get with [NM] of test map
		dActMapWidth= ptMap->GetWidth_NM (llCenter);
	}
	else
	{
		return bAssigned;
	}
							// end getting data of actual map

	CLatLon llCenterMiddle;
	llCenterMiddle.SetLatLon(dLat,dLon);		// actual center of screen, NOT center of map

	m_ScaledMapList.PrepareNewList(m_nActMapIndex, lActMapScale);

	short nCnt = GetMapCnt();


	for (int i=0; i<nCnt; i++)
	{
		ptMap = NULL;
		ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i);
		if (ptMap != NULL)
		{
			if (ptMap->IsCalibrated())
			{
				CLatLon OldTL;
				CLatLon OldBR;
				CLatLon llTestCenterMiddle;

				OldTL = ptMap->GetBorderLatLon (CALPOS_TL);
				OldBR = ptMap->GetBorderLatLon (CALPOS_BR);
				llTestCenterMiddle.SetLatLon ((OldTL.GetLat() + OldBR.GetLat())/2,
										 (OldTL.GetLon() + OldBR.GetLon())/2);

				if (this->IsGoodCenterMap (llCenterMiddle, ptMap))
				{
					BOOL bAutoSelMapInScaledMapList = m_ScaledMapList.HasAutoSelMap();

					BOOL bAutoSel = ptMap->IsAutoSel();		// new map
					 
					if (!bAutoSelMapInScaledMapList || !bAutoSel)
					{			// only one autoSel map is allowed in m_ScaledMapList!
						// calculates map scale: 1 : XX
						long lMapScale = ptMap->GetScale (pHorzResDPI.x);	
						CString szName = ptMap->GetName();

						m_ScaledMapList.Add(szName, i, bAutoSel, lMapScale);
						bAssigned = TRUE;
					}
				} // good center map
			} // calibrated
		}
	} // end for all maps

	return bAssigned;
}

/************************************************************************
 *  ScMpDoc.cpp   		 		 N e x t M a p M i d d l e				*
 *  Purpose: Searches new resolution map to replace actual map			*
 *  if bFound=FALSE, ptInitialIncreaseCnt returns the increase cnt,		*
 *  for which the m_nActMapIndex map is to be shown the first time		*
 ************************************************************************/
BOOL CScanMapDoc::NextMapMiddle (CSize* ptSizeTotal, BOOL bLarger, BOOL* ptMapChanged)
{
	BOOL	bFound=FALSE;

	short nNewIndex = -1;
	*ptMapChanged = FALSE;

	// don't look for next map index, if current map is scaled up or down
	if (m_nIncreaseCnt == 0)
		 nNewIndex = m_ScaledMapList.GetNextMapIndex(bLarger);	// get other map
	else nNewIndex = m_nActMapIndex;							// use current map

	if (nNewIndex >= 0)
	{		
		if (nNewIndex != m_nActMapIndex)
		{					// switch to new map, stay on increase cnt 0
			m_nActMapIndex	= nNewIndex;

			bFound = this->ActivateMap (nNewIndex, ptSizeTotal);
			*ptMapChanged = bFound;
		}
		else
		{					// scale current map up or down
			bFound = TRUE;		// use same map as before
		}
	}

	return bFound;
}


/************************************************************************
 *  ScMpDoc.cpp   		 		 N e x t M a p E a s t					*
 *  Purpose: Searches map next of east side of actual map				*
 ************************************************************************/
BOOL CScanMapDoc::NextMapEast (CSize* ptSizeTotal, double dActLat)
{
BOOL	bFound=FALSE;
CScannedMap* ptMap=NULL;
CLatLon OldTR;
CLatLon OldBR;
CLatLon llCenterRight;
short i, nCnt, nMaxQuality, nActIndex;
double	dActMapWidth;

if (m_nActMapIndex < 0)
	return FALSE;

						// begin getting data of actual map
ptMap = (CScannedMap*)m_ScannedMaps.GetAt(m_nActMapIndex);
if (ptMap != NULL)
	{
	OldTR = ptMap->GetBorderLatLon (CALPOS_TR);
	OldBR = ptMap->GetBorderLatLon (CALPOS_BR);
	llCenterRight.SetLatLon ((OldTR.GetLat() + OldBR.GetLat())/2,
							 (OldTR.GetLon() + OldBR.GetLon())/2);

							// get center lat lon of test map 
	CLatLon llCenter = ptMap->GetCenterLatLon();
								// get with [NM] of test map
	dActMapWidth= ptMap->GetWidth_NM (llCenter);
	}
else{
	return bFound;
	}
// end getting data of actual map

nCnt = GetMapCnt();

nActIndex = -1;
nMaxQuality=0;


for (i=0; i<nCnt; i++)
	{
	if (i != m_nActMapIndex)
		{			// dont compare with actual map!!
		ptMap = NULL;
		ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i);
		if (ptMap != NULL)
			{
			CLatLon llTestCenterRight;
			OldTR = ptMap->GetBorderLatLon (CALPOS_TR);
			OldBR = ptMap->GetBorderLatLon (CALPOS_BR);

			double dTestLat;
			if(OldBR.GetLat() < dActLat && dActLat < OldTR.GetLat())
					dTestLat = dActLat;
			else	dTestLat = (OldTR.GetLat() + OldBR.GetLat())/2;

			llTestCenterRight.SetLatLon (dTestLat,
									 (OldTR.GetLon() + OldBR.GetLon())/2);


			if (ptMap->IsAutoSel() &&
				this->IsGoodNextMap (llCenterRight, llTestCenterRight, dActMapWidth, 
										90, ptMap, &nMaxQuality))
				{
				nActIndex = i;
				}
			} // ptMap
		}
	} // end for all maps

if (nActIndex >= 0)
	{		
	bFound = this->ActivateMap (nActIndex, ptSizeTotal);
	}
return bFound;
}

/************************************************************************
 *  ScMpDoc.cpp   		 		 N e x t M a p W e s t					*
 *  Purpose: Searches map next of west side of actual map				*
 ************************************************************************/
BOOL CScanMapDoc::NextMapWest (CSize* ptSizeTotal, double dActLat)
{
BOOL	bFound=FALSE;
CScannedMap* ptMap=NULL;
CLatLon OldTL;
CLatLon OldBL;
CLatLon llCenterLeft;
short i, nCnt, nMaxQuality, nActIndex;
double	dActMapWidth;

if (m_nActMapIndex < 0)
	return FALSE;

						// begin getting data of actual map
ptMap = (CScannedMap*)m_ScannedMaps.GetAt(m_nActMapIndex);
if (ptMap != NULL)
	{
	OldTL = ptMap->GetBorderLatLon (CALPOS_TL);
	OldBL = ptMap->GetBorderLatLon (CALPOS_BL);
	llCenterLeft.SetLatLon ((OldTL.GetLat() + OldBL.GetLat())/2,
							(OldTL.GetLon() + OldBL.GetLon())/2);

							// get center lat lon of test map 
	CLatLon llCenter = ptMap->GetCenterLatLon();
								// get with [NM] of test map
	dActMapWidth= ptMap->GetWidth_NM (llCenter);
	}
else{
	return bFound;
	}
// end getting data of actual map

nCnt = GetMapCnt();

nActIndex = -1;
nMaxQuality=0;


for (i=0; i<nCnt; i++)
	{
	if (i != m_nActMapIndex)
		{			// dont compare with actual map!!
		ptMap = NULL;
		ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i);
		if (ptMap != NULL)
			{
			CLatLon llTestCenterLeft;
			OldTL = ptMap->GetBorderLatLon (CALPOS_TL);
			OldBL = ptMap->GetBorderLatLon (CALPOS_BL);

			double dTestLat;
			if(OldBL.GetLat() < dActLat && dActLat < OldTL.GetLat())
					dTestLat = dActLat;
			else	dTestLat = (OldTL.GetLat() + OldBL.GetLat())/2;


			llTestCenterLeft.SetLatLon (dTestLat,
									(OldTL.GetLon() + OldBL.GetLon())/2);

			if (ptMap->IsAutoSel() &&
				this->IsGoodNextMap (llCenterLeft, llTestCenterLeft, dActMapWidth,
									270, ptMap, &nMaxQuality))
				{
				nActIndex = i;
				}
			} // ptMap
		}
	} // end for all maps

if (nActIndex >= 0)
	{			
	bFound = this->ActivateMap (nActIndex, ptSizeTotal);
	}
return bFound;
}

/************************************************************************
 *  ScMpDoc.cpp   		 		 N e x t M a p S o u t h				*
 *  Purpose: Searches map next of south side of actual map				*
 ************************************************************************/
BOOL CScanMapDoc::NextMapSouth (CSize* ptSizeTotal, double dActLon)
{
BOOL	bFound=FALSE;
CScannedMap* ptMap=NULL;
CLatLon OldBR;
CLatLon OldBL;
CLatLon llCenterSouth;
short i, nCnt, nMaxQuality, nActIndex;
double	dActMapWidth;

if (m_nActMapIndex < 0)
	return FALSE;

						// begin getting data of actual map
ptMap = (CScannedMap*)m_ScannedMaps.GetAt(m_nActMapIndex);
if (ptMap != NULL)
	{
	OldBR = ptMap->GetBorderLatLon (CALPOS_BR);
	OldBL = ptMap->GetBorderLatLon (CALPOS_BL);
	llCenterSouth.SetLatLon ((OldBR.GetLat() + OldBL.GetLat())/2,
							(OldBR.GetLon() + OldBL.GetLon())/2);

							// get center lat lon of test map 
	CLatLon llCenter = ptMap->GetCenterLatLon();
								// get with [NM] of test map
	dActMapWidth= ptMap->GetWidth_NM (llCenter);
	}
else{
	return bFound;
	}
// end getting data of actual map

nCnt = GetMapCnt();

nActIndex = -1;
nMaxQuality=0;

for (i=0; i<nCnt; i++)
	{
	if (i != m_nActMapIndex)
		{			// dont compare with actual map!!
		ptMap = NULL;
		ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i);
		if (ptMap != NULL)
			{
			CLatLon llTestCenterSouth;
			OldBR = ptMap->GetBorderLatLon (CALPOS_BR);
			OldBL = ptMap->GetBorderLatLon (CALPOS_BL);

			double dTestLon;
			if(OldBL.GetLon() < dActLon && dActLon < OldBR.GetLon())
					dTestLon = dActLon;
			else	dTestLon = (OldBR.GetLon() + OldBL.GetLon())/2;

			llTestCenterSouth.SetLatLon ((OldBR.GetLat() + OldBL.GetLat())/2,
									dTestLon);

			if (ptMap->IsAutoSel() &&
				this->IsGoodNextMap (llCenterSouth, llTestCenterSouth, dActMapWidth,
									180, ptMap, &nMaxQuality))
				{
				nActIndex = i;
				}
			} // ptMap
		}
	} // end for all maps

if (nActIndex >= 0)
	{			
	bFound = this->ActivateMap (nActIndex, ptSizeTotal);
	}
return bFound;
}

/************************************************************************
 *  ScMpDoc.cpp   		 		 N e x t M a p N o r t h				*
 *  Purpose: Searches map next of north side of actual map				*
 ************************************************************************/
BOOL CScanMapDoc::NextMapNorth (CSize* ptSizeTotal, double dActLon)
{
BOOL	bFound=FALSE;
CScannedMap* ptMap=NULL;
CLatLon OldTR;
CLatLon OldTL;
CLatLon llCenterNorth;
short i, nCnt, nMaxQuality, nActIndex;
double	dActMapWidth;

if (m_nActMapIndex < 0)
	return FALSE;

						// begin getting data of actual map
ptMap = (CScannedMap*)m_ScannedMaps.GetAt(m_nActMapIndex);
if (ptMap != NULL)
	{
	OldTR = ptMap->GetBorderLatLon (CALPOS_TR);
	OldTL = ptMap->GetBorderLatLon (CALPOS_TL);
	llCenterNorth.SetLatLon ((OldTR.GetLat() + OldTL.GetLat())/2,
							 (OldTR.GetLon() + OldTL.GetLon())/2);

							// get center lat lon of test map 
	CLatLon llCenter = ptMap->GetCenterLatLon();
								// get with [NM] of test map
	dActMapWidth= ptMap->GetWidth_NM (llCenter);
	}
else{
	return bFound;
	}
// end getting data of actual map

nCnt = GetMapCnt();

nActIndex = -1;
nMaxQuality=0;

for (i=0; i<nCnt; i++)
	{
	if (i != m_nActMapIndex)
		{			// dont compare with actual map!!
		ptMap = NULL;
		ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i);
		if (ptMap != NULL)
			{
			CLatLon llTestCenterNorth;
			OldTR = ptMap->GetBorderLatLon (CALPOS_TR);
			OldTL = ptMap->GetBorderLatLon (CALPOS_TL);

			if (ptMap->GetName().Compare("Frankfurt")==0 ||
				ptMap->GetName().Compare("Hannover")==0)
				int bDebug=TRUE;

			double dTestLon;
			if(OldTL.GetLon() < dActLon && dActLon < OldTR.GetLon())
					dTestLon = dActLon;
			else	dTestLon = (OldTR.GetLon() + OldTL.GetLon())/2;

			llTestCenterNorth.SetLatLon ((OldTR.GetLat() + OldTL.GetLat())/2,
									 dTestLon);

			if (ptMap->IsAutoSel() &&
				this->IsGoodNextMap (llCenterNorth, llTestCenterNorth, dActMapWidth,
									0, ptMap, &nMaxQuality))
				{
				nActIndex = i;
				}
			} // ptMap
		}
	} // end for all maps

if (nActIndex >= 0)
	{			
	bFound = this->ActivateMap (nActIndex, ptSizeTotal);
	}
return bFound;
}

/************************************************************************
 *  ScMpDoc.cpp	  		 		 C h a n g e 							*
 ************************************************************************/
BOOL CScanMapDoc::Change(CWnd* ptWnd)
{
BOOL bChanged=FALSE;

CScanMapDlg ScMpDlg(ptWnd, this);
   
int RetVal = ScMpDlg.DoModal();

switch (RetVal)
	{
	case IDOK:  
		{ 
		CString szPath = this->GetPathName(); 
		if (ScMpDlg.WriteDataBaseIntoDoc()) 
			{             
		//	this->OnSaveDocument (szPath);   // activate to convert map file only
 			bChanged = TRUE;
 			}    
		this->SetAutoSelect(FALSE);		// avoid activating other map to view route
		} 
		break;
	case IDCANCEL:
		break;
	}	
return bChanged;
} 


/************************************************************************
 *  ScMpDoc.cpp	  	 C h a n g e C a l i b r a t i o n					*
 ************************************************************************/
BOOL CScanMapDoc::ChangeCalibration(CScanMapView* ptScMpView)
{
	BOOL bOK=FALSE;

	if (!ptScMpView->IsDrawn())
		return bOK;

	CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(m_nActMapIndex);
	if (ptMap != NULL)
	{					 
		int RetVal;

		switch (ptMap->GetProjection())
		{
		case LAMBERT:
			{
				CScMpCalibDlg ScanMapCalibDlg((CWnd*)ptScMpView, this);  										 
				RetVal = ScanMapCalibDlg.DoModal();	
			}
			break;
		case LINEAR:
			{
				CLinearCalibDlg ScanMapCalibDlg((CWnd*)ptScMpView, this);  										 
				RetVal = ScanMapCalibDlg.DoModal();	
			}
			break;
		case MERCATOR:
			break;
		case POLAR:
			break;
		}

			
		switch (RetVal)
		{
		case IDOK: 
			if (ptMap->IsCalibrated())
			{
				ptScMpView->Invalidate(FALSE);	// remove green circles
			}
			else
			{
				short nCalIndex = ptMap->GetActCalIndex();
				if (nCalIndex >= 0)
				{
					CALIBRATEPOS CalPos = ptMap->FixPtIndexToCalibPos(nCalIndex);
					switch (CalPos)
					{
					case CALPOS_TL:
						ptScMpView->ScrollVertMax (FALSE);
						ptScMpView->ScrollHorzMax (FALSE);
						break;

					case CALPOS_TR:
						ptScMpView->ScrollVertMax (FALSE);
						ptScMpView->ScrollHorzMax (TRUE);
						break;

					case CALPOS_BR:
						ptScMpView->ScrollVertMax (TRUE);
						ptScMpView->ScrollHorzMax (TRUE);
						break;

					case CALPOS_BL:
						ptScMpView->ScrollVertMax (TRUE);
						ptScMpView->ScrollHorzMax (FALSE);
						break;
					}			// message with next instruction 
					AfxMessageBox (ptMap->GetActCalInfoText());	
								// no parameter: GetActCalInfoText uses m_nActCalIndex
								// to decide which text to show
				}
			}

			bOK=TRUE;
			break;

		case IDCANCEL:
			ptScMpView->Invalidate(FALSE);	// remove green circle
			break;
		}	
	}

return bOK;
} 

/************************************************************************
 *  ScMpDoc.cpp				S a v e M o u s e P o s						*
 ************************************************************************/
void CScanMapDoc::SaveMousePos(CPoint pMouseLP)
{
CScannedMap* ptMap = (CScannedMap*)m_ScannedMaps.GetAt(m_nActMapIndex);

if (ptMap != NULL)
	{		   
	ptMap->SaveMousePos (pMouseLP);
	}
}

/************************************************************************
 *  ScMpDoc.cpp				A c t i v a t e I m p o r t M o d e			*
 ************************************************************************/
void CScanMapDoc::ActivateImportMode (CString szBmpPath)
{
m_szDestBmpPath = szBmpPath;
m_bImport = TRUE;
}


/************************************************************************
 *  ScMpDoc.cpp				O n O p e n D o c u m e n t					*
 ************************************************************************/
BOOL CScanMapDoc::OnOpenDocument(LPCTSTR pszPathName)
{
BOOL bOpened = FALSE;

bOpened = CDocument::OnOpenDocument(pszPathName); 

if (bOpened)
	{						
	CScannedMap* ptMap = NULL;
	if (m_nActMapIndex >= 0 && m_nActMapIndex<m_ScannedMaps.GetSize())
		{
		if ((ptMap = (CScannedMap*)m_ScannedMaps.GetAt(m_nActMapIndex)) != NULL)
			{				 // load bitmap of actual selected map
			}
		}
	}
return bOpened;
}

/************************************************************************
 *  ScMpDoc.cpp	  		 		 S e r i a l i z e 						*
 ************************************************************************/
void CScanMapDoc::Serialize(CArchive& ar)
{ 
int	i, nMapCnt;
   
if (ar.IsStoring())
	{
		// TODO: add storing code here 
	ar << (WORD)m_nActMapIndex;
		 
	ar << m_ScannedMaps.GetSize(); 
	for (i=0; i<m_ScannedMaps.GetSize(); i++)
		{
		CScannedMap* ptMap = NULL;

		if ((ptMap = (CScannedMap*)m_ScannedMaps.GetAt(i)) != NULL)
			{
			ptMap->Serialize (ar);
			}
		}
	}
else{
	WORD Word;
		// TODO: add loading code here  

	ar >> Word;  
	if (!m_bImport)			// don't change act index while importing
		m_nActMapIndex = (short)Word;

	ar >> nMapCnt;

	m_bImported = FALSE;
	for (i=0; i<nMapCnt; i++)
		{
		CScannedMap* ptMap = (CScannedMap*)new CScannedMap ();
		ptMap->Serialize (ar);

		if (m_bImport)	
			{
			short nIndex;

			CString szTitle = ptMap->GetTitle();	   // N50E0100.bmp
			ptMap->SetPath(m_szDestBmpPath + "\\" + szTitle);


			nIndex = MapNameToIndex (ptMap->GetName());
			if (nIndex > -1)
				{			// Name of imported map already used
				DeleteMap (nIndex);	// delete old map entry from document list
				}

			nIndex = MapPathToIndex (ptMap->GetPath());
			if (nIndex > -1)
				{			// full file name of imported map already used
				DeleteMap (nIndex);	// delete old map entry from document list
				}

			if(bDemo)
			{
				if (szTitle.CompareNoCase("Germany.bmp") == 0)
					ptMap->InvertActivateState (SHOW_AIRSPACE);
			}

			nIndex = this->GetSortedIndex (m_ScannedMaps, ptMap);
			m_ScannedMaps.InsertAt (nIndex, ptMap);
			m_bImported = TRUE;
			}	
		else{			// add map only, if corresponding file exists
			if (CInitDoc::FileExists(ptMap->GetPath()))
				{
				m_ScannedMaps.Add (ptMap);
				}
			else{
				CString szMsg;
				szMsg = CDlgTool::Format (IDF_DELMISSINGMAP, ptMap->GetName());
				if (AfxMessageBox ((LPCSTR)szMsg, MB_YESNO) == IDNO)
					m_ScannedMaps.Add (ptMap);
				}
			}
		}

	if (m_bImport)				// deactivate import mode
		m_bImport = FALSE;

	nMapCnt = m_ScannedMaps.GetSize();
	if (m_nActMapIndex >= nMapCnt)		// bad index
		m_nActMapIndex = (nMapCnt > 0)? nMapCnt-1 : -1;
	}
}


 
