// ScMpView.cpp : implementation of the CScanMapView class
//
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include <AFXPRIV.H>			// for WM_INITIALUPDATE
#include <math.h>				// for sin, cos

#include "pf.h"

#include "InitDoc.h"  
#include "MetDoc.h"
#include "LocDoc.h"
#include "PlaneDoc.h"

#include "Calc.h"			// this statement must be after PlaneDoc.h
#include "MainFrm.h"		// for EnableNextCmd

#include "WayDoc.h"
#include "TrackDoc.h"

#include "Map.h"			// RectAlignRight
#include "ElevDoc.h"		// CriticalLegPoints, GetElevCnt		
#include "ScMpDoc.h"
#include "ScMpView.h"                                           

#include "InterMap.h"

#include "..\CPPTOOLS\Vektor.h"                                       
#include "Symbols.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

extern CInitDoc* 		ptInit;
extern CFontDoc* 		ptFontTool;
extern CDimDoc* 		ptDim;      

extern CLoadDoc*		ptLoad; 
extern CCalcDoc*		ptCalc;
extern CAppendixDoc*	ptAppend;  
extern CMetDoc*			ptMet;
extern CPlaneDoc*		ptPlaneDoc;

extern CTrackDoc*		ptTrackDoc;

extern CElevDoc*		ptElevDoc;
extern CScanMapDoc*		ptScMpDoc;
extern BOOL bMMText;


/////////////////////////////////////////////////////////////////////////////
// CScanMapView

IMPLEMENT_DYNCREATE(CScanMapView, CTextViewTool)

BEGIN_MESSAGE_MAP(CScanMapView, CTextViewTool)
	//{{AFX_MSG_MAP(CScanMapView)
	ON_WM_SIZE()
	ON_WM_MOUSEMOVE()
	ON_WM_LBUTTONDOWN()
	ON_WM_CONTEXTMENU()
	ON_COMMAND(ID_WY_INS, OnMenuInsert)
	ON_COMMAND(ID_WY_APP, OnMenuAppend)
	ON_COMMAND(ID_WY_BEAR, OnMenuBear)
	ON_COMMAND(ID_WY_DEL, OnMenuDelete)
	ON_COMMAND(ID_WY_ACT, OnMenuActivate)
	ON_COMMAND(ID_WY_PLALT, OnMenuPlAlt)
	ON_COMMAND_RANGE(ID_WY_INS_FIRST, ID_WY_INS_LAST, OnMenuInsMapLoc)
	ON_COMMAND_RANGE(ID_WY_APP_FIRST, ID_WY_APP_LAST, OnMenuAppMapLoc)
	ON_COMMAND_RANGE(ID_WY_BEAR_FIRST, ID_WY_BEAR_LAST, OnMenuBearMapNav)
	ON_COMMAND_RANGE(ID_WY_DEL_FIRST, ID_WY_DEL_LAST, OnMenuDelRteWpt)
	ON_COMMAND_RANGE(ID_WY_ACT_FIRST, ID_WY_ACT_LAST, OnMenuActRteWpt)
	ON_COMMAND_RANGE(ID_WY_PLALT_FIRST, ID_WY_PLALT_LAST, OnMenuPlAltRteWpt)

	ON_COMMAND_RANGE(ID_ME_LOCEDIT_FIRST, ID_ME_LOCEDIT_LAST, OnMenuMeEditMapLoc)
	ON_COMMAND_RANGE(ID_ME_AIRSPACEEDIT_FIRST, ID_ME_AIRSPACEEDIT_LAST, OnMenuMeEditAirspace)

	ON_UPDATE_COMMAND_UI(ID_WY_INS, OnUpdateWyIns)
	ON_UPDATE_COMMAND_UI(ID_WY_APP, OnUpdateWyApp)
	ON_UPDATE_COMMAND_UI(ID_WY_BEAR, OnUpdateWyBear)
	ON_UPDATE_COMMAND_UI(ID_WY_DEL, OnUpdateWyDel)
	ON_UPDATE_COMMAND_UI(ID_WY_ACT, OnUpdateWyAct)
	ON_UPDATE_COMMAND_UI(ID_WY_PLALT, OnUpdateWyPlAlt)
	ON_COMMAND(ID_EDIT_COPY, OnEditCopy)
	ON_COMMAND(ID_ME_CENTER, OnMenuCenter)
	ON_COMMAND(ID_ME_LARGER	, OnMenuLarger)
	ON_COMMAND(ID_ME_SMALLER, OnMenuSmaller)
	ON_COMMAND(ID_TB_CENTER, OnTBMapCenter)
	ON_COMMAND(ID_TB_LARGER, OnTBMapLarger)
	ON_COMMAND(ID_TB_SMALLER, OnTBMapSmaller)
	ON_COMMAND(ID_ME_AIRPORT, OnShowAirport)
	ON_COMMAND(ID_ME_REPORT, OnShowReport)
	ON_COMMAND(ID_ME_INTERSECT, OnMShowIntersect)
	ON_COMMAND(ID_ME_NDB, OnShowNdb)
	ON_COMMAND(ID_ME_VOR, OnShowVor)
	ON_COMMAND(ID_ME_USR, OnShowUsr)
	ON_COMMAND(ID_ME_SAIL, OnShowSail)
	ON_COMMAND(ID_ME_HELI, OnShowHeli)
	ON_COMMAND(ID_ME_MAP, OnShowMAP)
	ON_COMMAND(ID_ME_LABEL, OnShowLabel)
	ON_COMMAND(ID_ME_ELEV, OnShowElev)
	ON_COMMAND(ID_ME_AIRSPACE, OnShowAirspace)
	ON_UPDATE_COMMAND_UI(ID_ME_AIRPORT, OnUpdateAirport)
	ON_UPDATE_COMMAND_UI(ID_ME_INTERSECT, OnUpdateIntersect)
	ON_UPDATE_COMMAND_UI(ID_ME_NDB, OnUpdateNdb)
	ON_UPDATE_COMMAND_UI(ID_ME_REPORT, OnUpdateReport)
	ON_UPDATE_COMMAND_UI(ID_ME_USR, OnUpdateUsr)
	ON_UPDATE_COMMAND_UI(ID_ME_SAIL, OnUpdateSail)
	ON_UPDATE_COMMAND_UI(ID_ME_HELI, OnUpdateHeli)
	ON_UPDATE_COMMAND_UI(ID_ME_MAP, OnUpdateMAP)
	ON_UPDATE_COMMAND_UI(ID_ME_VOR, OnUpdateVor)
	ON_UPDATE_COMMAND_UI(ID_ME_LABEL, OnUpdateLabel)
	ON_UPDATE_COMMAND_UI(ID_ME_ELEV, OnUpdateElev)
	ON_UPDATE_COMMAND_UI(ID_ME_AIRSPACE, OnUpdateAirspace)
	ON_WM_KEYDOWN()
	//}}AFX_MSG_MAP
	// Standard printing commands
	ON_COMMAND(ID_FILE_PRINT, CTextViewTool::OnFilePrint)
	ON_COMMAND(ID_FILE_PRINT_PREVIEW, CTextViewTool::OnFilePrintPreview)
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CScanMapView construction/destruction

CScanMapView::CScanMapView()
{
	// TODO: add construction code here
ptFontTool->GetParams (m_FontName, &m_wPointSize);

m_CharX = m_LineY = 0;
m_BCharX = m_BLineY = 0;

CreateNewFont(&m_PlainFont);
CreateBoldFont(&m_BoldFont);
m_pDib = NULL;
m_ptMap = NULL;
m_bDrawFixPoint = FALSE;

m_ptInterMap = NULL;
m_ptInterMap = new CInteractiveMap (this);

m_ptColor = ptInit->GetColorPtr();
m_bOnDrawScroll=FALSE;
m_bDrawn = FALSE;

									// for changing size of scanned map
m_sOrigImage.cx = m_sOrigImage.cy = 0;
m_bTBMapCenter = FALSE;
m_bTBMapLarger = FALSE;
m_bTBMapSmaller = FALSE;
}

CScanMapView::~CScanMapView()
{
m_PlainFont.DeleteObject ();
m_BoldFont.DeleteObject ();
if (m_ptInterMap != NULL)
	delete m_ptInterMap;
}

/************************************************************************
 *  ScMpView.cpp    	 	 C h e c k L o a d e d						*
 ************************************************************************/
void CScanMapView::CheckLoaded()
{
	if (m_pDib != NULL)
	{
		if (!m_pDib->IsLoaded())
			m_pDib->Read();
	}
}


/************************************************************************
 *  ScMpView.cpp				F r e e M e m o r y						*
 *	Purpose: Called from CMainFrame::SwitchToView to free memory 		*
 *			before creating MapView Image from ElevArrays				*
 ************************************************************************/
void CScanMapView::FreeMemory()
{
	if (m_pDib != NULL)
		m_pDib->DisposeImagePtr();
}

/************************************************************************
 *  ScMpView.cpp				CheckCalibration						*
 *	Purpose: check calibration of activated map							*
 ************************************************************************/
void CScanMapView::CheckCalibration()
{
	CSize sBmpLP;
	CDC* pDC = GetDC();			// get total size LP in ABSOLUTE (positive) values

	pDC->SetMapMode(MM_LOMETRIC);
	if (m_pDib->GetSizeLP (pDC, &sBmpLP))
	{
		this->CheckCalibration(sBmpLP);
	}

	ReleaseDC(pDC);
}


/************************************************************************
 *  ScMpView.cpp				U p d a t e F o n t s 					*
 ************************************************************************/
void CScanMapView::UpdateFonts (CDC* pDC)
{
ptFontTool->GetParams (m_FontName, &m_wPointSize);  // make font unchangeable
//_fstrcpy ((LPSTR)m_FontName, (LPSTR)"Arial");
//m_wPointSize = 8;
m_lfHeight = this->Y_PointToLP (pDC, m_wPointSize);  

m_PlainFont.DeleteObject ();
CreateNewFont(&m_PlainFont);
this->GetCharSize (pDC, &m_PlainFont, &m_CharX, &m_LineY); 

m_BoldFont.DeleteObject ();
CreateBoldFont(&m_BoldFont);
this->GetCharSize (pDC, &m_BoldFont, &m_BCharX, &m_BLineY);  
}

/************************************************************************
 *  ScMpView.cpp		O n I n i t i a l U p d a t e					*
 ************************************************************************/
void CScanMapView::OnInitialUpdate()
{
	CTextViewTool::OnInitialUpdate();
	// TODO: calculate the total size of this view

CDC* pDC = GetDC();    
pDC->SetMapMode(MM_LOMETRIC);

this->UpdateFonts (pDC);

ReleaseDC(pDC);     
}


/////////////////////////////////////////////////////////////////////////////
// CScanMapView drawing
/************************************************************************
 *  ScMpView.cpp   				  S e t X T a b s 	// unused !			*
 ************************************************************************/
void CScanMapView::SetXTabs (CDC* pDC, int CharX)
{
int X_TAB, LogPixXToUse, LogPixYToUse;
GetLogPixToUse(pDC, (int*)&LogPixXToUse, (int*)&LogPixYToUse);   


X_TAB = 4*CharX;
m_dX = LogPixXToUse - 2*X_TAB;
m_dY = LogPixYToUse;
m_Xleft  = X_TAB;
m_Xright = m_dX;			// overwrite later


int nStartY = 2*m_LineY;
int nStartX = X_TAB;
m_rMaxRect.SetRect (nStartX,nStartY, (int)(nStartX+2100), nStartY+2900);

m_Xleft  = m_rMaxRect.left;
m_Xright = m_rMaxRect.right;
m_dX = m_Xright - m_Xleft;
}



/****************************************************************************
 *	ScMpView.cpp					C i r c l e								*
 ****************************************************************************/
void CScanMapView::Circle (CDC* pDC, long nX0, long nY0, short nRadius, BOOL bFilled)
{
CSymbols::Ellipse (pDC, nX0-nRadius, nY0-nRadius, nX0+nRadius, nY0+nRadius, bFilled);
}

/****************************************************************************
 *	ScMpView.cpp				D r a w F i x P o i n t						*
 ****************************************************************************/
void CScanMapView::DrawFixPoint(CDC* pDC, CPoint pLP)
{
	CPen* ptNewPen = new CPen (PS_SOLID, 10, RGB (255, 255, 0));
	if (ptNewPen != NULL)
	{
		CPen* ptOldPen = pDC->SelectObject (ptNewPen);
		this->Circle (pDC,  pLP.x, pLP.y, 50, FALSE);
		pDC->SelectObject (ptOldPen);	  
		delete ptNewPen;
	}
}										   	

/************************************************************************
 *  ScanMapView.cpp				T e x t R o t a t e						*
 *  fAngle  < 0, right end moves up										*
 *  fAngle == 0, horizontal												*
 *  fAngle  > 0, right end moves down									*
 ************************************************************************/
void CScanMapView::TextRotate (CDC* pDC, long lX, long lY, double dAngle, CString szText)
{ 
	 short nAngleTenths = (short)(dAngle*10);

     CFont fntRotated;

     LOGFONT logFont;
     memset( &logFont, 0, sizeof(logFont));

     _tcscpy(logFont.lfFaceName, _T("Arial"));
     logFont.lfHeight = m_lfHeight;     
     logFont.lfUnderline = false;
     logFont.lfItalic = false;
     logFont.lfWeight = FW_NORMAL;

     logFont.lfOrientation = nAngleTenths;		// 450: turn right 45 degrees 
     logFont.lfEscapement = nAngleTenths;
     VERIFY(fntRotated.CreateFontIndirect(&logFont));

     CFont* pOldFont = (CFont*)pDC->SelectObject(&fntRotated);
//	 int OldBkMode = pDC->SetBkMode (TRANSPARENT);

     pDC->TextOut(lX, lY, szText);
	 
//	 pDC->SetBkMode (OldBkMode);
     pDC->SelectObject(pOldFont);
}

/************************************************************************
 *  ScMpView.cpp			R e c t A l i g n R i g h t					*
 ************************************************************************/
void CScanMapView::RectAlignRight (CDC* pDC, int Xright, int Ybot, int LineY,
				short nSizeLP, BOOL bFrame)
{
short	X0, Y0, X1, Y1;
short	dX;

dX	= abs(LineY) / 2;

X0	= Xright - nSizeLP - dX;
X1	= Xright + dX;
Y0	= Ybot + LineY;
Y1	= Y0 - (short)(1.4*LineY);

if (bFrame)
    {
    pDC->Rectangle (X0, Y0, X1+1, Y1+1); /* left, top, right, bottom	*/
    }
else{
    CBrush* ptBrush;
    CRect    rErase;
    rErase.SetRect(X0, Y0, X1+1, Y1+1); /* left, top, right, bottom	*/
    ptBrush = (CBrush*)GetStockObject (WHITE_BRUSH);
    pDC->FillRect (rErase, ptBrush);
    }
}

/************************************************************************
 *  ScMpView.cpp			R e c t A l i g n L e f t					*
 ************************************************************************/
void CScanMapView::RectAlignLeft (CDC* pDC, int Xleft, int Ybot, int LineY,
				short nSizeLP, BOOL bFrame)
{
short	X0, Y0, X1, Y1;
short	dX;

dX	= abs(LineY) / 2;

X0	= Xleft - dX;
X1	= Xleft + nSizeLP + dX;
Y0	= Ybot + LineY;
Y1	= Y0 - (short)(1.4*LineY);

if (bFrame)
    {
    pDC->Rectangle (X0, Y0, X1+1, Y1+1); /* left, top, right, bottom	*/
    }
else{
    CBrush* ptBrush;
    CRect    rErase;
    rErase.SetRect(X0, Y0, X1+1, Y1+1); /* left, top, right, bottom	*/
    ptBrush = (CBrush*)GetStockObject (WHITE_BRUSH);
    pDC->FillRect (rErase, ptBrush);
    }
}

/************************************************************************
 *  ScMpView.cpp				G e t M a x R e c t L P					*
 ************************************************************************/
void CScanMapView::GetMaxRectLP (CDC* pDC, CWnd* ptWnd, LPRECT lpRectMax)
{
					 //normal: CapsX = 52, LineY = 60
if (m_bForPrinter)
    {
    int Width_pix, Height_lines;
    POINT   Offset, Pixels;

    Width_pix = pDC->GetDeviceCaps (HORZRES);	/* 960 pixels	*/
    Height_lines = pDC->GetDeviceCaps (VERTRES);	/* 841 lines	*/

    Pixels.x = Width_pix;
    Pixels.y = Height_lines;
	
	pDC->DPtoLP((LPPOINT)&Pixels, 1);   /* W=2032, H=-2967	    */
//    Offset.x = (int)(0.05* Pixels.x);	    /* start 05% of pagewidth */
//    Offset.y = (int)(0.08* Pixels.y);	    /* start 08% of pageheight*/
//    Pixels.x = (int)(0.9 * Pixels.x);	    /* use 90% of pagewidth */
//    Pixels.y = (int)(0.85* Pixels.y);	    /* use 85% of pageheight*/
    Offset.x = (int)(0.00* Pixels.x);	    /* start 0% of pagewidth */
    Offset.y = (int)(0.00* Pixels.y);	    /* start 0% of pageheight*/
    Pixels.x = (int)(1.00* Pixels.x);	    /* use 100% of pagewidth */
    Pixels.y = (int)(1.00* Pixels.y);	    /* use 100% of pageheight*/
    SetRect (lpRectMax, Offset.x, Offset.y, 
						Pixels.x, Pixels.y);
    }
else{					    /* use actual window size*/
    ptWnd->GetClientRect (lpRectMax);
	pDC->DPtoLP((LPPOINT)lpRectMax, 2);
    }
}  

/************************************************************************
 *  MapView.cpp				G e t U s e a b l e R e c t L P				*
 ************************************************************************/
CRect CScanMapView::GetUseableRectLP (CDC* pDC)
{
CRect rView, rBorderLP;

if (m_bForPrinter && pDC != NULL)
	{
	this->GetMaxRectLP (pDC, this, rView);
	short nFrameX = (short)((double)(rView.right - rView.left) / 50);
	short nFrameY = (short)((double)(rView.bottom - rView.top) / 70);

	rBorderLP.top	= rView.top		+ 3*nFrameY;
	rBorderLP.left	= rView.left	+ 6*nFrameX;
	rBorderLP.right	= rView.right	- 0*nFrameX;
	rBorderLP.bottom = rView.bottom	- 1*nFrameY;
	}
else{
	CClientDC dc (this);
	dc.SetMapMode (MM_LOMETRIC);

	GetClientRect (rView);
	dc.DPtoLP((LPPOINT)&rView, 2);
	rBorderLP.top	= rView.top		;
	rBorderLP.left	= rView.left	;
	rBorderLP.right	= rView.right	;
	rBorderLP.bottom = rView.bottom	;
	}  

return rBorderLP;
}


/************************************************************************
 *  ScMpView.cpp	   			G e t L a b e l S t a r t L P 			*
 ************************************************************************/
void CScanMapView::GetLabelStartLP(long LPx1, long LPy1, double dAngle, long LineY,
								   long* ptLPx, long* ptLPy)
{
	long lTextVektLen;
	double dRad;

	lTextVektLen = 2*LineY;			

	dRad = pi * dAngle / 180;
	if (bMMText) *ptLPx = LPx1 + (long)(sin (dRad) * lTextVektLen);
		else	 *ptLPx = LPx1 - (long)(sin (dRad) * lTextVektLen);
	*ptLPy = LPy1 - (long)(cos (dRad) * lTextVektLen);
}

/************************************************************************
 *  ScMpView.cpp	   			D r a w L a b e l 						*
 ************************************************************************/
void CScanMapView::DrawLabel (CDC* pDC, long LPx1, long LPy1, double TextAngle, long LineY, 
				BOOL bRightAlign, CString szLoc, short nSizeLP, BOOL bActWpt)
{	
long	LPx2, LPy2, dY, dYtext;

GetLabelStartLP(LPx1, LPy1, TextAngle, LineY, &LPx2, &LPy2);


CPen DashPen (PS_DASH, 1, RGB (0, 0, 0));
CBrush* ptOldBrush;
CPen* ptOldPen = (CPen*)pDC->SelectObject (&DashPen);

						 				
CBrush brBackGround (bActWpt? m_ptColor->GetColScMapSel() : m_ptColor->GetColScMapLabel());
ptOldBrush = (CBrush*)pDC->SelectObject (&brBackGround);

int OldBkMode = pDC->SetBkMode (TRANSPARENT);

dY	= LineY / 3;							/* see: RectAlignXXXX		*/
dYtext = LineY / 6;

pDC->MoveTo (LPx1, LPy1);
pDC->LineTo (LPx2, LPy2+dY);											

COLORREF	OldTextCol = pDC->GetTextColor();
pDC->SetTextColor(RGB (0, 0, 0));			// switch to black text color
if (bRightAlign)	
	{										/* true: show frame			*/
	CMap::RectAlignRight (pDC, LPx2, LPy2+dY, LineY, nSizeLP, TRUE);	
	this->TextAlignRight (pDC, LPx2, LPy2+dYtext, szLoc);
	}
else{										/* true: show frame			*/
	CMap::RectAlignLeft (pDC, LPx2, LPy2+dY, LineY, nSizeLP, TRUE);	
	this->TextAlignLeft (pDC, LPx2, LPy2+dYtext, szLoc);
	}
pDC->SetTextColor(OldTextCol);				// switch back to colored map text
		
pDC->MoveTo (LPx1, LPy1);

pDC->SetBkMode (OldBkMode);
//pDC->SelectStockObject (WHITE_BRUSH);
pDC->SelectObject (ptOldBrush);
pDC->SelectObject (ptOldPen);
}	

/****************************************************************************
 *	ScMpView.c				S c M p C r i t i c a l L e g					*
 ****************************************************************************/
void ScMpCriticalLeg (CLatLon& LLenter, CLatLon& LLleave, long lElev_ft, void* ptData)
{
	if (ptData != NULL)
	{
	CScanMapView* ptView = (CScanMapView*)ptData;

	CDC*			pDC = ptView->GetDCPtr ();
	CScannedMap*	ptMap = ptView->GetMapPtr ();
	CColorDoc*		ptColor = ptView->GetColorDocPtr ();


	DRECT rMap = ptMap->GetLatLonBorder();
	double dEnterLon, dLeaveLon;
	if (CLatLon::GetMapLon(rMap.left, LLenter.GetLon(), rMap.right, &dEnterLon) &&
		CLatLon::GetMapLon (rMap.left, LLleave.GetLon(), rMap.right, &dLeaveLon))
		{
		long LPx0, LPy0, LPx1, LPy1;				
		ptMap->LatLonToLP (dEnterLon, LLenter.GetLat(), &LPx0, &LPy0);
		ptMap->LatLonToLP (dLeaveLon, LLleave.GetLat(), &LPx1, &LPy1);

		COLORREF LegCol = ptColor->GetColCriticalLine();
		ptColor->DrawLeg (pDC, LPx0, LPy0, LPx1, LPy1, 
						TRUE /* bScMap */, LegCol);
		}
	}

}


/****************************************************************************
 *	ScMpView.c				D r a w R o u t e O n M a p						*
 ****************************************************************************/
void CScanMapView::DrawRouteOnMap (CDC* pDC, CScannedMap* ptMap)
{
	CWayDoc* ptWayDoc = GetDocument();
	ASSERT_VALID(ptWayDoc);

	// TODO: add draw code for native data here            
if (ptWayDoc == NULL)
	return;

short i;
long LPx0, LPy0, LPx1, LPy1;				
double dLat0, dLon0, dLon1, Way0Lon;
COLORREF LegCol;
BOOL bDrawn;
BOOL bShowActWpt = FALSE;

CLatLon LLold;			// for ScMpCriticalLeg
long	lAlt_old;

dLon0 = Way0Lon = 0.;

for (i=0; i<ptWayDoc->GetWayPointCnt(); i++)
	{
	CWayPoint WayPt;
	if (ptWayDoc->GetWayPointPtr (i, &WayPt))
		{
		CLatLon LLact(WayPt.GetLat(), WayPt.GetLon());	// for ScMpCriticalLeg
		long lAlt = WayPt.GetActAlt_ft();

		if (i==0)
			{
			dLon1 = WayPt.GetLon();		
			}
		else{
			dLon1 = dLon0 + WayPt.FlightAngle (Way0Lon, WayPt.GetLon());
			}
			
		bDrawn = FALSE;

		if (i==0)	
			{
			if (ptMap->IsLatLonInRect (WayPt.GetLat(), dLon1))
				{
				ptMap->LatLonToLP (dLon1, WayPt.GetLat(), &LPx1, &LPy1);
			//	pDC->MoveTo (LPx1, LPy1);  // for DrawLabel
				bDrawn = TRUE;
				}
			}
		else{
			double dVisLat0, dVisLon0, dVisLat1, dVisLon1;
			if (ptMap->IsVisibleLine (dLat0, dLon0, WayPt.GetLat(), dLon1, 
					&dVisLat0, &dVisLon0, &dVisLat1, &dVisLon1))
				{
				ptMap->LatLonToLP (dVisLon0, dVisLat0, &LPx0, &LPy0);
				ptMap->LatLonToLP (dVisLon1, dVisLat1, &LPx1, &LPy1);

				m_ptColor->DrawLeg (pDC, LPx0, LPy0, LPx1, LPy1, 
								TRUE /* bScMap */, LegCol);

				if (ptElevDoc != NULL && ptMap->IsActivated(SHOW_ELEV))
					{
					this->SetDCPtr (pDC);	// for ScMpCriticalLeg
					this->SetMapPtr (ptMap);
					BOOL bOrtho = TRUE;
					ptElevDoc->CriticalLegPoints(LLold, lAlt_old, LLact, lAlt,
							ScMpCriticalLeg, bOrtho, this);
					}
				bDrawn = TRUE;
				}
			}


		Way0Lon = WayPt.GetLon();

		dLat0 = WayPt.GetLat();
		dLon0 = dLon1;									/* Kartenpunkt			*/

		LLold = LLact;				// for ScMpCriticalLeg
		lAlt_old = lAlt;
		}

	bShowActWpt = (i==ptWayDoc->GetActInd()) && !m_bForPrinter;
	LegCol = m_ptColor->GetLegColor (/*bScMap*/ TRUE, bShowActWpt);

	if (bDrawn)
	{
		if (ptMap->IsActivated(SHOW_LABEL) || bShowActWpt)
			{												/* name of waypoint		*/
			double	TextAngle;
			BOOL	bRightAlign;
			short	nSizeLP;
			CString	szNameForRte;     
 		
			TextAngle = ptWayDoc->LabelDirection (i, &bRightAlign);
			nSizeLP = WayPt.GetTextWidth(pDC);
			szNameForRte = WayPt.GetNameForRte ();
			this->DrawLabel (pDC, LPx1, LPy1, TextAngle,
				m_LineY, bRightAlign, szNameForRte, nSizeLP, bShowActWpt);
			}

		short nBearInd;
		for (nBearInd=0; nBearInd<2; nBearInd++)
			{
			if (WayPt.HasBearing (nBearInd))
				{
				double dNavLat, dNavLon, dMapLon;
				long   lLPx0, lLPy0, lLPx1, lLPy1;
				if (WayPt.GetBearLoc (nBearInd, &dNavLat, &dNavLon))
					{			// line from nav facility to WayPt
					ptMap->LatLonToLP (dNavLon, dNavLat, &lLPx0, &lLPy0);

					dMapLon = dNavLon + CLatLon::FlightAngle (dNavLon, WayPt.GetLon());
					ptMap->LatLonToLP (dMapLon, WayPt.GetLat(), &lLPx1, &lLPy1);
					m_ptColor->DrawBear (pDC, lLPx0, lLPy0,	// nav facility or enter pt
											lLPx1, lLPy1,		// wpt on other side
											TRUE /* bScMap */);
					}
				}
			}
		} // bDrawn
	}

									// second loop to draw check points
for (i=0; i<ptWayDoc->GetWayPointCnt(); i++)
	{
	CWayPoint WayPt;
	if (ptWayDoc->GetWayPointPtr (i, &WayPt))
		{							 // Kartenpunkt
		ptMap->LatLonToLP (WayPt.GetLon(), WayPt.GetLat(), &LPx1, &LPy1);
		m_ptColor->DrawCheckPoint (pDC, LPx1, LPy1, TRUE/* bScMap */, WayPt.IsUpDown());
		}
	}
}

/****************************************************************************
 *	ScMpView.c				D r a w M v M p R t e O n M a p					*
 ****************************************************************************/
void CScanMapView::DrawMvMpRteOnMap (CDC* pDC, CScannedMap* ptMap)
{
	CWayDoc* ptWayDoc = GetDocument();
	ASSERT_VALID(ptWayDoc);

	// TODO: add draw code for native data here            
if (ptWayDoc == NULL)
	return;

short i;
long LPx0, LPy0, LPx1, LPy1;				
double dLat0, dLon0, dLon1, Way0Lon;

dLon0 = Way0Lon = 0.;

for (i=0; i<ptWayDoc->GetWayPointCnt(); i++)
	{
	CWayPoint WayPt;
	if (ptWayDoc->GetWayPointPtr (i, &WayPt))
		{
		CLatLon LLact(WayPt.GetLat(), WayPt.GetLon());	// for ScMpCriticalLeg

		if (i==0)
			{
			dLon1 = WayPt.GetLon();		
			}
		else{
			dLon1 = dLon0 + WayPt.FlightAngle (Way0Lon, WayPt.GetLon());
			}
			
		if (i==0)	
			{
			if (ptMap->IsLatLonInRect (WayPt.GetLat(), dLon1))
				{
				ptMap->LatLonToLP (dLon1, WayPt.GetLat(), &LPx1, &LPy1);
				}
			}
		else{
			double dVisLat0, dVisLon0, dVisLat1, dVisLon1;
			if (ptMap->IsVisibleLine (dLat0, dLon0, WayPt.GetLat(), dLon1, 
					&dVisLat0, &dVisLon0, &dVisLat1, &dVisLon1))
				{
				ptMap->LatLonToLP (dVisLon0, dVisLat0, &LPx0, &LPy0);
				ptMap->LatLonToLP (dVisLon1, dVisLat1, &LPx1, &LPy1);

				m_ptColor->DrawMvMpLeg (pDC, LPx0, LPy0, LPx1, LPy1); 
				}
			}


		Way0Lon = WayPt.GetLon();

		dLat0 = WayPt.GetLat();
		dLon0 = dLon1;									/* Kartenpunkt			*/
		}
	}
}


/****************************************************************************
 *	ScMpView.c					D r a w T r a c k O n M a p	 				*
 ****************************************************************************/
void CScanMapView::DrawTrackOnMap (CDC* pDC, CScannedMap* ptMap)
{
	// TODO: add draw code for native data here            
short	nNameSizeLP, nTimeGrid;
long	lTimeMin, lTimeMax, lLabelTime;
CString	szTime;

ptTrackDoc->GetTimeRange (&nTimeGrid, &lTimeMin, &lTimeMax);
TimeScaleTextX ((double)0, (double)nTimeGrid, &szTime);  
nNameSizeLP = GetTextWidth(pDC, szTime);
  
lLabelTime = lTimeMin;

this->DrawMvMpRteOnMap(pDC, ptMap);		// draw actual route as target

if (ptTrackDoc != NULL)
	{
	BOOL	bFirst = TRUE;
	BOOL	bInside = FALSE;
	short i;
	long LPx1, LPy1;
	double fTrackLat, fTrackLon;				
	double dLon0, dLon1, Track0Lon;
	short	nStartIndex = 0;

	dLon0 = Track0Lon = 0.;

	m_ptColor->SelectScanRoutePen (pDC, TRUE);


	CMovingMapDoc* ptMovingMap = ptInit->GetMovingMapDocPtr();
	if (ptMovingMap != NULL)
		{					// get first track index to draw
		TrackSymbol Symbol = TRACK_LINE;
		
		if (ptMovingMap->IsOnView())
			Symbol = ptMovingMap->GetTrackSymbol();

		switch (Symbol)
			{
			case TRACK_OFF:	nStartIndex=ptTrackDoc->GetCnt()-1; break;
			case TRACK_LINE: nStartIndex=0; break;
			case TRACK_TAIL: nStartIndex=ptTrackDoc->GetTimedIndex (5); break;
			}
		}


	long LPx0=0;				// LP of last track point
	long LPy0=0;
	BOOL bLastInside = FALSE;

	for (i=nStartIndex; i<ptTrackDoc->GetCnt(); i++)
		{
		CTrackPoint TrackPt;
		if (ptTrackDoc->GetTrackPointPtr (i, &TrackPt))
			{
			TrackPt.GetLatLon (&fTrackLat, &fTrackLon);
			if (TrackPt.IsStart()) 
				bFirst = TRUE;

			if (bFirst)
				{
				dLon1 = fTrackLon;		
				}
			else{
				dLon1 = dLon0 + CLatLon::FlightAngle (Track0Lon, fTrackLon);
				}
				
			ptMap->LatLonToLP (dLon1, fTrackLat, &LPx1, &LPy1);
			bInside = ptMap->IsInside(LPx1, LPy1);

			if (bInside)
			{		// move or draw
				if (bFirst)
				{
					pDC->MoveTo (LPx1, LPy1);
				}
				else
				{
					if (bLastInside)
					{	// track stays inside of map
						pDC->LineTo (LPx1, LPy1);
					}
					else
					{	// track has entered map from outside
						long LPx, LPy;		// point on border
						if (ptMap->CrossingBorder(LPx0, LPy0, LPx1, LPy1, &LPx, &LPy))
						{
							pDC->MoveTo (LPx, LPy);		// move to point on border
							pDC->LineTo (LPx1, LPy1);	// draw visible part of leg
						}
					}
				}
			}
			else
			{		// outside of map
				if (bFirst)
				{
					// do nothing
				}
				else
				{
					if (bLastInside)
					{		// track has left map
						long LPx, LPy;		// point on border
						if (ptMap->CrossingBorder(LPx0, LPy0, LPx1, LPy1, &LPx, &LPy))
						{	// draw visible part of leaving leg
							pDC->LineTo (LPx, LPy);
						}
					}
					else
					{	// start and end of track are outside of map
						// but middle part of leg could be visible on map!!!
						// check for two points on border
						long VertLPx, VertLPy;		// point on border
						long HorzLPx, HorzLPy;		// point on border
						if (ptMap->CrossingBorders(LPx0, LPy0, LPx1, LPy1, 
										&VertLPx, &VertLPy, &HorzLPx, &HorzLPy))
						{	// draw visible part of outside leg
							pDC->MoveTo (VertLPx, VertLPy);
							pDC->LineTo (HorzLPx, HorzLPy);
						}
					}
				}
			}

			Track0Lon = fTrackLon;
			dLon0 = dLon1;									/* Kartenpunkt			*/

			LPx0 = LPx1;				// last LP point of track
			LPy0 = LPy1;
			bLastInside = bInside;
			bFirst = FALSE;
			}

		if (ptMap->IsActivated(SHOW_LABEL))
			{												/* name of waypoint		*/
			if (ptTrackDoc->TrackNeedsLabel(i, nTimeGrid, &lLabelTime))		
				{
				double TextAngle;
				CTrackPoint TrackPt;
				BOOL	 bRightAlign;  
 				long	lSec;
				
				TextAngle = ptTrackDoc->LabelDirection (i, &bRightAlign);
				
				if (ptTrackDoc->GetTrackPointPtr (i, &TrackPt))
					{
					lSec = TrackPt.GetSec();
					TimeScaleTextX ((double)lSec, (double)nTimeGrid, &szTime);
					this->DrawLabel (pDC, LPx1, LPy1, TextAngle,
						m_LineY, bRightAlign, szTime, nNameSizeLP);
					}
				}
			}
		}

	if (ptMovingMap != NULL && bLastInside)
		ptMovingMap->DrawActPosSymbol(pDC, LPx1, LPy1);

	m_ptColor->SelectScanRoutePen (pDC, FALSE);
	}	
}


/************************************************************************
 *  ScMpView.cpp    		 D r a w S c a n n e d M a p s				*
 *  Purpose: Draws frames of maps to be used for enlarged view			*
 ************************************************************************/
void CScanMapView::DrawScannedMaps (CDC* pDC, CScannedMap* ptMap, DRECT rLatLon)
{
short i, j;
CColorDoc* ptColor = ptInit->GetColorPtr();

long lMapScale = this->GetScale(pDC, ptMap);

for (i=0; i<ptScMpDoc->GetMapCnt(); i++)
	{
	if (i == ptScMpDoc->GetActMapIndex ())
		continue;

	CScannedMap* ptTestMap = ptScMpDoc->GetMapPtr (i);
	if (ptTestMap != NULL)
		{
		CLatLon MapTL = ptTestMap->GetBorderLatLon(CALPOS_TL);
		CLatLon MapBR = ptTestMap->GetBorderLatLon(CALPOS_BR);

		long lTestMapScale = this->GetScale(pDC, ptTestMap);
		BOOL bUsableToEnlarge = (lTestMapScale < lMapScale);
		BOOL bVisibleOnMap = (CLatLon::IsLatLonInRect (MapBR.GetLat(), MapBR.GetLon(), rLatLon) ||
			CLatLon::IsLatLonInRect (MapTL.GetLat(), MapTL.GetLon(), rLatLon));
		BOOL bAutoSel = ptTestMap->IsAutoSel();

		if (!bAutoSel && bUsableToEnlarge && bVisibleOnMap)
			{
			long lXstart, lYstart, lXTopRight, lYTopRight, lX, lY;
			CLatLon llBorder;
								  
			CPen* ptNewPen = new CPen (PS_SOLID, 5, ptColor->GetColMapFrame ());
			CPen* ptOldPen = pDC->SelectObject (ptNewPen);
			for (j=0; j<4; j++)
				{
				llBorder = ptTestMap->GetBorderLatLon(j);

				double dMapLon;
				CLatLon::GetMapLon(rLatLon.left, llBorder.GetLon(), rLatLon.right, &dMapLon);

				ptMap->LatLonToLP (dMapLon, llBorder.GetLat(), &lX, &lY);

				if (j==0)
					{
					lXstart = lX;
					lYstart = lY;
					pDC->MoveTo(lX, lY);
					}
				else{
					if (j==1)
						{
						lXTopRight = lX;
						lYTopRight = lY;
						}
					pDC->LineTo(lX, lY);

					if (j==2)
						{		// bottom right: draw first part of X
						pDC->MoveTo (lXstart, lYstart);
						pDC->LineTo(lX, lY);
						}

					if (j==3)
						{		// bottom right: draw first part of X
						pDC->MoveTo (lXTopRight, lYTopRight);
						pDC->LineTo(lX, lY);
						}
					}
				}
			pDC->LineTo(lXstart, lYstart);
			pDC->SelectObject (ptOldPen);	 
			delete ptNewPen;
			}
		}
	}
}

/************************************************************************
 *  ScMpView.cpp   		 	D r a w E l e v A r e a s 					*
 ************************************************************************/
void CScanMapView::DrawElevAreas (CDC* pDC, CScannedMap* ptMap)
{
	short i;
	CColorDoc* ptColor = ptInit->GetColorPtr();

	if (ptElevDoc != NULL)
	{
												// get rect around map
		DRECT rLatLon = ptMap->GetLatLonBorder();

		
		short nStartInd, nEndInd;		// Elev Index
		CLatLon Map_TL (rLatLon.top, rLatLon.left);
		CLatLon Map_BR (rLatLon.bottom, rLatLon.right);

		if (ptElevDoc->GetMinMaxIndex (Map_TL, Map_BR, &nStartInd, &nEndInd))
		{
			CPen* ptNewPen = new CPen (PS_SOLID, 3, ptColor->GetColElevValue ());
			CPen* ptOldPen = pDC->SelectObject (ptNewPen);

			for (i=nStartInd; i<=nEndInd; i++)
			{							// for all Elevs of actual mercator map
				CElev Elev;
				if (ptElevDoc->GetElevPtr (i, &Elev))
				{
					CLatLon ElevBR = Elev.GetPosBR();
					CLatLon ElevTL = Elev.GetPosTL();

					double dRightLon, dLeftLon;
					BOOL bRightIn = CLatLon::IsLatLonInRect (ElevBR.GetLat(), ElevBR.GetLon(), rLatLon, &dRightLon);
					BOOL bLeftIn = CLatLon::IsLatLonInRect (ElevTL.GetLat(), ElevTL.GetLon(), rLatLon, &dLeftLon);
												// use help bools to avoid undefined dLeftLon
					if (bRightIn || bLeftIn)	// if statement doesn`t execute 2. statement, if 1. is TRUE!!
					{
						long lXstart, lYstart, lX, lY;

						ptMap->LatLonToLP (dLeftLon, ElevTL.GetLat(), &lX, &lY);

						lXstart = lX;
						lYstart = lY;
						pDC->MoveTo(lX, lY);			// top left

						ptMap->LatLonToLP (dRightLon, ElevTL.GetLat(), &lX, &lY);
						pDC->LineTo (lX, lY);			// top right

						ptMap->LatLonToLP (dRightLon, ElevBR.GetLat(), &lX, &lY);
						pDC->LineTo (lX, lY);			// bottom right

						Elev.DrawValue (pDC, lYstart, lXstart, lY, lX);

						ptMap->LatLonToLP (dLeftLon, ElevBR.GetLat(), &lX, &lY);
						pDC->LineTo (lX, lY);			// bottom left

						pDC->LineTo(lXstart, lYstart);	// top left

					}
				}
			}


			pDC->SelectObject (ptOldPen);	 
			delete ptNewPen; 

		}
	}
}

#ifdef __LATLONTOLP_FOR_DOUBLE__
/****************************************************************************
 *	ScMpView.cpp				D i s t a n c e T o L P 	 				*
 *  Purpose: Converts a distance, specified by value and dimension to LP	*
 *  LP values do depend on position on earth, specified by dLat, dLon.		*
 ****************************************************************************/
long CScanMapView::DistanceToLP(CScannedMap* ptMap, double dLat, double dLon, double* LPx, double* LPy,
							double dDist, short nDistDim)
{
	long lRadius = 0;

	ptMap->LatLonToLP (dLon, dLat, LPx, LPy);	// center of arc

				// convert dRadius to LP here...
	double dDist_NM = ptDim->ConvertDist (dDist, nDistDim, DIM_NM);

				// calc LatLon, dRadius_NM north of center
	CLatLon llCenter(dLat, dLon);
	CLatLon llNorth;
	short nRWK = 0;
	llNorth.FromRelKoords (llCenter, nRWK, dDist_NM);
	double dXn, dYn;
	ptMap->LatLonToLP (llNorth.GetLon(), llNorth.GetLat(), &dXn, &dYn);	// center of arc

	lRadius = (long)fabs(dYn - *LPy);

	return lRadius;

}
#endif

/****************************************************************************
 *	ScMpView.cpp				D i s t a n c e T o L P 	 				*
 *  Purpose: Converts a distance, specified by value and dimension to LP	*
 *  LP values do depend on position on earth, specified by dLat, dLon.		*
 ****************************************************************************/
long CScanMapView::DistanceToLP(CScannedMap* ptMap, double dLat, double dLon, long* LPx, long* LPy,
							double dDist, short nDistDim)
{
	long lRadius = 0;

	ptMap->LatLonToLP (dLon, dLat, LPx, LPy);	// center of arc

				// convert dRadius to LP here...
	double dDist_NM = ptDim->ConvertDist (dDist, nDistDim, DIM_NM);

				// calc LatLon, dRadius_NM north of center
	CLatLon llCenter(dLat, dLon);
	CLatLon llNorth;
	short nRWK = 0;
	llNorth.FromRelKoords (llCenter, nRWK, dDist_NM);
	long lXn, lYn;
	ptMap->LatLonToLP (llNorth.GetLon(), llNorth.GetLat(), &lXn, &lYn);	// center of arc

	lRadius = abs(lYn - *LPy);

	return lRadius;
}


/************************************************************************
 *  ScMpView.cpp	  			D r a w S e g m e n t s 				*
 *  Purpose: draws alls segments of an airspace,						*
 *  dLonOffset (360, -360) is used to draw on antipod maps				*
 *  returns FALSE, if not all segments are drawn						*
 ************************************************************************/
BOOL CScanMapView::DrawSegments (CDC* pDC, CScannedMap* ptMap, CAirspace* ptAirspace,
								CAirspaceSegmentDoc* ptSegDoc, long lStartIndex, double dLonOffset)
{
	BOOL bDrawn = FALSE;
	BOOL bCompleteDrawn = TRUE;
	CString szIdentifier;
	CString szText;
	short i = (short)lStartIndex;
	BOOL bSegmentsAvailable = (i>=0);		// segments available for current airspace
	BOOL bAntipodAirspace = ptAirspace->IsAntipod();

	while (bSegmentsAvailable)
	{      
		CAirspaceSegment* ptAirspaceSegment = NULL;
		
		if (ptSegDoc->GetAirspaceSegmentPtr (i, &ptAirspaceSegment))
		{
							// segment of same airspace identifier?
			if (i == lStartIndex)
			{
				szIdentifier = ptAirspaceSegment->GetIdentifier();
			}
			else
			{
				bSegmentsAvailable = (szIdentifier.Compare(ptAirspaceSegment->GetIdentifier()) == 0);
			}

	//		if ((szIdentifier.CompareNoCase("ED00040") == 0) &&
	//			ptAirspaceSegment->GetSegmentNumber() == 670)
	//			int s=1;

			if (bSegmentsAvailable)
			{
				BOOL bLP_OK = TRUE;
				double dLat, dLon;
				long lX1, lY1, lX2, lY2, lX0, lY0, lRadius;
				lX1 = lY1 = lX2 = lY2 = lX0 = lY0 = lRadius = ptAirspaceSegment->GetUndefinedLong();

								// try to get all LP values
				dLat = ptAirspaceSegment->GetLat1();
				dLon = ptAirspaceSegment->GetLon1();
				if (dLat != NO_KOORD && dLon != NO_KOORD)
				{
					if (bAntipodAirspace)
					{
						if (dLon < 0)
							dLon += 360;
					}
					dLon += dLonOffset;
					ptMap->LatLonToLP (dLon, dLat, &lX1, &lY1);

					if (lX1 <= -32768 || lX1 > 32768 ||
						lY1 <= -32768 || lY1 > 32768)
						bLP_OK = FALSE;
				}

				dLat = ptAirspaceSegment->GetLat2();
				dLon = ptAirspaceSegment->GetLon2();

				if (dLat != NO_KOORD && dLon != NO_KOORD)
				{
					if (bAntipodAirspace)
					{
						if (dLon < 0)
							dLon += 360;
					}
					dLon += dLonOffset;
					ptMap->LatLonToLP (dLon, dLat, &lX2, &lY2);

					if (lX2 <= -32768 || lX2 > 32768 ||
						lY2 <= -32768 || lY2 > 32768)
						bLP_OK = FALSE;
				}
				

				dLat = ptAirspaceSegment->GetGeodeticLat0();
				dLon = ptAirspaceSegment->GetGeodeticLon0();
				if (dLat != NO_KOORD && dLon != NO_KOORD)
				{				// center of arc defined							.
					if (bAntipodAirspace)
					{
						if (dLon < 0)
							dLon += 360;
					}
					dLon += dLonOffset;

								// convert lat/lon and dRadius to LP here..
					double dRadius = ptAirspaceSegment->GetRadius();
					short nDistDim = ptAirspaceSegment->GetDistDim();
					lRadius = this->DistanceToLP(ptMap, dLat, dLon, &lX0, &lY0, dRadius, nDistDim);

					if (lX0 <= -32768 || lX0 > 32768 ||
						lY0 <= -32768 || lY0 > 32768)
						bLP_OK = FALSE;
				}
				else
				{
					lX0 = 0;
					lY0 = 0;
					lRadius = 0;
				}
					
				if (bLP_OK)	
				{	// draw segment here...
						ptAirspaceSegment->Draw(pDC, lX1,lY1, lX2,lY2, lX0,lY0,lRadius);
						bDrawn = TRUE;
				}
				else	
				{
					bCompleteDrawn = FALSE;
				}
			}

			i++;
		}
		else
		{
			bSegmentsAvailable = FALSE;	
		}
	}

	return (bDrawn && bCompleteDrawn);
}   

/****************************************************************************
 *	MapView.cpp				D r a w A i r s p a c e s  						*
 ****************************************************************************/
void CScanMapView::DrawAirspaces (CDC* pDC, CScannedMap* ptMap)
{
													// get rect around map
	DRECT rLatLon = ptMap->GetLatLonBorder();
	long lMapScale = this->GetScale(pDC, ptMap);
	int oldAlign = pDC->SetTextAlign(TA_CENTER);

	CAirspaceRegionsDoc* ptAirspaceRegionsDoc = ptInit->GetAirspaceRegionsDocPtr();
	if (ptAirspaceRegionsDoc != NULL)
	{
		short nRegionIndex = 0;
		short nAirspaceIndex = 0;
		CAirspace* ptAirspace = NULL;

		while (ptAirspaceRegionsDoc->GetAirspaceInRect(rLatLon, &ptAirspace, 
			&nRegionIndex, &nAirspaceIndex))
		{
			if (ptAirspace != NULL)
			{
				CAirspaceDoc* ptAirspaceDoc = ptAirspaceRegionsDoc->GetDocument();
				CAirspaceSegmentDoc* ptSegDoc = ptAirspaceDoc->GetSegmentDocPtr();


				long lStartIndex = ptAirspace->GetFirstSegmentIndex();

				CColorDoc* ptColor = ptInit->GetColorPtr();
						
									// color depends on kind of airpace
				CPen* ptNewPen = ptAirspace->CreatePen(pDC);
				CPen* ptOldPen = (CPen*)pDC->SelectObject (ptNewPen);

				double dLonOffset = 0.;
				if (lStartIndex >= 0)
				{			// to calculate temp. long values for airpsaces on antipod maps
					if (rLatLon.left < -180)
					{		// antipod map lon: -206 ... -138
						if (ptAirspace->GetMaxLon() > 0)
						{		// east lon airpace
							dLonOffset = -360;
						}
					}

					if (rLatLon.right > 180)
					{		// antipod map lon: 154 ... 222
						if (ptAirspace->GetMinLon() < 0)
						{		// west lon airspace
							dLonOffset = 360.;
						}
					}
				}

				BOOL bCompleteDrawn = DrawSegments(pDC, ptMap, ptAirspace, ptSegDoc, lStartIndex, dLonOffset);

				if (bCompleteDrawn && ptMap->IsActivated(SHOW_LABEL))
				{
					double dMaxSegCourse=0;
					CLatLon llMaxSegCenter = ptAirspaceDoc->GetMaxSegmentCenter(ptAirspace, lStartIndex, &dMaxSegCourse);
					double dAngle = ptAirspace->CourseToTextAngle(dMaxSegCourse);
					double dLabelAngle = ptAirspace->CourseToLabelPosition(dMaxSegCourse);

					long lX, lY;
					CString szLabel = ptAirspace->GetLabelText();
					double dLon = llMaxSegCenter.GetLon();
					if (ptAirspace->IsAntipod())
					{
						if (dLon < 0)
							dLon += 360;
					}
					dLon += dLonOffset;
					ptMap->LatLonToLP (dLon, llMaxSegCenter.GetLat(), &lX, &lY);


					long lLabelX = lX;
					long lLabelY = lY;
					GetLabelStartLP(lX, lY, dLabelAngle, m_LineY/2, &lLabelX, &lLabelY);

					this->TextRotate(pDC, lLabelX, lLabelY, dAngle, szLabel);
				}

				pDC->SelectObject (ptOldPen);	 
				delete ptNewPen;
			}
		}
	}

	pDC->SetTextAlign(oldAlign);
}



/************************************************************************
 *  ScMpView.cpp    	 	 I s F o r R o u t e						*
 ************************************************************************/
BOOL CScanMapView::IsForRoute ()
{
BOOL bForRoute = TRUE;
if (ptTrackDoc != NULL)
	bForRoute = !(ptTrackDoc->GetCnt() > 0);
return bForRoute;
}


/************************************************************************
 *  ScMpView.cpp    			 G e t S c a l e						*
 *  Purpose: Get map scale												*
 ************************************************************************/
long CScanMapView::GetScale(CDC* pDC, CScannedMap* ptMap, BOOL bConsiderScrollPos)
{
	long lMapScale = 0;
								// calculates map scale: 1 : XX
	CPoint pHorzResDPI;
	pHorzResDPI.x = pDC->GetDeviceCaps (LOGPIXELSX);
	pHorzResDPI.y = pDC->GetDeviceCaps (LOGPIXELSY);
	pDC->DPtoLP (&pHorzResDPI);

	if (bConsiderScrollPos)
	{
		CPoint pScrollLP;
		pScrollLP = this->GetScrollPosition();
		pHorzResDPI -= pScrollLP;
	}

	lMapScale = ptMap->GetScale (pHorzResDPI.x);	

	return lMapScale;
}


/************************************************************************
 *  ScMpView.cpp    	 	 U p d a t e W h i t e A r e a s			*
 ************************************************************************/
void CScanMapView::UpdateWhiteAreas (CDC* pDC, CScannedMap* ptMap, 
					CRect& rClientDP, CRect& rMapDP)
{
CRect rClient;
this->GetClientRectLP (rClient);

CRect rMap = ptMap->GetLPRect ();			// get scaled size

long lX = rClient.right - rMap.right;
long lY = rClient.bottom - rMap.bottom;

CPen* ptOldPen = (CPen*)pDC->SelectStockObject (NULL_PEN);		  // NULL_PEN, WHITE_PEN 

CBrush* ptOldBrush = (CBrush*)pDC->SelectStockObject (LTGRAY_BRUSH); 

if (lX > 0)
	{			// l,	t,	r,	b
	CRect rFill (	rMap.right,    max (rClient.top, rMap.top), // gut fr graue Flche 
					rClient.right, min (rClient.bottom, rMap.bottom));
//	CRect rFill (	rMap.right,    rClient.top, 	 // gut, wenn nur Rest Wei gelscht wird
//					rClient.right, rClient.bottom);

	ptMap->Clip(pDC, rClientDP);	// set clip rect to client area
	pDC->Rectangle (rFill);
	ptMap->Clip(pDC, rMapDP);		// reset clip rect to map area
	}

BOOL bVertFill = (bMMText? (lY > 0) : (lY < 0));
if (bVertFill)
	{			// l,	t,	r,	b
	CRect rFill (min (rClient.left, rMap.left),	   // gut fr graue Flche
				rMap.bottom, 
				max (rClient.right, rMap.right),
				rClient.bottom);		   
//	CRect rFill (rClient.left,rMap.bottom, 		 // gut, wenn nur Rest Wei gelscht wird
//				rClient.right, rClient.bottom);

	ptMap->Clip(pDC, rClientDP);	// set clip rect to client area
	pDC->Rectangle (rFill);
	ptMap->Clip(pDC, rMapDP);		// reset clip rect to map area
	}

pDC->SelectObject (ptOldBrush);

pDC->SelectObject (ptOldPen);	  
}



/************************************************************************
 *  ScMpView.cpp    	 	 D r a w S c a n n e d M a p				*
 ************************************************************************/
BOOL CScanMapView::DrawScannedMap (CDC* pDC, BOOL bDraw)
{
	BOOL	bEndOfPage = FALSE;
	CPoint	pStart;
	CString szText;

	ActivateDrawing (bDraw);

	CFont* ptOldFont;
	ptOldFont = (CFont*)pDC->SelectObject (&m_PlainFont);


	CRect rMapDP, rClientDP, rClipDP;		// for clip region
													
	if (m_pDib != NULL)
	{
 		long lStartX, lStartY, lWidth, lHeight;  
		double dFaktX = 1;
		double dFaktY = 1;
		CSize	bmSize;
		int		nMapMode = MM_LOMETRIC;

		pDC->SetMapMode(nMapMode);

		if (m_pDib->GetSizeLP (pDC, &bmSize))
		{				// bmSize: AbsoluteLP coordinates	
			m_sOrigImage = bmSize;
													// absolute size
	//		if (!m_bForPrinter)		// otherwise scrollbars will be deleted after printing
	//			m_sizeTotal = ptScMpDoc->OriginalToScaledLP (m_sOrigImage);


			if (nMapMode == MM_LOMETRIC)
				bmSize.cy = -bmSize.cy;				// bmSize now in LP (y < 0)

			if (m_bDraw)
			{	
				CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());

				if (ptMap != NULL)
				{		
					if (m_bForPrinter)
					{
						BOOL bShowFullMap = FALSE;
						if (bShowFullMap)
						{
							CRect rAreaLP;								// ICAO Frankfurt
							this->GetMaxRectLP (pDC, this, rAreaLP);	// right=1980 bottom=-2855

							lStartX = rAreaLP.left;						// 0
							lStartY = rAreaLP.top;						// 0

														// bmSize.x=1872, bmSize.y=1189 [printer device LP]
								// get faktor to show total bitmap (bmSize) on rAreaLP (printer)
							CSize sDraw = GetDrawSize (bmSize,	// total size of bmp on printer
												CSize (rAreaLP.Width(), rAreaLP.Height()));
																// cx=1980 cy=-1257


							// lWidth =11700, lHeight = 7430  => dFakt = 0.169   => OK
							ptMap->GetSizeLP (&lWidth, &lHeight);
									// get faktors for x and y. Required, if ScreenResX != ScreenResY
							dFaktX = (double)sDraw.cx / lWidth;
							dFaktY = fabs((double)sDraw.cy / lHeight);	// fakt always positive!!



							pStart = CPoint(lStartX, lStartY);
							m_pDib->Stretch (pDC, pStart, sDraw);  // calls StretchDIBits(...)

							ptMap->SetStartPoint(pStart);
						}
						else
						{		// show part of map, visible on client area
							CRect rClientLP;					// get size of visible client area
							this->GetClientRectLP(rClientLP);	//  right=3172, bottom=-2027
																// this will cover the full printer area

							CRect rAreaLP;							// available printer area:
							this->GetMaxRectLP (pDC, this, rAreaLP);// right=1980 bottom=-2855
											
																	// lWidth =11700, lHeight = 7430
							ptMap->GetSizeLP (&lWidth, &lHeight);	// total (virtual) size of bmp on printer
																	// may be larger than printer area

										// use unscaled total bmp size (lWidth, -lHeight) here!
							CSize sDraw = GetDrawSize (CSize(lWidth, -lHeight), 
														CSize(rClientLP.Width(), rClientLP.Height()),
														CSize(rAreaLP.Width(), rAreaLP.Height()));
																	// cx=7303, cy=-4673

													// sDraw now doesn't consider scale factor
							CPoint pScrollLP;		// because pScrollLP already considers it!!
							pScrollLP = this->GetScrollPosition();		// x=4270  y=-3172
							lStartX = (long)(-pScrollLP.x * sDraw.cx / lWidth);	// -2665
							lStartY = (long)(-pScrollLP.y * fabs((double)sDraw.cy / lHeight));	// 1979
							pStart = CPoint(lStartX, lStartY);

													// after pStart is calcuated, 
													// sDraw has to be multiplicated with scale factor
							double dScFaktX = ptScMpDoc->GetScaleFaktX();
							double dScFaktY = ptScMpDoc->GetScaleFaktY();
							sDraw.cx = (long)(dScFaktX * sDraw.cx);		// put scale factor into sDraw
							sDraw.cy = (long)(dScFaktY * sDraw.cy);
													// now sDraw considers scale factor

													// get factor to get scaled map on printer
							dFaktX = (double)sDraw.cx / lWidth;			// dFaktX=0.62  dFaktY=0.62
							dFaktY = fabs((double)sDraw.cy / lHeight);	// fakt always positive!!


							m_pDib->Stretch (pDC, pStart, sDraw);  // calls StretchDIBits(...)

							ptMap->SetStartPoint(pStart);
						}
					}  // end: m_bForPrinter
					else
					{
						lStartX = 0;
						lStartY = 0;
						pStart = CPoint(lStartX, lStartY);
						ptMap->SetStartPoint(pStart);

						m_sizeTotal = ptScMpDoc->OriginalToScaledLP (m_sOrigImage);

						CSize sDraw;
						sDraw = m_sizeTotal;

						if (nMapMode == MM_LOMETRIC)
							sDraw.cy = -sDraw.cy;

						dFaktX = ptScMpDoc->GetScaleFaktX();
						dFaktY = ptScMpDoc->GetScaleFaktY();

						m_pDib->Stretch (pDC, pStart, sDraw);  // calls StretchDIBits(...)
					}

					ptMap->SetScaleFaktX (dFaktX);
					ptMap->SetScaleFaktY (dFaktY);

											// begin set clip rect to map rect
													// while printing:        		
					rMapDP = ptMap->GetLPRect();	// top left right bottom  		
					rMapDP.OffsetRect (pStart);						// add printing offset

					this->GetMaxRectLP (pDC, this, rClientDP);		// -230, 100, 1809, -2448

					pDC->LPtoDP (&rMapDP);		// map rect in DP:	   0, 0, 7165, 10519
					pDC->LPtoDP (&rClientDP);	// max client rect DP: 544, 237, 4280, 5789

					//rClipDP = m_bForPrinter? rMapDP : rClientDP;
					rClipDP = rClientDP;
					ptMap->Clip(pDC, rClipDP);			/* draw on map only		*/
											// end set clip rect to map rect



					if (ptMap->IsCalibrated())
					{
						BOOL bConsiderScrollPos = TRUE;
						long lMapScale = this->GetScale(pDC, ptMap, bConsiderScrollPos);

													// get rect around map
						DRECT rLatLon = ptMap->GetLatLonBorder();

													// draw map symbols on scanned map!
						CSymbols MapSymbols (pDC, m_bForPrinter, abs(m_LineY), 
													lMapScale, ptMap->GetTextColor());
						CLocation Loc;
						if (MapSymbols.Quality() > VQ_NONE)
						{
							short nActRegIndex = 0;
							CLaLoList* ptLaLoList = ptInit->GetLaLoListPtr();
							if (ptLaLoList->PrepareGetLoc (rLatLon))
							{
								while (ptLaLoList->GetLocationInRect (&Loc, rLatLon, &nActRegIndex))
								{
									ptMap->DrawLocation (Loc, MapSymbols);
								}
							}
						}
		// Achtung: Wenn hier fr eine Karte mehrere Regionen nacheinander aktiviert werden mssen,
		// dann kann sich die aktive Region, die im LocDoc gespeichert ist, ndern.
		// Im KoordDlg erscheint dann die Region, die als letzte zum Zeichnen aktiviert wurde.


						CMapDoc* ptMapDoc = ptInit->GetMapDocPtr();
						if (ptMapDoc->IsRouteInScanMap())
							this->DrawRouteOnMap (pDC, ptMap);
						else
							this->DrawTrackOnMap (pDC, ptMap);

						if (ptMap->IsActivated(SHOW_MAP))
							this->DrawScannedMaps(pDC, ptMap, rLatLon);

						if (ptMap->IsActivated(SHOW_ELEV))
							this->DrawElevAreas(pDC, ptMap);

						if (ptMap->IsActivated(SHOW_AIRSPACE))
							this->DrawAirspaces(pDC, ptMap);
					} // ptMap calibrated
				
				if (!m_bForPrinter)	// temporarily expands clip area to client rect 	
					this->UpdateWhiteAreas (pDC, ptMap, rClientDP, rClipDP);					
				} // ptMap != NULL
			} // m_bDraw

			m_bDrawn = TRUE;
		} // m_pDib->GetSizeLP
		else
		{		// bad Select (hBitmap in CImage::MakeBitmap = NULL)
			m_bDrawn = FALSE;
			CString szMsg;
			if (m_ptMap != NULL)	szMsg.Format(IDF_BAD_BMP, (LPCTSTR)m_ptMap->GetPath());
					else			szMsg = "Kann Karte nicht zeigen";
			AfxMessageBox(szMsg);
			return bEndOfPage;
		}
	} // m_pDib != NULL

													
	CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
	if (ptMap != NULL)
	{
		if (!ptMap->IsCalibrated())
		{						// while calibrating actual map:	
			short i;				// draw fix points of actual map
			for (i=0; i<ptMap->GetFixPtCnt (); i++)
			{
				CMapFixPt* ptFixPt = ptMap->GetFixPtPtr (i);
				if (ptFixPt != NULL)
				{		
					CPoint ScaledFixPointLP = ptScMpDoc->OriginalToScaledLP(ptFixPt->GetLP());
					this->DrawFixPoint(pDC, ScaledFixPointLP); 
				}
			}
		}

						// reset clip rect to client area
		ptMap->Clip(pDC, rClientDP);
	}

	pDC->SelectObject (ptOldFont);
	return bEndOfPage;
}


/************************************************************************
 *  ScMpView.cpp   			  P r i n t P a g e							*
 ************************************************************************/
void CScanMapView::PrintPage (CDC* pDC, int nPageToPrint)
{
int	nPage = 1;

BOOL bDraw;

do	{
	bDraw = (nPage == nPageToPrint);
	m_bMore = DrawScannedMap (pDC, bDraw);
	nPage++;
	} while (!bDraw);
}                                  


/************************************************************************
 *  ScMpView.cpp    	 		 O n D r a w							*
 ************************************************************************/
void CScanMapView::OnDraw(CDC* pDC)
{
	// TODO: add draw code for native data here            
m_bDraw=TRUE;

DrawScannedMap (pDC, m_bDraw);

if (m_bDrawFixPoint)
	{
//	this->DrawFixPoint(pDC, m_ActPointLP); 
	}
}

/************************************************************************
 *  ScMpView.cpp			P r e p a r e X T a b s						* 
 ************************************************************************/
void CScanMapView::PrepareXTabs (CDC* pDC)
{
pDC->SetMapMode(MM_LOMETRIC);

this->UpdateFonts (pDC);					// defines m_CharX
}

/************************************************************************
 *  ScMpView.cpp   		 	C h e c k C a l i b r a t i o n 			*
 ************************************************************************/
void CScanMapView::CheckCalibration (CSize sSizeLP)
{
	CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
	if (ptMap != NULL) 
	{
		CSize sizeDP;
		if (m_pDib->GetSize(&sizeDP))
		{				// change calibration for new bmp size
			ptMap->SetSizeDP (sizeDP);
			ptMap->SetSizeLP (sSizeLP.cx, sSizeLP.cy);
		}
	}
}

/************************************************************************
 *  ScMpView.cpp			G e t C l i e n t R e c t L P				*
 ************************************************************************/
void CScanMapView::GetClientRectLP (CRect& rClientLP)
{
this->GetClientRect(rClientLP);		// bottom=339, right=702
CClientDC dc (this);
dc.SetMapMode (MM_LOMETRIC);
dc.DPtoLP(&rClientLP);			// bottom=-1356, right=2808
}


/************************************************************************
 *  ScMpView.cpp			G e t A b s o l u t e B m p S i z e L P		*
 ************************************************************************/
BOOL CScanMapView::GetAbsoluteBmpSizeLP (CSize* ptActualBmpLP)
{
	BOOL	bOK = FALSE;

	ptActualBmpLP->cx = 100;
	ptActualBmpLP->cy = 100;

	if (m_pDib != NULL)
	{
		CSize sBmpLP;
		CDC* pDC = GetDC();			// get total size LP in ABSOLUTE (positive) values
		pDC->SetMapMode(MM_LOMETRIC);
		if (m_pDib->GetSizeLP (pDC, &sBmpLP))
			bOK = TRUE;

		ReleaseDC(pDC);

		if (bOK)
		{					// needs ABSOLUTE size of bmp (no negative values!)
			*ptActualBmpLP = ptScMpDoc->OriginalToScaledLP (sBmpLP);
			
			this->CheckCalibration (sBmpLP);	// use normal size!!
		}
	}

return bOK;
}

/************************************************************************
 *  ScMpView.cpp				O n U p d a t e 						* 
 *  Called on FileNew, FileOpen, Doc.UpdateAllViews 					*
 ************************************************************************/
void CScanMapView::OnUpdate (CView* pSender, LPARAM lHint, CObject* pHint)
{
//CWayDoc* pDoc = GetDocument();

if (this->GetAbsoluteBmpSizeLP(&m_sizeTotal))	// checks calibration
	{									 
	SetScrollSizes(MM_LOMETRIC, m_sizeTotal);
	}
// In OnUpdate werden die Scrollbars eingestellt.
// In OnPrepareDC darf dies nicht stattfinden, weil sonst ein doppelter
// Bildaufbau erfolgt!!


CTextViewTool::OnUpdate (pSender, lHint, pHint);
}


/************************************************************************
 *  ScMpView.cpp   			 O n P r e p a r e D C						*
 ************************************************************************/
void CScanMapView::OnPrepareDC(CDC* pDC, CPrintInfo* pInfo) 
{
	// TODO: Add your specialized code here and/or call the base class
CWayDoc* pDoc = GetDocument();
ASSERT_VALID(pDoc);  

if (pDoc != NULL)
	{	 
	m_bForPrinter = pDC->IsPrinting();	  // important to get correct update after printing

// In OnUpdate werden die Scrollbars eingestellt.
// In OnPrepareDC darf dies nicht stattfinden, weil sonst ein doppelter
// Bildaufbau erfolgt!!


	pDC->SetMapMode(MM_LOMETRIC);

	this->UpdateFonts (pDC);					// defines m_CharX
	}
		
if (!m_bForPrinter)
	CTextViewTool::OnPrepareDC(pDC, pInfo);	 // sets bContinuePrinting = TRUE if SetMaxPage is set
}

/////////////////////////////////////////////////////////////////////////////
// CScanMapView printing

/************************************************************************
 *  ScMpView.cpp   		  O n P r e p a r e P r i n t i n g				*
 ************************************************************************/
BOOL CScanMapView::OnPreparePrinting(CPrintInfo* pInfo)
{			 // called before the display of the print dialog
	// default preparation
m_bMore = TRUE;	               

UINT nMinPage = 1;						// Number of the first page of the document
UINT nMaxPage = 1;						// Number of the last page of the document
pInfo->SetMinPage (nMinPage);
pInfo->SetMaxPage (nMaxPage);

return DoPreparePrinting(pInfo);
}

/************************************************************************
 *  ScMpView.cpp   		 	O n B e g i n P r i n t i n g				*
 ************************************************************************/
void CScanMapView::OnBeginPrinting(CDC* pDC, CPrintInfo* pInfo)
{			   // called after the print dialog exists
	// TODO: add extra initialization before printing
CWayDoc* pDoc = GetDocument();
ASSERT_VALID(pDoc);  

if (pDoc != NULL)
	{	        
//	pDC->SetMapMode(MM_LOMETRIC);

//	this->UpdateFonts (pDC);					// defines m_CharX

//    this->SetXTabs (pDC, m_CharX);
	}
							// 2. possibility to set the document length 
}

/************************************************************************
 *  ScMpView.cpp   		 		 O n P r i n t							*
 ************************************************************************/
void CScanMapView::OnPrint(CDC* pDC, CPrintInfo* pInfo) 
{
	// TODO: Add your specialized code here and/or call the base class
this->PrintPage (pDC, pInfo->m_nCurPage);
}

/************************************************************************
 *  ScMpView.cpp   		 		 O n E n d P r i n t i n g				*
 ************************************************************************/
void CScanMapView::OnEndPrinting(CDC* pDC, CPrintInfo* pInfo) 
{

}

/////////////////////////////////////////////////////////////////////////////
// CScanMapView diagnostics

#ifdef _DEBUG
void CScanMapView::AssertValid() const
{
	CTextViewTool::AssertValid();
}

void CScanMapView::Dump(CDumpContext& dc) const
{
	CTextViewTool::Dump(dc);
}

CWayDoc* CScanMapView::GetDocument() // non-debug version is inline
{
	ASSERT(m_pDocument->IsKindOf(RUNTIME_CLASS(CWayDoc)));
	return (CWayDoc*)m_pDocument;
}
#endif //_DEBUG


/************************************************************************
 *  ScMpView.cpp   		 		 S c r o l l V e r t M a x				*
 ************************************************************************/
void CScanMapView::ScrollVertMax (BOOL bMax)
{
CPoint pScrollLP;
pScrollLP = this->GetScrollPosition();
CSize sMapLP = m_sizeTotal;			// (3028, 4448)	  m_sizeTotal
									// same as GetTotalSize

CRect rClientLP;					// get size of client area
this->GetClientRectLP(rClientLP);
if (!bMMText)
	{	 
	rClientLP.top		= -rClientLP.top;
	rClientLP.bottom	= -rClientLP.bottom;
	}
														// always positive
//int	dMaxHorzScroll = sMapLP.cx - rClientLP.Width();		// 220
int dMaxVertScroll = sMapLP.cy - rClientLP.Height();	// 3092


if (dMaxVertScroll > 0)
	{
	pScrollLP.y = bMax? dMaxVertScroll : 0;
	if (!bMMText)	 
		pScrollLP.y = -pScrollLP.y;

	this->ScrollToPosition (pScrollLP);
	}
}

/************************************************************************
 *  ScMpView.cpp   		 		 S c r o l l H o r z M a x				*
 ************************************************************************/
void CScanMapView::ScrollHorzMax (BOOL bMax)
{
CPoint pScrollLP;
pScrollLP = this->GetScrollPosition();

CSize sMapLP = m_sizeTotal;			// (3028, 4448)	  m_sizeTotal
									// same as GetTotalSize

CRect rClientLP;					// get size of client area
this->GetClientRectLP(rClientLP);
if (!bMMText)
	{	 
	rClientLP.top		= -rClientLP.top;
	rClientLP.bottom	= -rClientLP.bottom;
	}

int	dMaxHorzScroll = sMapLP.cx - rClientLP.Width();		// 220
//int dMaxVertScroll = sMapLP.cy - rClientLP.Height();	// 3092


if (dMaxHorzScroll > 0)
	{
	pScrollLP.x = bMax? dMaxHorzScroll : 0;
	this->ScrollToPosition (pScrollLP);			// initiates view update!!
	}
}

/************************************************************************
 *  ScMpView.cpp   		 		 S c r o l l T o						*
 *  Purpose: scrolls without drawing									*
 ************************************************************************/
void CScanMapView::ScrollTo (CPoint pScrollLP)
{
	if (!bMMText)	 
		pScrollLP.y = -pScrollLP.y;

	CPoint pScrollDP;
	pScrollDP.x = pScrollLP.x;
	pScrollDP.y = pScrollLP.y;

	CClientDC dc (this);
	dc.SetMapMode(MM_LOMETRIC);
	dc.LPtoDP(&pScrollDP);

	SetScrollPos(SB_HORZ, pScrollDP.x);
	SetScrollPos(SB_VERT, pScrollDP.y);
}

/////////////////////////////////////////////////////////////////////////////
// CScanMapView message handlers

void CScanMapView::OnSize(UINT nType, int cx, int cy) 
{
CTextViewTool::OnSize(nType, cx, cy);

	// TODO: Code fr die Behandlungsroutine fr Nachrichten hier einfgen

}


/************************************************************************
 *  ScMpView.cpp   		 		O n M o u s e M o v e					*
 ************************************************************************/
void CScanMapView::OnMouseMove(UINT nFlags, CPoint point) 
{
	// TODO: Add your message handler code here and/or call default

	m_pMouseLP = point;

	if (m_bTBMapCenter)
	{
		::SetCursor (::LoadCursor(NULL, IDC_CROSS));
	}
	else
	{								// is moving map running?
		BOOL bRecording = FALSE;
		CMovingMapDoc* ptMovingMap = ptInit->GetMovingMapDocPtr();
		if (ptMovingMap != NULL)
		{							
			bRecording = ptMovingMap->IsRecording();
		}

		if (!bRecording)	// this check is also required in CMapView::OnMouseMove!!
		{					// mouse move handling, only if moving map is NOT running!
			CClientDC dc (this);
			dc.SetMapMode (MM_LOMETRIC);
			dc.DPtoLP (&m_pMouseLP);

			CPoint pScrollLP;
			pScrollLP = this->GetScrollPosition();

			double dLat, dLon;
			long lLatLP = m_pMouseLP.y + pScrollLP.y;
			long lLonLP = m_pMouseLP.x + pScrollLP.x;

			CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
			if (ptMap != NULL)
			{
				if (ptMap->IsCalibrated())
				{
					ptMap->LPtoLatLon (lLonLP, lLatLP, &dLat, &dLon); 
					m_ptInterMap->SetLatLon (dLat, dLon);
					m_ptInterMap->StartPosChangedTimer(10);		// 10 1/10 sec = 1 sec
				}
				else
				{
					::SetCursor (::LoadCursor(NULL, IDC_CROSS));

					dLat = lLatLP;				// while testing...
					dLon = lLonLP;   
				}
			}
		}
	}

	CTextViewTool::OnMouseMove(nFlags, point);
}

/************************************************************************
 *  ScMpView.cpp   		 	O n L B u t t o n D o w n					*
 ************************************************************************/
void CScanMapView::OnLButtonDown(UINT nFlags, CPoint point) 
{
	// TODO: Add your message handler code here and/or call default
	m_pMouseLP = point;

	CClientDC dc (this);
	dc.SetMapMode (MM_LOMETRIC);
	dc.DPtoLP (&m_pMouseLP);

	CPoint pScrollLP;
	pScrollLP = this->GetScrollPosition();
	
	CTextViewTool::OnLButtonDown(nFlags, point);
					   			// store global position for OnDraw with scroll offset 
	m_ActPointLP = m_pMouseLP +	pScrollLP;


	if (m_bTBMapCenter)
	{
		m_bTBMapCenter = FALSE;
		this->OnMenuCenter();
	}
	else
	{
		CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
		if (ptMap != NULL)
		{
			if (ptMap->IsCalibrated())
			{
				if (ptMap->IsActivated(SHOW_MAP))
				{
					double  dLat, dLon;			// get mouse position
					ptMap->LPtoLatLon (m_ActPointLP.x, m_ActPointLP.y, &dLat, &dLon); 
					CMainFrame* ptMainFrm = (CMainFrame*)this->GetParent();
					ptMainFrm->EnableNextCmd();
					m_ptInterMap->OnShowScannedMap (dLat, dLon, ptMainFrm, ptScMpDoc->GetCmdID());
				}
				else
				{
					m_ptInterMap->OnActivateLeg (GetDocument());
				}
			}
			else
			{					 // show calibration dialog to get actual lat lon
				m_bDrawFixPoint = TRUE;
				this->DrawFixPoint(&dc, m_pMouseLP);// draw directly at mouse position (no scroll offset needed!)


				CPoint pOriginalLP = ptScMpDoc->ScaledToOriginalLP(m_ActPointLP);
				ptScMpDoc->SaveMousePos(pOriginalLP);
									 // map may be calibrated in SaveMousePos

				if (!ptMap->IsCalibrated())
				{
					if (ptScMpDoc->ChangeCalibration(this)) 
					{
						m_bDrawFixPoint = !ptMap->IsCalibrated();
					}
					else
					{					// abort selected in CScMpCalibDlg
						m_bDrawFixPoint = FALSE;
						BOOL bFound = FALSE;
						if (this->IsForRoute())
						{
							CWayDoc* ptWayDoc = GetDocument();
							bFound = ptScMpDoc->ActivateNearestMap(ptWayDoc);
						}
						else
						{
							bFound = ptScMpDoc->ActivateNearestMap(ptTrackDoc);
						}

						if (!bFound)
						{
							ptScMpDoc->SetActMapIndex(-1);

							ptScMpDoc->SetAborted(TRUE);
							WPARAM wParam = MAKELONG (ID_VIEW_LAST, 0);
							(this->GetParent())->PostMessage(WM_COMMAND, wParam, 0);
						}
						else
						{
							this->InvalidateRect(NULL);		// remove yellow circle
						}
					}
				}
				else
				{					   // calibration finished!

					m_bDrawFixPoint = FALSE;
					this->InvalidateRect(NULL);		// remove yellow circle
				}
			}
		}
	}

//	::SetCursor (::LoadCursor(NULL, IDC_ARROW));
}

/************************************************************************
 *  ScMpView.cpp	  		 	O n T B M a p C e n t e r				*
 ************************************************************************/
void CScanMapView::OnTBMapCenter() 
{
	// TODO: Add your command handler code here
m_bTBMapCenter = TRUE;	
}

/************************************************************************
 *  ScMpView.cpp	  		 	O n T B M a p L a r g e r				*
 ************************************************************************/
void CScanMapView::OnTBMapLarger() 
{
	// TODO: Add your command handler code here
m_bTBMapLarger = TRUE;
OnMenuLarger();
m_bTBMapLarger = FALSE;	
}

/************************************************************************
 *  ScMpView.cpp	  		 	O n T B M a p S m a l l e r				*
 ************************************************************************/
void CScanMapView::OnTBMapSmaller() 
{
	// TODO: Add your command handler code here
m_bTBMapSmaller = TRUE;
OnMenuSmaller();	
m_bTBMapSmaller = FALSE;	
}

/************************************************************************
 *  ScMpView.cpp	  		 	O n C o n t e x t M e n u				*
 ************************************************************************/
void CScanMapView::OnContextMenu(CWnd* pWnd, CPoint point) 
{
	// TODO: Add your message handler code here
short nVirtKey;			// If the high-order bit is 1, the key is down; 
						//	otherwise, it is up.

nVirtKey = GetKeyState(VK_CONTROL); 
if ((nVirtKey & 0x8000) == 0x8000)
	{
	m_ptInterMap->ShowMapPopupMenu (point);
	}
else{
	m_ptInterMap->ShowWayPopupMenu (point, GetDocument());
	}
}

/************************************************************************
 *  ScMpView.cpp	  		 	O n M e n u X X X						*
 ************************************************************************/

void CScanMapView::OnMenuAppend() 
{
	// TODO: Add your command handler code here
OnMenuAppMapLoc(ID_WY_APP_FIRST);  // use first and only entry
}

void CScanMapView::OnMenuInsert() 
{
	// TODO: Add your command handler code here
OnMenuInsMapLoc(ID_WY_INS_FIRST);  // use first and only entry
}

void CScanMapView::OnMenuBear() 
{
	// TODO: Add your command handler code here
OnMenuBearMapNav(ID_WY_BEAR_FIRST);  // use first and only entry	
}

void CScanMapView::OnMenuDelete() 
{
	// TODO: Add your command handler code here
OnMenuDelRteWpt(ID_WY_DEL_FIRST);  // use first and only entry	
}

void CScanMapView::OnMenuActivate() 
{
	// TODO: Add your command handler code here
OnMenuActRteWpt(ID_WY_ACT_FIRST);  // use first and only entry	
}

void CScanMapView::OnMenuPlAlt() 
{
	// TODO: Add your command handler code here
OnMenuPlAltRteWpt(ID_WY_PLALT_FIRST);  // use first and only entry	
}


/************************************************************************
 *  ScMpView.cpp	  		 	O n U p d a t e X X X					*
 ************************************************************************/

void CScanMapView::OnUpdateWyApp(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
m_ptInterMap->OnUpdateWyApp(pCmdUI);	
}

void CScanMapView::OnUpdateWyIns(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
m_ptInterMap->OnUpdateWyIns(pCmdUI);	
}

void CScanMapView::OnUpdateWyBear(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
m_ptInterMap->OnUpdateWyBear(pCmdUI);	
}

void CScanMapView::OnUpdateWyDel(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
m_ptInterMap->OnUpdateWyDel(pCmdUI);	
}

void CScanMapView::OnUpdateWyAct(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
m_ptInterMap->OnUpdateWyAct(pCmdUI);	
}

void CScanMapView::OnUpdateWyPlAlt(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
m_ptInterMap->OnUpdateWyPlAlt(pCmdUI);	
}



/************************************************************************
 *  ScMpView.cpp	  	 	O n M e n u A p p M a p L o c				*
 ************************************************************************/
void CScanMapView::OnMenuAppMapLoc(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuAppMapLoc (nID);
}

/************************************************************************
 *  ScMpView.cpp	  	 	O n M e n u I n s M a p L o c				*
 ************************************************************************/
void CScanMapView::OnMenuInsMapLoc(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuInsMapLoc (nID);
}

/************************************************************************
 *  ScMpView.cpp	  	 	O n M e n u B e a r M a p N a v				*
 ************************************************************************/
void CScanMapView::OnMenuBearMapNav(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuBearMapNav (nID);
}

/************************************************************************
 *  ScMpView.cpp	  	 	O n M e n u D e l R t e W p t				*
 ************************************************************************/
void CScanMapView::OnMenuDelRteWpt(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuDelRteWpt (nID);
}

/************************************************************************
 *  ScMpView.cpp	  	 	O n M e n u A c t R t e W p t				*
 ************************************************************************/
void CScanMapView::OnMenuActRteWpt(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuActRteWpt (nID);
}

/************************************************************************
 *  ScMpView.cpp	  	 	O n M e n u P l A l t R t e W p t			*
 ************************************************************************/
void CScanMapView::OnMenuPlAltRteWpt(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuPlAltRteWpt (nID);
}

/************************************************************************
 *  ScMpView.cpp	  	 O n M e n u L o c E d i t M a p L o c			*
 ************************************************************************/
void CScanMapView::OnMenuMeEditMapLoc(UINT nID) 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
m_ptInterMap->OnMenuMeEditMapLoc (nID);
}

/************************************************************************
 *  ScMpView.cpp	   O n M e n u L o c E d i t M a p A i r s p a c e	*
 ************************************************************************/
void CScanMapView::OnMenuMeEditAirspace(UINT nID) 
{
	CWayDoc* ptWayDoc = GetDocument();
	m_ptInterMap->OnMenuMeEditAirspace (nID, ptWayDoc);
}

/************************************************************************
 *  ScMpView.cpp	  	 	O n E d i t C o p y							*
 ************************************************************************/
void CScanMapView::OnEditCopy() 
{
	// TODO: Add your command handler code here
this->CopyToClipboard();		
}

/****************************************************************************
 *	MapView.cpp						OnMenuCenter							*
 ****************************************************************************/										 
void CScanMapView::OnMenuCenter() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
	CRect rClientLP;					// get size of client area
	this->GetClientRectLP(rClientLP);

	CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
	if (ptMap != NULL)
	{
		double dLat, dLon;
		ptMap->LPtoLatLon (m_ActPointLP.x, m_ActPointLP.y, &dLat, &dLon);

		BOOL bRedraw = TRUE;
		if (DoCenter (dLat, dLon, bRedraw))
			this->Invalidate(FALSE);
	}
}


/************************************************************************
 *  ScMpView.cpp	  		 	ScrollToScaledPos 						*
 ************************************************************************/
void CScanMapView::ScrollToScaledPos (CRect rClientLP)
{
	CPoint  pScrollPos;			// change scrollbars

	long lCenterLPx, lCenterLPy;			// centered pos on map in LP koords
	
	CPoint pScrollLP;
	pScrollLP = this->GetScrollPosition();
	
	CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
	if (ptMap != NULL)
	{
		double dLat, dLon;
							// get logical coords of center
		lCenterLPx = pScrollLP.x + (long)(0.5 * rClientLP.Width());
		lCenterLPy = pScrollLP.y + (long)(0.5 * rClientLP.Height());
							// get lat lon of centered point
		ptMap->LPtoLatLon (lCenterLPx, lCenterLPy, &dLat, &dLon);

		ptMap->SetScaleFaktX(ptScMpDoc->GetScaleFaktX());
		ptMap->SetScaleFaktY(ptScMpDoc->GetScaleFaktY());
							
							// get new logical coords after map is scaled
		ptMap->LatLonToLP(dLon, dLat, &lCenterLPx, &lCenterLPy);
	}


							// calc scroll pos (LP) to center new CenterLP
	pScrollPos.x = lCenterLPx - (long)(0.5 * rClientLP.Width());
	pScrollPos.y = lCenterLPy - (long)(0.5 * rClientLP.Height());


	CSize sVirt = m_sizeTotal;
	CSize pageSize (rClientLP.Width(), rClientLP.Height());		// page scroll
	sVirt.cy    = abs(sVirt.cy);				// to avoid negative scroll size
	pageSize.cy = abs(pageSize.cy);				// to avoid negative scroll size
	CSize lineSize (pageSize.cx/4, pageSize.cy/4);		// line scroll
	SetScrollSizes (MM_LOMETRIC, sVirt, pageSize, lineSize);

										// avoid moving image out of client area
	if (pageSize.cx > sVirt.cx)
		pScrollPos.x = 0;				// horizontal scroll bar hidden
	if (pageSize.cy > sVirt.cy)
		pScrollPos.y = 0;				// vertical scroll bar hidden


	if (pScrollPos.x < 0) pScrollPos.x = 0;		// avoid negative values
	if (bMMText)
		{
		if (pScrollPos.y < 0) pScrollPos.y = 0;		// avoid negative values
		}
	else{
		if (pScrollPos.y > 0) pScrollPos.y = 0;		// avoid positive values
		}

	ScrollToPosition (pScrollPos);			// logical 
}

/************************************************************************
 *  ScMpView.cpp   		 G e t R i g h t S c r e e n L a t L o n		*
 *  Returns position of point on the middle of the right side of map	*
 ************************************************************************/
BOOL CScanMapView::GetRightScreenLatLon(double* ptLat, double* ptLon)
{
	BOOL bDone=FALSE;

							// get center of screen LP
	CRect rScreenLP = GetUseableRectLP();

	CPoint	pCenterRightLP;
	pCenterRightLP.x = rScreenLP.right;
	pCenterRightLP.y = (rScreenLP.bottom + rScreenLP.top)/2;

							// get map relative LP
	CPoint pScrollLP;
	pScrollLP = this->GetScrollPosition();

	long lLatLP = pCenterRightLP.y + pScrollLP.y;
	long lLonLP = pCenterRightLP.x + pScrollLP.x;

							// get lat lon of center of screen
	CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
	if (ptMap != NULL)
	{
		if (ptMap->IsCalibrated())
		{
			ptMap->LPtoLatLon (lLonLP, lLatLP, ptLat, ptLon); 
			bDone = TRUE;
		}
	}

	return bDone;
}

/************************************************************************
 *  ScMpView.cpp   		 G e t L e f t S c r e e n L a t L o n			*
 *  Returns position of point on the middle of the left side of map		*
 ************************************************************************/
BOOL CScanMapView::GetLeftScreenLatLon(double* ptLat, double* ptLon)
{
	BOOL bDone=FALSE;

							// get center of screen LP
	CRect rScreenLP = GetUseableRectLP();

	CPoint	pCenterLeftLP;
	pCenterLeftLP.x = rScreenLP.left;
	pCenterLeftLP.y = (rScreenLP.bottom + rScreenLP.top)/2;

							// get map relative LP
	CPoint pScrollLP;
	pScrollLP = this->GetScrollPosition();

	long lLatLP = pCenterLeftLP.y + pScrollLP.y;
	long lLonLP = pCenterLeftLP.x + pScrollLP.x;

							// get lat lon of center of screen
	CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
	if (ptMap != NULL)
	{
		if (ptMap->IsCalibrated())
		{
			ptMap->LPtoLatLon (lLonLP, lLatLP, ptLat, ptLon); 
			bDone = TRUE;
		}
	}

	return bDone;
}

/************************************************************************
 *  ScMpView.cpp   		 G e t T o p S c r e e n L a t L o n			*
 *  Returns position of point on the middle of the top side of map		*
 ************************************************************************/
BOOL CScanMapView::GetTopScreenLatLon(double* ptLat, double* ptLon)
{
	BOOL bDone=FALSE;

							// get center of screen LP
	CRect rScreenLP = GetUseableRectLP();

	CPoint	pCenterTopLP;
	pCenterTopLP.x = (rScreenLP.right + rScreenLP.left)/2;
	pCenterTopLP.y = rScreenLP.top;

							// get map relative LP
	CPoint pScrollLP;
	pScrollLP = this->GetScrollPosition();
	long lLatLP = pCenterTopLP.y + pScrollLP.y;
	long lLonLP = pCenterTopLP.x + pScrollLP.x;

							// get lat lon of center of screen
	CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
	if (ptMap != NULL)
	{
		if (ptMap->IsCalibrated())
		{
			ptMap->LPtoLatLon (lLonLP, lLatLP, ptLat, ptLon); 
			bDone = TRUE;
		}
	}

	return bDone;
}

/************************************************************************
 *  ScMpView.cpp   		 G e t B o t t o m S c r e e n L a t L o n		*
 *  Returns position of point on the middle of the bottom side of map	*
 ************************************************************************/
BOOL CScanMapView::GetBottomScreenLatLon(double* ptLat, double* ptLon)
{
	BOOL bDone=FALSE;

									// get bottom center of screen LP
	CRect rScreenLP = GetUseableRectLP();

	CPoint	pCenterBottomLP;
	pCenterBottomLP.x = (rScreenLP.right + rScreenLP.left)/2;
	pCenterBottomLP.y = rScreenLP.bottom;

							// get map relative LP
	CPoint pScrollLP;
	pScrollLP = this->GetScrollPosition();
	long lLatLP = pCenterBottomLP.y + pScrollLP.y;
	long lLonLP = pCenterBottomLP.x + pScrollLP.x;

							// get lat lon of center of screen
	CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
	if (ptMap != NULL)
	{
		if (ptMap->IsCalibrated())
		{
			ptMap->LPtoLatLon (lLonLP, lLatLP, ptLat, ptLon); 
			bDone = TRUE;
		}
	}

	return bDone;
}

/************************************************************************
 *  ScMpView.cpp   		 G e t C e n t e r S c r e e n L a t L o n		*
 ************************************************************************/
BOOL CScanMapView::GetCenterScreenLatLon(double* ptLat, double* ptLon)
{
	BOOL bDone=FALSE;

							// get center of screen LP
	CRect rScreenLP = GetUseableRectLP();
	CPoint	pCenterLP;
	pCenterLP.x = (rScreenLP.right + rScreenLP.left)/2;
	pCenterLP.y = (rScreenLP.bottom + rScreenLP.top)/2;

							// get map relative LP
	CPoint pScrollLP;
	pScrollLP = this->GetScrollPosition();

	long lLatLP = pCenterLP.y + pScrollLP.y;
	long lLonLP = pCenterLP.x + pScrollLP.x;

							// get lat lon of center of screen
	CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
	if (ptMap != NULL)
	{
		if (ptMap->IsCalibrated())
		{
			ptMap->LPtoLatLon (lLonLP, lLatLP, ptLat, ptLon); 
			bDone = TRUE;
		}
	}

	return bDone;
}

/****************************************************************************
 *	MapView.cpp						OnMenuLarger							*
 ****************************************************************************/										 
void CScanMapView::OnMenuLarger() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
	CRect rClientLP;					// get size of client area
	this->GetClientRectLP(rClientLP);

	CDC* pDC = GetDC();			// to get resolution of screen in NextMapMiddle
	double dLat, dLon;

	if (m_bTBMapLarger)		// initiated via toolbar "+"
	{						// set dLat/dLon to center of screen
		GetCenterScreenLatLon(&dLat, &dLon);
	}
	else
	{			// initiated via context menu
				// set dLat/dLon to last mouse position, saved in
				// OnMouseMove: m_ptInterMap->SetLatLon (dLat, dLon);
		CMapDoc* ptMapDoc = ptInit->GetMapDocPtr();
		CLatLon llMousePos = ptMapDoc->GetLatLon();
		dLat = llMousePos.GetLat();
		dLon = llMousePos.GetLon();
	}


	ptScMpDoc->AssignIncreaseCountersFor(pDC, dLat, dLon);
			
			// first try to get higher resolution map
	BOOL bLarger = TRUE;
	BOOL bMapChanged = FALSE;
	if (ptScMpDoc->NextMapMiddle(&m_sizeTotal, bLarger, &bMapChanged))
	{
		if (bMapChanged)
		{								
			this->GetAbsoluteBmpSizeLP(&m_sizeTotal);   // checks calibration

			SetScrollSizes(MM_LOMETRIC, m_sizeTotal);

			BOOL bRedraw = TRUE;
			DoCenter (dLat, dLon, bRedraw);
		}
		else
		{		// try to enlarge current map
			ptScMpDoc->ChangeIncreaseCnt(1);

										// calc new total size
			if (ptScMpDoc->CalcActFakt (m_sOrigImage, &m_sizeTotal))
			{
				this->ScrollToScaledPos (rClientLP);
				this->Invalidate(FALSE);
			}
			else
			{	// limit of short reached: stop increasing map !!
				// Shorts are used in LPtoDP ((LPPOINT)...) in ScrollToScaledPos in SetScrollSizes!!
				ptScMpDoc->ChangeIncreaseCnt(-1);
				ptScMpDoc->CalcActFakt (m_sOrigImage, &m_sizeTotal);
				MessageBeep (MB_OK);
			}
		}
	}
}	

/****************************************************************************
 *	MapView.cpp						OnMenuSmaller							*
 ****************************************************************************/										 
void CScanMapView::OnMenuSmaller() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
	CRect rClientLP;					// get size of client area
	this->GetClientRectLP(rClientLP);

	CDC* pDC = GetDC();			// to get resolution of screen in NextMapMiddle
	double dLat, dLon;

	if (m_bTBMapSmaller)	// initiated via toolbar "-"
	{						// set dLat/dLon to center of screen
		GetCenterScreenLatLon(&dLat, &dLon);
	}
	else	
	{			// initiated via context menu
				// set dLat/dLon to last mouse position, saved in
				// OnMouseMove: m_ptInterMap->SetLatLon (dLat, dLon);
		CMapDoc* ptMapDoc = ptInit->GetMapDocPtr();
		CLatLon llMousePos = ptMapDoc->GetLatLon();
		dLat = llMousePos.GetLat();
		dLon = llMousePos.GetLon();
	}

	ptScMpDoc->AssignIncreaseCountersFor(pDC, dLat, dLon);

			// first try to get lower resolution map
	BOOL bLarger = FALSE;
	BOOL bMapChanged = FALSE;
	if (ptScMpDoc->NextMapMiddle(&m_sizeTotal, bLarger, &bMapChanged))
	{
		if (bMapChanged)
		{
			this->GetAbsoluteBmpSizeLP(&m_sizeTotal);   // checks calibration

			SetScrollSizes(MM_LOMETRIC, m_sizeTotal);
			BOOL bRedraw = TRUE;
			DoCenter (dLat, dLon, bRedraw);
		}
		else
		{		// reduce size of current map
			ptScMpDoc->ChangeIncreaseCnt(-1);
												// calc new total size
			if (ptScMpDoc->CalcActFakt (m_sOrigImage, &m_sizeTotal))
			{
				this->ScrollToScaledPos (rClientLP);
				this->Invalidate(FALSE);
			}
		}
	}
}			

/************************************************************************
 *  ScMpView.cpp	  	 	O n S h o w x x x							*
 ************************************************************************/
void CScanMapView::OnShowAirport() 
{
	// TODO: Add your command handler code here
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		{
		ptMap->InvertActivateState (SHOW_APT);
		this->Invalidate(FALSE);   // dont erase before update
		}
	}
}

void CScanMapView::OnShowReport() 
{
	// TODO: Add your command handler code here
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		{
		ptMap->InvertActivateState (SHOW_REP);
		this->Invalidate(FALSE);   // dont erase before update
		}
	}
}

void CScanMapView::OnMShowIntersect() 
{
	// TODO: Add your command handler code here
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		{
		ptMap->InvertActivateState (SHOW_INT);
		this->Invalidate(FALSE);   // dont erase before update
		}
	}
}

void CScanMapView::OnShowNdb() 
{
	// TODO: Add your command handler code here
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		{
		ptMap->InvertActivateState (SHOW_NDB);
		this->Invalidate(FALSE);   // dont erase before update
		}
	}
}

void CScanMapView::OnShowVor() 
{
	// TODO: Add your command handler code here
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		{
		ptMap->InvertActivateState (SHOW_VOR);
		this->Invalidate(FALSE);   // dont erase before update
		}
	}
}

void CScanMapView::OnShowUsr() 
{
	// TODO: Add your command handler code here
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		{
		ptMap->InvertActivateState (SHOW_USR);
		this->Invalidate(FALSE);   // dont erase before update
		}
	}
}

void CScanMapView::OnShowSail() 
{
	// TODO: Add your command handler code here
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		{
		ptMap->InvertActivateState (SHOW_SAIL);
		this->Invalidate(FALSE);   // dont erase before update
		}
	}
}

void CScanMapView::OnShowHeli() 
{
	// TODO: Add your command handler code here
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		{
		ptMap->InvertActivateState (SHOW_HELI);
		this->Invalidate(FALSE);   // dont erase before update
		}
	}
}

void CScanMapView::OnShowMAP() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		{
		ptMap->InvertActivateState (SHOW_MAP);
		this->Invalidate(FALSE);   // dont erase before update
		}
	}
}


void CScanMapView::OnShowLabel() 
{
	// TODO: Add your command handler code here
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		{
		ptMap->InvertActivateState (SHOW_LABEL);
		this->Invalidate(FALSE);   // dont erase before update
		}
	}
}

void CScanMapView::OnShowElev() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		{
		ptMap->InvertActivateState (SHOW_ELEV);
		this->Invalidate(FALSE);   // dont erase before update
		}
	}
}

void CScanMapView::OnShowAirspace() 
{
	// TODO: Code fr Befehlsbehandlungsroutine hier einfgen
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		{
		ptMap->InvertActivateState (SHOW_AIRSPACE);
		this->Invalidate(FALSE);   // dont erase before update
		}
	}
}


/************************************************************************
 *  ScMpView.cpp	  	 	O n U p d a t e x x x						*
 ************************************************************************/
void CScanMapView::OnUpdateShow (CCmdUI* pCmdUI, short nType)
{
CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	if (ptMap->IsCalibrated())
		pCmdUI->SetCheck (ptMap->IsActivated (nType));
	}	

}

void CScanMapView::OnUpdateAirport(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
this->OnUpdateShow (pCmdUI, SHOW_APT);	
}

void CScanMapView::OnUpdateIntersect(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
this->OnUpdateShow (pCmdUI, SHOW_INT);	
}

void CScanMapView::OnUpdateLabel(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
this->OnUpdateShow (pCmdUI, SHOW_LABEL);	
}

void CScanMapView::OnUpdateElev(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
	BOOL bEnable = FALSE;

	if (ptElevDoc != NULL)
		bEnable = (ptElevDoc->GetElevCnt() > 0);

	this->OnUpdateShow (pCmdUI, SHOW_ELEV);	
	pCmdUI->Enable(bEnable);	
}

void CScanMapView::OnUpdateAirspace(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
	BOOL bEnable = FALSE;

	CAirspaceRegionsDoc* ptAirspaceRegionsDoc = ptInit->GetAirspaceRegionsDocPtr();
	if (ptAirspaceRegionsDoc != NULL)
		bEnable = (ptAirspaceRegionsDoc->GetCnt() > 0);

	this->OnUpdateShow (pCmdUI, SHOW_AIRSPACE);	
	pCmdUI->Enable(bEnable);	
}

void CScanMapView::OnUpdateNdb(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
this->OnUpdateShow (pCmdUI, SHOW_NDB);	
}

void CScanMapView::OnUpdateReport(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
this->OnUpdateShow (pCmdUI, SHOW_REP);	
}

void CScanMapView::OnUpdateUsr(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
this->OnUpdateShow (pCmdUI, SHOW_USR);	
}

void CScanMapView::OnUpdateSail(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
this->OnUpdateShow (pCmdUI, SHOW_SAIL);	
}

void CScanMapView::OnUpdateHeli(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
this->OnUpdateShow (pCmdUI, SHOW_HELI);	
}

void CScanMapView::OnUpdateMAP(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
this->OnUpdateShow (pCmdUI, SHOW_MAP);	
}

void CScanMapView::OnUpdateVor(CCmdUI* pCmdUI) 
{
	// TODO: Add your command update UI handler code here
this->OnUpdateShow (pCmdUI, SHOW_VOR);	
}

/************************************************************************
 *  ScMpView.cpp   		 		O n K e y D o w n			 			*
 ************************************************************************/
void CScanMapView::OnKeyDown(UINT nChar, UINT nRepCnt, UINT nFlags) 
{
	// TODO: Code fr die Behandlungsroutine fr Nachrichten hier einfgen und/oder Standard aufrufen
	double dLat, dLon;
	CPoint pScrollLP;
	pScrollLP = this->GetScrollPosition();			// (220, 0)
	CSize sMapLP = m_sizeTotal;								// (3028, 4448)
															// same as GetTotalSize

	CRect rClientLP;					// get size of client area
	this->GetClientRectLP(rClientLP);
	if (!bMMText)
		{	 
		pScrollLP.y			= -pScrollLP.y;
		rClientLP.top		= -rClientLP.top;
		rClientLP.bottom	= -rClientLP.bottom;
		}

	short EPS = 4;
	int	dMaxHorzScroll = sMapLP.cx - rClientLP.Width();		// 220
	int dMaxVertScroll = sMapLP.cy - rClientLP.Height();	// 3092

	BOOL bNewMap=FALSE;
	switch (nChar)
	{							
	case 'v':
	case 'V':
		{
			CWayPoint WayPt;
			CWayDoc* ptWayDoc = GetDocument();
			if (ptWayDoc != NULL)
			{
				short i = ptWayDoc->GetActInd();
				if (ptWayDoc->GetWayPointPtr (i, &WayPt))
				{
					if (ptScMpDoc->GetIncreaseCnt() != 0)
					{
						ptScMpDoc->SetIncreaseCnt(0);
						ptScMpDoc->CalcActFakt (m_sOrigImage, &m_sizeTotal);
					}

					CLatLon LLact(WayPt.GetLat(), WayPt.GetLon());
				
					short nNewIndex=-1;				// index of visual approach chart
					if (ptScMpDoc->GetVACIndexFor (LLact, &nNewIndex))
					{
						short nOldIndex = ptScMpDoc->GetActMapIndex();
						if (nNewIndex != nOldIndex)
						{					// activate visual approach chart
							ptScMpDoc->ActivateMap (nNewIndex);
						}
						else
						{					// try to activate ICAO card
							CLatLon LLactual;
							ptWayDoc->GetActLatLonPtr(&LLactual);
							ptScMpDoc->ActivateScMapForLatLon(LLactual);
						}
						
						
						m_bOnDrawScroll=TRUE;	// activate scrolling after update
						m_bHorz = TRUE;			// move horz bar left
						m_bMax = TRUE;			// call OnInitialUpdate

						if (m_bOnDrawScroll)
						{					// initiating view update here!

							this->GetAbsoluteBmpSizeLP(&m_sizeTotal);   // checks calibration

							SetScrollSizes(MM_LOMETRIC, m_sizeTotal);	// needs absolute LP koords
							
							BOOL bRedraw = TRUE;
							this->DoCenter (WayPt.GetLat(), WayPt.GetLon(), bRedraw);
						}					
					}
				}
			}
		}
		break;

	case VK_LEFT:
		if (dMaxHorzScroll<0 || pScrollLP.x == 0)
		{
			GetLeftScreenLatLon(&dLat, &dLon);

			if (ptScMpDoc->GetIncreaseCnt() != 0)
			{
				ptScMpDoc->SetIncreaseCnt(0);
				ptScMpDoc->CalcActFakt (m_sOrigImage, &m_sizeTotal);
			}

			if (ptScMpDoc->NextMapWest(&m_sizeTotal, dLat))
			{
				m_bOnDrawScroll=TRUE;	// activate scrolling after update
				m_bHorz = TRUE;			// move horz bar left
				m_bMax = TRUE;			// call OnInitialUpdate

				if (m_bOnDrawScroll)
				{					// initiating view update here!

					this->GetAbsoluteBmpSizeLP(&m_sizeTotal);   // checks calibration

					SetScrollSizes(MM_LOMETRIC, m_sizeTotal);
					BOOL bRedraw = TRUE;
					DoCenter (dLat, dLon, bRedraw);
				} 
			}
			else
			{
				AfxMessageBox (IDS_NO_MAP_WEST);
			}
		}
		break;
	case VK_RIGHT:
		if (dMaxHorzScroll<0 || pScrollLP.x >= dMaxHorzScroll-EPS)
		{
			GetRightScreenLatLon(&dLat, &dLon);

			if (ptScMpDoc->GetIncreaseCnt() != 0)
			{
				ptScMpDoc->SetIncreaseCnt(0);
				ptScMpDoc->CalcActFakt (m_sOrigImage, &m_sizeTotal);
			}

			if (ptScMpDoc->NextMapEast(&m_sizeTotal, dLat))
			{
				m_bOnDrawScroll=TRUE;	// activate scrolling after update
				m_bHorz = TRUE;			// move horz bar left
				m_bMax = FALSE;			// call OnInitialUpdate

				if (m_bOnDrawScroll)
				{					// initiating view update here!
					this->GetAbsoluteBmpSizeLP(&m_sizeTotal);   // checks calibration

					SetScrollSizes(MM_LOMETRIC, m_sizeTotal);
					BOOL bRedraw = TRUE;
					DoCenter (dLat, dLon, bRedraw);
				} 
			}
			else
			{
				AfxMessageBox (IDS_NO_MAP_EAST);
			}
		}
		break;
	case VK_UP:
		if (dMaxVertScroll<0 || pScrollLP.y == 0)
		{
			GetTopScreenLatLon(&dLat, &dLon);
			
			if (ptScMpDoc->GetIncreaseCnt() != 0)
			{
				ptScMpDoc->SetIncreaseCnt(0);
				ptScMpDoc->CalcActFakt (m_sOrigImage, &m_sizeTotal);
			}

			if (ptScMpDoc->NextMapNorth(&m_sizeTotal, dLon))
			{
				m_bOnDrawScroll=TRUE;	// activate scrolling after update
				m_bHorz = FALSE;		// move vert bar left
				m_bMax = TRUE;			// call OnInitialUpdate

				if (m_bOnDrawScroll)
				{					// initiating view update here!
					this->GetAbsoluteBmpSizeLP(&m_sizeTotal);   // checks calibration

					SetScrollSizes(MM_LOMETRIC, m_sizeTotal);
					BOOL bRedraw = TRUE;
					DoCenter (dLat, dLon, bRedraw);
				} 
			}
			else
			{
				AfxMessageBox (IDS_NO_MAP_NORTH);
			}
		}
		break;
	case VK_DOWN:				  // 1474		   1476		- 4
		if (dMaxVertScroll<0 || pScrollLP.y >= dMaxVertScroll-EPS)
		{
			GetBottomScreenLatLon(&dLat, &dLon);

			if (ptScMpDoc->GetIncreaseCnt() != 0)
			{
				ptScMpDoc->SetIncreaseCnt(0);
				ptScMpDoc->CalcActFakt (m_sOrigImage, &m_sizeTotal);
			}

			if (ptScMpDoc->NextMapSouth(&m_sizeTotal, dLon))
			{
				m_bOnDrawScroll=TRUE;	// activate scrolling after update
				m_bHorz = FALSE;		// move vert bar left
				m_bMax = FALSE;			// call OnInitialUpdate

				if (m_bOnDrawScroll)
				{					// initiating view update here!
					this->GetAbsoluteBmpSizeLP(&m_sizeTotal);   // checks calibration

					SetScrollSizes(MM_LOMETRIC, m_sizeTotal);
					BOOL bRedraw = TRUE;
					DoCenter (dLat, dLon, bRedraw);
				} 
			}
			else
			{
				AfxMessageBox (IDS_NO_MAP_SOUTH);
			}
		}
		break;
	}

if (m_bOnDrawScroll)
		m_bOnDrawScroll=FALSE;		  // don't do KeyDown after new update
else    CTextViewTool::OnKeyDown(nChar, nRepCnt, nFlags);
}

/************************************************************************
 *  ScMpView.cpp   		 		D o C e n t e r				 			*
 ************************************************************************/
BOOL CScanMapView::DoCenter (double dLat, double dLon, BOOL bRedraw)
{
BOOL bMapFound=FALSE;

CPoint pScrollLP;
pScrollLP = this->GetScrollPosition();			// (220, 0)
CSize sMapLP = m_sizeTotal;								// (3028, 4448)
														// same as GetTotalSize

CRect rClientLP;					// get size of client area
this->GetClientRectLP(rClientLP);
if (!bMMText)
	{	 
	pScrollLP.y			= -pScrollLP.y;
	rClientLP.top		= -rClientLP.top;
	rClientLP.bottom	= -rClientLP.bottom;
	}

short EPS = 4;
int	dMaxHorzScroll = sMapLP.cx - rClientLP.Width();		// 220
int dMaxVertScroll = sMapLP.cy - rClientLP.Height();	// 3092


CScannedMap* ptMap = ptScMpDoc->GetMapPtr (ptScMpDoc->GetActMapIndex ());
if (ptMap != NULL)
	{
	long lMapLPx, lMapLPy;			// act pos on map in LP koords
	ptMap->LatLonToLP (dLon, dLat, &lMapLPx, &lMapLPy);
	CPoint PosLP(lMapLPx, lMapLPy);
	if (!bMMText)
		PosLP.y			= -PosLP.y;		// make always positive!!

	CPoint pNewScrollLP;				// get optimal scroll values
	pNewScrollLP.x = PosLP.x - (long)(0.5 * rClientLP.Width());
	pNewScrollLP.y = PosLP.y - (long)(0.5 * rClientLP.Height());


	if (pNewScrollLP.x < 0) pNewScrollLP.x = 0;
	if (pNewScrollLP.y < 0) pNewScrollLP.y = 0;
			
	if (pNewScrollLP.x > dMaxHorzScroll) pNewScrollLP.x = dMaxHorzScroll;
	if (pNewScrollLP.y > dMaxVertScroll) pNewScrollLP.y = dMaxVertScroll;
	
						// set new scroll here...	
	SetScrollSizes(MM_LOMETRIC, m_sizeTotal);	// needs absolute LP koords
	this->ScrollTo (pNewScrollLP);		// scroll without drawing


	ptInit->SetLatLonToCenter(dLat, dLon);
	m_ptInterMap->SetLatLon(dLat, dLon);	// show act pos in status bar
	m_ptInterMap->StartPosChangedTimer(1);	// 1/10 sec

	if (bRedraw)
		this->Invalidate(FALSE);		// redraw without erasing screen
	else
		this->InvalidateRect(NULL);		// invalidate only
	
	bMapFound = TRUE;
	}

return bMapFound;
}

