/************************************************************************
 *  						S c a n M a p . c p p	  					*
 ************************************************************************/
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include <afxpriv.h>	// for CPreviewDC class

#include "pf.h"
#include "DimDoc.h"    
#include "math.h"							// for: fabs
#include "Location.h"
#include "ScanMap.h"

#include "..\CPPTOOLS\Vektor.h"                                       
#include "Symbols.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

extern CDimDoc*		ptDim;

/////////////////////////////////////////////////////////////////////////////
// CScannedMap
IMPLEMENT_SERIAL(CScannedMap, CObject, 0 /* schema number*/ )

#define pi 3.1415926

/************************************************************************
 *  ScanMap.cpp				C S c a n M a p					Constructor	*
 ************************************************************************/
CScannedMap::CScannedMap()
{ 
	this->Reset();
}

/************************************************************************
 *  ScanMap.cpp				C S c a n n e d M a p			Constructor	*
 ************************************************************************/
CScannedMap::CScannedMap(CString szPath)
{ 
	this->Reset();
	m_szPath = szPath;
}

/************************************************************************
 *  ScanMap.cpp				C S c a n n e d M a p			Constructor	*
 ************************************************************************/
CScannedMap::CScannedMap(CScannedMap& Source)
{ 
*this = Source;				// uses operator=
}

/************************************************************************
 *  ScanMap.cpp				~ C S c a n n e d M a p			Destructor	*
 ************************************************************************/
CScannedMap::~CScannedMap()
{
DeleteArrayOf (&m_FixPts);
} 

/************************************************************************
 *  ScanMap.cpp					operator=								*
 ************************************************************************/
const CScannedMap& CScannedMap::operator=(const CScannedMap& Source)
{
short i;
DeleteArrayOf (&m_FixPts);


for (i=0; i< Source.m_FixPts.GetSize(); i++)
	{
	CMapFixPt* ptFixPt = (CMapFixPt*)Source.m_FixPts.GetAt (i);  
	if (ptFixPt != NULL)
		{
		CMapFixPt* ptCopiedFix = new CMapFixPt (*ptFixPt);
		if (ptCopiedFix != NULL)
			m_FixPts.Add (ptCopiedFix);       	// add new Element   
		}
	} 
	     	
m_vKegelPol = Source.m_vKegelPol;
m_bCalibrated = Source.m_bCalibrated;
memcpy (m_ptShowFlags, Source.m_ptShowFlags, FLAG_CNT*sizeof (BOOL));


m_nSizeLine = Source.m_nSizeLine;
m_nSizeBear = Source.m_nSizeBear;;

m_ColLine	= Source.m_ColLine;
m_ColBear	= Source.m_ColBear; 
m_ColLabel	= Source.m_ColLabel;
m_ColSel	= Source.m_ColSel;

m_ColText	= Source.m_ColText;

m_szName = Source.m_szName;

for (i=0; i<4; i++)
	m_Border[i] = Source.m_Border[i];

m_bAntipod = Source.m_bAntipod;
m_bAutoSel = Source.m_bAutoSel;
m_Type = Source.m_Type;
m_szPath = Source.m_szPath;			// path to *.bmp file

m_lSizeX_DP = Source.m_lSizeX_DP;
m_lSizeY_DP = Source.m_lSizeY_DP;

m_lBmpSizeX = Source.m_lBmpSizeX;
m_lBmpSizeY = Source.m_lBmpSizeY;
m_nActCalIndex = Source.m_nActCalIndex;

m_dFaktX	= Source.m_dFaktX;
m_dFaktY	= Source.m_dFaktY;
m_pStart.x	= Source.m_pStart.x;
m_pStart.y	= Source.m_pStart.y;

m_bRelativ	= Source.m_bRelativ;
m_nCourse	= Source.m_nCourse;				// relative definition of point 2
m_dDist		= Source.m_dDist;
m_nDistDim	= Source.m_nDistDim;

return *this;
}

/************************************************************************
 *	ScanMap.cpp						R e s e t	 						*
 ************************************************************************/										 
void CScannedMap::Reset ()
{
DeleteArrayOf (&m_FixPts);

m_bCalibrated = FALSE;
memset (m_ptShowFlags, 0, FLAG_CNT*sizeof (BOOL));

m_nSizeLine = 10;
m_nSizeBear = 5;

m_ColLine= RGB (255, 255,0);
m_ColBear = RGB (255, 0, 0); 
m_ColLabel = RGB (255, 255,0);
m_ColSel = RGB (0, 255, 0);

m_ColText = RGB (0,0,0);
	 
m_szName.Empty();

m_bAntipod = FALSE;
m_bAutoSel = TRUE;
m_Type = LAMBERT;
m_szPath.Empty();			// path to *.bmp file
m_lSizeX_DP	= 0;
m_lSizeY_DP = 0;
m_lBmpSizeX = 0;
m_lBmpSizeY = 0;
m_nActCalIndex = -1;

m_dFaktX	= 1.;
m_dFaktY	= 1.;
m_pStart.x	= 0;
m_pStart.y	= 0;

m_bRelativ = FALSE;			// LINEAR Projection:
m_nCourse = 135;			// relative definition of point 2
m_dDist = 5;
m_nDistDim = DIM_NM;
}


/************************************************************************
 *	ScanMap.cpp					A c t i v a t e 						*
 ************************************************************************/										 
void CScannedMap::Activate (short nTypeID, BOOL bActivate)
{
*(m_ptShowFlags + nTypeID) = bActivate;
}

/************************************************************************
 *	ScanMap.cpp					I s A c t i v a t e d					*
 ************************************************************************/										 
BOOL CScannedMap::IsActivated (short nTypeID)
{
return *(m_ptShowFlags + nTypeID);
}

/************************************************************************
 *	ScanMap.cpp			I n v e r t A c t i v a t e S t a t e			*
 ************************************************************************/										 
void CScannedMap::InvertActivateState (short nTypeID)
{
*(m_ptShowFlags + nTypeID) = !(*(m_ptShowFlags + nTypeID));
}

/************************************************************************
 *  ScanMap.cpp			F i x P t I n d e x T o C a l i b P o s			*
 ************************************************************************/
CALIBRATEPOS CScannedMap::FixPtIndexToCalibPos(short nIndex)
{
CALIBRATEPOS CalPos;

switch (m_Type)
	{
	case LAMBERT:
		switch(nIndex)
		{
		case 0:	CalPos = CALPOS_TL; break;
		case 1:	CalPos = CALPOS_TR; break;
		case 2:	CalPos = CALPOS_BR; break;
		case 3:	CalPos = CALPOS_BL; break;
		}
		break;

	case LINEAR:
		switch(nIndex)
		{
		case 0:	CalPos = CALPOS_TL; break;
		case 1:	CalPos = CALPOS_BR; break;
		}
		break;

	case MERCATOR:
		CalPos=CALPOS_TL;		// not implemented
		break;

	case POLAR:
		CalPos=CALPOS_TL;		// not implemented
		break;
	}
	return CalPos;
}

/************************************************************************
 *  ScanMap.cpp			C a l i b P o s T o F i x P t I n d e x			*
 ************************************************************************/
short CScannedMap::CalibPosToFixPtIndex (CALIBRATEPOS CalPos)
{
	short nIndex = -1;

	switch (m_Type)
	{
	case LAMBERT:
		switch(CalPos)
		{
		case CALPOS_TL: nIndex = 0; break;
		case CALPOS_TR: nIndex = 1; break;
		case CALPOS_BR: nIndex = 2; break;
		case CALPOS_BL: nIndex = 3; break;
		}
		break;

	case LINEAR:
		switch(CalPos)
		{
		case CALPOS_TL: nIndex = 0; break;
		case CALPOS_BR: nIndex = 1; break;
		}
		break;

	case MERCATOR:
		nIndex = 0;		// not implemented
		break;

	case POLAR:
		nIndex = 0;		// not implemented
		break;
	}

	return nIndex;
}

/************************************************************************
 *  ScanMap.cpp				D e l e t e A r r a y O f 					*
 ************************************************************************/
void CScannedMap::DeleteArrayOf (CObArray* ptArray)
{
int i, nEntryCnt;	
											
nEntryCnt = ptArray->GetSize();
for (i=0; i<nEntryCnt; i++)
	{
	CObject* ptEntry;
	if ((ptEntry = ptArray->GetAt(i)) != NULL)
		{
		delete ptEntry;				// delete original element
		}
	}
ptArray->RemoveAll();
}

/****************************************************************************
 *	ScanMap.cpp				G e t L a m b e r t S c a l e					*
 ****************************************************************************/										 
long CScannedMap::GetLambertScale (long lHorzResLPI)
{
long lScale, dPixX;
double fDist, dLon;

CMapFixPt* ptBR = this->GetFixPtPtr (2);	 // bottom right  
CMapFixPt* ptBL = this->GetFixPtPtr (3);	 // bottom left   


dPixX = ptBR->GetLP_X() - ptBL->GetLP_X();		// visible X-pixels
dLon  = ptBR->GetLon() - ptBL->GetLon();		// visible lon degrees
fDist = dLon * 60;								// visible horizontal distance

lScale = (long)(fDist * 185000 * lHorzResLPI / (dPixX * 2.54));
return lScale;
}

/****************************************************************************
 *	ScanMap.cpp					G e t L i n e a r S c a l e					*
 ****************************************************************************/										 
long CScannedMap::GetLinearScale (long lHorzResLPI)
{
	long lScale;

							// get lat lon of map
	DRECT rLatLon = this->GetLatLonBorder();
	CLatLon llBL(rLatLon.bottom, rLatLon.left);		// use ptFP1
	CLatLon llBR(rLatLon.bottom, rLatLon.right);

								// get pixel
	long lLP1x, lLP1y;
	this->LatLonToLP (llBL.GetLon(), llBL.GetLat(), &lLP1x, &lLP1y);

	long lLP2x, lLP2y;
	this->LatLonToLP (llBR.GetLon(), llBR.GetLat(), &lLP2x, &lLP2y);

	long dPixX = labs(lLP2x - lLP1x);

	double dWidth_NM, dCourse;
	dWidth_NM = llBR.LoxoDist (llBL, &dCourse);		

	lScale = (long)(dWidth_NM * 185000 * lHorzResLPI / (dPixX * 2.54));

	return lScale;
}

/****************************************************************************
 *	ScanMap.cpp				G e t M e r c a t o r S c a l e					*
 ****************************************************************************/										 
long CScannedMap::GetMercatorScale (long lHorzResLPI)
{
long lScale, dPixX;
double fDist, mLat, dLon;

CMapFixPt* ptTL = this->GetFixPtPtr (0);	 // top left  
CMapFixPt* ptBR = this->GetFixPtPtr (2);	 // bottom right  
CMapFixPt* ptBL = this->GetFixPtPtr (3);	 // bottom left   


dPixX = ptBR->GetLP_X() - ptBL->GetLP_X();		// visible X-pixels
dLon  = ptBR->GetLon() - ptBL->GetLon();		// visible lon degrees
mLat  = (ptTL->GetLat() + ptBL->GetLat())/2;	// average lat degree
fDist = dLon * 60 * cos (mLat*pi/180);			// visible horizontal distance

lScale = (long)(fDist * 185000 * lHorzResLPI / (dPixX * 2.54));
return lScale;
}

/****************************************************************************
 *	ScanMap.cpp					G e t P o l a r S c a l e					*
 ****************************************************************************/										 
long CScannedMap::GetPolarScale (long lHorzResLPI)
{
long lScale=1;
return lScale;
}

/****************************************************************************
 *	ScanMap.cpp						G e t S c a l e							*
 ****************************************************************************/										 
long CScannedMap::GetScale (long lHorzResLPI)	
{
long lScale;

switch (m_Type)
	{
	case LAMBERT:
		lScale = this->GetLambertScale (lHorzResLPI);
		break;

	case LINEAR:
		lScale = this->GetLinearScale (lHorzResLPI);
		break;

	case MERCATOR:
		lScale = this->GetMercatorScale (lHorzResLPI);
		break;

	case POLAR:
		lScale = this->GetPolarScale (lHorzResLPI);
		break;
	}
return lScale;
}


/************************************************************************
 *  ScanMap.cpp						C l i p								*
 ************************************************************************/
void CScannedMap::Clip(CDC* pDC, CRect rClipDP)
{
if (pDC->IsKindOf (RUNTIME_CLASS(CPreviewDC)))
	{							// adjust clip rect for preview
	CPreviewDC* pPrevDC = (CPreviewDC*)pDC;

	pPrevDC->PrinterDPtoScreenDP (&rClipDP.TopLeft());
	pPrevDC->PrinterDPtoScreenDP (&rClipDP.BottomRight());

					// now offset the result by teh viewport
					// of the print preview window:
	CPoint ptOrg;
	::GetViewportOrgEx (pDC->m_hDC, &ptOrg);
	rClipDP += ptOrg;
	}


CRgn rClip;			// create the clip region and select it:
rClip.CreateRectRgn(rClipDP.left, rClipDP.top,
					rClipDP.right,rClipDP.bottom);
pDC->SelectClipRgn (&rClip);
}

/************************************************************************
 *  ScanMap.cpp				G e t L P R e c t							*
 ************************************************************************/
CRect CScannedMap::GetLPRect ()
{
CRect rRect(0, 0, (long)(m_dFaktX * m_lBmpSizeX), -(long)(m_dFaktY * m_lBmpSizeY));

return rRect;
}

/************************************************************************
 *  ScanMap.cpp					S e t S i z e D P						*
 ************************************************************************/
void CScannedMap::SetSizeDP (CSize SizeDP)
{
m_lSizeX_DP = SizeDP.cx;
m_lSizeY_DP = SizeDP.cy;
}

/************************************************************************
 *  ScanMap.cpp					G e t S i z e D P						*
 ************************************************************************/
void CScannedMap::GetSizeDP (long* ptSizeX, long* ptSizeY)
{
*ptSizeX = m_lSizeX_DP;
*ptSizeY = m_lSizeY_DP;
}

/************************************************************************
 *  ScanMap.cpp					S e t S i z e L P						*
 ************************************************************************/
BOOL CScannedMap::SetSizeLP (long lSizeX, long lSizeY)
{
BOOL bChanged;
bChanged = this->CheckCalibration(lSizeX, lSizeY);

m_lBmpSizeX = lSizeX;
m_lBmpSizeY = lSizeY;
return bChanged;
}

/************************************************************************
 *  ScanMap.cpp					G e t S i z e L P						*
 ************************************************************************/
void CScannedMap::GetSizeLP (long* ptSizeX, long* ptSizeY)
{
*ptSizeX = m_lBmpSizeX;
*ptSizeY = m_lBmpSizeY;
}

/************************************************************************
 *  ScanMap.cpp			G e t S c a l e d S i z e L P					*
 ************************************************************************/
void CScannedMap::GetScaledSizeLP (long* ptSizeX, long* ptSizeY)
{
*ptSizeX = (long)(m_dFaktX * m_lBmpSizeX);
*ptSizeY = (long)(m_dFaktY * m_lBmpSizeY);
}

/************************************************************************
 *  ScanMap.cpp					G e t T i t l e							*
 ************************************************************************/
CString CScannedMap::GetTitle()
{
short nSlash = m_szPath.ReverseFind ('\\');
CString szTitle = m_szPath.Right(m_szPath.GetLength() - nSlash - 1);
return szTitle;
}

/************************************************************************
 *  ScanMap.cpp				C a l i b r a t e L a m b e r t				*
 ************************************************************************/
BOOL CScannedMap::CalibrateLambert ()
{
m_bCalibrated = FALSE;

CMapFixPt* ptTL; 
CMapFixPt* ptTR;
CMapFixPt* ptBR; 
CMapFixPt* ptBL; 

CPoint FixLP;
CVektor vR_left, vR_right, vE, vSP;

if (this->GetFixPtCnt () == 4)
	{
	ptTL = this->GetFixPtPtr (0);
	ptTR = this->GetFixPtPtr (1);
	ptBR = this->GetFixPtPtr (2);
	ptBL = this->GetFixPtPtr (3);

	FixLP = ptTL->GetLP();
	CVektor vTL(FixLP.x, FixLP.y);

	FixLP = ptTR->GetLP();
	CVektor vTR(FixLP.x, FixLP.y);

	FixLP = ptBL->GetLP();
	CVektor vBL(FixLP.x, FixLP.y);

	FixLP = ptBR->GetLP();
	CVektor vBR(FixLP.x, FixLP.y);

	vR_left = vBL - vTL;
	vR_right= vBR - vTR;		

	vE.Set (vTR.x - vTL.x, vTR.y - vTL.y);
	if (vE.LGS2 (vR_left, -vR_right))
		{
		double dLat, dLon;

		m_vKegelPol = vTL + vR_left * vE.x;
		//vS2 = vTR + vR_right* vE.y;
		m_bCalibrated = TRUE;

											// calculate max visible lat lon values
		this->GetLambertLatLon (0, 0, &dLat, &dLon); 
		m_Border[0].SetLatLon (dLat, dLon);			// top left
		this->GetLambertLatLon (m_lBmpSizeX, 0, &dLat, &dLon); 
		m_Border[1].SetLatLon (dLat, dLon);			// top right
		this->GetLambertLatLon (m_lBmpSizeX, -m_lBmpSizeY, &dLat, &dLon); 
		m_Border[2].SetLatLon (dLat, dLon);			// bottom right
		this->GetLambertLatLon (0, -m_lBmpSizeY, &dLat, &dLon); 
		m_Border[3].SetLatLon (dLat, dLon);			// bottom left
		}
	}
return m_bCalibrated;
}


/************************************************************************
 *  ScanMap.cpp				G e t B o r d e r L a t L o n 				*
 ************************************************************************/
CLatLon CScannedMap::GetBorderLatLon (short i)
{
return m_Border[i];
}

/************************************************************************
 *  ScanMap.cpp				G e t L a t L o n B o r d e r				*
 ************************************************************************/
DRECT CScannedMap::GetLatLonBorder ()				   // Index and position:
{													   // 0    1
DRECT rBorder;										   // 3    2

									// use top left or top right
rBorder.top = max (m_Border[0].GetLat(), m_Border[1].GetLat());	

									// use bottom right or bottom left
rBorder.bottom = min (m_Border[2].GetLat(), m_Border[3].GetLat());

									// use top left or bottom left
rBorder.left = min (m_Border[0].GetLon(), m_Border[3].GetLon());

									// use	top right or bottom right		
rBorder.right = max (m_Border[1].GetLon(), m_Border[2].GetLon());	

return rBorder;
}

/************************************************************************
 *  ScanMap.cpp				I s N e a r B o r d e r						*
 *  Purpose: Returns TRUE, if dLat/dLon is 5 mm before border of map	*
 ************************************************************************/
BOOL CScannedMap::IsNearBorder(double dLat, double dLon, int nChangeMapLimit_mm)
{
BOOL bNearBorder;
long lEPS = 10 * nChangeMapLimit_mm;	// default: 50 1/10 mm
long lSizeX, lSizeY;
CRect rMapLP;
											// get logical area of map
this->GetScaledSizeLP (&lSizeX, &lSizeY);
rMapLP.SetRect (0,0, lSizeX, -lSizeY);					
										
long lLPx, lLPy;
this->LatLonToLP (dLon, dLat, &lLPx, &lLPy);
					
bNearBorder = !(((rMapLP.left + lEPS) <= lLPx && lLPx <= (rMapLP.right - lEPS)) &&
			    ((rMapLP.top  - lEPS) >= lLPy && lLPy >= (rMapLP.bottom + lEPS)));


return bNearBorder;
}

/************************************************************************
 *  ScanMap.cpp				G e t F r e e F a k t						*
 *	Input:  lLPx0, lLPy0: logical coords of point[N-x-1]	(x > 0)		*
 *			lLPx1, lLPy1: logical coords of point[N-1]					*
 *	Output:	dFreeFakt = 1 (best) ... dFreeFakt = 0 (worst case)			*
 ************************************************************************/
double CScannedMap::GetFreeFakt (long lLPx0, long lLPy0, long lLPx1, long lLPy1)
{
	double dFreeFakt = 0;
	double dFaktX, dFaktY;

	long lRichtX = lLPx1 - lLPx0;		// get direction of movement
	long lRichtY = lLPy1 - lLPy0;

										// get logical coords of map border
	long lLPleft, lLPtop, lLPright, lLPbottom, lLPwidth, lLPheight;

	CRect rRectLP = this->GetLPRect ();
	lLPleft		= rRectLP.left;
	lLPright	= rRectLP.right;
	lLPtop		= rRectLP.top;
	lLPbottom	= rRectLP.bottom;


lLPwidth = lLPright  - lLPleft;		// get logical widht and height of map
lLPheight= lLPbottom - lLPtop;


long lStepX, lStepY;

if (lRichtX < 0)
	{			// moving west
	lStepX = (lLPx1 - lLPleft) / lRichtX;
	dFaktX = (double)lStepX / ((double)lLPwidth/ lRichtX);
	}
else{
	if (lRichtX > 0)
		{		// moving east
		lStepX = (lLPright - lLPx1) / lRichtX;
		dFaktX = (double)lStepX / ((double)lLPwidth/ lRichtX);
		}
	else{				// no horizontal movement
		dFaktX = 1;		// no space limit
		}
	}

if (lRichtY > 0)
	{			// moving north
	lStepY = (lLPy1 - lLPtop)/lRichtY;
	dFaktY = (double)lStepY / ((double)lLPheight / lRichtY);
	}
else{
	if (lRichtY < 0)
		{		// moving south
		lStepY = (lLPbottom - lLPy1)/lRichtY;
		dFaktY = (double)lStepY / ((double)lLPheight / lRichtY);
		}
	else{				// no vertical movement
		dFaktY = 1;		// no space limit
		}
	}

if (dFaktX < 0)
	{
	dFaktX *= -1;
	AfxMessageBox ("CScannedMap::GetFreeFakt dFaktX < 0)");
	}
	
if (dFaktY < 0)
	{
	dFaktY *= -1;
	AfxMessageBox ("CScannedMap::GetFreeFakt dFaktY < 0)");	
	}
if (dFaktX > 1)
	{
	dFaktX = 1;
	AfxMessageBox ("CScannedMap::GetFreeFakt dFaktX > 1)");
	}
	
if (dFaktY > 1)
	{
	dFaktY = 1;
	AfxMessageBox ("CScannedMap::GetFreeFakt dFaktY > 1)");	
	}
	
	
if (dFaktX < dFaktY)	dFreeFakt = dFaktX;
		else			dFreeFakt = dFaktY;
		
return dFreeFakt;
}

/************************************************************************
 *  ScanMap.cpp				G e t C e n t e r F a k t					*
 *	Input:  lLPx, lLPy: logical coords of point to center				*
 *	Output:	dCenterFakt = 1 (best) ... dCenterFakt = 0 (worst case)		*
 ************************************************************************/
double CScannedMap::GetCenterFakt (long lLPx, long lLPy)
{
	double dCenterFakt = 0;
	double dFaktX, dFaktY;

										// get logical coords of map border
	long lLPleft, lLPtop, lLPright, lLPbottom, lLPwidth, lLPheight, lLPcenterX, lLPcenterY;

	CRect rRectLP = this->GetLPRect ();
	lLPleft		= rRectLP.left;
	lLPright	= rRectLP.right;
	lLPtop		= rRectLP.top;
	lLPbottom	= rRectLP.bottom;


	if ((lLPleft<= lLPx && lLPx <= lLPright) &&
		(lLPtop >= lLPy && lLPy >= lLPbottom))
	{			// act point visible on this map

		lLPwidth = lLPright  - lLPleft;		// get logical width and height of map
		lLPheight= lLPbottom - lLPtop;

		lLPcenterX = lLPleft + (lLPwidth / 2);	// logical center of map
		lLPcenterY = lLPtop + (lLPheight / 2);

		dFaktX = 1 - 2 * (double)labs(lLPx - lLPcenterX) / lLPwidth;
		dFaktY = 1 - 2 * (double)labs(lLPy - lLPcenterY) / labs(lLPheight);


		if (dFaktX < 0)		// lLPx on left border of map
		{
			dFaktX = 0;
		}

		if (dFaktY < 0)		// lLPy on bottom border of map
		{
			dFaktY = 0;
		}

		if (dFaktX > 1)		// rounding error with perfect centered lLPx
		{
			dFaktX = 1;
		}
		
		if (dFaktY > 1)		// rounding error with perfect centered lLPy
		{
			dFaktY = 1;
		}


	}
	else
	{			// act point outside map
		dFaktX = dFaktY = 0;
	}

	
if (dFaktX < dFaktY)	dCenterFakt = dFaktX;
		else			dCenterFakt = dFaktY;
		
return dCenterFakt;
}

/************************************************************************
 *  ScanMap.cpp				G e t C e n t e r L a t L o n				*
 ************************************************************************/
CLatLon CScannedMap::GetCenterLatLon()
{
CLatLon llCenter;
long	lX_LP, lY_LP;
double	dLat, dLon;

lX_LP =  (long)(m_dFaktX*m_lBmpSizeX)/2;
lY_LP = -(long)(m_dFaktY*m_lBmpSizeY)/2;

this->LPtoLatLon (lX_LP, lY_LP, &dLat, &dLon);

llCenter.SetLatLon (dLat, dLon);
return llCenter;
}

/************************************************************************
 *  ScanMap.cpp					GetDiagonalDist_NM						*
 ************************************************************************/
double CScannedMap::GetDiagonalDist_NM()
{
	double dDist_NM, wab;
	CVektor A, B;

	DRECT rLatLon = this->GetLatLonBorder();

	CLatLon TL(rLatLon.top, rLatLon.left);
	CLatLon BR(rLatLon.bottom, rLatLon.right);

	TL.ToXYZ(&A);					/* Koordinaten von Top Left						*/
	BR.ToXYZ(&B);					/* Koordinaten von Bottom Right					*/
	wab = A.VektorAngle (B);		/* Winkel zwischen Kartenmittelpunkt und B		*/		
									/* get distance between TL and BR of map		*/
	dDist_NM = 60 * (wab * 180./pi);	/* Dist von A nach B in NM					*/

	return dDist_NM;
}

/****************************************************************************
 *	ScanMap.cpp				G e t W i d t h _ N M							*
 ****************************************************************************/
double CScannedMap::GetWidth_NM (CLatLon& llReference)
{
long	lX_LP, lY_LP;
double dWidth_NM, dLat,dLon, dCourse;

this->LatLonToLP (llReference.GetLon(), llReference.GetLat(), 
					&lX_LP, &lY_LP);

this->LPtoLatLon (0, lY_LP, &dLat, &dLon);
CLatLon llLeft(dLat, dLon);

this->LPtoLatLon ((long)(m_dFaktX * m_lBmpSizeX), lY_LP, &dLat, &dLon);
CLatLon llRight(dLat, dLon);

dWidth_NM = llRight.LoxoDist (llLeft, &dCourse);

return dWidth_NM;
}

/****************************************************************************
 *	ScanMap.cpp					S e t S c a l e F a k t	X 					*
 ****************************************************************************/
BOOL CScannedMap::SetScaleFaktX (double dFaktX) 
{
BOOL bChanged = (fabs (m_dFaktX - dFaktX) > 0.001); 
m_dFaktX	 = dFaktX;
return bChanged;
}

/****************************************************************************
 *	ScanMap.cpp					S e t S c a l e F a k t	Y 					*
 ****************************************************************************/
BOOL CScannedMap::SetScaleFaktY (double dFaktY) 
{
BOOL bChanged = (fabs (m_dFaktY - dFaktY) > 0.001); 
m_dFaktY	 = dFaktY;
return bChanged;
}

/****************************************************************************
 *	ScanMap.cpp					D r a w L o c a t i o n						*
 ****************************************************************************/
void CScannedMap::DrawLocation (CLocation& Loc, CSymbols& Symbols)
{
double dLat = Loc.GetLat();
double dLon = Loc.GetLon();
double dMapLon;

if (IsLatLonInRect (dLat, dLon, &dMapLon))
	{  
	long	LPx0, LPy0;
	CString	szName;
	                 
	this->LatLonToLP(dMapLon, dLat, &LPx0, &LPy0);

	if (Symbols.Quality() < VQ_MEDIUM)
		{
		szName = Loc.GetIndicator();
		if (szName.GetLength()==0)
			szName = Loc.GetName();
		}
	else{
		szName = Loc.GetName();
		}

	short nType = Loc.GetCategory ();
	short nVar  = Loc.GetVariation ();	
						   
	switch (nType)
		{
		case WP_AIRPORT:
			if (this->IsActivated (SHOW_APT))
				Symbols.APT_Draw (LPx0, LPy0, szName, nVar, Loc.GetRwyDir(), Loc.IsGras()); 
			break;
		case WP_REPORT:
			if (this->IsActivated (SHOW_REP))
				Symbols.REP_Draw (LPx0, LPy0, szName); 
			break;
		case WP_INTERSECTION:
			if (this->IsActivated (SHOW_INT))
				Symbols.INT_Draw (LPx0, LPy0, Loc.GetIndicator()); 
			break;
		case WP_NDB:
			if (this->IsActivated (SHOW_NDB))
				Symbols.NDB_Draw (LPx0, LPy0, Loc.GetIndicator()); 
			break;
		case WP_VOR:
			if (this->IsActivated (SHOW_VOR))
				Symbols.VOR_Draw (LPx0, LPy0, szName, Loc.GetIndicator(), nVar); 
			break;
		case WP_USER:
			if (this->IsActivated (SHOW_USR))
				Symbols.USR_Draw (LPx0, LPy0, szName); 
			break;
		case WP_SAIL:
			if (this->IsActivated (SHOW_SAIL))
				Symbols.SAIL_Draw (LPx0, LPy0, szName, Loc.IsGras()); 
			break;
		case WP_HELI:
			if (this->IsActivated (SHOW_HELI))
				Symbols.HELI_Draw (LPx0, LPy0, szName, Loc.IsGras()); 
			break;
		}
	}          
}

/************************************************************************
 *  ScanMap.cpp				C a l i b r a t e L i n e a r				*
 ************************************************************************/
BOOL CScannedMap::CalibrateLinear ()
{
	m_bCalibrated = FALSE;

	CMapFixPt* ptFP1; 
	CMapFixPt* ptFP2;
	CPoint FixLP1, FixLP2;

	if (this->GetFixPtCnt () == 2)
	{
		ptFP1 = this->GetFixPtPtr (0);
		ptFP2 = this->GetFixPtPtr (1);

		FixLP1 = ptFP1->GetLP();
		FixLP2 = ptFP2->GetLP();

		long dX = labs(FixLP2.x - FixLP1.x);
		long dY = labs(FixLP2.y - FixLP1.y);

		if (dX != 0 && dY != 0)
		{
			BOOL bSpreadX = (m_lBmpSizeX/dX < 4);
			BOOL bSpreadY = (m_lBmpSizeY/dY < 4);

			if (bSpreadX && bSpreadY)
			{			// calibration area covers more than 1/4 of image
				double dQuality = 0;
				dQuality = (dX < dY)? (double)dX/dY : (double)dY/dX;
				m_bCalibrated = ( 1 >= dQuality && dQuality > 0.1);
			}
		}

		if (m_bCalibrated)
		{
			double dLat, dLon;

											// calculate max visible lat lon values
			this->GetLinearLatLon (0, 0, &dLat, &dLon); 
			m_Border[0].SetLatLon (dLat, dLon);			// top left
			this->GetLinearLatLon (m_lBmpSizeX, 0, &dLat, &dLon); 
			m_Border[1].SetLatLon (dLat, dLon);			// top right
			this->GetLinearLatLon (m_lBmpSizeX, -m_lBmpSizeY, &dLat, &dLon); 
			m_Border[2].SetLatLon (dLat, dLon);			// bottom right
			this->GetLinearLatLon (0, -m_lBmpSizeY, &dLat, &dLon); 
			m_Border[3].SetLatLon (dLat, dLon);			// bottom left
		}

	}

return m_bCalibrated;
}

/************************************************************************
 *  ScanMap.cpp				C a l i b r a t e M e r c a t o r			*
 ************************************************************************/
BOOL CScannedMap::CalibrateMercator ()
{
m_bCalibrated = FALSE;

return m_bCalibrated;
}

/************************************************************************
 *  ScanMap.cpp				C a l i b r a t e P o l a r					*
 ************************************************************************/
BOOL CScannedMap::CalibratePolar ()
{
m_bCalibrated = FALSE;

return m_bCalibrated;
}

/************************************************************************
 *  ScanMap.cpp				 C a l i b r a t e							*
 ************************************************************************/
 BOOL CScannedMap::Calibrate () 
 {
 switch (m_Type)
	{
	case LAMBERT:
		m_bCalibrated = this->CalibrateLambert();
		break;

	case LINEAR:
		m_bCalibrated = this->CalibrateLinear();
		break;

	case MERCATOR:
		m_bCalibrated = this->CalibrateMercator();
		break;

	case POLAR:
		m_bCalibrated = this->CalibratePolar();
		break;
	}

return m_bCalibrated;
}


/************************************************************************
 *  ScanMap.cpp				C h e c k C a l i b r a t i o n				*
 *  Purpose: Adjust Fixpoints and PolVektor for changed logical size	*
 ************************************************************************/
BOOL CScannedMap::CheckCalibration (long lSizeX, long lSizeY)
{
BOOL bChanged = FALSE;
						
if (m_bCalibrated)
	{							// map was already calibrated			
	if (m_lBmpSizeX != lSizeX || m_lBmpSizeY != lSizeY)
		{						// and there is a new bmp size
		short i;
		double dFaktX = (double)lSizeX / m_lBmpSizeX;
		double dFaktY = (double)lSizeY / m_lBmpSizeY;

		for (i=0; i<m_FixPts.GetSize(); i++)
			{					// convert fix points
			CMapFixPt* ptFix = NULL;

			if ((ptFix = (CMapFixPt*)m_FixPts.GetAt(i)) != NULL)
				{
				long lX_LP = ptFix->GetLP_X();
				long lY_LP = ptFix->GetLP_Y();
		
				lX_LP = (long)(dFaktX * lX_LP +0.5);

				if (lY_LP > 0)
					{		
					lY_LP = (long)(dFaktY * lY_LP +0.5);
					}
				else{
					lY_LP = -lY_LP;
					lY_LP = (long)(dFaktY * lY_LP +0.5);
					lY_LP = -lY_LP;
					}

				ptFix->SetLP_X (lX_LP);
				ptFix->SetLP_Y (lY_LP);
				}
			}

								// set new bmp size
		m_lBmpSizeX = lSizeX;
		m_lBmpSizeY = lSizeY;

								// calibrate with new size
		bChanged = this->Calibrate();
		}
	}
return bChanged;
}

/************************************************************************
 *  ScanMap.cpp				G e t L a m b e r t L a t L o n				*
 ************************************************************************/
void CScannedMap::GetLambertLatLon (long lX_LP, long lY_LP, double* ptLat, double* ptLon) 
{
double dLat=0.;
double dLon=0.;


CPoint FixLP;
CVektor vR_left, vR_right, vE, vSP;

if (m_bCalibrated)
	{
	CMapFixPt* ptTL = this->GetFixPtPtr (0);
	CMapFixPt* ptTR = this->GetFixPtPtr (1);
	CMapFixPt* ptBR = this->GetFixPtPtr (2);
	CMapFixPt* ptBL = this->GetFixPtPtr (3);

	FixLP = ptTL->GetLP();
	CVektor vTL(FixLP.x, FixLP.y);

	FixLP = ptTR->GetLP();
	CVektor vTR(FixLP.x, FixLP.y);

	FixLP = ptBL->GetLP();
	CVektor vBL(FixLP.x, FixLP.y);

	FixLP = ptBR->GetLP();
	CVektor vBR(FixLP.x, FixLP.y);


										// calculate latitude 1. version

	CVektor vPol_TopLeft = m_vKegelPol - vTL;	// kegelspitze -> fixpunt oben links  
	CVektor vPol_BotLeft = m_vKegelPol - vBL;	// kegelspitze -> fixpunt unten links 
												// kegelspitze -> neuer Punkt          
 	CVektor vPol_X = m_vKegelPol - CVektor(lX_LP, lY_LP);

	double dPol_X = vPol_X.Abs();					// distance Pol -> new point
	double dPol_BL = vPol_BotLeft.Abs ();			// distance Pol -> fixpoint bottom left


 										// calculate latitude 
	CVektor vTopBot = vTL - vBL;
	double dTopBot = vTopBot.Abs();

	BOOL bSouthHemis = (m_vKegelPol.y < 0);
	if (bSouthHemis)
			dLat = ptBL->GetLat() - (ptTL->GetLat() - ptBL->GetLat()) * (dPol_BL- dPol_X) / dTopBot;
	else	dLat = ptBL->GetLat() + (ptTL->GetLat() - ptBL->GetLat()) * (dPol_BL- dPol_X) / dTopBot;



										// calculate longitude
	CVektor vPol_BotRight = m_vKegelPol - vBR;	 // kegelspitze -> fixpunt unten rechts

	double dAngleKegel  = vPol_BotLeft.VektorAngle (vPol_BotRight);
	double dAngleX_l	= vPol_BotLeft.VektorAngle (vPol_X);	// always positive !!
	double dLonDiff = (ptBR->GetLon() - ptBL->GetLon()) * dAngleX_l / dAngleKegel;

										 // check if right or left of bottom left fixpt
	double dAngleX_r	= vPol_BotRight.VektorAngle (vPol_X);
	if (dAngleX_r > dAngleKegel)		 // X left of BottomLeft
		 dLon = ptBL->GetLon() - dLonDiff; 
	else dLon = ptBL->GetLon() + dLonDiff;
	}

*ptLat = dLat;
*ptLon = dLon;
}

/************************************************************************
 *  ScanMap.cpp				G e t L i n e a r L a t L o n				*
 ************************************************************************/
void CScannedMap::GetLinearLatLon (long lX_LP, long lY_LP, double* ptLat, double* ptLon) 
{
	double dLat=0.;
	double dLon=0.;

	if (m_bCalibrated)
	{
		CMapFixPt* ptFP1; 
		CMapFixPt* ptFP2;
		CPoint FixLP1, FixLP2;

		if (this->GetFixPtCnt () == 2)
		{
			ptFP1 = this->GetFixPtPtr (0);
			ptFP2 = this->GetFixPtPtr (1);

											// get range of geographically koords
			double dLatRange = ptFP2->GetLat() - ptFP1->GetLat();
			double dLonRange = ptFP2->GetLon() - ptFP1->GetLon();

											// get range of logically koords
			FixLP1 = ptFP1->GetLP();
			FixLP2 = ptFP2->GetLP();

			long dX = FixLP2.x - FixLP1.x;
			long dY = FixLP2.y - FixLP1.y;

			if (dX != 0 && dY != 0)
			{
				dLon = (double)(lX_LP-FixLP1.x)*(dLonRange/dX) + ptFP1->GetLon();
				dLat = (double)(lY_LP-FixLP1.y)*(dLatRange/dY) + ptFP1->GetLat();
			}
		}
	}

	*ptLat = dLat;
	*ptLon = dLon;
}


/************************************************************************
 *  ScanMap.cpp				G e t M e r c a t o r L a t L o n			*
 ************************************************************************/
void CScannedMap::GetMercatorLatLon (long lX_LP, long lY_LP, double* ptLat, double* ptLon) 
{
double dLat=0.;
double dLon=0.;






*ptLat = dLat;
*ptLon = dLon;
}

/************************************************************************
 *  ScanMap.cpp				G e t P o l a r L a t L o n					*
 ************************************************************************/
void CScannedMap::GetPolarLatLon (long lX_LP, long lY_LP, double* ptLat, double* ptLon) 
{
double dLat=0.;
double dLon=0.;






*ptLat = dLat;
*ptLon = dLon;
}



/************************************************************************
 *  ScanMap.cpp				 L P t o L a t L o n						*
 ************************************************************************/
void CScannedMap::LPtoLatLon (long lX_LP, long lY_LP, double* ptLat, double* ptLon) 
{									// convert from scaled LP
lX_LP = (long)((double)(lX_LP - m_pStart.x) / m_dFaktX);
lY_LP = (long)((double)(lY_LP - m_pStart.y) / m_dFaktY);

switch (m_Type)
	{
	case LAMBERT:
		this->GetLambertLatLon (lX_LP, lY_LP, ptLat, ptLon);
		break;

	case LINEAR:
		this->GetLinearLatLon (lX_LP, lY_LP, ptLat, ptLon);
		break;

	case MERCATOR:
		this->GetMercatorLatLon (lX_LP, lY_LP, ptLat, ptLon);
		break;

	case POLAR:
		this->GetPolarLatLon (lX_LP, lY_LP, ptLat, ptLon);
		break;
	}
}

/************************************************************************
 *  ScanMap.cpp				G e t L a m b e r t L P						*
 ************************************************************************/
void CScannedMap::GetLambertLP (double dLon, double dLat, long* ptX_LP, long* ptY_LP) 
{
CPoint FixLP;
CVektor vR_left, vX;

if (m_bCalibrated)
	{
	CMapFixPt* ptTL = this->GetFixPtPtr (0);
	CMapFixPt* ptTR = this->GetFixPtPtr (1);
	CMapFixPt* ptBR = this->GetFixPtPtr (2);
	CMapFixPt* ptBL = this->GetFixPtPtr (3);

	FixLP = ptTL->GetLP();
	CVektor vTL(FixLP.x, FixLP.y);

	FixLP = ptTR->GetLP();
	CVektor vTR(FixLP.x, FixLP.y);

	FixLP = ptBL->GetLP();
	CVektor vBL(FixLP.x, FixLP.y);

	FixLP = ptBR->GetLP();
	CVektor vBR(FixLP.x, FixLP.y);


   	BOOL bSouthHemis = (m_vKegelPol.y < 0);

										// calculate distance from pol to X
	CVektor vTopBot = vTL - vBL;
	CVektor vPol_BotLeft = m_vKegelPol - vBL;	// kegelspitze -> fixpunt unten links 

	double dPol_X;
	if (bSouthHemis)
		 dPol_X = vPol_BotLeft.Abs() - vTopBot.Abs() * (ptBL->GetLat()-dLat) / 
											(ptTL->GetLat()-ptBL->GetLat());
	else dPol_X = vPol_BotLeft.Abs() + vTopBot.Abs() * (ptBL->GetLat()-dLat) / 
											(ptTL->GetLat()-ptBL->GetLat());

										// calculate angle btn. Pol_BotLeft and Pol_X
 	CVektor vPol_BotRight = m_vKegelPol - vBR;	 // kegelspitze -> fixpunt unten rechts
	double dAngleKegel  = vPol_BotLeft.VektorAngle (vPol_BotRight);
	double dAngleX      = dAngleKegel * (dLon-ptBL->GetLon()) / 
										(ptBR->GetLon()-ptBL->GetLon());
	if (bSouthHemis)
		dAngleX = -dAngleX;

										// calculate vX on dLat and line Pol_BotLeft
	CVektor vR_left;
	vR_left.Normalize(vBL - vTL);		// Einheitsvektor

	if (bSouthHemis)
			vX	= m_vKegelPol - vR_left * dPol_X;
	else	vX	= m_vKegelPol + vR_left * dPol_X;

										// Rotate vX with dAngleX => vX on line Pol_X
	vX.Rotate(m_vKegelPol, dAngleX);
	}

*ptX_LP = (long)vX.x;
*ptY_LP = (long)vX.y;
}

/************************************************************************
 *  ScanMap.cpp				G e t L i n e a r L P						*
 ************************************************************************/
void CScannedMap::GetLinearLP (double dLon, double dLat, long* ptX_LP, long* ptY_LP) 
{
	*ptX_LP = 0;
	*ptY_LP = 0;

	if (m_bCalibrated)
	{
		CMapFixPt* ptFP1; 
		CMapFixPt* ptFP2;
		CPoint FixLP1, FixLP2;

		if (this->GetFixPtCnt () == 2)
		{
			ptFP1 = this->GetFixPtPtr (0);
			ptFP2 = this->GetFixPtPtr (1);

											// get range of geographically koords
			double dLatRange = ptFP2->GetLat() - ptFP1->GetLat();
			double dLonRange = ptFP2->GetLon() - ptFP1->GetLon();

											// get range of logically koords
			FixLP1 = ptFP1->GetLP();
			FixLP2 = ptFP2->GetLP();

			long dX = FixLP2.x - FixLP1.x;
			long dY = FixLP2.y - FixLP1.y;

			*ptX_LP = (long)((dLon - ptFP1->GetLon()) / (dLonRange/dX) + FixLP1.x);
			*ptY_LP = (long)((dLat - ptFP1->GetLat()) / (dLatRange/dY) + FixLP1.y);
		}
	}
}

/************************************************************************
 *  ScanMap.cpp				G e t M e r c a t o r L P					*
 ************************************************************************/
void CScannedMap::GetMercatorLP (double dLon, double dLat, long* ptX_LP, long* ptY_LP) 
{
CVektor vX;



*ptX_LP = (long)vX.x;
*ptY_LP = (long)vX.y;
}

/************************************************************************
 *  ScanMap.cpp				G e t P o l a r L P							*
 ************************************************************************/
void CScannedMap::GetPolarLP (double dLon, double dLat, long* ptX_LP, long* ptY_LP) 
{
CVektor vX;



*ptX_LP = (long)vX.x;
*ptY_LP = (long)vX.y;
}

/************************************************************************
 *  ScanMap.cpp				 L a t L o n T o L P 						*
 ************************************************************************/
void CScannedMap::LatLonToLP(double dLon, double dLat, long* ptX_LP, long* ptY_LP)
{
DRECT rLatLon = this->GetLatLonBorder ();
BOOL bAntipod = (rLatLon.right > 180);
dLon = (bAntipod && dLon < 0)? 360 + dLon : dLon;

switch (m_Type)
	{
	case LAMBERT:
		this->GetLambertLP (dLon, dLat, ptX_LP, ptY_LP);
		break;

	case LINEAR:
		this->GetLinearLP (dLon, dLat, ptX_LP, ptY_LP);
		break;

	case MERCATOR:
		this->GetMercatorLP (dLon, dLat, ptX_LP, ptY_LP);
		break;

	case POLAR:
		this->GetPolarLP (dLon, dLat, ptX_LP, ptY_LP);
		break;
	}
								// convert to scaled LP
*ptX_LP = (long)(m_dFaktX * (*ptX_LP)) + m_pStart.x;
*ptY_LP = (long)(m_dFaktY * (*ptY_LP)) + m_pStart.y;
}

/************************************************************************
 *  ScanMap.cpp    		 	C r o s s i n g B o r d e r					*
 *  Calculates factors L1,L2, where leg meets border of map				*
 ************************************************************************/
BOOL CScannedMap::CalcBorderPoint(CVektor& V1, CVektor& V2, CVektor& E, double* ptL1, double* ptL2)
{
	BOOL bCalculated = FALSE;
	CVektor Res(E);
	bCalculated = Res.LGS2(V1, V2);
	if (bCalculated)
	{
		*ptL1 = Res.X();
		*ptL2 = Res.Y();
	}

	return bCalculated;
}

/************************************************************************
 *  ScanMap.cpp    		 	I s I n s i d e								*
 ************************************************************************/
BOOL CScannedMap::IsInside(long LPx1, long LPy1)
{
	BOOL bInside = TRUE;
	long LPxmin = 0;
	long LPymax = 0;
	long LPxmax, LPymin;

	GetScaledSizeLP (&LPxmax, &LPymin);
	LPymin = - LPymin;

	if (LPx1 < LPxmin || LPx1 > LPxmax)
		bInside = FALSE;

	if (LPy1 < LPymin || LPy1 > LPymax)
		bInside = FALSE;

	return bInside;
}

/************************************************************************
 *  ScanMap.cpp    		 	C r o s s i n g B o r d e r					*
 ************************************************************************/
BOOL CScannedMap::CrossingBorder(long LPx0, long LPy0, long LPx1, long LPy1, 
								 long* ptLPx, long* ptLPy)
{
	BOOL bCrossed = FALSE;
	long LPxmin = 0;
	long LPymax = 0;
	long LPxmax, LPymin;

	GetScaledSizeLP (&LPxmax, &LPymin);
	LPymin = - LPymin;

	long dX = LPx1-LPx0;
	long dY = LPy1-LPy0;
	if (dX != 0 || dY != 0)
	{
		CVektor Rleg(dX, dY);
		Rleg.Normalize();

		CVektor B, E;		// B: Border direction, E: Result of matrix
		double dL1, dL2;	// faktors to get meeting point
		long LPx, LPy;		// point where leg meets border of map

		if (LPx0 > LPxmax || LPx1 > LPxmax)
		{	// rechts
			B.Set(0,-1);
			E.Set(LPxmax-LPx0, -LPy0);
			if (this->CalcBorderPoint(Rleg, B, E, &dL1, &dL2))
			{
				LPx = LPxmax;
				LPy = (long)dL2;
				if (LPymin <= LPy && LPy <= LPymax)
					bCrossed = TRUE;
			}
		}
		if (!bCrossed && (LPx0 < LPxmin || LPx1 < LPxmin))
		{	// links
			B.Set(0,-1);
			E.Set(-LPx0, -LPy0);
			if (this->CalcBorderPoint(Rleg, B, E, &dL1, &dL2))
			{
				LPx = LPxmin;
				LPy = (long)dL2;
				if (LPymin <= LPy && LPy <= LPymax)
					bCrossed = TRUE;
			}
		}
		if (!bCrossed && (LPy0 > LPymax || LPy1 > LPymax))
		{	// oben
			B.Set(-1,0);
			E.Set(-LPx0, -LPy0);
			if(this->CalcBorderPoint(Rleg, B, E, &dL1, &dL2))
			{
				LPx = (long)dL2;
				LPy = LPymax;
				if (LPxmin <= LPx && LPx <=LPxmax)
					bCrossed = TRUE;
			}
		}
		if (!bCrossed && (LPy0 < LPymin || LPy1 < LPymin))
		{	// unten
			B.Set(-1,0);
			E.Set(-LPx0, LPymin-LPy0);
			if (this->CalcBorderPoint(Rleg, B, E, &dL1, &dL2))
			{
				LPx = (long)dL2;
				LPy = LPymin;
				if (LPxmin <= LPx && LPx <=LPxmax)
					bCrossed = TRUE;
			}
		}

		if (bCrossed)
		{
			*ptLPx = LPx;
			*ptLPy = LPy;
		}
	}	// no movement
	return bCrossed;
}

/************************************************************************
 *  ScanMap.cpp    		 	C r o s s i n g B o r d e r	s				*
 *  get visible part of long leg										*
 ************************************************************************/
BOOL CScannedMap::CrossingBorders(long LPx0, long LPy0, long LPx1, long LPy1, 
								 long* ptLPx0, long* ptLPy0, long* ptLPx1, long* ptLPy1)
{
	BOOL bCrossedHorz = FALSE;
	BOOL bCrossedVert = FALSE;
	long LPxmin = 0;
	long LPymax = 0;
	long LPxmax, LPymin;
	
	GetScaledSizeLP (&LPxmax, &LPymin);
	LPymin = - LPymin;

	long dX = LPx1-LPx0;
	long dY = LPy1-LPy0;
	if (dX != 0 || dY != 0)
	{
		CVektor Rleg(dX, dY);
		Rleg.Normalize();

		CVektor B, E;		// B: Border direction, E: Result of matrix
		double dL1, dL2;	// faktors to get meeting point
		long LPx, LPy;		// point where leg meets border of map

		if ((LPx0 < LPxmax && LPx1> LPxmax ) ||
			(LPx1 < LPxmax && LPx0 > LPxmax))
		{	// rechts
			B.Set(0,-1);
			E.Set(LPxmax-LPx0, -LPy0);
			if (this->CalcBorderPoint(Rleg, B, E, &dL1, &dL2))
			{
				LPx = LPxmax;
				LPy = (long)dL2;
				if (LPymin <= LPy && LPy <= LPymax)
					bCrossedVert = TRUE;
			}
		}
		if (!bCrossedVert && 
			((LPx0 < LPxmin && LPx1> LPxmin ) ||
			(LPx1 < LPxmin && LPx0 > LPxmin)))
		{	// links
			B.Set(0,-1);
			E.Set(-LPx0, -LPy0);
			if (this->CalcBorderPoint(Rleg, B, E, &dL1, &dL2))
			{
				LPx = LPxmin;
				LPy = (long)dL2;
				if (LPymin <= LPy && LPy <= LPymax)
					bCrossedVert = TRUE;
			}
		}

		if (bCrossedVert)
		{
			*ptLPx0 = LPx;
			*ptLPy0 = LPy;

			if ((LPy0 < LPymax && LPy1 > LPymax) ||
				(LPy1 < LPymax && LPy0 > LPymax))
			{	// oben
				B.Set(-1,0);
				E.Set(-LPx0, -LPy0);
				if(this->CalcBorderPoint(Rleg, B, E, &dL1, &dL2))
				{
					LPx = (long)dL2;
					LPy = LPymax;
					if (LPxmin <= LPx && LPx <=LPxmax)
						bCrossedHorz = TRUE;
				}
			}
			if (!bCrossedHorz && 
				((LPy0 < LPymin && LPy1 > LPymin) ||
				 (LPy1 < LPymin && LPy0 > LPymin)))
			{	// unten
				B.Set(-1,0);
				E.Set(-LPx0, LPymin-LPy0);
				if (this->CalcBorderPoint(Rleg, B, E, &dL1, &dL2))
				{
					LPx = (long)dL2;
					LPy = LPymin;
					if (LPxmin <= LPx && LPx <=LPxmax)
						bCrossedHorz = TRUE;
				}
			}
		}

		if (bCrossedHorz)
		{
			*ptLPx1 = LPx;
			*ptLPy1 = LPy;
		}
	}	// no movement
	return (bCrossedVert && bCrossedHorz);
}

/************************************************************************
 *  ScanMap.cpp    		 		I s E q u a l							*
 ************************************************************************/
BOOL CScannedMap::IsEqual(const CScannedMap& Map)
{
double	EPS_wi = 0.00001;			/* Genauigkeit zum Winkelvergleich	*/

if (m_bCalibrated != Map.m_bCalibrated) return FALSE;

if (m_bCalibrated)
	{
	short i;
	for (i=0; i<4; i++)
		{
		if (fabs (m_Border[i].GetLat() - Map.m_Border[i].GetLat()) > EPS_wi)	return FALSE; 
		if (fabs (m_Border[i].GetLon() - Map.m_Border[i].GetLon()) > EPS_wi)	return FALSE; 
		}
	}

									// compare line width and color setting
if (m_nSizeLine != Map.m_nSizeLine) return FALSE;
if (m_nSizeBear != Map.m_nSizeBear) return FALSE;

if (m_ColLine != Map.m_ColLine) return FALSE;
if (m_ColBear != Map.m_ColBear) return FALSE;
if (m_ColLabel != Map.m_ColLabel) return FALSE;
if (m_ColSel != Map.m_ColSel) return FALSE;

if (m_ColText != Map.m_ColText) return FALSE;


if (m_szPath.Compare (Map.m_szPath) != 0) return FALSE;
if (m_Type != Map.m_Type) return FALSE;
if (m_bAntipod != Map.m_bAntipod) return FALSE;
if (m_bAutoSel != Map.m_bAutoSel) return FALSE;
if (m_szName.Compare (Map.m_szName) != 0) return FALSE;

if (m_lSizeX_DP != Map.m_lSizeX_DP) return FALSE;
if (m_lSizeY_DP != Map.m_lSizeY_DP) return FALSE;

if (m_lBmpSizeX != Map.m_lBmpSizeX) return FALSE;
if (m_lBmpSizeY != Map.m_lBmpSizeY) return FALSE;

if (m_nActCalIndex != Map.m_nActCalIndex) return FALSE;

if (m_dFaktX	!= Map.m_dFaktX) return FALSE;
if (m_dFaktY	!= Map.m_dFaktY) return FALSE;
if (m_pStart	!= Map.m_pStart) return FALSE;

if (m_bRelativ	!= Map.m_bRelativ) return FALSE;
if (m_nCourse	!= Map.m_nCourse) return FALSE;
if (m_dDist		!= Map.m_dDist) return FALSE;
if (m_nDistDim	!= Map.m_nDistDim) return FALSE;
 
return TRUE;
}

	
/************************************************************************
 *  ScanMap.cpp    		 	G e t A c t C a l I n f o T e x t			*
 ************************************************************************/
CString CScannedMap::GetActCalInfoText (short nIndex)
{
CString szInfo;
short nStrOffset;

if (nIndex == -2)
	nIndex = m_nActCalIndex;

if (nIndex > -1)
	{
	CALIBRATEPOS CalPos = this->FixPtIndexToCalibPos(nIndex);
	switch (m_Type)
		{
		case LAMBERT:
			nStrOffset = CalPos;
			szInfo.LoadString (IDS_LAM_CALIB_MSG_TL+nStrOffset);
			break;

		case LINEAR:
			nStrOffset = (CalPos==CALPOS_BR)? 1 : 0;
			szInfo.LoadString (IDS_LIN_CALIB_MSG_TL+nStrOffset);
			break;

		case MERCATOR:
			szInfo.LoadString (IDS_LIN_CALIB_MSG_TL);	// dummy ID
			break;

		case POLAR:
			szInfo.LoadString (IDS_LIN_CALIB_MSG_TL);	// dummy ID
			break;
		}
	}

return szInfo;
}

/************************************************************************
 *  ScanMap.cpp				I s R e a d y T o C a l i b r a t e			*
 ************************************************************************/
BOOL CScannedMap::IsReadyToCalibrate ()
{
BOOL bReady;
short	i, nReqPts;
short	nDefinedCnt=0;

switch (m_Type)
	{
	case LAMBERT:	nReqPts = 4;	break;
	case LINEAR:	nReqPts = 2;	break;
	case MERCATOR:	nReqPts = 0;	break;	// not implemented
	case POLAR:		nReqPts = 0;	break;  // not implemented
	}

for (i=0; i<nReqPts; i++)
	{
	CMapFixPt* ptFixPt = (CMapFixPt*)m_FixPts.GetAt(i);
	if (ptFixPt != NULL)
		{
		if (ptFixPt->IsDefined())
			nDefinedCnt++;
		}
	}

bReady = (nReqPts == nDefinedCnt);
return bReady;
}

/************************************************************************
 *  ScanMap.cpp					C h e c k A n t i p o d					*
 ************************************************************************/
void CScannedMap::CheckAntipod()
{
switch (m_Type)
	{
	case LAMBERT:
		{				// if left latutude > right latitude => antipod			   
		double dLonLeft, dLonRight;
		CMapFixPt* ptFixPt = this->GetFixPtPtr (CalibPosToFixPtIndex(CALPOS_TL));
		dLonLeft = ptFixPt->GetLon();

		ptFixPt = this->GetFixPtPtr (CalibPosToFixPtIndex(CALPOS_TR));
		dLonRight = ptFixPt->GetLon();		 // dLonRight is negative!!
		
		m_bAntipod = (dLonRight < dLonLeft);
		if (m_bAntipod)
			{					// TR and BR must have same longitude values!!
			dLonRight = 360. + dLonRight;	 // new longitude now > 180

			ptFixPt->SetLon (dLonRight); // new TR longitude now > 180
			ptFixPt = this->GetFixPtPtr (CalibPosToFixPtIndex(CALPOS_BR));
			ptFixPt->SetLon (dLonRight); // new BR longitude now > 180
			}
		}
		break;

	case LINEAR:
	{				// if left latutude > right latitude => antipod			   
		double dLon1, dLon2;

		CMapFixPt* ptFixPt1 = this->GetFixPtPtr (CalibPosToFixPtIndex(CALPOS_TL));
		dLon1 = ptFixPt1->GetLon();

		CMapFixPt* ptFixPt2 = this->GetFixPtPtr (CalibPosToFixPtIndex(CALPOS_BR));
		dLon2 = ptFixPt2->GetLon();
		
		m_bAntipod = (fabs (dLon2 - dLon1) > 180);
		if (m_bAntipod)
		{				// point1 and point2 must have positive longitude values!!
			if (dLon1 < 0)
			{
				dLon1 = 360. + dLon1;	 // new longitude now > 180
				ptFixPt1->SetLon (dLon1); // new BR longitude now > 180
			}
			if (dLon2 < 0)
			{
				dLon2 = 360. + dLon2;	 // new longitude now > 180
				ptFixPt2->SetLon (dLon2); // new BR longitude now > 180
			}
		}
	}
	break; 

	case MERCATOR:
		break;

	case POLAR:
		break;
	}
}


/************************************************************************
 *  ScanMap.cpp			I s L a t L o n V i s i b l e 					*
 ************************************************************************/
BOOL CScannedMap::IsLatLonVisible(double dLat, double dLon)
{
	BOOL bLatLonOnMap = FALSE;
	long lSizeX, lSizeY;
	long lX_LP, lY_LP;
	CRect rMapLP;

											// get logical area of map
	this->GetScaledSizeLP (&lSizeX, &lSizeY);
	rMapLP.SetRect (0,0, lSizeX, -lSizeY);
	this->LatLonToLP (dLon, dLat, &lX_LP, &lY_LP);
	if ((rMapLP.left <= lX_LP && lX_LP <= rMapLP.right) &&
		(rMapLP.top >= lY_LP && lY_LP >= rMapLP.bottom))
	{
		bLatLonOnMap = TRUE;
	}

	return bLatLonOnMap;
}

/************************************************************************
 *  ScanMap.cpp			I s L a t L o n I n R e c t 					*
 ************************************************************************/
BOOL CScannedMap::IsLatLonInRect (double dLat, double dLon, double* ptMapLon)
{
BOOL bInRect = FALSE;

DRECT rLatLon = this->GetLatLonBorder();

	double dMapLon;
	
	BOOL bLonOK = CLatLon::GetMapLon (rLatLon.left, dLon, rLatLon.right, &dMapLon);
	if (ptMapLon != NULL)
		*ptMapLon = dMapLon;

	if (bLonOK &&
		rLatLon.bottom<=dLat && dLat<=rLatLon.top)
		bInRect = TRUE;

return bInRect;
}


/************************************************************************
 *  ScanMap.cpp			G e t L a t L o n L i m i t0 					*
 *  Purpose: Get new values for dLat0 and dLon0							*
 *	Sucht Punkt auf Grokreis von dLat0/dLon0 nach dLat1/dLon1			*
 *			der mglichst nah, am Zentrum der Karte,					*
 *			aber noch auerhalb der Karte liegt							*
 ************************************************************************/
BOOL CScannedMap::GetLatLonLimit0 (double dLat0, double dLon0, double dLat1, double dLon1, 
								  double* ptVisLat0, double* ptVisLon0)
{
	BOOL		bFound = FALSE;
	BOOL		bStop = FALSE;
	CVektor		A,B, X;
	CLatLon		LocA, LocB, LocX;
	double	    LastCourse, Course, wab, dLastDist, dDist;

	CLatLon Loc0(dLat0, dLon0);
	CLatLon Loc1(dLat1, dLon1);

	CLatLon CenterLoc = this->GetCenterLatLon();
									/* get distance between Loc0 and CenterLoc		*/
	dLastDist = CenterLoc.Distance (Loc0, &LastCourse);

	LocA = Loc0;
	LocB = Loc1;
	do
	{
		LocA.ToXYZ(&A);					/* Koordinaten von Ort A			*/
		LocB.ToXYZ(&B);					/* Koordinaten von Ort B			*/
		wab = A.VektorAngle (B);		/* Winkel zwischen Ortsvektoren		*/
									/* calculate waypoint btn LocA and LocB */
		X = A.GrossKreis (B, wab/2);	/* X : Ort nach Winkel wab/2		*/

		LocX.FromXYZ (X);			/* ptLocX: Ort nach Winkel wab/2	*/
		if (this->IsLatLonInRect (LocX.GetLat(), LocX.GetLon()))
		{							// LocX inside of Map
			bFound = TRUE;
			bStop = TRUE;
		}
		else
		{							// LocX outside of Map
			dDist = CenterLoc.Distance (LocX, &Course);

			double CourseDiff = CLatLon::FlightAngle (Course, LastCourse);
			if (fabs(CourseDiff) < 90)
			{				// LocX same of map as Loc0
				if (dDist < dLastDist)
				{
					dLastDist = dDist;
					LocA = LocX;		// LocX liegt nher am Kartenmittelpunkt als LocA
				}
				else
				{
					bStop = TRUE;
				}
			}
			else 
			{				// LocX on opposite side as Loc0
				LocB = LocX;
			}
		}

	} while (!bStop);

	if (!bFound)
	{
		if (dLastDist < this->GetDiagonalDist_NM())
			bFound = TRUE;
	}

	*ptVisLat0 = LocA.GetLat();
	*ptVisLon0 = LocA.GetLon();

	return bFound;
}

/************************************************************************
 *  ScanMap.cpp			 G e t L a t L o n L i m i t 1 					*
 *  Purpose: Get new values for dLat1 and dLon1							*
 *	Sucht Punkt auf Grokreis von dLat0/dLon0 nach dLat1/dLon1			*
 *			der mglichst nah, am Zentrum der Karte,					*
 *			aber noch auerhalb der Karte liegt							*
 ************************************************************************/
BOOL CScannedMap::GetLatLonLimit1 (double dLat0, double dLon0, double dLat1, double dLon1, 
								  double* ptVisLat1, double* ptVisLon1)
{
	BOOL		bFound = FALSE;
	BOOL		bStop = FALSE;
	CVektor		A,B, X;
	CLatLon		LocA, LocB, LocX;
	double	    LastCourse, Course, wab, dLastDist, dDist;

	CLatLon Loc0(dLat0, dLon0);
	CLatLon Loc1(dLat1, dLon1);

	CLatLon CenterLoc = this->GetCenterLatLon();

									/* get distance between CenterLoc and Loc1		*/
	dLastDist = CenterLoc.Distance (Loc1, &LastCourse);

	LocA = Loc0;
	LocB = Loc1;
	do
	{
		LocA.ToXYZ(&A);					/* Koordinaten von Ort A			*/
		LocB.ToXYZ(&B);					/* Koordinaten von Ort B			*/
		wab = A.VektorAngle (B);		/* Winkel zwischen Ortsvektoren		*/
									/* calculate waypoint btn LocA and LocB */
		X = A.GrossKreis (B, wab/2);	/* X : Ort nach Winkel wab/2		*/

		LocX.FromXYZ (X);			/* ptLocX: Ort nach Winkel wab/2	*/
		if (this->IsLatLonInRect (LocX.GetLat(), LocX.GetLon()))
		{							// LocX inside of Map
			bFound = TRUE;
			bStop = TRUE;
		}
		else
		{							// LocX outside of Map
			dDist = CenterLoc.Distance (LocX, &Course);

			double CourseDiff = CLatLon::FlightAngle (Course, LastCourse);
			if (fabs(CourseDiff) < 90)
			{				// LocX same of map as Loc1
				if (dDist < dLastDist)
				{
					dLastDist = dDist;
					LocB = LocX;		// LocX liegt nher am Kartenmittelpunkt als LocB
				}
				else
				{
					bStop = TRUE;
				}
			}
			else 
			{				// LocX on opposite side as Loc1
				LocA = LocX;
			}
		}

	} while (!bStop);

	if (!bFound)
	{
		if (dLastDist < this->GetDiagonalDist_NM())
			bFound = TRUE;
	}

	*ptVisLat1 = LocB.GetLat();
	*ptVisLon1 = LocB.GetLon();

	return bFound;
}

// replaced by new version, see below... after test, remove this together
// with GetLatLonLimit0 and GetLatLonLimit1 
/****************************************************************************
 *	ScanMap.cpp				I s V i s i b l e L i n e 						*
 ****************************************************************************/
/*BOOL CScannedMap::IsVisibleLine (double dLat0, double dLon0, double dLat1, double dLon1, 
				double* ptVisLat0, double* ptVisLon0, double* ptVisLat1, double* ptVisLon1)
{
	BOOL bDraw = FALSE;

	BOOL bStartInRect = this->IsLatLonInRect (dLat0, dLon0);
	BOOL bEndInRect   = this->IsLatLonInRect (dLat1, dLon1);

	if (bStartInRect)
	{
		*ptVisLon0  = dLon0;
		*ptVisLat0	= dLat0;

		if (bEndInRect)
		{				// start and end in DRECT
			*ptVisLon1  = dLon1;
			*ptVisLat1	= dLat1;
			bDraw = TRUE;
		}
		else
		{				// start in DRECT, but end is outside of DRECT
						// calculate lat lon of point leaving DRECT
			if (this->GetLatLonLimit1 (dLat0, dLon0, dLat1, dLon1, ptVisLat1, ptVisLon1))
				bDraw = TRUE;
		}
	}
	else
	{						// start outside of DRECT
		if (bEndInRect)
		{				// end in DRECT, but start is outside of DRECT
						// calculate lat lon of point entering DRECT
		 	if (this->GetLatLonLimit0 (dLat0, dLon0, dLat1, dLon1, ptVisLat0, ptVisLon0))
			{
				*ptVisLon1  = dLon1;
				*ptVisLat1	= dLat1;
				bDraw = TRUE;
			}
		}
		else
		{				// start and end is outside of DRECT
						// check if DRECT is btn start and end
			if (this->GetLatLonLimit0 (dLat0, dLon0, dLat1, dLon1, ptVisLat0, ptVisLon0) &&
				this->GetLatLonLimit1 (dLat0, dLon0, dLat1, dLon1, ptVisLat1, ptVisLon1))
				bDraw = TRUE;

		}
	}
return bDraw;
}
*/

/************************************************************************
 *  ScanMap.cpp			 G r o s s K r e i s F o r L o n				*
 *  Purpose: get point on orthodrome A - B for specified dLon			*
 *	return: dLat of found point											*
 ************************************************************************/
BOOL CScannedMap::GrossKreisForLon(CVektor A, CVektor B, double dLon, BOOL bLeftBorder, CLatLon* ptLocX)
{
	BOOL bDefined = FALSE;

	double dLonDiff, wax, wax_min, wax_max, dEps;
	int		nCnt = 0;
	CVektor X;
	CLatLon LocTest;


	dEps = 0.00001;
	wax_min = 0;
	wax_max = A.VektorAngle (B);			// Winkel zwischen Ortsvektoren	

	do
	{
		wax = (wax_min + wax_max) / 2;
		X = A.GrossKreis (B, wax);			// X : Ort nach Winkel wax			
		LocTest.FromXYZ (X);				// LocX: Ort nach Winkel wax

		if (bLeftBorder)
			dLonDiff = LocTest.GetLon() - dLon;	// check resulting lon
		else
			dLonDiff = dLon - LocTest.GetLon();

		if (dLonDiff >= 0) wax_max = wax;
			else		   wax_min = wax;
		
		nCnt++;
	} while (fabs(wax_max - wax_min) > dEps && nCnt < 100);

	if (nCnt < 100)
	{
		*ptLocX = LocTest;
		bDefined = TRUE;
	}

	return bDefined;
}

/************************************************************************
 *  ScanMap.cpp			 G r o s s K r e i s F o r L a t				*
 *  Purpose: get point on orthodrome A - B for specified dLon			*
 *	return: dLat of found point											*
 ************************************************************************/
BOOL CScannedMap::GrossKreisForLat(CVektor A, CVektor B, double dLat, BOOL bTopBorder, CLatLon* ptLocX)
{
	BOOL bDefined = FALSE;

	double dLatDiff, wax, wax_min, wax_max, dEps;
	int		nCnt = 0;
	CVektor X;
	CLatLon LocTest;


	dEps = 0.00001;
	wax_min = 0;
	wax_max = A.VektorAngle (B);			// Winkel zwischen Ortsvektoren	

	do
	{
		wax = (wax_min + wax_max) / 2;
		X = A.GrossKreis (B, wax);			// X : Ort nach Winkel wax			
		LocTest.FromXYZ (X);				// LocX: Ort nach Winkel wax

		if (bTopBorder)
			dLatDiff = LocTest.GetLat() - dLat;	// check resulting lat
		else
			dLatDiff = dLat - LocTest.GetLat();

		if (dLatDiff >= 0) wax_max = wax;
			else		   wax_min = wax;
		
		nCnt++;
	} while (fabs(wax_max - wax_min) > dEps && nCnt < 100);

	if (nCnt < 100)
	{
		*ptLocX = LocTest;
		bDefined = TRUE;
	}

	return bDefined;
}


/************************************************************************
 *  ScanMap.cpp			 G e t L a t L o n L i m i t 					*
 *  Purpose: Get new values for dLat1 and dLon1							*
 *	Sucht ersten Punkt auf Grokreis von dLat0/dLon0 nach dLat1/dLon1	*
 *			der von LocA aus gesehen am Kartenrand liegt				*
 ************************************************************************/
BOOL CScannedMap::GetLatLonLimit (double dLat0, double dLon0, double dLat1, double dLon1, 
								  double* ptVisLat0, double* ptVisLon0)
{
	BOOL		bFound = FALSE;
	CVektor		A,B, X;
	CLatLon		LocX, LocTest;
	double	    dEps, wab, dDist, dDistMin, dCourse;

	dEps = 0.00001;

	CLatLon LocA(dLat0, dLon0);
	CLatLon LocB(dLat1, dLon1);

	DRECT rLatLon = this->GetLatLonBorder();


	LocA.ToXYZ(&A);							/* Koordinaten von Ort A			*/
	LocB.ToXYZ(&B);							/* Koordinaten von Ort B			*/
	wab = A.VektorAngle (B);				/* Winkel zwischen Ortsvektoren		*/
	dDistMin = LocA.Distance (LocB, &dCourse);
	
	if (fabs(dLon1 - dLon0) > dEps)
	{										// Unterschiedliche Lngengrade
		BOOL bLeftBorder;							
		double dLonX = rLatLon.left;		// Schnittpunkt mit linkem Kartenrand?  // dLonX = 9.8928844801992 (soll)
		bLeftBorder = TRUE;
		if (GrossKreisForLon(A, B, dLonX, bLeftBorder, &LocTest))		// LocTest = ortho point for dLonX
		{
			dDist = LocA.Distance (LocTest, &dCourse);	// mit wax = 0.16910 ergbe sich Lat = 9.8971454983480 
			if (dDist < dDistMin)
			{
				if ((rLatLon.bottom <= LocTest.GetLat()) && (LocTest.GetLat() <= rLatLon.top))
				{
					dDistMin = dDist;
					LocX = LocTest;
					bFound = TRUE;
				}
			}
		}

		dLonX = rLatLon.right;				// Schnittpunkt mit rechtem Kartenrand ?

		bLeftBorder = FALSE;
		if (GrossKreisForLon(A, B, dLonX, bLeftBorder, &LocTest))		// LocTest = ortho point for dLonX
		{
			dDist = LocA.Distance (LocTest, &dCourse);
			if (dDist < dDistMin)
			{
				if ((rLatLon.bottom <= LocTest.GetLat()) && (LocTest.GetLat() <= rLatLon.top))
				{
					dDistMin = dDist;
					LocX = LocTest;
					bFound = TRUE;
				}
			}
		}
	}
	
	if (fabs(dLat1 - dLat0) > dEps)
	{		
		BOOL bTopBorder;					// Unterschiedliche Breitengrade
		double dLatX = rLatLon.bottom;		// Schnittpunkt mit unterem Kartenrand ?
		bTopBorder = TRUE;
		if (GrossKreisForLat(A, B, dLatX, bTopBorder, &LocTest))	// LocTest = ortho point for dLatX
		{
			dDist = LocA.Distance (LocTest, &dCourse);
			if (dDist < dDistMin)
			{
				if ((rLatLon.left <= LocTest.GetLon()) && (LocTest.GetLon() <= rLatLon.right))
				{
					dDistMin = dDist;
					LocX = LocTest;
					bFound = TRUE;
				}
			}
		}

		dLatX = rLatLon.top;				// Schnittpunkt mit oberem Kartenrand ?
		bTopBorder = FALSE;
		if (GrossKreisForLat(A, B, dLatX, bTopBorder, &LocTest))	// LocTest = ortho point for dLatX
		{
			dDist = LocA.Distance (LocTest, &dCourse);
			if (dDist < dDistMin)
			{
				if ((rLatLon.left <= LocTest.GetLon()) && (LocTest.GetLon() <= rLatLon.right))
				{
					dDistMin = dDist;
					LocX = LocTest;
					bFound = TRUE;
				}
			}
		}
	}
	
	if (bFound)
	{
		*ptVisLat0 = LocX.GetLat();
		*ptVisLon0 = LocX.GetLon();
	}
	else
	{
		*ptVisLat0 = NO_KOORD;
		*ptVisLon0 = NO_KOORD;
	}

	return bFound;
}


/****************************************************************************
 *	ScanMap.cpp				I s V i s i b l e L i n e 						*
 *  used in:	CScanMapView::DrawRouteOnMap and							*
 *				CScanMapView::DrawMvMpRteOnMap								*
 ****************************************************************************/
BOOL CScannedMap::IsVisibleLine (double dLat0, double dLon0, double dLat1, double dLon1, 
				double* ptVisLat0, double* ptVisLon0, double* ptVisLat1, double* ptVisLon1)
{
	BOOL bDraw = FALSE;

	BOOL bStartInRect = this->IsLatLonInRect (dLat0, dLon0);
	BOOL bEndInRect   = this->IsLatLonInRect (dLat1, dLon1);

	if (bStartInRect)
	{
		*ptVisLon0  = dLon0;
		*ptVisLat0	= dLat0;

		if (bEndInRect)
		{				// start and end in DRECT
			*ptVisLon1  = dLon1;
			*ptVisLat1	= dLat1;
			bDraw = TRUE;
		}
		else
		{				// start in DRECT, but end is outside of DRECT
						// calculate lat lon of point leaving DRECT
			if (this->GetLatLonLimit (dLat1, dLon1, dLat0, dLon0, ptVisLat1, ptVisLon1))
				bDraw = TRUE;
		}
	}
	else
	{						// start outside of DRECT
		if (bEndInRect)
		{				// end in DRECT, but start is outside of DRECT
						// calculate lat lon of point entering DRECT
			if (this->GetLatLonLimit (dLat0, dLon0, dLat1, dLon1, ptVisLat0, ptVisLon0))
			{
				*ptVisLon1  = dLon1;
				*ptVisLat1	= dLat1;
				bDraw = TRUE;
			}
		}
		else
		{				// start and end is outside of DRECT
						// check if DRECT is btn start and end
			if (this->GetLatLonLimit (dLat0, dLon0, dLat1, dLon1, ptVisLat0, ptVisLon0) &&
				this->GetLatLonLimit (dLat1, dLon1, dLat0, dLon0, ptVisLat1, ptVisLon1))
				bDraw = TRUE;
		}
	}
return bDraw;
}


/************************************************************************
 *  ScanMap.cpp			S e a r c h N e x t C a l i b I n d e x			*
 ************************************************************************/
void CScannedMap::SearchNextCalibIndex(short* ptCalIndex)
{	
		// not ready to calibrate: search first undefined fix point
short nCalIndex = 0;
short i;
CMapFixPt* ptFixPt;	
BOOL	bDefined = TRUE;
for (i=0; i<this->GetFixPtCnt() && bDefined; i++)
	{
	bDefined = FALSE;
	ptFixPt = this->GetFixPtPtr (i);
	if (ptFixPt != NULL)
		{
		bDefined = ptFixPt->IsDefinedLP();
		if (bDefined)
			nCalIndex++;
		}
	}

if (nCalIndex == this->GetFixPtCnt())
	 *ptCalIndex = -1;
else *ptCalIndex = nCalIndex;	 
}

/************************************************************************
 *  ScanMap.cpp					T r y T o C a l i b r a t e				*
 ************************************************************************/
BOOL CScannedMap::TryToCalibrate()
{
BOOL bCalibrated = FALSE;
									// check if ready to calibrate
if (this->IsReadyToCalibrate())
	{
	this->CheckAntipod();
	if (this->Calibrate ())
		{
		AfxMessageBox (IDS_MAP_CALIBRATED);
		bCalibrated = TRUE;
		}
	else{
		AfxMessageBox (IDS_MAP_NOT_CALIBRATED);
		}
	}	

this->SetCalibrated (bCalibrated);

return bCalibrated;
}


/************************************************************************
 *  ScanMap.cpp					S a v e M o u s e P o s					*
 ************************************************************************/
BOOL CScannedMap::SaveMousePos (CPoint pMouse)
{
BOOL bOK = TRUE;

									// set mouse value into actual fix point
CMapFixPt* ptFixPt = this->GetFixPtPtr (m_nActCalIndex);
if (ptFixPt != NULL)
	{
	ptFixPt->SetLP(pMouse);

									// check if ready to calibrate
	bOK = this->TryToCalibrate();
	}
return bOK;
}


/************************************************************************
 *  ScanMap.cpp				A p p e n d F i x P t						*
 ************************************************************************/
BOOL CScannedMap::AppendFixPt (CMapFixPt* ptFixPt)
{
BOOL bAppended = FALSE;
short nCnt = GetFixPtCnt();

if (ptFixPt != NULL)
	{  
	m_FixPts.Add (ptFixPt);       	// add new Element  
	bAppended = (GetFixPtCnt() == nCnt+1);      
	}
return bAppended;
}

/************************************************************************
 *  ScanMap.cpp				C h a n g e F i x P t						*
 ************************************************************************/
BOOL CScannedMap::ChangeFixPt (short nIndex, CMapFixPt* ptNewFixPt)
{
BOOL bChanged = FALSE;
short nCnt = GetFixPtCnt();

CMapFixPt* ptOldFixPt = (CMapFixPt*)m_FixPts.GetAt(nIndex);
if (ptOldFixPt != NULL)
	{  
	delete ptOldFixPt;						// delete old map ptr
	}

if (ptNewFixPt != NULL)
	{  
	m_FixPts.SetAt (nIndex, ptNewFixPt);    // set new map ptr 
	bChanged = (GetFixPtCnt() == nCnt);      
	}

return bChanged;
}

/************************************************************************
 *  ScanMap.cpp					D e l e t e F i x P t					*
 ************************************************************************/
BOOL CScannedMap::DeleteFixPt (short nIndex)
{
BOOL bDeleted = FALSE;
short nCnt = GetFixPtCnt();

CMapFixPt* ptFixPt = (CMapFixPt*)m_FixPts.GetAt(nIndex);
if (ptFixPt != NULL)
	{  
	m_FixPts.RemoveAt(nIndex);
	delete ptFixPt;
	bDeleted = (GetFixPtCnt() == nCnt-1);      
	}

return bDeleted;
}

/************************************************************************
 *  ScanMap.cpp				G e t F i x P t C n t						*
 ************************************************************************/
short CScannedMap::GetFixPtCnt ()
{
return (short)m_FixPts.GetSize();
}

/************************************************************************
 *  ScanMap.cpp				G e t F i x P t P t r						*
 ************************************************************************/
CMapFixPt* CScannedMap::GetFixPtPtr (short nIndex)
{
CMapFixPt* ptFixPt=NULL;
if (nIndex>=0  &&  nIndex < GetFixPtCnt())
	{
	ptFixPt = (CMapFixPt*)m_FixPts.GetAt (nIndex);  
	} 
return ptFixPt;      
}

/************************************************************************
 *  ScanMap.cpp				W r i t e L o g F i l e						*
 ************************************************************************/
void CScannedMap::WriteLogFile()
{
short i;

CFile* ptFile = new CFile ();

if (ptFile != NULL)
	{
	char szBuffer[256];
	short nLen;
	CString szFileTitle = m_szName + ".txt";

	ptFile->Open(szFileTitle, CFile::modeCreate | CFile::modeNoInherit | CFile::modeWrite, NULL); 

	nLen = sprintf (szBuffer, "Name: %s\r\n", (LPCTSTR)m_szName);
	ptFile->Write ((const void*)szBuffer, (UINT)nLen);

	nLen = sprintf (szBuffer, "Typ: ");
	switch (m_Type)
		{
		case 	LAMBERT:	nLen += sprintf (szBuffer+nLen, "%s\r\n", "Lambert"); break;
		case	LINEAR:		nLen += sprintf (szBuffer+nLen, "%s\r\n", "Linear"); break;
		case	MERCATOR:	nLen += sprintf (szBuffer+nLen, "%s\r\n", "Mercator"); break;
		case	POLAR:		nLen += sprintf (szBuffer+nLen, "%s\r\n", "Polar"); break;
		default:  nLen += sprintf (szBuffer+nLen, "%s: %d\r\n", "Unbekannt", m_Type); break;
		}

	ptFile->Write ((const void*)szBuffer, (UINT)nLen);

	nLen = sprintf (szBuffer, "Pfad: %s\r\n", (LPCTSTR)m_szPath);
	ptFile->Write ((const void*)szBuffer, (UINT)nLen);

	nLen = sprintf (szBuffer, "Antipod: %s\r\n", m_bAntipod? "Ja":"Nein");
	ptFile->Write ((const void*)szBuffer, (UINT)nLen);

	nLen = sprintf (szBuffer, "Autosel: %s\r\n", m_bAutoSel? "Ja":"Nein");
	ptFile->Write ((const void*)szBuffer, (UINT)nLen);
	 
	nLen = sprintf (szBuffer, "BmpSizeDP: X=%ld Y=%ld\r\n\r\n", m_lSizeX_DP, m_lSizeY_DP);
	ptFile->Write ((const void*)szBuffer, (UINT)nLen);

	nLen = sprintf (szBuffer, "BmpSizeLP: X=%ld Y=%ld\r\n\r\n", m_lBmpSizeX, m_lBmpSizeY);
	ptFile->Write ((const void*)szBuffer, (UINT)nLen);


 	nLen = sprintf (szBuffer, "Anzahl der Fixpunkte: %ld\r\n", m_FixPts.GetSize());
	ptFile->Write ((const void*)szBuffer, (UINT)nLen);

	for (i=0; i<m_FixPts.GetSize(); i++)
		{
		CMapFixPt* ptFix = NULL;

		if ((ptFix = (CMapFixPt*)m_FixPts.GetAt(i)) != NULL)
			{
			nLen = sprintf (szBuffer, "Lon = %08.4f  LP_X = %ld\r\n", ptFix->GetLon(), ptFix->GetLP_X());
			ptFile->Write ((const void*)szBuffer, (UINT)nLen);

			nLen = sprintf (szBuffer, "Lat = %08.4f  LP_Y = %ld\r\n", ptFix->GetLat(), ptFix->GetLP_Y());
			ptFile->Write ((const void*)szBuffer, (UINT)nLen);
			}
	    }

	nLen = sprintf (szBuffer, "Geeicht: %s\r\n", m_bCalibrated? "Ja":"Nein");
	ptFile->Write ((const void*)szBuffer, (UINT)nLen);

	if (m_Type == LAMBERT)
	{
		nLen = sprintf (szBuffer, "KegelPol: x=%f  y=%f  z=%f\r\n\r\n", m_vKegelPol.x, m_vKegelPol.y, m_vKegelPol.z);
		ptFile->Write ((const void*)szBuffer, (UINT)nLen);
	}	

	nLen = sprintf (szBuffer, "Sichtbare Kartengrenze:\r\n");
	ptFile->Write ((const void*)szBuffer, (UINT)nLen);

	for (i=0; i<4; i++)			   // max. visible lat lon values
		{
		nLen = sprintf (szBuffer, "Lat = %08.4f  Lon = %08.4f\r\n", m_Border[i].GetLat(), m_Border[i].GetLon());
		ptFile->Write ((const void*)szBuffer, (UINT)nLen);
		}


	ptFile->Close();
	delete ptFile;
	ptFile = NULL;
	}
}


/************************************************************************
*  ScanMap.cpp    		 		 S e r i a l i z e						*
 ************************************************************************/
void CScannedMap::Serialize(CArchive& ar)
{ 
int i, nFixPtCnt, nCnt, nVersion;

if (ar.IsStoring())
	{		// TODO: add storing code here 
	nVersion = ACT_MAP_VERS;  
	ar << (WORD)nVersion;

	ar << (WORD)m_Type;
	ar << m_szName;
	ar << m_szPath;			// path to *.bmp file
	ar << m_bAntipod;
	ar << m_bAutoSel;

	ar << m_lSizeX_DP;
	ar << m_lSizeY_DP;
	ar << m_lBmpSizeX;
	ar << m_lBmpSizeY;

	ar << m_FixPts.GetSize();

	for (i=0; i<m_FixPts.GetSize(); i++)
		{
		CMapFixPt* ptFix = NULL;

		if ((ptFix = (CMapFixPt*)m_FixPts.GetAt(i)) != NULL)
			{
			ptFix->Serialize (ar);
			}
	    }

	ar << m_bCalibrated;

	switch (m_Type)
	{
	case LAMBERT:
   		ar << m_vKegelPol.x;
   		ar << m_vKegelPol.y;
   		ar << m_vKegelPol.z;
		break;
	case LINEAR:
		ar << m_bRelativ;
		ar << m_nCourse;				// relative definition of point 2
		ar << m_dDist;
		ar << (WORD)m_nDistDim;
		break;
	}


	for (i=0; i<4; i++)			   // max. visible lat lon values
		m_Border[i].Serialize(ar);  
	
	ar << (int)FLAG_CNT; 
	for (i=0; i<(int)FLAG_CNT; i++)
		ar << m_ptShowFlags[i];	// show APT, REP, INT, NDB, VOR, USR, LABEL, MAPS, ELEV, AIRSPACE

	ar << m_nSizeLine;
	ar << m_nSizeBear;

	ar << m_ColLine;
	ar << m_ColBear; 
	ar << m_ColLabel;
	ar << m_ColSel;
	ar << m_ColText;
	}
else{       // TODO: add loading code here   
	WORD	Word;
	ar >> Word;		nVersion = (short)Word;

	ar >> Word;		m_Type = (PROJECTION)Word;
	ar >> m_szName;
	ar >> m_szPath;			// path to *.bmp file
 	ar >> m_bAntipod;
 	ar >> m_bAutoSel;

	ar >> m_lSizeX_DP;
	ar >> m_lSizeY_DP;
	ar >> m_lBmpSizeX;
	ar >> m_lBmpSizeY;

	ar >> nFixPtCnt;

	DeleteArrayOf (&m_FixPts);
	for (i=0; i<nFixPtCnt; i++)
		{
		CMapFixPt* ptFix = (CMapFixPt*)new CMapFixPt ();
		if (ptFix != NULL)
			{
			ptFix->Serialize (ar);
			m_FixPts.Add (ptFix);
			}
		}

	ar >> m_bCalibrated;

	switch (m_Type)
	{
	case LAMBERT:
   		ar >> m_vKegelPol.x;
   		ar >> m_vKegelPol.y;
   		ar >> m_vKegelPol.z;
		break;
	case LINEAR:
		ar >> m_bRelativ;
		ar >> m_nCourse;				// relative definition of point 2
		ar >> m_dDist;
		ar >> Word;		m_nDistDim = (short)Word;
		break;
	}


	for (i=0; i<4; i++)			   // max. visible lat lon values
		m_Border[i].Serialize(ar);  
	
	 
	ar >> nCnt; 
	for (i=0; i<nCnt; i++)
		ar >> m_ptShowFlags[i];	// show APT, REP, INT, NDB, VOR, USR, LABEL, MAPS, ELEV, AIRSPACE

	for (i=nCnt; i<FLAG_CNT; i++)	// if nCnt < FLAG_CNT:
		m_ptShowFlags[i] = FALSE;	// set rest = FALSE 
	
	ar >> m_nSizeLine;
	ar >> m_nSizeBear;

	ar >> m_ColLine;
	ar >> m_ColBear; 
	ar >> m_ColLabel;
	ar >> m_ColSel;
	ar >> m_ColText;


//	WriteLogFile();
	}
} 
  
/////////////////////////////////////////////////////////////////////////////
// CScannedMap commands
