// TrackDoc.cpp : implementation of the CTrackDoc class
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "pf.h"

#include "InitDoc.h"
#include "DimDoc.h"  

#include "TrackDlg.h"
#include "GpsTrDlg.h"

#include "TrackDoc.h"
#include "TrackKmlDoc.h"

#include "TrackVw.h"

#ifdef _DEBUG
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

extern CInitDoc* 		ptInit;
extern CDimDoc*			ptDim;
extern CScanMapDoc*		ptScMpDoc;
extern BOOL				bDemo;

/************************************************************************
 *  TrackDoc.cpp			T i m e S c a l e T e x t X					*
 ************************************************************************/
short TimeScaleTextX (double dXval, double dXGrid, CString* ptScale)
{        
short	nLen;
char	szScale[32];							
CString	szScaleFormat;
short	nMultDaysHours, nHour, nMin, nSec;
//char	szDim[20];

nMultDaysHours = (short)(dXval / 3600);			// nMultDaysHours > 24 if track continues next day
nHour = nMultDaysHours;

while (nHour > 24) 
	nHour -= 24;								// show only up to 24 hours in X-axis
nMin = (short)((dXval - 3600.*nMultDaysHours)/60);
nSec = (short)(dXval - 3600.*nMultDaysHours - 60.*nMin);

szScaleFormat.LoadString (IDF_TIME_SCALE);
if (dXGrid < 60)
	{
	nLen = sprintf (szScale, szScaleFormat, nMin, nSec);
	}
else{
	nLen = sprintf (szScale, szScaleFormat, nHour, nMin);
	}

*ptScale = (CString)szScale;
return nLen;
}
					   

/************************************************************************
 *  TrackDoc.cpp    		 	G e t T e x t W i d t h					*
 ************************************************************************/
short GetTextWidth(CDC* pDC, CString szText)
{  
short	nTextLen;
CSize	TextSize;

TextSize = pDC->GetTextExtent(szText);
nTextLen = (short)TextSize.cx;
return nTextLen;
}




/////////////////////////////////////////////////////////////////////////////
// CTrackDoc

IMPLEMENT_DYNCREATE(CTrackDoc, CDocument)

BEGIN_MESSAGE_MAP(CTrackDoc, CDocument)
	//{{AFX_MSG_MAP(CTrackDoc)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CTrackDoc construction/destruction

CTrackDoc::CTrackDoc()
{
	// TODO: add one-time construction code here
m_bConverted = FALSE;
m_bDebug = false;			// not in Init!!
this->Init();
}

CTrackDoc::~CTrackDoc()
{
this->DeleteArrayOf (&m_TrackPts);
}

/************************************************************************
 *  TrackDoc.cpp				o p e r a t o r  = 						*
 ************************************************************************/
CTrackDoc& CTrackDoc::operator =(const CTrackDoc& doc)
{
	BOOL bOK = TRUE;
	
	this->OnNewDocument();		// calls DeleteContents
	
	_fmemcpy (&m_Header, &doc.m_Header, 1);

	this->SetPathName(doc.GetPathName());

	m_nSimulateCnt = doc.m_nSimulateCnt;
	m_bDebug = doc.m_bDebug;

	for (int i=0; i<doc.m_TrackPts.GetSize() && bOK; i++)
	{
		LPTRACKPT lpTrackPt = (LPTRACKPT)doc.m_TrackPts.GetAt(i);
		CTrackPoint TrackPt(lpTrackPt);
		bOK = this->AppendWpt (TrackPt);
	}

	return *this;
}

/************************************************************************
 *  TrackDoc.cpp				I n L e g a l A r e a 					*
 ************************************************************************/
BOOL CTrackDoc::InLegalArea()
{
BOOL	bDeleted;
short i;
double	dMinLat, dMaxLat, dMinLon, dMaxLon;

if (!bDemo) return TRUE;

dMinLat = 48.5; dMaxLat = 50.5; dMinLon = 6.5; dMaxLon = 9.5;

bDeleted = FALSE;

i=0;
while (i<m_TrackPts.GetSize())
    {
	double fLat, fLon;

	LPTRACKPT lpTrackPt = (LPTRACKPT)m_TrackPts.GetAt(i);
	CTrackPoint TrackPt(lpTrackPt);

	TrackPt.GetLatLon (&fLat, &fLon);

	if (fLat < dMinLat || fLat > dMaxLat ||
		fLon < dMinLon || fLon > dMaxLon)
		{
		TRACE ("Deleting TrackDoc:InLegalArea LPTRACKPT %d\n", i);
		delete lpTrackPt;
		m_TrackPts.RemoveAt(i);
		bDeleted = TRUE;
		}
	else{
		i++;
		}
	}

this->SetModifiedFlag (FALSE);

if (bDeleted)
	{
	if (m_TrackPts.GetSize() == 0)
		{
		this->OnNewDocument();
		}

	AfxMessageBox (IDS_LEG_AREA);
	return FALSE;
	}
return TRUE;
}

/************************************************************************
 *  TrackDoc.cpp			O n N e w D o c u m e n t					*
 ************************************************************************/
BOOL CTrackDoc::OnNewDocument()
{
	if (!CDocument::OnNewDocument())
		return FALSE;

	// TODO: add reinitialization code here
	// (SDI documents will reuse this document)
this->Init();

return TRUE;
}

/************************************************************************
 *  TrackDoc.cpp			O n O p e n D o c u m e n t					*
 ************************************************************************/
BOOL CTrackDoc::OnOpenDocument(LPCTSTR pszPathName)
{
BOOL bOpened = FALSE;

if (bDemo)
	this->SetModifiedFlag(FALSE);

if (this->SaveModified())
	{
	this->SetPathName (pszPathName);
	bOpened = CDocument::OnOpenDocument(pszPathName); 

	if (bOpened)
		{
		InLegalArea();

		if (m_bConverted)
			{
			if (this->OnSaveDocument (pszPathName))
				{
				m_bConverted = FALSE;
				if (ptInit->IsSaveSignal())
					MessageBeep(MB_OK);	
				}
			}


		ptScMpDoc->ActivateNearestMap(this);
		}
	}

return bOpened;
}

/************************************************************************
 *  TrackDoc.cpp			O n S a v e D o c u m e n t					*
 ************************************************************************/
BOOL CTrackDoc::OnSaveDocument(const char* pszPathName)
{
BOOL bSaved = FALSE;

if (bDemo)
	{
	AfxMessageBox (IDS_SAVE_DISABLED);
	return bSaved;
	}
else{
	bSaved = CDocument::OnSaveDocument(pszPathName); 
	this->SetModifiedFlag (FALSE);
	}

return bSaved;
}


/************************************************************************
 *  TrackDoc.cpp				I n i t									*
 ************************************************************************/
void CTrackDoc::Init()
{
_fmemset (&m_Header, 0, sizeof (TRACKHEADER));

CString szVers;
szVers.LoadString (IDS_TRK_VERS2);
_fstrcpy (m_Header.szVersion, (LPCTSTR)szVers);
m_Header.nAltDim = DIM_FEET;				// DIM_METER
m_Header.nSpdDim = DIM_KT;					// DIM_KMH
m_Header.nDistDim = DIM_METER;

CString szTitle;
szTitle.LoadString (IDS_NO_TITLE);
this->SetPathName(szTitle);

m_nSimulateCnt = -1;
}

/****************************************************************************
 *	TrackDoc.cpp			S e t D i m e n s i o n s						*
 ****************************************************************************/
void CTrackDoc::SetDimensions (short nAltDim, short nSpdDim, short nDistDim)
{
m_Header.nAltDim = nAltDim;
m_Header.nSpdDim = nSpdDim;
m_Header.nDistDim= nDistDim;
}


/************************************************************************
 *  TrackDoc.cpp			D e l e t e C o n t e n t s					*
 *  Called from: OnFileNew and OnFileOpen								*
 ************************************************************************/
void CTrackDoc::DeleteContents()
{
this->DeleteArrayOf (&m_TrackPts);
}

/************************************************************************
 *  TrackDoc.cpp			D e l e t e A r r a y O f 					*
 ************************************************************************/
void CTrackDoc::DeleteArrayOf (CPtrArray* ptArray)
{
int i, nEntryCnt;	
											
nEntryCnt = ptArray->GetSize();
for (i=0; i<nEntryCnt; i++)
	{
	LPTRACKPT ptEntry;
	if ((ptEntry = (LPTRACKPT)ptArray->GetAt(i)) != NULL)
		{
		delete ptEntry;				// delete original element
		}
	}
ptArray->RemoveAll();
}          

/************************************************************************
 *  TrackDoc.cpp			G e t T i m e R a n g e 					*
 ************************************************************************/
void CTrackDoc::GetTimeRange (short* ptGridX, long* ptMinX, long* ptMaxX)
{
	short	nGridX, i, nCnt, nDays;
	long	lPrevTime, lTimeMin, lTimeMax;
	CTrackPoint TrackPt;

	lPrevTime = lTimeMin = lTimeMax = 0;

	nCnt = this->GetCnt();
	nDays = 0;
	i=0;
	while (i<nCnt)
	{			
		if (this->GetTrackPointPtr(i++, &TrackPt))
		{
			if (i==1)
			{				
				lTimeMin = TrackPt.GetSec();				// get first time
				lPrevTime = lTimeMin;
			}
			else
			{		
										// compare times
				lTimeMax = TrackPt.GetSec() + (long)nDays * 24*3600;
				if (lTimeMax < lPrevTime)
					nDays++;
				
				if (nDays > 0)
				{
					TrackPt.SetHours(TrackPt.GetHours() + (unsigned char)(nDays * 24));
					lTimeMax = TrackPt.GetSec();

					//m_TrackPts.SetAt(i-1, &TrackPt);	// store changed time
					this->ChangeWpt(i-1, TrackPt);		// store changed time
				}

				lPrevTime = lTimeMax;
			}
		}
	}

								  nGridX = 1;
if ((lTimeMax-lTimeMin) >   10	) nGridX = 6;
if ((lTimeMax-lTimeMin) >   60	) nGridX = 30;
if ((lTimeMax-lTimeMin) >  300	) nGridX = 60;	// 1 Min
if ((lTimeMax-lTimeMin) >  600	) nGridX = 120;	// 2 Min
if ((lTimeMax-lTimeMin) > 1200	) nGridX = 240;	// 4 Min
if ((lTimeMax-lTimeMin) > 2400	) nGridX = 480;	// 8 Min
if ((lTimeMax-lTimeMin) > 4800	) nGridX = 600;	// 10 Min
if ((lTimeMax-lTimeMin) > 6000	) nGridX =1200;	// 20 Min

												/* rounded map borders		*/
*ptMinX = nGridX * (long)((float)lTimeMin / nGridX);
*ptMaxX = nGridX * (long)((float)lTimeMax / nGridX) + nGridX;
*ptGridX = nGridX;
}

/************************************************************************
 *  TrackDoc.cpp				S e t N a m e							*
 ************************************************************************/
void CTrackDoc::SetName (CString szName)
{
	this->SetTitle ((LPCTSTR)szName);
	CString szPath = ptInit->GetActualPath() + this->GetTitle() + ".trk";
	this->SetPathName(szPath);
}

/************************************************************************
 *  TrackDoc.cpp				S e t N a m e							*
 ************************************************************************/
void CTrackDoc::SetName (unsigned long lSystemTime)
{
	CString szName;
	time_t	ltime;
	struct	tm LogTime, *ptLogTime;

	ltime = lSystemTime;
	ptLogTime = &LogTime;
	ptLogTime = gmtime(&ltime);

	unsigned int nDay = (UINT)ptLogTime->tm_mday;
	unsigned int nMonth = (UINT)ptLogTime->tm_mon + 1;
	unsigned int nYear	= (UINT)ptLogTime->tm_year + 1900;

	szName.Format ("%04d%02d%02d", nYear, nMonth, nDay);
	this->SetName (szName);
}

/************************************************************************
 *  TrackDoc.cpp			G e t M a p B o r d e r	 					*
 ************************************************************************/
void CTrackDoc::GetMapBorder (CBorder* ptBorder)
{
short i;
double 		fLat, fLon;
short nCnt = m_TrackPts.GetSize();

for (i=0; i<nCnt; i++)
	{
	LPTRACKPT lpTrackPt = (LPTRACKPT)m_TrackPts.GetAt(i);
	CTrackPoint TrackPt(lpTrackPt);

	TrackPt.GetLatLon (&fLat, &fLon);

	if (i==0)	ptBorder->Init (fLat, fLon);
		else	ptBorder->Expand (fLat, fLon);
	}
ptBorder->Finish ();
}

/****************************************************************************
 *	TrackDoc.cpp			G e t A l t M a p B o r d e r s					*
 ****************************************************************************/										 
void CTrackDoc::GetAltMapBorders (long* ptMinX, long* ptMaxX, 
					long* ptMinY, long* ptMaxY, 
					short* ptGridX, short* ptGridY)
{
BOOL	bData = FALSE;
int i;
short	nCnt, nAltDim;
short	nGridY;
long	lAlt, lAltMin, lAltMax;

nCnt = this->GetCnt();
lAltMin = lAltMax = 0;
											// get time range for X-axis
this->GetTimeRange (ptGridX, ptMinX, ptMaxX);


											// get altitude range for Y-axis
for (i=0; i<nCnt; i++)						
	{
	CTrackPoint TrackPt;
	if (this->GetTrackPointPtr (i, &TrackPt))
		{						   
		if (TrackPt.GetAlt (&lAlt))
			{
			if (lAlt < 0)
				lAlt = 0;

			if (bData)
				{									
				if (lAlt > lAltMax) lAltMax = lAlt;
				if (lAlt < lAltMin) lAltMin = lAlt;
				}
			else{				
				bData = TRUE;				
				lAltMin = lAltMax = lAlt;				
				}
			}
		}
	}


nAltDim = this->GetAltDim();
lAltMin = (long)ptDim->ConvertDist ((double)lAltMin, nAltDim, ptDim->Altitude());
lAltMax = (long)ptDim->ConvertDist((double)lAltMax, nAltDim, ptDim->Altitude());

if (lAltMin > 0)
	lAltMin = 0;

//nGridY = 5;								// minimal alt grid
//if ((lAltMax-lAltMin) >    50	) nGridY = 10;
//if ((lAltMax-lAltMin) >   100	) nGridY = 50;
//if ((lAltMax-lAltMin) >   500	) nGridY = 100;
//if ((lAltMax-lAltMin) >  1000	) nGridY = 500;
nGridY = 500;								// minimal alt grid
if ((lAltMax-lAltMin) >  5000	) nGridY = 1000;
if ((lAltMax-lAltMin) > 10000	) nGridY = 5000;
if ((lAltMax-lAltMin) > 30000	) nGridY = 10000;

												/* rounded map borders		*/
*ptMinY = nGridY * (long)((float)lAltMin / nGridY);
*ptMaxY = nGridY * (long)((float)lAltMax / nGridY) + nGridY;
*ptGridY = nGridY;
}

/****************************************************************************
 *	TrackDoc.cpp			G e t S p e e d M a p B o r d e r s				*
 ****************************************************************************/										 
void CTrackDoc::GetSpeedMapBorders (long* ptMinX, long* ptMaxX, 
					short* ptMinY, short* ptMaxY, 
					short* ptGridX, short* ptGridY)
{          
BOOL	bData = FALSE;
int i;
short	nCnt, nSpeedDim, nPrefSpeedDim;
short	nSpeed, nSpeedMin, nSpeedMax, nGridY;

nCnt = this->GetCnt();
nSpeedMin = nSpeedMax = 0;
											// get time range for X-axis
this->GetTimeRange (ptGridX, ptMinX, ptMaxX);

											// get speed range for Y-axis
for (i=0; i<nCnt; i++)							
	{
	CTrackPoint TrackPt;
	if (this->GetTrackPointPtr (i, &TrackPt))
		{
		if (TrackPt.GetSpeed (&nSpeed))
			{
			if (nSpeed < 0)
				nSpeed = 0;

			if (bData)
				{						
				if (nSpeed > nSpeedMax) nSpeedMax = nSpeed;
				if (nSpeed < nSpeedMin) nSpeedMin = nSpeed;
				}
			else{	
				bData = TRUE;									
				nSpeedMin = nSpeedMax = nSpeed;		
				}
			}             
		}
	}

nSpeedDim = this->GetSpeedDim();												
nPrefSpeedDim = ptDim->DistDimToSpeedDim(ptDim->Distance());
nSpeedMin = (short)ptDim->ConvertSpeed ((double)nSpeedMin, nSpeedDim, nPrefSpeedDim);
nSpeedMax = (short)ptDim->ConvertSpeed((double)nSpeedMax, nSpeedDim, nPrefSpeedDim);


								  nGridY = 5;
if ((nSpeedMax-nSpeedMin) >  50	) nGridY = 10;
if ((nSpeedMax-nSpeedMin) > 100	) nGridY = 50;
if ((nSpeedMax-nSpeedMin) > 200	) nGridY = 100;
if ((nSpeedMax-nSpeedMin) > 300	) nGridY = 200;
if ((nSpeedMax-nSpeedMin) > 400	) nGridY = 300;
if ((nSpeedMax-nSpeedMin) > 500	) nGridY = 400;
if ((nSpeedMax-nSpeedMin) > 600	) nGridY = 500;

												/* rounded map borders		*/
*ptMinY = nGridY * (short)((float)nSpeedMin / nGridY);
*ptMaxY = nGridY * (short)((float)nSpeedMax / nGridY) + nGridY;
*ptGridY = nGridY;
}


/************************************************************************
 *  TrackDoc.cpp			G e t S p e e d S t r i n g					*
 *  Purpose: returns string like "89 kt"								*
 ************************************************************************/
CString CTrackDoc::GetSpeedString (short nIndex, short nDestSpeedDim)
{
	CString szSpeed, szDim;

	short nSpeed = 0;

	CTrackPoint TrackPt;
	if (this->GetTrackPointPtr (nIndex, &TrackPt))
	{						   
		if (TrackPt.GetSpeed (&nSpeed))
		{
			double fConvert;
			fConvert = ptDim->ConvertSpeed (nSpeed, this->GetSpeedDim(), nDestSpeedDim);
			nSpeed = (short)(fConvert+0.5);
		}
	}

	ptDim->GetDimCstring(&szDim, nDestSpeedDim);
	szSpeed.Format ("%4d %s", nSpeed, szDim);

	return szSpeed;
}

/************************************************************************
 *  TrackDoc.cpp			G e t A l t i t u d e						*
 *  Purpose: returns altitude of trackpoint at specified index			*
 ************************************************************************/
BOOL CTrackDoc::GetAltitude (short nIndex, short nDestAltDim, long* ptAlt)
{
	BOOL bSet = FALSE;

	CTrackPoint TrackPt;
	if (this->GetTrackPointPtr (nIndex, &TrackPt))
	{				
		if (TrackPt.GetAlt (ptAlt))
		{
			double fConvert;
			fConvert = ptDim->ConvertDist (*ptAlt, this->GetAltDim(), nDestAltDim);
			*ptAlt = (long)(fConvert+0.5);
			bSet = TRUE;
		}
	}

	return bSet;
}

/************************************************************************
 *  TrackDoc.cpp			G e t A l t i t u d e S t r i n g			*
 *  Purpose: returns string like "1500 ft"								*
 ************************************************************************/
CString CTrackDoc::GetAltitudeString (short nIndex, short nDestAltDim)
{
	CString szAlt = "---";
	long lAlt = 0;

	if (this->GetAltitude(nIndex, nDestAltDim, &lAlt))
	{
		CString szDim;
		ptDim->GetDimCstring(&szDim, nDestAltDim);
		szAlt.Format ("%ld %s", lAlt, szDim);
	}
	return szAlt;
}

/************************************************************************
 *  TrackDoc.cpp			A v e r a g e A n g l e 					*
 *  Purpose: returns angle between Angle1 and Angle2	[degree]		*
 *	Example: Angle1=330, Angle2=10, AverageAngle = 350					*
 ************************************************************************/
double CTrackDoc::AverageAngle (double Angle1, double Angle2)
{
double AngleDiff, AverageAngle;

AngleDiff =  Angle1 - Angle2;
if (AngleDiff < 0) AngleDiff *= -1;

AverageAngle = (Angle1 + Angle2) / 2;
if (AngleDiff > 180) AverageAngle += 180;

if (AverageAngle >= 360)	AverageAngle -= 360;

return AverageAngle;
}


/****************************************************************************
 *	TrackDoc.cpp			T r a c k N e e d s L a b e l 	 				*
 ****************************************************************************/
BOOL CTrackDoc::TrackNeedsLabel(short i, short nTimeGrid, long* ptLabelTime)
{
BOOL	bNeedLabel = FALSE;
long	lTrackTime;
long	lTimeDiff;
short	nCnt;
CTrackPoint TrackPt;

nCnt = this->GetCnt();

if (this->GetTrackPointPtr (i, &TrackPt))
	{
	lTrackTime = TrackPt.GetSec();

	if (i==0)
		{
		bNeedLabel = TRUE;
		(*ptLabelTime) += nTimeGrid;
		}
		
	if (i>0 && i<nCnt-1)
		{
		lTimeDiff = lTrackTime - *ptLabelTime;
		
		if (lTimeDiff > 0)
			{
			long	lFakt;
			bNeedLabel = (lTimeDiff < 3);
			
			lFakt = (lTimeDiff / nTimeGrid) + 1;
			(*ptLabelTime) += lFakt * nTimeGrid;
			}
		}
	}
	
if (i==nCnt-1)
	{
	bNeedLabel = TRUE;
	}
	
return bNeedLabel;
}

/************************************************************************
 *  TrackDoc.cpp	      		G e t C o u r s e 						*
 ************************************************************************/
BOOL CTrackDoc::GetCourse (double* ptCourse)
{
BOOL bCourse = false;
CTrackPoint ActTrackPt;
CLatLon LLPrev;
double	Average = -1;
short	nCnt = 0;
short	nMaxTextCnt = 3;

*ptCourse = Average;

short nWayNum = this->GetCnt() - 1;		// start at last recorded point
do	{
	if (nWayNum>=0)
		{
		double	fActLat, fActLon;
		if (this->GetTrackPointPtr (nWayNum, &ActTrackPt))
			{
			ActTrackPt.GetLatLon (&fActLat, &fActLon);	// get lat lon

			if (LLPrev.IsValid())
				{					// calculate course
				double TT;
				CLatLon LLAct (fActLat, fActLon);
				double dDist = LLAct.LoxoDist (LLPrev, &TT);
				if (dDist > 0)
					{
					nCnt++;
					Average = (Average == -1)? TT : AverageAngle (Average, TT);				
					}
				}
				
			LLPrev.SetLatLon (fActLat, fActLon);
			}

		nWayNum--;				// check previous point
		}
	} while (nWayNum>0 && nCnt<nMaxTextCnt);
	
bCourse = (nCnt>=nMaxTextCnt);
*ptCourse = Average;

return bCourse;
}	
	

/************************************************************************
 *  TrackDoc.cpp       		L a b e l D i r e c t i o n 				*
 ************************************************************************/
double CTrackDoc::LabelDirection (short i, BOOL* ptRightAlign)
{
double	fPrevLat, fPrevLon;
double	fActLat, fActLon;
double	fNextLat, fNextLon;
CTrackPoint ActTrackPt;
CTrackPoint	PrevTrackPt, NextTrackPt;
double	TextAngle;
double	Average = -1;


short	nCnt = this->GetCnt();

if (this->GetTrackPointPtr (i, &ActTrackPt))
	{
	ActTrackPt.GetLatLon (&fActLat, &fActLon);
	CLatLon ActLatLon (fActLat, fActLon);

	if (i>0 && i < (nCnt - 1))
		{
		double		PrevTT, NextTT;

		this->GetTrackPointPtr (i-1, &PrevTrackPt);
		this->GetTrackPointPtr (i+1, &NextTrackPt);

		PrevTrackPt.GetLatLon (&fPrevLat, &fPrevLon);
		NextTrackPt.GetLatLon (&fNextLat, &fNextLon);

		ActLatLon.LoxoDist (fPrevLat, fPrevLon, &PrevTT);
		ActLatLon.LoxoDist (fNextLat, fNextLon, &NextTT);
		
		Average = AverageAngle (PrevTT, NextTT);
		}
	else{
		short	nWayNum;
		short	nTest = 0;
		short	nStep = nCnt/100;
		BOOL	bStop = FALSE;
		double	TT;
		
		if (i==0)
			{											/* first waypoint		*/
			do	{							
				nWayNum = i+ nStep*nTest+1;
				if (nWayNum < nCnt/10)
					{
					nTest++;
					this->GetTrackPointPtr (nWayNum, &NextTrackPt);
					NextTrackPt.GetLatLon (&fNextLat, &fNextLon);
					ActLatLon.LoxoDist (fNextLat, fNextLon, &TT);
		
					Average = (Average == -1)? TT : AverageAngle (Average, TT);
					}
				else{
					bStop = TRUE;
					}
				} while (nTest < 10 && !bStop);
			}
			
		
		if (i == (nCnt - 1))			
			{											/* last waypoint		*/
			do	{							
				nWayNum = i- nStep*nTest-1;
				if (nWayNum > nCnt - nCnt/10)
					{
					nTest++;
					this->GetTrackPointPtr (nWayNum, &PrevTrackPt);
					PrevTrackPt.GetLatLon (&fPrevLat, &fPrevLon);

					ActLatLon.LoxoDist (fPrevLat, fPrevLon, &TT);

					Average = (Average == -1)? TT : AverageAngle (Average, TT);
					}
				else{
					bStop = TRUE;
					}
				} while (nTest < 10 && !bStop);
			}
		}
	}

	
if (Average>=0 && Average<180)
	{
	TextAngle = Average + 180;
	*ptRightAlign = TRUE;
	}
else{			
	TextAngle = Average - 180;
	*ptRightAlign = FALSE;
	}
	
return TextAngle;
}


/****************************************************************************
 *	TrackDoc.cpp					F i l t e r								*
 ****************************************************************************/
void CTrackDoc::Filter ()
{
CTrackPoint TrkPt;
short i, nLastID;
double	dActLat, dActLon, dLastLat, dLastLon;
BOOL	bCompare, bDeleteLast;

bCompare = FALSE;
nLastID = 0;
dLastLat = dLastLon = 0.;

i=0;
while (i < m_TrackPts.GetSize())
	{
	bDeleteLast = FALSE;
	
	if (this->GetTrackPointPtr (i, &TrkPt))
		{
		if (TrkPt.GetLatLon (&dActLat, &dActLon))
			{
			if (bCompare)
				{
				bDeleteLast = ((dActLat == dLastLat) && (dActLon == dLastLon));
				}
				
			if (bDeleteLast)
				{
				this->DeleteWpt (nLastID);
				}
			else{
				dLastLat = dActLat;
				dLastLon = dActLon;
				nLastID = i;
				bCompare = TRUE;
				i++;
				}
			}
		else{					// invalid Lat, Lon
			this->DeleteWpt (i);
			}
		}
	}
}

/****************************************************************************
 *	Trackdoc.cpp				G e t T i m e d I n d e x 	 				*
 *  Purpose: returns index of Track point nMaxMinutes before end of track	*
 ****************************************************************************/
short CTrackDoc::GetTimedIndex (short nMaxMinutes)
{
	BOOL bLast = TRUE;
	BOOL bStop = FALSE;
	long lMaxSeconds, lActSeconds;
	short i;

	for (i=this->GetCnt()-1; i>=0 && !bStop; i--)
	{							// start with last recorded point
		CTrackPoint TrackPt;
		if (this->GetTrackPointPtr (i, &TrackPt))
			{
			short nHour, nMin, nSec;
			TrackPt.GetTime(&nHour, &nMin, &nSec);
			lActSeconds = (long)nHour*3600 + (long)nMin*60 + nSec;

			if (bLast)
			{									// end of track
				lMaxSeconds = lActSeconds;
				bLast = FALSE;
			}
			else
			{	// get difference
				if (lMaxSeconds < lActSeconds)
					lMaxSeconds += 24 * 3600;	// around midnight!!

				bStop = (lMaxSeconds - lActSeconds) > (nMaxMinutes * 60);
			}
		}
	}

	return i+1;		// to compensate last i-- of loop
}

/************************************************************************
 *	TrackDoc.cpp			S e t S i m u l a t e C n t			 		*
 ************************************************************************/
void CTrackDoc::SetSimulateCnt(short nCnt)
{
	m_nSimulateCnt = nCnt;
}

/************************************************************************
 *	TrackDoc.cpp				G e t C n t		 						*
 ************************************************************************/
short CTrackDoc::GetCnt ()
{
	short nCnt;
	if (m_nSimulateCnt >=0)	nCnt = m_nSimulateCnt;
					else	nCnt = (short)m_TrackPts.GetSize();
	return nCnt;
}

/************************************************************************
 *  TrackDoc.cpp			G e t T r a c k P o i n t P t r				*
 ************************************************************************/
LPTRACKPT CTrackDoc::GetTrackPointPtr (short nIndex)
{
LPTRACKPT lpTrack=NULL;
if (nIndex>=0 && nIndex<m_TrackPts.GetSize())
	{
	lpTrack = (LPTRACKPT)m_TrackPts.GetAt (nIndex);  
	} 
return lpTrack;      
}

/************************************************************************
 *  TrackDoc.cpp			G e t T r a c k P o i n t P t r				*
 ************************************************************************/
BOOL CTrackDoc::GetTrackPointPtr (short nIndex, CTrackPoint* ptTrackPt)
{
BOOL bOK = FALSE;

if (nIndex >= 0 && nIndex < m_TrackPts.GetSize())
	{
	LPTRACKPT lpTrack = (LPTRACKPT)m_TrackPts.GetAt(nIndex);
	ptTrackPt->SetPtr (lpTrack);
	bOK = TRUE;
	}

return bOK;
}


/************************************************************************
 *  TrackDoc.cpp				A p p e n d W p t						*
 ************************************************************************/
BOOL CTrackDoc::AppendWpt (CTrackPoint TrackPt)
{
BOOL bAppended = FALSE;

LPTRACKPT lpTrackPt = TrackPt.CreatePtr();
if (lpTrackPt != NULL)
	{  
	m_TrackPts.Add (lpTrackPt);       	// add new Element  
	bAppended = TRUE;      
	}
return bAppended;
}

/************************************************************************
 *  TrackDoc.cpp				C h a n g e W p t						*
 ************************************************************************/
BOOL CTrackDoc::ChangeWpt (short nIndex, CTrackPoint TrackPt)
{
BOOL bChanged = FALSE;

if (nIndex >= 0 && nIndex < m_TrackPts.GetSize())
	{ 
	LPTRACKPT lpTrackPt = (LPTRACKPT)m_TrackPts.GetAt(nIndex);
	if (lpTrackPt != NULL)
		{  
	//	TRACE ("CTrackDoc::ChangeWpt %d\n", nIndex);

		TrackPt.GetPtr (lpTrackPt);				// copy TrackPt into memory of array
		bChanged = TRUE;      
		}
	}

return bChanged;
}

/************************************************************************
 *  TrackDoc.cpp				D e l e t e W p t						*
 ************************************************************************/
BOOL CTrackDoc::DeleteWpt (short nIndex)
{
BOOL bDeleted = FALSE;

if (nIndex >= 0 && nIndex < m_TrackPts.GetSize())
	{
	LPTRACKPT lpTrackPt = (LPTRACKPT)m_TrackPts.GetAt(nIndex);
	if (lpTrackPt != NULL)
		{  
		m_TrackPts.RemoveAt(nIndex);
		delete lpTrackPt;
		bDeleted = TRUE;      
		}
	}

return bDeleted;
}

/****************************************************************************
 *  TrackDoc.cpp				R e c e i v e N M E A 						*
 ****************************************************************************/
BOOL CTrackDoc::ReceiveNMEA (CNmea* ptNmea, CTrackPoint* ptActTrackPt, BOOL* ptValid, CFile* ptFile)
{
char		szBuffer[256];
BOOL 		bTimeOut = FALSE;
BOOL		bStop = FALSE;   
BOOL		bInvalid = FALSE;
CNmeaEvent 	NMEAData;	
short 		nRMCcnt = 0; 
BOOL		bWaitForGRMZ = TRUE;
BOOL		bStored = FALSE;  

do	{
	bTimeOut = ptNmea->GetNextNMEAEvent (&NMEAData, ptFile);
	bStop = bTimeOut;	   // evtl. test mouse down button here

	switch (NMEAData.GetFunction())
		{
		case FN_BWC:
			break;
		case FN_GLL:
			{
			long	lLen;
			NMEA_GLL GLL;
	//		SysBeep (20);
			
			NMEAData.SplitGLL (&GLL);
			lLen = sprintf (szBuffer, "\nLat=%f, Lon=%f, H=%d, M=%d, S=%f, Valid=%c", 
					GLL.fLat, GLL.fLon, GLL.nHour, GLL.nMin, GLL.fSec, GLL.cDataState);
			if (ptFile != NULL)
				ptFile->Write ((const void*)szBuffer, (UINT)lLen);
			}
			break;
			
		case FN_RMB:
			{
			long	lLen;
			NMEA_RMB RMB;
			
			NMEAData.SplitRMB (&RMB);
			
			lLen = sprintf (szBuffer, "\nValid=%c CTerr=%f Steer %c, Orig=%s, Dest=%s", 
					RMB.cDataState, RMB.fCrossTrackErr_NM, RMB.cDirToSteer, RMB.szOrigWpt, RMB.szDestWpt);
			if (ptFile != NULL)
				ptFile->Write ((const void*)szBuffer, (UINT)lLen);

			lLen = sprintf (szBuffer, "\nLat=%f, Lon=%f, Dist=%f, Bear=%f, Vel=%f, Arrival=%c", 
					RMB.fDestLat, RMB.fDestLon, RMB.fRangeToDest_NM, RMB.fBrngToDest_Dgr_T, RMB.fVelClDest_kt, RMB.cArrivalState);
			if (ptFile != NULL)
				ptFile->Write ((const void*)szBuffer, (UINT)lLen);
			}
			break;

		case FN_RMC:
			{
			long	lLen;
			NMEA_RMC RMC;
			
			NMEAData.SplitRMC (&RMC);
			
			lLen = sprintf (szBuffer, "\nH=%d M=%d S=%f, Valid=%c, Lat=%f, Lon=%f", 
					RMC.nHour, RMC.nMin, RMC.fSec, RMC.cDataState, RMC.fLat, RMC.fLon); 
			if (ptFile != NULL)
				ptFile->Write ((const void*)szBuffer, (UINT)lLen);
			
			lLen = sprintf (szBuffer, "\nGndSpd=%f, GndCrs=%f, D=%d M=%d Y=%d, Var=%f", 
					RMC.fSpeedGnd_kt, RMC.fCourseGnd_Dgr_T, RMC.nDay, RMC.nMonth, RMC.nYear, RMC.fVar);
			if (ptFile != NULL)
				ptFile->Write ((const void*)szBuffer, (UINT)lLen);
			
			ptActTrackPt->SetLatLon (RMC.fLat, RMC.fLon, RMC.bLatLon);
			ptActTrackPt->SetSpeed (RMC.fSpeedGnd_kt, RMC.bGndParams);
			ptActTrackPt->SetTime (RMC.nHour, RMC.nMin, RMC.fSec);

			if (RMC.cDataState == DS_VALID)
					nRMCcnt++;
			else	bInvalid = TRUE;  
			
			if (nRMCcnt > 2 && bWaitForGRMZ) 
				bWaitForGRMZ = FALSE;
				
			if (!bWaitForGRMZ)
				{ 						// store position and speed without altitude!
				ptActTrackPt->SetAlt (0, FALSE);
				this->SetDimensions (DIM_FEET, DIM_KT);
				bStored = this->AppendWpt (*ptActTrackPt);
				nRMCcnt = 0; 
 				}
			}
			break;
			
		case FN_RTE:
			break;
		case FN_XTE:
			break;
			
		case FN_GRM:
			{
			long	lLen;
			switch (NMEAData.GetFirstDataChar ())
				{
				case 'Z':
					{
					short nAltDim, nSpdDim;

					NMEA_GRMZ GRMZ;
					NMEAData.SplitGRMZ (&GRMZ);
					lLen = sprintf (szBuffer, "\nAlt=%d, Dim=%c, Val=%d", 
							GRMZ.nAlt, GRMZ.cDim, GRMZ.nVal);
					if (ptFile != NULL)
						ptFile->Write ((const void*)szBuffer, (UINT)lLen);
					
					
					nAltDim = (GRMZ.cDim == 'f')? DIM_FEET : DIM_METER;
			//		nSpdDim = (GRMZ.cDim == 'f')? DIM_KT : DIM_KMH;
					nSpdDim = DIM_KT;
					ptActTrackPt->SetAlt (GRMZ.nAlt, GRMZ.bAlt);
					
					if (nRMCcnt == 1)
						{
						this->SetDimensions (nAltDim, nSpdDim);
						bStored = this->AppendWpt (*ptActTrackPt);
						}
					nRMCcnt = 0; 
					}
					break;
					
				default:
					break;
				}
			}
			
		default:
			break;
		}
	} while (!bStop && !bStored && !bInvalid);

*ptValid = !bInvalid;				
return bTimeOut;			
}


/************************************************************************
 *  TrackDoc.cpp		 		 C h a n g e 							*
 ************************************************************************/
BOOL CTrackDoc::Change(CWnd* ptWnd)
{
BOOL bOK = FALSE;

CGpsDoc* ptGpsDoc = ptInit->GetGpsPtr();
CTrackDlg NMEATrackDlg(ptWnd, this, ptGpsDoc);

int RetVal = NMEATrackDlg.DoModal();
switch (RetVal)
	{
	case IDOK:  
		break;
	case IDCANCEL:					// Fertig
	//	this->UpdateAllViews(NULL);  // does nothing. TrackDoc owns no views!!!
		ptWnd->Invalidate (TRUE);	// force update after dlg disappears
 		bOK = TRUE; 
		break;
	}	

return bOK;
} 

/************************************************************************
 *  TrackDoc.cpp					O p e n								*
 ************************************************************************/
BOOL CTrackDoc::Open(CWnd* ptWnd) 
{
	// TODO: Add your control notification handler code here
BOOL bOpened = FALSE;
							// ask for new file name
//static char szFilter[] = "DeltaDoc (*.ddc) | *.ddc | Text Files (*.txt) | *.txt | All Files (*.*) | *.* ||"; 
static char BASED_CODE szFilter[] = "Aufzeichnung (*.trk)|*.trk||"; 

CFileDialog dlg(TRUE, 	  					// File Open dialog
				(LPCTSTR)"trk",				// default extension: "*.trk"
				NULL,						// no initial filename
				OFN_HIDEREADONLY |	  		// dwFlags
				OFN_OVERWRITEPROMPT,
				szFilter,
				ptWnd);						// parent window

/*
CString szPath = ptInit->GetActualPath();	// without NameExt
dlg.m_ofn.lpstrInitialDir = (LPCTSTR)szPath;

  dlg.m_ofn.lpstrTitle = "ffnen Sie eine TRK-Datei"; // sets title of the open dialog
  dlg.m_ofn.Flags |=
   OFN_FILEMUSTEXIST | // only allow existing files
   OFN_PATHMUSTEXIST | // only allow existing directories
   OFN_NOCHANGEDIR |  // !!!! DON'T CHANGE WORKING DIR
   OFN_READONLY;  // don't try to open for write
*/

int ret = dlg.DoModal();
if (ret == IDOK)
	{
	CString szNameNew = dlg.GetPathName();  
	if (this->OnOpenDocument (szNameNew))
		{
		bOpened = TRUE;
		}
	}

return bOpened;
}


/************************************************************************
 *  TrackDoc.cpp					S a v e A s							*
 ************************************************************************/
BOOL CTrackDoc::SaveAs(CWnd* ptWnd, BOOL bConfirm)
{
	// TODO: Add your control notification handler code here
BOOL bSaved=FALSE;
BOOL bDoSave;

if (bConfirm)
	{
	CString szText;
	CString szFileExt = this->GetTitle();
	szText.Format(IDF_SAVE_FILE, (LPCTSTR)szFileExt);
	bDoSave = (AfxMessageBox ((LPCTSTR)szText, MB_YESNO) == IDYES);
	}
else{
	bDoSave = TRUE;
	}

if (bDoSave)
	{							// ask for new file name
	static char szFilter[] = "Aufzeichnung (*.trk)|*.trk||"; 

	CString szInitialName;
	szInitialName = this->GetTitle();

	CFileDialog dlg(FALSE, 	  					// File SaveAs dialog
					(LPCTSTR)"trk",				// default extension "*.trk"
					(LPCTSTR)szInitialName,		// initial filename
					OFN_HIDEREADONLY |	  		// dwFlags
					OFN_OVERWRITEPROMPT,
					szFilter,
					ptWnd);						// parent window

//	CString szPath = ptInit->GetActualPath();	// without NameExt
//	dlg.m_ofn.lpstrInitialDir = (LPCTSTR)szPath;

	int ret = dlg.DoModal();
	if (ret == IDOK)
		{
		CString szPath = dlg.GetPathName();  
		this->SetPathName (szPath);

		if (this->OnSaveDocument (szPath))
			{	
			bSaved = TRUE;
			}
		}
	}

return (bSaved || !bDoSave);
}

/************************************************************************
 *  TrackDoc.cpp		 		R e a d F r o m G P S 					*
 ************************************************************************/
BOOL CTrackDoc::ReadFromGPS(CWnd* ptWnd)
{
	BOOL bOK = FALSE;


	BOOL bDebug = FALSE;
	CString szTestGPS;	// debug gps data transfer if track title starts with TestGPS
	szTestGPS.LoadString(IDS_TESTGPS);
	bDebug = (this->GetTitle().Find((LPCTSTR)szTestGPS) == 0);

	
							// old track in memory?

	if (this->GetCnt() > 1)
	{							// old track in memory!
		BOOL bNewDoc = TRUE;

		if (this->IsModified())
			bNewDoc = this->SaveAs(ptWnd, TRUE);// bConfirm = TRUE

		if (bNewDoc || bDemo)
		{							// old track is saved
			this->OnNewDocument();	// delete old track
		}
		else
		{							// saving old track was aborted
			return FALSE;		// don't get new track
		}
	}

								// start reading new track
CGpsDoc* ptGpsDoc = ptInit->GetGpsPtr();
this->SetDebug(bDebug);
CReadGpsTrackDlg GpsTrackDlg(ptWnd, this, ptGpsDoc);

int RetVal = GpsTrackDlg.DoModal();
switch (RetVal)
	{
	case IDOK:  
		this->InLegalArea();
		this->SaveAs(ptWnd);
	//	this->UpdateAllViews(NULL);  // does nothing. TrackDoc owns no views!!!
		ptWnd->Invalidate (TRUE);	// force update after dlg disappears
		bOK = TRUE;
		break;
	case IDCANCEL:					// Abbruch
		bOK = FALSE;
		break;
	}	

return bOK;
} 


/************************************************************************
 *  WayDoc.cpp  		 	 W r i t e T o K M L						*
 ************************************************************************/
BOOL CTrackDoc::WriteToKML(CWnd* ptWnd)
{  
	BOOL 	bOK = FALSE;

	this->InLegalArea();

							// ask for new file name
	static char szFilter[] = "ExportKML (*.kml)|*.kml||"; 

	CString szInitialName;
	szInitialName = this->GetTitle();
												// remove ".trk" from name
	short nDot = szInitialName.ReverseFind('.');
	if (nDot > -1)
	{
		szInitialName = szInitialName.Left(nDot);
	}
	else
	{
		CString szNoTitle;
		szNoTitle.LoadString(IDS_NO_TITLE);
		szInitialName = szNoTitle;
	}

	CString szExt(".kml");						// add ".kml"
	szInitialName += szExt;


	CFileDialog dlg(FALSE, 	  					// File SaveAs dialog
					(LPCTSTR)"kml",				// default extension "*.kml"
					(LPCTSTR)szInitialName,		// initial filename
					OFN_HIDEREADONLY |	  		// dwFlags
					OFN_OVERWRITEPROMPT,
					szFilter,
					ptWnd);						// parent window

//	CString szPath = ptInit->GetActualPath();	// without NameExt
//	dlg.m_ofn.lpstrInitialDir = (LPCTSTR)szPath;

	int ret = dlg.DoModal();
	if (ret == IDOK)
	{
		CString szBuffer;
		CString szFileTitle = dlg.GetPathName();	// full path

		CString szRouteName = dlg.GetFileName();	// EDFEEDDR.kml
		nDot = szRouteName.ReverseFind('.');
		if (nDot > -1)
			szRouteName = szRouteName.Left(nDot);

		CTrackKmlDoc* ptKmlDoc = new CTrackKmlDoc();
		if (ptKmlDoc != NULL)
		{
			if (ptKmlDoc->ConvertFromTrackDoc(this, szFileTitle))
			{
				if (ptKmlDoc->IsModified())
				{
					BOOL bConfirm=FALSE;
					if (ptKmlDoc->OnSaveDocument (ptKmlDoc->GetPathName(), bConfirm))
					{
						CString szMsg;
						szMsg.Format(IDF_FILESAVED, (LPCTSTR)szFileTitle);
						AfxMessageBox((LPCTSTR)szMsg);
					}
				}
			}

			delete ptKmlDoc;
		}
	} 

	return bOK;	
} 


/////////////////////////////////////////////////////////////////////////////
// CTrackDoc serialization

/************************************************************************
 *  TrackDoc.cpp  		 		V e r s T o H e a d e r 				*
 ************************************************************************/
void CTrackDoc::VersToHeader (char* ptHeader, short nVersion)
{      
	CString szVers;
	switch (nVersion)
	{
		case 1:	szVers.LoadString (IDS_TRK_VERS1);	break; 
	//	case 2:	szVers.LoadString (IDS_TRK_VERS2);	break; 
		case ACT_TRACK_VERS:	szVers.LoadString (IDS_TRK_VERS2);	break; 
		default: szVers.Empty();	break;
	}

	strcpy (ptHeader, (LPCTSTR)szVers);
}      

/************************************************************************
 *  TrackDoc.cpp  		 		 H e a d e r T o V e r s 				*
 ************************************************************************/
short CTrackDoc::HeaderToVers (char* ptHeader)
{  
	short nVersion = 0;
	CString szTestVers; 

	szTestVers.LoadString (IDS_TRK_VERS1); 
	if (szTestVers.Compare (ptHeader) == 0)
		nVersion = 1;

	szTestVers.LoadString (IDS_TRK_VERS2);  
	if (szTestVers.Compare (ptHeader) == 0)
		nVersion = 2;

	return nVersion;
} 

/************************************************************************
 *  TrackDoc.cpp  		 		C a l c T o t a l D i s t				*
 *  Purpose: Calc dist btn llPrev and actTrkPt and add it to ptLastDist	*
 ************************************************************************/
BOOL CTrackDoc::CalcTotalDist(CTrackPoint& actTrkPt, CLatLon& llPrev, double* ptLastDist)
{
					// calculate dist
	double dLat, dLon, dPrevTT, dDist_NM;

	actTrkPt.GetLatLon(&dLat, &dLon);
	if(actTrkPt.IsStart())
	{
		(*ptLastDist) = 0;
	}
	else
	{
		CLatLon LLAct (dLat, dLon);
		dDist_NM = LLAct.LoxoDist (llPrev, &dPrevTT);

		(*ptLastDist) += ptDim->ConvertDist (dDist_NM, DIM_NM, DIM_METER);
	}

	return TRUE;
}

/************************************************************************
 *  TrackDoc.cpp  		 		S e r i a l i z e						*
 ************************************************************************/
void CTrackDoc::Serialize(CArchive& ar)
{
	short i, nVersion;

	if (ar.IsStoring())
	{
			// TODO: add storing code here
		this->VersToHeader (m_Header.szVersion, ACT_TRACK_VERS);	
		ar.Write (&m_Header, MAX_HEADER_SIZE);    

		short nCnt = this->GetCnt();
		for (i=0; i<nCnt; i++)
		{
			CTrackPoint TrackPt;
			if (this->GetTrackPointPtr (i, &TrackPt))
			{
				TrackPt.Serialize (ar, ACT_TRACK_VERS); 
			}
		}
	}
	else
	{
		// TODO: add loading code here
		this->DeleteArrayOf (&m_TrackPts);

		ar.Read (&m_Header, sizeof (TRACKHEADER));
		nVersion = this->HeaderToVers (m_Header.szVersion);  

		if (nVersion < ACT_TRACK_VERS)
		{	  
			CString szText;
			szText.Format(IDF_CONVERT_TR, (LPCTSTR)this->GetTitle());
			if (AfxMessageBox ((LPCTSTR)szText, MB_YESNO) == IDYES) 
			{ 
				m_bConverted = TRUE; 
			}
			else
			{ 
        		AfxMessageBox (IDS_BAD_TK_VERS);
        		return;
        	}
        }

		CFile* ptFile = ar.GetFile();
		DWORD	dwLength = (DWORD)ptFile->GetLength();
		  
		DWORD dwBlockSize;  			// get number of entries:
		switch (nVersion)
			{
			case 1:	dwBlockSize = sizeof (TRACKTYPE1);	break;
			case ACT_TRACK_VERS:	dwBlockSize = sizeof (TRACKTYPE);	break;
			}
		short nCnt = (short)((dwLength - MAX_HEADER_SIZE) / dwBlockSize);
		

		CLatLon	llPrev;			// used for convertion only
		double	dTotalDist=0.;
		BOOL	bValid = TRUE;

		for (i=0; i<nCnt; i++)
		{
			CTrackPoint TrackPt; 

			TrackPt.Serialize (ar, nVersion); 
			
			if (nVersion == 1)
			{				// calc distance from start to each track point	

				if (i==0)
					m_Header.nDistDim = DIM_METER;
				else							
					bValid = CalcTotalDist(TrackPt, llPrev, &dTotalDist);


				TrackPt.SetDist((long)dTotalDist, bValid);		// dist from start to this TrackPt

				double dLat, dLon;					// store llPrev for next call of CalcTotalDist
				TrackPt.GetLatLon(&dLat, &dLon);
				llPrev.SetLatLon(dLat, dLon);
			}

			this->AppendWpt (TrackPt);
		} 
	}
}

/////////////////////////////////////////////////////////////////////////////
// CTrackDoc diagnostics

#ifdef _DEBUG
void CTrackDoc::AssertValid() const
{
	CDocument::AssertValid();
}

void CTrackDoc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CTrackDoc commands
