// waydoc.cpp : implementation of the CWayDoc class
// (c) Copyright Softwareentwicklung Heinz Ldert 2008
// http://www.preflight.de

#include "stdafx.h"
#include "math.h"					// for fabs()
#include "pf.h"

#include "MainFrm.h"
#include "InitDoc.h"
#include "MetDoc.h"
#include "PlaneDoc.h"
#include "Calc.h"			// this statement must be after PlaneDoc.h

#include "FlPlDoc.h"

#include "ElevDoc.h"		
#include "WayDoc.h"
#include "WayKmlDoc.h"

#include "WayDlg.h"
#include "GpsRdDlg.h"
#include "GpsWrDlg.h"
#include "RteView.h"

#ifdef _DEBUG3
#undef THIS_FILE
static char BASED_CODE THIS_FILE[] = __FILE__;
#endif

extern CInitDoc* 	ptInit;  
extern CDimDoc*		ptDim;            
extern CMetDoc*			ptMet; 
extern CPlaneDoc*		ptPlaneDoc;

extern CFlightPlanDoc*	ptFlPlDoc;
extern CScanMapDoc*		ptScMpDoc;
extern CElevDoc*		ptElevDoc;
extern CElevArrayDoc*	ptElevArrayDoc;
extern BOOL	bDemo;


/////////////////////////////////////////////////////////////////////////////
// CWayDoc

IMPLEMENT_DYNCREATE(CWayDoc, CDocument)

BEGIN_MESSAGE_MAP(CWayDoc, CDocument)
	//{{AFX_MSG_MAP(CWayDoc)
	ON_COMMAND(ID_FILE_SAVE, OnFileSave)
	ON_COMMAND(ID_FILE_SAVE_AS, OnFileSaveAs)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CWayDoc construction/destruction

CWayDoc::CWayDoc()
{
	// TODO: add one-time construction code here
m_bConverted	= FALSE;     
m_nActIndex		= -1; 
m_bAddHomeBase	= TRUE;		// call NoHomeBase() before OnNewDocument
							// if empty doc is required!

m_ptCalculation = new CCalculate (this);


this->SetFlPlManual(FALSE);

this->SetLevel("VFR");
this->SetRoute("");
this->SetAlternate("", 1);
this->SetAlternate("", 2);
this->SetOtherInfo("");
}

CWayDoc::~CWayDoc()
{
this->DeleteArrayOf (&m_WayPts);
delete m_ptCalculation;
}

/************************************************************************
 *  WayDoc.cpp					o p e r a t o r  = 						*
 ************************************************************************/
CWayDoc& CWayDoc::operator =(const CWayDoc& doc)
{
	BOOL bOK = TRUE;

	this->NoHomeBase();

	this->OnNewDocument();		// calls DeleteContents
	
	_fmemcpy (&m_Header, &doc.m_Header, 1);
	_fmemcpy (&m_FlPlInfo, &doc.m_FlPlInfo, 1);
	m_nActIndex		= doc.m_nActIndex;
 	m_bConverted	= doc.m_bConverted; 
	m_bAddHomeBase	= doc.m_bAddHomeBase;

	this->SetPathName(doc.GetPathName());

	for (int i=0; i<doc.m_WayPts.GetSize() && bOK; i++)
	{
		LPWAY lpWayPt = (LPWAY)doc.m_WayPts.GetAt(i);
		CWayPoint WayPt(lpWayPt);
		bOK = this->AppendWpt (WayPt);
	}

	m_Alternate1 = doc.m_Alternate1;
	m_Alternate2 = doc.m_Alternate2;

	return *this;
}

/************************************************************************
 *  WayDoc.cpp				D e l e t e A r r a y O f 					*
 ************************************************************************/
void CWayDoc::DeleteArrayOf (CPtrArray* ptArray)
{
int i, nEntryCnt;	
											
nEntryCnt = ptArray->GetSize();
for (i=0; i<nEntryCnt; i++)
	{
	LPWAY ptEntry;
	if ((ptEntry = (LPWAY)ptArray->GetAt(i)) != NULL)
		{
		TRACE ("Deleting WayDoc LPWAY %d\n", i);
		delete ptEntry;				// delete original element
		}
	}
ptArray->RemoveAll();
}          

/************************************************************************
 *  WayDoc.cpp				G e t T i m e R a n g e 					*
 ************************************************************************/
void CWayDoc::GetTimeRange (short* ptGridX, long* ptMinX, long* ptMaxX)
{
short	nGridX;
long	lTimeMin, lTimeMax;

lTimeMin = lTimeMax = 0;

CONSUMTYPE* ptC = m_ptCalculation->GetConsPtr();
lTimeMin = 0;
lTimeMax = (long)ptC->fEnRoute_min * 60;


nGridX = 10;
if ((lTimeMax-lTimeMin) >   1 * 60) nGridX =   30;	// 30 sec
if ((lTimeMax-lTimeMin) >   5 * 60) nGridX =  120;	// 2 Min
if ((lTimeMax-lTimeMin) >  10 * 60) nGridX =  300;	// 5 Min
if ((lTimeMax-lTimeMin) >  30 * 60) nGridX =  600;	// 10 Min
if ((lTimeMax-lTimeMin) >  60 * 60) nGridX = 1200;	// 20 Min
if ((lTimeMax-lTimeMin) > 120 * 60) nGridX = 2400;	// 40 Min
if ((lTimeMax-lTimeMin) > 240 * 60) nGridX = 3600;	// 60 Min
if ((lTimeMax-lTimeMin) > 360 * 60) nGridX = 7200;	// 120 Min (2h)
if ((lTimeMax-lTimeMin) > 720 * 60) nGridX =14400;	// 240 Min (4h)
if ((lTimeMax-lTimeMin) >1440 * 60) nGridX =28800;	// 480 Min (8h)

												/* rounded map borders		*/
*ptMinX = nGridX * (long)((float)lTimeMin / nGridX);
*ptMaxX = nGridX * (long)((float)lTimeMax / nGridX) + nGridX;
*ptGridX = nGridX;
}

/****************************************************************************
 *	WayDoc.cpp			G e t E l e v M a p B o r d e r s					*
 ****************************************************************************/										 
void CWayDoc::GetElevMapBorders (long* ptMinX, long* ptMaxX, 
					long* ptMinY, long* ptMaxY, 
					short* ptGridX, short* ptGridY)
{
BOOL	bData = FALSE;
int i;
short	nCnt, nAltDim;
short	nGridY;
long	lAlt, lAltMin, lAltMax;

nCnt = this->GetWayPointCnt();
lAltMin = lAltMax = 0;
											// get time range for X-axis
this->GetTimeRange (ptGridX, ptMinX, ptMaxX);


											// get altitude range for Y-axis
for (i=0; i<nCnt; i++)						
	{
	CWayPoint WayPt;
	if (this->GetWayPointPtr (i, &WayPt))
		{						   
		lAlt = WayPt.GetActAlt_ft ();
		if (bData)
			{									
			if (lAlt > lAltMax) lAltMax = lAlt;
			if (lAlt < lAltMin) lAltMin = lAlt;
			}
		else{				
			bData = TRUE;				
			lAltMin = lAltMax = lAlt;				
			}
		}
	}


nAltDim = DIM_FEET;
lAltMin = (long)ptDim->ConvertDist ((double)lAltMin, nAltDim, ptDim->Altitude());
lAltMax = (long)ptDim->ConvertDist((double)lAltMax, nAltDim, ptDim->Altitude());

if (lAltMin > 0)
	lAltMin = 0;

//nGridY = 5;								// minimal alt grid
//if ((lAltMax-lAltMin) >    50	) nGridY = 10;
//if ((lAltMax-lAltMin) >   100	) nGridY = 50;
//if ((lAltMax-lAltMin) >   500	) nGridY = 100;
//if ((lAltMax-lAltMin) >  1000	) nGridY = 500;
nGridY = 500;								// minimal alt grid
if ((lAltMax-lAltMin) >  5000	) nGridY = 1000;
if ((lAltMax-lAltMin) > 10000	) nGridY = 5000;
if ((lAltMax-lAltMin) > 30000	) nGridY = 10000;

												/* rounded map borders		*/
*ptMinY = nGridY * (long)((float)lAltMin / nGridY);
*ptMaxY = nGridY * (long)((float)lAltMax / nGridY) + nGridY;
*ptGridY = nGridY;
}


/************************************************************************
 *  WayDoc.cpp				I n s e r t O r t h o						*
 ************************************************************************/
BOOL CWayDoc::InsertOrtho ()
{
BOOL bInserted;

short nOldCnt = m_WayPts.GetSize();
m_ptCalculation->InsertOrtho ();
if (bDemo) this->SetModifiedFlag(FALSE);

bInserted = (m_WayPts.GetSize() > nOldCnt);
return bInserted;
}

/************************************************************************
 *  WayDoc.cpp				I n s e r t S e m i							*
 ************************************************************************/
void CWayDoc::InsertSemi ()
{
m_ptCalculation->InsertSemi ();			// changing altitudes
if (bDemo) this->SetModifiedFlag(FALSE);
}

/************************************************************************
 *  WayDoc.cpp				I n s e r t U p D o w n						*
 ************************************************************************/
BOOL CWayDoc::InsertUpDown ()
{
BOOL bChanged = FALSE;

if (ptPlaneDoc != NULL)
	{
	CPlane	ActPlane;
	short nIndex = ptPlaneDoc->GetActIndex();
	if (ptPlaneDoc->GetPlanePtr(nIndex, &ActPlane))
		{									/* do prepare calculations		*/
		m_ptCalculation->SetPlaneSpeed (ActPlane);

		if (this->IsElev())
			this->DoCalcElev();			// may have changed ActAlt

		if (this->IsSemi())
			m_ptCalculation->InsertSemi ();		// get actual heights before
													// calculating up down points
		bChanged = m_ptCalculation->InsertUpDown (&ActPlane, ptMet);
		}
	}


if (bDemo) this->SetModifiedFlag(FALSE);

return bChanged;
}

/************************************************************************
 *  WayDoc.cpp				D e l e t e U p D o w n						*
 ************************************************************************/
void CWayDoc::DeleteUpDown()
{
m_ptCalculation->DeleteUpDown ();
if (bDemo) this->SetModifiedFlag(FALSE);
}


/************************************************************************
 *  WayDoc.cpp				D e l e t e O r t h o						*
 ************************************************************************/
void CWayDoc::DeleteOrtho()
{
m_ptCalculation->DeleteOrtho ();
if (bDemo) this->SetModifiedFlag(FALSE);
}

/************************************************************************
 *  WayDoc.cpp				CreateAndGetBearList						*
 ************************************************************************/
BOOL CWayDoc::CreateAndGetBearList (CPtrArray* ptList)
{
	BOOL	bCreated = FALSE;

	for (int i=0; i<m_WayPts.GetSize(); i++)	 // for all legs:
    {
		CWayPoint WayPt;
		if (this->GetWayPointPtr (i, &WayPt))
		{
			if (WayPt.IsUpDown())
			{
				BEARINGUPDATE* ptBU = new BEARINGUPDATE;

				memset (ptBU, 0, sizeof (BEARINGUPDATE));
				ptBU->nWptIndex = i;

 				for (int j=0; j<2; j++)
				{
					if (WayPt.HasBearing (j))
					{
						CString szNav;
						if (WayPt.GetBearNameAndCat (j, &szNav, &ptBU->Facility[j].nCategory))
						{
							strcpy (ptBU->Facility[j].szName, (LPCTSTR)szNav);
						}
					}
				}

				ptList->Add(ptBU);
				bCreated = TRUE;
			}
		} // if (GetWayPointPtr...
	} // for all legs

	return bCreated;
}

/************************************************************************
 *  WayDoc.cpp				SetAndRemoveBearList						*
 ************************************************************************/
BOOL CWayDoc::SetAndRemoveBearList(CPtrArray* ptList)
{
	BOOL	bChanged = FALSE;
	short	nBearListInd = 0;

	for (int i=0; i<m_WayPts.GetSize(); i++)	 // for all legs:
    {
		BOOL	bUpdated = FALSE;
		CWayPoint WayPt;
		if (this->GetWayPointPtr (i, &WayPt))
		{
			if (WayPt.IsUpDown() && (nBearListInd < ptList->GetSize()))
			{
				BEARINGUPDATE* ptBU = (BEARINGUPDATE*)ptList->GetAt(nBearListInd++);
				if (ptBU != NULL)
				{
					if (ptBU->nWptIndex == i)
					{
						for (int j=0; j<2; j++)
						{
							if (*ptBU->Facility[j].szName > 0)
							{			// use NameList to get Nav from szNav...
								CLocation NavLoc;
								CString szNav (ptBU->Facility[j].szName);
								short	nCat = ptBU->Facility[j].nCategory;
								if (WayPt.NameCatToLoc (szNav, nCat, &NavLoc))
								{
									WayPt.CalculateBearing (j, NavLoc);
									bUpdated = TRUE;
								}
							}
						}
					}
				delete ptBU;
				}				

				if (bUpdated)
				{
					if (this->ChangeWpt(i, WayPt))
						bChanged = TRUE;
				}
			}
		} // if (GetWayPointPtr...
	} // for all legs
				
	ptList->RemoveAll();

	return bChanged;
}


/************************************************************************
 *  WayDoc.cpp				U p d a t e U p D o w n 					*
 ************************************************************************/
void CWayDoc::UpdateUpDown()
{
if (m_ptCalculation != NULL)
	{
	if (this->IsUpDown())
		{				// update updown calculation
		short nOldActIndex = m_nActIndex;
	
		CPtrArray	Bearings;		// array of BEARINGUPDATE
		BOOL bBU = this->CreateAndGetBearList (&Bearings);

		this->DeleteUpDown();
		if (!this->InsertUpDown())
			this->SetHeaderFlag (WY_UPDOWN, FALSE);

		m_nActIndex = nOldActIndex;
		if (bBU) this->SetAndRemoveBearList(&Bearings);
		}
	}
}

/************************************************************************
 *  WayDoc.cpp				U p d a t e O r t h o	 					*
 ************************************************************************/
void CWayDoc::UpdateOrtho()
{
if (m_ptCalculation != NULL)
	{
	if (this->IsOrtho())
		{				// update updown calculation
		this->DeleteOrtho();
		if (!this->InsertOrtho())
			this->SetHeaderFlag (WY_ORTHO, FALSE);
		}
	}
}

/************************************************************************
 *  WayDoc.cpp				I n L e g a l A r e a 						*
 ************************************************************************/
BOOL CWayDoc::InLegalArea()
{
BOOL	bDeleted;
short i;
double	dMinLat, dMaxLat, dMinLon, dMaxLon;

if (!bDemo) return TRUE;

dMinLat = 48.5; dMaxLat = 50.5; dMinLon = 6.5; dMaxLon = 9.5;
bDeleted = FALSE;

i=0;
while (i<m_WayPts.GetSize())
    {
	double fLat, fLon;

	LPWAY lpWayPt = (LPWAY)m_WayPts.GetAt(i);
	CWayPoint WayPt(lpWayPt);
	fLat = WayPt.GetLat();
	fLon = WayPt.GetLon();
	if (fLat < dMinLat || fLat > dMaxLat ||
		fLon < dMinLon || fLon > dMaxLon)
		{
		TRACE ("Deleting WayDoc:InLegalArea LPWAY %d\n", i);
		delete lpWayPt;
		m_WayPts.RemoveAt(i);
		bDeleted = TRUE;
		}
	else{
		i++;
		}
	}

this->SetModifiedFlag (FALSE);

if (bDeleted)
	{
	if (m_WayPts.GetSize() == 0)
		{
		this->OnNewDocument();
		}

	AfxMessageBox (IDS_LEG_AREA);
	return FALSE;
	}
return TRUE;
}


/************************************************************************
 *  WayDoc.cpp  		 	C h e c k P l a n n e d A l t				*
 *  Purpose: compare planned alt with elevation of stored in waypoints	*
 ************************************************************************/
void CWayDoc::CheckPlannedAlt()
{
	for (int i=1; i<m_WayPts.GetSize(); i++)	// don't check start elevation
    {
		LPWAY lpWayPt = (LPWAY)m_WayPts.GetAt(i);
		CWayPoint WayPt(lpWayPt);

		short nAltDim = WayPt.GetAltDim();
		long lReqPlannedAlt = WayPt.GetSafeAlt_MSL(nAltDim);
		if (lReqPlannedAlt != NO_ALT)
		{
			long lPlannedAlt = WayPt.GetPlannedAlt();
			if (lPlannedAlt < lReqPlannedAlt)
			{
								// specified planned altitude is too low!!
				CString szMsg;
				CString szDim;
				ptDim->GetDimCstring(&szDim, nAltDim);

				long lElevFromLoc;		// try to get elevation from location
				if (((CLocation)WayPt).GetAltitude(&lElevFromLoc))
				{
										// convert waypoint elevation into altitude dimension
					short nElevDim = ((CLocation)WayPt).GetElevDim();
					double fConvert;				
					fConvert = ptDim->ConvertDist((double)lElevFromLoc, nElevDim, nAltDim);
					lElevFromLoc = (long)(fConvert + 0.5);	


				// "Soll die Flughhe am Ort %s (Elev. %ld %s) von %ld auf %ld %s erhht werden?"
					szMsg.Format (IDF_ASK_CHANGE_ALT, (LPCTSTR)WayPt.GetNameForRte(),
														lElevFromLoc, (LPCTSTR)szDim,
														lPlannedAlt, lReqPlannedAlt, (LPCTSTR)szDim);
					int iRet = AfxMessageBox (szMsg, MB_YESNO);
					if (iRet == IDYES)
					{
						WayPt.SetPlannedAlt (lReqPlannedAlt);
						this->ChangeWpt (i, WayPt);
					}
				}
			}
		}
	}
}



/************************************************************************
 *  WayDoc.cpp  			C h e c k A l t O n W p t s					*
 *  Purpose: compare planned altitude of all waypoints with elevation	*
 *	found in ElevArray. If planned alt < elevation,	increase planned alt*
 ************************************************************************/
BOOL CWayDoc::CheckAltOnWpts()
{
	BOOL bIncreased = FALSE;

	CWayPoint WptAct;
	long	lActPlAlt;
	short	nPlAltDim;

	long lSafeAlt_m = (long)(ptDim->ConvertDist (500., DIM_FEET, DIM_METER) + 0.5);


	for (int i=0; i<m_WayPts.GetSize(); i++)	
    {
		LPWAY lpWayPt = (LPWAY)m_WayPts.GetAt(i);
		WptAct.SetPtr(lpWayPt);
		lActPlAlt = WptAct.GetPlannedAlt();
		nPlAltDim = WptAct.GetAltDim();

							// get elev at WptAct from 1km grid
		short nElevFrGrid_m;
		if (ptElevArrayDoc != NULL)
		{
			if (ptElevArrayDoc->GetElev_m((CLatLon)WptAct, &nElevFrGrid_m))
			{	
				long lElevFrGrid = (long)(ptDim->ConvertDist(nElevFrGrid_m, DIM_METER, nPlAltDim) + 0.5);
				long lRequiredPlAlt = NO_ALT;

				if (i==0 || i == m_WayPts.GetSize()-1)
				{				// start or land location
					if (lActPlAlt < lElevFrGrid)
						lRequiredPlAlt = lElevFrGrid;
				}
				else
				{				// other waypoint of route
					if (lActPlAlt < lElevFrGrid + lSafeAlt_m)
						lRequiredPlAlt = lElevFrGrid + lSafeAlt_m;
				}


				if (lRequiredPlAlt != NO_ALT)
				{					// increase planned alt of WptAct!
			//		AfxMessageBox ("increase planned alt of WptAct!");

					WptAct.SetPlannedAlt(lRequiredPlAlt);
					this->ChangeWpt(i, WptAct);
					bIncreased = TRUE;
				}
			} // good lElevFrGrid
		} // good ptElevArrayDoc
	} // next leg

	return bIncreased;
}


/************************************************************************
 *  WayDoc.cpp  		 	C h e c k A l t O n R t e					*
 *  Purpose: on every leg:												*
 *	compare planned alt and required alt at 50 (nCheckPts) positions.	*
 *	where (required alt - planned alt) is maximum, insert new WayPoint	*
 *	with planned alt = required alt.									*
 *	Repeat this procedure, until no more WayPoints are inserted			*
 ************************************************************************/
BOOL CWayDoc::CheckAltOnRte(BOOL bForAutoRoute)
{
	BOOL bInserted = FALSE;

	this->CheckAltOnWpts();	// make sure, that original wpts are height enough!	

	BOOL bNewLoop;
	do
	{
		bNewLoop = FALSE;
		CWayPoint WptOld, WptAct;
		long lOldPlAlt_m, lActPlAlt_m;

		long lSafeAlt_m = (long)(ptDim->ConvertDist (500., DIM_FEET, DIM_METER) + 0.5);

		for (int i=0; i<m_WayPts.GetSize(); i++)	
		{
			LPWAY lpWayPt = (LPWAY)m_WayPts.GetAt(i);
			WptAct.SetPtr(lpWayPt);
			lActPlAlt_m = WptAct.GetPlannedAlt(DIM_METER);

			if (i>0)
			{

				double dCheckDist_NM;
				double dLimitToStartWpt_NM = 5;
				double dLimitToLandWpt_NM = 5;
				short const nCheckPts = 50;		// resolution to check elev
				
								// get loxodrome dist btn. WptOld and WptAct
				double dLegDist_NM = WptOld.LoxoDist((CLatLon)WptAct);
				double dStepDist_NM = dLegDist_NM / nCheckPts;


								// get LocX where elev correction is maximal
				CLatLon LocX;
				double dDistToLocX_NM;
				long	lElevAtLocX_m;
				long lElevCorrection_m=0;

				for (int j=1; j<nCheckPts-1; j++)
				{				// don't check first and last point of leg again (see CheckAltOnWpts)
					dCheckDist_NM = j * dStepDist_NM;

					CLatLon llCheck;		// get position to check
					llCheck = CLatLon::PointOnLoxoLegAtDist (dCheckDist_NM, (CLatLon)WptOld, WptAct);

										// get elev at llCheck from 1km grid
					short nElevFrGrid_m;
					if (ptElevArrayDoc != NULL)
					{
						if (ptElevArrayDoc->GetElev_m(llCheck, &nElevFrGrid_m))
						{					
							long lPlannedAlt_m = (long)(dCheckDist_NM * 
								(lActPlAlt_m - lOldPlAlt_m)/dLegDist_NM + lOldPlAlt_m + 0.5);
							long lRequiredPlAlt_m = nElevFrGrid_m + lSafeAlt_m;
							long lActCorrection_m = lRequiredPlAlt_m - lPlannedAlt_m;

							if (lActCorrection_m			> lElevCorrection_m &&
											dCheckDist_NM	> dLimitToStartWpt_NM &&
								(dLegDist_NM-dCheckDist_NM)	> dLimitToLandWpt_NM)
							{			// save maximum elev correction and position
								lElevCorrection_m = lActCorrection_m;
								LocX = llCheck;
								lElevAtLocX_m = nElevFrGrid_m;
								dDistToLocX_NM = dCheckDist_NM;
							}
						}	// good nElevFrGrid_m
					} // good ptElevArrayDoc 
				} // 	next checkPt of leg...


				if (lElevCorrection_m > 0)
				{		// LocX with max elev correction is defined!
						
									// insert WptX with planned alt=elevX + 500ft

					CWayPoint NavWpt;  
					double dConvert;
													/* get basis data of new loc		*/
					this->GetWayPointPtr (i-1, &NavWpt);
					NavWpt.DeleteBearing(0);		/* delete 1. bearing				*/
					NavWpt.DeleteBearing(1);		/* delete 2. bearing				*/
					
					NavWpt.SetLatLon(LocX.GetLat(), LocX.GetLon());
					NavWpt.SetAutoRoute(bForAutoRoute);
					NavWpt.SetElev(true);
					NavWpt.SetCategory(WP_USER);

					CString szLatLon;
					NavWpt.CreateLatLonStr(&szLatLon);
					CString szName("EL_");
					szName += szLatLon;
					NavWpt.SetName(szName);

					NavWpt.SetIndicator("");
					NavWpt.SetFrequency(NO_FREQ);
					NavWpt.SetRange(0);

					dConvert = ptDim->ConvertDist(lElevAtLocX_m, DIM_METER, NavWpt.GetElevDim());
					NavWpt.SetAltitude((long)(dConvert + 0.5));

					long lPlAlt = NavWpt.GetSafeAlt_MSL(NavWpt.GetAltDim());
					NavWpt.SetPlannedAlt(lPlAlt);

					short nVar = (short)(dDistToLocX_NM * 
						(WptAct.GetVariation() - WptOld.GetVariation())/dLegDist_NM + 
						WptOld.GetVariation() + 0.5);
					NavWpt.SetVariation(nVar);

					if (this->InsertWpt(i, NavWpt))
					{
						i++;
						bInserted = TRUE;
						bNewLoop = TRUE;
					}
				}


			} // i > 0
			
			WptOld = WptAct;
			lOldPlAlt_m = lActPlAlt_m;
		} // next leg
	} while (bNewLoop);

	return bInserted;
}


/************************************************************************
 *  WayDoc.cpp  		 O p t i m i z e  P l a n n e d A l t			*
 *  Purpose: find Wpt order, where planned alt is V-shaped. If middle 	*
 *	waypoint is of type WT_ELEV, then delete it, else increase pl. alt	*
 ************************************************************************/
BOOL CWayDoc::OptimizePlannedAlt()
{
	BOOL bOptimized = FALSE;
	BOOL bDeleted;

	CWayPoint Wpt0, Wpt1, WptAct;
	long lPlAlt0_ft, lPlAlt1_ft, lPlAltAct_ft;

	for (int i=0; i<m_WayPts.GetSize(); i++)	
	{
		bDeleted = FALSE;
		LPWAY lpWayPt = (LPWAY)m_WayPts.GetAt(i);
		WptAct.SetPtr(lpWayPt);
		lPlAltAct_ft = WptAct.GetPlannedAlt(DIM_FEET);

		if (i>1)
		{
			if (lPlAlt0_ft > lPlAlt1_ft &&
				lPlAlt1_ft < lPlAltAct_ft)
			{			// V - shape flight
				if (Wpt1.IsElev())
				{
					i--;	// decrease loop index !!
					this->DeleteWpt(i);	// deletes Wpt1
					bDeleted = TRUE;
				}
				else
				{						// get loxodrome dist of both legs
					double dLegDistA = Wpt0.LoxoDist((CLatLon)Wpt1);
					double dLegDistB = Wpt1.LoxoDist((CLatLon)WptAct);

					long lPlAlt_ft = (long)(dLegDistA * (lPlAltAct_ft - lPlAlt0_ft)/
									(dLegDistA + dLegDistB) + lPlAlt0_ft);

					double fConvert;
					fConvert = ptDim->ConvertDist (lPlAlt_ft, DIM_FEET, ptDim->Altitude());
					Wpt1.SetPlannedAlt( (long)(fConvert + 0.5));
					Wpt1.SetAltDim (ptDim->Altitude());

					this->ChangeWpt(i-1, Wpt1);
				}
				bOptimized = TRUE;
			}
		} // i > 1
		
		
		if (i>0 && !bDeleted)
		{
			Wpt0 = Wpt1;
			lPlAlt0_ft = lPlAlt1_ft;
		}

		Wpt1 = WptAct;
		lPlAlt1_ft = lPlAltAct_ft;

	} // next leg

	return bOptimized;
}


/************************************************************************
 *  WayDoc.cpp					U p d a t e								*
 ************************************************************************/
BOOL CWayDoc::Update()
{
BOOL	bDoSave = FALSE;
short	i, j;
BOOL	bChanged, bUpdated;

bChanged = FALSE;

for (i=0; i<m_WayPts.GetSize(); i++)	 // for all legs:
    {
	CWayPoint WayPt;					
	CString szNav;
	bUpdated = FALSE;

	if (this->GetWayPointPtr (i, &WayPt))
		{
		CString szNameForRte = WayPt.GetNameForRte ();
		CLocation StoredLoc;
		if (WayPt.NameCatToLoc (szNameForRte, WayPt.GetCategory(), &StoredLoc))
			{
			if (!StoredLoc.IsEqual((CLocation)WayPt))
				{						// ask user WHICH_KOORD to use
				int		iRet;

				CString szText;
				szText.Format(IDF_WHICH_KOORD, (LPCTSTR)WayPt.GetName());
				iRet = AfxMessageBox ((LPCTSTR)szText, MB_YESNO);
				if (iRet == IDYES)
					{					// use stored koordinates
					WayPt.SetLoc (StoredLoc);
					bUpdated = TRUE;
					}
				}
			}

 		for (j=0; j<2; j++)
			{
			if (WayPt.HasBearing (j))
				{
				short nCat;
				CLocation NavLoc;
										// use NameList to get Nav from szNav...
				if (WayPt.GetBearNameAndCat (j, &szNav, &nCat))
					{
					if (WayPt.NameCatToLoc (szNav, nCat, &NavLoc))
						{
						WayPt.CalculateBearing (j, NavLoc);
						bUpdated = TRUE;
						}
					}
				}
			}


		if (bUpdated)
			{
			this->ChangeWpt (i, WayPt);			// change wpt
			bChanged = TRUE;
			}
		} // if (GetWayPointPtr...
	} // for all legs: 

if (bChanged && !bDemo)
	{
	bDoSave = TRUE;
	}
return bDoSave;
}


/************************************************************************
 *  WayDoc.cpp				I n v e r t R o u t e						*
 ************************************************************************/
void CWayDoc::InvertRoute ()
{
CWayPoint	LastWpt, FirstWpt;
int	i,n, NumbOfLoops;

NumbOfLoops = m_WayPts.GetSize()/2;		/* if cnt=5 => NumbOfLoops = 2	*/
n = m_WayPts.GetSize() - 1;				/* max index = 4				*/


for (i=0; i<NumbOfLoops; i++)			/* i=0 contains start field		*/
    {			
	this->GetWayPointPtr (i, &FirstWpt);/* get first way data			*/
	this->GetWayPointPtr (n, &LastWpt);	/* get last way data			*/

	if (i==0)							/* exchange ELEV and PlatzRunde	*/
		{
		if (LastWpt.GetAltitude() != NO_ALT)	/* set new first alt	*/
			{
			LastWpt.SetPlannedAlt(LastWpt.GetAltitude());
			LastWpt.SetAltDim(LastWpt.GetElevDim());
			}

		if (FirstWpt.GetAltitude() != NO_ALT)	/* set new last alt		*/
			{
			double	fHelp, fAlt;
			fAlt = ptDim->ConvertDist (1000., DIM_FEET, FirstWpt.GetElevDim());
			fHelp = (FirstWpt.GetAltitude() + fAlt)/100 + .5; /*  / 100	*/

			FirstWpt.SetPlannedAlt((long)fHelp*100);	   /*  * 100	*/
			FirstWpt.SetAltDim (FirstWpt.GetElevDim());
			}
		}
 
	this->ChangeWpt (i, LastWpt);			/* put last to first		*/
	this->ChangeWpt (n, FirstWpt);			/* put first to last		*/
    n--;
    }
}



/************************************************************************
 *  WayDoc.cpp					C o p y W p t s							*
 ************************************************************************/
void CWayDoc::CopyWpts (CPtrArray* ptDest, CPtrArray* ptSource)
{
short i;
for (i=0; i<ptSource->GetSize(); i++) 
	{
	LPWAY	lpDocWay = NULL;         

	lpDocWay = (LPWAY)ptSource->GetAt (i);    
	if (lpDocWay != NULL)
		{
		TRACE ("Create WayDoc::CopyWpts LPWAY %d\n", i);

		LPWAY lpCpyWay = new WAYTYPE;	  // need no class!!
		_fmemcpy (lpCpyWay, lpDocWay, sizeof(WAYTYPE)); 
		ptDest->Add (lpCpyWay);    
		}
	}
}        

/************************************************************************
 *  WayDoc.cpp				G e t R o u t e 							*
 ************************************************************************/
void CWayDoc::GetRoute (CPtrArray* ptWayList)
{    
                              	// copy database from Doc to Dlg
this->DeleteArrayOf (ptWayList);
this->CopyWpts (ptWayList, &m_WayPts);   
}   

/************************************************************************
 *  WayDoc.cpp				S e t R o u t e		 						*
 ************************************************************************/
void CWayDoc::SetRoute (CPtrArray* ptWayList)
{     
                          		 // copy database from Dlg to Doc
this->DeleteArrayOf (&m_WayPts);
this->CopyWpts (&m_WayPts, ptWayList);   
}

/************************************************************************
 *  WayDoc.cpp				D e l e t e R o u t e						*
 ************************************************************************/
void CWayDoc::DeleteRoute ()
{     
	this->DeleteArrayOf (&m_WayPts);

	for (int i=1; i<=2; i++)
	{		// delete alternate way points of this doc and indicators of flight plan
		CWayPoint UndefWpt;
		this->SetAlternateWayPointPtr(i, UndefWpt);
		this->SetAlternate("", i);			// indicaor for flight plan
	}
}

/************************************************************************
 *  WayDoc.cpp  		 	S e a r c h I n R o u t e 					*
 ************************************************************************/
BOOL CWayDoc::SearchInRoute (CString szNameForRte, CWayPoint* ptWpt)
{
	BOOL bFound = false;
	
	for (int i=0; i<m_WayPts.GetSize() && !bFound; i++)
	{
		CWayPoint WayPt;
		if (this->GetWayPointPtr (i, &WayPt))
		{
			CString szTestName = WayPt.GetNameForRte();
			if (szTestName.Compare (szNameForRte) == 0)
			{
				bFound = TRUE;
				*ptWpt = WayPt;
			}
		}
	}

	return bFound;
}


/************************************************************************
 *  WayDoc.cpp  		 	F i n d B e s t N a m e 					*
 ************************************************************************/
BOOL CWayDoc::FindBestName (char* ptName, short CmpLen)
{
	BOOL bFound = false;
	
	for (int i=0; i<m_WayPts.GetSize() && !bFound; i++)
	{
		CWayPoint WayPt;
		if (this->GetWayPointPtr (i, &WayPt))
		{
			CString szTestName = WayPt.GetNameForRte();
			if (_fstrnicmp ((LPSTR)ptName, (LPCTSTR)szTestName, CmpLen) == 0)
			{
				_fstrcpy ((LPSTR)ptName, (LPCTSTR)szTestName);
				bFound = true;
			}
		}
	}
	return bFound;
}

/************************************************************************
 *  WayDoc.cpp				S a m e L o c A r o u n d					*
 ************************************************************************/
BOOL CWayDoc::SameLocAround (short nIndex, CLocation Loc)
{
#define EPS_KOORD 0.0001
BOOL	bSame = FALSE;
short	i, nLastIndex;

if (nIndex >= 0)
	{
	nLastIndex = (nIndex+1 < m_WayPts.GetSize())? nIndex + 1 : nIndex;
	
	for (i=nIndex; i<=nLastIndex && !bSame; i++)
		{
		LPWAY lpWayPt = (LPWAY)m_WayPts.GetAt(i);
		CWayPoint WayPt(lpWayPt);

		if (fabs (WayPt.GetLat() - Loc.GetLat()) < EPS_KOORD &&
			fabs (WayPt.GetLon() - Loc.GetLon()) < EPS_KOORD)
			{
			bSame = TRUE;
			}
		} 
	}
return bSame;
}

/************************************************************************
 *  WayDoc.cpp				A p p e n d W p t							*
 ************************************************************************/
BOOL CWayDoc::AppendWpt (CWayPoint WayPt)
{
BOOL bAppended = FALSE;

LPWAY lpWayPt = WayPt.CreatePtr();
if (lpWayPt != NULL)
	{  
	TRACE ("Create WayDoc::AppendWpt LPWAY %d\n", m_WayPts.GetSize());

	m_WayPts.Add (lpWayPt);       	// add new Element  
	bAppended = TRUE;      
	}
return bAppended;
}

/************************************************************************
 *  WayDoc.cpp				I n s e r t W p t							*
 ************************************************************************/
BOOL CWayDoc::InsertWpt (short nIndex, CWayPoint WayPt)
{
BOOL bInserted = FALSE;

LPWAY lpWayPt = WayPt.CreatePtr();
if (lpWayPt != NULL)
	{  
	TRACE ("Create WayDoc::InsertWpt LPWAY %d\n", nIndex);
	m_WayPts.InsertAt (nIndex, lpWayPt);       // insert new Element 

	if(m_nActIndex >= nIndex)
		m_nActIndex++;

	bInserted = TRUE;      
	}

return bInserted;
}

/************************************************************************
 *  WayDoc.cpp				C h a n g e W p t							*
 ************************************************************************/
BOOL CWayDoc::ChangeWpt (short nIndex, CWayPoint WayPt)
{
BOOL bChanged = FALSE;

if (nIndex >= 0 && nIndex < m_WayPts.GetSize())
	{ 
	LPWAY lpWayPt = (LPWAY)m_WayPts.GetAt(nIndex);
	if (lpWayPt != NULL)
		{  
	//	TRACE ("CWayDoc::ChangeWpt %d\n", nIndex);

		WayPt.GetPtr (lpWayPt);				// copy WayPt into memory of array
		bChanged = TRUE;      
		}
	}

return bChanged;
}

/************************************************************************
 *  WayDoc.cpp				D e l e t e W p t							*
 ************************************************************************/
BOOL CWayDoc::DeleteWpt (short nIndex)
{
BOOL bDeleted = FALSE;

if (nIndex >= 0 && nIndex < m_WayPts.GetSize())
	{ 
	LPWAY lpWayPt = (LPWAY)m_WayPts.GetAt(nIndex);
	if (lpWayPt != NULL)
		{  
		TRACE ("Delete WayDoc::DeleteWpt LPWAY %d\n", nIndex);
		delete lpWayPt;
		m_WayPts.RemoveAt(nIndex);

		if(m_nActIndex >= nIndex)
			{
			m_nActIndex--;
			if (m_nActIndex<0 && this->GetWayPointCnt()>0)
				m_nActIndex=0;
			}

		bDeleted = TRUE;      
		}
	}

return bDeleted;
}


/************************************************************************
 *  WayDoc.cpp			G e t M a x L o c N a m e L e n					*
 ************************************************************************/
BOOL CWayDoc::GetMaxLocNameLen (CDC* pDC, BOOL bPair)
{
	short	nThisTextLen;
	short i, nMaxNameLen;
	CString szHelpName;

	nMaxNameLen = 0;  
	short	nCnt = m_WayPts.GetSize();
	for (i=0; i<nCnt; i++)
    {
		LPWAY lpWayPt = (LPWAY)m_WayPts.GetAt(i);
		CWayPoint WayPt(lpWayPt);

 		if ((i==0 || i==nCnt-1) && bPair)
			 szHelpName = WayPt.GetLocNamePair ();
		else szHelpName = WayPt.GetNameForRte ();

		CSize	TextSize;
		TextSize = pDC->GetTextExtent(szHelpName, strlen(szHelpName));
		nThisTextLen = (short)TextSize.cx;

		if (nThisTextLen > nMaxNameLen) nMaxNameLen = nThisTextLen;
    }


	short nAlternateCnt = this->GetAlternateCnt();
	for (i=0; i<nAlternateCnt; i++)
	{
		CWayPoint WayPt;
		if (this->GetAlternateWayPointPtr(i+1, &WayPt))
		{
			szHelpName = WayPt.GetLocNamePair ();

			CSize	TextSize;
			TextSize = pDC->GetTextExtent(szHelpName, strlen(szHelpName));
			nThisTextLen = (short)TextSize.cx;

			if (nThisTextLen > nMaxNameLen) nMaxNameLen = nThisTextLen;
		}
	}

	return nMaxNameLen;
}


/************************************************************************
 *  WayDoc.cpp			G e t M a p B o r d e r	 						*
 ************************************************************************/
void CWayDoc::GetMapBorder (CBorder* ptBorder, BOOL bShowNav)
{
short i, j;
double 		fLat, fLon;
short nCnt = m_WayPts.GetSize();

for (i=0; i<nCnt; i++)
	{
	LPWAY lpWayPt = (LPWAY)m_WayPts.GetAt(i);
	CWayPoint WayPt(lpWayPt);
//	CLocation Loc = WayPt.GetLoc();
	CLocation Loc(WayPt);

	fLat = Loc.GetLat();
	fLon = Loc.GetLon();

	if (i==0)	ptBorder->Init (fLat, fLon);
		else	ptBorder->Expand (fLat, fLon);
		
	if (bShowNav)
		{
		for (j=0; j<2; j++)
			{
			if (WayPt.GetBearLoc (j, &fLat, &fLon))
				{				
				ptBorder->Expand (fLat, fLon);
				}
			}
		}
	}
ptBorder->Finish ();
}


/************************************************************************
 *  WayDoc.cpp			G e t W a y P o i n t C n t 					*
 ************************************************************************/
short CWayDoc::GetWayPointCnt ()
{
return (short)m_WayPts.GetSize();
}

/************************************************************************
 *  WayDoc.cpp				G e t W a y P o i n t P t r					*
 ************************************************************************/
LPWAY CWayDoc::GetWayPointPtr (short nIndex)
{
LPWAY lpWay=NULL;
if (nIndex>=0 && nIndex<m_WayPts.GetSize())
	{
	lpWay = (LPWAY)m_WayPts.GetAt (nIndex);  
	} 
return lpWay;      
}

/************************************************************************
 *  WayDoc.cpp			G e t W a y P o i n t P t r						*
 ************************************************************************/
BOOL CWayDoc::GetWayPointPtr (short nIndex, CWayPoint* ptWayPt)
{
BOOL bOK = FALSE;

if (nIndex >= 0 && nIndex < m_WayPts.GetSize())
	{
	LPWAY lpWay = (LPWAY)m_WayPts.GetAt(nIndex);
	ptWayPt->SetPtr (lpWay);
	bOK = TRUE;
	}

return bOK;
}

/************************************************************************
 *  WayDoc.cpp		G e t A l t e r n a t e W a y P o i n t P t r		*
 *  nID = 1, gets first alternate, nID = 2, gets second alternate		* 
 ************************************************************************/
BOOL CWayDoc::GetAlternateWayPointPtr (short nID, CWayPoint* ptWayPt)
{
	BOOL bOK = FALSE;

	if (HasAlternates())
	{
		if (nID == 1)	*ptWayPt = m_Alternate1;
			else		*ptWayPt = m_Alternate2;

		bOK = ptWayPt->IsValid();
	}

	return bOK;
}

/************************************************************************
 *  WayDoc.cpp		S e t A l t e r n a t e W a y P o i n t P t r		*
 *  nID = 1, sets first alternate, nID = 2, sets second alternate		* 
 ************************************************************************/
void CWayDoc::SetAlternateWayPointPtr (short nID, CWayPoint& WayPt)
{
		// allowed to set invalid way points in order to remove alternate
	if (nID == 1)	m_Alternate1 = WayPt;
		else		m_Alternate2 = WayPt;
}


/************************************************************************
 *  WayDoc.cpp			G e t A l t e r n a t e C n t					*
 ************************************************************************/
short CWayDoc::GetAlternateCnt()
{
	short nCnt = 0;

	if (HasAlternates())
	{
		if (m_Alternate1.IsValid())
			nCnt++;
		if (m_Alternate2.IsValid())
			nCnt++;
	}

	return nCnt;
}


/************************************************************************
 *  WayDoc.cpp			G e t A c t W a y P o i n t P t r 				*
 ************************************************************************/
BOOL CWayDoc::GetActWayPointPtr (CWayPoint* ptWayPt)
{
	BOOL bValid = FALSE;

	if (this->GetWayPointCnt() > 0)
		bValid = this->GetWayPointPtr(m_nActIndex, ptWayPt);
	return bValid;
}

/************************************************************************
 *  WayDoc.cpp			G e t A c t L o c a t i o n	P t r				*
 ************************************************************************/
BOOL CWayDoc::GetActLocationPtr (CLocation* ptLoc)
{
	BOOL bValid = FALSE;

	CWayPoint WayPt;
	if (this->GetActWayPointPtr(&WayPt))
	{
		*ptLoc = WayPt.GetLoc();
		bValid = ptLoc->IsValid();
	}

	return bValid;
}


/************************************************************************
 *  WayDoc.cpp			G e t A c t L a t L o n P t r					*
 ************************************************************************/
BOOL CWayDoc::GetActLatLonPtr (CLatLon* ptLatLon)
{
	BOOL bValid = FALSE;

	CWayPoint WayPt;
	if (this->GetActWayPointPtr(&WayPt))
	{
		*ptLatLon = (CLatLon)WayPt;
		bValid = ptLatLon->IsValid();
	}

	return bValid;
}


/************************************************************************
 *  WayDoc.cpp				S e t A c t I n d							*
 ************************************************************************/
void CWayDoc::SetActInd (short nIndex)
{
	m_nActIndex = nIndex;
	ptInit->ResetLatLonToCenter();	// invalidate latlon to center
			// if latlon to center is NOT valid, act wpt will be centered
}


/************************************************************************
 *  WayDoc.cpp			O n N e w D o c u m e n t						*
 ************************************************************************/
BOOL CWayDoc::OnNewDocument()
{
	if (!CDocument::OnNewDocument())	// calls DeleteContents
		return FALSE;
	// TODO: add reinitialization code here
	// (SDI documents will reuse this document)


	CFrameWnd* ptWnd = ptInit->GetMainWndPtr();

	CString szTitle;
	szTitle.LoadString(IDS_NO_TITLE);
	this->SetTitle((LPCTSTR)szTitle);

				// use home base as first way point
	if (m_bAddHomeBase)		// use NoHomeBase() before OnNewDocument
							//	to disable this feature
	{
		CLocation HomeBase;
		HomeBase = ptInit->GetHomeBase ();

		CWayPoint WayPt;
		WayPt.SetLoc (HomeBase);
		WayPt.SetPlannedAlt (HomeBase.GetAltitude());
		WayPt.SetAltDim (HomeBase.GetElevDim());
		if (this->AppendWpt (WayPt))
			this->SetActInd (0);	
	}

	ptInit->ResetLatLonToCenter();


	m_bAddHomeBase = TRUE;		// may be deactivated for reading GPS route

	if (ptScMpDoc != NULL)
	{
		ptScMpDoc->ActivateNearestMap(this);
	}

	return TRUE;
}

/************************************************************************
 *  PlaneDoc.cpp			D e l e t e C o n t e n t s					*
 *  Called from: OnFileNew and OnFileOpen								*
 ************************************************************************/
void CWayDoc::DeleteContents()
{
	this->InitHeader();
	this->DeleteArrayOf (&m_WayPts);

	if (m_ptCalculation != NULL)
	{
		delete m_ptCalculation;
		m_ptCalculation = NULL;
	}

	m_ptCalculation = new CCalculate (this);

	if (ptFlPlDoc != NULL)
		ptFlPlDoc->SetDataMode (DATA_AUTOMATIC);
}

/************************************************************************
 *  WayDoc.cpp				O n O p e n D o c u m e n t					*
 ************************************************************************/
BOOL CWayDoc::OnOpenDocument(LPCTSTR pszPathName)
{
BOOL	bDoSave;
BOOL	bOpened = FALSE;


//virtual void SetPathName( LPCTSTR lpszPathName, BOOL bAddToMRU = TRUE );
this->SetPathName (pszPathName);
bOpened = CDocument::OnOpenDocument(pszPathName); 

if (bOpened)
	{
	this->InLegalArea();
	bDoSave = this->Update();

	if (m_bConverted || bDoSave)
		{
		if (this->OnSaveDocument (pszPathName, FALSE))		// don't confirm
			{
			m_bConverted = FALSE;
			if (ptInit->IsSaveSignal())
				MessageBeep(MB_OK);	
			}
		}

	CLatLon LLtoCenter;
	this->GetActLatLonPtr(&LLtoCenter);
	ptScMpDoc->ActivateScMapForLatLon(LLtoCenter);

	CFrameWnd* ptWnd = ptInit->GetMainWndPtr();
	if (ptWnd != NULL)
		{
		eView nView = ((CMainFrame*)ptWnd)->GetTypeOfView();

		ptInit->ResetLatLonToCenter();

		if (nView < D_TRACK)
			{
			this->UpdateAllViews(NULL); 

		//		don't center initial vector graphik. 
		//		Instead: call GetMapBorder in CRouteView::OnUpdate
		//		to show whole area 
		//	if (nView == D_MAP)
		//		((CMainFrame*)ptWnd)->CenterLatLonOnMp (LLtoCenter);

			if (nView == D_SCANMAP)
				((CMainFrame*)ptWnd)->CenterLatLonOnScMp (LLtoCenter);
			}
		else{
			WPARAM wParam = MAKELONG (ID_VIEW_COURSE, 0);
			ptWnd->PostMessage(WM_COMMAND, wParam, 0);
			}
		}
	}

return bOpened;
}

/************************************************************************
 *  WayDoc.cpp				O n S a v e D o c u m e n t					*
 ************************************************************************/
BOOL CWayDoc::OnSaveDocument(const char* pszPathName, BOOL bConfirm)
{
BOOL bSaved = FALSE;
BOOL bDoSave;

if (bConfirm)
	{
	CString szText;
	CString szFileExt = this->GetTitle();
	szText.Format(IDF_SAVE_FILE, (LPCTSTR)szFileExt);
	bDoSave = (AfxMessageBox ((LPCSTR)szText, MB_YESNO) == IDYES);
	}
else{
	bDoSave = TRUE;
	}

if (bDoSave)
	{
	if (*pszPathName == 0)
		{			// path name is "" after OnNewDocument
		this->OnFileSaveAs();
		bSaved = !this->IsModified();
		}
	else{
		bSaved = CDocument::OnSaveDocument(pszPathName); 
		}
	}        
else{   
    this->SetModifiedFlag (FALSE);
	}
return bSaved;
}

/************************************************************************
 *  WayDoc.cpp  		 		 C h a n g e 							*
 ************************************************************************/
BOOL CWayDoc::Change(CWnd* ptWnd)
{  
BOOL 	bChanged = FALSE;
CWayDlg WayDlg(ptWnd, this);

int RetVal = WayDlg.DoModal();
switch (RetVal)
	{
	case IDOK:
		{   
		if (WayDlg.WriteRouteIntoDoc()) 
			{        
			this->SetActInd(this->GetWayPointCnt()-1);

			this->InLegalArea();
			this->CheckPlannedAlt();

			this->UpdateAllViews(NULL);  
 			bChanged = TRUE;
 			}     
		}
 		break;
	case IDCANCEL: 
        this->SetModifiedFlag(FALSE);
		break;
	}
return bChanged;	
} 

/************************************************************************
 *  WayDoc.cpp  		 		R e a d F r o m G P S					*
 ************************************************************************/
BOOL CWayDoc::ReadFromGPS(CWnd* ptWnd)
{  
	BOOL 	bChanged = FALSE;

	if (!bDemo)
	{
		if (this->IsModified())
		{
			BOOL bConfirm=TRUE;
			this->OnSaveDocument (this->GetPathName(), bConfirm);  
		}
	}


CGpsDoc* ptGpsDoc = ptInit->GetGpsPtr();
CReadFromGpsDlg ReadDlg(ptWnd, this, ptGpsDoc);

int RetVal = ReadDlg.DoModal();
switch (RetVal)
	{
	case IDOK:
 	//	ptWnd->InvalidateRect(NULL); 

		this->InLegalArea();
		if (this->Update())
			this->OnSaveDocument(this->GetPathName()); 

		this->UpdateAllViews(NULL);  
 		bChanged = TRUE;
 		break;
	case IDCANCEL: 
        this->SetModifiedFlag(FALSE);
		break;
	}
return bChanged;	
} 

/************************************************************************
 *  WayDoc.cpp  		 	 W r i t e T o G P S 						*
 ************************************************************************/
BOOL CWayDoc::WriteToGPS(CWnd* ptWnd)
{  
BOOL 	bOK = FALSE;

this->InLegalArea();

CGpsDoc* ptGpsDoc = ptInit->GetGpsPtr();
CWriteToGpsDlg WriteDlg(ptWnd, this, ptGpsDoc);

int RetVal = WriteDlg.DoModal();
switch (RetVal)
	{
	case IDOK:
		bOK = TRUE;
 		break;
	case IDCANCEL: 
		break;
	}
return bOK;	
} 


/************************************************************************
 *  WayDoc.cpp  		 	 C r e a t e N a m e						*
 *  uses indicator of first and last wpt to create name of route		*
 ************************************************************************/
CString CWayDoc::CreateName()
{
	CString szRteName;	

	CWayPoint WayPt;
	if (this->GetWayPointPtr(0, &WayPt))
		szRteName = WayPt.GetRteNamePart();

	if (this->GetWayPointPtr(this->GetWayPointCnt()-1, &WayPt))
		szRteName += WayPt.GetRteNamePart();

	return szRteName;
}

/************************************************************************
 *  WayDoc.cpp  		 C a t e g o r y T o G C A S t y p e			*
 *  Purpose: Converts PreFlight waypoint category into GCAS TypeFC		*
 ************************************************************************/
BOOL CWayDoc::CategoryToGCAStype (short nCategory, CString* ptType)
{
	BOOL bDefined = FALSE;

	switch (nCategory)
	{
	case WP_AIRPORT:
		*ptType = "0"; 
		bDefined = TRUE;
		break;
	case WP_REPORT:
		*ptType = "100";
		bDefined = TRUE;
		break;
	case WP_INTERSECTION:
		*ptType = "100";		// WP_INTERSECTION like WP_REPORT
		bDefined = TRUE;
		break;
	case WP_NDB:
		*ptType = "5"; 
		bDefined = TRUE;
		break;
	case WP_VOR:
		*ptType = "4"; 
		bDefined = TRUE;
		break;
	case WP_USER:
		*ptType = "8"; 
		bDefined = TRUE;
		break;
	case WP_SAIL:
		*ptType = "0";
		bDefined = TRUE;
		break;
	case WP_HELI:
		*ptType = "0";
		bDefined = TRUE;
		break;
	}

	return bDefined;

}

/************************************************************************
 *  WayDoc.cpp  		 	 W r i t e T o G C A S						*
 ************************************************************************/
BOOL CWayDoc::WriteToGCAS(CWnd* ptWnd)
{  
	BOOL 	bOK = FALSE;

	this->InLegalArea();

							// ask for new file name
	static char szFilter[] = "ExportGCAS (*.xml)|*.xml||"; 

	CString szInitialName;
	short nLen;
	szInitialName = this->GetTitle();
												// remove ".way" from name
	short nDot = szInitialName.ReverseFind('.');
	if (nDot > -1)
	{
		szInitialName = szInitialName.Left(nDot);
	}
	else
	{
		CString szNoTitle;
		szNoTitle.LoadString(IDS_NO_TITLE);
		if (szInitialName.Compare(szNoTitle)==0)
			szInitialName = CreateName();
	}

	CString szExt(".xml");						// add ".xml"
	szInitialName += szExt;


	CFileDialog dlg(FALSE, 	  					// File SaveAs dialog
					(LPCTSTR)"xml",				// default extension "*.xml"
					(LPCTSTR)szInitialName,		// initial filename
					OFN_HIDEREADONLY |	  		// dwFlags
					OFN_OVERWRITEPROMPT,
					szFilter,
					ptWnd);						// parent window

//	CString szPath = ptInit->GetActualPath();	// without NameExt
//	dlg.m_ofn.lpstrInitialDir = (LPCTSTR)szPath;

	int ret = dlg.DoModal();
	if (ret == IDOK)
		{
		CString szBuffer;
		CString szFileTitle = dlg.GetPathName();	// full path

		CString szRouteName = dlg.GetFileName();	// EDFEEDDR.xml
		nDot = szRouteName.ReverseFind('.');
		if (nDot > -1)
			szRouteName = szRouteName.Left(nDot);

		CFile* ptFile = new CFile ();

		if (ptFile != NULL)
			{
				ptFile->Open(szFileTitle, CFile::modeCreate | CFile::modeNoInherit | CFile::modeWrite, NULL); 

																	// <Route name="EDFEEDDR">
				szBuffer.Format(IDF_GCAS_RTE1, (LPCTSTR)szRouteName);
				nLen = szBuffer.GetLength();
				ptFile->Write ((const void*)szBuffer, (UINT)nLen);


				for (int i=0; i<this->GetWayPointCnt(); i++)
				{
					CWayPoint WayPt;
					
					if (this->GetWayPointPtr(i, &WayPt))
					{

						szBuffer.Format(IDF_GCAS_WPT1, i+1);		// <RoutePoint id="1">
						nLen = szBuffer.GetLength();
						ptFile->Write ((const void*)szBuffer, (UINT)nLen);
						

							CString szName;
							szBuffer.Format(IDS_GCAS_NAME1);			// <Name>
							nLen = szBuffer.GetLength();
							ptFile->Write ((const void*)szBuffer, (UINT)nLen);


																		// EDKA Aachen
							CString szIndicator = WayPt.GetIndicator();
							if (szIndicator.GetLength() == 4)
								szName = szIndicator + " ";
							szName+= WayPt.GetName();
							nLen = szName.GetLength();
							ptFile->Write ((const void*)szName, (UINT)nLen);

							szBuffer.Format(IDS_GCAS_NAME2);			// </Name>
							nLen = szBuffer.GetLength();
							ptFile->Write ((const void*)szBuffer, (UINT)nLen);




							
							CString szType;
							short nCategory = WayPt.GetCategory();			
							if (this->CategoryToGCAStype (nCategory, &szType))
							{
								szBuffer.Format(IDS_GCAS_TYPEFC1);			// <TypeFC>
								nLen = szBuffer.GetLength();
								ptFile->Write ((const void*)szBuffer, (UINT)nLen);

								nLen = szType.GetLength();					
								ptFile->Write ((const void*)szType, (UINT)nLen);

								szBuffer.Format(IDS_GCAS_TYPEFC2);			// </TypeFC>
								nLen = szBuffer.GetLength();
								ptFile->Write ((const void*)szBuffer, (UINT)nLen);
							}
							// WP_USER has no TypeFC information in GCAS




							szBuffer.Format(IDS_GCAS_FREQ1);			// <Frequency>
							nLen = szBuffer.GetLength();
							ptFile->Write ((const void*)szBuffer, (UINT)nLen);

							float fFreq;
							if (WayPt.GetFrequency(&fFreq))
							{
								CString szFreq;
								if (WayPt.GetCategory() == WP_NDB)
									 szFreq.Format ("%dK", (long)(fFreq*1000 + 0.5));
								else szFreq.Format ("%dM", (long)(fFreq*1000 + 0.5));

								nLen = szFreq.GetLength();
								ptFile->Write ((const void*)szFreq, (UINT)nLen);
							}
							szBuffer.Format(IDS_GCAS_FREQ2);			// </Frequency>
							nLen = szBuffer.GetLength();
							ptFile->Write ((const void*)szBuffer, (UINT)nLen);





							szBuffer.Format(IDS_GCAS_KEY1);			// <Key>
							nLen = szBuffer.GetLength();
							ptFile->Write ((const void*)szBuffer, (UINT)nLen);

							CString szKey;
							szKey.Format ("WP%04d", i);
							nLen = szKey.GetLength();
							ptFile->Write ((const void*)szKey, (UINT)nLen);

							szBuffer.Format(IDS_GCAS_KEY2);			// </Key>
							nLen = szBuffer.GetLength();
							ptFile->Write ((const void*)szBuffer, (UINT)nLen);



							szBuffer.Format(IDS_GCAS_LANDING1);		// <Landing>
							nLen = szBuffer.GetLength();
							ptFile->Write ((const void*)szBuffer, (UINT)nLen);

							CString szLanding;	// "WAHR" if last wpt of rte
							if (i == this->GetWayPointCnt()-1)	szLanding = "WAHR";
									else						szLanding = "FALSCH";
							nLen = szLanding.GetLength();
							ptFile->Write ((const void*)szLanding, (UINT)nLen);

							szBuffer.Format(IDS_GCAS_LANDING2);		// </Landing>
							nLen = szBuffer.GetLength();
							ptFile->Write ((const void*)szBuffer, (UINT)nLen);




							long lAlt_ft = (long)ptDim->ConvertDist(WayPt.GetAltitude(), WayPt.GetAltDim(), DIM_FEET);
							szBuffer.Format(IDF_GCAS_COORD1, lAlt_ft);	// <Coord alt=632">
							nLen = szBuffer.GetLength();
							ptFile->Write ((const void*)szBuffer, (UINT)nLen);

							short nDegr, nMin, nSec, nDir;
							char szSign[2];
							memset (szSign, 0, 2);

							CString szLat;
							CLatLon::AngleToDMS (WayPt.GetLat(), TRUE, &nDegr, &nMin, &nSec, &nDir);
							if ((char)nDir == 'S')
								szSign[0] = '-';
							szLat.Format(IDF_GCAS_COORD_LAT, szSign, nDegr, nMin, nSec);

							CString szLon;
							CLatLon::AngleToDMS (WayPt.GetLon(), FALSE, &nDegr, &nMin, &nSec, &nDir);
							if ((char)nDir == 'W')
								szSign[0] = '-';
							szLon.Format(IDF_GCAS_COORD_LON, szSign, nDegr, nMin, nSec);

							CString szCoord = szLat + szLon;			//50.49.23.0000,06.11.11.0000
							nLen = szCoord.GetLength();
							ptFile->Write ((const void*)szCoord, (UINT)nLen);



							szBuffer.Format(IDS_GCAS_COORD2);			// </Coord>
							nLen = szBuffer.GetLength();
							ptFile->Write ((const void*)szBuffer, (UINT)nLen);






						szBuffer.Format(IDS_GCAS_WPT2);				// </RoutePoint>
						nLen = szBuffer.GetLength();
						ptFile->Write ((const void*)szBuffer, (UINT)nLen);
					}

				}



				szBuffer.Format(IDS_GCAS_RTE2);						// </Route>
				nLen = szBuffer.GetLength();
				ptFile->Write ((const void*)szBuffer, (UINT)nLen);

				ptFile->Close();
				delete ptFile;
				ptFile = NULL;
			}
	
		}
	return bOK;	
} 


/************************************************************************
 *  WayDoc.cpp  		 	 W r i t e T o K M L						*
 ************************************************************************/
BOOL CWayDoc::WriteToKML(CWnd* ptWnd)
{  
	BOOL 	bOK = FALSE;

	this->InLegalArea();

							// ask for new file name
	static char szFilter[] = "Export (*.kml)|*.kml||"; 

	CString szInitialName;
	szInitialName = this->GetTitle();
												// remove ".way" from name
	short nDot = szInitialName.ReverseFind('.');
	if (nDot > -1)
	{
		szInitialName = szInitialName.Left(nDot);
	}
	else
	{
		CString szNoTitle;
		szNoTitle.LoadString(IDS_NO_TITLE);
		if (szInitialName.Compare(szNoTitle)==0)
			szInitialName = CreateName();
	}

	CString szExt(".kml");						// add ".kml"
	szInitialName += szExt;


	CFileDialog dlg(FALSE, 	  					// File SaveAs dialog
					(LPCTSTR)"kml",				// default extension "*.kml"
					(LPCTSTR)szInitialName,		// initial filename
					OFN_HIDEREADONLY |	  		// dwFlags
					OFN_OVERWRITEPROMPT,
					szFilter,
					ptWnd);						// parent window

//	CString szPath = ptInit->GetActualPath();	// without NameExt
//	dlg.m_ofn.lpstrInitialDir = (LPCTSTR)szPath;

	int ret = dlg.DoModal();
	if (ret == IDOK)
	{
		CString szBuffer;
		CString szFileTitle = dlg.GetPathName();	// full path

		CString szRouteName = dlg.GetFileName();	// EDFEEDDR.kml
		nDot = szRouteName.ReverseFind('.');
		if (nDot > -1)
			szRouteName = szRouteName.Left(nDot);

		CWayKmlDoc* ptKmlDoc = new CWayKmlDoc();
		if (ptKmlDoc != NULL)
		{
			if (ptKmlDoc->ConvertFromWayDoc(this, szFileTitle))
			{
				if (ptKmlDoc->IsModified())
				{
					BOOL bConfirm=FALSE;
					if (ptKmlDoc->OnSaveDocument (ptKmlDoc->GetPathName(), bConfirm))
					{
						CString szMsg;
						szMsg.Format(IDF_FILESAVED, (LPCTSTR)szFileTitle);
						AfxMessageBox((LPCTSTR)szMsg);
					}
				}
			}

			delete ptKmlDoc;
		}
	} 
	return bOK;	
} 


/************************************************************************
 *  WayDoc.cpp			  G e t N e a r e s t F a c i l i t y			*
 *  Sucht in "LaLoList" die Navigationsanlage, die dem Punkt "ptActLoc" *
 *  am nchsten liegt und speichert sie unter "ptNavLoc" ab.			*
 *  ptMinDist = Abstand von ActLoc zur nchsten Navigationsanlage.		*
 ************************************************************************/
BOOL CWayDoc::GetNearestFacility (CLocation ActLoc, CLocation* ptNavLoc, double* ptMinDist)
{
BOOL		bNavFound = FALSE;
CLocation	Loc;
short		nActRegIndex;

DRECT rMinMap = CLatLon::RadiusToDRECT (ActLoc.GetLat(), ActLoc.GetLon(), 2); // degrees

CLaLoList* ptLaLoList = ptInit->GetLaLoListPtr();
if (ptLaLoList->PrepareGetLoc (rMinMap))
	{
	while (ptLaLoList->GetLocationInRect (&Loc, rMinMap, &nActRegIndex))
		{							// is Loc a facility near to ActLoc?
		double dDist;
		if (Loc.IsFacilityInRange(ActLoc, &dDist))
			{
			if (!bNavFound)
				{					// first nav found	
				*ptMinDist = dDist;
				*ptNavLoc = Loc;
				bNavFound = TRUE;
				}
			else{					// check for minimum
				if (dDist < *ptMinDist)
					{
					*ptMinDist = dDist;
					*ptNavLoc = Loc;
					}
				}
			}
		}
	}

return bNavFound;
}

/************************************************************************
 *  WayDoc.cpp			G e t N e w I n d e x							*
 *  Sucht in "way" ab Wegpunkt "Ind0" den Flugabschnitt, in den			*
 *  der neue Wegpunkt Lat/Lon eingefgt werden kann.					*
 *  Die gesamte Flugstrecke vom Punkt "Ind0" bis zum neuen Wegpunkt		*
 *  M U S S nahezu geradlinig verlaufen!!!								*
 *  Output: *ptInd  (Index des neuen Wegpunktes)						*
 ************************************************************************/
void CWayDoc::GetNewIndex (CLatLon NavLatLon, short Ind0, short* ptInd)
{
short		i;
double		Dist, DistX, dRWK;
CLatLon		LatLon;
CWayPoint	Wpt;

			/* Distance btn new waypoint and waypoint "Ind0"*/
i = Ind0;
if (this->GetWayPointPtr(i++, &Wpt))
	{
	DistX = NavLatLon.Distance ((CLatLon)Wpt, &dRWK);

	LatLon = (CLatLon)Wpt;		/* Distance btn waypoint "Ind0" and	*/
								/* the following waypoints		*/
	do  {
		this->GetWayPointPtr (i++, &Wpt);
		Dist = LatLon.Distance ((CLatLon)Wpt, &dRWK);
		} while (Dist < DistX && i<this->GetWayPointCnt());
	}
*ptInd = i - 1;
}

/************************************************************************
 *  WayDoc.cpp		     G e t N e w P o i n t							*
 *  Bestimmt Mittelpunkt zwischen Loc0 und Loc1 und sucht				*
 *  Navigationsanlage in der Nhe des Mittelpunktes.					*
 *  Return: TRUE, wenn Nav gefunden, LocX = Nav  ODER					*
 *		  wenn kein Nav gefunden, ptLocX = Mittelpunkt					*
 *	    FALSE, wenn Loc0 und Loc1 schon dicht genug zusammen liegen		*
 ************************************************************************/
BOOL CWayDoc::GetNewPoint (CLocation Loc0, CLocation* ptLocX, CLocation Loc1, short Ind0)
{
CVektor		A,B, X;
double	    wab, dDist, fQual, Nmax;
CLocation   NavLoc;
double	    Dmax;
BOOL		bOK = FALSE;

/* activate calculation for distances > "Dmax" NM (default: 40 NM)*/
CCalcDoc* ptCalc = ptInit->GetCalcPtr();
Dmax = ptDim->ConvertDist (ptCalc->GetRouteDiff(), ptCalc->GetRouteDim(), DIM_NM);

						/* get distance between Loc0 and Loc1		*/
Loc0.ToXYZ(&A);					/* Koordinaten von Ort A			*/
Loc1.ToXYZ(&B);					/* Koordinaten von Ort B			*/
wab = A.VektorAngle (B);		/* Winkel zwischen Ortsvektoren		*/
dDist = 60 * (wab * 180./pi);	/* Dist von A nach B in NM			*/

if (dDist <= Dmax) return bOK;	/* waypoints are close enough	*/

							/* nRouteQual = 2 is illegal minimum	*/
fQual = (double)ptCalc->GetRouteQual() + .5;
Nmax = dDist/fQual;				/* max. dist. btn MP and next nav	*/

				/* calculate waypoint btn Loc0 and Loc1 */
X = A.GrossKreis (B, wab/2);	/* X : Ort nach Winkel wab/2		*/
*ptLocX = Loc1;
ptLocX->FromXYZ (X);			/* ptLocX: Ort nach Winkel wab/2	*/


//NavLoc = *ptLocX;		
								/* NavLoc: nearest Nav to ptLocX	*/
if (this->GetNearestFacility (*ptLocX, &NavLoc, &dDist))
    {
    if (dDist <= Nmax)
		{
		short	Ind;
		CWayPoint NavWpt;  
									    /* get basis data of new loc		*/
		this->GetWayPointPtr (Ind0, &NavWpt);
		NavWpt.DeleteBearing(0);		/* delete 1. bearing				*/
		NavWpt.DeleteBearing(1);		/* delete 2. bearing				*/
	    
		NavWpt.SetLoc (NavLoc);   
		
		if (NavWpt.GetAltitude() == NO_ALT)
		{
			short nElevFrGrid_m;
			if (ptElevArrayDoc != NULL)
			{
				if (ptElevArrayDoc->GetElev_m((CLatLon)NavWpt, &nElevFrGrid_m))
				{	
					long lAlt = (long)(ptDim->ConvertDist(nElevFrGrid_m, DIM_METER, NavWpt.GetElevDim()) + 0.5);
					NavWpt.SetAltitude(lAlt);
				}
			}
		}


		long lPrefPlannedAlt = ptCalc->GetPlAlt(NavWpt.GetAltDim());
		long lMinSafeAlt = NavWpt.GetSafeAlt_MSL (NavWpt.GetAltDim());         

		long lOldPlannedAlt = NavWpt.GetPlannedAlt();
		long lNewPlannedAlt = lOldPlannedAlt;

		                  /* set PlAlt according to Elev, SafeAlt or preferred alt	*/
		if (lMinSafeAlt != NO_ALT)
		{								// planned alt < Loc.Elev + Loc.SafeAlt_GND 
			if (lNewPlannedAlt < lMinSafeAlt)
				lNewPlannedAlt = lMinSafeAlt;	
		}
	
		if (lNewPlannedAlt < lPrefPlannedAlt)
			lNewPlannedAlt = lPrefPlannedAlt;

		
		if (lNewPlannedAlt != lOldPlannedAlt)	
		{     
			NavWpt.SetPlannedAlt(lNewPlannedAlt);	

			if (!this->IsElev())
			{
				CString	szText, szDim;
				ptDim->GetDimCstring (&szDim, ptDim->Altitude());
				szText.Format(IDF_CHANGE_ALT, 
								(LPCTSTR)NavWpt.GetName(), 
								lOldPlannedAlt,		
								lNewPlannedAlt, 
								(LPCTSTR)szDim);
				AfxMessageBox ((LPCTSTR)szText);
			}
		}


		this->GetNewIndex ((CLatLon)NavWpt, Ind0, &Ind);

		NavWpt.SetAutoRoute(TRUE);
		if (this->InsertWpt (Ind, NavWpt))
			{
			*ptLocX = NavLoc;		    /* ptLocX: Nav Facility	*/
			bOK = TRUE;
			}
		}
    }
return bOK;
}

/************************************************************************
 *  WayDoc.cpp			     A d d N a v								*
 *  Sucht nach Navigationsanlagen, die auf der Strecke zwischen			*
 *  Loc0 und Loc1 liegen und fgt sie in die Flugstrecke "way" ein.		*
 ************************************************************************/
void CWayDoc::AddNav (CLocation Loc0, CLocation LocX, 
						CLocation Loc1, short Ind0)
{
CLocation   LocX0, LocX1;

if (this->GetNewPoint (Loc0, &LocX0, LocX, Ind0))
	 this->AddNav (Loc0, LocX0, LocX, Ind0);

if (this->GetNewPoint (LocX, &LocX1, Loc1, Ind0))
	 this->AddNav (LocX, LocX1, Loc1, Ind0);
}


/************************************************************************
 *  WayDoc.cpp  		 		A u t o R o u t e						*
 ************************************************************************/
BOOL CWayDoc::AutoRoute() 
{
BOOL bAdded = FALSE;
short	nFirstCnt, nLastCnt, i;
short	nNewInd, nLastInd, NewPoints;
CWayPoint Wpt;
CLocation Loc0, Loc1, LocX;

BeginWaitCursor();		// display the hourglass cursor

nFirstCnt = GetWayPointCnt();
nNewInd	 = 0;
for (i=0; i<nFirstCnt-1; i++)
    {
    nLastCnt = GetWayPointCnt();
    nLastInd = nNewInd;

	this->GetWayPointPtr (nNewInd, &Wpt);
	Loc0 = (CLocation)Wpt;
	this->GetWayPointPtr (nNewInd + 1, &Wpt);
	Loc1 = (CLocation)Wpt;

    if (this->GetNewPoint (Loc0, &LocX, Loc1, nNewInd))
		this->AddNav (Loc0, LocX, Loc1, nNewInd);

    NewPoints = GetWayPointCnt() - nLastCnt;
    nNewInd = nLastInd + NewPoints + 1;
    }

bAdded = (GetWayPointCnt() > nFirstCnt);


EndWaitCursor();		// remove the hourglass cursor
return bAdded;	
}



/************************************************************************
 *  WayDoc.cpp				D e l e t e A u t o R o u t e				*
 ************************************************************************/
void CWayDoc::DeleteAutoRoute ()
{
CWayPoint theWay;
int i=1;						/* i=0 contains start field	*/
while (i < this->GetWayPointCnt ())
	{
	this->GetWayPointPtr (i, &theWay);

	if (theWay.IsAutoRoute())
		this->DeleteWpt (i);
	else	i++;
	}
}

/****************************************************************************
 *	ElevView.cpp				D o C a l c E l e v							*
 ****************************************************************************/
void CWayDoc::DoCalcElev ()
{
	if (ptElevDoc != NULL)
	{
		short		i, nCnt;
		short		nAltDim;
		CLatLon		LLold;
		CWayPoint	WayPt;

		nCnt	= this->GetWayPointCnt();
		nAltDim = DIM_FEET;

		for (i=0; i<nCnt; i++)
		{
			if (this->GetWayPointPtr (i, &WayPt))
			{
				CLatLon LLact(WayPt.GetLat(), WayPt.GetLon());

				if (i>0)	
				{
					short nStartInd, nEndInd;		// Elev Index
					if (ptElevDoc->GetMinMaxIndex (LLold, LLact, &nStartInd, &nEndInd))
					{
						int j;
						BOOL bAdded = FALSE;


						for (j=nStartInd; j<=nEndInd; j++)
						{							// for all Elevs around actual leg
							if (ptElevDoc->AddElevChanges(LLold, LLact, j))
							{
								bAdded = TRUE;
							}
						}

						if (bAdded)
						{
							long lSafeElev_ft=0;
							for (j=0; j<ptElevDoc->GetElevChangeCnt(); j++)
							{		// get all strored elev changes for actual leg in sorted order	
								ELEVPOS	From, To;
								long	lElev_ft;

								if (ptElevDoc->GetElevChange(j, &From, &To, &lElev_ft))
								{
									if (lElev_ft > lSafeElev_ft)
										lSafeElev_ft = lElev_ft;
								}
							}
							lSafeElev_ft += 500;

							ptElevDoc->RemoveElevChanges();
						

							if (i > 1)
							{			// don`t change altitide of first way point
								CWayPoint PrevWpt;
								if (this->GetWayPointPtr (i-1, &PrevWpt))
								{
									if (PrevWpt.GetActAlt_ft() < lSafeElev_ft)
									{
										PrevWpt.SetActAlt_ft(lSafeElev_ft);
										this->ChangeWpt(i-1, PrevWpt);
									}
								}
							}


							if (i < nCnt-1)
							{			// don't change altitude of last way point
								if (WayPt.GetActAlt_ft() < lSafeElev_ft)
								{
									WayPt.SetActAlt_ft(lSafeElev_ft);
									this->ChangeWpt(i, WayPt);	
								}
							}
						}
					}

				}  // i > 0

				LLold = LLact;
			}	// if (GetWayPointPtr...
		} // for all waypoints
	}
}


/************************************************************************
 *  WayDoc.cpp				S e t W a y N a m e							*
 ************************************************************************/
void CWayDoc::SetWayName ()
{
CString szName1;
CString szName2;
short	nLen1, nLen2, nPart1, nPart2;
CWayPoint WayPt;

if (this->GetWayPointPtr (0, &WayPt))
    {                          
    szName1 = WayPt.GetNameForRte ();
    nLen1 = szName1.GetLength();
    }

if (this->GetWayPointPtr (m_WayPts.GetSize()-1, &WayPt))
    {
    szName2 = WayPt.GetNameForRte ();
    nLen2 = szName2.GetLength();
    }

nPart1 = min (8*nLen1 / (nLen1 + nLen2), nLen1);
nPart2 = min (8*nLen2 / (nLen1 + nLen2), nLen2);
szName1 = szName1.Left (nPart1);
szName2 = szName2.Left (nPart2);

this->SetTitle ((LPCTSTR)(szName1+szName2));
CString szPath = ptInit->GetActualPath() + this->GetTitle() + ".way";
this->SetPathName(szPath);
}



/************************************************************************
 *  WayDoc.cpp				D r a w V a r i a t i o n					*
 ************************************************************************/
void CWayDoc::DrawVariation (CDC* pDC, int X, int Y, int CapsX, short nVar)
{   
short	Grad, Dir;
int	nLen;

if (nVar < 0)	Dir = 'W';
	else	Dir = 'E';

if (nVar < 0) nVar *=-1;
Grad = nVar;
 		           
CString szText;      
char	szBuffer[64];
szText.LoadString (IDS_F_VAR);
nLen = sprintf (szBuffer, (LPCTSTR)szText, Grad);
pDC->TextOut (X-CapsX, Y, (LPSTR)szBuffer, nLen);

szText.LoadString (IDS_F_CHAR);
nLen = sprintf (szBuffer, (LPCTSTR)szText, Dir);
pDC->TextOut (X, Y, (LPSTR)szBuffer, nLen);
}  

/************************************************************************
 *  WayDoc.cpp				A v e r a g e A n g l e 					*
 *  Purpose: returns angle between Angle1 and Angle2	[degree]		*
 *	Example: Angle1=330, Angle2=10, AverageAngle = 350					*
 ************************************************************************/
double CWayDoc::AverageAngle (double Angle1, double Angle2)
{
double AngleDiff, AverageAngle;

AngleDiff =  Angle1 - Angle2;
if (AngleDiff < 0) AngleDiff *= -1;

AverageAngle = (Angle1 + Angle2) / 2;
if (AngleDiff > 180) AverageAngle += 180;

if (AverageAngle >= 360)	AverageAngle -= 360;

return AverageAngle;
}

/************************************************************************
 *  WayDoc.cpp	       		L a b e l D i r e c t i o n 				*
 ************************************************************************/
double CWayDoc::LabelDirection (short i, BOOL* ptRightAlign)
{
CWayPoint ActWayPt;
double	TextAngle;
double	Average = -1;

short	nCnt = this->GetWayPointCnt();

if (this->GetWayPointPtr (i, &ActWayPt))
	{
	if (i>0 && i < (nCnt - 1))
		{
		CWayPoint	PrevWayPt, NextWayPt;
		double		PrevTT, NextTT;

		this->GetWayPointPtr (i-1, &PrevWayPt);
		this->GetWayPointPtr (i+1, &NextWayPt);


		ActWayPt.LoxoDist ( PrevWayPt.GetLat(),
							PrevWayPt.GetLon(), &PrevTT);
		ActWayPt.LoxoDist (	NextWayPt.GetLat(),
							NextWayPt.GetLon(), &NextTT);
		
		Average = AverageAngle (PrevTT, NextTT);
		}
	else{
		double	TT;

		if (i==0)
			{							/* first waypoint		*/
			CWayPoint	NextWayPt;

 			this->GetWayPointPtr (i+1, &NextWayPt);

			ActWayPt.LoxoDist ( NextWayPt.GetLat(), 
								NextWayPt.GetLon(), &TT);
			}
			
		if (i == (nCnt - 1))			
			{							/* last waypoint		*/
			CWayPoint	PrevWayPt;
			this->GetWayPointPtr (i-1, &PrevWayPt);

			ActWayPt.LoxoDist ( PrevWayPt.GetLat(),
								PrevWayPt.GetLon(), &TT);
			}
		Average = TT;
		}
	}
	
if (Average>=0 && Average<180)
	{
	TextAngle = Average + 180;
	*ptRightAlign = TRUE;
	}
else{			
	TextAngle = Average - 180;
	*ptRightAlign = FALSE;
	}
	
return TextAngle;
}


/////////////////////////////////////////////////////////////////////////////
// CWayDoc serialization

/************************************************************************
 *  WayDoc.cpp  		 		 H e a d e r T o V e r s 				*
 ************************************************************************/
short CWayDoc::HeaderToVers (char* ptHeader)
{  
short nVersion = 0;
CString szTestVers; 

szTestVers.LoadString (IDS_WAY_VERS1); 
if (szTestVers.Compare (ptHeader) == 0)
	nVersion = 1;

szTestVers.LoadString (IDS_WAY_VERS2);  
if (szTestVers.Compare (ptHeader) == 0)
	nVersion = 2;

szTestVers.LoadString (IDS_WAY_VERS22);  
if (szTestVers.Compare (ptHeader) == 0)
	nVersion = 3;

szTestVers.LoadString (IDS_WAY_VERS223);  
if (szTestVers.Compare (ptHeader) == 0)
	nVersion = 4;

szTestVers.LoadString (IDS_WAY_VERS300);  
if (szTestVers.Compare (ptHeader) == 0)
	nVersion = 5;

szTestVers.LoadString (IDS_WAY_VERS);  
if (szTestVers.Compare (ptHeader) == 0)
	nVersion = ACT_WAY_VERS;

return nVersion;
} 
     
/************************************************************************
 *  WayDoc.cpp  		 		V e r s T o H e a d e r 				*
 ************************************************************************/
void CWayDoc::VersToHeader (LPSTR lpHeader, short nVersion)
{      
CString szVers;
switch (nVersion)
	{
	case 1:	szVers.LoadString (IDS_WAY_VERS1);		break; 
	case 2:	szVers.LoadString (IDS_WAY_VERS2);		break; 
	case 3:	szVers.LoadString (IDS_WAY_VERS22);		break; 
	case 4:	szVers.LoadString (IDS_WAY_VERS223);	break; 
	case 5:	szVers.LoadString (IDS_WAY_VERS300);	break; 
	case ACT_WAY_VERS:	szVers.LoadString (IDS_WAY_VERS);	break; 
	default: szVers.Empty();	break;
    }

_fstrcpy (lpHeader, (LPCTSTR)szVers);
}      

/************************************************************************
 *  WayDoc.cpp				I n i t H e a d e r							*
 ************************************************************************/
void CWayDoc::InitHeader ()
{  
CString		szVers;

szVers.LoadString (IDS_WAY_VERS);
_fstrcpy (m_Header.szVersion, (LPCTSTR)szVers);
m_Header.cFlags		= 0;			// WY_REVERSE, WY_UPDOWN, ...
m_Header.cDummy		= 0;
m_Header.nWptCnt	= 0;
m_Header.nDummy2	= 0;  
}

/************************************************************************
 *  WayDoc.cpp    		 		S e t F l a g B i t						*
 ************************************************************************/
void CWayDoc::SetFlagBit (unsigned char* lpFlag, short BitMask, BOOL bSet)
{					/* BitMask: 1, 2, 4, 8, 16, ...	*/
if (bSet)   *lpFlag = *lpFlag | (unsigned char)BitMask;
    else    *lpFlag = *lpFlag & (unsigned char)(~BitMask);	/* ALT 126: ~	*/
}

/****************************************************************************
 *	WayDoc.cpp				S e t H e a d e r F l a g						*
 ****************************************************************************/
void CWayDoc::SetHeaderFlag (short nBit, BOOL bState)
{
SetFlagBit(&m_Header.cFlags, nBit, bState);
this->SetModifiedFlag (TRUE);
}

/****************************************************************************
 *	WayDoc.cpp				G e t H e a d e r F l a g						*
 ****************************************************************************/
BOOL CWayDoc::GetHeaderFlag (short nBit)
{
BOOL bState = FALSE;
bState = ((m_Header.cFlags & nBit) == nBit);

return bState;
}

/************************************************************************
 *	WayDoc.cpp				S e t P l A l t T o A c t A l t				*
 ************************************************************************/
BOOL CWayDoc::SetPlAltToActAlt()
{
	BOOL bChanged = FALSE;

	for (int i=0; i<m_WayPts.GetSize(); i++)
	{
		CWayPoint WayPt;
		if (this->GetWayPointPtr(i, &WayPt))
		{
			// get PlAlt, convert it to DIM_FEET and store it in ActAlt
			long lPlAlt = WayPt.GetPlannedAlt();
			long lNewAlt = (long)ptDim->ConvertDist(lPlAlt, WayPt.GetAltDim(), DIM_FEET);
			WayPt.SetActAlt_ft (lNewAlt);
			if (ChangeWpt (i, WayPt))
				bChanged = TRUE;
		}
	}
	return bChanged;
}

/************************************************************************
 *	WayDoc.cpp				S e t A c t A l t T o P l A l t				*
 ************************************************************************/
BOOL CWayDoc::SetActAltToPlAlt()
{
	BOOL bChanged = FALSE;

	for (int i=0; i<m_WayPts.GetSize(); i++)
	{
		CWayPoint WayPt;
		if (this->GetWayPointPtr(i, &WayPt))
		{
			// get ActAlt, convert it to actual nAltDim and store it in PlannedAlt
			long lActAlt = WayPt.GetActAlt_ft();
			long lNewAlt = (long)ptDim->ConvertDist(lActAlt, DIM_FEET, WayPt.GetAltDim());
			WayPt.SetPlannedAlt(lNewAlt);
			if (ChangeWpt (i, WayPt))
				bChanged = TRUE;
		}
	}
	return bChanged;
}

/************************************************************************
 *	WayDoc.cpp					S e t R e v e r s e						*
 ************************************************************************/
void CWayDoc::SetReverse (BOOL bActiv)
{
this->SetHeaderFlag (WY_REVERSE, bActiv);
}

/************************************************************************
 *	WayDoc.cpp					S e t U p D o w n						*
 ************************************************************************/
void CWayDoc::SetUpDown (BOOL bActiv) 
{
	this->SetHeaderFlag (WY_UPDOWN, bActiv);

	if (bActiv)	this->InsertUpDown ();
		else	this->DeleteUpDown();
}

/************************************************************************
 *	WayDoc.cpp					S e t S e m i							*
 ************************************************************************/
void CWayDoc::SetSemi (BOOL bActiv)
{
	this->SetHeaderFlag (WY_SEMI, bActiv);

	short nOldActIndex = m_nActIndex;
	CPtrArray	Bearings;		// array of BEARINGUPDATE
	BOOL bBU = this->CreateAndGetBearList (&Bearings);

	if (this->IsUpDown())				// if up down points activated
		this->DeleteUpDown();			// remove them before

	if (this->IsElev())
		this->DoCalcElev();

	if (bActiv)	
		this->InsertSemi ();			// changing altitudes

	if (this->IsUpDown()) 
		this->InsertUpDown ();			// and calculate new up down points

	m_nActIndex = nOldActIndex;
	if (bBU) this->SetAndRemoveBearList(&Bearings);
}

/************************************************************************
 *	WayDoc.cpp					S e t E l e v							*
 ************************************************************************/
void CWayDoc::SetElev (BOOL bActiv)
{
	this->SetHeaderFlag (WY_ELEV, bActiv);

	short nOldActIndex = m_nActIndex;
	CPtrArray	Bearings;		// array of BEARINGUPDATE
	BOOL bBU = this->CreateAndGetBearList (&Bearings);

	if (this->IsUpDown())				// if up down points activated
		this->DeleteUpDown();			// remove them before

	if (bActiv)
		this->DoCalcElev();

	if (this->IsSemi())	
		this->InsertSemi ();			// changing altitudes

	if (this->IsUpDown()) 
		this->InsertUpDown ();			// and calculate new up down points

	m_nActIndex = nOldActIndex;
	if (bBU) this->SetAndRemoveBearList(&Bearings);
}

/************************************************************************
 *	WayDoc.cpp					S e t O r t h o							*
 ************************************************************************/
void CWayDoc::SetOrtho (BOOL bActiv)
{
	this->SetHeaderFlag (WY_ORTHO, bActiv);

	if (bActiv)	this->InsertOrtho ();
		else	this->DeleteOrtho();
}

/************************************************************************
 *	WayDoc.cpp					S e t A u t o R o u t e					*
 ************************************************************************/
BOOL CWayDoc::SetAutoRoute (BOOL bActiv)
{	
BOOL bChanged;

this->SetHeaderFlag (WY_AUTO, bActiv);

short nOldCnt = m_WayPts.GetSize();
if (bActiv)	this->AutoRoute();
	else	this->DeleteAutoRoute();

if (bActiv)
{
	this->CheckAltOnRte(true);		// use ElevArray
	this->OptimizePlannedAlt();

	if (this->IsElev())
	{
		this->SetPlAltToActAlt();	// define ActAlt = PlannedAlt
		this->DoCalcElev();			// change ActAlt
		this->SetActAltToPlAlt();	// set PlannedAlt = ActAlt
	}
}


if (bDemo) this->SetModifiedFlag(FALSE);

bChanged = (m_WayPts.GetSize() != nOldCnt);
return bChanged;
}

/************************************************************************
 *	WayDoc.cpp					S e t A l t e r n a t e s				*
 ************************************************************************/
void CWayDoc::SetAlternates (BOOL bActiv)
{
	this->SetHeaderFlag (WY_ALTERNATES, bActiv);
}

/************************************************************************
 *	WayDoc.cpp					I s R e v e r s e						*
 ************************************************************************/
BOOL CWayDoc::IsReverse ()
{
return this->GetHeaderFlag (WY_REVERSE);
}

/************************************************************************
 *	WayDoc.cpp						I s U p D o w n 					*
 ************************************************************************/
BOOL CWayDoc::IsUpDown ()
{
return this->GetHeaderFlag (WY_UPDOWN);
}


/************************************************************************
 *	WayDoc.cpp						I s S e m i							*
 ************************************************************************/
BOOL CWayDoc::IsSemi ()
{
return this->GetHeaderFlag (WY_SEMI);
}

/************************************************************************
 *	WayDoc.cpp						I s E l e v							*
 ************************************************************************/
BOOL CWayDoc::IsElev ()
{
return this->GetHeaderFlag (WY_ELEV);
}


/************************************************************************
 *	WayDoc.cpp						I s O r t h o						*
 ************************************************************************/
BOOL CWayDoc::IsOrtho ()
{
return this->GetHeaderFlag (WY_ORTHO);
}

/************************************************************************
 *	WayDoc.cpp						I s A u t o	R o u t e				*
 ************************************************************************/
BOOL CWayDoc::IsAutoRoute ()
{
return this->GetHeaderFlag (WY_AUTO);
}

/************************************************************************
 *	WayDoc.cpp					H a s A l t e r n a t e	s				*
 ************************************************************************/
BOOL CWayDoc::HasAlternates ()
{
return this->GetHeaderFlag (WY_ALTERNATES);
}


/************************************************************************
 *  WayDoc.cpp	  		S e t F l i g h t P l a n P a r a m s 			*
 ************************************************************************/
void CWayDoc::SetFlightPlanParams (BOOL bManual, CString szLevel, CString szRoute,
				CString szAlternate1, CString szAlternate2, CString szOtherInfo)
{
	if (this->FlPlInfoChanged (bManual,
							   szLevel,
							   szRoute,
							   szAlternate1,  
							   szAlternate2,  
							   this->GetOtherInfo()))
	{       
		this->SetFlPlManual (bManual);
		this->SetLevel(szLevel);
		this->SetRoute(szRoute);

		if (szAlternate1.GetLength() == 0 && szAlternate2.GetLength() > 0)
		{
			this->SetAlternate(szAlternate2, 1);
			this->SetAlternate(szAlternate1, 2);
		}
		else
		{
			this->SetAlternate(szAlternate1, 1);
			this->SetAlternate(szAlternate2, 2);
		}

		this->SetOtherInfo(szOtherInfo);
		this->SetModifiedFlag (TRUE);

		BOOL bHasAlternates = FALSE;	

		for (int i=1; i<=2; i++)
		{
			BOOL bGetNewAlternate = FALSE;

			CString szAlternate = this->GetAlternate(i);	
			if (szAlternate.GetLength() > 0)
			{			// ICAO indicator of alternate defined in flight plan
				CWayPoint wptAlternate;
				if (this->GetAlternateWayPointPtr(i, &wptAlternate))
				{		// alternate defined in waydoc
					if (wptAlternate.GetIndicator().CompareNoCase((LPCTSTR)szAlternate) == 0)
					{		// same wpt already defined in waydoc
						bHasAlternates = TRUE;	
					}
					else
					{		// need to change alternate of waydoc
						bGetNewAlternate = TRUE;
					}
				}
				else
				{		// no alternate defined in waydoc
					bGetNewAlternate = TRUE;
				}
			}
			else
			{		// no ICAO indicator defined in flight plan
				CWayPoint wptAlternate;
				if (this->GetAlternateWayPointPtr(i, &wptAlternate))
				{		// but alternate defined in waydoc
					wptAlternate.Reset();	// remove it
					this->SetAlternateWayPointPtr(i, wptAlternate);
				}
			}

			if (bGetNewAlternate)
			{
				CWayPoint wptAlternate;

				CLaLoList* ptLaLoList = ptInit->GetLaLoListPtr();
				if (ptLaLoList != NULL)
				{
					CQuickFind*	ptQuickFind = ptInit->GetQuickFindPtr();
					CNameList*	ptNameList	= ptQuickFind->GetNameListPtr();

					long lFirstIndex, lLastIndex;
					BOOL	bMultNames = FALSE;

					if (ptNameList->GetSortedIndexRange ((LPCTSTR)szAlternate, &lFirstIndex, &lLastIndex))
					{
						short	nRegIndex;
						bMultNames = (lFirstIndex != lLastIndex);
						if (!bMultNames)
						{
							CLocation FoundLoc;
							if (ptNameList->GetLoc (&FoundLoc, lFirstIndex, &nRegIndex))
							{
								wptAlternate.SetLoc (FoundLoc);

								short nNewDim = FoundLoc.GetElevDim();
								long l1000ft = (long)ptDim->ConvertDist(1000, DIM_FEET, nNewDim);
								wptAlternate.SetPlannedAlt(FoundLoc.GetAltitude() + l1000ft);
								wptAlternate.SetAltDim(nNewDim);

								this->SetAlternateWayPointPtr(i, wptAlternate);
								bHasAlternates = TRUE;	
							} // loc found
						}
					} // GetSortedIndexRange
				}
			} // bGetNewAlternate
		} // for

		this->SetAlternates(bHasAlternates);
	} // FlPlInfoChanged
}


/************************************************************************
 *  WayDoc.cpp    		 		 S e t L e v e l						*
 ************************************************************************/
void CWayDoc::SetLevel(CString szLevel)
{    
if (szLevel.GetLength() >= LEVEL_SIZE)
	_fstrcpy (m_FlPlInfo.szLevel, (LPCTSTR)szLevel.Left (LEVEL_SIZE-1));
else
	_fstrcpy (m_FlPlInfo.szLevel, (LPCTSTR)szLevel); 
}

/************************************************************************
 *  WayDoc.cpp    		 		 S e t R o u t e						*
 ************************************************************************/
void CWayDoc::SetRoute(CString szRoute)
{    
if (szRoute.GetLength() >= ROUTE_SIZE)
	_fstrcpy (m_FlPlInfo.szRoute, (LPCTSTR)szRoute.Left (ROUTE_SIZE-1));
else
	_fstrcpy (m_FlPlInfo.szRoute, (LPCTSTR)szRoute); 
}

/************************************************************************
 *  WayDoc.cpp    		 S e t A l t e r n a t e						*
 ************************************************************************/
void CWayDoc::SetAlternate(CString szAlternate, short nID)
{    
if (szAlternate.GetLength() >= ALTERNATE_SIZE)
	_fstrcpy ((nID==1)? m_FlPlInfo.szAlternate1 :
					m_FlPlInfo.szAlternate2 , (LPCTSTR)szAlternate.Left (ALTERNATE_SIZE-1)); 
else
	_fstrcpy ((nID==1)? m_FlPlInfo.szAlternate1 :
					m_FlPlInfo.szAlternate2 , (LPCTSTR)szAlternate); 
}

/************************************************************************
 *  WayDoc.cpp    		 		S e t O t h e r I n f o					*
 ************************************************************************/
void CWayDoc::SetOtherInfo(CString szOtherInfo)
{    
if (szOtherInfo.GetLength() >= OTHER_INFO_SIZE)
	_fstrcpy (m_FlPlInfo.szOtherInfo, (LPCTSTR)szOtherInfo.Left (OTHER_INFO_SIZE-1));
else
	_fstrcpy (m_FlPlInfo.szOtherInfo, (LPCTSTR)szOtherInfo); 
}



/************************************************************************
 *  WayDoc.cpp    			G e t L e v e l								*
 ************************************************************************/
CString CWayDoc::GetLevel()
{         
CString szLevel = (CString)m_FlPlInfo.szLevel;
return szLevel;
}

/************************************************************************
 *  WayDoc.cpp    			G e t R o u t e								*
 ************************************************************************/
CString CWayDoc::GetRoute()
{         
CString szRoute = (CString)m_FlPlInfo.szRoute;
return szRoute;
}

/************************************************************************
 *  WayDoc.cpp    			G e t A l t e r n a t e						*
 ************************************************************************/
CString CWayDoc::GetAlternate(short nID)
{         
CString szAlternate = (CString)((nID == 1)? m_FlPlInfo.szAlternate1 :
											m_FlPlInfo.szAlternate2);
return szAlternate;
}

/************************************************************************
 *  WayDoc.cpp    			G e t O t h e r I n f o						*
 ************************************************************************/
CString CWayDoc::GetOtherInfo()
{         
CString szOtherInfo = (CString)m_FlPlInfo.szOtherInfo;
return szOtherInfo;
}

/************************************************************************
 *  WayDoc.cpp    			F l P l I n f o C h a n g e d				*
 ************************************************************************/
BOOL CWayDoc::FlPlInfoChanged (BOOL		bManual,
							   CString szLevel,
							   CString szRoute,
							   CString szAlternate1,  
							   CString szAlternate2,  
							   CString szOtherInfo)
{
if (bManual != m_FlPlInfo.bManual) return TRUE;
if (szLevel.Compare (this->GetLevel()) != 0) return TRUE;
if (szRoute.Compare (this->GetRoute()) != 0) return TRUE;
if (szAlternate1.Compare (this->GetAlternate(1)) != 0) return TRUE;
if (szAlternate2.Compare (this->GetAlternate(2)) != 0) return TRUE;
if (szOtherInfo.Compare (this->GetOtherInfo()) != 0) return TRUE;

return FALSE;
}

/************************************************************************
 *  WayDoc.cpp  		 		S e r i a l i z e						*
 ************************************************************************/
void CWayDoc::Serialize(CArchive& ar)
{
short i, nVersion;

if (ar.IsStoring())
	{
		// TODO: add storing code here
	nVersion = ACT_WAY_VERS;  
	VersToHeader ((LPSTR)&m_Header.szVersion, nVersion);
	this->SetWptCnt (m_WayPts.GetSize());	// put wpt cnt into header
	
	ar.Write (&m_Header, sizeof (WAYHEADER));    
	for (i=0; i<m_WayPts.GetSize(); i++)
		{
		LPWAY lpWayPt = (LPWAY)m_WayPts.GetAt(i);
		CWayPoint WayPt(lpWayPt);
		WayPt.Serialize (ar);  
		}      

	 // changed in mainfrm.cpp: CMainFrame::OnMiscFlpl()
/*	this->SetFlPlManual (ptFlPlDoc->GetDataMode() == DATA_MANUAL);
	this->SetLevel(ptFlPlDoc->GetLevel());
	this->SetRoute(ptFlPlDoc->GetRoute());
	this->SetAlternate(ptFlPlDoc->GetAlternate1(), 1);
	this->SetAlternate(ptFlPlDoc->GetAlternate2(), 2);
	this->SetOtherInfo(ptFlPlDoc->GetOtherInfo());	*/
	ar.Write (&m_FlPlInfo, sizeof (FLPLINFO));

	if (HasAlternates())
	{
		m_Alternate1.Serialize(ar);
		m_Alternate2.Serialize(ar);
	}

	m_bConverted = FALSE;		
	}
else{
		// TODO: add loading code here
	m_bConverted = FALSE;
	this->DeleteArrayOf (&m_WayPts);

	ar.Read (&m_Header, sizeof (WAYHEADER));
	nVersion = this->HeaderToVers (m_Header.szVersion);  
	
	if (nVersion < ACT_WAY_VERS)
		{	  
		CString szText;
	    szText.Format(IDS_CONVERT_WAY, (LPCTSTR)this->GetTitle());
		if (AfxMessageBox ((LPCTSTR)szText, MB_YESNO) == IDYES) 
			{ 
			m_bConverted = TRUE; 
            }
        else{ 
        	AfxMessageBox (IDS_BAD_WAY_VERS);
        	return;
        	}
        }
	
	if (nVersion < 3)
		this->InitHeader ();


	CFile* ptFile = ar.GetFile();
	DWORD	dwLength = (DWORD)ptFile->GetLength();

	short	nCnt;			// get number of entries:	 
	DWORD dwBlockSize;  			
	switch (nVersion)
		{
		case 1:	
			dwBlockSize = sizeof (WAYTYPE1);
			nCnt = (short)((dwLength - MAX_HEADER_SIZE) / dwBlockSize);	
			break;
		case 2:
		case 3:
		case 4:
			dwBlockSize = sizeof (WAYTYPE);
			nCnt = (short)((dwLength - MAX_HEADER_SIZE) / dwBlockSize);
			break;

		case 5:
		case ACT_WAY_VERS:
			nCnt = this->GetWptCnt ();	   // get wpt cnt from header
			break;
        }
	
	for (i=0; i<nCnt; i++)
		{
		CWayPoint WayPt; 
		WayPt.Serialize (ar, nVersion); 
		this->AppendWpt (WayPt);
		} 

//	if (nVersion < ACT_WAY_VERS)
		{	//	header has only stored WY_UPDOWN, WY_SEMI, WY_ORTHO flag
		this->UpdateUpDown();	  // add UPDOWN to route
		this->UpdateOrtho();	  // add ORTHO to route
		}

	nCnt = m_WayPts.GetSize();
	if (nCnt > 0)
		this->SetActInd (0);
	
		if (nVersion > 4)
			{
			ar.Read (&m_FlPlInfo, sizeof (FLPLINFO));

			if (HasAlternates())
			{
				m_Alternate1.Serialize(ar);
				m_Alternate2.Serialize(ar);
			}
		}
	}
}

/////////////////////////////////////////////////////////////////////////////
// CWayDoc diagnostics

#ifdef _DEBUG
void CWayDoc::AssertValid() const
{
	CDocument::AssertValid();
}

void CWayDoc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CWayDoc commands

/************************************************************************
 *  WayDoc.cpp  		 	O n F i l e S a v e							*
 ************************************************************************/
void CWayDoc::OnFileSave() 
{
	// TODO: Add your command handler code here
if (bDemo)
	{
	AfxMessageBox (IDS_SAVE_DISABLED);
	return;
	}

CDocument::OnFileSave();
}

/************************************************************************
 *  WayDoc.cpp  		 	O n F i l e S a v e A s						*
 ************************************************************************/
void CWayDoc::OnFileSaveAs() 
{
	// TODO: Add your command handler code here
CString szOldTitle(this->GetTitle());

if (bDemo)
	{
	AfxMessageBox (IDS_SAVE_DISABLED);
	return;
	}

CDocument::OnFileSaveAs();

CString szNewTitle(this->GetTitle());
if (szNewTitle.CompareNoCase((LPCTSTR)szOldTitle) != 0)
	{				   // uptate view after changing file name	
	this->UpdateAllViews(NULL);		
	}
}
